/* uSim system.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include "stdio.h"
#include "digits.h"
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include <stdlib.h>

#include "memory.h"
#include "system.h"
#include "cpu.h"
#include "monitor.h"
#include "bdev.h"
#include "cdev.h"
#include "clock.h"
#include "hostfile.h"

/**********************************************************************/
#pragma mark *** SYSTEM INTERRUPTS ***

unsigned long gCheckForInterrupts;

/* SystemInterrupt() handles system interrupt activity. */
void SystemInterrupt(void)
{

	CDevPoll();

}

/**********************************************************************/
#pragma mark *** SYSTEM FLAGS ***

Byte gSystemFlags;

#pragma mark GetSystemFlags
/* GetSystemFlags() returns the current system flags. */
/* SystemInterrupt() will be called as a side effect. */
static inline unsigned GetSystemFlags(void);

/* SetSystemFlags() sets or clears system flags. */
void SetSystemFlags(unsigned on, unsigned off)
{

	gSystemFlags = (gSystemFlags & ~off) | on;

}

/**********************************************************************/
#pragma mark *** SYSFLG PORT ***

static Byte SYSFLG;

enum SysFlg {
	SYSSW0 = kSystemSwitch0,
	SYSSW1 = kSystemSwitch1,
	SYSSW2 = kSystemSwitch2,
	SYSSW3 = kSystemSwitch3,
	SYSSW4 = kSystemSwitch4,
	SYSSW5 = kSystemSwitch5,
	SYSSW6 = kSystemSwitch6,
	SYSSW7 = kSystemSwitch7,
	SYSLT0 = kSystemLight0,
	SYSLT1 = kSystemLight1,
	SYSLT2 = kSystemLight2,
	SYSLT3 = kSystemLight3,
	SYSLT4 = kSystemLight4,
	SYSLT5 = kSystemLight5,
	SYSLT6 = kSystemLight6,
	SYSLT7 = kSystemLight7,
	SYSRES = kSystemReset,
	SYSMON = kSystemMonitor,
	SYSHLT = kSystemHalt,
	SYSBRK = kSystemBreak
};

/* sysflg() implements the SYSFLG port. */
static void sysflg(Byte *input, Byte output)
{

	if (input)
		*input = gSystemFlags;
	else
		gSystemFlags = output;

}

/**********************************************************************/
#pragma mark *** SYSTEM IDENTIFICATION ***

static unsigned long gSystemID = kSystemID;

/* GetSystemID() returns the current system ID. */
unsigned long GetSystemID(void)
{

	return gSystemID;

}

/* SetSystemID() sets the system ID. */
void SetSystemID(unsigned long systemID)
{

	gSystemID = systemID;

}

/* ResetSystemID() resets the system ID to its default value. */
void ResetSystemID(void)
{

	gSystemID = kSystemID;

}

/**********************************************************************/
#pragma mark *** SYSID0 SYSID1 SYSID2 SYSID3 PORTS ***

static Byte SYSID0;
static Byte SYSID1;
static Byte SYSID2;
static Byte SYSID3;

/* sysid0() implements the SYSID0 port. */
static void sysid0(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = (GetSystemID() >> 0) & 0x000000FF;

}

/* sysid1() implements the SYSID1 port. */
static void sysid1(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = (GetSystemID() >> 8) & 0x000000FF;

}

/* sysid2() implements the SYSID2 port. */
static void sysid2(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = (GetSystemID() >> 16) & 0x000000FF;

}

/* sysid3() implements the SYSID3 port. */
static void sysid3(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = (GetSystemID() >> 24) & 0x000000FF;

}

/**********************************************************************/
#pragma mark *** BANK0 BANK1 BANK2 BANK3 PORTS ***

#define	MAXBNK kMaxBank
#define	BNKSIZ kBankSize
#define	RAMSIZ (RamSize() / 1024)
#define	ROMSIZ (RomSize() / 1024)
#define	MINROM MinRomBank()
#define	MAXROM MaxRomBank() - 1
#define	MINRAM MinRamBank()
#define	MAXRAM MaxRamBank() - 1

static Byte BANK0;
static Byte BANK1;
static Byte BANK2;
static Byte BANK3;

/* bank0() implements the BANK0 port. */
static void bank0(Byte *input, Byte output)
{

	if (input)
		*input = RdBank(0);
	else
		WrBank(0, output);

}

/* bank1() implements the BANK1 port. */
static void bank1(Byte *input, Byte output)
{

	if (input)
		*input = RdBank(1);
	else
		WrBank(1, output);

}

/* bank2() implements the BANK2 port. */
static void bank2(Byte *input, Byte output)
{

	if (input)
		*input = RdBank(2);
	else
		WrBank(2, output);

}

/* bank3() implements the BANK3 port. */
static void bank3(Byte *input, Byte output)
{

	if (input)
		*input = RdBank(3);
	else
		WrBank(3, output);

}

/**********************************************************************/
#pragma mark *** DMAHI DMALO PORTS ***

static Byte DMAHI;
static Byte DMALO;

/* dmahi() implements the DMAHI port. */
static void dmahi(Byte *input, Byte output)
{

	if (input)
		*input = gDMA.byte.high;
	else
		gDMA.byte.high = output;

}

/* dmalo() implements the DMALO port. */
static void dmalo(Byte *input, Byte output)
{

	if (input)
		*input = gDMA.byte.low;
	else
		gDMA.byte.low = output;

}

/**********************************************************************/
#pragma mark *** DSKNUM DKSCTL SECHI SECLO TRKHI TRKLO PORTS ***

static Byte DSKNUM;
static Byte DSKCTL;
static Byte SECHI;
static Byte SECLO;
static Byte TRKHI;
static Byte TRKLO;

enum {
	MAXDSK = kMaxBDev,
	MAXXLT = kMaxSPT,
	MAXALV = kMaxALV,
	MAXCSV = kMaxCKS,
	MAXPB  = kMaxPB,
	SECSIZ = kBDevSectorSize,
	DSKRD  = 0x01,
	DSKWR  = 0x02,
	DSKOPN = 0x04,
	DSKCLS = 0x08,
	DSKST  = 0x10,
	DSKPB  = 0x20,
	DSKBSY = 0x40,
	DSKERR = 0x80
};

static Byte      gDSKNUM;
static Byte      gDSKST;
static WordBytes gDSKSEC;
static WordBytes gDSKTRK;

/* dsknum() implements the DSKNUM port. */
static void dsknum(Byte *input, Byte output)
{

	if (input)
		*input = gDSKNUM;
	else
		gDSKNUM = output;

}

/* dskctl() implements the DSKCTL port. */
static void dskctl(Byte *input, Byte output)
{
	BDevPtr bDevPtr = BDevIndexToPtr(gDSKNUM);
	Byte buffer[kBDevSectorSize];
	int result;

	if (input) {
		if (bDevPtr == 0)
			*input = DSKERR;
		else switch (BDevStatus(bDevPtr, 0)) {
		case kBDevStatusReadWrite:
			*input = gDSKST | DSKOPN | DSKRD | DSKWR;
			break;
		case kBDevStatusReadOnly:
			*input = gDSKST | DSKOPN | DSKRD;
			break;
		case kBDevStatusClosed:
			*input = DSKCLS | DSKERR;
			break;
		default:
			*input = gDSKST | DSKERR;
			break;
		}
		gDSKST = 0;
	}
	else if (bDevPtr != 0) {
		switch (output) {
		case DSKOPN:
			RdBytes(buffer, gDMA.word, kBDevSectorSize);
			result = BDevOpen(bDevPtr, (char *)buffer, 0);
			break;
		case DSKCLS:
			BDevClose(bDevPtr);
			result = BDevStatus(bDevPtr, 0);
			break;
		case DSKST:
			result = BDevStatus(bDevPtr, (char *)buffer);
			WrBytes(gDMA.word, buffer, kBDevSectorSize);
			break;
		case DSKPB:
			result = BDevInstallParameters(bDevPtr, gDMA.word);
			break;
		case DSKRD:
			result =
				BDevRead(bDevPtr, gDSKTRK.word, gDSKSEC.word, buffer);
			WrBytes(gDMA.word, buffer, kBDevSectorSize);
			break;
		case DSKWR:
			RdBytes(buffer, gDMA.word, kBDevSectorSize);
			result =
				BDevWrite(bDevPtr, gDSKTRK.word, gDSKSEC.word, buffer);
			break;
		}
		gDSKST = (result == kBDevStatusError) ? DSKERR : 0;
	}

}

/* sech() implements the SECHI port. */
static void sechi(Byte *input, Byte output)
{

	if (input)
		*input = gDSKSEC.byte.high;
	else
		gDSKSEC.byte.high = output;

}

/* seclo() implements the SECLO port. */
static void seclo(Byte *input, Byte output)
{

	if (input)
		*input = gDSKSEC.byte.low;
	else
		gDSKSEC.byte.low = output;

}

/* trkhi() implements the TRKHI port. */
static void trkhi(Byte *input, Byte output)
{

	if (input)
		*input = gDSKTRK.byte.high;
	else
		gDSKTRK.byte.high = output;

}

/* trklo() implements the TRKLO port. */
static void trklo(Byte *input, Byte output)
{

	if (input)
		*input = gDSKTRK.byte.low;
	else
		gDSKTRK.byte.low = output;

}

/**********************************************************************/
#pragma mark *** DEVCTL DEVDAT PORTS ***

static Byte DEVCTL;
static Byte DEVDAT;

static CDevPtr gCDevPtr;

/* devctl() implements the DEVCTL port. */
static void devctl(Byte *input, Byte output)
{
	char name[kBDevSectorSize];

	if (input)
		*input = (gCDevPtr != 0) ? CDevStatus(gCDevPtr, 0) : 0;

	else if ((gCDevPtr = CDevIndexToPtr(output & 0x0F)) != 0) {
		switch (output & 0x30) {
		case DEVOPN:
			RdBytes((Byte *)name, gDMA.word, kBDevSectorSize);
			CDevOpen(gCDevPtr, name);
			break;
		case DEVNAM:
			CDevStatus(gCDevPtr, name);
			WrBytes(gDMA.word, (Byte *)name, kBDevSectorSize);
			break;
		case DEVCLS:
			CDevClose(gCDevPtr);
			break;
		case DEVST:
			break;
		}
	}

}

/* devdat() implements the DEVDAT port. */
static void devdat(Byte *input, Byte output)
{

	if (input)
		*input = (gCDevPtr != 0) ? CDevInput(gCDevPtr) : 0;

	else if (gCDevPtr != 0)
		CDevOutput(gCDevPtr, output);

}

/**********************************************************************/
#pragma mark *** CLOCK0 CLOCK1 CLOCK2 CLOCK3 PORTS ***

static unsigned long gCLOCK;

static Byte CLOCK0;
static Byte CLOCK1;
static Byte CLOCK2;
static Byte CLOCK3;

/* clock0() implements the CLOCK0 port. */
static void clock0(Byte *input, Byte output)
{
#pragma unused(output)

	if (input) {
		gCLOCK = GetClock();
		*input = (gCLOCK >> 0) & 0x000000FF;
	}

}

/* clock1() implements the CLOCK1 port. */
static void clock1(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = (gCLOCK >> 8) & 0x000000FF;

}

/* clock2() implements the CLOCK2 port. */
static void clock2(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = (gCLOCK >> 16) & 0x000000FF;

}

/* clock3() implements the CLOCK3 port. */
static void clock3(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = (gCLOCK >> 24) & 0x000000FF;

}

/**********************************************************************/
#pragma mark *** TIMRD PORT ***

static Byte TIMRD;
#define 	TIMS   0
#define 	TIMM   1
#define 	TIMH   2
#define 	TIMDHI 3
#define 	TIMDLO 4

static TimeOfDay gTimeOfDay;
static Byte gTimeOfDayResult;

/* timrd() implements the TIMRD port. */
static void timrd(Byte *input, Byte output)
{

	if (input)
		*input = gTimeOfDayResult;
	else switch (output) {
	case TIMS:
		GetTimeOfDay(&gTimeOfDay);
		gTimeOfDayResult = UnsignedToBCD(gTimeOfDay.seconds);
		break;
	case TIMM:
		gTimeOfDayResult = UnsignedToBCD(gTimeOfDay.minutes);
		break;
	case TIMH:
		gTimeOfDayResult = UnsignedToBCD(gTimeOfDay.hours);
		break;
	case TIMDHI:
		gTimeOfDayResult = (gTimeOfDay.days >> 8) & 0xFF;
		break;
	case TIMDLO:
		gTimeOfDayResult = (gTimeOfDay.days >> 0) & 0xFF;
		break;
	default:
		gTimeOfDayResult = 0;
	}

}

/**********************************************************************/
#pragma mark *** FILCTL FCBHI FCBLO PORTS ***

static Byte FILCTL;
static Byte FCBHI;
static Byte FCBLO;

enum FilCtl {
	FILOPN = 0,
	FILCLS = 1,
	FILDEL = 2,
	FILMAK = 3,
	FILRD  = 4,
	FILWR  = 5,
	FILOK  = 0x00,
	FILERR = 0xFF
};

static WordBytes gFileFCB;
static Byte gFILresult;

/* filctl() implements the FILCTL port. */
static void filctl(Byte *input, Byte output)
{
	Byte buffer[kBDevSectorSize];
	FileFCB fcb;

	if (input)
		*input = gFILresult;
	else switch (output) {
	case FILOPN:
		RdBytes((Byte *)&fcb, gFileFCB.word + 1, sizeof(fcb));
		gFILresult = HostFileOpen(&fcb) ? FILOK : FILERR;
		WrBytes(gFileFCB.word + 1, (Byte *)&fcb, sizeof(fcb));
		break;
	case FILCLS:
		RdBytes((Byte *)&fcb, gFileFCB.word + 1, sizeof(fcb));
		gFILresult = HostFileClose(&fcb) ? FILOK : FILERR;
		WrBytes(gFileFCB.word + 1, (Byte *)&fcb, sizeof(fcb));
		break;
	case FILDEL:
		RdBytes((Byte *)&fcb, gFileFCB.word + 1, sizeof(fcb));
		gFILresult = HostFileDelete(&fcb) ? FILOK : FILERR;
		WrBytes(gFileFCB.word + 1, (Byte *)&fcb, sizeof(fcb));
		break;
	case FILMAK:
		RdBytes((Byte *)&fcb, gFileFCB.word + 1, sizeof(fcb));
		gFILresult = HostFileMake(&fcb) ? FILOK : FILERR;
		WrBytes(gFileFCB.word + 1, (Byte *)&fcb, sizeof(fcb));
		break;
	case FILRD:
		RdBytes((Byte *)&fcb, gFileFCB.word + 1, sizeof(fcb));
		fcb.count = kBDevSectorSize;
		gFILresult =
			HostFileRead(&fcb, (char *)buffer) ? FILOK : FILERR;
		WrBytes(gDMA.word, buffer, kBDevSectorSize);
		WrBytes(gFileFCB.word + 1, (Byte *)&fcb, sizeof(fcb));
		break;
	case FILWR:
		RdBytes((Byte *)&fcb, gFileFCB.word + 1, sizeof(fcb));
		RdBytes(buffer, gDMA.word, kBDevSectorSize);
		fcb.count = kBDevSectorSize;
		gFILresult =
			HostFileWrite(&fcb, (char *)buffer) ? FILOK : FILERR;
		WrBytes(gFileFCB.word + 1, (Byte *)&fcb, sizeof(fcb));
		break;
	default:
		gFILresult = FILERR;
		break;
	}

}

/* fcbhi() implements the FCBHI port. */
static void fcbhi(Byte *input, Byte output)
{

	if (input)
		*input = gFileFCB.byte.high;
	else
		gFileFCB.byte.high = output;

}

/* fcblo() implements the FCBLO port. */
static void fcblo(Byte *input, Byte output)
{

	if (input)
		*input = gFileFCB.byte.low;
	else
		gFileFCB.byte.low = output;

}

/**********************************************************************/
#pragma mark *** I/O PORTS ***

#define kMaxSystemPort 256

PortFunction gSystemPort[kMaxSystemPort];

/* unused() implements the UNUSED ports. */
static void unused(Byte *input, Byte output)
{
#pragma unused(output)

	if (input)
		*input = 0;

}

/* SetupSystemPorts() prepares the gSystemPort[] */
int SetupSystemPorts(void)
{
	unsigned i;

	for (i = 0; i < kMaxSystemPort; i++)
		gSystemPort[i] = unused;

	i = 0;

	/* System Flags */
	gSystemPort[SYSFLG = i++] = sysflg;

	/* System ID */
	gSystemPort[SYSID0 = i++] = sysid0;
	gSystemPort[SYSID1 = i++] = sysid1;
	gSystemPort[SYSID2 = i++] = sysid2;
	gSystemPort[SYSID3 = i++] = sysid3;

	/* Memory Mapping */
	gSystemPort[BANK0 = i++] = bank0;
	gSystemPort[BANK1 = i++] = bank1;
	gSystemPort[BANK2 = i++] = bank2;
	gSystemPort[BANK3 = i++] = bank3;
	gSystemPort[DMAHI = i++] = dmahi;
	gSystemPort[DMALO = i++] = dmalo;

	/* Character Devices */
	gSystemPort[DEVCTL = i++] = devctl;
	gSystemPort[DEVDAT = i++] = devdat;
	gCDevPtr = 0;

	/* Disk Devices */
	gSystemPort[DSKNUM = i++] = dsknum;
	gSystemPort[DSKCTL = i++] = dskctl;
	gSystemPort[SECHI = i++] = sechi;
	gSystemPort[SECLO = i++] = seclo;
	gSystemPort[TRKHI = i++] = trkhi;
	gSystemPort[TRKLO = i++] = trklo;
	gDSKNUM = 0;
	gDSKST = 0;
	gDSKSEC.word = 0;
	gDSKTRK.word = 0;

	/* Time of Day */
	gSystemPort[CLOCK0 = i++] = clock0;
	gSystemPort[CLOCK1 = i++] = clock1;
	gSystemPort[CLOCK2 = i++] = clock2;
	gSystemPort[CLOCK3 = i++] = clock3;
	gSystemPort[TIMRD = i++] = timrd;
	gCLOCK = 0;

	/* Host Files */
	gSystemPort[FILCTL = i++] = filctl;
	gSystemPort[FCBHI = i++] = fcbhi;
	gSystemPort[FCBLO = i++] = fcblo;
	gFILresult = 0;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return 0 if there was an error. */
error:
	return 0;

}

#pragma mark SystemInput
/* SystemInput() reads a byte from an I/O port. */
static inline void SystemInput(Byte port, Byte *value);

#pragma mark SystemOutput
/* SystemOutput() writes a byte to an I/O port. */
static inline void SystemOutput(Byte port, Byte value);

/**********************************************************************/
#pragma mark *** SYSTEM.EQU ***

/* GenerateSystemEqu() generates the system equate file. */
int GenerateSystemEqu(const char *file)
{
	FILE *f;

	printf("GENERATING: %s\n", file);

	if ((f = fopen(file, "w")) == 0) {
		printf("?ERROR\n");
		goto error;
	}

	fprintf(f, "; " kProgram " " CPU " SIMULATOR " kVersion "\n");
	fprintf(f, "; %s GENERATED BY " kProgram " main.c\n", file);
	fprintf(f, "\n");
	fprintf(f, ";SYSTEM FLAGS\n");
	fprintf(f, "SYSFLG	EQU	%d	; SYSTEM CONTROL/STATUS PORT\n", SYSFLG);
	fprintf(f, "SYSSW0	EQU	%d	; SYSTEM SWITCH BIT #0\n", SYSSW0);
	fprintf(f, "SYSSW1	EQU	%d	; SYSTEM SWITCH BIT #1\n", SYSSW1);
	fprintf(f, "SYSSW2	EQU	%d	; SYSTEM SWITCH BIT #2\n", SYSSW2);
	fprintf(f, "SYSSW3 	EQU	%d	; SYSTEM SWITCH BIT #3\n", SYSSW3);
	fprintf(f, "SYSSW4	EQU	%d	; SYSTEM SWITCH BIT #4\n", SYSSW4);
	fprintf(f, "SYSSW5	EQU	%d	; SYSTEM SWITCH BIT #5\n", SYSSW5);
	fprintf(f, "SYSSW6	EQU	%d	; SYSTEM SWITCH BIT #6\n", SYSSW6);
	fprintf(f, "SYSSW7	EQU	%d	; SYSTEM SWITCH BIT #7\n", SYSSW7);
	fprintf(f, "SYSLT0	EQU	%d	; SYSTEM LIGHT BIT #0\n", SYSLT0);
	fprintf(f, "SYSLT1	EQU	%d	; SYSTEM LIGHT BIT #1\n", SYSLT1);
	fprintf(f, "SYSLT2	EQU	%d	; SYSTEM LIGHT BIT #2\n", SYSLT2);
	fprintf(f, "SYSLT3	EQU	%d	; SYSTEM LIGHT BIT #3\n", SYSLT3);
	fprintf(f, "SYSLT4	EQU	%d	; SYSTEM LIGHT BIT #4\n", SYSLT4);
	fprintf(f, "SYSLT5	EQU	%d	; SYSTEM LIGHT BIT #5\n", SYSLT5);
	fprintf(f, "SYSLT6	EQU	%d	; SYSTEM LIGHT BIT #6\n", SYSLT6);
	fprintf(f, "SYSLT7	EQU	%d	; SYSTEM LIGHT BIT #7\n", SYSLT7);
	fprintf(f, "SYSRES	EQU	%d	; RESET IF BIT SET\n", SYSRES);
	fprintf(f, "SYSMON	EQU	%d	; MONITOR IF BIT SET\n", SYSMON);
	fprintf(f, "SYSHLT	EQU	%d	; HALT IF BIT SET\n", SYSHLT);
	fprintf(f, "SYSBRK	EQU	%d	; BREAK IF BIT SET\n", SYSBRK);
	fprintf(f, "\n");
	fprintf(f, "; SYSTEM IDENTIFICATION\n");
	fprintf(f, "SYSID0	EQU	%d	; SYSTEM ID LOW WORD, LOW BYTE PORT\n", SYSID0);
	fprintf(f, "SYSID1	EQU	%d	; SYSTEM ID LOW WORD, HIGH BYTE PORT\n", SYSID1);
	fprintf(f, "SYSID2	EQU	%d	; SYSTEM ID HIGH WORD, LOW BYTE PORT\n", SYSID2);
	fprintf(f, "SYSID3	EQU	%d	; SYSTEM ID HIGH WORD, HIGH BYTE PORT\n", SYSID3);
	fprintf(f, "\n");
	fprintf(f, "; MEMORY MANAGEMENT\n");
	fprintf(f, "ROMSIZ	EQU	%d	; TOTAL KILOBYTES ROM\n", ROMSIZ);
	fprintf(f, "RAMSIZ	EQU	%d	; TOTAL KILOBYTES RAM\n", RAMSIZ);
	fprintf(f, "BANK0	EQU	%d	; MEMORY BANK (0000H-3FFFH) PORT\n", BANK0);
	fprintf(f, "BANK1	EQU	%d	; MEMORY BANK (4000H-7FFFH) PORT\n", BANK1);
	fprintf(f, "BANK2	EQU	%d	; MEMORY BANK (8000H-BFFFH) PORT\n", BANK2);
	fprintf(f, "BANK3	EQU	%d	; MEMORY BANK (C000H-FFFFH) PORT\n", BANK3);
	fprintf(f, "BNKSIZ	EQU	%ld	; TOTAL BYTES IN A MEMORY BANK\n", BNKSIZ);
	fprintf(f, "MINROM	EQU	%d	; FIRST ROM INDEX\n", MINROM);
	fprintf(f, "MAXROM	EQU	%d	; LAST ROM INDEX\n", MAXROM);
	fprintf(f, "MINRAM	EQU	%d	; FIRST RAM INDEX\n", MINRAM);
	fprintf(f, "MAXRAM	EQU	%d	; LAST RAM INDEX\n", MAXRAM);
	fprintf(f, "DMAHI	EQU	%d	; DMA HIGH BYTE PORT\n", DMAHI);
	fprintf(f, "DMALO	EQU	%d	; DMA LOW BYTE PORT\n", DMALO);
	fprintf(f, "\n");
	fprintf(f, "; CHARACTER STREAM DEVICE\n");
	fprintf(f, "DEVCTL	EQU	%d	; DEVICE CONTROL/STATUS PORT\n", DEVCTL);
	fprintf(f, "DEVTTY	EQU	%d	; TTY CONSOLE DEVICE\n", DEVTTY);
	fprintf(f, "DEVCRT	EQU	%d	; CRT CONSOLE DEVICE\n", DEVCRT);
	fprintf(f, "DEVUC1	EQU	%d	; USER DEFINED CONSOLE DEVICE #1\n", DEVUC1);
	fprintf(f, "DEVUC2	EQU	%d	; USER DEFINED CONSOLE DEVICE #2\n", DEVUC2);
	fprintf(f, "DEVPTR	EQU	%d	; PAPER TAPE READER DEVICE\n", DEVPTR);
	fprintf(f, "DEVUR1	EQU	%d	; USER DEFINED READER DEVICE #1\n", DEVUR1);
	fprintf(f, "DEVUR2	EQU	%d	; USER DEFINED READER DEVICE #2\n", DEVUR2);
	fprintf(f, "DEVUR3	EQU	%d	; USER DEFINED READER DEVICE #3\n", DEVUR3);
	fprintf(f, "DEVPTP	EQU	%d	; PAPER TAPE PUNCH DEVICE\n", DEVPTP);
	fprintf(f, "DEVUP1	EQU	%d	; USER DEFINED PUNCH DEVICE #1\n", DEVUP1);
	fprintf(f, "DEVUP2	EQU	%d	; USER DEFINED PUNCH DEVICE #2\n", DEVUP2);
	fprintf(f, "DEVUP3	EQU	%d	; USER DEFINED PUNCH DEVICE #3\n", DEVUP3);
	fprintf(f, "DEVLPT	EQU	%d	; LINE PRINTER DEVICE\n", DEVLPT);
	fprintf(f, "DEVUL1	EQU	%d	; USER DEFINED LINE PRINTER DEVICE #1\n", DEVUL1);
	fprintf(f, "DEVUL2	EQU	%d	; USER DEFINED LINE PRINTER DEVICE #2\n", DEVUL2);
	fprintf(f, "DEVUL3	EQU	%d	; USER DEFINED LINE PRINTER DEVICE #3\n", DEVUL3);
	fprintf(f, "DEVOPN	EQU	%d	; OPEN COMMAND\n", DEVOPN);
	fprintf(f, "DEVNAM	EQU	%d	; NAME COMMAND\n", DEVNAM);
	fprintf(f, "DEVCLS	EQU	%d	; CLOSE COMMAND\n", DEVCLS);
	fprintf(f, "DEVST	EQU	%d	; STATUS COMMAND\n", DEVST);
	fprintf(f, "DEVERR	EQU	%d	; ERROR STATUS\n", DEVERR);
	fprintf(f, "DEVRD	EQU	%d	; READABLE STATUS\n", DEVRD);
	fprintf(f, "DEVWR	EQU	%d	; WRITABLE STATUS\n", DEVWR);
	fprintf(f, "DEVRW	EQU	%d	; READ/WRITE READY STATUS\n", DEVRW);
	fprintf(f, "DEVDAT	EQU	%d	; DEVICE DATA PORT\n", DEVDAT);
	fprintf(f, "\n");
	fprintf(f, "; DISK DEVICE\n");
	fprintf(f, "MAXDSK	EQU	%d	; NUMBER OF DISK DEVICES\n", MAXDSK);
	fprintf(f, "MAXXLT	EQU	%d	; SIZE OF DISK XLT\n", MAXXLT);
	fprintf(f, "MAXALV	EQU	%d	; SIZE OF DISK ALV\n", MAXALV);
	fprintf(f, "MAXCKS	EQU	%d	; SIZE OF DISK CSV\n", MAXCSV);
	fprintf(f, "MAXPB	EQU	%d	; SIZE OF DISK PB\n", MAXPB);
	fprintf(f, "DSKNUM	EQU	%d	; DISK SELECT PORT\n", DSKNUM);
	fprintf(f, "DSKCTL	EQU	%d	; DISK CONTROL/STATUS PORT\n", DSKCTL);
	fprintf(f, "SECSIZ	EQU	%d	; TOTAL BYTES IN SECTOR\n", SECSIZ);
	fprintf(f, "DSKOPN	EQU	%d	; DISK OPEN STATUS/COMMAND\n", DSKOPN);
	fprintf(f, "DSKCLS	EQU	%d	; DISK CLOSE STATUS/COMMAND\n", DSKCLS);
	fprintf(f, "DSKRD	EQU	%d	; DISK READ STATUS/COMMAND\n", DSKRD);
	fprintf(f, "DSKWR	EQU	%d	; DISK WRITE STATUS/COMMAND\n", DSKWR);
	fprintf(f, "DSKST	EQU	%d	; DISK STATUS COMMAND\n", DSKST);
	fprintf(f, "DSKPB	EQU	%d	; DISK PARAMETER BLOCK COMMAND\n", DSKPB);
	fprintf(f, "DSKBSY  EQU	%d	; DISK BUSY STATUS\n", DSKBSY);
	fprintf(f, "DSKERR	EQU	%d	; DISK ERROR STATUS\n", DSKERR);
	fprintf(f, "SECHI	EQU	%d	; SECTOR HIGH BYTE PORT\n", SECHI);
	fprintf(f, "SECLO	EQU	%d	; SECTOR LOW BYTE PORT\n", SECLO);
	fprintf(f, "TRKHI	EQU	%d	; TRACK HIGH BYTE PORT\n", TRKHI);
	fprintf(f, "TRKLO	EQU	%d	; TRACK LOW BYTE PORT\n", TRKLO);
	fprintf(f, "\n");
	fprintf(f, "; CLOCK DEVICE\n");
	fprintf(f, "CLOCK0	EQU	%d	; LOW WORD, LOW BYTE PORT\n", CLOCK0);
	fprintf(f, "CLOCK1	EQU	%d	; LOW WORD, HIGH BYTE PORT\n", CLOCK1);
	fprintf(f, "CLOCK2	EQU	%d	; HIGH WORD, LOW BYTE PORT\n", CLOCK2);
	fprintf(f, "CLOCK3	EQU	%d	; HIGH WORD, HIGH BYTE PORT\n", CLOCK3);
	fprintf(f, "\n");
	fprintf(f, "; TIME OF DAY DEVICE\n");
	fprintf(f, "TIMRD	EQU	%d	; TIME OF DAY PORT\n", TIMRD);
	fprintf(f, "TIMS	EQU	%d	; READ SECONDS COMMAND\n", TIMS);
	fprintf(f, "TIMM	EQU	%d	; READ MINUTES COMMAND\n", TIMM);
	fprintf(f, "TIMH	EQU	%d	; READ READ HOURS COMMAND\n", TIMH);
	fprintf(f, "TIMDHI	EQU	%d	; READ DAYS, HIGH BYTE COMMAND\n", TIMDHI);
	fprintf(f, "TIMDLO	EQU	%d	; READ DAYS, LOW BYTE COMMAND\n", TIMDLO);
	fprintf(f, "\n");
	fprintf(f, "; HOST FILE DEVICE\n");
	fprintf(f, "FILCTL	EQU	%d	; CONTROL/STATUS PORT\n", FILCTL);
	fprintf(f, "FILOPN	EQU	%d	; OPEN FILE COMMAND\n", FILOPN);
	fprintf(f, "FILCLS	EQU	%d	; CLOSE FILE COMMAND\n", FILCLS);
	fprintf(f, "FILDEL	EQU	%d	; DELETE FILE COMMAND\n", FILDEL);
	fprintf(f, "FILMAK	EQU	%d	; MAKE FILE COMMAND\n", FILMAK);
	fprintf(f, "FILRD	EQU	%d	; READ FILE COMMAND\n", FILRD);
	fprintf(f, "FILWR	EQU	%d	; WRITE FILE COMMAND\n", FILWR);
	fprintf(f, "FILOK	EQU	%d	; FILE OK STATUS\n", FILOK);
	fprintf(f, "FILERR	EQU	%d	; FILE ERROR STATUS\n", FILERR);
	fprintf(f, "FCBHI	EQU	%d	; FCB HIGH BYTE PORT\n", FCBHI);
	fprintf(f, "FCBLO	EQU	%d	; FCB LOW BYTE PORT\n", FCBLO);
	fprintf(f, "\n");

	fclose(f);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return 0 if there was an error. */
error:
	return 0;

}
