/* uSim cpu.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#undef GENERATE_TABLES
#undef CALCULATE_TABLES
#define kTablesFile "CPUTAB.H"

/* To generate the tables file:
 * 1) #define GENERATE_TABLES
 * 2) Execute Cpu().
 * 3) Move the tables file to the source directory.
 * 4) #undef GENERATE_TABLES
 */

#ifdef GENERATE_TABLES
#define CALCULATE_TABLES
#endif

#include "stdio.h"

#include "memory.h"
#include "system.h"
#include "cpu.h"
#include "monitor.h"

/* The CPU State. See cpu.h for details. */

static CpuState gCpuState;

#define A        (gCpuState.af.byte.high)
#define F        (gCpuState.af.byte.low)
#define B        (gCpuState.bc.byte.high)
#define C        (gCpuState.bc.byte.low)
#define D        (gCpuState.de.byte.high)
#define E        (gCpuState.de.byte.low)
#define H        (gCpuState.hl.byte.high)
#define L        (gCpuState.hl.byte.low)
#define SP       (gCpuState.sp.word)
#define SP_H     (gCpuState.sp.byte.high)
#define SP_L     (gCpuState.sp.byte.low)
#define PC       (gCpuState.pc.word)
#define PC_H     (gCpuState.pc.byte.high)
#define PC_L     (gCpuState.pc.byte.low)
#define AF       (gCpuState.af.word)
#define BC       (gCpuState.bc.word)
#define DE       (gCpuState.de.word)
#define HL       (gCpuState.hl.word)

#ifdef Z80
#define AF_PRIME (gCpuState.prime.af.word)
#define BC_PRIME (gCpuState.prime.bc.word)
#define DE_PRIME (gCpuState.prime.de.word)
#define HL_PRIME (gCpuState.prime.hl.word)
#define IX       (gCpuState.ix.word)
#define IX_H     (gCpuState.ix.byte.high)
#define IX_L     (gCpuState.ix.byte.low)
#define IY       (gCpuState.iy.word)
#define IY_H     (gCpuState.iy.byte.high)
#define IY_L     (gCpuState.iy.byte.low)
#define I        (gCpuState.i)
#define R        (gCpuState.r)
#define IM       (gCpuState.im)
#endif

#define X        (x.word)
#define X_H      (x.byte.high)
#define X_L      (x.byte.low)

#define Y        (y.word)
#define Y_H      (y.byte.high)
#define Y_L      (y.byte.low)

#define SIGN_FLAG       (F & SIGN)
#define ZERO_FLAG       (F & ZERO)
#define HALFCARRY_FLAG (F & HALFCARRY)
#define OVERFLOW_FLAG   (F & OVERFLOW)
#define PARITY_FLAG     (F & PARITY)
#define SUBTRACT_FLAG   (F & SUBTRACT)
#define CARRY_FLAG      (F & CARRY)

enum {
	CARRY_SHIFT       = 0,
	SUBTRACT_SHIFT    = 1,
	PARITY_SHIFT      = 2,
	OVERFLOW_SHIFT    = 2,
	MAGIC1_SHIFT      = 3,
	HALFCARRY_SHIFT   = 4,
	MAGIC2_SHIFT      = 5,
	ZERO_SHIFT        = 6,
	SIGN_SHIFT        = 7
};

/**********************************************************************/
#pragma mark CPU TABLES

#ifdef CALCULATE_TABLES

#pragma mark gSignZeroMagic2HalfCarryMagic1Overflow[]
static const Byte gSignZeroMagic2HalfCarryMagic1Overflow[256];
#pragma mark gSignZeroMagic2HalfCarryMagic1OverflowSubtract[]
static const Byte gSignZeroMagic2HalfCarryMagic1OverflowSubtract[256];
#pragma mark gSignZeroMagic2Magic1Parity[]
static const Byte gSignZeroMagic2Magic1Parity[256];
#pragma mark gSignZeroMagic2Magic1[]
static const Byte gSignZeroMagic2Magic1[256];
#pragma mark gHalfCarryOverflowCarry[]
static const Byte gHalfCarryOverflowCarry[512];
#pragma mark gHalfCarryCarry[]
static const Byte gHalfCarryCarry[512];

#ifdef GENERATE_TABLES

static int GenerateTables(const char *file)
{
	FILE *f;
	unsigned short x;

	printf("GENERATING %s\n", file);

	if ((f = fopen(file, "w")) == 0)
		goto error;

	fprintf(f, "/* %s GENERATED BY " kProgram " cpu.c */\n\n", file);

	fprintf(f,
	        "static const Byte "
	        "gSignZeroMagic2HalfCarryMagic1Overflow[256] = {\n");
	for (x = 0; x < 256; x++) {
		fprintf(f, (x % 16) ? " " : "	");
		fprintf(f, "%d,", gSignZeroMagic2HalfCarryMagic1Overflow[x]);
		fprintf(f, ((x + 1) % 16) ? "" : "\n");
	}
	fprintf(f, "};\n\n");

	fprintf(f,
	        "static const Byte "
	        "gSignZeroMagic2HalfCarryMagic1OverflowSubtract[256]"
	        " = {\n");
	for (x = 0; x < 256; x++) {
		fprintf(f, (x % 16) ? " " : "	");
		fprintf(f,
		        "%d,",
		        gSignZeroMagic2HalfCarryMagic1OverflowSubtract[x]);
		fprintf(f, ((x + 1) % 16) ? "" : "\n");
	}
	fprintf(f, "};\n\n");

	fprintf(f,
	        "static const Byte "
	        "gSignZeroMagic2Magic1Parity[256] = {\n");
	for (x = 0; x < 256; x++) {
		fprintf(f, (x % 16) ? " " : "	");
		fprintf(f, "%d,", gSignZeroMagic2Magic1Parity[x]);
		fprintf(f, ((x + 1) % 16) ? "" : "\n");
	}
	fprintf(f, "};\n\n");

	fprintf(f,
	        "static const Byte "
	        "gSignZeroMagic2Magic1[256] = {\n");
	for (x = 0; x < 256; x++) {
		fprintf(f, (x % 16) ? " " : "	");
		fprintf(f, "%d,", gSignZeroMagic2Magic1[x]);
		fprintf(f, ((x + 1) % 16) ? "" : "\n");
	}
	fprintf(f, "};\n\n");

	fprintf(f,
	        "static const Byte "
	        "gHalfCarryOverflowCarry[512] = {\n");
	for (x = 0; x < 512; x++) {
		fprintf(f, (x % 16) ? " " : "	");
		fprintf(f, "%d,", gHalfCarryOverflowCarry[x]);
		fprintf(f, ((x + 1) % 16) ? "" : "\n");
	}
	fprintf(f, "};\n\n");

	fprintf(f,
	        "static const Byte "
	        "gHalfCarryCarry[512] = {\n");
	for (x = 0; x < 512; x++) {
		fprintf(f, (x % 16) ? " " : "	");
		fprintf(f, "%d,", gHalfCarryCarry[x]);
		fprintf(f, ((x + 1) % 16) ? "" : "\n");
	}
	fprintf(f, "};\n\n");

	fclose(f);

	return 1;

error:
	if (f != 0)
		fclose(f);
	return 0;

}

#endif

static void CalculateTables(void)
{
	unsigned x;
	unsigned parityCount;

	for (x = 0; x < 256; x++) {
		parityCount = 1;
		if (x & 0x01) parityCount++;
		if (x & 0x02) parityCount++;
		if (x & 0x04) parityCount++;
		if (x & 0x08) parityCount++;
		if (x & 0x10) parityCount++;
		if (x & 0x20) parityCount++;
		if (x & 0x40) parityCount++;
		if (x & 0x80) parityCount++;

		*(Byte *)&(gSignZeroMagic2HalfCarryMagic1OverflowSubtract[x]) =
			(x & (SIGN | MAGIC1 | MAGIC2)) |
			((x == 0) << ZERO_SHIFT) |
			(((x & 0x0F) == 0x0F) << HALFCARRY_SHIFT) |
			((x == 0x7F) << OVERFLOW_SHIFT) |
			SUBTRACT;
		*(Byte *)&(gSignZeroMagic2HalfCarryMagic1Overflow[x]) =
			(x & (SIGN | MAGIC1 | MAGIC2)) |
			((x == 0) << ZERO_SHIFT) |
			(((x & 0x0F) == 0) << HALFCARRY_SHIFT) |
			((x == 0x80) << OVERFLOW_SHIFT);
		*(Byte *)&(gSignZeroMagic2Magic1Parity[x]) =
			(x & (SIGN | MAGIC1 | MAGIC2)) |
			((x == 0) << ZERO_SHIFT) |
			((parityCount & 1) ? PARITY : 0);
		*(Byte *)&(gSignZeroMagic2Magic1[x]) =
			(x & (SIGN | MAGIC1 | MAGIC2)) |
			((x == 0) << ZERO_SHIFT);

	}

	for (x = 0; x < 512; x++) {
		*(Byte *)&(gHalfCarryOverflowCarry[x]) =
		    (x & HALFCARRY) |
		    (((x >> 6) ^ (x >> 5)) & OVERFLOW) |
		    ((x >> 8) & CARRY);
		*(Byte *)&(gHalfCarryCarry[x]) =
		    (x & HALFCARRY) |
		    ((x >> 8) & CARRY);
	}

#ifdef GENERATE_TABLES
	GenerateTables(kTablesFile);
#endif
}

#else

#include kTablesFile

#endif

/**********************************************************************/
#pragma mark UOP1, UOP2, UOP3, UOP4, HALT, NOP, DI, EI, IM

#define OPCODE(X) static inline void X(void)

static inline void _UOP(Word opLen)
{
	PC -= opLen - 1;
}

OPCODE(UOP1) { _UOP(1); }
OPCODE(UOP2) { _UOP(2); }
OPCODE(UOP3) { _UOP(3); }
OPCODE(UOP4) { _UOP(4); }

OPCODE(HALT)
{
	PC -= 1;
	SetSystemFlags(kSystemHalt, 0);
}

OPCODE(NOP)
{
	;
}

OPCODE(DI) { }
OPCODE(EI) { }

#ifdef Z80
OPCODE(IM_0) { IM = 0; }
OPCODE(IM_1) { IM = 1; }
OPCODE(IM_2) { IM = 2; }
#endif

/**********************************************************************/
#pragma mark IN, INI, INIR, IND, INDR

/* 8-bit input from iNN. */
OPCODE(IN_A_iNN) { SystemInput(RdByte(PC++), &A); }

/* 8-bit input from iC. */
#ifdef Z80
OPCODE(IN_A_iC) { SystemInput(C, &A); }
OPCODE(IN_B_iC) { SystemInput(C, &B); }
OPCODE(IN_C_iC) { SystemInput(C, &C); }
OPCODE(IN_D_iC) { SystemInput(C, &D); }
OPCODE(IN_E_iC) { SystemInput(C, &E); }
OPCODE(IN_H_iC) { SystemInput(C, &H); }
OPCODE(IN_L_iC) { SystemInput(C, &L); }
#endif

#ifdef Z80

/* Input byte, increment pointer. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(INI)
{
	SystemInput(C, RwByte(HL++));
	F = (F & ~ZERO) | SUBTRACT | ((--B == 0) << ZERO_SHIFT);
}

/* Input byte, increment pointer, repeat until counter is zero. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(INIR)
{
	do SystemInput(C, RwByte(HL++)); while (--B);
	F |= ZERO | SUBTRACT;
}

/* Input byte, decrement pointer. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(IND)
{
	SystemInput(C, RwByte(HL--));
	F = (F & ~ZERO) | SUBTRACT | ((--B == 0) << ZERO_SHIFT);
}

/* Input byte, decrement pointer, repeat until counter is zero. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(INDR)
{
	do SystemInput(C, RwByte(HL--)); while (--B);
	F |= ZERO | SUBTRACT;
}

#endif

/**********************************************************************/
#pragma mark OUT, OTI, OTIR, OTD, OTDR

/* 8-bit output to iNN. */
OPCODE(OUT_iNN_A) { SystemOutput(RdByte(PC++), A); }

/* 8-bit output to iC. */
#ifdef Z80
OPCODE(OUT_iC_A) { SystemOutput(C, A); }
OPCODE(OUT_iC_B) { SystemOutput(C, B); }
OPCODE(OUT_iC_C) { SystemOutput(C, C); }
OPCODE(OUT_iC_D) { SystemOutput(C, D); }
OPCODE(OUT_iC_E) { SystemOutput(C, E); }
OPCODE(OUT_iC_H) { SystemOutput(C, H); }
OPCODE(OUT_iC_L) { SystemOutput(C, L); }
#endif

#ifdef Z80

/* Output byte, increment pointer. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(OTI)
{
	SystemOutput(C, RdByte(HL++));
	F = (F & ~ZERO) | SUBTRACT | ((--B == 0) << ZERO_SHIFT);
}

/* Output byte, increment pointer, repeat until counter is zero. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(OTIR)
{
	do SystemOutput(C, RdByte(HL++)); while (--B);
	F |= ZERO | SUBTRACT;
}

/* Output byte, decrement pointer. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(OTD)
{
	SystemOutput(C, RdByte(HL--));
	F = (F & ~ZERO) | SUBTRACT | ((--B == 0) << ZERO_SHIFT);
}

/* Output byte, decrement pointer, repeat until counter is zero. */
/* C = port address */
/* HL = pointer */
/* B = counter, Z if zero */
OPCODE(OTDR)
{
	do SystemOutput(C, RdByte(HL--)); while (--B);
	F |= ZERO | SUBTRACT;
}

#endif

/**********************************************************************/
#pragma mark LD, LDI, LDIR, LDD, LDDR

/* Special case to access the I register. */
#ifdef Z80
OPCODE(LD_I_A) { I = A; }
OPCODE(LD_A_I) { A = I; }
#endif

/* 8-bit load to the A register. */
OPCODE(LD_A_A) { A = A; }
OPCODE(LD_A_B) { A = B; }
OPCODE(LD_A_C) { A = C; }
OPCODE(LD_A_D) { A = D; }
OPCODE(LD_A_E) { A = E; }
OPCODE(LD_A_H) { A = H; }
OPCODE(LD_A_L) { A = L; }
#ifdef Z80
OPCODE(LD_A_IXH) { A = IX_H; }
OPCODE(LD_A_IXL) { A = IX_L; }
OPCODE(LD_A_IYH) { A = IY_H; }
OPCODE(LD_A_IYL) { A = IY_L; }
#endif
OPCODE(LD_A_NN) { A = RdByte(PC++); }
OPCODE(LD_A_iBC) { A = RdByte(BC); }
OPCODE(LD_A_iDE) { A = RdByte(DE); }
OPCODE(LD_A_iHL) { A = RdByte(HL); }
#ifdef Z80
OPCODE(LD_A_iIX_NN) { A = RdByte(IX + (char)RdByte(PC++)); }
OPCODE(LD_A_iIY_NN) { A = RdByte(IY + (char)RdByte(PC++)); }
#endif
OPCODE(LD_A_iNNNN)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	A = RdByte(X);
}

/* 8-bit load to the B register. */
OPCODE(LD_B_A) { B = A; }
OPCODE(LD_B_B) { B = B; }
OPCODE(LD_B_C) { B = C; }
OPCODE(LD_B_D) { B = D; }
OPCODE(LD_B_E) { B = E; }
OPCODE(LD_B_H) { B = H; }
OPCODE(LD_B_L) { B = L; }
#ifdef Z80
OPCODE(LD_B_IXH) { B = IX_H; }
OPCODE(LD_B_IXL) { B = IX_L; }
OPCODE(LD_B_IYH) { B = IY_H; }
OPCODE(LD_B_IYL) { B = IY_L; }
#endif
OPCODE(LD_B_NN) { B = RdByte(PC++); }
OPCODE(LD_B_iHL) { B = RdByte(HL); }
#ifdef Z80
OPCODE(LD_B_iIX_NN) { B = RdByte(IX + (char)RdByte(PC++)); }
OPCODE(LD_B_iIY_NN) { B = RdByte(IY + (char)RdByte(PC++)); }
#endif

/* 8-bit load to the C register. */
OPCODE(LD_C_A) { C = A; }
OPCODE(LD_C_B) { C = B; }
OPCODE(LD_C_C) { C = C; }
OPCODE(LD_C_D) { C = D; }
OPCODE(LD_C_E) { C = E; }
OPCODE(LD_C_H) { C = H; }
OPCODE(LD_C_L) { C = L; }
#ifdef Z80
OPCODE(LD_C_IXH) { C = IX_H; }
OPCODE(LD_C_IXL) { C = IX_L; }
OPCODE(LD_C_IYH) { C = IY_H; }
OPCODE(LD_C_IYL) { C = IY_L; }
#endif
OPCODE(LD_C_NN) { C = RdByte(PC++); }
OPCODE(LD_C_iHL) { C = RdByte(HL); }
#ifdef Z80
OPCODE(LD_C_iIX_NN) { C = RdByte(IX + (char)RdByte(PC++)); }
OPCODE(LD_C_iIY_NN) { C = RdByte(IY + (char)RdByte(PC++)); }
#endif

/* 8-bit load to the D register. */
OPCODE(LD_D_A) { D = A; }
OPCODE(LD_D_B) { D = B; }
OPCODE(LD_D_C) { D = C; }
OPCODE(LD_D_D) { D = D; }
OPCODE(LD_D_E) { D = E; }
OPCODE(LD_D_H) { D = H; }
OPCODE(LD_D_L) { D = L; }
#ifdef Z80
OPCODE(LD_D_IXH) { D = IX_H; }
OPCODE(LD_D_IXL) { D = IX_L; }
OPCODE(LD_D_IYH) { D = IY_H; }
OPCODE(LD_D_IYL) { D = IY_L; }
#endif
OPCODE(LD_D_NN) { D = RdByte(PC++); }
OPCODE(LD_D_iHL) { D = RdByte(HL); }
#ifdef Z80
OPCODE(LD_D_iIX_NN) { D = RdByte(IX + (char)RdByte(PC++)); }
OPCODE(LD_D_iIY_NN) { D = RdByte(IY + (char)RdByte(PC++)); }
#endif

/* 8-bit load to the E register. */
OPCODE(LD_E_A) { E = A; }
OPCODE(LD_E_B) { E = B; }
OPCODE(LD_E_C) { E = C; }
OPCODE(LD_E_D) { E = D; }
OPCODE(LD_E_E) { E = E; }
OPCODE(LD_E_H) { E = H; }
OPCODE(LD_E_L) { E = L; }
#ifdef Z80
OPCODE(LD_E_IXH) { E = IX_H; }
OPCODE(LD_E_IXL) { E = IX_L; }
OPCODE(LD_E_IYH) { E = IY_H; }
OPCODE(LD_E_IYL) { E = IY_L; }
#endif
OPCODE(LD_E_NN) { E = RdByte(PC++); }
OPCODE(LD_E_iHL) { E = RdByte(HL); }
#ifdef Z80
OPCODE(LD_E_iIX_NN) { E = RdByte(IX + (char)RdByte(PC++)); }
OPCODE(LD_E_iIY_NN) { E = RdByte(IY + (char)RdByte(PC++)); }
#endif

/* 8-bit load to the H register. */
OPCODE(LD_H_A) { H = A; }
OPCODE(LD_H_B) { H = B; }
OPCODE(LD_H_C) { H = C; }
OPCODE(LD_H_D) { H = D; }
OPCODE(LD_H_E) { H = E; }
OPCODE(LD_H_H) { H = H; }
OPCODE(LD_H_L) { H = L; }
OPCODE(LD_H_NN) { H = RdByte(PC++); }
OPCODE(LD_H_iHL) { H = RdByte(HL); }
#ifdef Z80
OPCODE(LD_H_iIX_NN) { H = RdByte(IX + (char)RdByte(PC++)); }
OPCODE(LD_H_iIY_NN) { H = RdByte(IY + (char)RdByte(PC++)); }
#endif

OPCODE(LD_L_A) { L = A; }
OPCODE(LD_L_B) { L = B; }
OPCODE(LD_L_C) { L = C; }
OPCODE(LD_L_D) { L = D; }
OPCODE(LD_L_E) { L = E; }
OPCODE(LD_L_H) { L = H; }
OPCODE(LD_L_L) { L = L; }
OPCODE(LD_L_NN) { L = RdByte(PC++); }
OPCODE(LD_L_iHL) { L = RdByte(HL); }
#ifdef Z80
OPCODE(LD_L_iIX_NN) { L = RdByte(IX + (char)RdByte(PC++)); }
OPCODE(LD_L_iIY_NN) { L = RdByte(IY + (char)RdByte(PC++)); }
#endif

/* 8-bit load to the IXH register. */
#ifdef Z80
OPCODE(LD_IXH_A) { IX_H = A; }
OPCODE(LD_IXH_B) { IX_H = B; }
OPCODE(LD_IXH_C) { IX_H = C; }
OPCODE(LD_IXH_D) { IX_H = D; }
OPCODE(LD_IXH_E) { IX_H = E; }
OPCODE(LD_IXH_IXH) { IX_H = IX_H; }
OPCODE(LD_IXH_IXL) { IX_H = IX_L; }
OPCODE(LD_IXH_IYH) { IX_H = IY_H; }
OPCODE(LD_IXH_IYL) { IX_H = IY_L; }
OPCODE(LD_IXH_NN) { IX_H = RdByte(PC++); }
#endif

/* 8-bit load to the IXL register. */
#ifdef Z80
OPCODE(LD_IXL_A) { IX_L = A; }
OPCODE(LD_IXL_B) { IX_L = B; }
OPCODE(LD_IXL_C) { IX_L = C; }
OPCODE(LD_IXL_D) { IX_L = D; }
OPCODE(LD_IXL_E) { IX_L = E; }
OPCODE(LD_IXL_IXH) { IX_L = IX_H; }
OPCODE(LD_IXL_IXL) { IX_L = IX_L; }
OPCODE(LD_IXL_IYH) { IX_L = IY_H; }
OPCODE(LD_IXL_IYL) { IX_L = IY_L; }
OPCODE(LD_IXL_NN) { IX_L = RdByte(PC++); }
#endif

/* 8-bit load to the IYH register. */
#ifdef Z80
OPCODE(LD_IYH_A) { IY_H = A; }
OPCODE(LD_IYH_B) { IY_H = B; }
OPCODE(LD_IYH_C) { IY_H = C; }
OPCODE(LD_IYH_D) { IY_H = D; }
OPCODE(LD_IYH_E) { IY_H = E; }
OPCODE(LD_IYH_IXH) { IY_H = IX_H; }
OPCODE(LD_IYH_IXL) { IY_H = IX_L; }
OPCODE(LD_IYH_IYH) { IY_H = IY_H; }
OPCODE(LD_IYH_IYL) { IY_H = IY_L; }
OPCODE(LD_IYH_NN) { IY_H = RdByte(PC++); }
#endif

/* 8-bit load to the IYL register. */
#ifdef Z80
OPCODE(LD_IYL_A) { IY_L = A; }
OPCODE(LD_IYL_B) { IY_L = B; }
OPCODE(LD_IYL_C) { IY_L = C; }
OPCODE(LD_IYL_D) { IY_L = D; }
OPCODE(LD_IYL_E) { IY_L = E; }
OPCODE(LD_IYL_IXH) { IY_L = IX_H; }
OPCODE(LD_IYL_IXL) { IY_L = IX_L; }
OPCODE(LD_IYL_IYH) { IY_L = IY_H; }
OPCODE(LD_IYL_IYL) { IY_L = IY_L; }
OPCODE(LD_IYL_NN) { IY_L = RdByte(PC++); }
#endif

/* 8-bit indirect load to iBC. */
OPCODE(LD_iBC_A) { WrByte(BC, A); }

/* 8-bit indirect load to iDE. */
OPCODE(LD_iDE_A) { WrByte(DE, A); }

/* 8-bit indirect load to iHL. */
OPCODE(LD_iHL_A) { WrByte(HL, A); }
OPCODE(LD_iHL_B) { WrByte(HL, B); }
OPCODE(LD_iHL_C) { WrByte(HL, C); }
OPCODE(LD_iHL_D) { WrByte(HL, D); }
OPCODE(LD_iHL_E) { WrByte(HL, E); }
OPCODE(LD_iHL_H) { WrByte(HL, H); }
OPCODE(LD_iHL_L) { WrByte(HL, L); }
OPCODE(LD_iHL_NN) { WrByte(HL, RdByte(PC++)); }

/* 8-bit indirect load to iIX+NN. */
#ifdef Z80
OPCODE(LD_iIX_NN_A) { WrByte(IX + (char)RdByte(PC++), A); }
OPCODE(LD_iIX_NN_B) { WrByte(IX + (char)RdByte(PC++), B); }
OPCODE(LD_iIX_NN_C) { WrByte(IX + (char)RdByte(PC++), C); }
OPCODE(LD_iIX_NN_D) { WrByte(IX + (char)RdByte(PC++), D); }
OPCODE(LD_iIX_NN_E) { WrByte(IX + (char)RdByte(PC++), E); }
OPCODE(LD_iIX_NN_H) { WrByte(IX + (char)RdByte(PC++), H); }
OPCODE(LD_iIX_NN_L) { WrByte(IX + (char)RdByte(PC++), L); }
OPCODE(LD_iIX_NN_NN)
{
	WrByte(IX + (char)RdByte(PC + 0), RdByte(PC + 1));
	PC += 2;
}
#endif

/* 8-bit indirect load to iIY+NN. */
#ifdef Z80
OPCODE(LD_iIY_NN_B) { WrByte(IY + (char)RdByte(PC++), B); }
OPCODE(LD_iIY_NN_C) { WrByte(IY + (char)RdByte(PC++), C); }
OPCODE(LD_iIY_NN_D) { WrByte(IY + (char)RdByte(PC++), D); }
OPCODE(LD_iIY_NN_E) { WrByte(IY + (char)RdByte(PC++), E); }
OPCODE(LD_iIY_NN_H) { WrByte(IY + (char)RdByte(PC++), H); }
OPCODE(LD_iIY_NN_L) { WrByte(IY + (char)RdByte(PC++), L); }
OPCODE(LD_iIY_NN_A) { WrByte(IY + (char)RdByte(PC++), A); }
OPCODE(LD_iIY_NN_NN)
{
	WrByte(IY + (char)RdByte(PC + 0), RdByte(PC + 1));
	PC += 2;
}
#endif

/* 8-bit indirect load to iNNNN. */
OPCODE(LD_iNNNN_A)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	WrByte(X, A);
}

/* 16-bit load to the BC register. */
OPCODE(LD_BC_NNNN) { C = RdByte(PC++); B = RdByte(PC++); }
#ifdef Z80
OPCODE(LD_BC_iNNNN)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	C = RdByte(X);
	B = RdByte(X + 1);
}
#endif

/* 16-bit load to the DE register. */
OPCODE(LD_DE_NNNN) { E = RdByte(PC++); D = RdByte(PC++); }
#ifdef Z80
OPCODE(LD_DE_iNNNN)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	E = RdByte(X);
	D = RdByte(X + 1);
}
#endif

/* 16-bit load to the HL register. */
OPCODE(LD_HL_NNNN) { L = RdByte(PC++); H = RdByte(PC++); }
OPCODE(LD_HL_iNNNN)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	L = RdByte(X);
	H = RdByte(X + 1);
}

/* 16-bit load to the SP register. */
OPCODE(LD_SP_HL) { SP = HL; }
#ifdef Z80
OPCODE(LD_SP_IX) { SP = IX; }
OPCODE(LD_SP_IY) { SP = IY; }
#endif
OPCODE(LD_SP_NNNN) { SP_L = RdByte(PC++); SP_H = RdByte(PC++); }
#ifdef Z80
OPCODE(LD_SP_iNNNN)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	SP_L = RdByte(X);
	SP_H = RdByte(X + 1);
}
#endif

/* 16-bit load to the IX register. */
#ifdef Z80
OPCODE(LD_IX_NNNN) { IX_L = RdByte(PC++); IX_H = RdByte(PC++); }
OPCODE(LD_IX_iNNNN)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	IX_L = RdByte(X);
	IX_H = RdByte(X + 1);
}
#endif

/* 16-bit load to the IY register. */
#ifdef Z80
OPCODE(LD_IY_NNNN) { IY_L = RdByte(PC++); IY_H = RdByte(PC++); }
OPCODE(LD_IY_iNNNN)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	IY_L = RdByte(X);
	IY_H = RdByte(X + 1);
}
#endif

/* 16-bit indirect load to iNNNN. */
#ifdef Z80
OPCODE(LD_iNNNN_BC)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	WrByte(X, C);
	WrByte(X + 1, B);
}
OPCODE(LD_iNNNN_DE)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	WrByte(X, E);
	WrByte(X + 1, D);
}
#endif
OPCODE(LD_iNNNN_HL)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	WrByte(X, L);
	WrByte(X + 1, H);
}
#ifdef Z80
OPCODE(LD_iNNNN_SP)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	WrByte(X, SP_L);
	WrByte(X + 1, SP_H);
}
OPCODE(LD_iNNNN_IX)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	WrByte(X, IX_L);
	WrByte(X + 1, IX_H);
}
OPCODE(LD_iNNNN_IY)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	WrByte(X, IY_L);
	WrByte(X + 1, IY_H);
}
#endif

#ifdef Z80

/* Move bytes, increment pointer. */
/* DE = destination pointer */
/* HL = source pointer */
/* BC = counter, PE if zero */
OPCODE(LDI)
{
	unsigned long result;
	WrByte(DE++, result = RdByte(HL++));
	result += A;
	F = (F & (SIGN | ZERO | CARRY)) |
	    (result & MAGIC1) |
	    ((result & 2) << HALFCARRY_SHIFT) |
	    ((--BC != 0) << OVERFLOW_SHIFT);
}

/* Move bytes, increment pointer, repeat until counter is zero. */
/* DE = destination pointer */
/* HL = source pointer */
/* BC = counter, PE if zero */
OPCODE(LDIR)
{
	unsigned long result;
	do WrByte(DE++, result = RdByte(HL++));
	while (--BC != 0);
	result += A;
	F = (F & (SIGN | ZERO | CARRY)) |
	    (result & MAGIC1) |
	    ((result & 2) << HALFCARRY_SHIFT);
}

/* Move bytes, decrement pointer. */
/* DE = destination pointer */
/* HL = source pointer */
/* BC = counter, PE if zero */
OPCODE(LDD)
{
	unsigned long result;
	WrByte(DE--, result = RdByte(HL--));
	result += A;
	F = (F & (SIGN | ZERO | CARRY)) |
	    (result & MAGIC1) |
	    ((result & 2) << HALFCARRY_SHIFT) |
	    ((--BC != 0) << OVERFLOW_SHIFT);
}

/* Move bytes, decrement pointer, repeat until counter is zero. */
/* DE = destination pointer */
/* HL = source pointer */
/* BC = counter, PE if zero */
OPCODE(LDDR)
{
	unsigned long result;
	do WrByte(DE--, result = RdByte(HL--));
	while (--BC != 0);
	result += A;
	F = (F & (SIGN | ZERO | CARRY)) |
	    (result & MAGIC1) |
	    ((result & 2) << HALFCARRY_SHIFT);
}

#endif

/**********************************************************************/
#pragma mark EX, EXX

/* Exchange DE and HL registers. */
OPCODE(EX_DE_HL)
{
	WordBytes x;
	X = DE;
	DE = HL;
	HL = X;
}

/* Exchange HL register with top of stack. */
OPCODE(EX_iSP_HL)
{
	WordBytes x;
	X_L = RdByte(SP);
	WrByte(SP, L);
	X_H = RdByte(SP + 1);
	WrByte(SP + 1, H);
	HL = X;
}

#ifdef Z80

/* Exchange IX register with top of stack. */
OPCODE(EX_iSP_IX)
{
	WordBytes x;
	X_L = RdByte(SP);
	WrByte(SP, IX_L);
	X_H = RdByte(SP + 1);
	WrByte(SP + 1, IX_H);
	IX = X;

}

/* Exchange IY register with top of stack. */
OPCODE(EX_iSP_IY)
{
	WordBytes x;
	X_L = RdByte(SP);
	WrByte(SP, IY_L);
	X_H = RdByte(SP + 1);
	WrByte(SP + 1, IY_H);
	IY = X;

}

#endif

#ifdef Z80

/* Exchange register sets AF. */
OPCODE(EX_AF_AF)
{
	Word x;
	x = AF;
	AF = AF_PRIME;
	AF_PRIME = x;
}

/* Exchange register sets BC, DE and HL. */
OPCODE(EXX)
{
	Word x;
	x = BC_PRIME; BC_PRIME = BC; BC = x;
	x = DE_PRIME; DE_PRIME = DE; DE = x;
	x = HL_PRIME; HL_PRIME = HL; HL = x;
}

#endif

/**********************************************************************/
#pragma mark DAA, NEG, CPL, SCF, CCF

/* Decimal Adjust Acumulator */
OPCODE(DAA)
{
	Word result = A;
	Byte half_carry = 0;

	if (SUBTRACT_FLAG) {
		if (HALFCARRY_FLAG || ((A & 0x0F) > 9)) {
			half_carry = ((A & 0x0F) < 6) << HALFCARRY_SHIFT;
			result -= 6;
			result &= 0xFF;
		}
		if (CARRY_FLAG || (A > 0x99))
			result -= 0x160;
	}
	else {
		if (HALFCARRY_FLAG || ((A & 0x0F) > 9)) {
			half_carry = ((A & 0x0F) > 9) << HALFCARRY_SHIFT;
			result += 6;
		}
		if (CARRY_FLAG || ((result & 0x1F0) > 0x90))
			result += 0x60;
	}

	A = result;

	F = (F & (CARRY | SUBTRACT)) |
	    ((result >> 8) & CARRY) |
	    half_carry |
	    gSignZeroMagic2Magic1Parity[A];

}

#ifdef Z80

/* Negate the A register. */
OPCODE(NEG)
{
	Byte a = A;
	F = (SUBTRACT) | gSignZeroMagic2Magic1[A = 0 - A] |
	    (((a & 0x0F) != 0) << HALFCARRY_SHIFT) |
	    ((a == 0x80) << OVERFLOW_SHIFT) |
	    ((a != 0) << CARRY_SHIFT);
}

#endif

/* Compliment (invert) the A register. */
OPCODE(CPL)
{
	A = ~A;
	F = (F & ~(MAGIC1 | MAGIC2)) |
	    (SUBTRACT) |
	    (HALFCARRY) |
	    (A & (MAGIC1 | MAGIC2));
}

/* Set the CARRY flag. */
OPCODE(SCF)
{
	F = (F & ~(MAGIC1 | MAGIC2 | SUBTRACT | HALFCARRY | CARRY)) |
	    (CARRY) |
	    (A & (MAGIC1 | MAGIC2));
}

/* Compliment (invert) the CARRY flag. */
/* HC gets previous C */
OPCODE(CCF)
{
	F = (F & ~(MAGIC1 | MAGIC2 | SUBTRACT | HALFCARRY | CARRY)) |
	    ((F & CARRY) << HALFCARRY_SHIFT) |
	    (~F & CARRY) |
	    (A & (MAGIC1 | MAGIC2));
}

/**********************************************************************/
#pragma mark INC, DEC

/* 8-bit increment */

static inline void _INC(Byte *bytePtr)
{
	F = (F & ~(SUBTRACT |
	           OVERFLOW |
	           HALFCARRY |
	           ZERO |
	           SIGN |
	           MAGIC1 |
	           MAGIC2)) |
	    gSignZeroMagic2HalfCarryMagic1Overflow[++(*bytePtr)];
}

OPCODE(INC_A) { _INC(&A); }
OPCODE(INC_B) { _INC(&B); }
OPCODE(INC_C) { _INC(&C); }
OPCODE(INC_D) { _INC(&D); }
OPCODE(INC_E) { _INC(&E); }
OPCODE(INC_H) { _INC(&H); }
OPCODE(INC_L) { _INC(&L); }
#ifdef Z80
OPCODE(INC_IXH) { _INC(&IX_H); }
OPCODE(INC_IXL) { _INC(&IX_L); }
OPCODE(INC_IYH) { _INC(&IY_H); }
OPCODE(INC_IYL) { _INC(&IY_L); }
#endif
OPCODE(INC_iHL) { _INC(RwByte(HL)); }
#ifdef Z80
OPCODE(INC_iIX_NN) { _INC(RwByte(IX + (char)RdByte(PC++))); }
OPCODE(INC_iIY_NN) { _INC(RwByte(IY + (char)RdByte(PC++))); }
#endif

/* 16-bit increment. */

OPCODE(INC_BC) { BC++; }
OPCODE(INC_DE) { DE++; }
OPCODE(INC_HL) { HL++; }
OPCODE(INC_SP) { SP++; }
#ifdef Z80
OPCODE(INC_IX) { IX++; }
OPCODE(INC_IY) { IY++; }
#endif

/* 8-bit decrement. */

static inline void _DEC(Byte *bytePtr)
{
	F = (F & ~(SUBTRACT |
	           OVERFLOW |
	           HALFCARRY |
	           ZERO |
	           SIGN |
	           MAGIC1 |
	           MAGIC2)) |
	     gSignZeroMagic2HalfCarryMagic1OverflowSubtract[--(*bytePtr)];
}

OPCODE(DEC_A) { _DEC(&A); }
OPCODE(DEC_B) { _DEC(&B); }
OPCODE(DEC_C) { _DEC(&C); }
OPCODE(DEC_D) { _DEC(&D); }
OPCODE(DEC_E) { _DEC(&E); }
OPCODE(DEC_H) { _DEC(&H); }
OPCODE(DEC_L) { _DEC(&L); }
#ifdef Z80
OPCODE(DEC_IXH) { _DEC(&IX_H); }
OPCODE(DEC_IXL) { _DEC(&IX_L); }
OPCODE(DEC_IYH) { _DEC(&IY_H); }
OPCODE(DEC_IYL) { _DEC(&IY_L); }
#endif
OPCODE(DEC_iHL){ _DEC(RwByte(HL)); }
#ifdef Z80
OPCODE(DEC_iIX_NN) { _DEC(RwByte(IX + (char)RdByte(PC++))); }
OPCODE(DEC_iIY_NN) { _DEC(RwByte(IY + (char)RdByte(PC++))); }
#endif

/* 16-bit decrement. */

OPCODE(DEC_BC) { BC--; }
OPCODE(DEC_DE) { DE--; }
OPCODE(DEC_HL) { HL--; }
OPCODE(DEC_SP) { SP--; }
#ifdef Z80
OPCODE(DEC_IX) { IX--; }
OPCODE(DEC_IY) { IY--; }
#endif

/**********************************************************************/
#pragma mark JP, JR

/* Jump to absolute address. */

static inline void _JP(Byte flag)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	if (flag != 0)
		PC = X;
}

OPCODE(JP_NNNN) { _JP(1); }
OPCODE(JP_NZ_NNNN) { _JP(!ZERO_FLAG); }
OPCODE(JP_Z_NNNN) { _JP(ZERO_FLAG); }
OPCODE(JP_NC_NNNN) {  _JP(!CARRY_FLAG); }
OPCODE(JP_C_NNNN) { _JP(CARRY_FLAG); }
OPCODE(JP_PO_NNNN) { _JP(!PARITY_FLAG); }
OPCODE(JP_PE_NNNN) { _JP(PARITY_FLAG); }
OPCODE(JP_P_NNNN) { _JP(!SIGN_FLAG); }
OPCODE(JP_M_NNNN) { _JP(SIGN_FLAG); }

/* Jump to indirect address. */

OPCODE(JP_iHL) { PC = HL; }
#ifdef Z80
OPCODE(JP_iIX) { PC = IX; }
OPCODE(JP_iIY) { PC = IY; }
#endif

/* Jump to PC-relative address. */

#ifdef Z80

static inline void _JR(Byte flag)
{
	WordBytes x;
	X_L = RdByte(PC++);
	if (flag)
		PC += (char)X_L;
}

OPCODE(JR_NN) { _JR(1); }
OPCODE(JR_NZ_NN) { _JR(!ZERO_FLAG); }
OPCODE(JR_Z_NN) { _JR(ZERO_FLAG); }
OPCODE(JR_NC_NN) { _JR(!CARRY_FLAG); }
OPCODE(JR_C_NN) { _JR(CARRY_FLAG); }

#endif

/* Decrement and jump to PC-relative address if zero. */
/* BC = counter */
#ifdef Z80
OPCODE(DJNZ_NN) { _JR(--B); }
#endif

/**********************************************************************/
#pragma mark PUSH, POP

/* Push a 16-bit register value onto the stack. */

OPCODE(PUSH_AF) { WrByte(--SP, A); WrByte(--SP, F); }
OPCODE(PUSH_BC) { WrByte(--SP, B); WrByte(--SP, C); }
OPCODE(PUSH_DE) { WrByte(--SP, D); WrByte(--SP, E); }
OPCODE(PUSH_HL) { WrByte(--SP, H); WrByte(--SP, L); }
#ifdef Z80
OPCODE(PUSH_IX) { WrByte(--SP, IX_H); WrByte(--SP, IX_L); }
OPCODE(PUSH_IY) { WrByte(--SP, IY_H); WrByte(--SP, IY_L); }
#endif

/* Pop a 16-bit register value from the stack. */

OPCODE(POP_AF) { F = RdByte(SP++); A = RdByte(SP++); }
OPCODE(POP_BC) { C = RdByte(SP++); B = RdByte(SP++); }
OPCODE(POP_DE) { E = RdByte(SP++); D = RdByte(SP++); }
OPCODE(POP_HL) { L = RdByte(SP++); H = RdByte(SP++); }
#ifdef Z80
OPCODE(POP_IX) { IX_L = RdByte(SP++); IX_H = RdByte(SP++); }
OPCODE(POP_IY) { IY_L = RdByte(SP++); IY_H = RdByte(SP++); }
#endif

/**********************************************************************/
#pragma mark CALL, RET, RST

/* Subroutine CALL. */

static inline void _CALL(Byte flag)
{
	WordBytes x;
	X_L = RdByte(PC++);
	X_H = RdByte(PC++);
	if (flag != 0) {
		WrByte(--SP, PC_H);
		WrByte(--SP, PC_L);
		PC = X;
	}
}

OPCODE(CALL_NNNN) { _CALL(1); }
OPCODE(CALL_NZ_NNNN) { _CALL(!ZERO_FLAG); }
OPCODE(CALL_Z_NNNN) { _CALL(ZERO_FLAG); }
OPCODE(CALL_NC_NNNN) { _CALL(!CARRY_FLAG); }
OPCODE(CALL_C_NNNN) { _CALL(CARRY_FLAG); }
OPCODE(CALL_PO_NNNN) { _CALL(!PARITY_FLAG); }
OPCODE(CALL_PE_NNNN) { _CALL(PARITY_FLAG); }
OPCODE(CALL_P_NNNN) { _CALL(!SIGN_FLAG); }
OPCODE(CALL_M_NNNN) { _CALL(SIGN_FLAG); }

/* Subroutine RETURN. */

static inline void _RET(Byte flag)
{
	WordBytes x;
	if (flag != 0) {
		X_L = RdByte(SP++);
		X_H = RdByte(SP++);
		PC = X;
	}
}

OPCODE(RET) { _RET(1); }
OPCODE(RET_NZ) { _RET(!ZERO_FLAG); }
OPCODE(RET_Z) { _RET(ZERO_FLAG); }
OPCODE(RET_NC) { _RET(!CARRY_FLAG); }
OPCODE(RET_C) { _RET(CARRY_FLAG); }
OPCODE(RET_PO) { _RET(!PARITY_FLAG); }
OPCODE(RET_PE) { _RET(PARITY_FLAG); }
OPCODE(RET_P) { _RET(!SIGN_FLAG); }
OPCODE(RET_M) { _RET(SIGN_FLAG); }

/* Special case to RETURN from interrupt handler. */
#ifdef Z80
OPCODE(RETN) { UOP2(); }
OPCODE(RETI) { UOP2(); }
#endif

/* Restart to standard subroutine. */

static inline void _RST(Word pc)
{

	WrByte(--SP, PC_H);
	WrByte(--SP, PC_L);
	PC = pc;

}

OPCODE(RST_00) { _RST(0x0000); }
OPCODE(RST_08) { _RST(0x0008); }
OPCODE(RST_10) { _RST(0x0010); }
OPCODE(RST_18) { _RST(0x0018); }
OPCODE(RST_20) { _RST(0x0020); }
OPCODE(RST_28) { _RST(0x0028); }
OPCODE(RST_30) { _RST(0x0030); }
OPCODE(RST_38) { _RST(0x0038); }

/**********************************************************************/
#pragma mark ADD, ADC

/* 8-bit add. */

static inline void _ADD(Byte value)
{
	unsigned long result = A + value;
	Byte bits = gHalfCarryOverflowCarry[A ^ value ^ result];
	F = gSignZeroMagic2Magic1[A = result] | bits;
}

OPCODE(ADD_A) { _ADD(A); }
OPCODE(ADD_B) { _ADD(B); }
OPCODE(ADD_C) { _ADD(C); }
OPCODE(ADD_D) { _ADD(D); }
OPCODE(ADD_E) { _ADD(E); }
OPCODE(ADD_H) { _ADD(H); }
OPCODE(ADD_L) { _ADD(L); }
#ifdef Z80
OPCODE(ADD_IXH) { _ADD(IX_H); }
OPCODE(ADD_IXL) { _ADD(IX_L); }
OPCODE(ADD_IYH) { _ADD(IY_H); }
OPCODE(ADD_IYL) { _ADD(IY_L); }
#endif
OPCODE(ADD_NN) { _ADD(RdByte(PC++)); }
OPCODE(ADD_iHL) { _ADD(RdByte(HL)); }
#ifdef Z80
OPCODE(ADD_iIX_NN) { _ADD(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(ADD_iIY_NN) { _ADD(RdByte(IY + (char)RdByte(PC++))); }
#endif

/* 16-bit add. */

static inline void _ADD_WORD(Word value)
{
	unsigned long result = HL + value;
	Byte bits = gHalfCarryCarry[(HL ^ value ^ result) >> 8];
	HL = result;
	F &= ~(MAGIC2 | HALFCARRY | MAGIC1 | SUBTRACT | CARRY);
	F |= (H & (MAGIC1 | MAGIC2)) | bits;
}

OPCODE(ADD_HL_BC) { _ADD_WORD(BC); }
OPCODE(ADD_HL_DE) { _ADD_WORD(DE); }
OPCODE(ADD_HL_HL) { _ADD_WORD(HL); }
OPCODE(ADD_HL_SP) { _ADD_WORD(SP); }

#ifdef Z80

static inline void _iADD_WORD(Word *x, Word value)
{
	unsigned long result = *x + value;
	Byte bits = gHalfCarryCarry[(*x ^ value ^ result) >> 8];
	*x = result;
	F &= ~(MAGIC2 | HALFCARRY | MAGIC1 | SUBTRACT | CARRY);
	F |= ((result >> 8) & (MAGIC1 | MAGIC2)) | bits;
}

OPCODE(ADD_IX_BC) { _iADD_WORD(&IX, BC); }
OPCODE(ADD_IX_DE) { _iADD_WORD(&IX, DE); }
OPCODE(ADD_IX_IX) { _iADD_WORD(&IX, IX); }
OPCODE(ADD_IX_SP) { _iADD_WORD(&IX, SP); }

OPCODE(ADD_IY_BC) { _iADD_WORD(&IY, BC); }
OPCODE(ADD_IY_DE) { _iADD_WORD(&IY, DE); }
OPCODE(ADD_IY_IY) { _iADD_WORD(&IY, IY); }
OPCODE(ADD_IY_SP) { _iADD_WORD(&IY, SP); }

#endif

/* 8-bit add with carry. */

static inline void _ADC(Byte value)
{
	unsigned long result = A + value + CARRY_FLAG;
	Byte bits = gHalfCarryOverflowCarry[A ^ value ^ result];
	F = gSignZeroMagic2Magic1[A = result] | bits;
}

OPCODE(ADC_A) { _ADC(A); }
OPCODE(ADC_B) { _ADC(B); }
OPCODE(ADC_C) { _ADC(C); }
OPCODE(ADC_D) { _ADC(D); }
OPCODE(ADC_E) { _ADC(E); }
OPCODE(ADC_H) { _ADC(H); }
OPCODE(ADC_L) { _ADC(L); }
#ifdef Z80
OPCODE(ADC_IXH) { _ADC(IX_H); }
OPCODE(ADC_IXL) { _ADC(IX_L); }
OPCODE(ADC_IYH) { _ADC(IY_H); }
OPCODE(ADC_IYL) { _ADC(IY_L); }
#endif
OPCODE(ADC_NN) { _ADC(RdByte(PC++)); }
OPCODE(ADC_iHL) { _ADC(RdByte(HL)); }
#ifdef Z80
OPCODE(ADC_iIX_NN) { _ADC(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(ADC_iIY_NN) { _ADC(RdByte(IY + (char)RdByte(PC++))); }
#endif

/* 16-bit add with carry. */

#ifdef Z80

static inline void _ADC_WORD(Word value)
{
	unsigned long result = HL + value + CARRY_FLAG;
	Byte bits = gHalfCarryOverflowCarry[(HL ^ value ^ result) >> 8];
	HL = result;
	F = (H & (SIGN | MAGIC2 | MAGIC1)) |
	    ((HL == 0) << ZERO_SHIFT) |
	    bits;
}

OPCODE(ADC_HL_BC) { _ADC_WORD(BC); }
OPCODE(ADC_HL_DE) { _ADC_WORD(DE); }
OPCODE(ADC_HL_HL) { _ADC_WORD(HL); }
OPCODE(ADC_HL_SP) { _ADC_WORD(SP); }

#endif

/**********************************************************************/
#pragma mark SUB, SBC

/* 8-bit subtract */

static inline void _SUB(short value)
{
	unsigned long result = (A - value) & 0x1FF;
	Byte bits = gHalfCarryOverflowCarry[A ^ value ^ result];
	F = (SUBTRACT) | gSignZeroMagic2Magic1[A = result] | bits;
}

OPCODE(SUB_A) { _SUB(A); }
OPCODE(SUB_B) { _SUB(B); }
OPCODE(SUB_C) { _SUB(C); }
OPCODE(SUB_D) { _SUB(D); }
OPCODE(SUB_E) { _SUB(E); }
OPCODE(SUB_H) { _SUB(H); }
OPCODE(SUB_L) { _SUB(L); }
#ifdef Z80
OPCODE(SUB_IXH) { _SUB(IX_H); }
OPCODE(SUB_IXL) { _SUB(IX_L); }
OPCODE(SUB_IYH) { _SUB(IY_H); }
OPCODE(SUB_IYL) { _SUB(IY_L); }
#endif
OPCODE(SUB_NN) { _SUB(RdByte(PC++)); }
OPCODE(SUB_iHL) { _SUB(RdByte(HL)); }
#ifdef Z80
OPCODE(SUB_iIX_NN) { _SUB(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(SUB_iIY_NN) { _SUB(RdByte(IY + (char)RdByte(PC++))); }
#endif

/* 8-bit subtract with carry */

static inline void _SBC(Byte value)
{
	unsigned long result = (A - value - CARRY_FLAG) & 0x1FF;
	Byte bits = gHalfCarryOverflowCarry[A ^ value ^ result];
	F = (SUBTRACT) | gSignZeroMagic2Magic1[A = result] | bits;
}

OPCODE(SBC_A) { _SBC(A); }
OPCODE(SBC_B) { _SBC(B); }
OPCODE(SBC_C) { _SBC(C); }
OPCODE(SBC_D) { _SBC(D); }
OPCODE(SBC_E) { _SBC(E); }
OPCODE(SBC_H) { _SBC(H); }
OPCODE(SBC_L) { _SBC(L); }
#ifdef Z80
OPCODE(SBC_IXH) { _SBC(IX_H); }
OPCODE(SBC_IXL) { _SBC(IX_L); }
OPCODE(SBC_IYH) { _SBC(IY_H); }
OPCODE(SBC_IYL) { _SBC(IY_L); }
#endif
OPCODE(SBC_NN) { _SBC(RdByte(PC++)); }
OPCODE(SBC_iHL) { _SBC(RdByte(HL)); }
#ifdef Z80
OPCODE(SBC_iIX_NN) { _SBC(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(SBC_iIY_NN) { _SBC(RdByte(IY + (char)RdByte(PC++))); }
#endif

/* 16-bit subtract with carry */

#ifdef Z80

static inline void _SBC_WORD(Word value)
{
	unsigned long result = (HL - value - CARRY_FLAG) & 0x1FFFF;
	Byte bits = gHalfCarryOverflowCarry[(HL ^ value ^ result) >> 8];
	HL = result;
	F = (SUBTRACT) |
	    (H & (SIGN | MAGIC2 | MAGIC1)) |
	    ((HL == 0) << ZERO_SHIFT) |
	    bits;
}

OPCODE(SBC_HL_BC) { _SBC_WORD(BC); }
OPCODE(SBC_HL_DE) { _SBC_WORD(DE); }
OPCODE(SBC_HL_HL) { _SBC_WORD(HL); }
OPCODE(SBC_HL_SP) { _SBC_WORD(SP); }

#endif

/**********************************************************************/
#pragma mark AND, OR, XOR

/* 8-bit and */

static inline void _AND(Byte value)
{
	F = (HALFCARRY) | gSignZeroMagic2Magic1Parity[A &= value];
}

OPCODE(AND_A) { _AND(A); }
OPCODE(AND_B) { _AND(B); }
OPCODE(AND_C) { _AND(C); }
OPCODE(AND_D) { _AND(D); }
OPCODE(AND_E) { _AND(E); }
OPCODE(AND_H) { _AND(H); }
OPCODE(AND_L) { _AND(L); }
#ifdef Z80
OPCODE(AND_IXH) { _AND(IX_H); }
OPCODE(AND_IXL) { _AND(IX_L); }
OPCODE(AND_IYH) { _AND(IY_H); }
OPCODE(AND_IYL) { _AND(IY_L); }
#endif
OPCODE(AND_NN) { _AND(RdByte(PC++)); }
OPCODE(AND_iHL) { _AND(RdByte(HL)); }
#ifdef Z80
OPCODE(AND_iIX_NN) { _AND(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(AND_iIY_NN) { _AND(RdByte(IY + (char)RdByte(PC++))); }
#endif

/* 8-bit or */

static inline void _OR(Byte value)
{
	F = gSignZeroMagic2Magic1Parity[A |= value];
}

OPCODE(OR_A) { _OR(A); }
OPCODE(OR_B) { _OR(B); }
OPCODE(OR_C) { _OR(C); }
OPCODE(OR_D) { _OR(D); }
OPCODE(OR_E) { _OR(E); }
OPCODE(OR_H) { _OR(H); }
OPCODE(OR_L) { _OR(L); }
#ifdef Z80
OPCODE(OR_IXH) { _OR(IX_H); }
OPCODE(OR_IXL) { _OR(IX_L); }
OPCODE(OR_IYH) { _OR(IY_H); }
OPCODE(OR_IYL) { _OR(IY_L); }
#endif
OPCODE(OR_NN) { _OR(RdByte(PC++)); }
OPCODE(OR_iHL) { _OR(RdByte(HL)); }
#ifdef Z80
OPCODE(OR_iIX_NN) { _OR(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(OR_iIY_NN) { _OR(RdByte(IY + (char)RdByte(PC++))); }
#endif

/* 8-bit exclusive or */

static inline void _XOR(Byte value)
{
	F = gSignZeroMagic2Magic1Parity[A ^= value];
}

OPCODE(XOR_A) { _XOR(A); }
OPCODE(XOR_B) { _XOR(B); }
OPCODE(XOR_C) { _XOR(C); }
OPCODE(XOR_D) { _XOR(D); }
OPCODE(XOR_E) { _XOR(E); }
OPCODE(XOR_H) { _XOR(H); }
OPCODE(XOR_L) { _XOR(L); }
#ifdef Z80
OPCODE(XOR_IXH) { _XOR(IX_H); }
OPCODE(XOR_IXL) { _XOR(IX_L); }
OPCODE(XOR_IYH) { _XOR(IY_H); }
OPCODE(XOR_IYL) { _XOR(IY_L); }
#endif
OPCODE(XOR_NN) { _XOR(RdByte(PC++)); }
OPCODE(XOR_iHL) { _XOR(RdByte(HL)); }
#ifdef Z80
OPCODE(XOR_iIX_NN) { _XOR(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(XOR_iIY_NN) { _XOR(RdByte(IY + (char)RdByte(PC++))); }
#endif

/**********************************************************************/
#pragma mark CP, CPI, CPIR, CPD, CPDR

/* 8-bit compare */

static inline void _CP(Byte value)
{
	unsigned long result = (A - value) & 0x1FF;
	F = (SUBTRACT) |
	    (result & SIGN) |
	    (((result & 0xFF) == 0) << ZERO_SHIFT) |
	    (value & (MAGIC1 | MAGIC2)) |
	    gHalfCarryOverflowCarry[A ^ value ^ result];
}

OPCODE(CP_A) { _CP(A); }
OPCODE(CP_B) { _CP(B); }
OPCODE(CP_C) { _CP(C); }
OPCODE(CP_D) { _CP(D); }
OPCODE(CP_E) { _CP(E); }
OPCODE(CP_H) { _CP(H); }
OPCODE(CP_L) { _CP(L); }
#ifdef Z80
OPCODE(CP_IXH) { _CP(IX_H); }
OPCODE(CP_IXL) { _CP(IX_L); }
OPCODE(CP_IYH) { _CP(IY_H); }
OPCODE(CP_IYL) { _CP(IY_L); }
#endif
OPCODE(CP_NN) { _CP(RdByte(PC++)); }
OPCODE(CP_iHL) { _CP(RdByte(HL)); }
#ifdef Z80
OPCODE(CP_iIX_NN) { _CP(RdByte(IX + (char)RdByte(PC++))); }
OPCODE(CP_iIY_NN) { _CP(RdByte(IY + (char)RdByte(PC++))); }
#endif

#ifdef Z80

/* Compare bytes, increment pointer. */
/* A = target */
/* HL = pointer */
/* BC = counter */
OPCODE(CPI)
{
	Byte value = RdByte(HL++);
	unsigned long result = A - value;
	unsigned long bits = A ^ value ^ result;
	F = (F & CARRY) |
	    (result & SIGN) |
	    (((result & 0xFF) == 0) << ZERO_SHIFT) |
	    ((result - ((bits >> 4) & 1)) & MAGIC1) |
	    (((result - ((bits & 16) >> 4)) & 2) << HALFCARRY_SHIFT) |
	    (bits & HALFCARRY) |
	    ((--BC != 0) << OVERFLOW_SHIFT) |
	    (SUBTRACT);
	if (((result & 15) == 8) && ((bits & 16) != 0))
		F &= ~MAGIC1;
}

/* Compare bytes, increment pointer, repeat until counter is zero. */
/* A = target */
/* HL = pointer */
/* BC = counter */
OPCODE(CPIR)
{
	Byte value;
	unsigned long result;
	unsigned long bits;
	int op;
	do {
		value = RdByte(HL++);
		result = A - value;
		op = (--BC != 0);
	} while (op && (result != 0));
	bits = A ^ value ^ result;
	F = (F & CARRY) |
	    (result & SIGN) |
	    (((result & 0xFF) == 0) << ZERO_SHIFT) |
	    ((result - ((bits >> 4) & 1)) & MAGIC1) |
	    (((result - ((bits & 16) >> 4)) & 2) << HALFCARRY_SHIFT) |
	    (bits & HALFCARRY) |
	    (op << OVERFLOW_SHIFT) |
	    (SUBTRACT);
	if (((result & 15) == 8) && ((bits & 16) != 0))
		F &= ~MAGIC1;
}

/* Compare bytes, decrement pointer. */
/* A = target */
/* HL = pointer */
/* BC = counter */
OPCODE(CPD)
{
	Byte value = RdByte(HL--);
	unsigned long result = A - value;
	unsigned long bits = A ^ value ^ result;
	F = (F & CARRY) |
	    (result & SIGN) |
	    (((result & 0xFF) == 0) << ZERO_SHIFT) |
	    ((result - ((bits >> 4) & 1)) & MAGIC1) |
	    (((result - ((bits & 16) >> 4)) & 2) << HALFCARRY_SHIFT) |
	    (bits & HALFCARRY) |
	    ((--BC != 0) << OVERFLOW_SHIFT) |
	    (SUBTRACT);
	if (((result & 15) == 8) && ((bits & 16) != 0))
		F &= ~MAGIC1;
}

/* Compare bytes, decrement pointer, repeat until counter is zero. */
/* A = target */
/* HL = pointer */
/* BC = counter */
OPCODE(CPDR)
{
	Byte value;
	unsigned long result;
	unsigned long bits;
	int op;
	do {
		value = RdByte(HL--);
		result = A - value;
		op = (--BC != 0);
	} while (op && (result != 0));
	bits = A ^ value ^ result;
	F = (F & CARRY) |
	    (result & SIGN) |
	    (((result & 0xFF) == 0) << ZERO_SHIFT) |
	    ((result - ((bits >> 4) & 1)) & MAGIC1) |
	    (((result - ((bits & 16) >> 4)) & 2) << HALFCARRY_SHIFT) |
	    (bits & HALFCARRY) |
	    (op << OVERFLOW_SHIFT) |
	    (SUBTRACT);
	if (((result & 15) == 8) && ((bits & 16) != 0))
		F &= ~MAGIC1;
}

#endif

/**********************************************************************/
#pragma mark RLCA, RRCA, RLA, RRA, RRD, RLD, RLC, RRC, RL, RR

OPCODE(RLCA)
{
	Byte result = (A << 1) | (A >> 7);
	F = (F & (SIGN | ZERO | OVERFLOW)) |
	    (result & (MAGIC1 | MAGIC2)) |
	    ((A >> 7) & CARRY);
	A = result;
}

OPCODE(RRCA)
{
	Byte result = (A >> 1) | (A << 7);
	F = (F & (SIGN | ZERO | OVERFLOW)) |
	    (result & (MAGIC2 | MAGIC1)) |
	    (A & CARRY);
	A = result;
}

OPCODE(RLA)
{
	Byte result = (A << 1) | (F & CARRY);
	F = (F & (SIGN | ZERO | OVERFLOW)) |
	    (result & (MAGIC1 | MAGIC2)) |
	    ((A >> 7) & CARRY);
	A = result;
}

OPCODE(RRA)
{
	Byte result = (A >> 1) | (F << 7);
	F = (F & (SIGN | ZERO | OVERFLOW)) |
	    (result & (MAGIC2 | MAGIC1)) |
	    (A & CARRY);
	A = result;
}

#ifdef Z80

OPCODE(RRD_iHL)
{
	Byte m = RdByte(HL);
	WrByte(HL, (m >> 4) | ((A & 0x0F) << 4));
	A = (A & 0xF0) | (m & 0x0F);
	F = gSignZeroMagic2Magic1Parity[A] | (F & CARRY);
}

OPCODE(RLD_iHL)
{
	Byte m = RdByte(HL);
	WrByte(HL, (m << 4) | (A &	0x0F));
	A = (A & 0xF0) | (m >> 4);
	F = gSignZeroMagic2Magic1Parity[A] | (F & CARRY);
}

#endif

#ifdef Z80

static inline void _RLC(Byte *value)
{
	Byte result = (*value << 1) | (*value >> 7);
	F = gSignZeroMagic2Magic1Parity[result & 0xFF] | (result & CARRY);
	*value = result;
}

OPCODE(RLC_A) { _RLC(&A); }
OPCODE(RLC_B) { _RLC(&B); }
OPCODE(RLC_C) { _RLC(&C); }
OPCODE(RLC_D) { _RLC(&D); }
OPCODE(RLC_E) { _RLC(&E); }
OPCODE(RLC_H) { _RLC(&H); }
OPCODE(RLC_L) { _RLC(&L); }
OPCODE(RLC_iHL) { _RLC(RwByte(HL)); }
OPCODE(RLC_iIX_NN) { _RLC(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RLC_iIY_NN) { _RLC(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

static inline void _RRC(Byte *value)
{
	Byte result = (*value >> 1) | (*value << 7);
	F = gSignZeroMagic2Magic1Parity[result] | ((result >> 7) & CARRY);
	*value = result;
}

OPCODE(RRC_A) { _RRC(&A); }
OPCODE(RRC_B) { _RRC(&B); }
OPCODE(RRC_C) { _RRC(&C); }
OPCODE(RRC_D) { _RRC(&D); }
OPCODE(RRC_E) { _RRC(&E); }
OPCODE(RRC_H) { _RRC(&H); }
OPCODE(RRC_L) { _RRC(&L); }
OPCODE(RRC_iHL) { _RRC(RwByte(HL)); }
OPCODE(RRC_iIX_NN) { _RRC(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RRC_iIY_NN) { _RRC(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

static inline void _RL(Byte *value)
{
	Byte result = (*value << 1) | CARRY_FLAG;
	F = gSignZeroMagic2Magic1Parity[result] | ((*value >> 7) & CARRY);
	*value = result;
}

OPCODE(RL_A) { _RL(&A); }
OPCODE(RL_B) { _RL(&B); }
OPCODE(RL_C) { _RL(&C); }
OPCODE(RL_D) { _RL(&D); }
OPCODE(RL_E) { _RL(&E); }
OPCODE(RL_H) { _RL(&H); }
OPCODE(RL_L) { _RL(&L); }
OPCODE(RL_iHL) { _RL(RwByte(HL)); }
OPCODE(RL_iIX_NN) { _RL(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RL_iIY_NN) { _RL(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

static inline void _RR(Byte *value)
{
	Byte result = (*value >> 1) | (CARRY_FLAG << 7);
	F = gSignZeroMagic2Magic1Parity[result] | (*value & CARRY);
	*value = result;
}

OPCODE(RR_A) { _RR(&A); }
OPCODE(RR_B) { _RR(&B); }
OPCODE(RR_C) { _RR(&C); }
OPCODE(RR_D) { _RR(&D); }
OPCODE(RR_E) { _RR(&E); }
OPCODE(RR_H) { _RR(&H); }
OPCODE(RR_L) { _RR(&L); }
OPCODE(RR_iHL) { _RR(RwByte(HL)); }
OPCODE(RR_iIX_NN) { _RR(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RR_iIY_NN) { _RR(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

/**********************************************************************/
#pragma mark SLA, SRA, SLIA, SRL

#ifdef Z80

static inline void _SLA(Byte *value)
{
	Byte result = *value << 1;
	F = gSignZeroMagic2Magic1Parity[result] | ((*value >> 7) & CARRY);
	*value = result;
}

OPCODE(SLA_A) { _SLA(&A); }
OPCODE(SLA_B) { _SLA(&B); }
OPCODE(SLA_C) { _SLA(&C); }
OPCODE(SLA_D) { _SLA(&D); }
OPCODE(SLA_E) { _SLA(&E); }
OPCODE(SLA_H) { _SLA(&H); }
OPCODE(SLA_L) { _SLA(&L); }
OPCODE(SLA_iHL) { _SLA(RwByte(HL)); }
OPCODE(SLA_iIX_NN) { _SLA(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SLA_iIY_NN) { _SLA(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

static inline void _SRA(Byte *value)
{
	Byte result = (*value >> 1) | (*value & 0x80);
	F = gSignZeroMagic2Magic1Parity[result] | (*value & CARRY);
	*value = result;
}

OPCODE(SRA_A) { _SRA(&A); }
OPCODE(SRA_B) { _SRA(&B); }
OPCODE(SRA_C) { _SRA(&C); }
OPCODE(SRA_D) { _SRA(&D); }
OPCODE(SRA_E) { _SRA(&E); }
OPCODE(SRA_H) { _SRA(&H); }
OPCODE(SRA_L) { _SRA(&L); }
OPCODE(SRA_iHL) { _SRA(RwByte(HL)); }
OPCODE(SRA_iIX_NN) { _SRA(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SRA_iIY_NN) { _SRA(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

static inline void _SLIA(Byte *value)
{
	Byte result = (*value << 1) | 1;
	F = gSignZeroMagic2Magic1Parity[result] | ((*value >> 7) & CARRY);
	*value = result;
}

OPCODE(SLIA_A) { _SLIA(&A); }
OPCODE(SLIA_B) { _SLIA(&B); }
OPCODE(SLIA_C) { _SLIA(&C); }
OPCODE(SLIA_D) { _SLIA(&D); }
OPCODE(SLIA_E) { _SLIA(&E); }
OPCODE(SLIA_H) { _SLIA(&H); }
OPCODE(SLIA_L) { _SLIA(&L); }
OPCODE(SLIA_iHL) { _SLIA(RwByte(HL)); }
OPCODE(SLIA_iIX_NN) { _SLIA(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SLIA_iIY_NN) { _SLIA(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

static inline void _SRL(Byte *value)
{
	Byte result = *value >> 1;
	F = gSignZeroMagic2Magic1Parity[result] | (*value & CARRY);
	*value = result;
}

OPCODE(SRL_A) { _SRL(&A); }
OPCODE(SRL_B) { _SRL(&B); }
OPCODE(SRL_C) { _SRL(&C); }
OPCODE(SRL_D) { _SRL(&D); }
OPCODE(SRL_E) { _SRL(&E); }
OPCODE(SRL_H) { _SRL(&H); }
OPCODE(SRL_L) { _SRL(&L); }
OPCODE(SRL_iHL) { _SRL(RwByte(HL)); }
OPCODE(SRL_iIX_NN) { _SRL(RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SRL_iIY_NN) { _SRL(RwByte(IY + (char)RdByte(PC - 2))); }

#endif

/**********************************************************************/
#pragma mark BIT, RES, SET

#ifdef Z80

/* Test one of 8 bits. */
/* M set if bit is 1 */

static inline void _BITM(Byte bit, Byte value)
{
	F &= ~(SIGN |
	       ZERO |
	       MAGIC2 |
	       HALFCARRY |
	       MAGIC1 |
	       OVERFLOW |
	       SUBTRACT);
	if (value & bit)
		F |= HALFCARRY | (bit & SIGN);
	else
		F |= HALFCARRY | ZERO | OVERFLOW;
}

static inline void _BIT(Byte bit, Byte value)
{
	F &= ~(SIGN |
	       ZERO |
	       MAGIC2 |
	       HALFCARRY |
	       MAGIC1 |
	       OVERFLOW |
	       SUBTRACT);
	if (value & bit)
		F |= HALFCARRY | (bit & SIGN);
	else
		F |= HALFCARRY | ZERO | OVERFLOW;
	F |= (value & (MAGIC1 | MAGIC2));
}

OPCODE(BIT_0_A) { _BIT(0x01, A); }
OPCODE(BIT_1_A) { _BIT(0x02, A); }
OPCODE(BIT_2_A) { _BIT(0x04, A); }
OPCODE(BIT_3_A) { _BIT(0x08, A); }
OPCODE(BIT_4_A) { _BIT(0x10, A); }
OPCODE(BIT_5_A) { _BIT(0x20, A); }
OPCODE(BIT_6_A) { _BIT(0x40, A); }
OPCODE(BIT_7_A) { _BIT(0x80, A); }

OPCODE(BIT_0_B) { _BIT(0x01, B); }
OPCODE(BIT_1_B) { _BIT(0x02, B); }
OPCODE(BIT_2_B) { _BIT(0x04, B); }
OPCODE(BIT_3_B) { _BIT(0x08, B); }
OPCODE(BIT_4_B) { _BIT(0x10, B); }
OPCODE(BIT_5_B) { _BIT(0x20, B); }
OPCODE(BIT_6_B) { _BIT(0x40, B); }
OPCODE(BIT_7_B) { _BIT(0x80, B); }

OPCODE(BIT_0_C) { _BIT(0x01, C); }
OPCODE(BIT_1_C) { _BIT(0x02, C); }
OPCODE(BIT_2_C) { _BIT(0x04, C); }
OPCODE(BIT_3_C) { _BIT(0x08, C); }
OPCODE(BIT_4_C) { _BIT(0x10, C); }
OPCODE(BIT_5_C) { _BIT(0x20, C); }
OPCODE(BIT_6_C) { _BIT(0x40, C); }
OPCODE(BIT_7_C) { _BIT(0x80, C); }

OPCODE(BIT_0_D) { _BIT(0x01, D); }
OPCODE(BIT_1_D) { _BIT(0x02, D); }
OPCODE(BIT_2_D) { _BIT(0x04, D); }
OPCODE(BIT_3_D) { _BIT(0x08, D); }
OPCODE(BIT_4_D) { _BIT(0x10, D); }
OPCODE(BIT_5_D) { _BIT(0x20, D); }
OPCODE(BIT_6_D) { _BIT(0x40, D); }
OPCODE(BIT_7_D) { _BIT(0x80, D); }

OPCODE(BIT_0_E) { _BIT(0x01, E); }
OPCODE(BIT_1_E) { _BIT(0x02, E); }
OPCODE(BIT_2_E) { _BIT(0x04, E); }
OPCODE(BIT_3_E) { _BIT(0x08, E); }
OPCODE(BIT_4_E) { _BIT(0x10, E); }
OPCODE(BIT_5_E) { _BIT(0x20, E); }
OPCODE(BIT_6_E) { _BIT(0x40, E); }
OPCODE(BIT_7_E) { _BIT(0x80, E); }

OPCODE(BIT_0_H) { _BIT(0x01, H); }
OPCODE(BIT_1_H) { _BIT(0x02, H); }
OPCODE(BIT_2_H) { _BIT(0x04, H); }
OPCODE(BIT_3_H) { _BIT(0x08, H); }
OPCODE(BIT_4_H) { _BIT(0x10, H); }
OPCODE(BIT_5_H) { _BIT(0x20, H); }
OPCODE(BIT_6_H) { _BIT(0x40, H); }
OPCODE(BIT_7_H) { _BIT(0x80, H); }

OPCODE(BIT_0_L) { _BIT(0x01, L); }
OPCODE(BIT_1_L) { _BIT(0x02, L); }
OPCODE(BIT_2_L) { _BIT(0x04, L); }
OPCODE(BIT_3_L) { _BIT(0x08, L); }
OPCODE(BIT_4_L) { _BIT(0x10, L); }
OPCODE(BIT_5_L) { _BIT(0x20, L); }
OPCODE(BIT_6_L) { _BIT(0x40, L); }
OPCODE(BIT_7_L) { _BIT(0x80, L); }

OPCODE(BIT_0_iHL) { _BITM(0x01, RdByte(HL)); }
OPCODE(BIT_1_iHL) { _BITM(0x02, RdByte(HL)); }
OPCODE(BIT_2_iHL) { _BITM(0x04, RdByte(HL)); }
OPCODE(BIT_3_iHL) { _BITM(0x08, RdByte(HL)); }
OPCODE(BIT_4_iHL) { _BITM(0x10, RdByte(HL)); }
OPCODE(BIT_5_iHL) { _BITM(0x20, RdByte(HL)); }
OPCODE(BIT_6_iHL) { _BITM(0x40, RdByte(HL)); }
OPCODE(BIT_7_iHL) { _BITM(0x80, RdByte(HL)); }

OPCODE(BIT_0_iIX_NN) { _BITM(0x01, RdByte(IX + (char)RdByte(PC - 2))); }
OPCODE(BIT_1_iIX_NN) { _BITM(0x02, RdByte(IX + (char)RdByte(PC - 2))); }
OPCODE(BIT_2_iIX_NN) { _BITM(0x04, RdByte(IX + (char)RdByte(PC - 2))); }
OPCODE(BIT_3_iIX_NN) { _BITM(0x08, RdByte(IX + (char)RdByte(PC - 2))); }
OPCODE(BIT_4_iIX_NN) { _BITM(0x10, RdByte(IX + (char)RdByte(PC - 2))); }
OPCODE(BIT_5_iIX_NN) { _BITM(0x20, RdByte(IX + (char)RdByte(PC - 2))); }
OPCODE(BIT_6_iIX_NN) { _BITM(0x40, RdByte(IX + (char)RdByte(PC - 2))); }
OPCODE(BIT_7_iIX_NN) { _BITM(0x80, RdByte(IX + (char)RdByte(PC - 2))); }

OPCODE(BIT_0_iIY_NN) { _BITM(0x01, RdByte(IY + (char)RdByte(PC - 2))); }
OPCODE(BIT_1_iIY_NN) { _BITM(0x02, RdByte(IY + (char)RdByte(PC - 2))); }
OPCODE(BIT_2_iIY_NN) { _BITM(0x04, RdByte(IY + (char)RdByte(PC - 2))); }
OPCODE(BIT_3_iIY_NN) { _BITM(0x08, RdByte(IY + (char)RdByte(PC - 2))); }
OPCODE(BIT_4_iIY_NN) { _BITM(0x10, RdByte(IY + (char)RdByte(PC - 2))); }
OPCODE(BIT_5_iIY_NN) { _BITM(0x20, RdByte(IY + (char)RdByte(PC - 2))); }
OPCODE(BIT_6_iIY_NN) { _BITM(0x40, RdByte(IY + (char)RdByte(PC - 2))); }
OPCODE(BIT_7_iIY_NN) { _BITM(0x80, RdByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

/* Reset (clear) one of 8 bits. */

static inline void _RES(Byte bit, Byte *value)
{
	*value &= ~bit;
}

OPCODE(RES_0_A) { _RES(0x01, &A); }
OPCODE(RES_1_A) { _RES(0x02, &A); }
OPCODE(RES_2_A) { _RES(0x04, &A); }
OPCODE(RES_3_A) { _RES(0x08, &A); }
OPCODE(RES_4_A) { _RES(0x10, &A); }
OPCODE(RES_5_A) { _RES(0x20, &A); }
OPCODE(RES_6_A) { _RES(0x40, &A); }
OPCODE(RES_7_A) { _RES(0x80, &A); }

OPCODE(RES_0_B) { _RES(0x01, &B); }
OPCODE(RES_1_B) { _RES(0x02, &B); }
OPCODE(RES_2_B) { _RES(0x04, &B); }
OPCODE(RES_3_B) { _RES(0x08, &B); }
OPCODE(RES_4_B) { _RES(0x10, &B); }
OPCODE(RES_5_B) { _RES(0x20, &B); }
OPCODE(RES_6_B) { _RES(0x40, &B); }
OPCODE(RES_7_B) { _RES(0x80, &B); }

OPCODE(RES_0_C) { _RES(0x01, &C); }
OPCODE(RES_1_C) { _RES(0x02, &C); }
OPCODE(RES_2_C) { _RES(0x04, &C); }
OPCODE(RES_3_C) { _RES(0x08, &C); }
OPCODE(RES_4_C) { _RES(0x10, &C); }
OPCODE(RES_5_C) { _RES(0x20, &C); }
OPCODE(RES_6_C) { _RES(0x40, &C); }
OPCODE(RES_7_C) { _RES(0x80, &C); }

OPCODE(RES_0_D) { _RES(0x01, &D); }
OPCODE(RES_1_D) { _RES(0x02, &D); }
OPCODE(RES_2_D) { _RES(0x04, &D); }
OPCODE(RES_3_D) { _RES(0x08, &D); }
OPCODE(RES_4_D) { _RES(0x10, &D); }
OPCODE(RES_5_D) { _RES(0x20, &D); }
OPCODE(RES_6_D) { _RES(0x40, &D); }
OPCODE(RES_7_D) { _RES(0x80, &D); }

OPCODE(RES_0_E) { _RES(0x01, &E); }
OPCODE(RES_1_E) { _RES(0x02, &E); }
OPCODE(RES_2_E) { _RES(0x04, &E); }
OPCODE(RES_3_E) { _RES(0x08, &E); }
OPCODE(RES_4_E) { _RES(0x10, &E); }
OPCODE(RES_5_E) { _RES(0x20, &E); }
OPCODE(RES_6_E) { _RES(0x40, &E); }
OPCODE(RES_7_E) { _RES(0x80, &E); }

OPCODE(RES_0_H) { _RES(0x01, &H); }
OPCODE(RES_1_H) { _RES(0x02, &H); }
OPCODE(RES_2_H) { _RES(0x04, &H); }
OPCODE(RES_3_H) { _RES(0x08, &H); }
OPCODE(RES_4_H) { _RES(0x10, &H); }
OPCODE(RES_5_H) { _RES(0x20, &H); }
OPCODE(RES_6_H) { _RES(0x40, &H); }
OPCODE(RES_7_H) { _RES(0x80, &H); }

OPCODE(RES_0_L) { _RES(0x01, &L); }
OPCODE(RES_1_L) { _RES(0x02, &L); }
OPCODE(RES_2_L) { _RES(0x04, &L); }
OPCODE(RES_3_L) { _RES(0x08, &L); }
OPCODE(RES_4_L) { _RES(0x10, &L); }
OPCODE(RES_5_L) { _RES(0x20, &L); }
OPCODE(RES_6_L) { _RES(0x40, &L); }
OPCODE(RES_7_L) { _RES(0x80, &L); }

OPCODE(RES_0_iHL) { _RES(0x01, RwByte(HL)); }
OPCODE(RES_1_iHL) { _RES(0x02, RwByte(HL)); }
OPCODE(RES_2_iHL) { _RES(0x04, RwByte(HL)); }
OPCODE(RES_3_iHL) { _RES(0x08, RwByte(HL)); }
OPCODE(RES_4_iHL) { _RES(0x10, RwByte(HL)); }
OPCODE(RES_5_iHL) { _RES(0x20, RwByte(HL)); }
OPCODE(RES_6_iHL) { _RES(0x40, RwByte(HL)); }
OPCODE(RES_7_iHL) { _RES(0x80, RwByte(HL)); }

OPCODE(RES_0_iIX_NN) { _RES(0x01, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RES_1_iIX_NN) { _RES(0x02, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RES_2_iIX_NN) { _RES(0x04, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RES_3_iIX_NN) { _RES(0x08, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RES_4_iIX_NN) { _RES(0x10, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RES_5_iIX_NN) { _RES(0x20, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RES_6_iIX_NN) { _RES(0x40, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(RES_7_iIX_NN) { _RES(0x80, RwByte(IX + (char)RdByte(PC - 2))); }

OPCODE(RES_0_iIY_NN) { _RES(0x01, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(RES_1_iIY_NN) { _RES(0x02, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(RES_2_iIY_NN) { _RES(0x04, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(RES_3_iIY_NN) { _RES(0x08, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(RES_4_iIY_NN) { _RES(0x10, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(RES_5_iIY_NN) { _RES(0x20, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(RES_6_iIY_NN) { _RES(0x40, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(RES_7_iIY_NN) { _RES(0x80, RwByte(IY + (char)RdByte(PC - 2))); }

#endif

#ifdef Z80

/* Set one of 8 bits. */

static inline void _SET(Byte bit, Byte *value)
{
	*value |= bit;
}

OPCODE(SET_0_A) { _SET(0x01, &A); }
OPCODE(SET_1_A) { _SET(0x02, &A); }
OPCODE(SET_2_A) { _SET(0x04, &A); }
OPCODE(SET_3_A) { _SET(0x08, &A); }
OPCODE(SET_4_A) { _SET(0x10, &A); }
OPCODE(SET_5_A) { _SET(0x20, &A); }
OPCODE(SET_6_A) { _SET(0x40, &A); }
OPCODE(SET_7_A) { _SET(0x80, &A); }

OPCODE(SET_0_B) { _SET(0x01, &B); }
OPCODE(SET_1_B) { _SET(0x02, &B); }
OPCODE(SET_2_B) { _SET(0x04, &B); }
OPCODE(SET_3_B) { _SET(0x08, &B); }
OPCODE(SET_4_B) { _SET(0x10, &B); }
OPCODE(SET_5_B) { _SET(0x20, &B); }
OPCODE(SET_6_B) { _SET(0x40, &B); }
OPCODE(SET_7_B) { _SET(0x80, &B); }

OPCODE(SET_0_C) { _SET(0x01, &C); }
OPCODE(SET_1_C) { _SET(0x02, &C); }
OPCODE(SET_2_C) { _SET(0x04, &C); }
OPCODE(SET_3_C) { _SET(0x08, &C); }
OPCODE(SET_4_C) { _SET(0x10, &C); }
OPCODE(SET_5_C) { _SET(0x20, &C); }
OPCODE(SET_6_C) { _SET(0x40, &C); }
OPCODE(SET_7_C) { _SET(0x80, &C); }

OPCODE(SET_0_D) { _SET(0x01, &D); }
OPCODE(SET_1_D) { _SET(0x02, &D); }
OPCODE(SET_2_D) { _SET(0x04, &D); }
OPCODE(SET_3_D) { _SET(0x08, &D); }
OPCODE(SET_4_D) { _SET(0x10, &D); }
OPCODE(SET_5_D) { _SET(0x20, &D); }
OPCODE(SET_6_D) { _SET(0x40, &D); }
OPCODE(SET_7_D) { _SET(0x80, &D); }

OPCODE(SET_0_E) { _SET(0x01, &E); }
OPCODE(SET_1_E) { _SET(0x02, &E); }
OPCODE(SET_2_E) { _SET(0x04, &E); }
OPCODE(SET_3_E) { _SET(0x08, &E); }
OPCODE(SET_4_E) { _SET(0x10, &E); }
OPCODE(SET_5_E) { _SET(0x20, &E); }
OPCODE(SET_6_E) { _SET(0x40, &E); }
OPCODE(SET_7_E) { _SET(0x80, &E); }

OPCODE(SET_0_H) { _SET(0x01, &H); }
OPCODE(SET_1_H) { _SET(0x02, &H); }
OPCODE(SET_2_H) { _SET(0x04, &H); }
OPCODE(SET_3_H) { _SET(0x08, &H); }
OPCODE(SET_4_H) { _SET(0x10, &H); }
OPCODE(SET_5_H) { _SET(0x20, &H); }
OPCODE(SET_6_H) { _SET(0x40, &H); }
OPCODE(SET_7_H) { _SET(0x80, &H); }

OPCODE(SET_0_L) { _SET(0x01, &L); }
OPCODE(SET_1_L) { _SET(0x02, &L); }
OPCODE(SET_2_L) { _SET(0x04, &L); }
OPCODE(SET_3_L) { _SET(0x08, &L); }
OPCODE(SET_4_L) { _SET(0x10, &L); }
OPCODE(SET_5_L) { _SET(0x20, &L); }
OPCODE(SET_6_L) { _SET(0x40, &L); }
OPCODE(SET_7_L) { _SET(0x80, &L); }

OPCODE(SET_0_iHL) { _SET(0x01, RwByte(HL)); }
OPCODE(SET_1_iHL) { _SET(0x02, RwByte(HL)); }
OPCODE(SET_2_iHL) { _SET(0x04, RwByte(HL)); }
OPCODE(SET_3_iHL) { _SET(0x08, RwByte(HL)); }
OPCODE(SET_4_iHL) { _SET(0x10, RwByte(HL)); }
OPCODE(SET_5_iHL) { _SET(0x20, RwByte(HL)); }
OPCODE(SET_6_iHL) { _SET(0x40, RwByte(HL)); }
OPCODE(SET_7_iHL) { _SET(0x80, RwByte(HL)); }

OPCODE(SET_0_iIX_NN) { _SET(0x01, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SET_1_iIX_NN) { _SET(0x02, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SET_2_iIX_NN) { _SET(0x04, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SET_3_iIX_NN) { _SET(0x08, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SET_4_iIX_NN) { _SET(0x10, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SET_5_iIX_NN) { _SET(0x20, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SET_6_iIX_NN) { _SET(0x40, RwByte(IX + (char)RdByte(PC - 2))); }
OPCODE(SET_7_iIX_NN) { _SET(0x80, RwByte(IX + (char)RdByte(PC - 2))); }

OPCODE(SET_0_iIY_NN) { _SET(0x01, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(SET_1_iIY_NN) { _SET(0x02, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(SET_2_iIY_NN) { _SET(0x04, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(SET_3_iIY_NN) { _SET(0x08, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(SET_4_iIY_NN) { _SET(0x10, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(SET_5_iIY_NN) { _SET(0x20, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(SET_6_iIY_NN) { _SET(0x40, RwByte(IY + (char)RdByte(PC - 2))); }
OPCODE(SET_7_iIY_NN) { _SET(0x80, RwByte(IY + (char)RdByte(PC - 2))); }

#endif

/**********************************************************************/
#pragma mark OPERATION TABLE

typedef void (*operation_t)(void);

#ifdef Z80

#define OPERATION(INDEX, CODE, JMP, N8080, NZ80, F8080, FZ80) FZ80,

static operation_t operation_DDCB[256] = {
#include "opddcb.h"
};
OPCODE(DDCB_OP) { PC++; (*operation_DDCB[RdByte(PC++)])(); }

static operation_t operation_DD[256] = {
#include "opdd.h"
};
OPCODE(DD_OP) { (*operation_DD[RdByte(PC++)])(); }

static operation_t operation_FDCB[256] = {
#include "opfdcb.h"
};
OPCODE(FDCB_OP) { PC++; (*operation_FDCB[RdByte(PC++)])(); }

static operation_t operation_FD[256] = {
#include "opfd.h"
};
OPCODE(FD_OP) { (*operation_FD[RdByte(PC++)])(); }

static operation_t operation_CB[256] = {
#include "opcb.h"
};
OPCODE(CB_OP) { (*operation_CB[RdByte(PC++)])(); }

static operation_t operation_ED[256] = {
#include "oped.h"
};
OPCODE(ED_OP) { (*operation_ED[RdByte(PC++)])(); }

static operation_t operation[256] = {
#include "op.h"
};

#else

#define OPERATION(INDEX, CODE, JMP, N8080, NZ80, F8080, FZ80) F8080,

static operation_t operation[256] = {
#include "op.h"
};

#endif

void Cpu(void)
{

#ifdef CALCULATE_TABLES
	CalculateTables();
#endif

	while (!GetSystemFlags() || !MonitorFlags(&gCpuState))
		(*operation[RdByte(PC++)])();

}
