/* dir.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include "stdio.h"
#include <string.h>
#include <stdlib.h>
#include "dir.h"

/* DirOpenPath() opens a directory using a search path. */
DirPtr DirOpenPath(const char *dir)
{
	char dirName[256];
	DirPtr dirPtr;
	int i;

	/* Search for the first instance of the directory. */
	dirPtr = 0;
	for (i = 0; gFOpenPath[i] != 0; i++) {
		sprintf(dirName, "%s%s", gFOpenPath[i], dir);
		if ((dirPtr = DirOpen(dirName)) != 0)
			break;
	}
	if (dirPtr == 0)
		dirPtr = DirOpen(dir);

	/* All done, no error, return pointer to DIR structure. */
	/* Return zero if there was an error. */
	return dirPtr;

}

/**********************************************************************/
#pragma mark *** MACINTOSH DIRECTORY ***

#ifdef MACINTOSH

#include <Files.h>

#pragma mark struct Dir
struct Dir {
	char path[kSizeofFilePath];
	struct DirEntry dirEntry;
	long ioDirID;
	int nextFile;
	short vRefNum;
};

extern unsigned char *CStrToPStr(unsigned char *p, const char *c);
extern char *PStrToCStr(char *c, const unsigned char *p);


/*
 * MacOS uses ':' to seperate path components.
 * An initial ':' means "start at the current directory".
 * No initial ':' means "start at the desktop".
 */

/* DirOpen() opens a directory, returns zero if error. */
DirPtr DirOpen(const char *path)
{
	VolumeParam volumeParam;
	CInfoPBRec cInfoPBRec;
	HFileInfo *hFileInfo = (HFileInfo*)&cInfoPBRec;
	DirInfo *dirInfo = (DirInfo*)&cInfoPBRec;
	DirPtr dirPtr = 0;
	Str255 name;
	short currentVRefNum;
	short vRefNum;
	unsigned i;
	OSErr err;

	/* Allocate the Dir structure. */
	if ((dirPtr = calloc(1, sizeof(Dir))) == 0)
		goto error;

	/* Remember the path. */
	strcpy(dirPtr->path, path);

	/* Get the vRefNum for the current volume. */
	volumeParam.ioCompletion = 0;
	volumeParam.ioNamePtr = 0;
	volumeParam.ioVRefNum = 0;
	if (PBGetVol((ParmBlkPtr)&volumeParam, 0))
		goto error;
	currentVRefNum = volumeParam.ioVRefNum;

	/* if the path contains an initial ':', use the current volume. */
	if (*path == ':')
		vRefNum = currentVRefNum;
	/* Otherwise, use the specified volume. */
	else {
		/* Extract the volume name from the path. */
		CStrToPStr(name, path);
		for (i = 1; i <= *name; i++) {
			if (name[i] == ':') {
				*name = i - 1;
				break;
			}
			path++;
		}
		*name += 1;
		name[*name] = ':';
		/* Get the vRefNum for the specified volume. */
		volumeParam.ioNamePtr = name;
		volumeParam.ioVRefNum = 0;
		if (PBSetVol((ParmBlkPtr)&volumeParam, 0))
			goto error;
		err = PBGetVol((ParmBlkPtr)&volumeParam, 0);
		vRefNum = volumeParam.ioVRefNum;
		volumeParam.ioNamePtr = 0;
		volumeParam.ioVRefNum = currentVRefNum;
		if (PBSetVol((ParmBlkPtr)&volumeParam, 0) || err)
			goto error;;
	}

	/* Get the catalog information for the path. */
	CStrToPStr(name, path);
	hFileInfo->ioNamePtr = name;
	hFileInfo->ioVRefNum = vRefNum;
	hFileInfo->ioFDirIndex = 0;
	hFileInfo->ioFVersNum = 0;
	dirInfo->ioDrDirID = 0;
	if (PBGetCatInfoSync(&cInfoPBRec) != 0)
		goto error;

	/* error if it is not a directory */
	if ((hFileInfo->ioFlAttrib & 0x10) == 0)
		goto error;

	/* setup the DIR structure */
	dirPtr->ioDirID = hFileInfo->ioDirID;
	dirPtr->nextFile = 1;
	dirPtr->vRefNum = vRefNum;

	/* All done, no error, return pointer to DIR structure. */
	return dirPtr;

	/* Return zero if there was an error. */
error:
	if (dirPtr != 0)
		free(dirPtr);
	return 0;

}

/* DirClose() closes a directory, returns non-zero if error. */
int DirClose(DirPtr dirPtr)
{

	if (dirPtr != 0)
		free(dirPtr);

	/* All done, no error, return zero. */
	return 0;

}

/* DirRead() reads the next directory entry, returns 0 if error */
DirEntryPtr DirRead(DirPtr dirPtr)
{
	DirEntryPtr dirEntryPtr = &dirPtr->dirEntry;
	CInfoPBRec cInfoPBRec;
	HFileInfo *hFileInfo = (HFileInfo*)&cInfoPBRec;

	/* Get the catalog information for the next file entry. */
	hFileInfo->ioNamePtr = (unsigned char*)dirEntryPtr->name;
	hFileInfo->ioVRefNum = dirPtr->vRefNum;
	hFileInfo->ioFDirIndex = dirPtr->nextFile++;
	hFileInfo->ioFVersNum = 0;
	hFileInfo->ioDirID = dirPtr->ioDirID;
	if (PBGetCatInfoSync(&cInfoPBRec))
		goto error;

	/* Prepare the file name. */
	PStrToCStr(dirEntryPtr->name, (unsigned char *)dirEntryPtr->name);

	/* Prepare the file path prefix. */
	sprintf(dirEntryPtr->pathPrefix, "%s:", dirPtr->path);

	/* Is this a regular file? */
	dirEntryPtr->isFile = ((hFileInfo->ioFlAttrib & 0x10) == 0);

	/* Set the file size. */
	dirEntryPtr->size = dirEntryPtr->isFile ? hFileInfo->ioFlLgLen : 0;

	/* Return a pointer to the DirEntry. */
	return dirEntryPtr;

	/* Return zero if there was an error. */
error:
	return 0;

}

#endif

/**********************************************************************/
#pragma mark *** DOS DIRECTORY ***

#ifdef DOS

#include <io.h>
#include <direct.h>

#pragma mark struct Dir
struct Dir {
	char path[kSizeofFilePath];
	struct DirEntry dirEntry;
	struct _finddata_t fileInfo;
	long handle;
	short status;
};

/*
 * MSDOS uses '\' to seperate path components.
 * Paths may begin with a drive specification.
 */

/* DirOpen() opens a directory, returns zero if error. */
DirPtr DirOpen(const char *path)
{
	char cwd[128];
	char dir[256];
	DirPtr dirPtr;

	/* Allocate the Dir structure. */
	if ((dirPtr = calloc(1, sizeof(Dir))) == 0)
		goto error;

	/* Remember the path. */
	strcpy(dirPtr->path, path);

	/* Get the current working directory and drive specification. */
	if (path[1] == ':') {
		_getdcwd(toupper(path[0]) - 'A' + 1, cwd, sizeof(cwd));
		path += 2;
	}
	else
		_getcwd(cwd, sizeof(cwd));

	/* If the path is "rooted" prepend the drive specification */
	/* otherwise fill in the entire cwd specification. */
	if (*path == '\\')
		sprintf(dir, "%c:%s\\*.*", cwd[0], path);
	else
		sprintf(dir, "%s%s%s\\*.*", cwd, path ? "\\" : "", path);

	/* Get directory information. */
	if ((dirPtr->handle = _findfirst(dir, &dirPtr->fileInfo)) == -1)
		goto error;

	/* All done, no error, return pointer to DIR structure. */
	return dirPtr;

	/* Return zero if there was an error. */
error:
	if (dirPtr != 0)
		free(dirPtr);
	return 0;

}

/* DirClose() closes a directory, returns non-zero if error. */
int DirClose(DirPtr dirPtr)
{

	if (dirPtr != 0)
		free(dirPtr);

	/* All done, no error, return zero. */
	return 0;

}

/* DirRead() reads the next directory entry, returns 0 if error */
DirEntryPtr DirRead(DirPtr dirPtr)
{
	DirEntryPtr dirEntryPtr = &dirPtr->dirEntry;

	/* Error if non-zero status. */
	if (dirPtr->status)
		goto error;

	/* Prepare the file name. */
	strcpy(dirEntryPtr->name, dirPtr->fileInfo.name);

	/* Prepare the file path prefix. */
	sprintf(dirEntryPtr->pathPrefix, "%s\\", dirPtr->path);

	/* Is this a regular file? */
	dirEntryPtr->isFile = !(dirPtr->fileInfo.attrib & _A_SUBDIR);

	/* Set the file size. */
	dirEntryPtr->size = dirEntryPtr->isFile ? dirPtr->fileInfo.size : 0;

	/* Get next directory entry. */
	dirPtr->status = _findnext(dirPtr->handle, &dirPtr->fileInfo);

	/* Return a pointer to the DirEntry. */
	return dirEntryPtr;

	/* Return zero if there was an error. */
error:
	return 0;

}

#endif

/**********************************************************************/
#pragma mark *** BSD DIRECTORY ***

#ifdef BSD

#include <sys/types.h>
#include <sys/dir.h>
#include <sys/stat.h>

#pragma mark struct Dir
struct Dir {
	char path[kSizeofFilePath];
	struct DirEntry dirEntry;
	DIR *dirPtr;
};

/*
 * MSDOS uses '\' to seperate path components.
 * Paths may begin with a drive specification.
 */

/* DirOpen() opens a directory, returns zero if error. */
DirPtr DirOpen(const char *path)
{
	char cwd[128];
	char dir[256];
	DirPtr dirPtr;

	/* Allocate the Dir structure. */
	if ((dirPtr = calloc(1, sizeof(Dir))) == 0)
		goto error;

	/* Remember the path. */
	strcpy(dirPtr->path, path);

	/* Get directory information. */
	if ((dirPtr->dirPtr = opendir(path)) == 0)
		goto error;

	/* All done, no error, return pointer to DIR structure. */
	return dirPtr;

	/* Return zero if there was an error. */
error:
	if (dirPtr != 0)
		free(dirPtr);
	return 0;

}

/* DirClose() closes a directory, returns non-zero if error. */
int DirClose(DirPtr dirPtr)
{

	if (dirPtr != 0) {
		if (dirPtr->dirPtr != 0)
			closedir(dirPtr->dirPtr);
		free(dirPtr);
	}

	/* All done, no error, return zero. */
	return 0;

}

/* DirRead() reads the next directory entry, returns 0 if error */
DirEntryPtr DirRead(DirPtr dirPtr)
{
	DirEntryPtr dirEntryPtr = &dirPtr->dirEntry;
	struct direct *directPtr;
	struct stat statStruct;

	/* Get the directory entry. */
	if ((directPtr = readdir(dirPtr->dirPtr)) == 0)
		goto error;

	/* Prepare the file name. */
	strcpy(dirEntryPtr->name, directPtr->d_name);

	/* Prepare the file path prefix. */
	sprintf(dirEntryPtr->pathPrefix, "%s/", dirPtr->path);

	/* Get the file information. */
	if (stat(dirPtr->path, &statStruct) != 0)
		goto error;

	/* Is this a regular file? */
	dirEntryPtr->isFile = (statStruct.st_mode & S_IFREG) != 0;

	/* Set the file size. */
	dirEntryPtr->size = dirEntryPtr->isFile ? statStruct.st_size : 0;

	/* Return a pointer to the DirEntry. */
	return dirEntryPtr;

	/* Return zero if there was an error. */
error:
	return 0;

}

#endif
