/* 
 * sprof_pt.c - sprof ptrace
 * MV, 4.6.1996 (23.5.1996 13.5.1996)
 *
 */

/*
 * To compile:  cc -W2 -O2 sprof_pt.c -c sprof_pt.o
 */

/*
 * sprof_pt: symbol prof ptrace
 *
 */

/* #define _POSIX_SOURCE */

#include <stdio.h>	/* fprintf() */
#include <stdlib.h>	/* atexit() */
#include <unistd.h>	/* getopt() */

#include <errno.h>	/* EEXIST */

#include <signal.h>  	/* SIGALRM, SIGTRAP */

#include <sys/ptrace.h>	/* arguments for ptrace() */

#ifdef LINUX 
#include <asm/ptrace.h> /* Linux 2.4: EIP */
#endif /* LINUX */

#include <sys/wait.h>	/* wstat... */
#include <errno.h>      /* errno */

#include "general.h"
#include "util.h"       /* for struct ucontext, if necessary */

/* #define USE_CALARM */ 	/* use child alarm ? */

/* for BSD: <machine/reg.h>, for AIX, HPUX: <sys/reg.h> */
#ifdef PTRACE_INCLUDE	/* special include to get register names? */
#include PTRACE_INCLUDE
#endif


#ifndef USE_CALARM
#define CSTOPSIG SIGPROF	/* SIGALRM */
#endif	/* USE_CALARM */

static pid_t child_pid;



#define IGNORED 0

static ad_t get_pc(void) {
  register int pc_i;
  /* request 3: read user registers */
  if ((pc_i = PTRACE(PTRACE_REQ_RD_USER, child_pid,  PTRACE_PC, IGNORED)) == -1) {
    perror("ptrace(READ_USER)");
    return (0);
  }
#ifdef DEBUG
  printf("get_pc:pc=%08x\n", (size_t)pc_i);
  printf("pc=%08x\n", (size_t)pc_i);
#endif
  return((ad_t)pc_i);
}


/* ****************************************************************************
 * sprof_alsig() - sprof interval timer signal handler
 *
 * Description:
 *   This signal handler is called, if an interval timer has expired.
 *
 * Parameters:
 *   int signum			signal number (not used)
 *   struct siginfo *si		signal info (not used, so  vovid *si)
 *   struct ucontext *uc	context information with machine context
 *
 * Return values:
 *   -
 *
 * ****************************************************************************
 */
static void sprof_alsig(SIGHAND_PARA) {
#ifndef USE_CALARM
  register int save_errno = errno;       /* save original errno */
#ifdef DEBUG
  fprintf(stderr, "SIGALARM. Sending signal CSTOPSIG (SIGPROF) to child...\n");
#endif

#ifdef DEBUG
  {
    size_t pc_i = (int)get_pc();
    printf("pc=%08x\n", pc_i);
  }
#endif

  if (kill(child_pid, CSTOPSIG) < 0) {
    perror("kill");
  }
  errno = save_errno;   /* restore errno */
#endif	/* not use CALARM */
}


/* Use our own SPROF_GET_PC ... */
#define SPROF_SIGHAND sprof_alsig
#define SPROF_GET_PROFTKS  real_ticks
/* (tbuf.tms_cutime + tbuf.tms_cstime) is empty until child terminates. */
#define SPROF_GET_PC (ad_t)get_pc()
#define SPROF_CREATOR "sprof_pt"
#include "sprof_lib.c"	/* INCLUDE THE LIBRARY */



#ifdef USE_CALARM
static ErrCode do_child(char *a[], int tick_rate_us) {
#else
static ErrCode do_child(char *a[]) {
#endif
  /* request 0: enable tracing on this process */
  (void)PTRACE(PTRACE_REQ_TRACEME, 0, 0, 0);
#ifdef DEBUG
  printf("child: %s before exec.\n", a[0]);
#endif

#ifdef USE_CALARM
    if (util_setitim(tick_rate_us) != OK) { return ERROR; }
#endif /* USE_CALARM */

  return(execv(a[0], a));  /* does never return, only on error */
}

#ifdef DEBUG
static char *sig_txt[32] = { 
 "NOSIG", "SIGHUP", "SIGINT", "SIGQUIT", "SIGILL",
 "SIGTRAP", "SIGABRT", "SIGEMT", "SIGFPE", "SIGKILL", "SIGBUS", "SIGSEGV",
 "SIGSYS", "SIGPIPE", "SIGALRM", "SIGTERM", "SIGUSR1", "SIGUSR2", 
 "SIGCHLD", "SIGPWR", "SIGWINCH", "SIGURG", "SIGPOLL", "SIGSTOP",
 "SIGTSTP", "SIGCONT", "SIGTTIN", "SIGTTOU", "SIGVTALRM", "SIGPROF", "SIGXCPU",
 "SIGXFSZ"
};
#endif


static void my_exit_func(void) {
  if (sprof_stop() != OK) {
    fprintf(stderr, "sprof_stop() failed.\n");
  }
}

static ErrCode do_ptrace(const int argc, char **argv, const int arg_num) {
  pid_t cpid;
#ifdef USE_CALARM
  int util_sig = 0;
  int pipe_fd[2];	/* we need a pipe to send the child the tick_rate */
  int tick_rate_us = 0;
  if (pipe(pipe_fd) < 0) {
    perror("pipe");
    return ERROR;
  }
#endif /* USE_CALARM */
  /* first, check, if file exist. We do not need the time assertion, so use
   * the same file.
   */
  if (util_check_stat(argv[arg_num], argv[arg_num]) != OK) {
    return ERROR;
  }

#ifndef USE_CALARM
  if (util_set_timtype(0) != OK)  return ERROR;
    /* we can only use ITIMER_REAL in the parent process! */
#endif /* USE_CALARM */

  if ((cpid = fork()) < 0) {
    perror(argv[0]);
    exit(1);
  } else if (cpid == 0) {	/* we're the child... */
#ifdef USE_CALARM
    close(pipe_fd[1]);	/* child: close write */
    if (read(pipe_fd[0], &tick_rate_us, sizeof(tick_rate_us)) != sizeof(tick_rate_us)) {
      fprintf(stderr, "Error: Child cannot read pipe.\n");
      close(pipe_fd[0]);
      return ERROR;
    }
    close(pipe_fd[0]);
    printf("child: received tick_rate_us=%d\n", tick_rate_us);
    return(do_child(&(argv[arg_num]), tick_rate_us));
#else /* USE_CALARM */
    return(do_child(&(argv[arg_num])));
#endif /* USE_CALARM */
  }

  if (atexit(my_exit_func) != OK) {
    perror("atexit");
    return ERROR;
  }

#ifdef USE_CALARM
  close(pipe_fd[0]);	/* parent: close read */
  if (sprof_start(argc - arg_num, &(argv[arg_num])) != OK) {
    fprintf(stderr, "sprof_start failed.\n");
    return ERROR;
  }
  if (sprof_inactivate() != OK)  return ERROR;
    /* we do not need timers and signals in the parent. */
  tick_rate_us = g_sp.tick_rate_us;
  write(pipe_fd[1], &tick_rate_us, sizeof(tick_rate_us)); /* send to child */
  close(pipe_fd[1]);
  (void)util_get_timtype(&util_sig);
#endif /* USE_CALARM */

  /* wait until child did exec... */
  {
    pid_t pid;
    int stat_loc;
    if ((pid = wait(&stat_loc)) < 0) {
      perror("wait");
      return ERROR;
    }
    /* did child send SIGTRAP? */
    if (WIFSTOPPED(stat_loc) && (WSTOPSIG(stat_loc) == SIGTRAP)) {
      child_pid = cpid;
#ifndef USE_CALARM
      if (sprof_start(argc - arg_num, &(argv[arg_num])) != OK) {
        fprintf(stderr, "sprof_start failed.\n");
        return ERROR;
      }
#endif /* !USE_CALARM */
    } else {
      fprintf(stderr, "Wait: Unexpected state/signal %d\n", WSTOPSIG(stat_loc));
      fprintf(stderr, "Child did not start properly.\n");
      return ERROR;
    }

    /* request 7: continue, possibly with signal */
    if (PTRACE(PTRACE_REQ_CONTINUE, pid, 1, 0) < 0) {
      perror("ptrace(CONTINUE)");
      return ERROR;
    }      
  }

  /* now start the measurement... */
  {
    pid_t pid = 0;
    int stat_loc;
    int sig = 0;
    do {
      if ((pid = wait(&stat_loc)) < 0) {
        if (errno == EINTR) {
          continue;	/* interrupted wait -> continue with wait */
        } else {
          perror("wait");
          return ERROR;
        }
      }
      /* printf("main: stat_loc=%d=%08x\n", stat_loc, (uint)stat_loc); */

      if (WIFSTOPPED(stat_loc)) {
#ifdef DEBUG
        printf("main: WIFSTOPPED: WSTOPSIG=%d (%s)\n", WSTOPSIG(stat_loc),
          sig_txt[WSTOPSIG(stat_loc)]);
#endif
        sig = WSTOPSIG(stat_loc);
#ifdef USE_CALARM
        if (sig == util_sig) {
#else /* USE_CALARM */
        if (sig == CSTOPSIG) {
#endif /* USE_CALARM */
#ifdef DEBUG
          printf("pid=%d, sig=%d: ", (int)pid, sig);
#endif
#ifdef UCONTEXT_GET_PC
          sprof_sighand(sig, NULL, NULL); /* call the standard signal handler */
#else
	  sprof_sighand(sig);
#endif /* UCONTEXT_GET_PC */
          sig = 0;
        }
      } else if (WIFEXITED(stat_loc)) {
        if (util_sigign() != OK) { }	/* ignore further signals */
        /* printf("main: WIFEXITED: WEXITSTATUS=%d\n", WEXITSTATUS(stat_loc)); */
        return (WEXITSTATUS(stat_loc));
      } else if (WIFSIGNALED(stat_loc)) {
        printf("main: WIFSIGNALED: WTERMSIG=%d\n", WTERMSIG(stat_loc));
        return ERROR;
      }

      /* request 7: continue, possibly with signal */
      if (PTRACE(PTRACE_REQ_CONTINUE, pid, 1, sig) < 0) {
        perror("ptrace(CONTINUE)");
        return ERROR;
      }      
    } while(1);
  }
}



/* ****************************
 *  read options, usage
 * **************************** */

static void usage(const char *pname) {
  fprintf(stderr, "%s %s -- Symbol Profiling (ptrace)\n", "SPROF_PT", "1.2");
  fprintf(stderr, "Marco Vieth, 4.10.1996\n");
  fprintf(stderr, "Usage: %s [sprof_options] program [prg options]\n", pname);
  fprintf(stderr, "[sprof_options]\n");
  fprintf(stderr, "\t-h\t help\n");
  fprintf(stderr, "\t-d\t debug\n");
}


struct option_s {
  int last_index;
  int debug_f;
};


static int read_opt(const int argc, char **argv, struct option_s *opts) {
  int n;
  int acnt;

  /* needed for Linux: find the first non-option element, so we can limit the */
  /* getopt() analysis */
  {
    int i;
    for (i = 1; i < argc; i++) {
      if (argv[i][0] != '-')  break;
    }
    acnt = i;
    opts->last_index = i;
  }
  while ((n = getopt(acnt, argv, "dh")) != EOF) {
    switch (n) {
      case 'd':         /* debug output */
        opts->debug_f = 1; /* atoi(optarg); */
      break;

      case '?':
      case 'h': 
        usage(argv[0]);
        return ERROR;
      /* break; */
      default:
      break;
    }
  }
  /* opts->last_index = optind; */
  return OK;
}


/* ****************************
 *  main
 * **************************** */


int main(const int argc, char **argv) {
  int rc = OK;
  struct option_s opts = { 0, 0 };

  if (read_opt(argc, argv, &opts) != OK)  return ERROR;

  if (opts.last_index >= argc) {
    usage(argv[0]);
    return ERROR;
  }
  if (opts.debug_f > 0) {
    fprintf(stderr, "Debug flag set.\n");
  }
  rc = do_ptrace(argc, argv, opts.last_index);
  return (rc);
}
/* End */
