/*
 * evcnt_ioctl.c -- Evcnt Device (ioctl cmd part)
 *
 * Version 0.2
 *
 * Copyright (C) 1996 by Marco Vieth
 *
 * 01.07.2000 0.21 MV  adapted for kernel 2.2x (get_fs_long, put_fs_long -> copy_from_user, copy_to_user)
 *
 */

/* This part is included from evcnt_ix86.c and prf_ix86.c */

#include "evcnt_feat_ix86.h"

#include <linux/version.h>

#define LINUX_V_NEWFS 0x020104  /* kernel version when 'file_operations' have been changed, maybe earlier */

#if LINUX_VERSION_CODE >= 0x020104

#include <asm/uaccess.h> /* for verify_area(), Linux kernel 2.2x */

#define get_fs_long(from)  ( {long to; copy_from_user(&to, from, sizeof(long)); to; })
#define put_fs_long(from, to) ({long f=from; copy_to_user(to, &f, sizeof(long)); })
/* #define memcpy_fromfs(to, from, n)  copy_from_user(to, from, n) */
/* #define memcpy_tofs(to, from, n)  copy_to_user(to, from, n) */
  /*
   * It would be a good idea to replace the old kernel functions
   * put_fs_long, get_fs_long, memcpy_tofs, memcpy_fromfs, verify_area
   * by the new kernel functions 
   * copy_to_user, copy_from_user, put_user, get_user.
   * These functions already do a verify_area()!
   * MV, 01.07.2000
   */

#endif /* LINUX_VERSION_CODE */


static int evcnt_feature_ioctl(unsigned int cmd, unsigned long arg) {
  register int rc = 0;
  register hint *arg_p = (hint *)arg;
  register hint tmp;

  switch (cmd) {
    case EV_GETTSC:
      if ((rc = verify_area(VERIFY_WRITE, (void *)arg_p, sizeof(hint))) != 0) {
        return (rc);
      }
      read_tsc(tmp.lo, tmp.hi);
      put_fs_long(tmp.lo, &arg_p->lo);
      put_fs_long(tmp.hi, &arg_p->hi);
      return (0);
    break;

#if defined(I586_EVCNT) || defined(I586MMX_EVCNT)
    case EV_GETFEATURE:
      if ((rc = verify_area(VERIFY_WRITE, (void *)arg_p, sizeof(hint))) != 0) {
        return (rc);
      }
      read_msr(tmp.lo, tmp.hi, MSR_FEATURE_R);
      tmp.lo &= (EV_FEA_BTB_DI | EV_FEA_VPIPE_DI | EV_FEA_L1_DI);
      tmp.hi = 0;
      put_fs_long(tmp.lo, &arg_p->lo);
      put_fs_long(tmp.hi, &arg_p->hi);
    break;

    case EV_SETFEATURE:
      if ((rc = verify_area(VERIFY_READ, (void *)arg_p, sizeof(hint))) != 0) {
        return (rc);
      }
      {
        unsigned long lo = get_fs_long(&arg_p->lo);
        read_msr(tmp.lo, tmp.hi, MSR_FEATURE_R);
        tmp.lo &= ~(EV_FEA_BTB_DI | EV_FEA_VPIPE_DI | EV_FEA_L1_DI);
        tmp.lo |= (lo & (EV_FEA_BTB_DI | EV_FEA_VPIPE_DI | EV_FEA_L1_DI));
        write_msr(tmp.lo, tmp.hi, MSR_FEATURE_R);
      }
      return (0);
    break;
#endif /* I586_EVCNT */

#if defined(I686_EVCNT) || defined(I686MMX_EVCNT) || defined(ATHLON_EVCNT)
    case EV_GETFEATURE:	/* not implemented/available on i686/Athlon */
    case EV_SETFEATURE:
      return -EINVAL;
    break;
#endif /* I686_EVCNT */

    case EV_GETMSR:
      if ((rc = verify_area(VERIFY_WRITE, (void *)arg_p, sizeof(hint[2]))) != 0) {
        return (rc);
      }
      {
        register unsigned long msr_reg = get_fs_long(&arg_p->lo);
        if (msr_reg >= EVCNT_MSR_NUM) {
          return -EINVAL;
        }
        arg_p++;	/* h[1] */
        read_msr(tmp.lo, tmp.hi, msr_reg);
        put_fs_long(tmp.lo, &arg_p->lo);
        put_fs_long(tmp.hi, &arg_p->hi);
      }
      return (0);
    break;

    case EV_SETMSR:
      if ((rc = verify_area(VERIFY_READ, (void *)arg_p, sizeof(hint[2]))) != 0) {
        return (rc);
      }
      {
        register unsigned long msr_reg = get_fs_long(&arg_p->lo);
        if (msr_reg >= EVCNT_MSR_NUM) {
          return -EINVAL;
        }
        arg_p++;	/* h[1] */
        tmp.lo = get_fs_long(&arg_p->lo);
        tmp.hi = get_fs_long(&arg_p->hi);
        write_msr(tmp.lo, tmp.hi, msr_reg);
      }
      return (0);
    break;

    default:
      rc = -EINVAL;
    break;    
  }
  return (rc);
}

#ifdef EVCNT_CYCNT
static hint tsc_base = {0, 0};	/* time stamp counter base */
#endif /* EVCNT_CYCNT */

static int evcnt_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
  unsigned long arg) {
  register int rc = 0;

#ifdef EVCNT_DEBUG
  printk("evcnt_ioctl, cmd: 0x%x, arg: 0x%x\n", cmd, arg);
#endif

  if (MINOR(inode->i_rdev) != EVCNT_MINOR) {
    return -ENODEV;	/* wrong minor number */
  }

  switch (cmd) {
    case EV_GETCNTS:
      {
        register hint *arg_p = (hint *)arg;	/* h[0] */
        register hint tmp;
        if ((rc = verify_area(VERIFY_WRITE, (void *)arg_p, sizeof(hint[EVCNT_CNUM]))) != 0) {
          return (rc);
        }
        read_msr(tmp.lo, tmp.hi, MSR_CNT0_R);
#if defined(I686_EVCNT) || defined(I686MMX_EVCNT)
        tmp.hi &= 0xff;	/* mask out b40..b63 */
#endif /* I686_EVCNT */
#if defined(ATHLON_EVCNT)
        tmp.hi &= 0xffff; /* mask out b49..b63 */
#endif /* ATHLON_EVCNT */

        /* printk("EV_GETCNTS[0]: lo=%ld, hi=%ld\n", tmp.lo, tmp.hi); */
        put_fs_long(tmp.lo, &arg_p->lo);
        put_fs_long(tmp.hi, &arg_p->hi);
        read_msr(tmp.lo, tmp.hi, MSR_CNT1_R);
        arg_p++;	/* h[1] */
#if defined(I686_EVCNT) || defined(I686MMX_EVCNT)
        tmp.hi &= 0xff;	/* mask out b40..b63 */
#endif /* I686_EVCNT */
#if defined(ATHLON_EVCNT)
        tmp.hi &= 0xffff; /* mask out b49..b63 */
#endif /* ATHLON_EVCNT */
        /* printk("EV_GETCNTS[1]: lo=%ld, hi=%ld\n", tmp.lo, tmp.hi); */
        put_fs_long(tmp.lo, &arg_p->lo);
        put_fs_long(tmp.hi, &arg_p->hi);
#ifdef EVCNT_CYCNT
        arg_p++;	/* h[2] */
        read_tsc(tmp.lo, tmp.hi);
        if (tmp.lo >= tsc_base.lo) {
          tmp.lo -= tsc_base.lo;
        } else {
          tmp.lo -= tsc_base.lo - ULONG_MAX;
          tmp.hi--;
        }
        tmp.hi -= tsc_base.hi;
        put_fs_long(tmp.lo, &arg_p->lo);
        put_fs_long(tmp.hi, &arg_p->hi);
#endif /* EVCNT_CYCNT */
        return (0);
      }
    break;

    case EV_SETCNTS:
      {
        register hint *arg_p = (hint *)arg;	/* h[0] */
        register hint tmp;
        if ((rc = verify_area(VERIFY_READ, (void *)arg_p, sizeof(hint[EVCNT_CNUM]))) != 0) {
          return (rc);
        }
        tmp.lo = get_fs_long(&arg_p->lo);
        tmp.hi = get_fs_long(&arg_p->hi);
        /* printk("EV_SETCNTS[0]: lo=%ld, hi=%ld\n", tmp.lo, tmp.hi); */
        write_msr(tmp.lo, tmp.hi, MSR_CNT0_R);
        arg_p++;	/* h[1] */
        tmp.lo = get_fs_long(&arg_p->lo);
        tmp.hi = get_fs_long(&arg_p->hi);
        /* printk("EV_SETCNTS[1]: lo=%ld, hi=%ld\n", tmp.lo, tmp.hi); */
        write_msr(tmp.lo, tmp.hi, MSR_CNT1_R);
#ifdef EVCNT_CYCNT
        /* arg_p++; h[2] not needed */
        /* and init the cycle counter ... */
        read_tsc(tsc_base.lo, tsc_base.hi);
#endif /* EVCNT_CYCNT */
        return (0);
      }
    break;

#if defined(I586_EVCNT) || defined(I586MMX_EVCNT)
    case EV_SETMODE:
      {
        register hint *arg_p = (hint *)arg;	/* h[0] */
        register hint tmp;
        register hint h;
        if ((rc = verify_area(VERIFY_READ, (void *)arg_p, sizeof(hint))) != 0) {
          return (rc);
        }
        /* set mode for counter 0 ... */
        read_msr(h.lo, h.hi, MSR_EVCTL_R);
        tmp.lo = get_fs_long(&arg_p->lo);	/* mode cnt 0 */
        tmp.hi = (tmp.lo >> 16) & 0xffff;	/* mode control cnt0 */
        tmp.lo &= 0xffff;
        h.lo &= ~(EVCTL_TYPE0 | EVCTL_CPL0 | EVCTL_ES0);
          /* clear counter 0: type, cpl, event (b0..5) */
        h.lo |= (tmp.lo & EVCTL_ES0) | ((tmp.hi << 6) & (EVCTL_TYPE0 | EVCTL_CPL0));
 	  /* set event, cpl,type */

        /* and for counter 1 ... */
        tmp.lo = get_fs_long(&arg_p->hi);	/* mode cnt 1 */
        tmp.hi = (tmp.lo >> 16) & 0xffff;	/* mode control cnt1 */
        tmp.lo &= 0xffff;
        h.lo &= ~(EVCTL_TYPE1 | EVCTL_CPL1 | EVCTL_ES1);
         /* clear counter 1: type, cpl, event */
        h.lo |= ((tmp.lo << 16) & EVCTL_ES1) | ((tmp.hi << 22) & 
          (EVCTL_TYPE1 | EVCTL_CPL1));  /* set event, cpl, type */
        write_msr(h.lo, h.hi, MSR_EVCTL_R);	/* write back */
      }
    break;
#endif /* I586_EVCNT */

#if defined(I686_EVCNT) || defined(I686MMX_EVCNT) || defined(ATHLON_EVCNT)
    case EV_SETMODE:
      {
        register hint *arg_p = (hint *)arg;	/* h[0] */
        register hint tmp;
        register hint h;
        if ((rc = verify_area(VERIFY_READ, (void *)arg_p, sizeof(hint))) != 0) {
          return (rc);
        }

        /* set mode for counter 0 ... */
        /* read_msr(h.lo, h.hi, MSR_EVCTL0_R); */
        h.hi = 0;	/* clear high dword */
        tmp.lo = get_fs_long(&arg_p->lo);  /* mode cnt 0 */
        tmp.hi = (tmp.lo >> 16) & 0xffff;  /* mode control cnt0 */
        tmp.lo &= 0xffff;
        h.lo = 0;  /* clear event select */
        h.lo |= (tmp.lo & (EVCTL_ES | EVCTL_US)) | ((tmp.hi << 16) & (EVCTL_TYPE | EVCTL_CPL)) | EVCTL_EI;
 	/* set event+unit mask, cpl, type, enable counting for both counters */
        write_msr(h.lo, h.hi, MSR_EVCTL0_R);	/* write event select 0 */

        /* and for counter 1 ... */
        /* read_msr(h.lo, h.hi, MSR_EVCTL1_R); */
        tmp.lo = get_fs_long(&arg_p->hi);		/* mode cnt 1 */
        tmp.hi = (tmp.lo >> 16) & 0xffff;	/* mode control cnt1 */
        tmp.lo &= 0xffff;
        h.lo = 0;  /* clear event select */
#if defined(ATHLON_EVCNT)
        h.lo |= (tmp.lo & (EVCTL_ES | EVCTL_US)) | ((tmp.hi << 16) & (EVCTL_TYPE | EVCTL_CPL)) | EVCTL_EI;
#else
        h.lo |= (tmp.lo & (EVCTL_ES | EVCTL_US)) | ((tmp.hi << 16) & (EVCTL_TYPE | EVCTL_CPL));
#endif
        /* set event, cpl, type */
        write_msr(h.lo, h.hi, MSR_EVCTL1_R);	/* write back */
      }
    break;
#endif /* I686_EVCNT */

    case EV_VERSION:
#ifdef I586_EVCNT
      rc = EV_VER_I586 + (1 << 8);
#endif /* I586_EVCNT */
#ifdef I586MMX_EVCNT
      rc = EV_VER_I586MMX + (1 << 8);
#endif /* I586MMX_EVCNT */

#ifdef I686_EVCNT
      rc = EV_VER_I686 + (1 << 8);
#endif /* I686_EVCNT */
#ifdef I686MMX_EVCNT
      rc = EV_VER_I686MMX + (1 << 8);
#endif /* I686MMX_EVCNT */

#ifdef ATHLON_EVCNT
      rc = EV_VER_ATHLON + (1 << 8);
#endif /* ATHLON_EVCNT */
    break;

    case EV_GETTSC:
    case EV_GETFEATURE:
    case EV_SETFEATURE:
    case EV_GETMSR:
    case EV_SETMSR:
      return(evcnt_feature_ioctl(cmd, arg));

    default:
      rc = -EINVAL;
    break;    
  }
  return (rc);
}  

/* end */
