/* ev_test_ix86.c */

/*
 * 19.01.2001 0.22 MV  adapted for AMD Athlon
 *
 */

#include <stdio.h>
#include <stdlib.h>  		/* atoi() */

#include <fcntl.h>              /* O_CREAT */
#include <unistd.h>             /* ioctl() */
#include <sys/ioctl.h>          /* ioctl */

#include "../../../include/general.h"   /* OK, ERROR */


#define TEST_MSR

#include "evcnt_ix86.h"


#include <limits.h>             /* ULONG_MAX */


struct ev_s {
  int fd;
};

static ErrCode ev_close(struct ev_s *evs) {
  if (close(evs->fd) < 0) {
    perror("close()");
    return ERROR;
  }
  return OK;
}

static ErrCode ev_read_cnts(struct ev_s *evs, hint h[EVCNT_CNUM]) {
  if (ioctl(evs->fd, EV_GETCNTS, h) < 0) {	/* get counters */
    perror("ioctl(EV_GETCNTS)");
    return ERROR;
  }
  return OK;
}

static ErrCode ev_write_cnts(struct ev_s *evs, const hint h[EVCNT_CNUM]) {
  if (ioctl(evs->fd, EV_SETCNTS, h) < 0) {	/* set counters */
    perror("ioctl(EV_SETCNTS)");
    return ERROR;
  }
  return OK;
}

static ErrCode ev_set_modes(struct ev_s *evs, const hint *h) {
#ifdef DEBUG
fprintf(stderr, "ev_set_modes(%x): h=%ld,%ld\n", EV_SETMODE, h->lo, h->hi);
#endif
  if (ioctl(evs->fd, EV_SETMODE, h) < 0) { /* set two modes */
    perror("ioctl(EV_SETMODE)");
    return ERROR;
  }
  return OK;
}

static ErrCode ev_open(struct ev_s *evs, const int ev[EVCNT_CNUM]) {
  /* open monitor device */
  hint modes;
  const hint initcnts[EVCNT_CNUM] = {{0}};
  const int type_ticks = 0;

#if defined(I586_EVCNT) || defined(I586MMX_EVCNT)
  const int cpl = 0x03;
  modes.lo = (ev[0] & 0xff) | ((cpl | (type_ticks << 2)) << 16);
  modes.hi = (ev[1] & 0xff) | ((cpl | (type_ticks << 2)) << 16);
#endif /* I586_EVCNT */

#if defined(I686_EVCNT) || defined(I686MMX_EVCNT) || defined(ATHLON_EVCNT)
  const int os_usr = 0x03;
  modes.lo = (ev[0] & 0xff) | ((os_usr | (type_ticks << 2)) << 16);
  modes.hi = (ev[1] & 0xff) | ((os_usr | (type_ticks << 2)) << 16);
#endif /* I686_EVCNT */

  {
    const int open_mode = O_RDONLY; /* O_CREAT; */
    if ((evs->fd = open(EVCNT_DEVICE, open_mode)) < 0) {
      evs->fd = 0;
      perror(EVCNT_DEVICE);
      return ERROR;
    }
    /* set counter modes */
    if (ev_set_modes(evs, &modes) != OK)  return ERROR;
    /* and initialize counters */
    if (ev_write_cnts(evs, initcnts) != OK)  return ERROR;
  }
  return OK;
}



#ifdef TEST_MSR

#include "evcnt_feat_ix86.h"

static ErrCode ev_read_msr(struct ev_s *evs, const int msr, hint *h) {
  hint h_tmp[2];
  h_tmp[0].lo = msr;
  h_tmp[0].hi = 0;
  if (ioctl(evs->fd, EV_GETMSR, h_tmp) < 0) {	/* get msr */
    perror("ioctl(EV_GETMSR)");
    return ERROR;
  }
  h->lo = h_tmp[1].lo;
  h->hi = h_tmp[1].hi;
  return OK;
}

static ErrCode ev_write_msr(struct ev_s *evs, const int msr, const hint *h) {
  hint h_tmp[2];
  h_tmp[0].lo = msr;
  h_tmp[0].hi = 0;
  h_tmp[1].lo = h->lo;
  h_tmp[1].hi = h->hi;
  if (ioctl(evs->fd, EV_SETMSR, h_tmp) < 0) {	/* set msr */
    perror("ioctl(EV_SETMSR)");
    return ERROR;
  }
  return OK;
}

static ErrCode test_msr(struct ev_s *evs) {
  int msr = 0xc1;
  hint h;
  msr = 0xc1;
  if (ev_read_msr(evs, msr, &h) != OK)  { return ERROR; }
  printf("1. read_msr(0x%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
    h.lo + h.hi * (double)ULONG_MAX);
  msr = 0xc2;
  if (ev_read_msr(evs, msr, &h) != OK)  { return ERROR; }
  printf("2. read_msr(0x%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
    h.lo + h.hi * (double)ULONG_MAX);
  msr = 0x186;
  if (ev_read_msr(evs, msr, &h) != OK)  { return ERROR; }
  printf("3. read_msr(0x%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
    h.lo + h.hi * (double)ULONG_MAX);
  msr = 0x187;
  if (ev_read_msr(evs, msr, &h) != OK)  { return ERROR; }
  printf("4. read_msr(0x%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
    h.lo + h.hi * (double)ULONG_MAX);
  msr = 0xc1;
  if (ev_read_msr(evs, msr, &h) != OK)  { return ERROR; }
  h.hi &= 0xff;  /* 0000120000e0503b */
  printf("5. read_msr(0x%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
    h.lo + h.hi * (double)ULONG_MAX);

  {
    hint h_mod;
    h_mod.lo = h.lo; /* 0x186: 0x430043;   0x187: 0x30045 */
    h_mod.hi = h.hi;
    if (ev_write_msr(evs, msr, &h_mod) != OK)  { return ERROR; }
  }
  if (ev_read_msr(evs, msr, &h) != OK)  { return ERROR; }
  printf("2. read_msr(0x%02x): %08lx%08lx (%.0f)\n", msr, h.hi, h.lo,
    h.lo + h.hi * (double)ULONG_MAX);
  return OK;
}
#endif /* TEST_MSR */



static ErrCode do_test(struct ev_s *evs, const int loops) {
  hint h[EVCNT_CNUM];
  register int i;
  /* printf("read event counters\n"); */
  for (i = 0; i < loops; i++) {
    register int j;
    if (ev_read_cnts(evs, h) != OK)  return ERROR;
    for (j = 0; j < EVCNT_CNUM; j++) {
      register double d = (double)h[j].lo + h[j].hi * (double)ULONG_MAX;
      printf("%10.0f ", d);
    }
    printf("\n");
  }
  return OK;
}


/* ****************************
 *  read options, usage
 * **************************** */

#define DEF_LOOPS 10

static void usage(const char *pname) {
  fprintf(stderr, "ev_test_ix86 - Test and change features of ix86, Athlon\n");
  fprintf(stderr, "Marco Vieth, 17.7.1997\n");
  fprintf(stderr, "Usage: %s [-f n]\n", pname);
  fprintf(stderr, "\t-l num\t loops (def. %d)\n", DEF_LOOPS);
  fprintf(stderr, "\t-e e1..\t event list for two counters\n");

  fprintf(stderr, "\t-h\t help\n");
  fprintf(stderr, "\t-d\t debug\n");
  fprintf(stderr, "\n");
  exit(1);  
}


struct option_s {
  int loops;
  char *events;
  int test_msr;
  int debug_f;
};


static int read_opt(int argc, char **argv, struct option_s *opts) {
  int n;
  while ((n = getopt(argc, argv, "de:hl:t")) != EOF) {
    switch (n) {
      case 'd':         /* debug output */
        opts->debug_f = 1;
      break;

      case 'e':         /* events */
        opts->events = optarg;
      break;

      case 'l':         /* loops */
        opts->loops = atoi(optarg);
      break;

      case 't':         /* test msr */
        opts->test_msr = 1;
      break;

      case 'h':  
      case '?':
        usage(argv[0]);
      break;
    }
  }
  return OK;
}


/* ****************************
 *  main
 * **************************** */

#define STAB_BASE 0     /* 0=auto detect base (strtol) */

int main(const int argc, char **argv) {
  struct option_s opts = { DEF_LOOPS, "", 0, 0};
  ErrCode rc = OK;
  struct ev_s evs;
  int ev[EVCNT_CNUM];

  if (read_opt(argc, argv, &opts) != OK)  { }

  {
    char *t = opts.events;
    char *tmp;
    ev[0] = strtol(t, &tmp, STAB_BASE);
    t = tmp + 1;
    if (*tmp != ',') {
      fprintf(stderr, "Warning: Unknown character in event list: %c\n", *tmp);
    }
    ev[1] = strtol(t, &tmp, STAB_BASE);
    ev[2] = -1;		/* cycle counter is always used. */
  }

  if (ev_open(&evs, ev) != OK)  { return ERROR; }
  {
    int i;
    for (i = 0; i < EVCNT_CNUM; i++) {
      if (ev[i] >= 0) {
        printf("  event %2d ", ev[i]);
      } else {
        printf("    cycles ");
      }
    }
    printf("\n");
  }
  if (opts.test_msr > 0) {
#ifdef TEST_MSR
    (void)test_msr(&evs);
#endif /* TEST_MSR */
  } else {
    do_test(&evs, opts.loops);
  }
  if (ev_close(&evs) != OK)  { return ERROR; }
  return (rc);
}
/* end */
