#!/usr/bin/perl -w
# dskinfo1.pl - show info of DSK file
#
# Marco Vieth, 27.11.1999
#

use strict;
use IO;
use Getopt::Std;


$::g_debug = 0;

###


#
# main
#
  my %opts = (
  );
  if (!getopts("s:hd", \%opts) or (@ARGV == 0) or exists($opts{'h'})) {
    # print STDERR "Unknown arg $ARGV[0]\n\n" if @ARGV > 0;
    die "Usage: $0 ...\n";
  }

  $::g_debug = $opts{'d'} || 0;

  my($dsk_file);
  foreach $dsk_file (@ARGV) {
    print "Processing file '$dsk_file'...\n";

    read_dskfile($dsk_file);
  }

  exit;

# end


sub fread_blk($$) {
  my($fh, $size) = @_;
  my($n, $buf);
  if (($n = $fh->read($buf, $size)) != $size) {
    warn "WARNING: fread_block: read $n bytes instead of $size!\n";
    return "";
  }
  if ($::g_debug > 0) {
    printf STDERR "DEBUG: bytes read: '%d', length(buf)='%d'\n", $n, length($buf);
  }
  return \$buf;
}

sub fwrite_blk($$$) {
  my($fh, $bufref, $size) = @_;
  my($n);
  if (($n = $fh->write($$bufref, $size)) != $size) {
    warn "WARNING: fwrite_block: written $n bytes instead of $size!\n";
    return 1;
  }
  if ($::g_debug > 0) {
    printf STDERR "DEBUG: bytes written: '%d', length(buf)='%d'\n", $n, length($$bufref);
  }
  return 0;
}


sub unpack_diskinfo($$) {
  my($inf, $disk_info_r) = @_;
  if ($::g_debug > 0) {
    printf STDERR "DEBUG: disk_info='%d'\n", length($$disk_info_r);
  }
  my(@di) = unpack("A34A14CCv", $$disk_info_r);
  my(@di_txt) = qw(ident creator tracks heads tsize);
  # printf STDERR "DEBUG: di_txt='@di_txt', size=%d\n", $#di_txt;

  $inf->print("Disk Info\n");
  my(%di, $i) = (), 0;
  foreach $_ (@di_txt) {
    $di{$_} = $di[$i++];
    $inf->printf("%-8s  = '%s'\n", ucfirst($_), $di{$_});
  }
  $inf->print("\n");
  $di{"ext"} = ($di{'tsize'} > 0) ? 0 : 1;

  my($tsize_num) = $di{'tracks'} * $di{'heads'};	# number of track sizes
  my(@tsizes);
  if ($di{'ext'} > 0) {
    $inf->print("Extended DSK format!\n");
    @tsizes = unpack("x52C${tsize_num}", $$disk_info_r);	# get HIGH bytes (offset 52)
    @tsizes = map { $_ * 0x100 } @tsizes;			# compute lengths (LOW=0)
  } else {
    @tsizes = split(' ', "$di{'tsize'} " x $tsize_num);		# set all track sizes
  }

  # printf STDERR "DEBUG: tsizes='@tsizes', size=%d\n", $#tsizes;
  $di{'tsizes'} = \@tsizes;

  if ($::g_debug > 0) {
    printf STDERR "DEBUG: tsizes = %s\n", "@{$di{'tsizes'}}";
  }
  $inf->printf("tsizes = %s\n\n", "@{$di{'tsizes'}}");

  return \%di;
}

sub unpack_trackinfo($$) {
  my($inf, $track_info_r) = @_;
  if ($::g_debug > 0) {
    printf STDERR "DEBUG: track_info='%d'\n", length($$track_info_r);
  }
  my(@ti) = unpack("A12x4CCx2CCCC", $$track_info_r);
  my(@ti_txt) = qw(ident track head bps spt gap3 fill);
  # printf STDERR "DEBUG: ti_txt='@ti_txt', size=%d\n", $#ti_txt;

  $inf->print("Track Info\n");
  my(%ti, $i) = (), 0;
  foreach $_ (@ti_txt) {
    $ti{$_} = $ti[$i++];
    $inf->printf("%-8s  = '%s'\n", ucfirst($_), $ti{$_});
  }

  my($sect_num) = $ti{'spt'};	# number of sectors
  my($off1) = 24;		# sectors start with offset 24
  my(@si_txt) = qw(s_trk s_hd s_sec s_bps st1 st2 ssize);
  while ($sect_num-- > 0) {
    my(@sect) = unpack("x${off1}CCCCCCv", $$track_info_r);	# get sector
    $i = 0;
    foreach $_ (@si_txt) {
      # printf "%-8s  = '%s'\n", ucfirst($_), $sect[$i];
      push(@{$ti{$_}}, $sect[$i++]);	# add parameter to array in hash
    }
    $off1 += 8;	# add length
  }

  for ($i = 0; $i < $ti{'spt'}; $i++) {
    $inf->print("Sector: ");
    foreach $_ (@si_txt) {
      $inf->printf("%s='%s' ", $_, @{$ti{$_}}[$i]);
    }
    $inf->print("\n");
  }
  $inf->print("\n");
  return \%ti;
}


sub read_dskfile($) {
  my($dsk_file) = @_;
  my($disk_info_size, $track_info_size) = (0x100, 0x100);

  my($fh) = new IO::File($dsk_file, "r") || (warn("WARNING: Cannot open file '$dsk_file'\n"), return 1);
  binmode($fh) || (warn("WARNING: Cannot set binary mode for '$dsk_file'"), return 1);


  my($inf_file);
  ($inf_file = $dsk_file) =~ s/\.dsk$//i;
  $inf_file .= ".inf";
  if (-r "$inf_file") {
    warn "WARNING: File '$inf_file' does already exist. Overwriting...\n";
  }
  my($inf) = new IO::File($inf_file, "w") || (warn("WARNING: Cannot create file '$inf_file'\n"), return 1);

  $inf->print("DSK file '$dsk_file'\n");

  my($disk_info_r) = fread_blk($fh,  $disk_info_size);
  my($di_r) = unpack_diskinfo($inf, $disk_info_r);
  # printf STDERR "DEBUG: di='%s'\n", "%$di_r";
  my($tracks) = $di_r->{'tracks'};
  my($tsize_idx) = 0;
  while ($tracks-- > 0) {
    my($heads) = $di_r->{'heads'};
    while($heads-- > 0) {
      my($track_info_r) = fread_blk($fh, $track_info_size);
      my($ti_r) = unpack_trackinfo($inf, $track_info_r);
      my($data_r) = fread_blk($fh, @{$di_r->{'tsizes'}}[$tsize_idx] - $track_info_size);
      $inf->printf("track data='%d'\n\n", length($$data_r));
      $tsize_idx++;
    }
  }
  $fh->close;
  $inf->close;
  return 0;
}
