// fcb.cpp
// Revision 16-may-2005


#include "fcb.h"

#include <algorithm>

#include <string.h>

using std::find;
using std::copy;
using std::fill_n;
using std::transform;


namespace {

const unsigned char SPACE= static_cast <unsigned char> (' ');
const unsigned char WILD= static_cast <unsigned char> ('?');

class stripbit7 {
public:
	char operator () (char c)
	{
		return c & 0x7F;
	}
};

inline std::string convert_string (const unsigned char * s, size_t len)
{
	size_t i= static_cast <int> (len);
	while (i > 0 && s [i - 1] == SPACE)
		--i;
	return std::string (reinterpret_cast <const char *> (s), i);
}

inline void assign_string (const std::string & s,
	unsigned char * dest, size_t len)
{
	size_t l= s.size ();
	if (l > len)
		l= len;
	s.copy (reinterpret_cast <char *> (dest), l);
	if (len > l)
		fill_n (dest + l, len - l, SPACE);
}

inline bool all_space (const unsigned char * s, size_t len)
{
	for (size_t i= 0; i < len; ++i)
		if (s [i] != SPACE)
			return false;
	return true;
}

inline bool some_wild_char (const unsigned char * s, size_t len)
{
	for (size_t i= 0; i < len; ++i)
		if (s [i] == WILD)
			return true;
	return false;
}

inline bool all_wild_char (const unsigned char * s, size_t len)
{
	for (size_t i= 0; i < len; ++i)
		if (s [i] != WILD)
			return false;
	return true;
}


} // namespace


FcbFile::FcbFile ()
{
	drivenum= 0;
	fill_n (filename, NAME_LEN, SPACE);
	fill_n (fileext, EXT_LEN, SPACE);
}

FcbFile::FcbFile (const std::string & s)
{
	drivenum= 0;
	setfile (s);
}

FcbFile::FcbFile (const unsigned char * fcbdata)
{
	drivenum= static_cast <size_t> (fcbdata [0] );
	setfile (fcbdata + 1);
}

bool FcbFile::all_blank () const
{
	return all_space (filename, NAME_LEN) &&
		all_space (fileext, EXT_LEN);
}

bool FcbFile::is_wild () const
{
	return some_wild_char (filename, NAME_LEN) ||
		some_wild_char (fileext, EXT_LEN);
}

bool FcbFile::all_wild () const
{
	return all_wild_char (filename, NAME_LEN) &&
		all_wild_char (fileext, EXT_LEN);
}

size_t FcbFile::getdrive () const
{
	return drivenum;
}

void FcbFile::getfile (unsigned char * name) const
{
	copy (filename, filename + NAME_LEN, name);
	copy (fileext, fileext + EXT_LEN, name + NAME_LEN);
}

std::string FcbFile::getfilename () const
{
	return convert_string (filename, NAME_LEN);
}

std::string FcbFile::getfileext () const
{
	return convert_string (fileext, EXT_LEN);
}

std::string FcbFile::getfile () const
{
	const std::string name= getfilename ();
	const std::string ext= getfileext ();
	if (! ext.empty () )
		return name + '.' + ext;
	else
		return name;
}

std::string FcbFile::getfullfile () const
{
	return std::string
		(reinterpret_cast <const char *> (filename), NAME_LEN) +
		'.' +
		std::string
		(reinterpret_cast <const char *> (fileext), EXT_LEN);
}

void FcbFile::setwild ()
{
	fill_n (filename, NAME_LEN, '?');
	fill_n (fileext, EXT_LEN, '?');
}

void FcbFile::setfile (const std::string & s)
{
	std::string file;
	std::string ext;
	const std::string::size_type point= s.find ('.');
	if (point != std::string::npos)
	{
		file= s.substr (0, point);
		ext= s.substr (point + 1);
	}
	else
		file= s;
	setfilename (file);
	setfileext (ext);
}

void FcbFile::setfile (const unsigned char * name)
{
	transform (name, name + NAME_LEN,
		filename, stripbit7 () );
	transform (name + NAME_LEN, name + NAME_LEN + EXT_LEN,
		fileext, stripbit7 () );
}

void FcbFile::setfilename (const std::string & s)
{
	assign_string (s, filename, NAME_LEN);
}

void FcbFile::setfileext (const std::string & s)
{
	assign_string (s, fileext, EXT_LEN);
}

bool operator < (const FcbFile & fcb1, const FcbFile & fcb2)
{
	if (fcb1.drivenum < fcb2.drivenum)
		return true;
	if (fcb1.drivenum > fcb2.drivenum)
		return false;
	int r= memcmp (fcb1.filename, fcb2.filename, FcbFile::NAME_LEN);
	if (r < 0)
		return true;
	if (r > 0)
		return false;
	r= memcmp (fcb1.fileext, fcb2.fileext, FcbFile::EXT_LEN);
	return r < 0;
}

std::ostream & operator << (std::ostream & os, const FcbFile & fcb)
{
	os.write (reinterpret_cast <const char *> (fcb.filename),
		FcbFile::NAME_LEN);
	os.put (' ');
	os.write (reinterpret_cast <const char *> (fcb.fileext),
		FcbFile::EXT_LEN);
	return os;
}


// End of fcb.cpp
