// disasm.cpp
// Revision 26-apr-2005


#ifndef DISASM_WITH_libz80


#include "disasm.h"
#include "cpu.h"

#include <sstream>
using std::ostringstream;
#include <map>

// For debugging.
#include <iostream>
using std::cerr;
using std::endl;
#include <assert.h>
#define ASSERT assert


namespace {


using std::string;

template <class C, size_t N>
size_t dim_array (C (&) [N])
{ return N; }

template <class C, size_t N>
size_t dim_array (const C (&) [N])
{ return N; }


class Disasm {
public:
	Disasm (const Cpu & cpu, z80word pos, std::string & op) :
		cpu (cpu),
		pos (pos),
		op (op)
	{ }
	z80word doit ();
	z80word decode ();
private:
	const Cpu & cpu;
	const z80word pos;
	std::string & op;

	enum InstIXY { InstIX, InstIY };

	static const char * regsimple (byte b);
	static const char * regsimpleIX (byte b);
	static const char * regsimpleIY (byte b);
	static const char * regsimpleIXY (byte b, InstIXY type);
	static const char * nameIXY (InstIXY type);
	static const char * regdouble (byte b);
	static const char * regdoubleq (byte b);
	static const char * condition (byte b);

	typedef z80word (Disasm::*decode_func) (byte b);
	typedef z80word (Disasm::*decode_func_IXY) (byte b, InstIXY type);

	#if 0
	struct idinst_elem {
		byte mask;
		byte pattern;
		decode_func decode;
	};
	#endif
	template <typename FUNC>
	struct idinst_elem_template {
		byte mask;
		byte pattern;
		FUNC decode;
	};
	typedef idinst_elem_template <decode_func> idinst_elem;
	typedef idinst_elem_template <decode_func_IXY> idinst_elem_IXY;

	static const idinst_elem idinst [];
	static const idinst_elem idinstCB [];
	static const idinst_elem idinstED [];
	//static const idinst_elem idinstDD [];
	//static const idinst_elem idinstFD [];
	static const idinst_elem_IXY idinstIXY [];
	//static const idinst_elem idinstDDCB [];
	//static const idinst_elem idinstFDCB [];
	static const idinst_elem_IXY idinstIXYCB [];

	#ifndef NDEBUG
public:
	//static bool checkidinst (const idinst_elem * pidinst, size_t n);
	template <class IDINST_ELEM>
	static bool checkidinst (const IDINST_ELEM * pidinst, size_t n);
	static bool checkidinsttables ();
private:
	#endif

	//static decode_func finddecodefunc
	//	(const idinst_elem * idinst, size_t n,byte b);
	template <typename FUNC>
	FUNC finddecodefunc
		(const idinst_elem_template <FUNC> * idinst, size_t n,byte b);

	z80word decoderelative ();
	void decode_n (z80word des);
	void decode_nn (z80word des);
	void decode_des (z80word des);

	z80word decodeCB (byte b);
	z80word decodeED (byte b);
	z80word decodeDD (byte b);
	z80word decodeFD (byte b);
	//z80word decodeDDCB (byte b);
	//z80word decodeFDCB (byte b);
	z80word decodeIXYCB (byte b, InstIXY type);

	// Grouped as in Zilog's Z80 Family CPU User Manual.

	// 8-bit Load Group.

	z80word decodeLD_r_r (byte b);
	z80word decodeLD_IXYHL_r (byte b, InstIXY type);
	//z80word decodeLD_IYHL_r (byte b);
	z80word decodeLD_r_IXYHL (byte b, InstIXY type);
	//z80word decodeLD_r_IYHL (byte b);
	z80word decodeLD_r_n (byte b);
	z80word decodeLD_IXYHL_n (byte b, InstIXY type);
	//z80word decodeLD_IYHL_n (byte b);
	z80word decodeLD_r__IXY_des_ (byte b, InstIXY type);
	//z80word decodeLD_r__IY_des_ (byte b);
	z80word decodeLD__IXY_des__r (byte b, InstIXY type);
	//z80word decodeLD__IY_des__r (byte b);
	z80word decodeLD__IXY_des__n (byte b, InstIXY type);
	//z80word decodeLD__IY_des__n (byte b);
	z80word decodeLD_A__nn_ (byte b);
	z80word decodeLD__nn__A (byte b);

	// 16-bit Load Group.

	z80word decodeLD_dd_nn (byte b);
	z80word decodeLD_IXY_nn (byte b, InstIXY type);
	//z80word decodeLD_IY_nn (byte b);
	z80word decodeLD_HL__nn_ (byte b);
	z80word decodeLD_dd__nn_ (byte b);
	z80word decodeLD_IXY__nn_ (byte b, InstIXY type);
	//z80word decodeLD_IY__nn_ (byte b);
	z80word decodeLD__nn__HL (byte b);
	z80word decodeLD__nn__dd (byte b);
	z80word decodeLD__nn__IXY (byte b, InstIXY type);
	//z80word decodeLD__nn__IY (byte b);
	z80word decodePUSH (byte b);
	z80word decodePOP (byte b);

	// 8-bit Arithmetic Group.

	z80word decodeADD_A_r (byte b);
	z80word decodeADD_A_n (byte b);
	z80word decodeADD_A__IXY_des_ (byte b, InstIXY type);
	//z80word decodeADD_A__IY_des_ (byte b);
	z80word decodeADD_A_IXYHL (byte b, InstIXY type);

	z80word decodeADC_A_r (byte b);
	z80word decodeADC_A_n (byte b);
	z80word decodeADC_A__IXY_des_ (byte b, InstIXY type);
	//z80word decodeADC_A__IY_des_ (byte b);
	z80word decodeADC_A_IXYHL (byte b, InstIXY type);

	z80word decodeSUB_r (byte b);
	z80word decodeSUB_n (byte b);
	z80word decodeSUB__IXY_des_ (byte b, InstIXY type);
	//z80word decodeSUB__IY_des_ (byte b);
	z80word decodeSUB_IXYHL (byte b, InstIXY type);

	z80word decodeSBC_A_r (byte b);
	z80word decodeSBC_A_n (byte b);
	z80word decodeSBC_A__IXY_des_ (byte b, InstIXY type);
	//z80word decodeSBC_A__IY_des_ (byte b);
	z80word decodeSBC_A_IXYHL (byte b, InstIXY type);

	z80word decodeAND_r (byte b);
	z80word decodeAND_n (byte b);
	z80word decodeAND__IXY_des_ (byte b, InstIXY type);
	//z80word decodeAND__IY_des_ (byte b);
	z80word decodeAND_IXYHL (byte b, InstIXY type);

	z80word decodeOR_r (byte b);
	z80word decodeOR_n (byte b);
	z80word decodeOR__IXY_des_ (byte b, InstIXY type);
	//z80word decodeOR__IY_des_ (byte b);
	z80word decodeOR_IXYHL (byte b, InstIXY type);

	z80word decodeXOR_r (byte b);
	z80word decodeXOR_n (byte b);
	z80word decodeXOR__IXY_des_ (byte b, InstIXY type);
	//z80word decodeXOR__IY_des_ (byte b);
	z80word decodeXOR_IXYHL (byte b, InstIXY type);

	z80word decodeCP_r (byte b);
	z80word decodeCP_n (byte b);
	z80word decodeCP__IXY_des_ (byte b, InstIXY type);
	//z80word decodeCP__IY_des_ (byte b);
	z80word decodeCP_IXYHL (byte b, InstIXY type);

	z80word decodeINC_r (byte b);
	z80word decodeINC__IXY_des_ (byte b, InstIXY type);
	//z80word decodeINC__IY_des_ (byte b);
	z80word decodeINC_IXYHL (byte b, InstIXY type);

	z80word decodeDEC_r (byte b);
	z80word decodeDEC__IXY_des_ (byte b, InstIXY type);
	//z80word decodeDEC__IY_des_ (byte b);
	z80word decodeDEC_IXYHL (byte b, InstIXY type);

	// 16-bit Arithmetic Group.

	z80word decodeADC_HL_dd (byte b);

	z80word decodeADD_HL_dd (byte b);
	z80word decodeADD_IXY_dd (byte b, InstIXY type);
	//z80word decodeADD_IY_dd (byte b);

	z80word decodeSBC_HL_dd (byte b);

	z80word decodeINC_dd (byte b);
	z80word decodeDEC_dd (byte b);

	// Rotate and Shitf Group.

	z80word decodeRLC_r (byte b);
	z80word decodeRLC__IXY_des_ (byte b, InstIXY type);
	//z80word decodeRLC__IY_des_ (byte b);

	z80word decodeRL_r (byte b);
	z80word decodeRL__IXY_des_ (byte b, InstIXY type);
	//z80word decodeRL__IY_des_ (byte b);

	z80word decodeRRC_r (byte b);
	z80word decodeRRC__IXY_des_ (byte b, InstIXY type);
	//z80word decodeRRC__IY_des_ (byte b);

	z80word decodeRR_r (byte b);
	z80word decodeRR__IXY_des_ (byte b, InstIXY type);
	//z80word decodeRR__IY_des_ (byte b);

	z80word decodeSLA_r (byte b);
	z80word decodeSLA__IXY_des_ (byte b, InstIXY type);
	//z80word decodeSLA__IY_des_ (byte b);

	z80word decodeSRA_r (byte b);
	z80word decodeSRA__IXY_des_ (byte b, InstIXY type);
	//z80word decodeSRA__IY_des_ (byte b);

	z80word decodeSRL_r (byte b);
	z80word decodeSRL__IXY_des_ (byte b, InstIXY type);
	//z80word decodeSRL__IY_des_ (byte b);

	z80word decodeSLL_r (byte b);
	z80word decodeSLL__IXY_des_ (byte b, InstIXY type);

	// Bit, Set, Reset and Test Group.

	z80word decodeBIT_b_r (byte b);
	z80word decodeBIT_b__IXY_des_ (byte b, InstIXY type);
	//z80word decodeBIT_b__IY_des_ (byte b);

	z80word decodeSET_b_r (byte b);
	z80word decodeSET_b__IXY_des_ (byte b, InstIXY type);
	//z80word decodeSET_b__IY_des_ (byte b);

	z80word decodeRES_b_r (byte b);
	z80word decodeRES_b__IXY_des_ (byte b, InstIXY type);
	//z80word decodeRES_b__IY_des_ (byte b);

	// Jump Group.

	z80word decodeJP (byte b);
	z80word decodeJP_c (byte b);
	z80word decodeJR (byte b);
	z80word decodeJR_c (byte b);
	z80word decodeDJNZ (byte b);

	// Call And Return Group.
	z80word decodeCALL (byte b);
	z80word decodeCALL_c (byte b);
	z80word decodeRET_c (byte b);
	z80word decodeRST (byte b);

	// Input and Output Group.
	z80word decodeIN_A__n_ (byte b);
	z80word decodeIN_r__C_ (byte b);
	z80word decodeOUT__n__A (byte b);
	z80word decodeOUT__C__r (byte b);
};

const char * Disasm::regsimple (byte b)
{
	switch (b)
	{
	case 0: return "B";
	case 1: return "C";
	case 2: return "D";
	case 3: return "E";
	case 4: return "H";
	case 5: return "L";
	case 6: return "(HL)";
	case 7: return "A";
	default:
		ASSERT (false);
		throw "Disassembler internal error";
	}
}

const char * Disasm::regsimpleIX (byte b)
{
	switch (b)
	{
	case 0: return "B";
	case 1: return "C";
	case 2: return "D";
	case 3: return "E";
	case 4: return "IXH";
	case 5: return "IXL";
	case 6: return "(HL)";
	case 7: return "A";
	default:
		ASSERT (false);
		throw "Disassembler internal error";
	}
}

const char * Disasm::regsimpleIY (byte b)
{
	switch (b)
	{
	case 0: return "B";
	case 1: return "C";
	case 2: return "D";
	case 3: return "E";
	case 4: return "IYH";
	case 5: return "IYL";
	case 6: return "(HL)";
	case 7: return "A";
	default:
		ASSERT (false);
		throw "Disassembler internal error";
	}
}

const char * Disasm::regsimpleIXY (byte b, InstIXY type)
{
	switch (b)
	{
	case 0: return "B";
	case 1: return "C";
	case 2: return "D";
	case 3: return "E";
	case 4: return (type == InstIX) ? "IXH" : "IYH";
	case 5: return (type == InstIX) ? "IXL" : "IYL";
	case 6: return "(HL)";
	case 7: return "A";
	default:
		ASSERT (false);
		throw "Disassembler internal error";
	}
}

const char * Disasm::nameIXY (InstIXY type)
{
	return (type == InstIX) ? "IX" : "IY";
}

const char * Disasm::regdouble (byte b)
{
	switch (b)
	{
	case 0: return "BC";
	case 1: return "DE";
	case 2: return "HL";
	case 3: return "SP";
	default:
		ASSERT (false);
		throw "Disassembler internal error";
	}
}

const char * Disasm::regdoubleq (byte b)
{
	switch (b)
	{
	case 0: return "BC";
	case 1: return "DE";
	case 2: return "HL";
	case 3: return "AF";
	default:
		ASSERT (false);
		throw "Disassembler internal error";
	}
}

const char * Disasm::condition (byte b)
{
	switch (b)
	{
	case 0: return "NZ";
	case 1: return "Z";
	case 2: return "NC";
	case 3: return "C";
	case 4: return "PO";
	case 5: return "PE";
	case 6: return "P";
	case 7: return "M";
	default:
		ASSERT (false);
		throw "Disassembler internal error";
	}
}

//Disasm::decode_func Disasm::finddecodefunc
//	(const idinst_elem * idinst, size_t n,byte b)
template <typename FUNC>
FUNC Disasm::finddecodefunc
	(const idinst_elem_template <FUNC> * idinst, size_t n,byte b)
{
	for (size_t i= 0; i < n; ++i)
	{
		const idinst_elem_template <FUNC> & id= idinst [i];
		if ( (b & id.mask) == id.pattern)
			return id.decode;
	}
	return 0;
}

const Disasm::idinst_elem Disasm::idinst []= {
	{ 0xFF, 0x22, & Disasm::decodeLD__nn__HL },
	{ 0xFF, 0x2A, & Disasm::decodeLD_HL__nn_ },
	{ 0xFF, 0x32, & Disasm::decodeLD__nn__A },
	{ 0xFF, 0x3A, & Disasm::decodeLD_A__nn_ },
	{ 0xCF, 0x01, & Disasm::decodeLD_dd_nn },
	{ 0xCF, 0xC1, & Disasm::decodePOP },
	{ 0xCF, 0xC5, & Disasm::decodePUSH },
	{ 0xFF, 0xC6, & Disasm::decodeADD_A_n },
	{ 0xFF, 0xCE, & Disasm::decodeADC_A_n },
	{ 0xFF, 0xD6, & Disasm::decodeSUB_n },
	{ 0xFF, 0xDE, & Disasm::decodeSBC_A_n },
	{ 0xC7, 0x04, & Disasm::decodeINC_r },
	{ 0xCF, 0x09, & Disasm::decodeADD_HL_dd },
	{ 0xC7, 0x05, & Disasm::decodeDEC_r },
	{ 0xCF, 0x03, & Disasm::decodeINC_dd },
	{ 0xCF, 0x0B, & Disasm::decodeDEC_dd },
	{ 0xC7, 0x06, & Disasm::decodeLD_r_n },
	{ 0xF8, 0x80, & Disasm::decodeADD_A_r },
	{ 0xFF, 0xE6, & Disasm::decodeAND_n },
	{ 0xFF, 0xC3, & Disasm::decodeJP },
	{ 0xC0, 0x40, & Disasm::decodeLD_r_r },
	{ 0xF8, 0xA0, & Disasm::decodeAND_r },
	{ 0xFF, 0xF6, & Disasm::decodeOR_n },
	{ 0xFF, 0xEE, & Disasm::decodeXOR_n },
	{ 0xFF, 0xFE, & Disasm::decodeCP_n },
	{ 0xFF, 0xCB, & Disasm::decodeCB },
	{ 0xFF, 0xED, & Disasm::decodeED },
	{ 0xFF, 0xDD, & Disasm::decodeDD },
	{ 0xFF, 0xFD, & Disasm::decodeFD },
	{ 0xF8, 0x88, & Disasm::decodeADC_A_r },
	{ 0xF8, 0x90, & Disasm::decodeSUB_r },
	{ 0xF8, 0x98, & Disasm::decodeSBC_A_r },
	{ 0xF8, 0xB0, & Disasm::decodeOR_r },
	{ 0xF8, 0xA8, & Disasm::decodeXOR_r },
	{ 0xF8, 0xB8, & Disasm::decodeCP_r },
	{ 0xC7, 0xC2, & Disasm::decodeJP_c },
	{ 0xFF, 0x18, & Disasm::decodeJR },
	{ 0xE7, 0x20, & Disasm::decodeJR_c },
	{ 0xFF, 0x10, & Disasm::decodeDJNZ },
	{ 0xFF, 0xCD, & Disasm::decodeCALL },
	{ 0xC7, 0xC4, & Disasm::decodeCALL_c },
	{ 0xC7, 0xC0, & Disasm::decodeRET_c },
	{ 0xC7, 0xC7, & Disasm::decodeRST },
	{ 0xFF, 0xDB, & Disasm::decodeIN_A__n_ },
	{ 0xFF, 0xD3, & Disasm::decodeOUT__n__A }
};

const Disasm::idinst_elem Disasm::idinstCB []= {
	{ 0xF8, 0x00, & Disasm::decodeRLC_r },
	{ 0xF8, 0x10, & Disasm::decodeRL_r },
	{ 0xF8, 0x08, & Disasm::decodeRRC_r },
	{ 0xF8, 0x18, & Disasm::decodeRR_r },
	{ 0xF8, 0x20, & Disasm::decodeSLA_r },
	{ 0xF8, 0x28, & Disasm::decodeSRA_r },
	{ 0xF8, 0x38, & Disasm::decodeSRL_r },
	{ 0xF8, 0x30, & Disasm::decodeSLL_r },
	{ 0xC0, 0x40, & Disasm::decodeBIT_b_r },
	{ 0xC0, 0xC0, & Disasm::decodeSET_b_r },
	{ 0xC0, 0x80, & Disasm::decodeRES_b_r },
};

const Disasm::idinst_elem Disasm::idinstED []= {
	{ 0xCF, 0x4B, & Disasm::decodeLD_dd__nn_ },
	{ 0xCF, 0x43, & Disasm::decodeLD__nn__dd },
	{ 0xCF, 0x4A, & Disasm::decodeADC_HL_dd },
	{ 0xCF, 0x42, & Disasm::decodeSBC_HL_dd },
	{ 0xC7, 0x40, & Disasm::decodeIN_r__C_ },
	{ 0xC7, 0x41, & Disasm::decodeOUT__C__r },
};

//const Disasm::idinst_elem Disasm::idinstDD []= {
const Disasm::idinst_elem_IXY Disasm::idinstIXY []= {
	{ 0xFF, 0xCB, & Disasm::decodeIXYCB },
	{ 0xC7, 0x46, & Disasm::decodeLD_r__IXY_des_ },
	{ 0xF8, 0x70, & Disasm::decodeLD__IXY_des__r },
	{ 0xFF, 0x36, & Disasm::decodeLD__IXY_des__n },
	{ 0xFF, 0x21, & Disasm::decodeLD_IXY_nn },
	{ 0xFF, 0x2A, & Disasm::decodeLD_IXY__nn_ },
	{ 0xFF, 0x22, & Disasm::decodeLD__nn__IXY },
	{ 0xFF, 0x86, & Disasm::decodeADD_A__IXY_des_ },
	{ 0xFF, 0x8E, & Disasm::decodeADC_A__IXY_des_ },
	{ 0xFF, 0x96, & Disasm::decodeSUB__IXY_des_ },
	{ 0xFF, 0x9E, & Disasm::decodeSBC_A__IXY_des_ },
	{ 0xCF, 0x09, & Disasm::decodeADD_IXY_dd },
	{ 0xFF, 0xA6, & Disasm::decodeAND__IXY_des_ },
	{ 0xFF, 0xB6, & Disasm::decodeOR__IXY_des_ },
	{ 0xFF, 0xAE, & Disasm::decodeXOR__IXY_des_ },
	{ 0xFF, 0xBE, & Disasm::decodeCP__IXY_des_ },
	{ 0xFF, 0x34, & Disasm::decodeINC__IXY_des_ },
	{ 0xFF, 0x35, & Disasm::decodeDEC__IXY_des_ },
	{ 0xC6, 0x44, & Disasm::decodeLD_r_IXYHL },
	{ 0xF0, 0x60, & Disasm::decodeLD_IXYHL_r },
	{ 0xF7, 0x26, & Disasm::decodeLD_IXYHL_n },
	{ 0xF7, 0x24, & Disasm::decodeINC_IXYHL },
	{ 0xF7, 0x25, & Disasm::decodeDEC_IXYHL },
	{ 0xFE, 0x84, & Disasm::decodeADD_A_IXYHL },
	{ 0xFE, 0x8C, & Disasm::decodeADC_A_IXYHL },
	{ 0xFE, 0x9C, & Disasm::decodeSBC_A_IXYHL },
	{ 0xFE, 0x94, & Disasm::decodeSUB_IXYHL },
	{ 0xFE, 0xA4, & Disasm::decodeAND_IXYHL },
	{ 0xFE, 0xAC, & Disasm::decodeXOR_IXYHL },
	{ 0xFE, 0xB4, & Disasm::decodeOR_IXYHL },
	{ 0xFE, 0xBC, & Disasm::decodeCP_IXYHL },
};

#if 0
const Disasm::idinst_elem Disasm::idinstFD []= {
	{ 0xFF, 0xCB, & Disasm::decodeFDCB },
	{ 0xC7, 0x46, & Disasm::decodeLD_r__IY_des_ },
	{ 0xF8, 0x70, & Disasm::decodeLD__IY_des__r },
	{ 0xFF, 0x36, & Disasm::decodeLD__IY_des__n },
	{ 0xFF, 0x21, & Disasm::decodeLD_IY_nn },
	{ 0xFF, 0x2A, & Disasm::decodeLD_IY__nn_ },
	{ 0xFF, 0x22, & Disasm::decodeLD__nn__IY },
	{ 0xFF, 0x86, & Disasm::decodeADD_A__IY_des_ },
	{ 0xFF, 0x8E, & Disasm::decodeADC_A__IY_des_ },
	{ 0xFF, 0x96, & Disasm::decodeSUB__IY_des_ },
	{ 0xFF, 0x9E, & Disasm::decodeSBC_A__IY_des_ },
	{ 0xCF, 0x09, & Disasm::decodeADD_IY_dd },
	{ 0xFF, 0xA6, & Disasm::decodeAND__IY_des_ },
	{ 0xFF, 0xB6, & Disasm::decodeOR__IY_des_ },
	{ 0xFF, 0xAE, & Disasm::decodeXOR__IY_des_ },
	{ 0xFF, 0xBE, & Disasm::decodeCP__IY_des_ },
	{ 0xFF, 0x34, & Disasm::decodeINC__IY_des_ },
	{ 0xFF, 0x35, & Disasm::decodeDEC__IY_des_ },
	{ 0xC6, 0x44, & Disasm::decodeLD_r_IYHL },
	{ 0xF0, 0x60, & Disasm::decodeLD_IYHL_r },
	{ 0xF7, 0x26, & Disasm::decodeLD_IYHL_n },
};
#endif

//const Disasm::idinst_elem Disasm::idinstDDCB []= {
const Disasm::idinst_elem_IXY Disasm::idinstIXYCB []= {
	{ 0xFF, 0x06, & Disasm::decodeRLC__IXY_des_ },
	{ 0xFF, 0x16, & Disasm::decodeRL__IXY_des_ },
	{ 0xFF, 0x0E, & Disasm::decodeRRC__IXY_des_ },
	{ 0xFF, 0x1E, & Disasm::decodeRR__IXY_des_ },
	{ 0xFF, 0x26, & Disasm::decodeSLA__IXY_des_ },
	{ 0xFF, 0x2E, & Disasm::decodeSRA__IXY_des_ },
	{ 0xFF, 0x3E, & Disasm::decodeSRL__IXY_des_ },
	{ 0xFF, 0x36, & Disasm::decodeSLL__IXY_des_ },
	{ 0xC7, 0x46, & Disasm::decodeBIT_b__IXY_des_ },
	{ 0xC7, 0xC6, & Disasm::decodeSET_b__IXY_des_ },
	{ 0xC7, 0x86, & Disasm::decodeRES_b__IXY_des_ },
};

#if 0
const Disasm::idinst_elem Disasm::idinstFDCB []= {
	{ 0xFF, 0x06, & Disasm::decodeRLC__IY_des_ },
	{ 0xFF, 0x16, & Disasm::decodeRL__IY_des_ },
	{ 0xFF, 0x0E, & Disasm::decodeRRC__IY_des_ },
	{ 0xFF, 0x1E, & Disasm::decodeRR__IY_des_ },
	{ 0xFF, 0x26, & Disasm::decodeSLA__IY_des_ },
	{ 0xFF, 0x2E, & Disasm::decodeSRA__IY_des_ },
	{ 0xFF, 0x3E, & Disasm::decodeSRL__IY_des_ },
	{ 0xC7, 0x46, & Disasm::decodeBIT_b__IY_des_ },
	{ 0xC7, 0xC6, & Disasm::decodeSET_b__IY_des_ },
	{ 0xC7, 0x86, & Disasm::decodeRES_b__IY_des_ },
};
#endif

#ifndef NDEBUG

unsigned int bitcount (byte b)
{
	unsigned int r= 0;
	if (b & 0x80) ++r;
	if (b & 0x40) ++r;
	if (b & 0x20) ++r;
	if (b & 0x10) ++r;
	if (b & 0x08) ++r;
	if (b & 0x04) ++r;
	if (b & 0x02) ++r;
	if (b & 0x01) ++r;
	return r;
}

//bool Disasm::checkidinst (const idinst_elem * idinst, size_t n)
template <class IDINST_ELEM>
bool Disasm::checkidinst (const IDINST_ELEM * idinst, size_t n)
{
	#if 0
	for (unsigned int b= 0; b < 256; ++b)
	{
		int i= static_cast <int> (n - 1);
		for ( ; i >= 0; --i)
		{
			const IDINST_ELEM & id= idinst [i];
			if ( (b & id.mask) == id.pattern)
				break;
		}
		if (i < 0)
			continue;
		const IDINST_ELEM & idant= idinst [i];
		for (int j= i - 1; j >= 0; --j)
		{
			const IDINST_ELEM & id= idinst [j];
			if ( (b & id.mask) == id.pattern &&
				bitcount (id.mask) < bitcount (idant.mask) )
			{
				cerr << "Byte " << hex2 (b) <<
					" masked by element " << i <<
					" is previously masked by element " <<
					j << endl;
				return false;
			}
		}
	}
	#endif
	std::vector <bool> reached (n);
	for (unsigned int b= 0; b < 256; ++b)
	{
		for (size_t i= 0; i < n; ++i)
		{
			const IDINST_ELEM & id= idinst [i];
			if ( (b & id.mask) == id.pattern)
			{
				reached [i]= true;
				break;
			}
		}
	}
	for (size_t i= 0; i < n; ++i)
		if (! reached [i] )
		{
			cerr << "Element " << i << " is never reached" << endl;
			return false;
		}
	return true;
}

bool Disasm::checkidinsttables ()
{
	if (! checkidinst (idinst, dim_array (idinst) ) )
	{
		cerr << "Table idinst incorrect" << endl;
		throw 0;
	}
	if (! checkidinst (idinstCB, dim_array (idinstCB) ) )
	{
		cerr << "Table idinstCB incorrect" << endl;
		throw 0;
	}
	if (! checkidinst (idinstED, dim_array (idinstED) ) )
	{
		cerr << "Table idinstED incorrect" << endl;
		throw 0;
	}
	if (! checkidinst (idinstIXY, dim_array (idinstIXY) ) )
	{
		cerr << "Table idinstDD incorrect" << endl;
		throw 0;
	}
	#if 0
	if (! checkidinst (idinstFD, dim_array (idinstFD) ) )
	{
		cerr << "Table idinstFD incorrect" << endl;
		throw 0;
	}
	#endif
	if (! checkidinst (idinstIXYCB, dim_array (idinstIXYCB) ) )
	{
		cerr << "Table idinstDDCB incorrect" << endl;
		throw 0;
	}
	#if 0
	if (! checkidinst (idinstFDCB, dim_array (idinstFDCB) ) )
	{
		cerr << "Table idinstFDCB incorrect" << endl;
		throw 0;
	}
	#endif
	return true;
}

bool indinstchecked= Disasm::checkidinsttables ();

#endif

typedef std::map <byte, string> simple_t;
simple_t simple;
simple_t simpleED;
simple_t simpleDD;
simple_t simpleFD;

bool init_maps ()
{
	//simple [0x]= "";

	simple [0x00]= "NOP";
	simple [0x02]= "LD (BC),A";
	simple [0x07]= "RLCA";
	simple [0x08]= "EX AF,AF'";
	simple [0x0A]= "LD A,(BC)";
	simple [0x0F]= "RRCA";
	simple [0x12]= "LD (DE),A";
	simple [0x17]= "RLA";
	simple [0x1A]= "LD A,(DE)";
	simple [0x1F]= "RRA";
	simple [0x27]= "DAA";
	simple [0x2F]= "CPL";
	simple [0x37]= "SCF";
	simple [0x3F]= "CCF";
	simple [0x76]= "HALT";
	simple [0xC9]= "RET";
	simple [0xD9]= "EXX";
	simple [0xE3]= "EX (SP),HL";
	simple [0xE9]= "JP (HL)";
	simple [0xEB]= "EX DE,HL";
	simple [0xF3]= "DI";
	simple [0xF9]= "LD SP,HL";
	simple [0xFB]= "EI";

	//simpleED [0x]= "";

	simpleED [0x44]= "NEG";
	simpleED [0x45]= "RETN";
	simpleED [0x46]= "IM 0";
	simpleED [0x47]= "LD I,A";
	simpleED [0x4D]= "RETI";
	simpleED [0x4F]= "LD R,A";
	simpleED [0x56]= "IM 1";
	simpleED [0x57]= "LD A,I";
	simpleED [0x5E]= "IM 2";
	simpleED [0x5F]= "LD A,R";
	simpleED [0x67]= "RRD";
	simpleED [0x6F]= "RLD";
	simpleED [0xA0]= "LDI";
	simpleED [0xA1]= "CPI";
	simpleED [0xA2]= "INI";
	simpleED [0xA3]= "OUTI";
	simpleED [0xA8]= "LDD";
	simpleED [0xA9]= "CPD";
	simpleED [0xAA]= "IND";
	simpleED [0xAB]= "OUTD";
	simpleED [0xB0]= "LDIR";
	simpleED [0xB1]= "CPIR";
	simpleED [0xB2]= "INIR";
	simpleED [0xB3]= "OTIR";
	simpleED [0xB8]= "LDDR";
	simpleED [0xB9]= "CPDR";
	simpleED [0xBA]= "INDR";
	simpleED [0xBB]= "OTDR";

	simpleDD [0x23]= "INC IX";
	simpleDD [0x29]= "ADD IX,IX";
	simpleDD [0x2B]= "DEC IX";
	simpleDD [0xE1]= "POP IX";
	simpleDD [0xE3]= "EX (SP),IX";
	simpleDD [0xE5]= "PUSH IX";
	simpleDD [0xE9]= "JP (IX)";
	simpleDD [0xF9]= "LD SP,IX";

	simpleFD [0x23]= "INC IY";
	simpleFD [0x29]= "ADD IY,IY";
	simpleFD [0x2B]= "DEC IY";
	simpleFD [0xE1]= "POP IY";
	simpleFD [0xE3]= "EX (SP),IY";
	simpleFD [0xE5]= "PUSH IY";
	simpleFD [0xE9]= "JP (IY)";
	simpleFD [0xF9]= "LD SP,IY";

	return true;
}

bool map_ready= init_maps ();

bool findsimple (const simple_t & m, byte b, string & str)
{
	simple_t::const_iterator it= m.find (b);
	if (it != m.end () )
	{
		str= it->second;
		return true;
	}
	else
		return false;
}

z80word Disasm::decoderelative ()
{
	z80word rel= cpu.memread (pos + 1);
	z80word dest= pos + 2;
	if (rel < 128)
		dest+= rel;
	else
		dest-= 256 - rel;
	ostringstream oss;
	oss << hex4 (dest);
	op+= oss.str ();
	return pos + 2;
}

void Disasm::decode_n (z80word des)
{
	byte n= cpu.memread (pos + des);
	ostringstream oss;
	oss << hex2 (n);
	op+= oss.str ();
}

void Disasm::decode_nn (z80word des)
{
	z80word nn= cpu.memreadw (pos + des);
	ostringstream oss;
	oss << hex4 (nn);
	op+= oss.str ();
}

void Disasm::decode_des (z80word des)
{
	int n= cpu.memread (pos + des);
	ostringstream oss;
	if (n < 128)
		oss << '+' << hex2 (n);
	else
		oss << '-' << hex2 (256 - n);
	oss << ')';
	op+= oss.str ();
}

z80word Disasm::decodeCB (byte)
{
	byte b= cpu.memread (pos + 1);
	//if (findsimple (simpleCB, b, op) )
	//	return pos + 2;

	decode_func decode=
		finddecodefunc (idinstCB, dim_array (idinstCB), b);
	if (decode)
		return (this->* decode) (b);

	op= "?=CB";
	return pos + 1;
}

z80word Disasm::decodeED (byte)
{
	byte b= cpu.memread (pos + 1);
	if (findsimple (simpleED, b, op) )
		return pos + 2;

	decode_func decode=
		finddecodefunc (idinstED, dim_array (idinstED), b);
	if (decode)
		return (this->* decode) (b);

	op= "?=ED";
	return pos + 1;
}

z80word Disasm::decodeDD (byte)
{
	byte b= cpu.memread (pos + 1);
	if (findsimple (simpleDD, b, op) )
		return pos + 2;

	//decode_func decode=
	//	finddecodefunc (idinstDD, dim_array (idinstDD), b);
	decode_func_IXY decode=
		finddecodefunc (idinstIXY, dim_array (idinstIXY), b);
	if (decode)
		return (this->* decode) (b, InstIX);

	op= "?=DD";
	return pos + 1;
}

z80word Disasm::decodeFD (byte)
{
	byte b= cpu.memread (pos + 1);
	if (findsimple (simpleFD, b, op) )
		return pos + 2;

	//decode_func decode=
	//	finddecodefunc (idinstFD, dim_array (idinstFD), b);
	decode_func_IXY decode=
		finddecodefunc (idinstIXY, dim_array (idinstIXY), b);
	if (decode)
		return (this->* decode) (b, InstIY);

	op= "?=FD";
	return pos + 1;
}

//z80word Disasm::decodeDDCB (byte)
z80word Disasm::decodeIXYCB (byte, InstIXY type)
{
	byte b= cpu.memread (pos + 3);
	//if (findsimple (simpleDDCB, b, op) )
	//	return pos + 2;

	//decode_func decode=
	//	finddecodefunc (idinstDDCB, dim_array (idinstDDCB), b);
	//if (decode)
	//	return (this->* decode) (b);
	decode_func_IXY decode=
		finddecodefunc (idinstIXYCB, dim_array (idinstIXYCB), b);
	if (decode)
		return (this->* decode) (b, type);

	op= (type == InstIX) ? "?=DD" : "?=FD";
	return pos + 1;
}

#if 0
z80word Disasm::decodeFDCB (byte)
{
	byte b= cpu.memread (pos + 3);
	//if (findsimple (simpleFDCB, b, op) )
	//	return pos + 2;

	decode_func decode=
		finddecodefunc (idinstFDCB, dim_array (idinstFDCB), b);
	if (decode)
		return (this->* decode) (b);

	op= "?=FD";
	return pos + 1;
}
#endif

z80word Disasm::decode ()
{
	byte b= cpu.memread (pos);
	if (findsimple (simple, b, op) )
		return pos + 1;

	decode_func decode=
		finddecodefunc (idinst, dim_array (idinst), b);
	if (decode)
		return (this->* decode) (b);

	ostringstream oss;
	oss << "?=" << hex2 (b);
	op= oss.str ();
	return pos + 1;
}

z80word Disasm::decodeLD_r_r (byte b)
{
	byte r1= (b >> 3) & 0x07;
	byte r2= b & 0x07;
	op= std::string ("LD ") + regsimple (r1) + ',' + regsimple (r2);
	return pos + 1;
}

z80word Disasm::decodeLD_IXYHL_r (byte b, InstIXY type)
{
	byte r1= (b >> 3) & 0x07;
	ASSERT (r1 == 4 || r1 == 5);
	byte r2= b & 0x07;
	op= std::string ("LD ") + regsimpleIXY (r1, type) + ',' +
		regsimpleIXY (r2, type);
	return pos + 2;
}

#if 0
z80word Disasm::decodeLD_IYHL_r (byte b)
{
	byte r1= (b >> 3) & 0x07;
	ASSERT (r1 == 4 || r1 == 5);
	byte r2= b & 0x07;
	op= std::string ("LD ") + regsimpleIY (r1) + ',' + regsimpleIY (r2);
	return pos + 2;
}
#endif

z80word Disasm::decodeLD_r_IXYHL (byte b, InstIXY type)
{
	byte r1= (b >> 3) & 0x07;
	byte r2= b & 0x07;
	ASSERT (r2 == 4 || r2 == 5);
	op= std::string ("LD ") + regsimpleIXY (r1, type) + ',' +
		regsimpleIXY (r2, type);
	return pos + 2;
}

#if 0
z80word Disasm::decodeLD_r_IYHL (byte b)
{
	byte r1= (b >> 3) & 0x07;
	byte r2= b & 0x07;
	ASSERT (r2 == 4 || r2 == 5);
	op= std::string ("LD ") + regsimpleIY (r1) + ',' + regsimpleIY (r2);
	return pos + 2;
}
#endif

z80word Disasm::decodeLD_r_n (byte b)
{
	byte r= (b >> 3) & 0x07;
	op= "LD ";
	op+= regsimple (r);
	op+= ',';
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeLD_IXYHL_n (byte b, InstIXY type)
{
	byte r= (b >> 3) & 0x07;
	ASSERT (r == 4 || r == 5);
	op= std::string ("LD ") + regsimpleIXY (r, type) + ',';
	decode_n (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeLD_IYHL_n (byte b)
{
	byte r= (b >> 3) & 0x07;
	ASSERT (r == 4 || r == 5);
	op= std::string ("LD ") + regsimpleIY (r) + ',';
	decode_n (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeLD_r__IXY_des_ (byte b, InstIXY type)
{
	op= "LD ";
	byte r= (b >> 3) & 0x07;
	op+= regsimple (r);
	op+= ",(";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeLD_r__IY_des_ (byte b)
{
	op= "LD ";
	byte r= (b >> 3) & 0x07;
	op+= regsimple (r);
	op+= ",(IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeLD__IXY_des__r (byte b, InstIXY type)
{
	op= "LD (";
	op+= nameIXY (type);
	decode_des (2);
	op+=",";
	op+= regsimple (b & 0x07);
	return pos + 3;
}

#if 0
z80word Disasm::decodeLD__IY_des__r (byte b)
{
	op= "LD (IY";
	decode_des (2);
	op+=",";
	op+= regsimple (b & 0x07);
	return pos + 3;
}
#endif

z80word Disasm::decodeLD__IXY_des__n (byte, InstIXY type)
{
	op= "LD (";
	op+= nameIXY (type);
	decode_des (2);
	op+= ',';
	decode_n (3);
	return pos + 4;
}

#if 0
z80word Disasm::decodeLD__IY_des__n (byte)
{
	op= "LD (IY";
	decode_des (2);
	op+= ',';
	decode_n (3);
	return pos + 4;
}
#endif

z80word Disasm::decodeLD_A__nn_ (byte)
{
	op= "LD A,(";
	decode_nn (1);
	op+=')';
	return pos + 3;
}

z80word Disasm::decodeLD__nn__A (byte)
{
	op= "LD (";
	decode_nn (1);
	op+= "),A";
	return pos + 3;
}

z80word Disasm::decodeLD_dd_nn (byte b)
{
	byte r= (b >> 4) & 0x03;
	//z80word nn= cpu.memreadw (pos + 1);
	//ostringstream oss;
	//oss << "LD " << regdouble (r) << ',' << hex4 << nn;
	//op= oss.str ();
	op= "LD ";
	op+= regdouble (r);
	op+=',';
	decode_nn (1);
	return pos + 3;
}

z80word Disasm::decodeLD_IXY_nn (byte, InstIXY type)
{
	op= "LD ";
	op+= nameIXY (type);
	op+= ',';
	decode_nn (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeLD_IY_nn (byte)
{
	op= "LD IY,";
	decode_nn (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeLD_HL__nn_ (byte)
{
	op= "LD HL,(";
	decode_nn (1);
	op += ')';
	return pos + 3;
}

z80word Disasm::decodeLD_dd__nn_ (byte b)
{
	op= "LD ";
	byte r= (b >> 4) & 0x03;
	op+= regdouble (r);
	op+= ",(";
	decode_nn (2);
	op+= ')';
	return pos + 4;
}

z80word Disasm::decodeLD_IXY__nn_ (byte, InstIXY type)
{
	op= "LD ";
	op+= nameIXY (type);
	op+= ",(";
	decode_nn (2);
	op+= ')';
	return pos + 4;
}

#if 0
z80word Disasm::decodeLD_IY__nn_ (byte)
{
	op= "LD IY,(";
	decode_nn (2);
	op+= ')';
	return pos + 4;
}
#endif

z80word Disasm::decodeLD__nn__HL (byte)
{
	op= "LD (";
	decode_nn (1);
	op+= "),HL";
	return pos + 3;
}

z80word Disasm::decodeLD__nn__dd (byte b)
{
	op= "LD (";
	decode_nn (2);
	op+= "),";
	byte r= (b >> 4) & 0x03;
	op+= regdouble (r);
	return pos + 4;
}

z80word Disasm::decodeLD__nn__IXY (byte, InstIXY type)
{
	op= "LD (";
	decode_nn (2);
	op+= "),";
	op+= nameIXY (type);
	return pos + 4;
}

#if 0
z80word Disasm::decodeLD__nn__IY (byte)
{
	op= "LD (";
	decode_nn (2);
	op+= "),IY";
	return pos + 4;
}
#endif

z80word Disasm::decodePUSH (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "PUSH ";
	op+= regdoubleq (r);
	return pos + 1;
}

z80word Disasm::decodePOP (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "POP ";
	op+= regdoubleq (r);
	return pos + 1;
}

z80word Disasm::decodeADD_A_r (byte b)
{
	byte r= b & 0x07;
	op= "ADD A,";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeADD_A_n (byte)
{
	op= "ADD A,";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeADD_A__IXY_des_ (byte, InstIXY type)
{
	op= "ADD A,(";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeADD_A__IY_des_ (byte)
{
	op= "ADD A,(IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeADD_A_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "ADD A,";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeADC_A_r (byte b)
{
	byte r= b & 0x07;
	op= "ADC A,";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeADC_A_n (byte)
{
	op= "ADC A,";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeADC_A__IXY_des_ (byte, InstIXY type)
{
	op= "ADC A,(";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeADC_A__IY_des_ (byte)
{
	op= "ADC A,(IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeADC_A_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "ADC A,";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeSUB_r (byte b)
{
	byte r= b & 0x07;
	op= "SUB ";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeSUB_n (byte)
{
	op= "SUB ";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeSUB__IXY_des_ (byte, InstIXY type)
{
	op= "SUB (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeSUB__IY_des_ (byte)
{
	op= "SUB (IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeSUB_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "SUB ";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeSBC_A_r (byte b)
{
	byte r= b & 0x07;
	op= "SBC A,";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeSBC_A_n (byte)
{
	op= "SBC A,";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeSBC_A__IXY_des_ (byte, InstIXY type)
{
	op= "SBC A,(";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeSBC_A__IY_des_ (byte)
{
	op= "SBC A,(IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeSBC_A_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "SBC A,";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeAND_r (byte b)
{
	byte r= b & 0x07;
	op= "AND ";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeAND_n (byte)
{
	op= "AND ";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeAND__IXY_des_ (byte, InstIXY type)
{
	op= "AND (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeAND__IY_des_ (byte)
{
	op= "AND (IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeAND_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "AND ";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeOR_r (byte b)
{
	byte r= b & 0x07;
	op= "OR ";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeOR_n (byte)
{
	op= "OR ";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeOR__IXY_des_ (byte, InstIXY type)
{
	op= "OR (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeOR__IY_des_ (byte)
{
	op= "OR (IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeOR_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "OR ";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeXOR_r (byte b)
{
	byte r= b & 0x07;
	op= "XOR ";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeXOR_n (byte)
{
	op= "XOR ";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeXOR__IXY_des_ (byte, InstIXY type)
{
	op= "XOR (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeXOR__IY_des_ (byte)
{
	op= "XOR (IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeXOR_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "XOR ";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeCP_r (byte b)
{
	byte r= b & 0x07;
	op= "CP ";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeCP_n (byte)
{
	op= "CP ";
	decode_n (1);
	return pos + 2;
}

z80word Disasm::decodeCP__IXY_des_ (byte, InstIXY type)
{
	op= "CP (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeCP__IY_des_ (byte)
{
	op= "CP (IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeCP_IXYHL (byte b, InstIXY type)
{
	byte r= b & 0x07;
	ASSERT (r == 4 || r == 5);
	op= "CP ";
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeINC_r (byte b)
{
	byte r= (b >> 3) & 0x07;
	op= "INC ";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeINC__IXY_des_ (byte, InstIXY type)
{
	op= "INC (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeINC__IY_des_ (byte)
{
	op= "INC (IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeINC_IXYHL (byte b, InstIXY type)
{
	op= "INC ";
	byte r= (b >> 3) & 0x07;
	ASSERT (r == 4 || r == 5);
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeDEC_r (byte b)
{
	byte r= (b >> 3) & 0x07;
	op= "DEC ";
	op+= regsimple (r);
	return pos + 1;
}

z80word Disasm::decodeDEC__IXY_des_ (byte, InstIXY type)
{
	op= "DEC (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 3;
}

#if 0
z80word Disasm::decodeDEC__IY_des_ (byte)
{
	op= "DEC (IY";
	decode_des (2);
	return pos + 3;
}
#endif

z80word Disasm::decodeDEC_IXYHL (byte b, InstIXY type)
{
	op= "DEC ";
	byte r= (b >> 3) & 0x07;
	ASSERT (r == 4 || r == 5);
	op+= regsimpleIXY (r, type);
	return pos + 2;
}

z80word Disasm::decodeADD_HL_dd (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "ADD HL,";
	op+= regdouble (r);
	return pos + 1;
}

z80word Disasm::decodeADC_HL_dd (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "ADC HL,";
	op+= regdouble (r);
	return pos + 2;
}

z80word Disasm::decodeSBC_HL_dd (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "SBC HL,";
	op+= regdouble (r);
	return pos + 2;
}

z80word Disasm::decodeADD_IXY_dd (byte b, InstIXY type)
{
	byte r= (b >> 4) & 0x03;
	op= "ADD ";
	op+= nameIXY (type);
	op+= ',';
	op+= regdouble (r);
	return pos + 2;
}

#if 0
z80word Disasm::decodeADD_IY_dd (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "ADD IY,";
	op+= regdouble (r);
	return pos + 2;
}
#endif

z80word Disasm::decodeINC_dd (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "INC ";
	op+= regdouble (r);
	return pos + 1;
}

z80word Disasm::decodeDEC_dd (byte b)
{
	byte r= (b >> 4) & 0x03;
	op= "DEC ";
	op+= regdouble (r);
	return pos + 1;
}

z80word Disasm::decodeRLC_r (byte b)
{
	byte r= b & 0x07;
	op= "RLC ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeRLC__IXY_des_ (byte, InstIXY type)
{
	op= "RLC (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeRLC__IY_des_ (byte)
{
	op= "RLC (IY";
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeRL_r (byte b)
{
	byte r= b & 0x07;
	op= "RL ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeRL__IXY_des_ (byte, InstIXY type)
{
	op= "RL (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeRL__IY_des_ (byte)
{
	op= "RL (IY";
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeRRC_r (byte b)
{
	byte r= b & 0x07;
	op= "RRC ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeRRC__IXY_des_ (byte, InstIXY type)
{
	op= "RRC (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeRRC__IY_des_ (byte)
{
	op= "RRC (IY";
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeRR_r (byte b)
{
	byte r= b & 0x07;
	op= "RR ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeRR__IXY_des_ (byte, InstIXY type)
{
	op= "RR (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeRR__IY_des_ (byte)
{
	op= "RR (IY";
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeSLA_r (byte b)
{
	byte r= b & 0x07;
	op= "SLA ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeSLA__IXY_des_ (byte, InstIXY type)
{
	op= "SLA (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeSLA__IY_des_ (byte)
{
	op= "SLA (IY";
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeSRA_r (byte b)
{
	byte r= b & 0x07;
	op= "SRA ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeSRA__IXY_des_ (byte, InstIXY type)
{
	op= "SRA (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeSRA__IY_des_ (byte)
{
	op= "SRA (IY";
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeSRL_r (byte b)
{
	byte r= b & 0x07;
	op= "SRL ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeSRL__IXY_des_ (byte, InstIXY type)
{
	op= "SRL (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

z80word Disasm::decodeSLL_r (byte b)
{
	byte r= b & 0x07;
	op= "SLL ";
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::decodeSLL__IXY_des_ (byte, InstIXY type)
{
	op= "SLL (";
	op+= nameIXY (type);
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeSRL__IY_des_ (byte)
{
	op= "SRL (IY";
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeBIT_b_r (byte b)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "BIT " << int (bit) << ',' << regsimple (b & 0x07);
	op= oss.str ();
	return pos + 2;
}

z80word Disasm::decodeBIT_b__IXY_des_ (byte b, InstIXY type)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "BIT " << int (bit) << ",(" << nameIXY (type);
	op= oss.str ();
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeBIT_b__IY_des_ (byte b)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "BIT " << int (bit) << ",(IY";
	op= oss.str ();
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeSET_b_r (byte b)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "SET " << int (bit) << ',' << regsimple (b & 0x07);
	op= oss.str ();
	return pos + 2;
}

z80word Disasm::decodeSET_b__IXY_des_ (byte b, InstIXY type)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "SET " << int (bit) << ",(" << nameIXY (type);
	op= oss.str ();
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeSET_b__IY_des_ (byte b)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "SET " << int (bit) << ",(IY";
	op= oss.str ();
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeRES_b_r (byte b)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "RES " << int (bit) << ',' << regsimple (b & 0x07);
	op= oss.str ();
	return pos + 2;
}

z80word Disasm::decodeRES_b__IXY_des_ (byte b, InstIXY type)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "RES " << int (bit) << ",(" << nameIXY (type);
	op= oss.str ();
	decode_des (2);
	return pos + 4;
}

#if 0
z80word Disasm::decodeRES_b__IY_des_ (byte b)
{
	byte bit= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "RES " << int (bit) << ",(IY";
	op= oss.str ();
	decode_des (2);
	return pos + 4;
}
#endif

z80word Disasm::decodeJP (byte)
{
	z80word nn= cpu.memreadw (pos + 1);
	ostringstream oss;
	oss << "JP " << hex4 (nn);
	op= oss.str ();
	return pos + 3;
}

z80word Disasm::decodeJP_c (byte b)
{
	z80word nn= cpu.memreadw (pos + 1);
	byte cond= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "JP " << condition (cond) << ',' << hex4 (nn);
	op= oss.str ();
	return pos + 3;
}

z80word Disasm::decodeJR (byte)
{
	op= "JR ";
	return decoderelative ();
}

z80word Disasm::decodeJR_c (byte b)
{
	op= "JR ";
	byte cond= (b >> 3) & 0x03;
	op+= condition (cond);
	op+=',';
	return decoderelative ();
}

z80word Disasm::decodeDJNZ (byte)
{
	op= "DJNZ ";
	return decoderelative ();
}

z80word Disasm::decodeCALL (byte)
{
	z80word nn= cpu.memreadw (pos + 1);
	ostringstream oss;
	oss << "CALL " << hex4 (nn);
	op= oss.str ();
	return pos + 3;
}

z80word Disasm::decodeCALL_c (byte b)
{
	z80word nn= cpu.memreadw (pos + 1);
	byte cond= (b >> 3) & 0x07;
	ostringstream oss;
	oss << "CALL " << condition (cond) << ',' << hex4 (nn);
	op= oss.str ();
	return pos + 3;
}

z80word Disasm::decodeRET_c (byte b)
{
	byte cond= (b >> 3) & 0x07;
	ostringstream oss;
	op= "RET ";
	op+= condition (cond);
	return pos + 1;
}

z80word Disasm::decodeRST (byte b)
{
	byte dest= b & 0x38;
	ostringstream oss;
	oss << "RST " << hex2 (dest);
	op= oss.str ();
	return pos + 1;
}

z80word Disasm::decodeIN_A__n_ (byte)
{
	byte n= cpu.memread (pos + 1);
	ostringstream oss;
	oss << "IN A,(" << hex2 (n) << ')';
	op= oss.str ();
	return pos + 2;
}

z80word Disasm::decodeIN_r__C_ (byte b)
{
	byte r= (b >> 3) & 0x07;
	op= "IN ";
	op+= regsimple (r);
	op+= ",(C)";
	return pos + 2;
}

z80word Disasm::decodeOUT__n__A (byte)
{
	op= "OUT (";
	decode_n (1);
	op+= "),A";
	return pos + 2;
}

z80word Disasm::decodeOUT__C__r (byte b)
{
	op= "OUT (C),";
	byte r= (b >> 3) & 0x07;
	op+= regsimple (r);
	return pos + 2;
}

z80word Disasm::doit ()
{
	z80word npos= decode ();
	ostringstream oss;
	#if 0
	for (z80word n= pos; n < npos; ++n)
	{
		oss << hex2 (cpu.memread (n) ) << ' ';
	}
	#else
	for (z80word n= pos; n < pos + 4; ++n)
	{
		if (n < npos)
			oss << hex2 (cpu.memread (n) ) << ' ';
		else
			oss << "   ";
	}
	#endif
	op= oss.str () + op;
	return npos;
}


} // namespace


z80word disasm (const Cpu & cpu, z80word pos, std::string & op)
{
	Disasm d (cpu, pos, op);
	return d.doit ();
}

z80word disasm_short (const Cpu & cpu, z80word pos, std::string & op)
{
	Disasm d (cpu, pos, op);
	return d.decode ();
}


#endif


// End of disasm.cpp
