// cpm.cpp
// Revision 16-may-2005


#include "cpm.h"

#include "cpu.h"
#include "fcb.h"
#include "directory.h"

#include <iostream>
#include <iomanip>
#include <sstream>
#include <vector>
#include <map>
#include <set>
#include <algorithm>
#include <stdexcept>
#include <memory>

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

#include <assert.h>
#define ASSERT assert

using std::ostream;
using std::cin;
using std::cout;
using std::cerr;
using std::flush;
using std::setw;
using std::setfill;
using std::istringstream;
using std::string;
using std::vector;
using std::map;
using std::set;
using std::fill;
using std::make_pair;
using std::for_each;
using std::runtime_error;
using std::logic_error;
using std::auto_ptr;


//***********************************************
//		Auxiliar functions.
//***********************************************


namespace {


ostream & crlf (ostream & os)
{
	os << "\r\n" << flush;
	return os;
}

inline byte cpmupper (byte c)
{
	return static_cast <byte> (toupper (c) );
}

string strupper (const string & str)
{
	string result;
	for (string::size_type i= 0; i < str.size (); ++i)
		result+= static_cast <char>
			(cpmupper (static_cast <byte> (str [i] ) ) );
	return result;
}

const string delimiters (" \t\r\n<=,!|>[].;:");

inline bool isdelimiter (byte c)
{
	return delimiters.find (c) != string::npos;
}

bool validchar (byte c)
{
	return ! isdelimiter (c);
}

bool validpasschar (byte c)
{
	// Provisional
	return ! isdelimiter (c);
}

inline byte conv_bcd (byte b)
{
	return (b / 10) * 16 + (b % 10);
}

void ltrim (string & str)
{
	string::size_type l= str.find_first_not_of (" \t");
	if (l != string::npos)
		str.erase (0, l);
}

inline size_t disknumfromletter (byte letter)
{
	return static_cast <size_t> (cpmupper (letter) - 'A');
}

inline byte letterfromdisknum (size_t disknum)
{
	return static_cast <byte> ('A' + disknum);
}

string normalize_dir (const string & path)
{
	if (path.empty () )
		return "./";
	else
	{
		if (path [path.size () -1 ] != '/')
			return path + '/';
		else
			return path;
	}
}

class GuardHandle {
public:
	GuardHandle (int h) :
		handle (h)
	{ }
	~GuardHandle ()
	{
		if (handle != -1)
			close (handle);
	}
	void release ()
	{
		handle= -1;
	}
private:
	int handle;

	GuardHandle (const GuardHandle &); // Forbidden
	void operator = (const GuardHandle &); // Forbidden
};


} // namespace


//***********************************************


// Value used for system trap.

const byte trapbyte= 0x76; // HALT code.

// Code for JP nn instruction, used in jump tables.
const byte jp_code= 0xC3;
// Code for RET instruction, used in .COM header
const byte ret_code= 0xC9;

const byte default_output_delimiter= '$';
const byte eof_char= 0x1A;

const byte ErrModeReturn=        0xFF;
const byte ErrModeDisplayReturn= 0xFE;
const byte ErrModeDefault=       0x00; // Or any other.

const z80word cpm_page_size= 0x0100;


// Address in page 0.

const z80word reset_call= 0;
const z80word bdos_call= 5;
const z80word default_fcb_1= 0x005C;
const z80word default_fcb_2= 0x006C;
const z80word default_pass_1= 0x0051;
const z80word default_pass_2= 0x0054;
const z80word default_dma_addr= 0x0080;
const z80word cmd_args= 0x0080;

// TPA begins in page 1.
const z80word tpa_start_addr= cpm_page_size;


// BIOS addresses.

const z80word default_stack_size= 64; // 32 16 bit entries.
const z80word default_stack= 0xFFFD;
const z80word default_stack_base= default_stack - default_stack_size;

const z80word num_bios_calls_2_2= 17; // CP/M 2.2 bios calls
const z80word num_bios_calls= 33; // CP/M Plus bios calls

const z80word bios_traps= default_stack_base - num_bios_calls;

// bios_base must be at page start.

const z80word aux_bios_base= bios_traps - num_bios_calls * 3;
const z80word bios_base= (aux_bios_base / cpm_page_size) * cpm_page_size;

const byte biosTIME= 0x1A;


// SCB

// Constants for several SCB variables.

const byte SCB_CPMVER= 0x05;
const byte SCB_CLPSUB= 0x0F;
const byte SCB_CLPRET= 0x10;
const byte SCB_CCPDSK= 0x13;
const byte SCB_CCPUSR= 0x14;
const byte SCB_CCPFL1= 0x17;
const byte SCB_CCPFL2= 0x18;
const byte SCB_CCPFL3= 0x19;
const byte SCB_PAGWID= 0x1A;
const byte SCB_PAGLEN= 0x1C;
const byte SCB_OUTDLM= 0x37;
const byte SCB_SELF=   0x3A;
const byte SCB_CRDMA=  0x3C;
const byte SCB_CRDSK=  0x3E;
const byte SCB_VINFO=  0x3F;
const byte SCB_RESEL=  0x41;
const byte SCB_FX=     0x43;
const byte SCB_USRCD=  0x44;
const byte SCB_MLTIO=  0x4A;
const byte SCB_ERMDE=  0x4B;
const byte SCB_DRVSEA= 0x4C;
const byte SCB_DRVTMP= 0x50;
const byte SCB_ERDSK=  0x51;
const byte SCB_DATE=   0x58;
const byte SCB_HOUR=   0x5A;
const byte SCB_MIN=    0x5B;
const byte SCB_SEC=    0x5C;
const byte SCB_CMBADR= 0x5D;
const byte SCB_MXTPA=  0x62;

// Evaluate scb position in memory.

const z80word scb_documented_size= 0x64;
const z80word scb_undoc_size= 0x34;
const z80word scb_size= scb_documented_size + scb_undoc_size;

const z80word scb_undoc_offset= 0x68;
const z80word scb_documented_offset= scb_undoc_offset + scb_undoc_size;

const z80word aux_scb_base= bios_base - scb_size - scb_undoc_offset;
const z80word scb_base= (aux_scb_base / cpm_page_size) * cpm_page_size;

const z80word scb_undoc_base= scb_base + scb_undoc_offset;
const z80word scb_documented_base= scb_base + scb_documented_offset;
const z80word scb_end= scb_documented_base + scb_documented_size;


// BDOS addresses.

// At the moment one page is enough for BDOS.
const z80word bdos_base_default= scb_base - cpm_page_size;

// Jump to BDOS must be at offset 6 from bdos_base_default
const z80word bdos_entry_offset= 6;

// At bdos entry address there are 3 bytes that will be set with a
// jump instruction to the position of the trap. The trap is placed
// in the next byte.
const z80word bdos_trap_addr= bdos_base_default + bdos_entry_offset + 3;

// Put here also the other traps.
const z80word endrun_trap_addr= bdos_trap_addr + 1;
const z80word endinit_trap_addr= endrun_trap_addr + 1;

// Bdos first page used as buffer for command line
const z80word default_console_buffer_size= 128;
const z80word default_console_buffer= endinit_trap_addr + 1;


// Fcb fields and auxiliary data.

const z80word fcb_extension= 12;
const z80word fcb_current_record= 32;
const z80word fcb_random_record= 33;
const z80word records_in_extension= 128;

const z80word sector_size= 128;

const size_t max_disks= 16; // Valid disks are A: to P:
const size_t invalid_fcb_disk= size_t (-1);


// RSX header fields.

const z80word rsx_serial= 0;
const z80word rsx_jump_start= 6;
const z80word rsx_start= 7;
const z80word rsx_jump_next= 9;
const z80word rsx_next= 10;
const z80word rsx_prev= 12;
const z80word rsx_remove= 14;
const z80word rsx_nonbank= 15;
const z80word rsx_name= 16;
const z80word rsx_loader= 24;
const z80word rsx_reserved= 25;


//*********************************************************
//		OpenFile
//*********************************************************


class OpenFile {
public:
	OpenFile (int handle_n);
	OpenFile (const OpenFile & of);
	~OpenFile ();
	int gethandle () const;
	void close ();
	int read (void * buffer, int count);
	int write (void * buffer, int count);
	long lseek (long offset, int whence);
private:
	void operator = (OpenFile &); // Forbidden
	class File;
	File * pf;
};

class OpenFile::File {
public:
	File (int handle_n);
	void addref ();
	void delref ();

	int gethandle () const;
	void open (int handle_n);
	void close ();
	int read (void * buffer, int count);
	int write (void * buffer, int count);
	long lseek (long offset, int whence);
private:
	File (const File &); // Forbidden
	void operator = (const File &); // Forbidden
	~File ();

	class AvoidWarning { };
	friend class AvoidWarning;

	size_t refcount;
	int handle;
};


OpenFile::File::File (int handle_n) :
	refcount (1),
	handle (handle_n)
{
	if (handle == -1)
		throw logic_error ("Open file with invalid handle");
}

OpenFile::File::~File ()
{
	close ();
}

void OpenFile::File::addref ()
{
	++refcount;
}

void OpenFile::File::delref ()
{
	if (--refcount == 0)
		delete this;
}

int OpenFile::File::gethandle () const
{
	return handle;
}

void OpenFile::File::close ()
{
	if (handle != -1)
	{
		::close (handle);
		handle= -1;
	}
}

int OpenFile::File::read (void * buffer, int count)
{
	return ::read (handle, buffer, count);
}

int OpenFile::File::write (void * buffer, int count)
{
	return ::write (handle, buffer, count);
}

long OpenFile::File::lseek (long offset, int whence)
{
	return ::lseek (handle, offset, whence);
}


OpenFile::OpenFile (int handle_n) :
	pf (new File (handle_n) )
{
}

OpenFile::OpenFile (const OpenFile & of) :
	pf (of.pf)
{
	pf->addref ();
}

OpenFile::~OpenFile ()
{
	pf->delref ();
}

int OpenFile::gethandle () const
{
	return pf->gethandle ();
}

void OpenFile::close ()
{
	pf->close ();
}

int OpenFile::read (void * buffer, int count)
{
	pf->read (buffer, count);
}

int OpenFile::write (void * buffer, int count)
{
	return pf->write (buffer, count);
}

long OpenFile::lseek (long offset, int whence)
{
	return pf->lseek (offset, whence);
}


//*********************************************************
//	BdosErr: launch BDOS errors as exceptions.
//*********************************************************


class BdosErrUncatched : public runtime_error {
public:
	BdosErrUncatched () :
		runtime_error ("BDOS error not catched")
	{ }
};

class BdosErr : public BdosErrUncatched {
public:
	BdosErr (byte h, byte l) :
		bdoserrcode ( (static_cast <z80word> (h) << 8) |
			static_cast <z80word> (l) )
	{ }
	BdosErr (z80word reghl) :
		bdoserrcode (reghl)
	{ }
	virtual bool serious () const { return false; }
	z80word getcode () const;
private:
	z80word bdoserrcode;
};


z80word BdosErr::getcode () const
{
	return bdoserrcode;
}

class BdosErrSerious : public BdosErr {
public:
	BdosErrSerious (byte h, byte l) :
		BdosErr (h, l)
	{ }
	BdosErrSerious (z80word reghl) :
		BdosErr (reghl)
	{ }
	virtual bool serious () const { return true; }
};


const z80word ErrNoFileMatch=       0x00FF;
const z80word ErrDiskIO=            0x01FF;
const z80word ErrInvalidDrive=      0x04FF;
const z80word ErrFileAlreadyExists= 0x08FF;
const z80word ErrInvalidFCB=        0x0009;


//*********************************************************
//		CPM::Internal
//*********************************************************


CpmOptions::CpmOptions () :
	debugz80 (false),
	debugsystem (false),
	prefix ('\0'),
	tpaend (0),
	default_disk ('A')
{
}

void CpmOptions::setdebugz80 (bool deb)
{
	debugz80= true;
}

void CpmOptions::setdebugsystem (bool deb)
{
	debugsystem= true;
}

void CpmOptions::setprefix (char newprefix)
{
	prefix= newprefix;
}

void CpmOptions::settpaend (unsigned short newtpaend)
{
	tpaend= newtpaend;
}

void CpmOptions::setdefaultdisk (char diskletter)
{
	default_disk= cpmupper (static_cast <byte> (diskletter) );
}


class CPM::Internal : public Cpu {
public:
	Internal (Console & console_n, Printer & printer_n,
		const CpmOptions & options);
	virtual ~Internal ();

	void setdisk (char disk, string path);
	char newdisk (string path);

	void interactive ();
	void runprogram (const string & filename, const string & args);
	void runcommandline (const string & commandline);

	enum TrapReturn {
		TrapContinue, TrapRet, TrapEndProgram, TrapEndInitRsx
	};
	typedef TrapReturn (CPM::Internal::* TrapFunc) (void);
	static TrapFunc gettrapfunc (z80word addr);
private:
	static const char default_bprefix= '@';

	Console & console;
	Printer & printer;

	bool debugz80_flag;
	bool debugsystem_flag;
	bool debugz80;
	bool debugsystem;

	typedef map <z80word, TrapFunc> trapfuncmap_t;
	typedef map <byte, TrapFunc> bdosmap_t;

	static trapfuncmap_t trapfuncmap;
	static bdosmap_t bdosmap;
	static bool funcmaps_inited;
	static bool funcmaps_init ();

	z80word bdos_base;
	z80word bdos_jump;

	//z80word program_return_code;

	//byte output_delimiter;

	char flag_disk_exist [max_disks];
	string disk_path [max_disks];

	void setprefix (char newprefix);

	void set_tpaend (z80word new_tpaend);
	z80word get_tpaend ();
	void set_native_tpaend (z80word new_tpaend);

	//size_t default_disk;

	void set_default_disk (size_t newdisk);
	void set_default_disk_letter (byte letter);
	size_t get_default_disk ();
	byte get_default_disk_letter ();

	void set_user (byte usernum);
	byte get_user ();

	void set_dma (z80word newdma);
	z80word get_dma ();

	void set_multi_sector (byte count);
	byte get_multi_sector ();

	byte get_error_mode ();

	void set_default_stack (z80word retaddr= 0);

	void makedisk (size_t disknum, const string & path);
	bool disk_exist (size_t disknum);
	string pathfromdisknum (size_t disknum);
	string pathfromdisknum_check (size_t disknum);

	void close_all ();

	// Open files.

	// Some programs, specifically Hi-Tech C, move the FCB
	// in memory after open it and continues using the copy.
	// Thus the memory address of the FCB can't be used
	// as index.
	#define MAP_BY_NAME

	#ifndef MAP_BY_NAME

	typedef map <z80word, OpenFile> fcbmap_t;

	#else

	//typedef map <string, OpenFile> fcbmap_t;
	typedef map <FcbFile, OpenFile> fcbmap_t;

	#endif
	fcbmap_t fcbmap;

	OpenFile & getopenfile (FcbFile & fcb);
	void insertopenfile (FcbFile & fcb, int handle);

	void close_fcb (FcbFile & fcb);
	void close_fcb_if (FcbFile & fcb);

	// Search for files.

	typedef map <z80word, FindFile> findfilemap_t;
	findfilemap_t findfilemap;

	void addfinder (z80word fcbaddr, FindFile & ff);
	void delfinder (z80word fcbaddr);
	void delallfinders ();
	FindFile & getfinder (z80word fcbaddr);

	void clean_fcb_entry (z80word fcbaddr);
	bool is_blank_fcb (z80word fcbaddr);
	bool is_blank_ext_fcb (z80word fcbaddr);

	OpenFile * getopenfile (z80word fcbaddr);
	void insertopenfile (z80word fcbaddr, int handle);

	//byte multi_sector_count;

	void memwritejump (z80word address, z80word destination);

	void read_console_buffer (z80word buffer);

	unsigned long fcb_get_random_record (z80word fcbaddr);

	void set_filename (FcbFile & fcbfile, z80word fcbaddr);

	size_t disknum_from_fcbnum (size_t fcbdrive);
	size_t disknum_from_fcb (z80word fcbaddr);
	string getdisk_from_fcb_check (z80word fcbaddr);
	string filename_from_fcb (z80word fcbaddr);
	string filespec_from_fcb (z80word fcbaddr);
	string extension_from_fcb (z80word fcbaddr);
	string filepath_from_fcb (z80word fcbaddr);
	string filepath_from_fcb_check (z80word fcbaddr);

	void setcomextension (z80word fcbaddr);
	bool open_file (z80word fcbaddr);
	bool make_file (z80word fcbaddr);
	//z80word dma_addr;

	z80word getSCB (byte scboff);
	z80word getbSCB (byte scboff);
	void setbSCB (byte scboff, byte value);
	void setwSCB (byte scboff, z80word value);

	//*********** Trap functions **********

	// End run trap.
	TrapReturn end_run ();
	TrapReturn end_rsx_init ();

	// BIOS functions.
	TrapReturn biosfuncBOOT ();
	TrapReturn biosfuncWBOOT ();
	TrapReturn biosfuncCONST ();
	TrapReturn biosfuncCONIN ();
	TrapReturn biosfuncCONOUT ();
	TrapReturn biosfuncLIST ();
	TrapReturn biosfuncAUXOUT ();
	TrapReturn biosfuncAUXIN ();
	TrapReturn biosfuncHOME ();
	TrapReturn biosfuncSELDSK ();
	TrapReturn biosfuncSETTRK ();
	TrapReturn biosfuncSETSEC ();
	TrapReturn biosfuncSETDMA ();
	TrapReturn biosfuncREAD ();
	TrapReturn biosfuncWRITE ();
	TrapReturn biosfuncLISTST ();
	TrapReturn biosfuncSECTRN ();
	TrapReturn biosfuncCONOST ();
	TrapReturn biosfuncAUXIST ();
	TrapReturn biosfuncAUXOST ();
	TrapReturn biosfuncDEVTBL ();
	TrapReturn biosfuncDEVINI ();
	TrapReturn biosfuncDRVTBL ();
	TrapReturn biosfuncMULTIO ();
	TrapReturn biosfuncFLUSH ();
	TrapReturn biosfuncMOVE ();
	TrapReturn biosfuncTIME ();
	TrapReturn biosfuncSELMEM ();
	TrapReturn biosfuncSETBNK ();
	TrapReturn biosfuncXMOVE ();
	TrapReturn biosfuncUSERF ();
	TrapReturn biosfuncRESERV1 ();
	TrapReturn biosfuncRESERV2 ();

	static const TrapFunc bios_table [];

	// RST.
	TrapReturn UnsupportedRst ();

	// BDOS functions.
	void bdos_return_hl (byte h, byte l);
	void bdos_return_hl (z80word reghl);
	void bdos_return_a (byte a);

	void setsearchresult (const string & result);
	void setsearchresult (const FcbFile & fcbfile);

	TrapReturn bdos ();
	z80word get_bdos_fcb ();

	TrapReturn bdos_system_reset ();                     // 00
	TrapReturn bdos_console_input ();                    // 01
	TrapReturn bdos_console_output ();                   // 02
	TrapReturn bdos_list_output ();                      // 05
	TrapReturn bdos_direct_console_io ();                // 06
	TrapReturn bdos_write_string ();                     // 09
	TrapReturn bdos_read_console_buffer ();              // 0A
	TrapReturn bdos_get_console_status ();               // 0B
	TrapReturn bdos_return_version_number ();            // 0C
	TrapReturn bdos_reset_disk_system ();                // 0D
	TrapReturn bdos_select_disk ();                      // 0E
	TrapReturn bdos_open_file ();                        // 0F
	TrapReturn bdos_close_file ();                       // 10
	TrapReturn bdos_search_for_first ();                 // 11
	TrapReturn bdos_search_for_next ();                  // 12
	TrapReturn bdos_delete_file ();                      // 13
	TrapReturn bdos_read_sequential ();                  // 14
	TrapReturn bdos_write_sequential ();                 // 15
	TrapReturn bdos_make_file ();                        // 16
	TrapReturn bdos_rename_file ();                      // 17
	TrapReturn bdos_return_current_disk ();              // 19
	TrapReturn bdos_set_dma_address ();                  // 1A
	TrapReturn bdos_map_read_only ();                    // 1D
	TrapReturn bdos_set_file_attributes ();              // 1E
	TrapReturn bdos_get_set_user_code ();                // 20
	TrapReturn bdos_read_random ();                      // 21
	TrapReturn bdos_write_random ();                     // 22
	TrapReturn bdos_compute_file_size ();                // 23
	TrapReturn bdos_set_random_record ();                // 24
	TrapReturn bdos_set_multi_sector_count ();           // 2C
	TrapReturn bdos_set_bdos_error_mode ();              // 2D
	TrapReturn bdos_get_set_system_control_block ();     // 31
	TrapReturn bdos_direct_bios_calls ();                // 32
	TrapReturn bdos_load_overlay ();                     // 3B
	TrapReturn bdos_call_resident_system_extension ();   // 3C
	TrapReturn bdos_free_blocks ();                      // 62
	TrapReturn bdos_truncate_file ();                    // 63
	TrapReturn bdos_get_read_file_date_and_pass_mode (); // 66
	TrapReturn bdos_get_date_and_time ();                // 69
	TrapReturn bdos_get_set_program_return_code ();      // 6C
	TrapReturn bdos_get_set_console_mode ();             // 6D
	TrapReturn bdos_get_set_output_delimiter ();         // 6E
	TrapReturn bdos_parse_filename ();                   // 98

	z80word parsefilename (z80word name,
		size_t & fcbnum, string & file, string & ext, string & pass);

public:
	void parsefilefcb (z80word & name, z80word fcbaddr);
	void cleardefaultfcb (z80word fcbaddr, z80word pass);
	void parsefilecmdline (z80word & pos, z80word endline,
		z80word fcbaddr, z80word pass);
	void setcmd_args (const string & args);
	void setarguments (const string & args);
	void showcurrentstate ();

	void showbdoserror (z80word errcode);

	TrapReturn runinstruction ();
	void runner ();
	void run ();
	TrapReturn checkrunner ();
	TrapReturn checkrun ();

	void prepareruntransient ();
	void endruntransient ();
	void runtransient ();
	void attachloader ();
	void attachrsx (z80word image, z80word len);
	bool loadfiletpa (int handle);
	bool loadprltpa (int handle);
	void tpastart ();
	string getcommandline ();
	void debugger ();
	bool loadprogram (string cmdupper);

private:
	// Built in commands.

	char bprefix;

	bool builtinEXIT (const string &);
	bool builtinPOKE (const string & strcmd);
	bool builtinGO (const string & strcmd);
	bool builtinLOAD (const string & strcmd);
	bool builtinDEBUG (const string & strcmd);
	bool builtinDIR (const string & strcmd);
	bool builtinERA (const string & strcmd);
	bool builtinUSER (const string & strcmd);

	typedef bool (CPM::Internal::* builtinfunc_t) (const string &);
	typedef map <string, builtinfunc_t> built_t;

	// This is to allow make const the builtin tables while
	// initializing them at runtime.

	class Initbuilt {
		Initbuilt ();
	public:
		built_t pr_built;
		built_t built;
		static const Initbuilt instance;
	};
	static const built_t & pr_built;
	static const built_t & built;
	static const built_t::const_iterator prbtend;
	static const built_t::const_iterator btend;

	builtinfunc_t find_builtin (const string & comm);
};


// Static vars.

const CPM::Internal::TrapFunc CPM::Internal::bios_table []=
{
	& CPM::Internal::biosfuncBOOT,
	& CPM::Internal::biosfuncWBOOT,
	& CPM::Internal::biosfuncCONST,
	& CPM::Internal::biosfuncCONIN,
	& CPM::Internal::biosfuncCONOUT,
	& CPM::Internal::biosfuncLIST,
	& CPM::Internal::biosfuncAUXOUT,
	& CPM::Internal::biosfuncAUXIN,
	& CPM::Internal::biosfuncHOME,
	& CPM::Internal::biosfuncSELDSK,
	& CPM::Internal::biosfuncSETTRK,
	& CPM::Internal::biosfuncSETSEC,
	& CPM::Internal::biosfuncSETDMA,
	& CPM::Internal::biosfuncREAD,
	& CPM::Internal::biosfuncWRITE,
	& CPM::Internal::biosfuncLISTST,
	& CPM::Internal::biosfuncSECTRN,
	& CPM::Internal::biosfuncCONOST,
	& CPM::Internal::biosfuncAUXIST,
	& CPM::Internal::biosfuncAUXOST,
	& CPM::Internal::biosfuncDEVTBL,
	& CPM::Internal::biosfuncDEVINI,
	& CPM::Internal::biosfuncDRVTBL,
	& CPM::Internal::biosfuncMULTIO,
	& CPM::Internal::biosfuncFLUSH,
	& CPM::Internal::biosfuncMOVE,
	& CPM::Internal::biosfuncTIME,
	& CPM::Internal::biosfuncSELMEM,
	& CPM::Internal::biosfuncSETBNK,
	& CPM::Internal::biosfuncXMOVE,
	& CPM::Internal::biosfuncUSERF,
	& CPM::Internal::biosfuncRESERV1,
	& CPM::Internal::biosfuncRESERV2
};

CPM::Internal::trapfuncmap_t CPM::Internal::trapfuncmap;

CPM::Internal::bdosmap_t CPM::Internal::bdosmap;

bool CPM::Internal::funcmaps_init ()
{
	trapfuncmap [endrun_trap_addr]=   & Internal::end_run;
	trapfuncmap [endinit_trap_addr]=  & Internal::end_rsx_init;

	// BIOS traps.

	for (size_t i= 0; i < num_bios_calls; ++i)
		trapfuncmap [bios_traps + i]= bios_table [i];

	// RST.

	for (int i= 0x08; i <= 0x38; i+= 0x08)
		trapfuncmap [i]= & Internal::UnsupportedRst;


	// BDOS functions.

	trapfuncmap [bdos_trap_addr]= & Internal::bdos;

	bdosmap [0x00]= & Internal::bdos_system_reset;
	bdosmap [0x01]= & Internal::bdos_console_input;
	bdosmap [0x02]= & Internal::bdos_console_output;
	bdosmap [0x05]= & Internal::bdos_list_output;
	bdosmap [0x06]= & Internal::bdos_direct_console_io;
	bdosmap [0x09]= & Internal::bdos_write_string;
	bdosmap [0x0A]= & Internal::bdos_read_console_buffer;
	bdosmap [0x0B]= & Internal::bdos_get_console_status;
	bdosmap [0x0C]= & Internal::bdos_return_version_number;
	bdosmap [0x0D]= & Internal::bdos_reset_disk_system;
	bdosmap [0x0E]= & Internal::bdos_select_disk;
	bdosmap [0x0F]= & Internal::bdos_open_file;
	bdosmap [0x10]= & Internal::bdos_close_file;
	bdosmap [0x11]= & Internal::bdos_search_for_first;
	bdosmap [0x12]= & Internal::bdos_search_for_next;
	bdosmap [0x13]= & Internal::bdos_delete_file;
	bdosmap [0x14]= & Internal::bdos_read_sequential;
	bdosmap [0x15]= & Internal::bdos_write_sequential;
	bdosmap [0x16]= & Internal::bdos_make_file;
	bdosmap [0x17]= & Internal::bdos_rename_file;
	bdosmap [0x19]= & Internal::bdos_return_current_disk;
	bdosmap [0x1A]= & Internal::bdos_set_dma_address;
	bdosmap [0x1D]= & Internal::bdos_map_read_only;
	bdosmap [0x1E]= & Internal::bdos_set_file_attributes;
	bdosmap [0x20]= & Internal::bdos_get_set_user_code;
	bdosmap [0x21]= & Internal::bdos_read_random;
	bdosmap [0x22]= & Internal::bdos_write_random;
	bdosmap [0x23]= & Internal::bdos_compute_file_size;
	bdosmap [0x24]= & Internal::bdos_set_random_record;
	bdosmap [0x2C]= & Internal::bdos_set_multi_sector_count;
	bdosmap [0x2D]= & Internal::bdos_set_bdos_error_mode;
	bdosmap [0x31]= & Internal::bdos_get_set_system_control_block;
	bdosmap [0x32]= & Internal::bdos_direct_bios_calls;
	bdosmap [0x3B]= & Internal::bdos_load_overlay;
	bdosmap [0x3C]= & Internal::bdos_call_resident_system_extension;
	bdosmap [0x62]= & Internal::bdos_free_blocks;
	bdosmap [0x63]= & Internal::bdos_truncate_file;
	bdosmap [0x66]= & Internal::bdos_get_read_file_date_and_pass_mode;
	bdosmap [0x69]= & Internal::bdos_get_date_and_time;
	bdosmap [0x6C]= & Internal::bdos_get_set_program_return_code;
	bdosmap [0x6D]= & Internal::bdos_get_set_console_mode;
	bdosmap [0x6E]= & Internal::bdos_get_set_output_delimiter;
	bdosmap [0x98]= & Internal::bdos_parse_filename;

	return true;
}

bool CPM::Internal::funcmaps_inited= CPM::Internal::funcmaps_init ();

//CPM::Internal::Internal (Console & console_n, Printer & printer_n,
//		bool debugz80_n, bool debugsystem_n) :
CPM::Internal::Internal (Console & console_n, Printer & printer_n,
		const CpmOptions & options) :
	console (console_n),
	printer (printer_n),

	debugz80_flag (options.debugz80),
	debugsystem_flag (options.debugsystem),
	debugz80 (false),
	debugsystem (false),
	//program_return_code (0),
	//output_delimiter (default_output_delimiter),
	//default_disk (static_cast <char> (options.default_disk - 'A') ),

	//multi_sector_count (1),
	//bprefix (default_bprefix)
	bprefix (options.prefix)
{
	ASSERT ( (sizeof (bios_table) / sizeof (* bios_table) ) ==
		num_bios_calls);

	if (debugsystem_flag)
	{
		cerr << "Cpu initialized." << crlf <<
			"\tMemory at " << static_cast <void *> (mem) <<
			"\tSCB at " << hex4 (scb_documented_base) <<
			crlf;
	}

	clear_all_mem (0xE7); // RST 20h, for testing.

	// Clear SCB.
	for (z80word i= scb_undoc_base; i < scb_end; ++i)
		memwrite (i, 0);

	// Initialize SCB values.

	set_native_tpaend (options.tpaend);

	setbSCB (SCB_CPMVER, 0x31); // CP/M, vesrion 3.1
	setbSCB (SCB_CLPSUB, 0); // Submit file drive.
	setwSCB (SCB_CLPRET, 0); // Program return code.
	// SCB_CCPDSK is set by set_default_disk
	// SCB_CCPUSR is set by set_user

	setbSCB (SCB_CCPFL1, 0x00);
	setbSCB (SCB_CCPFL2, 0x10); // SUB, then COM.
	setbSCB (SCB_CCPFL3, 0x00);
	setbSCB (SCB_PAGWID, static_cast <byte> (console.columns () - 1) );
	setbSCB (SCB_PAGLEN, static_cast <byte> (console.lines () ) );
	setbSCB (SCB_OUTDLM, default_output_delimiter);
	setwSCB (SCB_SELF, scb_documented_base);

	set_dma (default_dma_addr); // SCB_CRDMA

	set_default_disk_letter (options.default_disk); // SCB_CRDSK

	set_user (0); // SCB_USRCD

	set_multi_sector (1); // SCB_MLTIO

	setbSCB (SCB_ERMDE, ErrModeDefault);
	// Set drive search chain.
	setbSCB (SCB_DRVSEA, 0); // Current disk.
	setbSCB (SCB_DRVSEA + 1, 0xFF); // No more search.
	setbSCB (SCB_DRVSEA + 2, 0xFF); // Just in case...
	setbSCB (SCB_DRVSEA + 3, 0xFF);
	setbSCB (SCB_DRVTMP, 0); // Temporary: current disk.

	// SCB clock not initialized.

	setwSCB (SCB_CMBADR, 0); // Non banked system.

	// Set traps.

	for (trapfuncmap_t::iterator it= trapfuncmap.begin ();
		it != trapfuncmap.end ();
		++it)
	{
		memwrite (it->first, trapbyte);
	}

	// BIOS Jump table: Jumps to the position of the traps.
	for (int i= 0; i < num_bios_calls; ++i)
		memwritejump (bios_base + 3 * i, bios_traps + i);

	// Set Warm reset call in page 0.
	memwritejump (reset_call, bios_base + 3);

	bdos_jump= bdos_base + bdos_entry_offset;
	// Set BDOS Jump to trap.
	memwritejump (bdos_jump, bdos_trap_addr);

	// Set BDOS Call in page 0.
	memwritejump (bdos_call, bdos_jump);

	// Set disk A:
	setdisk ('A', string () );
}

CPM::Internal::~Internal ()
{
	close_all ();
}

void CPM::Internal::setprefix (char newprefix)
{
	bprefix= newprefix;
}

void CPM::Internal::set_tpaend (unsigned short new_tpaend)
{
	if (debugsystem_flag)
	{
		cerr << "Setting active TPA end to: " <<
			hex4 (new_tpaend) << crlf;
		if (new_tpaend <= tpa_start_addr ||
			new_tpaend > bdos_base_default + bdos_entry_offset)
		{
			cerr << "\tWARNING: TPA end out of limits." << crlf;
		}
	}

	setwSCB (SCB_MXTPA, new_tpaend);
}

z80word CPM::Internal::get_tpaend ()
{
	return getSCB (SCB_MXTPA);
}

void CPM::Internal::set_native_tpaend (z80word new_tpaend)
{
	if (new_tpaend == 0)
		new_tpaend= bdos_base_default;
	else
	{
		// Adjust to begin of page.
		new_tpaend&= 0xFF00;
	}

	if (debugsystem_flag)
		cerr << "Setting TPA end to: " << hex4 (new_tpaend) << crlf;

	if (new_tpaend <= tpa_start_addr || new_tpaend > bdos_base_default)
	{
		throw runtime_error ("TPA end limit invalid");
	}

	set_tpaend (new_tpaend + bdos_entry_offset);
	bdos_base= new_tpaend;
}

void CPM::Internal::set_default_disk (size_t newdisk)
{
	setbSCB (SCB_CRDSK, static_cast <byte> (newdisk) );
	setbSCB (SCB_CCPDSK, static_cast <byte> (newdisk) );
}

void CPM::Internal::set_default_disk_letter (byte letter)
{
	set_default_disk (disknumfromletter (letter) );
}

size_t CPM::Internal::get_default_disk ()
{
	return static_cast <size_t> (getbSCB (SCB_CRDSK) );
}

byte CPM::Internal::get_default_disk_letter ()
{
	return letterfromdisknum (get_default_disk () );
}

void CPM::Internal::set_user (byte usernum)
{
	setbSCB (SCB_USRCD, usernum);
	setbSCB (SCB_CCPUSR, usernum);
}

byte CPM::Internal::get_user ()
{
	return getbSCB (SCB_USRCD);
}

void CPM::Internal::set_dma (z80word newdma)
{
	if (debugsystem)
		cerr << "DMA address set to " <<
			hex4 (newdma) << crlf;
	setwSCB (SCB_CRDMA, newdma);
}

z80word CPM::Internal::get_dma ()
{
	return getSCB (SCB_CRDMA);
}

void CPM::Internal::set_multi_sector (byte count)
{
	setbSCB (SCB_MLTIO, count);
}

byte CPM::Internal::get_multi_sector ()
{
	return getbSCB (SCB_MLTIO);
}

byte CPM::Internal::get_error_mode ()
{
	return getbSCB (SCB_ERMDE);
}

void CPM::Internal::set_default_stack (z80word retaddr)
{
	set_sp (default_stack);
	memwritew (default_stack, retaddr);
}

void CPM::Internal::makedisk (size_t disknum, const string & path)
{
	ASSERT (disknum < max_disks);

	if (debugsystem_flag)
		cerr << "Setting disk " <<
			static_cast <char> (disknum + 'A')
			<< ": to " << path << crlf;

	disk_path [disknum]= path;
	flag_disk_exist [disknum]= 1;
}

void CPM::Internal::setdisk (char disk, string path)
{
	byte d= static_cast <byte> (disk);

	//d= cpmupper (d);
	//if (d < 'A' || d >= 'A' + max_disks)
	//	throw runtime_error ("Invalid disk letter");
	//size_t disknum= d - 'A';

	size_t disknum= disknumfromletter (disk);
	if (disknum >= max_disks)
		throw runtime_error ("Invalid disk letter");

	makedisk (disknum, normalize_dir (path) );
}

char CPM::Internal::newdisk (string path)
{
	path= normalize_dir (path);

	size_t disknum= invalid_fcb_disk;
	for (size_t i= 0; i < max_disks; ++i)
	{
		if (flag_disk_exist [i] && disk_path [i] == path)
			disknum= i;
	}

	if (disknum == invalid_fcb_disk)
	{
		for (size_t i= 0; i < max_disks; ++i)
		{
			if (! flag_disk_exist [i] )
				disknum= i;
		}
		if (disknum == invalid_fcb_disk)
			throw runtime_error ("Not enough disks");

		makedisk (disknum, path);
	}

	return static_cast <char> (disknum) + 'A';
}

bool CPM::Internal::disk_exist (size_t disknum)
{
	if (disknum < max_disks)
		return flag_disk_exist [disknum];
	else
		return false;
}

string CPM::Internal::pathfromdisknum (size_t disknum)
{
	// Must be called with a valid disk number.
	ASSERT (disknum < max_disks);
	ASSERT (flag_disk_exist [disknum] );

	return disk_path [disknum];
}

string CPM::Internal::pathfromdisknum_check (size_t disknum)
{
	if (disknum >= max_disks || ! flag_disk_exist [disknum] )
		throw BdosErrSerious (ErrInvalidDrive);
	return disk_path [disknum];
}

z80word CPM::Internal::getSCB (byte scboff)
{
	ASSERT (scboff < scb_documented_size);

	return memreadw (scb_documented_base + scboff);
}

z80word CPM::Internal::getbSCB (byte scboff)
{
	ASSERT (scboff < scb_documented_size);

	return memread (scb_documented_base + scboff);
}

void CPM::Internal::setbSCB (byte scboff, byte value)
{
	ASSERT (scboff < scb_documented_size);

	memwrite (scb_documented_base + scboff, value);
}

void CPM::Internal::setwSCB (byte scboff, z80word value)
{
	ASSERT (scboff < scb_documented_size - 1);

	memwritew (scb_documented_base + scboff, value);
}

void CPM::Internal::addfinder (z80word fcbaddr, FindFile & ff)
{
	//findfilemap [fcbaddr]= & ff;
	findfilemap.insert (make_pair (fcbaddr, ff) );
}

void CPM::Internal::delfinder (z80word fcbaddr)
{
	findfilemap_t::iterator it= findfilemap.find (fcbaddr);
	if (it != findfilemap.end () )
	{
		findfilemap.erase (fcbaddr);
	}
}

void CPM::Internal::delallfinders ()
{
	findfilemap.clear ();
}

FindFile & CPM::Internal::getfinder (z80word fcbaddr)
{
	findfilemap_t::iterator it= findfilemap.find (fcbaddr);
	if (it != findfilemap.end () )
		return it->second;
	else
		throw BdosErr (ErrInvalidFCB);
}

void CPM::Internal::clean_fcb_entry (z80word fcbaddr)
{
	{
		#ifndef MAP_BY_NAME

		fcbmap_t::iterator it= fcbmap.find (fcbaddr);
		if (it != fcbmap.end () )
			fcbmap.erase (fcbaddr);

		#else

		//string str= filename_from_fcb (fcbaddr);
		//fcbmap_t::iterator it= fcbmap.find (str);

		FcbFile fcb (mem + fcbaddr);
		close_fcb_if (fcb);
		//fcbmap_t::iterator it= fcbmap.find (fcb);
		//if (it != fcbmap.end () )
		//	fcbmap.erase (it);

		#endif
	}
	delfinder (fcbaddr);
}

bool CPM::Internal::is_blank_fcb (z80word fcbaddr)
{
	for (size_t i= 1; i < 12; ++i)
		if (memread (fcbaddr + i) != ' ')
			return false;
	return true;
}

bool CPM::Internal::is_blank_ext_fcb (z80word fcbaddr)
{
	for (size_t i= 9; i < 12; ++i)
		if (memread (fcbaddr + i) != ' ')
			return false;
	return true;
}

OpenFile * CPM::Internal::getopenfile (z80word fcbaddr)
{
	#ifndef MAP_BY_NAME

	fcbmap_t::iterator it= fcbmap.find (fcbaddr);

	#else

	//fcbmap_t::iterator it= fcbmap.find (filename_from_fcb (fcbaddr) );
	FcbFile fcb (mem + fcbaddr);
	fcbmap_t::iterator it= fcbmap.find (fcb);

	#endif

	if (it == fcbmap.end () )
		return NULL;
	return & it->second;
}

void CPM::Internal::insertopenfile (z80word fcbaddr, int handle)
{
	#ifndef MAP_BY_NAME

	fcbmap.insert (make_pair (fcbaddr, OpenFile (handle) ) );

	#else

	string str= filename_from_fcb (fcbaddr);
	fcbmap.insert (make_pair (str, OpenFile (handle) ) );

	#endif
}


void CPM::Internal::close_all ()
{
	fcbmap.clear ();
	delallfinders ();
}

OpenFile & CPM::Internal::getopenfile (FcbFile & fcb)
{
	fcbmap_t::iterator it= fcbmap.find (fcb);
	if (it == fcbmap.end () )
		throw BdosErr (ErrInvalidFCB);
	return it->second;
}

void CPM::Internal::insertopenfile (FcbFile & fcb, int handle)
{
	fcbmap.insert (make_pair (fcb, OpenFile (handle) ) );
}

void CPM::Internal::close_fcb (FcbFile & fcb)
{
	fcbmap_t::iterator it= fcbmap.find (fcb);

	if (it == fcbmap.end () )
		throw BdosErr (ErrInvalidFCB);
	fcbmap.erase (it);
}

void CPM::Internal::close_fcb_if (FcbFile & fcb)
{
	fcbmap_t::iterator it= fcbmap.find (fcb);
	if (it != fcbmap.end () )
		fcbmap.erase (it);
}

void CPM::Internal::memwritejump (z80word address, z80word destination)
{
	memwrite (address, jp_code);
	memwritew (address + 1, destination);
}

void CPM::Internal::read_console_buffer (z80word buffer)
{
	const byte len= memread (buffer);
	byte i= 0;
	bool waiting= true;
	while (waiting)
	{
		char c= console.charin ();
		switch (c)
		{
		case '\r':
		case '\n':
			waiting= false;
			break;
		case 0x7F:
		case 0x08:
			if (i > 0)
			{
				console.left ();
				console.charout (' ');
				console.left ();
				--i;
			}
			else
				console.charout (7);
			break;
		case 0x01: // Ctrl-A
			if (i > 0)
			{
				console.left ();
				--i;
			}
			else
				console.charout (7);
			break;
		default:
			if (i < len)
			{
				console.charout (c);
				memwrite (buffer + 2 + i, c);
				++i;
			}
			else
				console.charout (7);
		}
	} 
	memwrite (buffer + 1, i);
}

unsigned long CPM::Internal::fcb_get_random_record (z80word fcbaddr)
{
	z80word a= fcbaddr + fcb_random_record;
	return memread (a) +
		(static_cast <unsigned long> (memread (a + 1) ) << 8) +
		(static_cast <unsigned long> (memread (a + 2) ) << 16);
}

size_t CPM::Internal::disknum_from_fcbnum (size_t fcbdrive)
{
	if (fcbdrive > max_disks)
		return invalid_fcb_disk;
	if (fcbdrive == 0)
	{
		//fcbdrive= default_disk;
		fcbdrive= get_default_disk ();
	}
	else
		--fcbdrive;
	return fcbdrive;
}

size_t CPM::Internal::disknum_from_fcb (z80word fcbaddr)
{
	return disknum_from_fcbnum (static_cast <size_t> (memread (fcbaddr) ) );
}

string CPM::Internal::getdisk_from_fcb_check (z80word fcbaddr)
{
	return pathfromdisknum_check (disknum_from_fcb (fcbaddr) );
}

void CPM::Internal::set_filename (FcbFile & fcbfile, z80word fcbaddr)
{
	fcbfile.setfile (mem + fcbaddr + 1);
}

string CPM::Internal::filename_from_fcb (z80word fcbaddr)
{
	z80word l;
	for (l= 9; l > 1; --l)
		if ( (memread (fcbaddr + l - 1) & 0x7F) != ' ')
			break;
	string str;
	for (z80word i= 1; i < l; ++i)
		str+= memread (fcbaddr + i) & 0x7F;
	for (l= 12; l > 9; --l)
		if (memread (fcbaddr + l - 1) != ' ')
			break;
	if (l > 9)
	{
		str+= '.';
		for (z80word i= 9; i < l; ++i)
			str+= memread (fcbaddr + i);
	}
	return str;
}

string CPM::Internal::filespec_from_fcb (z80word fcbaddr)
{
	string str;
	byte disk= memread (fcbaddr);
	if (disk != 0)
	{
		str= string (1, static_cast <char> (disk - 1 + 'A') )
			+ ':';
	}
	str+= filename_from_fcb (fcbaddr);
	return str;
}

string CPM::Internal::extension_from_fcb (z80word fcbaddr)
{
	z80word l;
	for (l= 12; l > 9; --l)
		if (memread (fcbaddr + l - 1) != ' ')
			break;
	string str;
	if (l > 9)
	{
		for (z80word i= 9; i < l; ++i)
			str+= memread (fcbaddr + i);
	}
	return str;
}

string CPM::Internal::filepath_from_fcb (z80word fcbaddr)
{
	return pathfromdisknum (disknum_from_fcb (fcbaddr) ) +
		filename_from_fcb (fcbaddr);
}

string CPM::Internal::filepath_from_fcb_check (z80word fcbaddr)
{
	return pathfromdisknum_check (disknum_from_fcb (fcbaddr) ) +
		filename_from_fcb (fcbaddr);
}

void CPM::Internal::setcomextension (z80word fcbaddr)
{
	memwrite (fcbaddr +  9, 'C');
	memwrite (fcbaddr + 10, 'O');
	memwrite (fcbaddr + 11, 'M');
}

bool CPM::Internal::open_file (z80word fcbaddr)
{
	if (debugsystem)
		cerr << "open_file" << crlf;

	clean_fcb_entry (fcbaddr);

	size_t disknum= disknum_from_fcb (fcbaddr);

	#if 0
	if (disknum == invalid_fcb_disk)
		throw BdosErr (0x04FF);

	if (! disk_exist (disknum) )
		throw BdosErr (0x04FF);
	#endif

	//if (! disk_exist (disknum) )
	//	return false;
	string filename= pathfromdisknum_check (disknum) +
		filename_from_fcb (fcbaddr);

	FcbFile fcbfile;
	set_filename (fcbfile, fcbaddr);

	if (debugsystem)
	{
		cerr << "Opening '" << fcbfile << "' as '" <<
			filename << '\'' << crlf;
	}

	int h= open (filename.c_str (), O_RDWR);
	if (h == -1)
		return false;

	GuardHandle guard (h);
	insertopenfile (fcbaddr, h);
	guard.release ();

	// Testing.
	memwrite (fcbaddr + fcb_current_record, 0);

	return true;
}

bool CPM::Internal::make_file (z80word fcbaddr)
{
	clean_fcb_entry (fcbaddr);

	size_t disknum= disknum_from_fcb (fcbaddr);
	string filename= pathfromdisknum_check (disknum) +
		filename_from_fcb (fcbaddr);

	if (debugsystem)
		cerr << "Making '" << filename << '\'' << crlf;

	int h= open (filename.c_str (), O_RDWR | O_CREAT | O_EXCL, 0666);
	if (h == -1)
	{
		int e= errno;
		if (debugsystem)
			cerr << "open failed with '" << filename <<
				"': " << strerror (e) << crlf;
		z80word code;
		switch (e)
		{
		case EEXIST:
			code= ErrFileAlreadyExists; break;
		default:
			code= ErrDiskIO;
		}
		throw BdosErrSerious (code);
	}

	GuardHandle guard (h);
	insertopenfile (fcbaddr, h);
	guard.release ();

	// Testing.
	memwrite (fcbaddr + fcb_current_record, 0);

	return true;
}

CPM::Internal::TrapFunc CPM::Internal::gettrapfunc (z80word addr)
{
	trapfuncmap_t::iterator it= trapfuncmap.find (addr);
	if (it == trapfuncmap.end () )
		return 0;
	else
		return it->second;
}


CPM::Internal::TrapReturn CPM::Internal::end_run ()
{
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::end_rsx_init ()
{
	return TrapEndInitRsx;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncBOOT ()
{
	if (debugsystem)
		cerr << crlf << "BIOS BOOT" << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncWBOOT ()
{
	if (debugsystem)
		cerr << crlf << "BIOS WBOOT" << crlf;

	// Remove RSX flagged for removing.

	//z80word previous= bdos_call + 1;
	z80word previous= get_tpaend () + 1;
	z80word actual= memreadw (previous);

	z80word aux= 0;
	bool some_remains= false;
	while (actual < bdos_base)
	{
		if (actual <= aux)
		{
			if (debugsystem)
			{
				cerr << "RSX chain is incorrect. "
					"Clearing all RSX" <<
					crlf;
			}
			memwritew (bdos_call + 1, bdos_jump);
			set_tpaend (bdos_jump);
			break;
		}
		aux= actual;
		z80word base= actual & 0xFF00;
		actual= memreadw (base + rsx_next);
		bool is_loader= memread (base + rsx_loader) == 0xFF;
		if (memread (base + rsx_remove) == 0xFF)
		{
			if (! is_loader || ! some_remains)
			{
				if (debugsystem_flag)
				{
					if (is_loader)
						cerr << "Removing LOADER" <<
							crlf;
					else
						cerr << "Removing RSX 	" <<
							hex4 (base) <<
							crlf;
				}
				memwritew (previous, actual);
			}
		}
		else
		{
			some_remains= true;
			previous= base + rsx_next;
			if (debugsystem_flag)
			{
				if (is_loader)
					cerr << "Not removing LOADER" <<
						crlf;
				else
					cerr << "Not removing RSX " <<
						hex4 (base) <<
						crlf;
			}
		}
	}

	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncCONST ()
{
	// Entry: none.
	// Output: A= 0 -> No character waiting, 0xFF -> Yes.

	//if (debugsystem)
	//	cerr << "CONST called." << crlf;

	set_a (console.charavailable () ? 0xFF : 0x00);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncCONIN ()
{
	// Entry: none.
	// Output: A= Character

	//cerr << "CONIN called." << crlf;

	char c= console.charin ();
	set_a (c);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncCONOUT ()
{
	// Entry: character in C.
	// Output: none.

	console.charout (get_c () );
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncLIST ()
{
	printer.put (get_c () );
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncAUXOUT ()
{
	cerr << crlf << "BIOS AUXOUT unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncAUXIN ()
{
	cerr << crlf << "BIOS AUXIN unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncHOME ()
{
	cerr << crlf << "BIOS HOME unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncSELDSK ()
{
	cerr << crlf << "BIOS SELDSK unimplemented." << crlf <<
		"Setting disk to " << get_a () + 'A' << crlf;
	// Provisional.
	set_hl (0);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncSETTRK ()
{
	cerr << crlf << "BIOS SETTRK unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncSETSEC ()
{
	cerr << crlf << "BIOS SETSEC unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncSETDMA ()
{
	//dma_addr= get_bc ();
	set_dma (get_bc () );
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncREAD ()
{
	if (debugsystem)
		cerr << "BIOS READ called." << crlf;

	// Always return non recoverable error.
	set_a (0x01);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncWRITE ()
{
	if (debugsystem)
		cerr << "BIOS WRITE unimplemented." << crlf;

	// Always return physical error.
	set_a (0x01);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncLISTST ()
{
	// A=00h if list device is not ready to accept a character.
	// A=FFh if list device is ready to accept a character.

	set_a (0xFF);

	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncSECTRN ()
{
	cerr << crlf << "BIOS SECTRN unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncCONOST ()
{
	if (debugsystem)
		cerr << "BIOS CONOST called." << crlf;

	// Always return ready to send chars.
	set_a (0xFF);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncAUXIST ()
{
	cerr << crlf << "BIOS AUXIST unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncAUXOST ()
{
	cerr << crlf << "BIOS AUXOST unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncDEVTBL ()
{
	cerr << crlf << "BIOS DEVTBL unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncDEVINI ()
{
	if (debugsystem)
		cerr << "BIOS DEVINI called." << crlf;

	// Initialize character device specified in register C.
	// Actually nothing to do.
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncDRVTBL ()
{
	if (debugsystem)
		cerr << "BIOS DRVTBL called." << crlf;

	// Returns: No drive table, buffers not set up, no hashing.
	set_hl (0xFFFE);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncMULTIO ()
{
	if (debugsystem)
		cerr << "BIOS MULTIO called." << crlf;

	set_multi_sector (get_c () );
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncFLUSH ()
{
	if (debugsystem)
		cerr << "BIOS FLUSH called." << crlf;

	// Returns no error or not implemented.
	set_a (0);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncMOVE ()
{
	cerr << crlf << "BIOS MOVE unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncTIME ()
{
	if (get_c () != 0)
	{
		cerr << "BIOS set time unimplemented" << crlf;
		return TrapEndProgram;
	}

	time_t t= time (0);
	z80word day= (t - 252374400) / (60 * 60 * 24);
	struct tm * ptm= localtime (& t);

	setwSCB (SCB_DATE, day);
	setbSCB (SCB_HOUR, conv_bcd (ptm->tm_hour) );
	setbSCB (SCB_MIN, conv_bcd (ptm->tm_min) );
	setbSCB (SCB_SEC, conv_bcd (ptm->tm_sec) );
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncSELMEM ()
{
	cerr << crlf << "BIOS SELMEM unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncSETBNK ()
{
	cerr << crlf << "BIOS SETBNK unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncXMOVE ()
{
	cerr << crlf << "BIOS XMOVE unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncUSERF ()
{
	// Reserved for system implementor.

	cerr << crlf << "BIOS USERF unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncRESERV1 ()
{
	cerr << crlf << "BIOS RESERV1 unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::biosfuncRESERV2 ()
{
	cerr << crlf << "BIOS RESERV2 unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::UnsupportedRst ()
{
	cerr << crlf << "RST " << hex2 (get_pc () ) <<
		" unsupported." << crlf;
	z80word sp= get_sp ();
	z80word pos= memreadw (sp);
	cerr << "Return address: " << hex4 (pos) << 'h' << crlf;
	return TrapEndProgram;
}

void CPM::Internal::bdos_return_hl (byte h, byte l)
{
	set_h (h);
	set_b (h);
	set_l (l);
	set_a (l);
	if (debugsystem)
	{
		cerr << "BDOS return values: " << hex2 (h) << ' ' <<
			hex2 (l) << crlf;
	}
}

void CPM::Internal::bdos_return_hl (z80word reghl)
{
	byte regh= static_cast <byte> (reghl >> 8);
	byte regl= static_cast <byte> (reghl & 0xFF);
	bdos_return_hl (regh, regl);
}

void CPM::Internal::bdos_return_a (byte a)
{
	bdos_return_hl (0, a);
}

CPM::Internal::TrapReturn CPM::Internal::bdos ()
{
	byte func= get_c ();

	bdosmap_t::iterator it= bdosmap.find (func);
	if (it == bdosmap.end () )
	{
		cerr << crlf << "BDOS func " << hex2 (func) <<
			" unimplemented." << crlf;
		return TrapEndProgram;
	}

	setbSCB (SCB_FX, func);
	setbSCB (SCB_RESEL, 0);
	return (this->* it->second) ();
}

z80word CPM::Internal::get_bdos_fcb ()
{
	z80word fcbaddr= get_de ();
	setbSCB (SCB_RESEL, 0xFF);
	setwSCB (SCB_VINFO, fcbaddr);
	return fcbaddr;
}

// 00
CPM::Internal::TrapReturn CPM::Internal::bdos_system_reset ()
{
	if (debugsystem)
		cerr << "BDOS system reset called" << crlf;
	set_pc (0);
	return TrapContinue;
}

// 01
CPM::Internal::TrapReturn CPM::Internal::bdos_console_input ()
{
	byte c= console.charin ();
	if (c >= 0x20 || c == '\r' || c == '\n')
		console.charout (c);
	if (c == '\t')
		console.charout (' ');
	bdos_return_a (c);
	return TrapRet;
}

// 02
CPM::Internal::TrapReturn CPM::Internal::bdos_console_output ()
{
	console.charout (get_e () );
	return TrapRet;
}

// 05
CPM::Internal::TrapReturn CPM::Internal::bdos_list_output ()
{
	// Provisional
	//console.charout (get_e () );
	printer.put (get_e() );
	return TrapRet;
}

// 06
CPM::Internal::TrapReturn CPM::Internal::bdos_direct_console_io ()
{
	byte c= get_e ();
	switch (c)
	{
	case 0xFF: // Input / status
		if (console.charavailable () )
			bdos_return_a (console.charin () );
		else
			bdos_return_a (0);
		break;
	case 0xFE: // Status
		bdos_return_a (console.charavailable () ? 0xFF : 0x00);
		break;
	case 0xFD: // Input
		bdos_return_a (console.charin () );
		break;
	default: // Output
		console.charout (c);
	}
	return TrapRet;
}

// 09
CPM::Internal::TrapReturn CPM::Internal::bdos_write_string ()
{
	z80word pos= get_de ();

	//if (debugsystem)
	//	cerr << "(BDOS Write string at " << hex4 (pos) << ')' <<
	//		crlf;

	byte output_delimiter= getbSCB (SCB_OUTDLM);
	byte c;
	while ( (c= memread (pos) ) != output_delimiter)
	{
		console.charout (c);
		++pos;
	}
	return TrapRet;
}

// 0A
CPM::Internal::TrapReturn CPM::Internal::bdos_read_console_buffer ()
{
	if (debugsystem)
		cerr << "BDOS read console buffer called" << crlf;

	z80word addr= get_de ();

	read_console_buffer (addr);
	return TrapRet;
}

// 0B
CPM::Internal::TrapReturn CPM::Internal::bdos_get_console_status ()
{
	bdos_return_a (console.charavailable () );
	return TrapRet;
}

// 0C
CPM::Internal::TrapReturn CPM::Internal::bdos_return_version_number ()
{
	if (debugsystem)
		cerr << "BDOS return version number called" << crlf;

	//bdos_return_a (0x31); // CP/M, vesrion 3.1
	bdos_return_a (getbSCB (SCB_CPMVER) );
	return TrapRet;
}

// 0D
CPM::Internal::TrapReturn CPM::Internal::bdos_reset_disk_system ()
{
	if (debugsystem)
		cerr << "BDOS reset disk system called" << crlf;

	// Provisional: does nothing.
	return TrapRet;
}

// 0E
CPM::Internal::TrapReturn CPM::Internal::bdos_select_disk ()
{
	if (debugsystem)
		cerr << "BDOS select disk called" << crlf;

	// Provisonal.
	//default_disk= get_e ();
	set_default_disk (get_e () );

	bdos_return_a (0);
	return TrapRet;
}

// 0F
CPM::Internal::TrapReturn CPM::Internal::bdos_open_file ()
{
	if (debugsystem)
		cerr << "BDOS open file called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	if (debugsystem)
	{
		cerr << "Opening file, fcbaddr= " << hex4 (fcbaddr) <<
			"h : '";
		for (z80word i= 1; i < 9; ++i)
			cerr << memread (fcbaddr + i);
		cerr << '.';
		for (z80word i= 9; i < 12; ++i)
			cerr << memread (fcbaddr + i);
		cerr << '\'' << crlf;
	}

	if (open_file (fcbaddr) )
	{
		if (debugsystem)
			cerr << "Open succeded" << crlf;
		bdos_return_hl (0);
	}
	else
	{
		if (debugsystem)
			cerr << "Open failed" << crlf;
		bdos_return_hl (0x00FF);
	}
	return TrapRet;
}

// 10
CPM::Internal::TrapReturn CPM::Internal::bdos_close_file ()
{
	if (debugsystem)
		cerr << "BDOS close file called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	//string filename= filename_from_fcb (fcbaddr);

	//if (debugsystem)
	//	cerr << "Close file: " << filename << crlf;

	//OpenFile * pf= getopenfile (fcbaddr);
	//if (pf == NULL)
	//{
	//	throw BdosErr (ErrInvalidFCB);
	//}
	//pf->close ();
	//clean_fcb_entry (fcbaddr);

	FcbFile fcb (mem + fcbaddr);
	if (debugsystem)
		cerr << "Closing file: " << fcb << crlf;

	close_fcb (fcb);

	bdos_return_hl (0);
	return TrapRet;
}

void CPM::Internal::setsearchresult (const string & result)
{
	z80word dma_addr= get_dma ();

	memwrite (dma_addr, 0);
	z80word pos= dma_addr + 1;
	string::size_type n= 0;
	const string::size_type l= result.size ();
	for (int i= 0; i < 8; ++i, ++pos)
	{
		if (n < l && result [n] != '.')
		{
			memwrite (pos, result [n] );
			++n;
		}
		else
			memwrite (pos, ' ');
	}
	++n;
	for (int i= 0; i < 3; ++i, ++pos)
	{
		if (n < l)
		{
			memwrite (pos, result [n] );
			++n;
		}
		else
			memwrite (pos, ' ');
	}
}

void CPM::Internal::setsearchresult (const FcbFile & fcbfile)
{
	z80word dma_addr= get_dma ();

	memwrite (dma_addr, 0);
	fcbfile.getfile (mem + dma_addr + 1);
}

// 11
CPM::Internal::TrapReturn CPM::Internal::bdos_search_for_first ()
{
	if (debugsystem)
		cerr << "BDOS search for first called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	clean_fcb_entry (fcbaddr);

	size_t disknum= disknum_from_fcb (fcbaddr);

	#if 0
	if (disknum == invalid_fcb_disk || ! disk_exist (disknum) )
	{
		bdos_return_hl (0x04, 0xFF);
		return TrapRet;
	}
	#endif

	//string filemask= filename_from_fcb (fcbaddr);
	FcbFile fcbmask;
	fcbmask.setfile (mem + fcbaddr + 1);

	if (debugsystem)
	{
		//cerr << "Search for first: " << filemask << crlf;
		cerr << "Search for first: '" <<
			letterfromdisknum (disknum) <<
			'(' << hex2 (memread (fcbaddr) ) << ')' <<
			':' <<
			fcbmask.getfilename () <<
			'.' << fcbmask.getfileext () << '\'' << crlf;
	}

	//FindFile * pff= new FindFile (pathfromdisknum (disknum) );
	//auto_ptr <FindFile> pff
	//	(new FindFile (pathfromdisknum_check (disknum), debugsystem_flag) );

	FindFile ff (pathfromdisknum_check (disknum), debugsystem_flag);

	//const string result= pff->findfirst (filemask);
	//FcbFile result= pff->findfirst (fcbmask);
	FcbFile result;
	

	//if (! result.empty () )
	//if (pff->getResult () == FindFile::Result::Ok)

	//FindFile::Result::code findresult= pff->findfirst (fcbmask, result);
	FindFile::Result::code findresult= ff.findfirst (fcbmask, result);
	if (findresult == FindFile::Result::Ok)
	{
		if (debugsystem)
			cerr << "search for first returns '" << result <<
				'\'' << crlf;
		//findfilemap [fcbaddr]= pff.get ();
		addfinder (fcbaddr, ff);
		//pff.release ();
		setsearchresult (result);
		bdos_return_hl (0);
	}
	else
	{
		if (debugsystem)
			cerr << "search for first failed" << crlf;
		switch (findresult)
		{
		case FindFile::Result::NoFile:
			bdos_return_hl (ErrNoFileMatch);
			break;
		case FindFile::Result::NoDisk:
			bdos_return_hl (ErrDiskIO);
			break;
		}
	}
	return TrapRet;
}

// 12
CPM::Internal::TrapReturn CPM::Internal::bdos_search_for_next ()
{
	if (debugsystem)
		cerr << "BDOS search for next called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	#if 0
	findfilemap_t::iterator it= findfilemap.find (fcbaddr);
	if (it == findfilemap.end () )
	{
		if (debugsystem)
			cerr << "Called bdos search for next "
				"with invalid FCB" << crlf;
		bdos_return_hl (0, 0xFF);
		return TrapRet;
	}
	FindFile * pff= it->second;
	#endif
	FindFile & ff= getfinder (fcbaddr);

	//const string result= pff->findnext ();
	//FcbFile result= pff->findnextfcb ();
	FcbFile result;

	//if (! result.empty () )
	//if (pff->getResult () == FindFile::Result::Ok)
	//if (pff->findnext (result) == FindFile::Result::Ok)
	if (ff.findnext (result) == FindFile::Result::Ok)
	{
		setsearchresult (result);
		bdos_return_a (0);
	}
	else
		bdos_return_a (0xFF);
	return TrapRet;
}

// 13
CPM::Internal::TrapReturn CPM::Internal::bdos_delete_file ()
{
	if (debugsystem)
		cerr << "BDOS delete file called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	#if 0

	string filename= filename_from_fcb (fcbaddr);

	if (debugsystem)
		cerr << "Delete file: " << filename << crlf;

	const string diskpath= getdisk_from_fcb_check (fcbaddr);

	FindFile ff (diskpath, debugsystem_flag);

	string result;

	#else

	FcbFile fcbmask (mem + fcbaddr);

	if (debugsystem)
		cerr << "Delete file: '" << fcbmask << '\'' << crlf;

	const string diskpath= pathfromdisknum_check
		(disknum_from_fcbnum (fcbmask.getdrive () ) );

	//FcbFile result= ff.findfirst (fcbmask);
	//string result;

	FindFile ff (diskpath, fcbmask, debugsystem_flag);
	//const string filename= fcbmask.getfile ();

	//if (debugsystem)
	//	cerr << "Dir: " << diskpath << " File: '" << filename <<
	//		'\'' << crlf;

	#endif

	#if 0
	FindFile::Result::code findcode= ff.findfirst (filename, result);
	switch (findcode)
	{
	case FindFile::Result::Ok:
		break; // Continue processing.
	case FindFile::Result::NoFile:
		throw BdosErr (ErrInvalidDrive);
	default:
		throw BdosErr (ErrDiskIO);
	}
	#endif

	string result;
	FindFile::Result::code findcode;
	bool some_deleted= false;
	//do {
	while  ( (findcode= ff.get (result) ) == FindFile::Result::Ok)
	{
		//const string filefound= diskpath + result.getfile ();
		const string filefound= diskpath + result;
		if (debugsystem)
			cerr << "Erasing: '" << filefound << '\'' << crlf;
		if (unlink (filefound.c_str () ) != 0)
		{
			int e= errno;
			if (debugsystem)
			{
				cerr << "unlink error: " << strerror (e) <<
					crlf;
			}
			byte code_high;
			switch (e)
			{
			case ENOENT:
				code_high= 0x00; break;
			case EACCES:
				code_high= 0x03; break;
			default:
				code_high= 0x01;
			}
			throw BdosErr (code_high, 0xFF);
		}
		some_deleted= true;
	//} while ( (findcode= ff.findnext (result) ) == FindFile::Result::Ok);
	}

	if (findcode == FindFile::Result::NoFile)
	{
		if (some_deleted)
			bdos_return_hl (0);
		else
			bdos_return_hl (ErrNoFileMatch);
	}
	else
	{
		throw BdosErrSerious (ErrDiskIO);
	}

	return TrapRet;
}

// 14
CPM::Internal::TrapReturn CPM::Internal::bdos_read_sequential ()
{
	if (debugsystem)
		cerr << "BDOS read sequential called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	if (debugsystem)
		cerr << "Fcb= " << hex4 (fcbaddr) <<
			", file is: " << filename_from_fcb (fcbaddr) << crlf;

	//OpenFile * pf= getopenfile (fcbaddr);
	//if (pf == NULL)
	//{
	//	if (debugsystem)
	//		cerr << "Invalid FCB in read_sequential" << crlf;
	//	throw BdosErr (ErrInvalidFCB);
	//}

	FcbFile fcb (mem + fcbaddr);
	OpenFile & file= getopenfile (fcb);

	const byte multi_sector_count= get_multi_sector ();
	int readsize= sector_size * multi_sector_count;

	z80word current_record= memread (fcbaddr + fcb_current_record);
	byte extension= memread (fcbaddr + fcb_extension);
	long pos= extension;
	pos*= records_in_extension;
	pos+= current_record;
	pos*= sector_size;

	z80word dma_addr= get_dma ();

	if (debugsystem)
	{
		cerr << "Reading " << int (multi_sector_count) <<
			" sectors from " << filename_from_fcb (fcbaddr) <<
			" position " << pos <<
			" to " << hex4 (dma_addr) <<
			" current record is " << current_record <<
			crlf;
	}

	//if (pf->lseek (pos, SEEK_SET) == -1)
	if (file.lseek (pos, SEEK_SET) == -1)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}

	//int r= pf->read (mem + dma_addr, readsize);
	int r= file.read (mem + dma_addr, readsize);

	if (debugsystem)
		cerr << "Bytes readed: " << r << crlf;

	if (r == -1 || r == 0)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}

	// If there is an incomplete sector, fill with CP/M EOF char.
	for (int i= r; i < readsize; ++i)
		memwrite (dma_addr + i, eof_char);

	int sec= r / sector_size;
	if (r % sector_size)
		sec++;

	current_record+= sec;
	if (current_record >= records_in_extension)
	{
		current_record-= records_in_extension;
		++extension;
		memwrite (fcbaddr + fcb_extension, extension);
	}
	memwrite (fcbaddr + fcb_current_record,
		static_cast <byte> (current_record) );

	memwrite (fcbaddr + fcb_current_record,
		current_record);

	if (sec < multi_sector_count)
		bdos_return_hl (sec, 1); // EOF after sec sectors.
	else
		bdos_return_hl (0, 0);

	return TrapRet;
}

// 15
CPM::Internal::TrapReturn CPM::Internal::bdos_write_sequential ()
{
	if (debugsystem)
		cerr << "BDOS write sequential called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	//OpenFile * pf= getopenfile (fcbaddr);
	//if (pf == NULL)
	//{
	//	if (debugsystem)
	//		cerr << "Invalid FCB in write_sequential" << crlf;
	//	throw BdosErr (ErrInvalidFCB);
	//}

	FcbFile fcb (mem + fcbaddr);
	OpenFile & file= getopenfile (fcb);

	const byte multi_sector_count= get_multi_sector ();
	int writesize= sector_size * multi_sector_count;

	z80word current_record= memread (fcbaddr + fcb_current_record);
	byte extension= memread (fcbaddr + fcb_extension);
	long pos= extension;
	pos*= records_in_extension;
	pos+= current_record;
	pos*= sector_size;

	z80word dma_addr= get_dma ();

	if (debugsystem)
	{
		cerr << "Writing " << int (multi_sector_count) <<
			" sectors to " << filename_from_fcb (fcbaddr) <<
			" position " << pos <<
			" from " << hex4 (dma_addr) <<
			" current record is " << current_record <<
			crlf;
	}

	//if (pf->lseek (pos, SEEK_SET) == -1)
	if (file.lseek (pos, SEEK_SET) == -1)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}

	//int r= pf->write (mem + dma_addr, writesize);
	int r= file.write (mem + dma_addr, writesize);

	int sec= r / sector_size;
	if (r % sector_size)
		sec++;

	current_record+= sec;
	if (current_record >= records_in_extension)
	{
		current_record-= records_in_extension;
		++extension;
		memwrite (fcbaddr + fcb_extension, extension);
	}
	memwrite (fcbaddr + fcb_current_record,
		static_cast <byte> (current_record) );

	memwrite (fcbaddr + fcb_current_record,
		current_record);

	if (sec < multi_sector_count)
		bdos_return_hl (sec, 1); // EOF after sec sectors.
	else
		bdos_return_hl (0);
	return TrapRet;
}

// 16
CPM::Internal::TrapReturn CPM::Internal::bdos_make_file ()
{
	if (debugsystem)
		cerr << "BDOS make file called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	#if 0
	if (make_file (fcbaddr) )
		bdos_return_hl (0);
	else
		bdos_return_hl (ErrFileAlreadyExists);
	return TrapRet;
	#endif

	FcbFile fcb (mem + fcbaddr);
	size_t disknum= disknum_from_fcb (fcbaddr);

	string filename= pathfromdisknum_check (disknum) + fcb.getfile ();

	if (debugsystem)
		cerr << "Making '" << filename << '\'' << crlf;

	int h= open (filename.c_str (), O_RDWR | O_CREAT | O_EXCL, 0666);
	if (h == -1)
	{
		int e= errno;
		if (debugsystem)
			cerr << "open failed with '" << filename <<
				"': " << strerror (e) << crlf;
		z80word code;
		switch (e)
		{
		case EEXIST:
			code= ErrFileAlreadyExists; break;
		default:
			code= ErrDiskIO;
		}
		throw BdosErrSerious (code);
	}

	GuardHandle guard (h);
	insertopenfile (fcb, h);
	guard.release ();

	// Testing.
	memwrite (fcbaddr + fcb_current_record, 0);

	bdos_return_hl (0);
	return TrapRet;
}

// 17
CPM::Internal::TrapReturn CPM::Internal::bdos_rename_file ()
{
	if (debugsystem)
		cerr << "BDOS rename file called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	size_t disknum= disknum_from_fcb (fcbaddr);

	#if 0
	if (disknum == invalid_fcb_disk)
	{
		bdos_return_hl (0x04, 0xFF);
		return TrapRet;
	}
	#endif

	const string disk= pathfromdisknum_check (disknum);

	const string orig= disk + filename_from_fcb (fcbaddr);

	z80word pseudofcb= fcbaddr + 16;
	string dest= disk + filename_from_fcb (pseudofcb);

	if (debugsystem)
		cerr << "Renaming " << orig << " to " << dest << crlf;

	rename (orig.c_str (), dest.c_str () );
	bdos_return_hl (0);
	return TrapRet;
}

// 19
CPM::Internal::TrapReturn CPM::Internal::bdos_return_current_disk ()
{
	if (debugsystem)
		cerr << "BDOS return current disk called" << crlf;

	//bdos_return_hl (0, default_disk);
	bdos_return_hl (0, get_default_disk () );
	return TrapRet;
}

// 1A
CPM::Internal::TrapReturn CPM::Internal::bdos_set_dma_address ()
{
	if (debugsystem)
		cerr << "BDOS set dma address called" << crlf;

	#if 0
	dma_addr= get_de ();

	if (debugsystem)
		cerr << "DMA address set to " <<
			hex4 (dma_addr) << crlf;
	#endif

	set_dma (get_de () );

	bdos_return_hl (0);
	return TrapRet;
}

// 1D
CPM::Internal::TrapReturn CPM::Internal::bdos_map_read_only ()
{
	if (debugsystem)
		cerr << "BDOS map read only called" << crlf;
	// Provisional: no drive is read only.
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 1E
CPM::Internal::TrapReturn CPM::Internal::bdos_set_file_attributes ()
{
	if (debugsystem)
		cerr << "BDOS set file attributes called" << crlf;

	// Nada de momento
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 20
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_user_code ()
{
	byte newuser= get_e ();
	if (newuser == 0xFF)
	{
		// Get user.
		if (debugsystem)
			cerr << "BDOS Get User Code" << crlf;
		//bdos_return_hl (0, 0); // Provisional.
		//bdos_return_a (getbSCB (SCB_USRCD) );
		bdos_return_a (get_user () );
	}
	else
	{
		// Set user.
		if (debugsystem)
			cerr << "BDOS Set User Code to " <<
				int (newuser) << crlf;
		set_user (newuser);
	}
	return TrapRet;
}

// 21
CPM::Internal::TrapReturn CPM::Internal::bdos_read_random ()
{
	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	if (debugsystem)
		cerr << "BDOS read random at " <<
			fcb_get_random_record (fcbaddr) <<
			crlf;

	//OpenFile * pf= getopenfile (fcbaddr);
	//if (pf == NULL)
	//{
	//	bdos_return_hl (0, 9); // Invalid FCB
	//	return TrapRet;
	//}

	FcbFile fcb (mem + fcbaddr);
	OpenFile & file= getopenfile (fcb);

	unsigned long rec= fcb_get_random_record (fcbaddr);
	unsigned long ext= rec / records_in_extension;
	if (ext > 255)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}
	memwrite (fcbaddr + fcb_extension, static_cast <byte> (ext) );
	rec%= records_in_extension;
	memwrite (fcbaddr + fcb_current_record, static_cast <byte> (rec) );

	// Provisional.
	TrapReturn r= bdos_read_sequential ();

	if (debugsystem)
		cerr << crlf << "Read random done." << crlf;
	return r;
}

// 22
CPM::Internal::TrapReturn CPM::Internal::bdos_write_random ()
{
	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	if (debugsystem)
		cerr << "BDOS write random at " <<
			fcb_get_random_record (fcbaddr) <<
			crlf;

	OpenFile * pf= getopenfile (fcbaddr);
	if (pf == NULL)
	{
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}

	// Provisional
	pf->lseek (fcb_get_random_record (fcbaddr) * sector_size, SEEK_SET);

	unsigned long rec= fcb_get_random_record (fcbaddr);
	unsigned long ext= rec / records_in_extension;
	if (ext > 255)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}
	memwrite (fcbaddr + fcb_extension, static_cast <byte> (ext) );
	rec%= records_in_extension;
	memwrite (fcbaddr + fcb_current_record, static_cast <byte> (rec) );

	TrapReturn r= bdos_write_sequential ();

	if (debugsystem)
		cerr << crlf << "Write random done." << crlf;
	return r;
}

// 23
CPM::Internal::TrapReturn CPM::Internal::bdos_compute_file_size ()
{
	if (debugsystem)
		cerr << "BDOS compute file size called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	string filename= filename_from_fcb (fcbaddr);
	struct stat buf;
	if (stat (filename.c_str (), & buf) != 0)
	{
		bdos_return_hl (0, 0xFF);
		return TrapRet;
	}
	const unsigned long filesize= buf.st_size;
	unsigned long l= filesize / sector_size;
	if (filesize % sector_size)
		++l;
	memwrite (fcbaddr + fcb_random_record + 0, l & 0xFF);
	memwrite (fcbaddr + fcb_random_record + 2, (l >> 8) & 0xFF);
	memwrite (fcbaddr + fcb_random_record + 3, (l >> 16) & 0xFF);
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 24
CPM::Internal::TrapReturn CPM::Internal::bdos_set_random_record ()
{
	if (debugsystem)
		cerr << "BDOS set random record called" << crlf;

	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	string filename= filename_from_fcb (fcbaddr);

	if (debugsystem)
		cerr << "set random record to: " << filename << crlf;

	OpenFile * pf= getopenfile (fcbaddr);
	if (pf == NULL)
	{
		throw BdosErr (ErrInvalidFCB);
	}
	long pos= pf->lseek (0, SEEK_CUR);
	if (pos == -1)
	{
		throw BdosErrSerious (ErrDiskIO);
		return TrapRet;
	}
	long l= pos / 128;
	if (pos % 128)
		++l;
	memwrite (fcbaddr + fcb_random_record, l & 0xFF);
	memwrite (fcbaddr + fcb_random_record + 2, (l >> 8) & 0xFF);
	memwrite (fcbaddr + fcb_random_record + 3, (l >> 16) & 0xFF);
	bdos_return_hl (0);
	return TrapRet;
}

// 2C
CPM::Internal::TrapReturn CPM::Internal::bdos_set_multi_sector_count ()
{
	if (debugsystem)
		cerr << "BDOS set multi sector count called" << crlf;

	byte count= get_e ();
	if (count < 1 || count > 128)
		bdos_return_a (0xFF);
	else
	{
		if (debugsystem)
			cerr << "Setting multi-sector count to " <<
				int (count) << crlf;

		//multi_sector_count= count;
		set_multi_sector (count);
		bdos_return_a (0);
	}
	return TrapRet;
}

// 2D
CPM::Internal::TrapReturn CPM::Internal::bdos_set_bdos_error_mode ()
{
	if (debugsystem)
		cerr << "BDOS set bdos error mode called" << crlf;

	// E= error mode.
	setbSCB (SCB_ERMDE, get_e () );
	return TrapRet;
}

// 31
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_system_control_block ()
{
	z80word scbpb= get_de ();
	byte offset= memread (scbpb + 0);
	byte opervalue= memread (scbpb + 1);

	enum Operation { Get, SetB, SetW };
	Operation oper= Get;

	if (debugsystem)
	{
		cerr << "BDOS get/set system control block called" <<
			crlf;
	}

	byte byteval;
	z80word wordval;
	switch (opervalue)
	{
	case 0x00:
		if (debugsystem)
		{
			cerr << "\tReading " << hex2 (offset) << crlf;
		}
		break;
	case 0xFF:
		oper= SetB;
		byteval= memread (scbpb + 2);
		if (debugsystem)
		{
			cerr << "\tSet byte " << hex2 (offset) <<
				" to " << hex2 (byteval) << crlf;
		}
		break;
	case 0xFE:
		oper= SetW;
		wordval= memreadw (scbpb + 2);
		if (debugsystem)
		{
			cerr << "\tSet word " << hex2 (offset) <<
				" to " << hex4 (wordval) << crlf;
		}
		break;
	default:
		if (debugsystem)
			cerr << "Invalid SCB set operation" << crlf;
		bdos_return_hl (0xFF);
		return TrapRet;
	}

	z80word retval= 0;
	switch (offset)
	{
	case 0x0C:
		// Used in Year 2000 compliant version to select date format.
		switch (oper)
		{
		case Get:
			retval= 0x01; // UK format.
			break;
		default:
			if (debugsystem)
				cerr << "Set date format operation invalid." <<
					crlf;
		}
		break;
	case SCB_CLPSUB:
		switch (oper)
		{
		case Get:
			retval= getSCB (offset);
			break;
		case SetB:
			setbSCB (offset, byteval);
			break;
		default:
			cerr << "Invalid set submit drive word"  << crlf;
		}
		break;
	case SCB_CLPRET:
		switch (oper)
		{
		case Get:
			retval= getSCB (offset);
			break;
		case SetW:
			setbSCB (offset, wordval);
			break;
		default:
			cerr << "Invalid set error code byte"  << crlf;
		}
		break;
	case SCB_CCPFL1:
	case SCB_CCPFL2:
	case SCB_CCPFL3:
		switch (oper)
		{
		case Get:
			retval= getSCB (offset);
			break;
		case SetB:
			setbSCB (offset, byteval);
			break;
		default:
			cerr << "Invalid set flags word"  << crlf;
		}
		break;
	case SCB_PAGWID: // Console width (0 based)
		switch (oper)
		{
		case Get:
			retval= static_cast <byte> (console.columns () - 1);
			break;
		case SetB:
			setbSCB (SCB_PAGWID, byteval);
			break;
		default:
			if (debugsystem)
				cerr << "Page width operation invalid." <<
					crlf;
		}
		break;
	case SCB_PAGLEN: // Console page length
		switch (oper)
		{
		case Get:
			retval= static_cast <byte> (console.lines () );
			break;
		case SetB:
			setbSCB (SCB_PAGLEN, byteval);
			break;
		default:
			if (debugsystem)
				cerr << "Page length operation invalid." <<
					crlf;
		}
		break;
	case 0x2C: // Page mode
		switch (oper)
		{
		case Get:
			retval= 0;
			break;
		case SetB: // Set byte
			// Ignored
			break;
		default:
			if (debugsystem)
				cerr << "Page mode operation invalid." <<
					crlf;
		}
		break;
	case 0x3A: // SCB address
		switch (oper)
		{
		case Get:
			//retval= scb_documented_base;
			retval= getSCB (SCB_SELF);
			break;
		default:
			cerr << "Set SCB address inavlid operation" << crlf;
		}
		break;
	case SCB_DRVSEA:
	case SCB_DRVSEA + 1:
	case SCB_DRVSEA + 2:
	case SCB_DRVSEA + 3:
		switch (oper)
		{
		case Get:
			retval= getSCB (offset);
			break;
		case SetB:
			setbSCB (offset, byteval);
			break;
		default:
			cerr << "Invalid set word search drive"  << crlf;
		}
		break;
	case SCB_DRVTMP:
		switch (oper)
		{
		case Get:
			retval= getSCB (offset);
			break;
		case SetB:
			setbSCB (offset, byteval);
			break;
		default:
			cerr << "Invalid set word temporary drive"  << crlf;
		}
		break;
	case SCB_ERMDE:
		switch (oper)
		{
		case Get:
			retval= getSCB (offset);
			break;
		case SetB:
			setbSCB (offset, byteval);
			break;
		default:
			cerr << "Invalid set word error mode"  << crlf;
		}
		break;
	case SCB_DATE: // Date in days since 1 Jan 78
		switch (oper)
		{
		case Get:
			retval= 0;
			break;
		default:
			cerr << "Set date unsupported" << crlf;
		}
		break;
	case SCB_HOUR: // Hour in BCD
		switch (oper)
		{
		case Get:
			retval= 0;
			break;
		default:
			cerr << "Set hour unsupported" << crlf;
		}
		break;
	case SCB_MIN: // Minute in BCD
		switch (oper)
		{
		case Get:
			retval= 0;
			break;
		default:
			cerr << "Set minute unsupported" << crlf;
		}
		break;
	case SCB_SEC: // Second in BCD
		switch (oper)
		{
		case Get:
			retval= 0;
			break;
		default:
			cerr << "Set second unsupported" << crlf;
		}
		break;
	case SCB_MXTPA:
		switch (oper)
		{
		case Get:
			retval= getSCB (offset);
			break;
		case SetW:
			setbSCB (offset, wordval);
			break;
		default:
			cerr << "Invalid set tpa byte"  << crlf;
		}
		break;
	default:
		if (offset >= scb_documented_size)
		{
			bdos_return_hl (0);
			break;
		}
		switch (oper)
		{
		case Get:
			cerr << "Reading";
			break;
		case SetB:
			cerr << "Writing byte " <<
				hex2 (memread (scbpb + 2) ) <<
				"h in";
			break;
		case SetW:
			cerr << "Writing z80word " <<
				hex4 (memreadw (scbpb + 2) ) <<
				"h in";
			break;
		}
		cerr <<	" unsupported SCB field " <<
			hex2 (offset) << crlf;
	}
	bdos_return_hl (retval);
	return TrapRet;
}

// 32
CPM::Internal::TrapReturn
	CPM::Internal::bdos_direct_bios_calls ()
{
	if (debugsystem)
	{
		cerr << "BDOS direct bios calls" << crlf;
	}
	z80word biospb= get_de ();

	byte funcnum= memread (biospb);
	if (debugsystem)
	{
		cerr << "Bios function: " << hex2 (funcnum) << crlf;
	}
	

	set_a (memread (biospb + 1) );
	set_bc (memread (biospb + 2) );
	set_de (memread (biospb + 4) );
	set_hl (memread (biospb + 6) );

	TrapReturn r= (this->* (trapfuncmap [bios_traps +  funcnum]) ) ();

	//return TrapRet;
	return r;
}

// 3B
CPM::Internal::TrapReturn CPM::Internal::bdos_load_overlay ()
{
	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	z80word loadaddr= static_cast <z80word>
		(fcb_get_random_record (fcbaddr) & 0xFFFF);

	if (debugsystem)
	{
		cerr << "BDOS load overlay: " <<
			filename_from_fcb (fcbaddr) <<
			" at " << hex4 (loadaddr) <<
			crlf;
	}

	OpenFile * pf= getopenfile (fcbaddr);
	if (pf == NULL)
	{
		if (debugsystem)
			cerr << "Invalid FCB in load overlay" << crlf;
		throw BdosErr (ErrInvalidFCB);
	}

	if (pf->lseek (0, SEEK_SET) == -1)
		throw BdosErrSerious (ErrDiskIO);

	// Provisional.
	if (loadaddr == tpa_start_addr)
	{
		loadfiletpa (pf->gethandle () );
	}
	else
	{
		int r= pf->read (mem + loadaddr, 0xFFFF);
		if (r < 0)
			throw BdosErrSerious (ErrDiskIO);
		if (debugsystem)
			cerr << "Loaded " << r << " bytes " << crlf;
	}

	//bdos_return_hl (0);
	//return TrapRet;

	set_default_stack ();
	set_pc (loadaddr);
	return TrapContinue;
}

// 3C
CPM::Internal::TrapReturn
	CPM::Internal::bdos_call_resident_system_extension ()
{
	if (debugsystem)
	{
		cerr << "BDOS call resident system extension" << crlf;
		z80word rsxpb= get_de ();
		cerr << "Function: " << hex2 (memread (rsxpb) ) <<
			crlf;
	}
	bdos_return_hl (0xFF, 0xFF);
	return TrapRet;
}

// 62
CPM::Internal::TrapReturn CPM::Internal::bdos_free_blocks ()
{
	if (debugsystem)
	{
		cerr << "BDOS free blocks" << crlf;
	}

	// No need to do nothing under Aliados actually.
	bdos_return_hl (0);
	return TrapRet;
}

// 63

CPM::Internal::TrapReturn CPM::Internal::bdos_truncate_file ()
{
	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	if (debugsystem)
	{
		cerr << "BDOS truncate file" << crlf <<
			"File: " << filename_from_fcb (fcbaddr) << crlf;
	}

	// Provisional.
	bdos_return_hl (0);
	return TrapRet;
}

//  66
CPM::Internal::TrapReturn
	CPM::Internal::bdos_get_read_file_date_and_pass_mode ()
{
	//z80word fcbaddr= get_de ();
	z80word fcbaddr= get_bdos_fcb ();

	if (debugsystem)
	{
		cerr << "BDOS get read file date stamps and password mode" <<
			crlf << "File: " << filename_from_fcb (fcbaddr) <<
			crlf;
	}

	// Provisional
	bdos_return_hl (0, 0xFF); // File not found.
	return TrapRet;
}

// 69
CPM::Internal::TrapReturn CPM::Internal::bdos_get_date_and_time ()
{
	if (debugsystem)
	{
		cerr << "BDOS get date and time" << crlf;
	}
	z80word addr= get_de ();
	set_c (0);
	(this->* (bios_table [biosTIME]) ) ();

	z80word scbtime= scb_documented_base + SCB_DATE;
	memwritew (addr, memreadw (scbtime) );
	memwrite (addr + 2, memread (scbtime + 2) );
	memwrite (addr + 3, memread (scbtime + 3) );
	bdos_return_a (memread (scbtime + 4) );
	return TrapRet;
}

// 6C
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_program_return_code ()
{
	if (debugsystem)
		cerr << "BDOS get/set program return code called" << crlf;

	z80word code= get_de ();
	if (code == 0xFFFF)
	{
		//bdos_return_hl (program_return_code);
		bdos_return_hl (getSCB (SCB_CLPRET) );
	}
	else
	{
		//program_return_code= code;
		setwSCB (SCB_CLPRET, code);
	}
	return TrapRet;
}

// 6D
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_console_mode ()
{
	if (debugsystem)
		cerr << "BDOS get/set console mode called" << crlf;

	if (get_de () == 0xFFFF)
	{
		set_hl (0);
	}
	else
	{
		if (debugsystem)
			cerr << "Console mode set to " <<
				hex4 (get_de () ) << crlf;

		set_hl (0); // Provisional
	}
	return TrapRet;
}

// 6E
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_output_delimiter ()
{
	if (debugsystem)
		cerr << "BDOS get/set output delimiter called" << crlf;

	z80word code= get_de ();
	if (code == 0xFFFF)
		bdos_return_hl (getbSCB (SCB_OUTDLM) );
	else
		setbSCB (SCB_OUTDLM, static_cast <byte> (code & 0xFF) );
	return TrapRet;
}

// 98
CPM::Internal::TrapReturn CPM::Internal::bdos_parse_filename ()
{
	z80word pfcb= get_de ();

	if (debugsystem)
		cerr << "BDOS parse filename in " << hex4 (pfcb) << crlf;

	z80word name= memreadw (pfcb);
	z80word fcbaddr= memreadw (pfcb + 2);
	parsefilefcb (name, fcbaddr);

	bdos_return_hl (name); // End of string
	return TrapRet;
}

z80word CPM::Internal::parsefilename (z80word name,
	size_t & fcbnum, string & file, string & ext, string & pass)
{
	byte c;
	while ( (c= memread (name) ) == ' ' || c == '\t')
		++name;
	if (c == '\r' || c == 0)
		return 0;

	byte auxc= cpmupper (c);
	if (auxc >= 'A' && auxc <= 'P' && memread (name + 1) == ':')
	{
		fcbnum= disknumfromletter (auxc) + 1;
		name+= 2;
	}
	else
		fcbnum= 0;

	size_t i= 0;
	while ( (c= memread (name) ) != '.' && c != ' ' &&
		c != '\t' && c != '\r' && c != 0)
	{
		if (i < 8)
			file+= static_cast <char> (cpmupper (c) );
		++i;
		++name;
	}
	if (c == '.')
	{
		++name;
		i= 0;
		while ( (c= memread (name) ) != '.' && c != ' ' &&
			c != '\t' && c != '\r' && c != 0)
		{
			if (i < 3)
				ext+= static_cast <char> (cpmupper (c) );
			++i;
			++name;
		}
	}

	// Skip whitespace at end.
	if (c == ' ' || c == '\t')
	{
		z80word auxname= name;
		do {
			c= memread (++auxname);
		} while (c == ' ' || c == '\t');
		if (c == '\r' || c == '\0')
			return 0;
		else
			return name;
	}

	if (c == '\r' || c == '\0')
		return 0;
	else
		return name;
}

void CPM::Internal::parsefilefcb (z80word & name, z80word fcbaddr)
{
	if (debugsystem)
	{
		cerr << "Parsing in " << hex4 (fcbaddr) <<
			" to " << name << crlf;
	}

	memwrite (fcbaddr, 0); // Default disk

	// Clean file name and type to spaces.
	for (int i= 1; i < 12; ++i)
		memwrite (fcbaddr + i, ' ');
	// Clean next 3 bytes. 
	for (int i= 12; i < 16; ++i)
		memwrite (fcbaddr + i, 0);

	//Clean password to spaces
	//for (int i= 16; i < 24; ++i)
	//	memwrite (fcbaddr + i, ' ');
	memwrite (fcbaddr + 16, ' ');

	// Clean reserved area
	//for (int i= 24; i < 36; ++i)
	//	memwrite (fcbaddr + i, 0);

	size_t fcbnum= 0;
	string file, ext, pass;
	name= parsefilename (name, fcbnum, file, ext, pass);

	if (debugsystem)
	{
		cerr << "Parse result - file: '" << file <<
			"' ext: '" << ext <<
			"' pass: '" << pass << '\'' <<
			crlf;
	}

	memwrite (fcbaddr, static_cast <byte> (fcbnum) );
	string::size_type l= file.size ();
	if (l > 8)
		l= 8;
	for (string::size_type i= 0; i < l; ++i)
		memwrite (fcbaddr + 1 + i, file [i] );

	l= ext.size ();
	if (l > 3)
		l= 3;
	for (string::size_type i= 0; i < l; ++i)
		memwrite (fcbaddr + 9 + i, ext [i] );

	for (string::size_type i= 12; i < 16; ++i)
		memwrite (fcbaddr + i, 0);

	l= pass.size ();
	if (l < 8)
		pass+= string (8 - l, ' ');
	if (l > 8)
		l= 8;
	for (string::size_type i= 0; i < 8; ++i)
		memwrite (fcbaddr + 16 + i, pass [i] );
}

void CPM::Internal::cleardefaultfcb (z80word fcbaddr, z80word pass)
{
	memwrite (fcbaddr, 0); // Disk
	for (z80word i= 1; i < 12; ++i)
		memwrite (fcbaddr + i, ' ');
	memwrite (fcbaddr + fcb_extension, 0);

	memwritew (pass, 0); // Address
	memwrite (pass + 2, 0); // Length
}

void CPM::Internal::parsefilecmdline (z80word & pos, z80word endline,
	z80word fcbaddr, z80word pass)
{
	if (pos >= endline)
		return;
	if (pos < endline - 1 && isalpha (memread (pos) ) &&
		memread (pos + 1) == ':')
	{
		byte c= cpmupper (memread (pos) ) - 'A' + 1;
		memwrite (fcbaddr, c);
		pos+= 2;
	}
	size_t i= 0;
	byte c;
	while (pos < endline && validchar ( (c= memread (pos) ) ) )
	{
		if (i < 8)
		{
			if (c == '*')
			{
				while (i < 8)
				{
					memwrite (fcbaddr + i + 1, '?');
					++i;
				}
			}
			else
			{
				memwrite (fcbaddr + i + 1, c);
				++i;
			}
		}
		++pos;
	}
	if (pos >= endline)
		return;
	if (c == '.')
	{
		++pos;
		i= 0;
		while (pos < endline && validchar ( (c= memread (pos) ) ) )
		{
			if (i < 3)
			{
				if (c == '*')
				{
					while (i < 3)
					{
						memwrite (fcbaddr + i + 9, '?');
						++i;
					}
				}
				else
				{
					memwrite (fcbaddr + 9 + i, c);
					++i;
				}
			}
			++pos;
		}
	}
	if (pos >= endline)
		return;
	if (c == ';')
	{
		++pos;
		byte i= 0;
		z80word pospass= pos;
		while (pos < endline && validpasschar (memread (pos) ) )
		{
			++i;
			++pos;
		}
		if (i != 0)
		{
			memwritew (pass, pospass);
			memwrite (pass + 2, i);
		}
	}
}

void CPM::Internal::setcmd_args (const string & args)
{
	string::size_type l= args.size ();
	if (l > 127)
		l= 127;
	memwrite (cmd_args, static_cast <byte> (l) );
	for (z80word i= 0; i < l; ++i)
		memwrite (cmd_args + 1 + i, args [i] );
	for (z80word i= l; i < 127; ++i)
		memwrite (cmd_args + 1 + i, 0);
}

void CPM::Internal::setarguments (const string & args)
{
	if (debugsystem_flag)
		cerr << "Setting arguments to: " << args << crlf;

	setcmd_args (args);

	z80word pos= cmd_args + 1;
	const z80word endline= pos + memread (cmd_args);

	cleardefaultfcb (default_fcb_1, default_pass_1);
	cleardefaultfcb (default_fcb_2, default_pass_2);

	byte c;
	while (pos < endline && (c= memread (pos) ) == ' ' || c == '\t')
		++pos;

	if (pos < endline)
	{
		parsefilecmdline (pos, endline,
			default_fcb_1, default_pass_1);

		++pos; // Skip separator
		while (pos < endline &&
			(c= memread (pos) ) == ' ' || c == '\t')
		{
			++pos;
		}
		parsefilecmdline (pos, endline,
			default_fcb_2, default_pass_2);
	}
}

void CPM::Internal::showcurrentstate ()
{
	bool saveflag= debugz80;
	debugz80= false;

	z80word sp= get_sp ();
	z80word pc= get_pc ();
	cerr << hex4 (pc);

	#if ! defined USE_ImcZ80 || defined  DISASM_WITH_libz80

	disassembly (cerr);

	#else

	//cerr << ": " << hex2 << int (memread (pc) ) << dec;
	cerr << ": ";
	if (memread (pc)  == trapbyte && ( gettrapfunc (pc) ) != 0)
		cerr << "System trap";
	else
		disassembly (cerr);

	#endif

	// Flags.
	byte f= get_f ();
	cerr << " F=" << hex2 (f) <<
		' ' << ( flag_c () ? ' ' : 'N' ) << "C" <<
		' ' << ( flag_z () ? ' ' : 'N' ) << "Z" <<
		' ' << ( flag_s () ? ' ' : 'N' ) << "S";

	// Registers.
	cerr << " A=" << hex2 (get_a () ) <<
		" BC=" << hex4 (get_bc () ) <<
		" DE=" << hex4 (get_de () ) <<
		" HL=" << hex4 (get_hl () ) <<
		" IX=" << hex4 (get_ix () ) <<
		" IY=" << hex4 (get_iy () ) <<
		crlf;

	// Stack inspection.
	#if 1
	cerr <<
		" SP=" << hex4 (sp) << "->" <<
		//hex2 << int (memread (sp + 0) ) << ' ' <<
		hex2 (memread (sp + 1) ) << ' ' <<
		hex2 (memread (sp + 2) ) << ' ' <<
		hex2 (memread (sp + 3) ) << ' ' <<
		hex2 (memread (sp + 4) ) << ' ' <<
		hex2 (memread (sp + 5) ) << ' ' <<
		hex2 (memread (sp + 6) ) << ' ' <<
		hex2 (memread (sp + 7) ) << ' ' <<
		hex2 (memread (sp + 6) );
	#endif

	cerr << crlf;

	debugz80= saveflag;
}

void CPM::Internal::showbdoserror (z80word errcode)
{
	byte c;
	switch (errcode)
	{
	case ErrDiskIO:            c= 0x01; break;
	case ErrInvalidDrive:      c= 0x04; break;
	case ErrFileAlreadyExists: c= 0x08; break;
	default:
		c= 0x00;
	}

	cerr << "Aliados CP/M Error ";
	z80word fcbaddr= 0;
	if (c != 0)
	{
		if (getbSCB (SCB_RESEL) == 0xFF)
		{
			fcbaddr= getSCB (SCB_VINFO);
			size_t disknum= disknum_from_fcb (fcbaddr);
			byte letter;
			if (disknum == invalid_fcb_disk)
				letter= '?';
			else
				letter= letterfromdisknum (disknum);
			cerr << "on " << letter << ": ";
		}
	}
	switch (c)
	{
	case 0x01: cerr << "Disk I/O";      break;
	case 0x04: cerr << "Invalid Drive"; break;
	case 0x08: cerr << "File Exists";   break;
	default:   cerr << "UNKNOWN ERROR"; break;
	}
	cerr << crlf;

	cerr << "BDOS function = " <<
		static_cast <unsigned int> (getbSCB (SCB_FX) );
	if (fcbaddr)
	{
		FcbFile f (mem + fcbaddr);
		cerr << " File = " << f.getfullfile ();
	}
	cerr << crlf;
}

inline CPM::Internal::TrapReturn CPM::Internal::runinstruction ()
{
	z80word pc= get_pc ();
	byte b= memread (pc);
	TrapFunc trap;
	if (b == trapbyte && (trap= gettrapfunc (pc) ) != 0)
	{
		try
		{
			if (debugsystem || debugz80)
			{
				cout << flush;
				cerr << flush;
			}
			TrapReturn r= (this->* trap) ();
			if (debugsystem || debugz80)
			{
				cout << flush;
				cerr << flush;
			}
			switch (r)
			{
			case TrapContinue:
				// Nothing to do.
				break;
			case TrapRet:
				{
					z80word sp= get_sp ();
					z80word pc= memread (sp + 0) |
						(z80word (memread (sp + 1) )
						<< 8);
					set_pc (pc);
					set_sp (sp + 2);
				}
				break;
			case TrapEndProgram:
				if (debugsystem)
					cerr << "Ending program" << crlf;
			}
			return r;
		}
		catch (BdosErr & e)
		{
			z80word errcode= e.getcode ();

			if (e.serious () )
			switch (get_error_mode () )
			{
			case ErrModeReturn:
				break;
			case ErrModeDisplayReturn:
				showbdoserror (errcode);
				break;
			default:
				// Any other value, default mode.
				showbdoserror (errcode);
				return TrapEndProgram;
			}

			z80word sp= get_sp ();
			z80word pc= memread (sp + 0) |
				(z80word (memread (sp + 1) ) << 8);
			set_pc (pc);
			set_sp (sp + 2);
			bdos_return_hl (errcode);
			return TrapRet;
		}
	}
	else
	{
		// Avoid looping in HALT instruction.
		if (b == 0x76)
		{
			//set_pc (pc + 1);
			if (debugsystem)
				cerr << "HALT instruction executed" << crlf;
			return TrapEndProgram;
		}
		else
			execute ();
		return TrapContinue;
	}
}

void CPM::Internal::runner ()
{
	for (;;)
	{
		if (debugz80)
			showcurrentstate ();
		if (runinstruction () == TrapEndProgram)
			return;
	}
}

void CPM::Internal::run ()
{
	//multi_sector_count= 1;
	//dma_addr= default_dma_addr;

	debugsystem= debugsystem_flag;
	debugz80= debugz80_flag;

	runner ();

	//close_all ();

	debugsystem= false;
	debugz80= false;
}

CPM::Internal::TrapReturn CPM::Internal::checkrunner ()
{
	TrapReturn r;
	do
	{
		if (debugz80)
			showcurrentstate ();
		r= runinstruction ();
	}
	while (r != TrapEndProgram && r != TrapEndInitRsx);
	return r;
}

CPM::Internal::TrapReturn CPM::Internal::checkrun ()
{
	debugsystem= debugsystem_flag;
	debugz80= debugz80_flag;

	TrapReturn r= checkrunner ();

	debugsystem= false;
	debugz80= false;
	return r;
}

void CPM::Internal::prepareruntransient ()
{
	reset ();

	//multi_sector_count= 1;
	set_multi_sector (1);
	//dma_addr= default_dma_addr;
	set_dma (default_dma_addr);

	set_pc (tpa_start_addr);

	// Init stack, putting 0 as return address to return to system.
	//set_sp (default_stack);
	//memwritew (default_stack, 0);
	set_default_stack ();

	//output_delimiter= default_output_delimiter;
	setbSCB (SCB_OUTDLM, default_output_delimiter);
}

void CPM::Internal::endruntransient ()
{
	close_all ();
	printer.flush ();
}

void CPM::Internal::runtransient ()
{
	prepareruntransient ();

	run ();

	endruntransient ();
}

void CPM::Internal::attachloader ()
{
	// Attach a pseudo loader that does nothing.
	// Some RSX may need to have a loader, and
	// it helps in deattachment.

	//z80word old_jump= memreadw (bdos_call + 1);
	z80word old_jump= get_tpaend ();

	if (old_jump != bdos_jump)
	{
		// Some RSX already loaded.
		if (debugsystem_flag)
		{
			cerr << "Not attaching fake LOADER." << crlf;
		}
		return;
	}

	if (debugsystem_flag)
	{
		cerr << "Attaching fake LOADER." << crlf;
	}

	z80word old_base= old_jump & 0xFF00;
	z80word new_base= old_base - 0x0100;

	// Build the RSX prefix of the loader.
	memwrite (new_base + rsx_jump_start, jp_code);
	memwritew (new_base + rsx_start, old_jump);
	memwrite (new_base + rsx_jump_next, jp_code);
	memwritew (new_base + rsx_next, old_jump);

	// Previous field indicating first RSX.
	memwritew (new_base + rsx_prev, bdos_call + 2);

	// Set remove flag to true.
	memwritew (new_base + rsx_remove, 0xFF);

	memcpy (mem + new_base + rsx_name, "LOADER  ", 8);
	memwrite (new_base + rsx_loader, 0xFF);

	// Point BDOS jump to loader.
	memwritew (bdos_call + 1, new_base + rsx_jump_start);
	set_tpaend (new_base + rsx_jump_start);
}

void CPM::Internal::attachrsx (z80word image, z80word len)
{
	// Image points to the PRL header of the RSX.
	if (debugsystem_flag)
	{
		cerr << "RSX header:" << crlf <<
			"\tStart: " << hex4 (memreadw (image + rsx_start) ) <<
			crlf <<
			"\tNext: " << hex4 (memreadw (image + rsx_next) ) <<
			crlf <<
			"\tRemove flag: " <<
			hex2 (memread (image + rsx_remove) ) <<
			crlf <<
			"\tName: " << setw (8) <<
			reinterpret_cast <char *> (mem + image + rsx_name) <<
			crlf;
	}

	// Calculate installation address.
	//z80word old_jump= memreadw (bdos_call + 1);
	z80word old_jump= get_tpaend ();

	if (debugsystem_flag)
		cerr << "\tOld jump address: " << hex4 (old_jump) << crlf;

	z80word old_base= old_jump & 0xFF00;
	z80word new_base= (old_base - len) & 0xFF00;
	byte base_seg= new_base >> 8;

	// Move to installation location.
	for (z80word i= 0; i < len; ++i)
		memwrite (new_base + i, memread (image + i) );

	// Relocation.
	z80word reloc= image + len;
	//cerr << "Relocation map " << hex4 (reloc) << crlf;
	z80word pos= reloc;
	byte mask= 0;
	byte b;
	for (z80word i= 0; i < len; ++i)
	{
		if (mask == 0)
		{
			b= memread (pos++);
			//cerr << "Byte: " << hex2 (b) << crlf;
			mask= 0x80;
		}
		if (b & mask)
		{
			//cout << hex4 (i) <<
			//	" needs relocation" << crlf;
			memwrite (new_base + i,
				memread (new_base + i) + base_seg - 1);
		}
		mask>>= 1;
	}

	// Set previous field of old RSX (it must be one,
	// the loader at least).
	memwritew (old_base + rsx_prev, new_base + rsx_jump_next + 2);

	// Set jump to old BDOS jump.
	memwritew (new_base + rsx_next, old_jump);

	// Set previous field indicating this is the first RSX.
	memwritew (new_base + rsx_prev, bdos_call + 2);

	memwrite (new_base + rsx_loader, 0); // This is not the loader.

	// Point BDOS jump to the actual RSX.
	memwritew (bdos_call + 1, new_base + rsx_jump_start);
	set_tpaend (new_base + rsx_jump_start);
}

bool CPM::Internal::loadfiletpa (int handle)
{
	ASSERT (handle != -1);

	if (debugsystem_flag)
	{
		cerr << "Loading file in " <<
			static_cast <void *> (mem) << ", " <<
			static_cast <void *> (mem + tpa_start_addr) << crlf;
	}

	//int r= read (handle, mem + tpa_start_addr, bdos_base - tpa_start_addr);
	int r= read (handle, mem + tpa_start_addr,
		get_tpaend () - tpa_start_addr);
	if (r <= 0)
	{
		cerr << "Error reading: " << strerror (errno) << crlf;
		return false;
	}
	char c;

	if (read (handle, & c, 1) > 0)
	{
		cerr << "File too long" << crlf;
		return false;
	}

	// Check if it have header.
	if (memread (tpa_start_addr) != ret_code)
		return true;

	// Pre-initialisation code.

	if (debugsystem_flag)
		cerr << "Running pre-intialisation code" << crlf;

	// Initialize the stack, a RET will jump to the end program trap.
	//set_sp (default_stack);
	//memwritew (default_stack, endrun_trap_addr);

	set_default_stack (endinit_trap_addr);
	set_pc (tpa_start_addr + 3);
	TrapReturn initresult= checkrun ();

	if (initresult != TrapEndInitRsx)
	{
		if (debugsystem_flag)
			cerr << "Unexpected exit of initialization code" <<
				crlf;
		return false;
	}

	if (debugsystem_flag)
		cerr << "End of pre-intialisation code" << crlf;

	// Relocation and activation of RSX.
	byte num_rsx= memread (tpa_start_addr + 0x0F);
	if (num_rsx > 0)
		attachloader ();
	for (byte i= 0; i < num_rsx; ++i)
	{
		z80word recrsx= tpa_start_addr + 0x10 + i * 0x10;
		if (debugsystem_flag)
		{
			cerr << "RSX record " << int (i) << crlf <<
				"Offset: " << memreadw (recrsx + 0) << crlf <<
				"Length: " << memreadw (recrsx + 2) << crlf <<
				"NB flag: " << int (memread (recrsx + 4) ) <<
					crlf <<
				"Name: " << setw (8) <<
				reinterpret_cast <char *>
					(mem + recrsx + 6) <<
				crlf;
		}
		// Attach it if is not only for non-banked systems.
		if (memread (recrsx + 4) != 0xFF)
			attachrsx (memreadw (recrsx + 0) + tpa_start_addr,
				memreadw (recrsx + 2) );
	}

	// Move the main program to the start of tpa.
	z80word len= memreadw (tpa_start_addr + 1);
	for (z80word i= 0; i < len; ++i)
		memwrite (tpa_start_addr + i,
			memread (tpa_start_addr + 0x100 + i) );

	return true;
}

bool CPM::Internal::loadprltpa (int handle)
{
	//int r= read (handle, mem + tpa_start_addr, bdos_base - tpa_start_addr);
	int r= read (handle, mem + tpa_start_addr,
		get_tpaend () - tpa_start_addr);
	if (r <= 0)
		return false;
	char c;

	if (read (handle, & c, 1) > 0) // File is too long.
		return false;

	if (memread (tpa_start_addr) != 0)
	{
		cerr << "Invalid PRL header" << crlf;
		return false;
	}
	z80word len= memreadw (tpa_start_addr + 1);
	// Check if header + code + relocation bitmap fits in the file.
	if (len + (len + 7) / 8 + 0x100 > r)
	{
		cerr << "Invalid PRL header" << crlf;
		return false;
	}


	// Relocation not needed to run the prl in the tpa.

	// Move the code to start of tpa.
	for (z80word i= 0; i < len; ++i)
		memwrite (tpa_start_addr + i,
			memread (tpa_start_addr + 0x100 + i) );

	return true;
}


namespace {


class ConsoleActivator {
public:
	ConsoleActivator (Console & console_n);
	~ConsoleActivator ();
private:
	Console & console;
};

ConsoleActivator::ConsoleActivator (Console & console_n) :
	console (console_n)
{
	console.activate ();
}

ConsoleActivator::~ConsoleActivator ()
{
	console.deactivate ();
}


} // namespace


void CPM::Internal::tpastart ()
{
	ConsoleActivator conact (console);

	runtransient ();
}

#if 0
bool CPM::Internal::loadcommand (const string & filename)
{
	if (debugsystem_flag)
		cerr << "Loading command " << filename << crlf;

	int handle= open (filename.c_str (), O_RDONLY);
	if (handle == -1)
	{
		cerr << "File not loaded" << crlf;
		return false;
	}
	bool r= loadfiletpa (handle);
	close (handle);
	return r;
}
#endif

string CPM::Internal::getcommandline ()
{
	memwrite (default_console_buffer, default_console_buffer_size);
	read_console_buffer (default_console_buffer);
	cout << crlf;
	return string (
		reinterpret_cast <char *>
			(mem + default_console_buffer + 2),
		memread (default_console_buffer + 1) );
}

namespace {


string getcommand (const string & str)
{
	const string::size_type l= str.find_first_of (" \t");
	if (l != string::npos)
		return str.substr (0, l);
	else
		return str;
}

bool getnumber (const string & str, z80word & n)
{
	const char * cstr= str.c_str ();
	char * aux;

	unsigned long l= (* cstr == '#') ?
		strtoul (cstr + 1, & aux, 10) :
		strtoul (cstr, & aux, 16);

	if (* aux != '\0')
		return false;
	if (l > 65535)
		return false;
	n= static_cast <z80word> (l);
	return true;
}

bool getbytenumber (const string & str, byte & n)
{
	const char * cstr= str.c_str ();
	char * aux;

	unsigned long l= (* cstr == '#') ?
		strtoul (cstr + 1, & aux, 10) :
		strtoul (cstr, & aux, 16);

	if (* aux != '\0')
		return false;
	if (l > 255)
		return false;
	n= static_cast <byte> (l);
	return true;
}

class Debugger {
public:
	Debugger (CPM::Internal & cpmi);
	~Debugger ();
	void doit ();
private:
	CPM::Internal & cpmi;

	typedef vector <string> vparam;
	enum Result { Ok, Bad, End };

	typedef map <z80word, byte> breakpoint_t;
	breakpoint_t breakpoint;

	typedef Result (Debugger::* command) (const vparam &);
	typedef map <char, command> mapfunc_t;
	static const mapfunc_t mapfunc;

	static mapfunc_t createmap ();
	static command getfunc (char letter);

	z80word lastlistpos;
	z80word displaypos;

	enum Flag { FlagC, FlagZ, FlagM, FlagE, FlagI };
	enum Reg { RegB, RegD, RegH, RegS, RegP, RegIX, RegIY };

	static vparam parseparam (string cmd);

	Result co_go (const vparam & vp);
	Result co_putbreakpoint (const vparam & vp);
	Result co_quit (const vparam & vp);
	void examineflag (Flag f);
	void examineA ();
	void examinereg (Reg r);
	Result co_examine (const vparam & vp);
	Result co_trace (const vparam & vp);
	Result co_untrace (const vparam & vp);
	z80word displayline (z80word from, z80word to);
	Result co_display (const vparam & vp);
	Result co_set (const vparam & vp);
	Result co_fill (const vparam & vp);
	Result co_move (const vparam & vp);
	#if defined USE_ImcZ80 && ! defined DISASM_WITH_libz80
	z80word disassembly (z80word pos);
	Result co_list (const vparam & vp);
	#endif
};

Debugger::Debugger (CPM::Internal & cpmi) :
	cpmi (cpmi),
	lastlistpos (tpa_start_addr),
	displaypos (tpa_start_addr)
{ }

Debugger::~Debugger ()
{
	cpmi.endruntransient ();
}

Debugger::mapfunc_t Debugger::createmap ()
{
	mapfunc_t m;

	m ['G']= & Debugger::co_go;
	m ['P']= & Debugger::co_putbreakpoint;
	m ['Q']= & Debugger::co_quit;
	m ['X']= & Debugger::co_examine;
	m ['T']= & Debugger::co_trace;
	m ['U']= & Debugger::co_untrace;
	m ['D']= & Debugger::co_display;
	m ['S']= & Debugger::co_set;
	m ['F']= & Debugger::co_fill;
	m ['M']= & Debugger::co_move;

	#if defined USE_ImcZ80 && ! defined DISASM_WITH_libz80
	m ['L']= & Debugger::co_list;
	#endif

	return m;
}

const Debugger::mapfunc_t Debugger::mapfunc (Debugger::createmap () );

Debugger::command Debugger::getfunc (char letter)
{
	mapfunc_t::const_iterator it= mapfunc.find (letter);
	if (it != mapfunc.end () )
		return it->second;
	else
		return 0;
}

Debugger::vparam Debugger::parseparam (string cmd)
{
	vparam v;
	string::size_type pos= 0;
	while ( (pos= cmd.find (',', pos) ) != string::npos)
	{
		v.push_back (cmd.substr (0, pos) );
		cmd.erase (0, pos + 1);
	}
	if (! cmd.empty () )
		v.push_back (cmd);
	return v;
}

Debugger::Result Debugger::co_go (const vparam & vp)
{
	typedef set <z80word> local_t;
	local_t localbreakpoint;

	z80word pos= cpmi.get_pc ();
	size_t nargs= vp.size ();
	if (nargs > 0)
	{
		if (! vp [0].empty () && ! getnumber (vp [0], pos) )
			return Bad;
		for (size_t i= 1; i < nargs; ++i)
		{
			z80word breakpos;
			if (! getnumber (vp [i], breakpos) )
				return Bad;
			localbreakpoint.insert (breakpos);
		}
	}
	cpmi.set_pc (pos);

	const local_t::iterator localend= localbreakpoint.end ();
	const breakpoint_t::iterator breakpointend= breakpoint.end ();
	for (;;)
	{
		// Check local breakpoints.
		if (localbreakpoint.find (cpmi.get_pc () ) != localend)
		{
			lastlistpos= cpmi.get_pc ();
			return Ok;
		}

		CPM::Internal::TrapReturn r= cpmi.runinstruction ();
		if (r == CPM::Internal::TrapEndProgram)
			return End;

		// Check persistent breakpoints after execution, to avoid
		// start executing after stoping in one.
		breakpoint_t::iterator it= breakpoint.find (cpmi.get_pc () );
		if (it != breakpointend)
		{
			cout << hex2 (it->second) <<
				" PASS " << hex4 (it->first) << crlf;
			if (--it->second == 0)
			{
				it->second= 1;
				lastlistpos= cpmi.get_pc ();
				return Ok;
			}
		}

	}

	return End;
}

Debugger::Result Debugger::co_putbreakpoint (const vparam & vp)
{
	if (vp.empty () )
	{
		for (breakpoint_t::iterator it= breakpoint.begin ();
			it != breakpoint.end ();
			++it)
		{
			cout << hex2 (it->second) << ' ' <<
				hex4 (it->first) << crlf;
		}
		return Ok;
	}
	size_t narg= vp.size ();
	z80word breakpos;
	if (! getnumber (vp [0], breakpos) )
		return Bad;
	byte count= 1;
	if (narg > 1)
	{
		if (! getbytenumber (vp [1], count) )
			return Bad;
	}
	if (count == 0)
		breakpoint.erase (breakpos);
	else
		breakpoint [breakpos]= count;
	return Ok;
}

Debugger::Result Debugger::co_quit (const vparam &)
{
	return End;
}

void Debugger::examineflag (Flag f)
{
	bool value;
	switch (f)
	{
	case FlagC: value= cpmi.flag_c (); break;
	case FlagZ: value= cpmi.flag_z (); break;
	case FlagM: value= cpmi.flag_s (); break;
	case FlagE: value= cpmi.flag_pv (); break;
	case FlagI: value= cpmi.flag_h (); break;
	}
	cout << (value ? '1' : '0') << ' '<< flush;
	string strvalue= cpmi.getcommandline ();
	if (strvalue.empty () )
		return;
	if (strvalue == "0")
		value= false;
	else if (strvalue == "1")
		value= true;
	else
	{
		cout << '?' << crlf;
		return;
	}
	switch (f)
	{
	case FlagC: cpmi.setflag_c (value); break;
	case FlagZ: cpmi.setflag_z (value); break;
	case FlagM: cpmi.setflag_s (value); break;
	case FlagE: cpmi.setflag_pv (value); break;
	case FlagI: cpmi.setflag_h (value); break;
	}
}

void Debugger::examineA ()
{
	byte value= cpmi.get_a ();
	cout << hex2 (value) << ' ' << flush;
	string strvalue= cpmi.getcommandline ();
	if (strvalue.empty () )
		return;
	byte newval;
	if (! getbytenumber (strvalue, newval) )
	{
		cout << '?' << crlf;
		return;
	}
	cpmi.set_a (newval);
}

void Debugger::examinereg (Reg r)
{
	z80word value;
	switch (r)
	{
	case RegB: value= cpmi.get_bc (); break;
	case RegD: value= cpmi.get_de (); break;
	case RegH: value= cpmi.get_hl (); break;
	case RegS: value= cpmi.get_sp (); break;
	case RegP: value= cpmi.get_pc (); break;
	case RegIX: value= cpmi.get_ix (); break;
	case RegIY: value= cpmi.get_iy (); break;
	}
	cout << hex4 (value) << ' ' << flush;
	string strvalue= cpmi.getcommandline ();
	if (strvalue.empty () )
		return;
	if (! getnumber (strvalue, value) )
	{
		cout << '?' << crlf;
		return;
	}
	switch (r)
	{
	case RegB: cpmi.set_bc (value); break;
	case RegD: cpmi.set_de (value); break;
	case RegH: cpmi.set_hl (value); break;
	case RegS: cpmi.set_sp (value); break;
	case RegP: cpmi.set_pc (value); break;
	case RegIX: cpmi.set_ix (value); break;
	case RegIY: cpmi.set_iy (value); break;
	}
}

Debugger::Result Debugger::co_examine (const vparam & vp)
{
	switch (vp.size () )
	{
	case 0:
		cpmi.showcurrentstate ();
		break;
	case 1:
		{
			const string & str= vp [0];
			if (str == "C")
				examineflag (FlagC);
			else if (str == "Z")
				examineflag (FlagZ);
			else if (str == "M")
				examineflag (FlagM);
			else if (str == "E")
				examineflag (FlagE);
			else if (str == "I")
				examineflag (FlagI);
			else if (str == "A")
				examineA ();
			else if (str == "B")
				examinereg (RegB);
			else if (str == "D")
				examinereg (RegD);
			else if (str == "H")
				examinereg (RegH);
			else if (str == "S")
				examinereg (RegS);
			else if (str == "P")
				examinereg (RegP);
			else if (str == "IX")
				examinereg (RegIX);
			else if (str == "IY")
				examinereg (RegIY);
			else
				cout << '?' << crlf;
		}
		break;
	default:
		return Bad;
	}
	return Ok;
}

Debugger::Result Debugger::co_trace (const vparam & vp)
{
	z80word n= 1;
	switch (vp.size () )
	{
	case 0:
		// Nothing.
		break;
	case 1:
		if (! getnumber (vp [0], n) )
			return Bad;
		break;
	default:
		return Bad;
	}
	for ( ; n > 0; --n)
	{
		cpmi.showcurrentstate ();
		#if 0
		cpmi.execute ();
		#else
		CPM::Internal::TrapReturn r= cpmi.runinstruction ();
		if (r == CPM::Internal::TrapEndProgram)
			return End;
		#endif

	}
	lastlistpos= cpmi.get_pc ();
	return Ok;
}

Debugger::Result Debugger::co_untrace (const vparam & vp)
{
	z80word n= 1;
	switch (vp.size () )
	{
	case 0:
		// Nothing.
		break;
	case 1:
		if (! getnumber (vp [0], n) )
			return Bad;
		break;
	default:
		return Bad;
	}
	cpmi.showcurrentstate ();
	for ( ; n > 0; --n)
	{
		CPM::Internal::TrapReturn r= cpmi.runinstruction ();
		if (r == CPM::Internal::TrapEndProgram)
			return End;
	}
	lastlistpos= cpmi.get_pc ();
	return Ok;
}

z80word Debugger::displayline (z80word from, z80word to)
{
	ASSERT (to > from && (to - from <= 16) );

	cout << hex4 (from) << "  ";
	for (z80word i= from; i < to; ++i)
		cout << hex2 (cpmi.memread (i) ) << ' ';
	for (int i= 0; i < 16 - (to - from); ++i)
		cout << "   ";

	for (z80word i= from; i < to; ++i)
	{
		unsigned char c= cpmi.memread (i);
		if (c < 32 || c > 127)
			c= '.';
		cout << c;
	}
	cout << crlf;
	return to;
}

Debugger::Result Debugger::co_display (const vparam & vp)
{
	size_t narg= vp.size ();
	if (narg > 0)
	{
		if (! vp [0].empty () )
			if (! getnumber (vp [0], displaypos) )
				return Bad;
	}
	if (narg > 1)
	{
		if (narg > 2)
			return Bad;
		z80word endpos;
		if (! getnumber (vp [1], endpos) )
			return Bad;
		++endpos;
		while (displaypos < endpos)
		{
			z80word endline= displaypos + 16;
			if (endline > endpos)
				endline= endpos;
			displaypos= displayline (displaypos, endline);
		}
		return Ok;
	}
	for (int i= 0; i < 12; ++i)
		displaypos= displayline (displaypos, displaypos + 16);
	return Ok;
}

Debugger::Result Debugger::co_set (const vparam & vp)
{
	if (vp.size () != 1)
		return Bad;
	z80word pos;
	if (! getnumber (vp [0], pos) )
		return Bad;
	for (;;)
	{
		cout << hex4 (pos) << ' ' <<
			hex2 (cpmi.memread (pos) ) << ' ' << flush;
		string strvalue= cpmi.getcommandline ();
		if (strvalue == ".")
			break;
		if (strvalue.empty () )
			++pos;
		else
		{
			byte val;
			if (! getbytenumber (strvalue, val) )
				cout << '?' << crlf;
			else
			{
				cpmi.memwrite (pos, val);
				++pos;
			}
		}
	}
	return Ok;
}

Debugger::Result Debugger::co_fill (const vparam & vp)
{
	if (vp.size () != 3)
		return Bad;
	z80word from;
	if (! getnumber (vp [0], from) )
		return Bad;
	z80word to;
	if (! getnumber (vp [1], to) || to == 0xFFFF)
		return Bad;
	if (to < from)
		return Bad;
	byte b;
	if (! getbytenumber (vp [2], b) )
		return Bad;
	for (z80word i= from; i <= to; ++i)
		cpmi.memwrite (i, b);
	return Ok;
}

Debugger::Result Debugger::co_move (const vparam & vp)
{
	if (vp.size () != 3)
		return Bad;
	z80word from;
	if (! getnumber (vp [0], from) )
		return Bad;
	z80word to;
	if (! getnumber (vp [1], to) || to == 0xFFFF)
		return Bad;
	if (to < from)
		return Bad;
	z80word dest;
	if (! getnumber (vp [2], dest) )
		return Bad;
	z80word size= to - from + 1;

	if (dest > from && dest <= to)
	{
		dest+= size - 1;
		for ( ; size > 0; --size)
			cpmi.memwrite (dest--, cpmi.memread (to--) );
	}
	else
	{
		for ( ; size > 0; --size)
			cpmi.memwrite (dest++, cpmi.memread (from++) );
	}
	return Ok;
}

#if defined USE_ImcZ80 && ! defined DISASM_WITH_libz80

z80word Debugger::disassembly (z80word pos)
{
	cout << hex4 (pos) << ' ';
	if (cpmi.memread (pos)  == trapbyte &&
		( cpmi.gettrapfunc (pos) ) != 0)
	{
		cout << "System trap";
		++pos;
	}
	else
		pos= cpmi.disassembly (cout, pos);
	cout << crlf;
	return pos;
}

Debugger::Result Debugger::co_list (const vparam & vp)
{
	//z80word pos= cpmi.get_pc ();
	z80word pos= lastlistpos;
	size_t nargs= vp.size ();
	if (nargs > 0)
	{
		if (! getnumber (vp [0], pos) )
			return Bad;
		if (nargs > 1)
		{
			z80word posend;
			if (! getnumber (vp [1], posend) )
				return Bad;
			if (nargs > 2)
				return Bad;
			while (pos < posend)
				pos= disassembly (pos);
			lastlistpos= pos;
			return Ok;
		}
	}
	for (int i= 0; i < 11; ++i)
		pos= disassembly (pos);
	lastlistpos= pos;
	return Ok;
}

#endif

void Debugger::doit ()
{
	string cmdline;
	for (;;)
	{
		cout << '#' << flush;
		cmdline= cpmi.getcommandline ();
		ltrim (cmdline);
		if (cmdline.empty () )
			continue;
		//char lettercommand= cmdline [0];

		cmdline= strupper (cmdline);
		char lettercommand= cpmupper (cmdline [0] );
		cmdline.erase (0, 1);
		vparam vp= parseparam (cmdline);
		Result result= Bad;

		command f= getfunc (lettercommand);
		if (f)
			result= (this->*f) (vp);

		if (result == Bad)
			cout << '?' << crlf;
		if (result == End)
			break;
	}
}


} // namespace


void CPM::Internal::debugger ()
{
	Debugger deb (* this);
	deb.doit ();
}

bool CPM::Internal::loadprogram (string cmdupper)
{
	setcmd_args (cmdupper);
	z80word pos= cmd_args + 1;
	const z80word endline= pos + memread (cmd_args);

	cleardefaultfcb (default_fcb_1, default_pass_1);
	parsefilecmdline (pos, endline,
		default_fcb_1, default_pass_1);

	size_t numdisk= static_cast <size_t> (memread (default_fcb_1) );
	if (numdisk != 0 && is_blank_fcb (default_fcb_1) )
	{
		--numdisk;
		if (! disk_exist (numdisk) )
		{
			cout << "Invalid drive" << crlf;
		}
		else
		{
			//default_disk= numdisk;
			set_default_disk (numdisk);
		}
		return false;
	}

	string ext= extension_from_fcb (default_fcb_1);
	if (ext.empty () )
		setcomextension (default_fcb_1);
	else
	{
		if (ext != "COM" && ext != "PRL")
		{
			cerr << "Invalid extension: '" << ext << "'" << crlf;
			cout << filespec_from_fcb (default_fcb_1) <<
				'?' << crlf;
			return false;
		}
	}

	z80word n= pos - (cmd_args + 1);
	cmdupper.erase (0, n);

	if (debugsystem_flag)
	{
		cout << "Executing " <<
			filespec_from_fcb (default_fcb_1) << 
			cmdupper << crlf;
	}

	if (! open_file (default_fcb_1) )
	{
		cout << "Error opening command" << crlf;
		return false;
	}
	OpenFile * pof= getopenfile (default_fcb_1);
	if (! pof)
		throw logic_error ("Algo huele a podrido");

	if (ext == "PRL")
	{
		cerr << "Extension: '" << ext << '\'' << crlf;
		if (! loadprltpa (pof->gethandle () ) )
		{
			cout << "Error loading command" << crlf;
			return false;
		}
	}
	else
	{
		if (! loadfiletpa (pof->gethandle () ) )
		{
			cout << "Error loading command" << crlf;
			return false;
		}
	}

	close_all ();
	setarguments (cmdupper);
	prepareruntransient ();
	return true;
}

bool CPM::Internal::builtinEXIT (const string & /*cmdline*/)
{
	return true;
}

bool CPM::Internal::builtinPOKE (const string & strcmd)
{
	istringstream iss (strcmd);
	z80word pos;
	string strpos;
	iss >> strpos;
	if (! getnumber (strpos, pos) )
	{
		cout << "POKE " << strcmd << '?' << crlf;
		return false;
	}
	for (;;)
	{
		char c;
		iss >> c; // >> take care of white space.
		if (! iss)
			break;
		if (c == '"')
		{
			while ( (c= iss.get () ) != '"' && iss)
			{
				memwrite (pos, c);
				++pos;
			}
		}
		else
		{
			iss.unget ();
			string strnum;
			iss >> strnum;
			byte num;
			if (! getbytenumber (strnum, num) )
			{
				cout << "@POKE " << strcmd << '?' << crlf;
				return false;
			}
			memwrite (pos, num);
			++pos;
		}
	}
	return false;
}

bool CPM::Internal::builtinGO (const string & strcmd)
{
	z80word start= tpa_start_addr;
	string cmdupper= strupper (strcmd);
	istringstream iss (cmdupper);
	string strstart;
	iss >> strstart;
	if (iss)
	{
		if (! getnumber (strstart, start) )
		{
			cout << "GO " << strstart <<
				'?' << crlf;
			return false;
		}
		iss.get ();
		if (! iss.eof () )
		{
			cout << "GO " << cmdupper <<
				'?' << crlf;
			return false;
		}
	}

	prepareruntransient ();
	set_pc (start);
	run ();
	endruntransient ();
	return false;
}

bool CPM::Internal::builtinLOAD (const string & strcmd)
{
	string cmdupper= strupper (strcmd);
	if (cmdupper.empty () )
	{
		cout << "LOAD what?" << crlf;
		return false;
	}
	loadprogram (cmdupper);

	return false;
}

bool CPM::Internal::builtinDEBUG (const string & strcmd)
{
	string cmdupper= strupper (strcmd);
	if (! cmdupper.empty () )
	{
		if (! loadprogram (cmdupper) )
			return false;
	}
	else
		prepareruntransient ();

	debugger ();
	endruntransient ();

	return false;
}

bool CPM::Internal::builtinDIR (const string & strcmd)
{
	setarguments (strcmd);
	FcbFile fcbmask (mem + default_fcb_1);
	if (fcbmask.all_blank () )
		fcbmask.setwild ();
	size_t disknum= disknum_from_fcbnum (fcbmask.getdrive () );
	if (! disk_exist (disknum) )
	{
		cout << "Invalid disk" << crlf;
		return false;
	}

	//char diskletter= static_cast <char> (disknum + 'A');
	byte diskletter= letterfromdisknum (disknum);
	string path= pathfromdisknum (disknum);

	if (debugsystem_flag)
		cerr << "Mask used: '" << fcbmask <<
			"' " << "Directory used: " << path << crlf;

	// Evaluate columns used for listing.
	size_t files_line= (console.columns () - 2) / 15;
	if (files_line == 0)
		files_line= 1;

	FindFile ff (path, fcbmask, debugsystem_flag);
	FcbFile result;
	FindFile::Result::code findcode;

	size_t numlisted= 0;
	while ( (findcode= ff.get (result) ) == FindFile::Result::Ok)
	{
		if (numlisted % files_line == 0)
		{
			if (numlisted != 0)
				cout << crlf;
			cout << diskletter << ": ";
		}
		else
			cout << " : ";
		cout << result;
		++numlisted;
	}
	if (findcode != FindFile::Result::NoFile)
		cout << "Error reading directory";
	else
		if (numlisted == 0)
			cout << "No file";
	cout << crlf;
	return false;
}

bool CPM::Internal::builtinERA (const string & strcmd)
{
	setarguments (strcmd);
	FcbFile fcbmask (mem + default_fcb_1);
	if (fcbmask.all_blank () )
	{
		cout << "ERA?" << crlf;
		return false;
	}

	size_t disknum= disknum_from_fcbnum (fcbmask.getdrive () );
	if (! disk_exist (disknum) )
	{
		cout << "Invalid disk" << crlf;
		return false;
	}
	string path= pathfromdisknum (disknum);

	if (debugsystem_flag)
		cerr << "ERA " << path << '/' << fcbmask << crlf;

	if (fcbmask.all_wild () )
	{
		cout << "ALL FILES (Y/N)?" << flush;
		memwrite (cmd_args, 1);
		read_console_buffer (cmd_args);
		cout << crlf;
		if (memread (cmd_args + 1) != 1)
			return false;
		byte c= memread (cmd_args + 2);
		if (c != 'Y' && c != 'y')
			return false;
	}

	set_de (default_fcb_1);
	bdos_delete_file ();
	switch (get_hl () )
	{
	case 0:
		break;
	case ErrNoFileMatch:
		cout << "No file" << crlf;
		break;
	default:
		cout << "Disk error" << crlf;
	}

	return false;
}

bool CPM::Internal::builtinUSER (const string & strcmd)
{
	istringstream iss (strcmd);
	unsigned int usernum;
	iss >> usernum;
	if (! iss)
	{
		if (iss.eof () )
			cout << "USER?";
		else
			cout << strcmd << '?';
		cout << crlf;
	}
	else
	{
		set_user (static_cast <byte> (usernum) );
	}
	return false;
}

CPM::Internal::Initbuilt::Initbuilt ()
{
	pr_built ["EXIT"]=  & CPM::Internal::builtinEXIT;
	pr_built ["POKE"]=  & CPM::Internal::builtinPOKE;
	pr_built ["GO"]=    & CPM::Internal::builtinGO;
	pr_built ["LOAD"]=  & CPM::Internal::builtinLOAD;
	pr_built ["DEBUG"]= & CPM::Internal::builtinDEBUG;

	built ["DIR"]=    & CPM::Internal::builtinDIR;
	built ["ERA"]=    & CPM::Internal::builtinERA;
	built ["USER"]=   & CPM::Internal::builtinUSER;
}

const CPM::Internal::Initbuilt CPM::Internal::Initbuilt::instance;

const CPM::Internal::built_t & CPM::Internal::pr_built=
	CPM::Internal::Initbuilt::instance.pr_built;
const CPM::Internal::built_t & CPM::Internal::built=
	CPM::Internal::Initbuilt::instance.built;

const CPM::Internal::built_t::const_iterator CPM::Internal::prbtend=
	CPM::Internal::pr_built.end ();
const CPM::Internal::built_t::const_iterator CPM::Internal::btend=
	CPM::Internal::built.end ();


CPM::Internal::builtinfunc_t CPM::Internal::find_builtin
	(const string & comm)
{
	builtinfunc_t bfunc= 0;
	built_t::const_iterator it;
	if (bprefix == '\0')
	{
		it= pr_built.find (comm);
		if (it != prbtend)
		{
			bfunc= it->second;
		}
		else
		{
			it= built.find (comm);
			if (it != btend)
				bfunc= it->second;
		}
	}
	else
	{
		if (comm [0] == bprefix)
		{
			it= pr_built.find (comm.substr (1) );
			if (it != btend)
				bfunc= it->second;
		}
		else
		{
			it= built.find (comm);
			if (it != btend)
				bfunc= it->second;
		}
	}
	return bfunc;
}

void CPM::Internal::interactive ()
{
	ConsoleActivator conact (console);

	string cmdline;
	for (;;)
	{
		byte usernum= get_user ();
		if (usernum != 0)
		cout << static_cast <unsigned int> (usernum);
		cout << static_cast <char> (get_default_disk_letter () ) <<
			'>' << flush;

		cmdline= getcommandline ();
		ltrim (cmdline);

		if (cmdline.empty () )
			continue;
		if (cmdline [0] == ';')
			continue;

		string cmdupper= strupper (cmdline);

		string command= getcommand (cmdupper);

		builtinfunc_t bfunc= find_builtin (command);

		if (bfunc)
		{
			cmdupper.erase (0, command.size () );
			ltrim (cmdupper);
			bool r= (this->* bfunc) (cmdupper);
			if (r)
				break;
		}
		else
		{
			if (loadprogram (cmdupper) )
			{
				run ();
				endruntransient ();
			}
		}
		cout << crlf;
	}
}

void CPM::Internal::runprogram (const string & filename, const string & args)
{
	if (debugsystem_flag)
		cerr << "Loading command " << filename << crlf;

	int handle= open (filename.c_str (), O_RDONLY);
	if (handle == -1)
		throw runtime_error ("Error loading file: " + filename);

	GuardHandle guard (handle);
	bool r= loadfiletpa (handle);
	guard.release ();
	close (handle);

	if (! r)
		throw runtime_error ("Error loading program");

	setarguments (args);
	tpastart ();
}

void CPM::Internal::runcommandline (const string & commandline)
{
	string line= commandline;
	string command= getcommand (line);

	builtinfunc_t bfunc= find_builtin (command);

	if (bfunc)
	{
		line.erase (0, command.size () );
		ltrim (line);
		//bool r= (this->* bfunc) (line);
		//if (r)
		//	break;
		(void) (this->* bfunc) (line);
	}
	else
	{
		if (loadprogram (commandline) )
		{
			//run ();
			//endruntransient ();
			tpastart ();
		}
	}
}


//*********************************************************
//				CPM
//*********************************************************


CPM::CPM (Console & console_n, Printer & printer_n,
		const CpmOptions & options) :
	p (new Internal (console_n, printer_n, options) )
{

}

CPM::~CPM ()
{
	delete p;
}

void CPM::setdisk (char disk, const std::string & path)
{
	p->setdisk (disk, path);
}

char CPM::newdisk (const std::string & path)
{
	return p->newdisk (path);
}

void CPM::interactive ()
{
	p->interactive ();
}

void CPM::runprogram
		(const std::string & filename, const std::string & args)
{
	p->runprogram (filename, args);
}

void CPM::runcommandline (const std::string & commandline)
{
	p->runcommandline (commandline);
}


// End of cpm.cpp
