// directory.cpp
// Revision 17-mar-2004

#include "directory.h"

#include <algorithm>

#include <glob.h>

#include <vector>

namespace {

bool validcpmfilename (const std::string & result,
	std::string::size_type maxfilelen,
	std::string::size_type maxextlen)
{
	std::string file;
	std::string ext;
	const std::string::size_type point= result.find ('.');
	if (point != std::string::npos)
	{
		file= result.substr (0, point);
		ext= result.substr (point + 1);
	}
	else
		file= result;

	if (file.size () > maxfilelen)
		return false;
	if (ext.size () > maxextlen)
		return false;

	// Only one '.' allowed,
	if (ext.find ('.') != std::string::npos)
		return false;

	#if 0
	// No lowercase.
	for (std::string::size_type i= 0, l= file.size (); i < l; ++i)
		if (islower (file [i]) )
			return false;
	for (std::string::size_type i= 0, l= ext.size (); i < l; ++i)
		if (islower (ext [i]) )
			return false;
	#endif
	return true;
}

}

//***********************************************
//		FindFile::Internal
//***********************************************


class FindFile::Internal {
public:
	Internal ();
	~Internal ();
	std::string findfirst (const std::string & mask);
	std::string findnext ();
private:
	void closesearch ();
	bool finding;
	size_t n;
	std::vector <std::string> vfile;
};

FindFile::Internal::Internal () :
	finding (false)
{
}

FindFile::Internal::~Internal ()
{
	if (finding)
		closesearch ();
}

void FindFile::Internal::closesearch ()
{
	finding= false;
	vfile.clear ();
}

std::string FindFile::Internal::findfirst (const std::string & mask)
{
	if (finding)
		closesearch ();

	std::string file;
	std::string ext;
	const std::string::size_type pos= mask.find ('.');
	if (pos == std::string::npos)
		file= mask;
	else
	{
		file= mask.substr (0, pos);
		ext= mask.substr (pos + 1);
	}

	// CP/M allows that a ? replaces an empty position,
	// glob does not. Thus we replace the last '?' with
	// a '*', but save the max length allowed to test
	// the results against.
	const std::string::size_type maxfilelen= file.size ();
	const std::string::size_type maxextlen= ext.size ();
	int l= file.size () - 1;
	while (l >= 0 && file [l] == '?')
		--l;
	if (l >= 0)
	{
		file= file.substr (0, l + 1);
		if (file.size () < 8)
			file+= '*';
	}
	else
		file= "*";
	l= ext.size () - 1;
	while (l >= 0 && ext [l] == '?')
		--l;
	if (l >= 0)
	{
		ext= ext.substr (0, l + 1);
		if (ext.size () < 3)
			ext+= '*';
	}
	else
		ext= "*";

	std::string s= file + '.' + ext;

	int r;
	glob_t g;
	int flag= GLOB_NOSORT; // Don't sort, we do it.
	switch ( (r= glob (s.c_str (), flag, NULL, & g) ) )
	{
	case 0:
	case GLOB_NOMATCH:
		break;
	default:
		return std::string ();
	}

	// I want that FILE is found when searching for FILE.*,
	// thus we do a second search without extension in
	// that case.
	if (r == 0)
		flag|= GLOB_APPEND;
	if (ext == "*")
		glob (file.c_str (), flag, NULL, &g);

	if (g.gl_pathc == 0)
	{
		globfree (& g);
		return std::string ();
	}

	finding= true;
	for (size_t i= 0; i < static_cast <size_t> (g.gl_pathc); ++i)
	{
		std::string l (g.gl_pathv [i] );
		if (validcpmfilename (l, maxfilelen, maxextlen) )
			vfile.push_back (l);
	}

	globfree (& g);

	// As two glob operations were done, we need to
	// remove duplicates.	
	std::sort (vfile.begin (), vfile.end () );
	std::vector <std::string>::iterator it=
		unique (vfile.begin (), vfile.end () );
	vfile.erase (it, vfile.end () );

	// TODO: still a problem, if FILE and FILE. exists
	// CP/M see the two but can open only one.

	n= 0;
	return findnext ();
}

std::string FindFile::Internal::findnext ()
{
	if (! finding)
		return std::string ();
	if (n >= vfile.size () )
	{
		closesearch ();
		return std::string ();
	}
	return vfile [n++];
}

//***********************************************
//		FindFile
//***********************************************


FindFile::FindFile () :
	in (new Internal)
{
}

FindFile::~FindFile ()
{
	delete in;
}

std::string FindFile::findfirst (const std::string & mask)
{
	return in->findfirst (mask);
}

std::string FindFile::findnext ()
{
	return in->findnext ();
}


// End of directory.cpp
