// cpm.cpp
// Revision 20-mar-2004

#include "cpm.h"

#include "cpu.h"
#include "directory.h"

#include <vector>
#include <map>
#include <set>
#include <algorithm>
#include <sstream>
#include <stdexcept>

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

//#include <glob.h>

#include <assert.h>
#define ASSERT assert

//***********************************************
//	For testing and debugging.
//***********************************************

#include <iostream>
#include <iomanip>
using std::ostream;
using std::cin;
using std::cout;
using std::cerr;
using std::endl;
using std::flush;
using std::uppercase;
using std::setw;
using std::setfill;

//***********************************************
//		Auxiliar functions.
//***********************************************

namespace {

ostream & crlf (ostream & os)
{
	os << '\r' << endl;
	return os;
}

std::string strupper (const std::string & str)
{
	std::string result;
	for (std::string::size_type i= 0; i < str.size (); ++i)
		result+= static_cast <char> (toupper (str [i] ) );
	return result;
}

} // namespace

//***********************************************

// Value used for system trap.

const byte trapbyte= 0x76; // HALT code.

// Code for JP nn instruction, used in jump tables.
const byte jp_code= 0xC3;
// Code for RET instructio, used in .COM header
const byte ret_code= 0xC9;

// Address in page 0.

const word reset_call= 0;
const word bdos_call= 5;
const word tpa= 0x100;
const word default_fcb_1= 0x005C;
const word default_fcb_2= 0x006C;
const word default_pass_1= 0x0051;
const word default_pass_2= 0x0054;
const word default_dma_addr= 0x0080;
const word cmd_args= 0x0080;

// BIOS and BDOS addresses.

const word default_stack_size= 64; // 32 16 bit entries.
const word default_stack= 0xFFFD;
const word default_stack_base= default_stack - default_stack_size;

//const word num_bios_calls= 17; // CP/M 2.2 bios calls
const word num_bios_calls= 33; // CP/M Plus bios calls

const word bios_traps= default_stack_base - num_bios_calls;

// bios_base and bdos_base_default must be in 256 bytes page boundaries.

const word aux_bios_base= bios_traps - num_bios_calls * 3;
const word bios_base= (aux_bios_base / 256) * 256;

const word bdos_base_default= bios_base - 256;
// Jump to BDOS must be at offset 6 from bdos_base_default
const word bdos_jump_default= bdos_base_default + 6;
const word bdos_trap= bdos_jump_default + 3;
// Put here the end run trap for convenience.
const word end_run_trap= bdos_trap + 1;

// Bdos first page used as buffer for command line
const word default_console_buffer_size= 128;
const word default_console_buffer= end_run_trap + 1;

const word sector_size= 128;

// Fcb fields and auxiliary data.
const word fcb_extension= 12;
const word fcb_current_record= 32;
const word fcb_random_record= 33;
const word records_in_extension= 128;

// RSX header fields.
const word rsx_serial= 0;
const word rsx_jump_start= 6;
const word rsx_start= 7;
const word rsx_jump_next= 9;
const word rsx_next= 10;
const word rsx_prev= 12;
const word rsx_remove= 14;
const word rsx_nonbank= 15;
const word rsx_name= 16;
const word rsx_loader= 24;
const word rsx_reserved= 25;

//*********************************************************
//		OpenFile
//*********************************************************

class OpenFile {
public:
	OpenFile (int handle);
	int gethandle () const { return handle; }
	void close ();
	int read (void * buffer, int count)
	{
		return ::read (handle, buffer, count);
	}
	int write (void * buffer, int count)
	{
		return ::write (handle, buffer, count);
	}
	long lseek (long offset, int whence)
	{
		return ::lseek (handle, offset, whence);
	}
private:
	int handle;
};

OpenFile::OpenFile (int handle) :
	handle (handle)
{
}

void OpenFile::close ()
{
	::close (handle);
}

//*********************************************************
//		CPM::Internal
//*********************************************************

class CPM::Internal : public Cpu {
	Console & console;
	Printer & printer;
public:
	enum TrapReturn { TrapContinue, TrapEndProgram, TrapRet };
private:
	typedef TrapReturn (Internal::* TrapFunc) (void);
	typedef std::map <word, TrapFunc> trapfuncmap_t;
	static trapfuncmap_t trapfuncmap;
public:
	static TrapFunc gettrapfunc (word addr);
private:

	typedef std::map <byte, TrapFunc> bdosmap_t;
	static bdosmap_t bdosmap;

	word bdos_base;
	word bdos_jump;

	static bool funcmaps_inited;
	static bool funcmaps_init ();
public:
	Internal (Console & console, Printer & printer);
	virtual ~Internal ();

	void settpaend (unsigned short tpaend);

	void close_all ();

	bool debugsystem;
	bool debugz80;

	// Testing, to support Hi-Tech C.
	#define MAP_BY_NAME

	#ifndef MAP_BY_NAME
	typedef std::map <word, OpenFile> fcbmap_t;
	#else
	typedef std::map <std::string, OpenFile> fcbmap_t;
	#endif
	fcbmap_t fcbmap;

	typedef std::map <word, FindFile *> findfilemap_t;
	findfilemap_t findfilemap;

	void clean_fcb_entry (word fcb);
	OpenFile * getopenfile (word fcb);
	void insertopenfile (word fcb, int handle);

	byte multi_sector_count;

	void memwritejump (word address, word destination);

	void read_console_buffer (word buffer);

	unsigned long fcb_get_random_record (word fcb);

	std::string filename_from_fcb (word fcb);
	std::string extension_from_fcb (word fcb);
	void setcomextension (word fcb);
	bool open_file (word fcb);
	bool make_file (word fcb);
	word dma_addr;

	//*********** Trap functions **********

	// End run trap.
	TrapReturn end_run ();

	// BIOS functions.
	TrapReturn bios_COLDSTART ();
	TrapReturn bios_WARMSTART ();
	TrapReturn bios_CONSTATUS ();
	TrapReturn bios_CONINPUT ();
	TrapReturn bios_CONOUTPUT ();
	TrapReturn bios_LISTOUT ();
	TrapReturn bios_PUNCH ();
	TrapReturn bios_READER ();
	TrapReturn bios_HOME ();
	TrapReturn bios_SETDISD ();
	TrapReturn bios_SETTRACK ();
	TrapReturn bios_SETSECTOR ();
	TrapReturn bios_SETDMA ();
	TrapReturn bios_READDISK ();
	TrapReturn bios_WRITEDISK ();
	TrapReturn bios_LISTSTATUS ();
	TrapReturn bios_SECTORTRAN ();

	TrapReturn bios_USERF ();

	TrapReturn bios_unsupported ();

	// RST.
	TrapReturn UnsupportedRst ();

	// BDOS functions.
	TrapReturn bdos ();
	void bdos_return_hl (byte h, byte l);
	void bdos_return_hl (word reghl);
	TrapReturn bdos_system_reset (); // 00
	TrapReturn bdos_console_input (); // 01
	TrapReturn bdos_console_output (); // 02
	TrapReturn bdos_list_output (); // 05
	TrapReturn bdos_direct_console_io (); // 06
	TrapReturn bdos_write_string (); // 09
	TrapReturn bdos_read_console_buffer (); // 0A
	TrapReturn bdos_get_console_status (); // 0B
	TrapReturn bdos_return_version_number (); // 0C
	TrapReturn bdos_reset_disk_system (); // 0D
	TrapReturn bdos_select_disk (); // 0E
	TrapReturn bdos_open_file (); // 0F
	TrapReturn bdos_close_file (); // 10

	void setsearchresult (const std::string & result);

	TrapReturn bdos_search_for_first (); // 11
	TrapReturn bdos_search_for_next (); // 12
	TrapReturn bdos_delete_file (); // 13
	TrapReturn bdos_read_sequential (); // 14
	TrapReturn bdos_write_sequential (); // 15
	TrapReturn bdos_make_file (); //16
	TrapReturn bdos_rename_file (); // 17
	TrapReturn bdos_return_current_disk (); // 19
	TrapReturn bdos_set_dma_address (); // 1A
	TrapReturn bdos_map_read_only (); // 1D
	TrapReturn bdos_set_file_attributes (); // 1E
	TrapReturn bdos_get_set_user_code (); // 20
	TrapReturn bdos_read_random (); // 21
	TrapReturn bdos_write_random (); // 22
	TrapReturn bdos_compute_file_size (); // 23
	TrapReturn bdos_set_random_record (); // 24
	TrapReturn bdos_set_multi_sector_count (); // 2C
	TrapReturn bdos_set_bdos_error_mode (); // 2D
	TrapReturn bdos_get_set_system_control_block (); // 31
	TrapReturn bdos_call_resident_system_extension (); // 3C
	TrapReturn bdos_get_read_file_date_and_pass_mode (); // 66
	TrapReturn bdos_get_set_console_mode (); // 6D
	TrapReturn bdos_parse_filename (); // 98

	void parsefilefcb (word & name, word fcb);
	void cleardefaultfcb (word fcb, word pass);
	void parsefilecmdline (word & pos, word endline,
		word fcb, word pass);
	void setcmd_args (const std::string & args);
	void setarguments (const std::string & args);
	void showcurrentstate ();
	TrapReturn runinstruction ();
	void runner ();
	void run (bool debugsystem, bool debugz80);
	void prepareruntransient ();
	void endruntransient ();
	void runtransient (bool debugsystem, bool debugz80);
	void attachloader ();
	void attachrsx (word image, word len);
	bool loadfiletpa (int handle);
	bool loadprltpa (int handle);
	void tpastart (bool ndebugsystem, bool ndebugz80);
	bool loadcommand (const std::string & filename);
	std::string getcommandline ();
	void debugger ();
	void poke (const std::string & strcmd);
	void interactive (bool debugsystem, bool debugz80);
};

// Static vars.

CPM::Internal::trapfuncmap_t CPM::Internal::trapfuncmap;

CPM::Internal::bdosmap_t CPM::Internal::bdosmap;

bool CPM::Internal::funcmaps_init ()
{
	trapfuncmap [end_run_trap]=   & Internal::end_run;

	// BIOS traps.

	trapfuncmap [bios_traps +  0]= & Internal::bios_COLDSTART;
	trapfuncmap [bios_traps +  1]= & Internal::bios_WARMSTART;
	trapfuncmap [bios_traps +  2]= & Internal::bios_CONSTATUS;
	trapfuncmap [bios_traps +  3]= & Internal::bios_CONINPUT;
	trapfuncmap [bios_traps +  4]= & Internal::bios_CONOUTPUT;
	trapfuncmap [bios_traps +  5]= & Internal::bios_LISTOUT;
	trapfuncmap [bios_traps +  6]= & Internal::bios_PUNCH;
	trapfuncmap [bios_traps +  7]= & Internal::bios_READER;
	trapfuncmap [bios_traps +  8]= & Internal::bios_HOME;
	trapfuncmap [bios_traps +  9]= & Internal::bios_SETDISD;
	trapfuncmap [bios_traps + 10]= & Internal::bios_SETTRACK;
	trapfuncmap [bios_traps + 11]= & Internal::bios_SETSECTOR;
	trapfuncmap [bios_traps + 12]= & Internal::bios_SETDMA;
	trapfuncmap [bios_traps + 13]= & Internal::bios_READDISK;
	trapfuncmap [bios_traps + 14]= & Internal::bios_WRITEDISK;
	trapfuncmap [bios_traps + 15]= & Internal::bios_LISTSTATUS;
	trapfuncmap [bios_traps + 16]= & Internal::bios_SECTORTRAN;

	for (int i= 17; i < num_bios_calls; ++i)
		trapfuncmap [bios_traps + i]= & Internal::bios_unsupported;

	trapfuncmap [bios_traps + 30]= & Internal::bios_USERF;

	for (int i= 0x08; i <= 0x38; i+= 0x08)
		trapfuncmap [i]= & Internal::UnsupportedRst;


	trapfuncmap [bdos_trap]= & Internal::bdos;

	// BDOS functions.

	bdosmap [0x00]= & Internal::bdos_system_reset;
	bdosmap [0x01]= & Internal::bdos_console_input;
	bdosmap [0x02]= & Internal::bdos_console_output;
	bdosmap [0x05]= & Internal::bdos_list_output;
	bdosmap [0x06]= & Internal::bdos_direct_console_io;
	bdosmap [0x09]= & Internal::bdos_write_string;
	bdosmap [0x0A]= & Internal::bdos_read_console_buffer;
	bdosmap [0x0B]= & Internal::bdos_get_console_status;
	bdosmap [0x0C]= & Internal::bdos_return_version_number;
	bdosmap [0x0D]= & Internal::bdos_reset_disk_system;
	bdosmap [0x0E]= & Internal::bdos_select_disk;
	bdosmap [0x0F]= & Internal::bdos_open_file;
	bdosmap [0x10]= & Internal::bdos_close_file;
	bdosmap [0x11]= & Internal::bdos_search_for_first;
	bdosmap [0x12]= & Internal::bdos_search_for_next;
	bdosmap [0x13]= & Internal::bdos_delete_file;
	bdosmap [0x14]= & Internal::bdos_read_sequential;
	bdosmap [0x15]= & Internal::bdos_write_sequential;
	bdosmap [0x16]= & Internal::bdos_make_file;
	bdosmap [0x17]= & Internal::bdos_rename_file;
	bdosmap [0x19]= & Internal::bdos_return_current_disk;
	bdosmap [0x1A]= & Internal::bdos_set_dma_address;
	bdosmap [0x1D]= & Internal::bdos_map_read_only;
	bdosmap [0x1E]= & Internal::bdos_set_file_attributes;
	bdosmap [0x20]= & Internal::bdos_get_set_user_code;
	bdosmap [0x21]= & Internal::bdos_read_random;
	bdosmap [0x22]= & Internal::bdos_write_random;
	bdosmap [0x23]= & Internal::bdos_compute_file_size;
	bdosmap [0x24]= & Internal::bdos_set_random_record;
	bdosmap [0x2C]= & Internal::bdos_set_multi_sector_count;
	bdosmap [0x2D]= & Internal::bdos_set_bdos_error_mode;
	bdosmap [0x31]= & Internal::bdos_get_set_system_control_block;
	bdosmap [0x3C]= & Internal::bdos_call_resident_system_extension;
	bdosmap [0x66]= & Internal::bdos_get_read_file_date_and_pass_mode;
	bdosmap [0x6D]= & Internal::bdos_get_set_console_mode;
	bdosmap [0x98]= & Internal::bdos_parse_filename;

	return true;
}

bool CPM::Internal::funcmaps_inited= CPM::Internal::funcmaps_init ();

CPM::Internal::Internal (Console & console, Printer & printer) :
	console (console),
	printer (printer),
	bdos_base (bdos_base_default),
	bdos_jump (bdos_jump_default),
	//self (putvi (this) ),
	debugsystem (false),
	debugz80 (false),
	multi_sector_count (1)
{
	std::fill (mem, mem + sizeof (mem), 0xE7); // RST 20h, for testing.
	//std::fill (mem+ 0x100, mem + sizeof (mem), 0xE7); // RST 20h, for testing.
	//std::fill (mem, mem + sizeof (mem), 0x00);

	// Set traps.
	for (trapfuncmap_t::iterator it= trapfuncmap.begin ();
		it != trapfuncmap.end ();
		++it)
	{
		memwrite (it->first, trapbyte);
	}

	// BIOS Jump table: Jumps to the position of the traps.
	for (int i= 0; i < num_bios_calls; ++i)
		memwritejump (bios_base + 3 * i, bios_traps + i);

	// Set Warm reset call in page 0.
	memwritejump (reset_call, bios_base + 3);

	// Set BDOS Jump to trap.
	memwritejump (bdos_jump_default, bdos_trap);

	// Set BDOS Call in page 0.
	memwritejump (bdos_call + 0, bdos_jump_default);
}

CPM::Internal::~Internal ()
{
	close_all ();
}

void CPM::Internal::settpaend (unsigned short tpaend)
{
	if (tpaend <= tpa || tpaend >= bdos_base_default ||
		(tpaend & 0xF) != 0)
	{
		throw std::runtime_error ("TPA end limit invalid");
	}
	bdos_base= tpaend;
	bdos_jump= tpaend + 6;

	// Set BDOS Jump to trap.
	memwritejump (bdos_jump, bdos_trap);

	// Set BDOS Call in page 0.
	memwritejump (bdos_call + 0, bdos_jump);
}

void CPM::Internal::clean_fcb_entry (word fcb)
{
	{
		#ifndef MAP_BY_NAME
		fcbmap_t::iterator it= fcbmap.find (fcb);
		if (it != fcbmap.end () )
			fcbmap.erase (fcb);
		#else
		std::string str= filename_from_fcb (fcb);
		fcbmap_t::iterator it= fcbmap.find (str);
		if (it != fcbmap.end () )
			fcbmap.erase (str);
		#endif
	}
	{
		findfilemap_t::iterator it= findfilemap.find (fcb);
		if (it != findfilemap.end () )
		{
			delete it->second;
			findfilemap.erase (fcb);
		}
	}
}

OpenFile * CPM::Internal::getopenfile (word fcb)
{
	#ifndef MAP_BY_NAME
	fcbmap_t::iterator it= fcbmap.find (fcb);
	#else
	fcbmap_t::iterator it= fcbmap.find (filename_from_fcb (fcb) );
	#endif

	if (it == fcbmap.end () )
		return NULL;
	return & it->second;
}

void CPM::Internal::insertopenfile (word fcb, int handle)
{
	#ifndef MAP_BY_NAME
	fcbmap.insert (std::make_pair (fcb, OpenFile (handle) ) );
	#else
	std::string str= filename_from_fcb (fcb);
	fcbmap.insert (std::make_pair (str, OpenFile (handle) ) );
	#endif
}

void CPM::Internal::close_all ()
{
	{
		for (fcbmap_t::iterator it= fcbmap.begin ();
				it != fcbmap.end ();
				++it)
			it->second.close ();
	}
	{
		for (findfilemap_t::iterator it= findfilemap.begin ();
				it != findfilemap.end ();
				++it)
			delete it->second;
	}
	fcbmap.clear ();
	findfilemap.clear ();
}

void CPM::Internal::memwritejump (word address, word destination)
{
	#if 0
	mem [address + 0]= jp_code;
	mem [address + 1]= destination & 0xFF;
	mem [address + 2]= destination >> 8;
	#else
	memwrite (address, jp_code);
	memwritew (address + 1, destination);
	#endif
}

void CPM::Internal::read_console_buffer (word buffer)
{
	const byte len= memread (buffer);
	byte i= 0;
	bool waiting= true;
	while (waiting)
	{
		char c= console.charin ();
		switch (c)
		{
		case '\r':
		case '\n':
			waiting= false;
			break;
		case 0x7F:
		case 0x08:
			if (i > 0)
			{
				console.left ();
				console.charout (' ');
				console.left ();
				--i;
			}
			else
				console.charout (7);
			break;
		case 0x01: // Ctrl-A
			if (i > 0)
			{
				console.left ();
				--i;
			}
			else
				console.charout (7);
			break;
		default:
			if (i < len)
			{
				console.charout (c);
				memwrite (buffer + 2 + i, c);
				++i;
			}
			else
				console.charout (7);
		}
	} 
	memwrite (buffer + 1, i);
}

unsigned long CPM::Internal::fcb_get_random_record (word fcb)
{
	word a= fcb + fcb_random_record;
	return memread (a) +
		(static_cast <unsigned long> (memread (a + 1) ) << 8) +
		(static_cast <unsigned long> (memread (a + 2) ) << 16);
}

std::string CPM::Internal::filename_from_fcb (word fcb)
{
	word l;
	for (l= 9; l > 1; --l)
		if ( (memread (fcb + l - 1) & 0x7F) != ' ')
			break;
	std::string str;
	for (word i= 1; i < l; ++i)
		str+= memread (fcb + i) & 0x7F;
	for (l= 12; l > 9; --l)
		if (memread (fcb + l - 1) != ' ')
			break;
	if (l > 9)
	{
		str+= '.';
		for (word i= 9; i < l; ++i)
			str+= memread (fcb + i);
	}
	return str;
}

std::string CPM::Internal::extension_from_fcb (word fcb)
{
	word l;
	for (l= 12; l > 9; --l)
		if (memread (fcb + l - 1) != ' ')
			break;
	std::string str;
	if (l > 9)
	{
		for (word i= 9; i < l; ++i)
			str+= memread (fcb + i);
	}
	return str;
}

void CPM::Internal::setcomextension (word fcb)
{
	memwrite (fcb +  9, 'C');
	memwrite (fcb + 10, 'O');
	memwrite (fcb + 11, 'M');
}

bool CPM::Internal::open_file (word fcb)
{
	clean_fcb_entry (fcb);
	std::string filename= filename_from_fcb (fcb);
	if (debugsystem)
		cerr << "Opening '" << filename << '\'' << crlf;

	int h= open (filename.c_str (), O_RDWR);
	if (h == -1)
		return false;
	try
	{
		//fcbmap.insert (std::make_pair (fcb, OpenFile (h) ) );
		insertopenfile (fcb, h);
	}
	catch (...)
	{
		close (h);
		throw;
	}

	// Testing.
	memwrite (fcb + fcb_current_record, 0);

	return true;
}

bool CPM::Internal::make_file (word fcb)
{
	clean_fcb_entry (fcb);
	std::string filename= filename_from_fcb (fcb);
	if (debugsystem)
		cerr << "Making '" << filename << '\'' << crlf;

	int h= open (filename.c_str (), O_RDWR | O_CREAT | O_EXCL, 0666);
	if (h == -1)
		return false;
	try
	{
		//fcbmap.insert (std::make_pair (fcb, OpenFile (h) ) );
		insertopenfile (fcb, h);
	}
	catch (...)
	{
		close (h);
		throw;
	}

	// Testing.
	memwrite (fcb + fcb_current_record, 0);

	return true;
}

CPM::Internal::TrapFunc CPM::Internal::gettrapfunc (word addr)
{
	trapfuncmap_t::iterator it= trapfuncmap.find (addr);
	if (it == trapfuncmap.end () )
		return 0;
	else
		return it->second;
}


CPM::Internal::TrapReturn CPM::Internal::end_run ()
{
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_COLDSTART ()
{
	if (debugsystem)
		cerr << crlf << "BIOS COLDSTART" << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_WARMSTART ()
{
	if (debugsystem)
		cerr << crlf << "BIOS WARMSTART" << crlf;

	// Remove RSX flagged for removing.

	word previous= bdos_call + 1;
	word actual= memreadw (previous);
	word aux= 0;
	bool some_remains= false;
	while (actual < bdos_base)
	{
		if (actual <= aux)
		{
			cerr << "RSX chain is incorrect. Clearing all RSX" <<
				crlf;
			memwritew (bdos_call + 1, bdos_jump);
			break;
		}
		aux= actual;
		word base= actual & 0xFF00;
		actual= memreadw (base + rsx_next);
		bool is_loader= memread (base + rsx_loader) == 0xFF;
		if (memread (base + rsx_remove) == 0xFF)
		{
			if (! is_loader || ! some_remains)
			{
				if (is_loader)
					cerr << "Removing loader" << crlf;
				else
					cerr << "Removing " << hex4 (base) <<
						crlf;
				memwritew (previous, actual);
			}
		}
		else
		{
			some_remains= true;
			previous= base + rsx_next;
			if (is_loader)
				cerr << "Not removing loader" << crlf;
			else
				cerr << "Not removing " << hex4 (base) <<
					crlf;
		}
	}

	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_CONSTATUS ()
{
	// Entry: none.
	// Output: A= 0 -> No character waiting, 0xFF -> Yes.
	//if (debugsystem)
	//	cerr << "CONSTATUS called." << crlf;
	//z.R1.br.A= 0;
	//set_a (0);
	set_a (console.charavailable () ? 0xFF : 0x00);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::bios_CONINPUT ()
{
	// Entry: none.
	// Output: A= Character
	//cerr << "CONINPUT called." << crlf;
	//z.R1.br.A= 0;
	//char c= cin.get ();
	char c= console.charin ();
	//if (c == '\n')
	//	c= '\r';
	set_a (c);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::bios_CONOUTPUT ()
{
	// Entry: character in C.
	// Output: none.
	console.charout (get_c () );
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::bios_LISTOUT ()
{
	//cerr << crlf << "BIOS LISTOUT unimplemented." << crlf;
	//return TrapEndProgram;
	//console.charout (get_c () );
	printer.put (get_c () );
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::bios_PUNCH ()
{
	cerr << crlf << "BIOS PUNCH unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_READER ()
{
	cerr << crlf << "BIOS READER unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_HOME ()
{
	cerr << crlf << "BIOS HOME unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_SETDISD ()
{
	cerr << crlf << "BIOS SETDISD unimplemented." << crlf <<
		"Setting disk to " << get_a () + 'A' << crlf;
	// Provisional.
	set_hl (0);
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::bios_SETTRACK ()
{
	cerr << crlf << "BIOS SETTRACK unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_SETSECTOR ()
{
	cerr << crlf << "BIOS SETSECTOR unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_SETDMA ()
{
	//cerr << crlf << "BIOS SETDMA unimplemented." << crlf;
	//return TrapEndProgram;
	dma_addr= get_bc ();
	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::bios_READDISK ()
{
	cerr << crlf << "BIOS READDISK unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_WRITEDISK ()
{
	cerr << crlf << "BIOS WRITEDISK unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_LISTSTATUS ()
{
	// A=00h if list device is not ready to accept a character.
	// A=FFh if list device is ready to accept a character.

	set_a (0xFF);

	return TrapRet;
}

CPM::Internal::TrapReturn CPM::Internal::bios_SECTORTRAN ()
{
	cerr << crlf << "BIOS SECTORTRAN unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_USERF ()
{
	// Reserved for system implementor.

	cerr << crlf << "BIOS USERF unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bios_unsupported ()
{
	cerr << crlf << "BIOS CALL " <<
		//hex2 << (z.PC - bios_base) / 3 <<
		(get_pc ()  - bios_traps) <<
		" dec unimplemented." << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::UnsupportedRst ()
{
	cerr << crlf << "RST " << hex2 (get_pc () ) <<
		" unsupported." << crlf;
	word sp= get_sp ();
	word pos= memreadw (sp);
	cerr << "Return address: " << hex4 (pos) << 'h' << crlf;
	return TrapEndProgram;
}

CPM::Internal::TrapReturn CPM::Internal::bdos ()
{
	//bdosmap_t::iterator it= bdosmap.find (z.R1.br.C);
	byte func= get_c ();

	if (debugsystem)
		cerr << "Bdos func number " << hex2 (func) <<
			crlf;

	bdosmap_t::iterator it= bdosmap.find (func);
	if (it == bdosmap.end () )
	{
		cerr << crlf << "BDOS func " << hex2 (func) <<
			" unimplemented." << crlf;
		return TrapEndProgram;
	}
	return (this->* it->second) ();
}

void CPM::Internal::bdos_return_hl (byte h, byte l)
{
	//z.R1.br.H= h;
	//z.R1.br.B= h;
	//z.R1.br.L= l;
	//z.R1.br.A= l;
	set_h (h);
	set_b (h);
	set_l (l);
	set_a (l);
}

void CPM::Internal::bdos_return_hl (word reghl)
{
	set_hl (reghl);
	set_b (static_cast <byte> (reghl >> 8) );
	set_a (static_cast <byte> (reghl & 0xFF) );
}

// 00
CPM::Internal::TrapReturn CPM::Internal::bdos_system_reset ()
{
	if (debugsystem)
		cerr << "BDOS system reset called" << crlf;
	set_pc (0);
	return TrapContinue;
}

// 01
CPM::Internal::TrapReturn CPM::Internal::bdos_console_input ()
{
	byte c= console.charin ();
	if (c >= 0x20 || c == '\r' || c == '\n')
		console.charout (c);
	if (c == '\t')
		console.charout (' ');
	bdos_return_hl (0, c);
	return TrapRet;
}

// 02
CPM::Internal::TrapReturn CPM::Internal::bdos_console_output ()
{
	//cout << z.R1.br.E;
	//cout << get_e ();
	console.charout (get_e () );
	return TrapRet;
}

// 05
CPM::Internal::TrapReturn CPM::Internal::bdos_list_output ()
{
	// Provisional
	//console.charout (get_e () );
	printer.put (get_e() );
	return TrapRet;
}

// 06
CPM::Internal::TrapReturn CPM::Internal::bdos_direct_console_io ()
{
	//byte c= z.R1.br.E;
	byte c= get_e ();
	switch (c)
	{
	case 0xFF: // Input / status
		//bdos_return_hl (0, 0x0D);
		if (console.charavailable () )
			bdos_return_hl (0, console.charin () );
		else
			bdos_return_hl (0, 0);
		break;
	case 0xFE: // Status
		//bdos_return_hl (0, 0);
		bdos_return_hl (0, console.charavailable () ? 0xFF : 0x00);
		break;
	case 0xFD: // Input
		//bdos_return_hl (0, cin.get () );
		bdos_return_hl (0, console.charin () );
		break;
	default: // Output
		//cout << c;
		console.charout (c);
	}
	return TrapRet;
}

// 09
CPM::Internal::TrapReturn CPM::Internal::bdos_write_string ()
{
	//word pos= z.R1.wr.DE;
	word pos= get_de ();
	if (debugsystem)
		cerr << "(BDOS Write string at " << hex4 (pos) << ')' <<
			crlf;
	byte c;
	while ( (c= memread (pos) ) != '$')
	{
		//cout << c;
		console.charout (c);
		++pos;
	}
	//cout << std::flush;
	return TrapRet;
}

// 0A
CPM::Internal::TrapReturn CPM::Internal::bdos_read_console_buffer ()
{
	if (debugsystem)
		cerr << "BDOS read console buffer called" << crlf;
	word addr= get_de ();
	read_console_buffer (addr);
	return TrapRet;
}

// 0B
CPM::Internal::TrapReturn CPM::Internal::bdos_get_console_status ()
{
	bdos_return_hl (0, console.charavailable () );
	return TrapRet;
}

// 0C
CPM::Internal::TrapReturn CPM::Internal::bdos_return_version_number ()
{
	if (debugsystem)
		cerr << "BDOS return version number called" << crlf;
	bdos_return_hl (0, 0x31); // CP/M, vesrion 3.1
	return TrapRet;
}

// 0D
CPM::Internal::TrapReturn CPM::Internal::bdos_reset_disk_system ()
{
	if (debugsystem)
		cerr << "BDOS reset disk system called" << crlf;
	// Provisional: does nothing.
	return TrapRet;
}

// 0E
CPM::Internal::TrapReturn CPM::Internal::bdos_select_disk ()
{
	if (debugsystem)
		cerr << "BDOS select disk called" << crlf;
	// Actually ignored.
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 0F
CPM::Internal::TrapReturn CPM::Internal::bdos_open_file ()
{
	if (debugsystem)
		cerr << "BDOS open file called" << crlf;
	//word fcb= z.R1.wr.DE;
	word fcb= get_de ();
	if (debugsystem)
	{
		cerr << "Opening file, fcb= " << hex4 (fcb) <<
			"h : '";
		for (word i= 1; i < 9; ++i)
			cerr << memread (fcb + i);
		cerr << '.';
		for (word i= 9; i < 12; ++i)
			cerr << memread (fcb + i);
		cerr << '\'' << crlf;
	}

	if (open_file (fcb) )
	{
		if (debugsystem)
			cerr << "Open succeded" << crlf;
		bdos_return_hl (0, 0);
	}
	else
	{
		if (debugsystem)
			cerr << "Open failed" << crlf;
		bdos_return_hl (0, 0xFF);
	}
	return TrapRet;
}

// 10
CPM::Internal::TrapReturn CPM::Internal::bdos_close_file ()
{
	if (debugsystem)
		cerr << "BDOS close file called" << crlf;
	//word fcb= z.R1.wr.DE;
	word fcb= get_de ();
	std::string filename= filename_from_fcb (fcb);

	if (debugsystem)
		cerr << "Close file: " << filename << crlf;

	OpenFile * pf= getopenfile (fcb);
	if (pf == NULL)
	{
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}

	pf->close ();
	clean_fcb_entry (fcb);

	bdos_return_hl (0, 0);
	return TrapRet;
}

void CPM::Internal::setsearchresult (const std::string & result)
{
	memwrite (dma_addr, 0);
	word pos= dma_addr + 1;
	std::string::size_type n= 0;
	const std::string::size_type l= result.size ();
	for (int i= 0; i < 8; ++i, ++pos)
	{
		if (n < l && result [n] != '.')
		{
			memwrite (pos, result [n] );
			++n;
		}
		else
			memwrite (pos, ' ');
	}
	++n;
	for (int i= 0; i < 3; ++i, ++pos)
	{
		if (n < l)
		{
			memwrite (pos, result [n] );
			++n;
		}
		else
			memwrite (pos, ' ');
	}
}

// 11
CPM::Internal::TrapReturn CPM::Internal::bdos_search_for_first ()
{
	if (debugsystem)
		cerr << "BDOS search for first called" << crlf;
	word fcb= get_de ();
	clean_fcb_entry (fcb);
	std::string filemask= filename_from_fcb (fcb);

	if (debugsystem)
		cerr << "Search for first: " << filemask << crlf;

	FindFile * pff= new FindFile;
	const std::string result= pff->findfirst (filemask);
	if (! result.empty () )
	{
		findfilemap [fcb]= pff;
		setsearchresult (result);
		bdos_return_hl (0, 0);
	}
	else
		bdos_return_hl (0, 0xFF);
	return TrapRet;
}

// 12
CPM::Internal::TrapReturn CPM::Internal::bdos_search_for_next ()
{
	if (debugsystem)
		cerr << "BDOS search for next called" << crlf;
	word fcb= get_de ();
	findfilemap_t::iterator it= findfilemap.find (fcb);
	if (it == findfilemap.end () )
	{
		if (debugsystem)
			cerr << "Called bdos search for next "
				"with invalid FCB" << crlf;
		bdos_return_hl (0, 0xFF);
		return TrapRet;
	}
	FindFile * pff= it->second;
	const std::string result= pff->findnext ();
	if (! result.empty () )
	{
		setsearchresult (result);
		bdos_return_hl (0, 0);
	}
	else
		bdos_return_hl (0, 0xFF);
	return TrapRet;
}

// 13
CPM::Internal::TrapReturn CPM::Internal::bdos_delete_file ()
{
	if (debugsystem)
		cerr << "BDOS delete file called" << crlf;
	//word fcb= z.R1.wr.DE;
	word fcb= get_de ();
	std::string filename= filename_from_fcb (fcb);

	if (debugsystem)
		cerr << "Delete file: " << filename << crlf;

	unlink (filename.c_str () );
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 14
CPM::Internal::TrapReturn CPM::Internal::bdos_read_sequential ()
{
	if (debugsystem)
		cerr << "BDOS read sequential called" << crlf;

	word fcb= get_de ();

	if (debugsystem)
		cerr << "Fcb= " << hex4 (fcb) <<
			", file is: " << filename_from_fcb (fcb) << crlf;

	OpenFile * pf= getopenfile (fcb);
	if (pf == NULL)
	{
		if (debugsystem)
			cerr << "Invalid FCB" << crlf;
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}
	int readsize= sector_size * multi_sector_count;

	word current_record= memread (fcb + fcb_current_record);
	byte extension= memread (fcb + fcb_extension);
	long pos= extension;
	pos*= records_in_extension;
	pos+= current_record;
	pos*= sector_size;

	if (debugsystem)
		cerr << "Reading " << int (multi_sector_count) <<
			" sectors from " << filename_from_fcb (fcb) <<
			" position " << pos <<
			" to " << hex4 (dma_addr) <<
			" current record is " << current_record <<
			crlf;

	if (pf->lseek (pos, SEEK_SET) == -1)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}

	int r= pf->read (mem + dma_addr, readsize);

	if (debugsystem)
		cerr << "Bytes readed: " << r << crlf;

	if (r == -1 || r == 0)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}

	// If there is an incomplete sector, fill with CP/M EOF char.
	for (int i= r; i < readsize; ++i)
		memwrite (dma_addr + i, 0x1A);

	int sec= r / sector_size;
	if (r % sector_size)
		sec++;

	current_record+= sec;
	if (current_record >= records_in_extension)
	{
		current_record-= records_in_extension;
		++extension;
		memwrite (fcb + fcb_extension, extension);
	}
	memwrite (fcb + fcb_current_record, static_cast <byte> (current_record) );

	memwrite (fcb + fcb_current_record, current_record);

	if (sec < multi_sector_count)
		bdos_return_hl (sec, 1); // EOF after sec sectors.
	else
		bdos_return_hl (0, 0);

	return TrapRet;
}

// 15
CPM::Internal::TrapReturn CPM::Internal::bdos_write_sequential ()
{
	if (debugsystem)
		cerr << "BDOS write sequential called" << crlf;
	word fcb= get_de ();
	OpenFile * pf= getopenfile (fcb);
	if (pf == NULL)
	{
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}
	int writesize= sector_size * multi_sector_count;

	word current_record= memread (fcb + fcb_current_record);
	byte extension= memread (fcb + fcb_extension);
	long pos= extension;
	pos*= records_in_extension;
	pos+= current_record;
	pos*= sector_size;

	if (debugsystem)
		cerr << "Writing " << int (multi_sector_count) <<
			" sectors to " << filename_from_fcb (fcb) <<
			" position " << pos <<
			" from " << hex4 (dma_addr) <<
			" current record is " << current_record <<
			crlf;

	if (pf->lseek (pos, SEEK_SET) == -1)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}

	int r= pf->write (mem + dma_addr, writesize);

	int sec= r / sector_size;
	if (r % sector_size)
		sec++;

	current_record+= sec;
	if (current_record >= records_in_extension)
	{
		current_record-= records_in_extension;
		++extension;
		memwrite (fcb + fcb_extension, extension);
	}
	memwrite (fcb + fcb_current_record, static_cast <byte> (current_record) );

	memwrite (fcb + fcb_current_record, current_record);

	if (sec < multi_sector_count)
		bdos_return_hl (sec, 1); // EOF after sec sectors.
	else
		bdos_return_hl (0, 0);
	return TrapRet;
}

// 16
CPM::Internal::TrapReturn CPM::Internal::bdos_make_file ()
{
	if (debugsystem)
		cerr << "BDOS make file called" << crlf;
	//word fcb= z.R1.wr.DE;
	word fcb= get_de ();
	if (make_file (fcb) )
		bdos_return_hl (0, 0);
	else
		bdos_return_hl (8, 0xFF); // File exists.
	return TrapRet;
}

// 17
CPM::Internal::TrapReturn CPM::Internal::bdos_rename_file ()
{
	if (debugsystem)
		cerr << "BDOS rename file called" << crlf;
	//word fcb= z.R1.wr.DE;
	word fcb= get_de ();
	std::string orig= filename_from_fcb (fcb);
	word pseudofcb= fcb + 16;
	std::string dest= filename_from_fcb (pseudofcb);

	if (debugsystem)
		cerr << "Renaming " << orig << " to " << dest << crlf;

	rename (orig.c_str (), dest.c_str () );
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 19
CPM::Internal::TrapReturn CPM::Internal::bdos_return_current_disk ()
{
	if (debugsystem)
		cerr << "BDOS return current disk called" << crlf;

	// Return A:
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 1A
CPM::Internal::TrapReturn CPM::Internal::bdos_set_dma_address ()
{
	if (debugsystem)
		cerr << "BDOS set dma address called" << crlf;
	//dma_addr= z.R1.wr.DE;
	dma_addr= get_de ();

	if (debugsystem)
		cerr << "DMA address set to " <<
			hex4 (dma_addr) << crlf;

	bdos_return_hl (0, 0);
	return TrapRet;
}

// 1D
CPM::Internal::TrapReturn CPM::Internal::bdos_map_read_only ()
{
	if (debugsystem)
		cerr << "BDOS map read only called" << crlf;
	// Provisional: no drive is read only.
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 1E
CPM::Internal::TrapReturn CPM::Internal::bdos_set_file_attributes ()
{
	if (debugsystem)
		cerr << "BDOS set file attributes called" << crlf;

	// Nada de momento
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 20
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_user_code ()
{
	//if (z.R1.br.E == 0xFF)
	if (get_e () == 0xFF)
	{
		// Get user.
		if (debugsystem)
			cerr << "BDOS Get User Code" << crlf;
		bdos_return_hl (0, 0); // Provisional.
	}
	else
	{
		// Set user.
		// Provisional, does nothing.
		if (debugsystem)
			cerr << "BDOS Set User Code to " <<
				int (get_e () ) << crlf;
	}
	return TrapRet;
}

// 21
CPM::Internal::TrapReturn CPM::Internal::bdos_read_random ()
{
	word fcb= get_de ();

	if (debugsystem)
		cerr << "BDOS read random at " <<
			fcb_get_random_record (fcb) <<
			crlf;

	OpenFile * pf= getopenfile (fcb);
	if (pf == NULL)
	{
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}

	// Provisional
	//pf->lseek (fcb_get_random_record (fcb) * sector_size, SEEK_SET);

	unsigned long rec= fcb_get_random_record (fcb);
	unsigned long ext= rec / records_in_extension;
	if (ext > 255)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}
	memwrite (fcb + fcb_extension, static_cast <byte> (ext) );
	rec%= records_in_extension;
	memwrite (fcb + fcb_current_record, static_cast <byte> (rec) );

	TrapReturn r= bdos_read_sequential ();

	if (debugsystem)
		cerr << crlf << "Read random done." << crlf;
	return r;
}

// 22
CPM::Internal::TrapReturn CPM::Internal::bdos_write_random ()
{
	word fcb= get_de ();

	if (debugsystem)
		cerr << "BDOS write random at " <<
			fcb_get_random_record (fcb) <<
			crlf;

	OpenFile * pf= getopenfile (fcb);
	if (pf == NULL)
	{
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}

	// Provisional
	pf->lseek (fcb_get_random_record (fcb) * sector_size, SEEK_SET);

	unsigned long rec= fcb_get_random_record (fcb);
	unsigned long ext= rec / records_in_extension;
	if (ext > 255)
	{
		bdos_return_hl (0, 1); // EOF
		return TrapRet;
	}
	memwrite (fcb + fcb_extension, static_cast <byte> (ext) );
	rec%= records_in_extension;
	memwrite (fcb + fcb_current_record, static_cast <byte> (rec) );

	TrapReturn r= bdos_write_sequential ();

	if (debugsystem)
		cerr << crlf << "Write random done." << crlf;
	return r;
}

// 23
CPM::Internal::TrapReturn CPM::Internal::bdos_compute_file_size ()
{
	if (debugsystem)
		cerr << "BDOS compute file size called" << crlf;
	//word fcb= z.R1.wr.DE;
	word fcb= get_de ();
	std::string filename= filename_from_fcb (fcb);
	struct stat buf;
	if (stat (filename.c_str (), & buf) != 0)
	{
		bdos_return_hl (0, 0xFF);
		return TrapRet;
	}
	const unsigned long filesize= buf.st_size;
	unsigned long l= filesize / sector_size;
	if (filesize % sector_size)
		++l;
	memwrite (fcb + fcb_random_record + 0, l & 0xFF);
	memwrite (fcb + fcb_random_record + 2, (l >> 8) & 0xFF);
	memwrite (fcb + fcb_random_record + 3, (l >> 16) & 0xFF);
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 24
CPM::Internal::TrapReturn CPM::Internal::bdos_set_random_record ()
{
	if (debugsystem)
		cerr << "BDOS set random record called" << crlf;
	//word fcb= z.R1.wr.DE;
	word fcb= get_de ();

	std::string filename= filename_from_fcb (fcb);

	if (debugsystem)
		cerr << "set random record to: " << filename << crlf;

	OpenFile * pf= getopenfile (fcb);
	if (pf == NULL)
	{
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}
	long pos= pf->lseek (0, SEEK_CUR);
	if (pos == -1)
	{
		bdos_return_hl (0, 9); // Invalid FCB
		return TrapRet;
	}
	long l= pos / 128;
	if (pos % 128)
		++l;
	memwrite (fcb + fcb_random_record, l & 0xFF);
	memwrite (fcb + fcb_random_record + 2, (l >> 8) & 0xFF);
	memwrite (fcb + fcb_random_record + 3, (l >> 16) & 0xFF);
	bdos_return_hl (0, 0);
	return TrapRet;
}

// 2C
CPM::Internal::TrapReturn CPM::Internal::bdos_set_multi_sector_count ()
{
	if (debugsystem)
		cerr << "BDOS set multi sector count called" << crlf;
	//byte count= z.R1.br.E;
	byte count= get_e ();
	if (count > 128)
		bdos_return_hl (0, 0xFF);
	else
	{
		if (debugsystem)
			cerr << "Setting multi-sector count to " <<
				int (count) << crlf;

		multi_sector_count= count;
		bdos_return_hl (0, 0);
	}
	return TrapRet;
}

// 2D
CPM::Internal::TrapReturn CPM::Internal::bdos_set_bdos_error_mode ()
{
	if (debugsystem)
		cerr << "BDOS set bdos error mode called" << crlf;
	// E= error mode.
	// Provisional: Ignored.
	return TrapRet;
}

// 31
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_system_control_block ()
{
	if (debugsystem)
		cerr << "BDOS get/set system control block called" << crlf;

	word scbpb= get_de ();
	byte offset= memread (scbpb + 0);
	byte set= memread (scbpb + 1);

	switch (offset)
	{
	case 0x1A: // Console width (0 based)
		switch (set)
		{
		case 0: // Get
			//set_a (54);
			set_a (static_cast <byte> (console.columns () - 1) );
			break;
		case 0xFF: // Set byte
			// Ignored
			break;
		default:
			if (debugsystem)
				cerr << "Page width operation invalid." <<
					crlf;
		}
		break;
	case 0x1C: // Console page length
		switch (set)
		{
		case 0: // Get
			//set_a (54);
			set_a (static_cast <byte> (console.lines () ) );
			break;
		case 0xFF: // Set byte
			// Ignored
			break;
		default:
			if (debugsystem)
				cerr << "Page length operation invalid." <<
					crlf;
		}
		break;
	case 0x2C: // Page mode
		switch (set)
		{
		case 0: // Get
			//z.R1.br.A= 0;
			set_a (0);
			break;
		case 0xFF: // Set byte
			// Ignored
			break;
		default:
			if (debugsystem)
				cerr << "Page mode operation invalid." <<
					crlf;
		}
		break;
	case 0x50: // Temporary file drive
		switch (set)
		{
		case 0:
			set_a (0);
			break;
		case 0xFF:
			cerr << "Setting temporary drive" << crlf;
			// Ignored.
			break;
		default:
			cerr << "Set temporary invalid operation" << crlf;
		}
		break;
	default:
		switch (set)
		{
		case 0:
			cerr << "Reading";
			break;
		case 0xFF:
			cerr << "Writing byte " <<
				hex2 (memread (scbpb + 2) ) <<
				"h in";
			break;
		case 0xFE:
			cerr << "Writing word " <<
				hex4 (memreadw (scbpb + 2) ) <<
				"h in";
			break;
		default:
			cerr << "Unsupported operation";
		}
		cerr <<	" unsupported SCB field " <<
			hex2 (offset) << crlf;
	}
	return TrapRet;
}

//3C
CPM::Internal::TrapReturn
	CPM::Internal::bdos_call_resident_system_extension ()
{
	if (debugsystem)
	{
		cerr << "BDOS call resident system extension" << crlf;
		word rsxpb= get_de ();
		cerr << "Function: " << hex2 (memread (rsxpb) ) <<
			crlf;
	}
	bdos_return_hl (0xFF, 0xFF);
	return TrapRet;
}

// 66

CPM::Internal::TrapReturn
	CPM::Internal::bdos_get_read_file_date_and_pass_mode ()
{
	word fcb= get_de ();
	if (debugsystem)
	{
		cerr << "BDOS get read file date stamps and password mode" <<
			crlf << "File: " << filename_from_fcb (fcb) << endl;
	}

	// Provisional
	bdos_return_hl (0, 0xFF); // File not found.
	return TrapRet;
}

// 6D
CPM::Internal::TrapReturn CPM::Internal::bdos_get_set_console_mode ()
{
	if (debugsystem)
		cerr << "BDOS get/set console mode called" << crlf;
	//if (z.R1.wr.DE == 0xFFFF)
	if (get_de () == 0xFFFF)
	{
		//z.R1.wr.HL= 0; // Provisional
		set_hl (0);
	}
	else
	{
		if (debugsystem)
			cerr << "Console mode set to " <<
				//hex4 << z.R1.wr.DE << crlf;
				hex4 (get_de () ) << crlf;

		//z.R1.wr.HL= 0; // Provisional
		set_hl (0); // Provisional
	}
	return TrapRet;
}

// 98
CPM::Internal::TrapReturn CPM::Internal::bdos_parse_filename ()
{
	if (debugsystem)
		cerr << "BDOS parse filename called" << crlf;
	word pfcb= get_de ();

	if (debugsystem)
		cerr << "Parsing filename in pfcb " <<
			hex4 (pfcb) << 'h' << crlf;

	word name= memreadw (pfcb);
	word fcb= memreadw (pfcb + 2);
	parsefilefcb (name, fcb);
	bdos_return_hl (name);
	return TrapRet;
}

void CPM::Internal::parsefilefcb (word & name, word fcb)
{
	memwrite (fcb, 0); // Default disk
	// Clean file name and type to spaces.
	for (int i= 1; i < 12; ++i)
		memwrite (fcb + i, ' ');
	// Clean next 3 bytes. 
	for (int i= 12; i < 16; ++i)
		memwrite (fcb + i, 0);
	// Clean password to spaces
	//for (int i= 16; i < 24; ++i)
	//	memwrite (fcb + i, ' ');
	// Clean reserved area
	//for (int i= 24; i < 32; ++i)
	//	memwrite (fcb + i, 0);
	// Clean random record
	//memwrite (fcb + fcb_random_record + 0, 0);
	//memwrite (fcb + fcb_random_record + 1, 0);
	//memwrite (fcb + fcb_random_record + 2, 0);

	char c;
	while ( (c= memread (name) ) == ' ' || c == '\t')
		++name;
	if (c == '\r' || c == '\0')
	{
		name= 0;
		return;
	}

	int i= 0;
	while ( (c= memread (name) ) != '.' && c != ' ' && c != '\t' && c != 0)
	{
		if (i < 8)
			memwrite (fcb + i + 1, toupper (c) );
		++i;
		++name;
	}
	if (c == '.')
	{
		++name;
		i= 0;
		while ( (c= memread (name) ) != '.' && c != ' ' &&
			c != '\t' && c != 0)
		{
			if (i < 3)
				memwrite (fcb + i + 9, toupper (c) );
			++i;
			++name;
		}
	}
}

void CPM::Internal::cleardefaultfcb (word fcb, word pass)
{
	memwrite (fcb, 0); // Disk
	for (word i= 1; i < 12; ++i)
		memwrite (fcb + i, ' ');
	memwrite (fcb + fcb_extension, 0);

	memwritew (pass, 0); // Address
	memwrite (pass + 2, 0); // Length
}

namespace {

const std::string delimiters (" \t\r\n<=,!|>[].;:");

inline bool isdelimiter (byte c)
{
	return delimiters.find (c) != std::string::npos;
}

bool validchar (byte c)
{
	return ! isdelimiter (c);
}

bool validpasschar (byte c)
{
	// Provisional
	return ! isdelimiter (c);
}

} // namespace

void CPM::Internal::parsefilecmdline (word & pos, word endline,
	word fcb, word pass)
{
	if (pos >= endline)
		return;
	if (pos < endline - 2 && isalpha (memread (pos) ) &&
		memread (pos + 1) == ':')
	{
		byte c= toupper (memread (pos) ) - 'A' + 1;
		memwrite (fcb, c);
		pos+= 2;
	}
	size_t i= 0;
	byte c;
	while (pos < endline && validchar ( (c= memread (pos) ) ) )
	{
		if (i < 8)
			memwrite (fcb + i + 1, c);
		++i;
		++pos;
	}
	if (pos >= endline)
		return;
	if (c == '.')
	{
		++pos;
		i= 0;
		while (pos < endline && validchar ( (c= memread (pos) ) ) )
		{
			if (i < 3)
				memwrite (fcb + 9 + i, c);
			++i;
			++pos;
		}
	}
	if (pos >= endline)
		return;
	if (c == ';')
	{
		++pos;
		byte i= 0;
		word pospass= pos;
		while (pos < endline && validpasschar (memread (pos) ) )
		{
			++i;
			++pos;
		}
		if (i != 0)
		{
			memwritew (pass, pospass);
			memwrite (pass + 2, i);
		}
	}
}

void CPM::Internal::setcmd_args (const std::string & args)
{
	std::string::size_type l= args.size ();
	if (l > 127)
		l= 127;
	memwrite (cmd_args, static_cast <byte> (l) );
	for (word i= 0; i < l; ++i)
		memwrite (cmd_args + 1 + i, args [i] );
	for (word i= l; i < 127; ++i)
		memwrite (cmd_args + 1 + i, 0);
}

void CPM::Internal::setarguments (const std::string & args)
{
	setcmd_args (args);

	word pos= cmd_args + 1;
	const word endline= pos + memread (cmd_args);

	cleardefaultfcb (default_fcb_1, default_pass_1);
	cleardefaultfcb (default_fcb_2, default_pass_2);

	byte c;
	while (pos < endline && (c= memread (pos) ) == ' ' || c == '\t')
		++pos;

	if (pos < endline)
	{
		parsefilecmdline (pos, endline,
			default_fcb_1, default_pass_1);

		++pos; // Skip separator
		while (pos < endline &&
			(c= memread (pos) ) == ' ' || c == '\t')
		{
			++pos;
		}
		parsefilecmdline (pos, endline,
			default_fcb_2, default_pass_2);
	}
}

void CPM::Internal::showcurrentstate ()
{
	bool saveflag= debugz80;
	debugz80= false;

	//word sp= z.R1.wr.SP;
	word sp= get_sp ();
	//cerr << hex4 << z.PC;
	word pc= get_pc ();
	cerr << hex4 (pc);

	#if 1

	#if ! defined USE_ImcZ80 || defined  DISASM_WITH_libz80

	disassembly (cerr);

	#else

	//cerr << ": " << hex2 << int (memread (pc) ) << dec;
	cerr << ": ";
	if (memread (pc)  == trapbyte && ( gettrapfunc (pc) ) != 0)
		cerr << "System trap";
	else
		disassembly (cerr);

	#endif

	#endif

	// Flags.
	//byte f= z.R1.br.F;
	byte f= get_f ();
	cerr << " F=" << hex2 (f) <<

		//' ' << ( (f & F_C) ? ' ' : 'N' ) << "C" <<
		//' ' << ( (f & F_Z) ? ' ' : 'N' ) << "Z" <<
		//' ' << ( (f & F_S) ? ' ' : 'N' ) << "S";

		//' ' << ( flag_c (f) ? ' ' : 'N' ) << "C" <<
		//' ' << ( flag_z (f) ? ' ' : 'N' ) << "Z" <<
		//' ' << ( flag_s (f) ? ' ' : 'N' ) << "S";

		' ' << ( flag_c () ? ' ' : 'N' ) << "C" <<
		' ' << ( flag_z () ? ' ' : 'N' ) << "Z" <<
		' ' << ( flag_s () ? ' ' : 'N' ) << "S";

	// Registers.
	//cerr << " A=" << hex2 (z.R1.br.A) <<
	//	" BC=" << hex4 (z.R1.wr.BC) <<
	//	" DE=" << hex4 (z.R1.wr.DE) <<
	//	" HL=" << hex4 (z.R1.wr.HL) <<
	//	crlf;
	cerr << " A=" << hex2 (get_a () ) <<
		" BC=" << hex4 (get_bc () ) <<
		" DE=" << hex4 (get_de () ) <<
		" HL=" << hex4 (get_hl () ) <<
		" IX=" << hex4 (get_ix () ) <<
		" IY=" << hex4 (get_iy () ) <<
		crlf;

	// Stack inspection.
	#if 1
	cerr <<
		" SP=" << hex4 (sp) << "->" <<
		//hex2 << int (memread (sp + 0) ) << ' ' <<
		hex2 (memread (sp + 1) ) << ' ' <<
		hex2 (memread (sp + 2) ) << ' ' <<
		hex2 (memread (sp + 3) ) << ' ' <<
		hex2 (memread (sp + 4) ) << ' ' <<
		hex2 (memread (sp + 5) ) << ' ' <<
		hex2 (memread (sp + 6) ) << ' ' <<
		hex2 (memread (sp + 7) ) << ' ' <<
		hex2 (memread (sp + 6) );
	#endif

	cerr << crlf;

	debugz80= saveflag;
}

inline CPM::Internal::TrapReturn CPM::Internal::runinstruction ()
{
	word pc= get_pc ();
	byte b= memread (pc);
	TrapFunc trap;
	if (b == trapbyte && (trap= gettrapfunc (pc) ) != 0)
	{
		TrapReturn r= (this->* trap) ();
		if (r == TrapRet)
		{
			//word sp= z.R1.wr.SP;
			word sp= get_sp ();
			word pc= memread (sp + 0) |
				(word (memread (sp + 1) )
					<< 8);
			//z.PC= pc;
			set_pc (pc);
			//z.R1.wr.SP= sp + 2;
			set_sp (sp + 2);
		}
		return r;
	}
	else
	{
		// Avoid looping in HALT instruction.
		if (b == 0x76)
		{
			//set_pc (pc + 1);
			if (debugsystem)
				cerr << "HALT instruction executed" << crlf;
			return TrapEndProgram;
		}
		else
			execute ();
		return TrapContinue;
	}
}

void CPM::Internal::runner ()
{
	//TrapFunc trap;

	for (;;)
	{
		#if 1
		// Show debug info.

		if (debugz80)
		{
			#if 0

			debugz80= false;
			//word sp= z.R1.wr.SP;
			word sp= get_sp ();
			//cerr << hex4 << z.PC;
			word pc= get_pc ();
			cerr << hex4 (pc);

			#if 1

			#if ! defined USE_ImcZ80 || defined  DISASM_WITH_libz80

			// Current instruction.
			#ifdef USE_ImcZ80
			aux_set_pc (pc);
			#endif

			disassembly (cerr);

			#else

			//cerr << ": " << hex2 << int (memread (pc) ) << dec;
			cerr << ": ";
			if (memread (pc)  == trapbyte &&
					( gettrapfunc (pc) ) != 0)
				cerr << "System trap";
			else
				disassembly (cerr);

			#endif

			#endif

			// Flags.
			//byte f= z.R1.br.F;
			byte f= get_f ();
			cerr << " F=" << hex2 (f) <<
				//' ' << ( (f & F_C) ? ' ' : 'N' ) << "C" <<
				//' ' << ( (f & F_Z) ? ' ' : 'N' ) << "Z" <<
				//' ' << ( (f & F_S) ? ' ' : 'N' ) << "S";
				' ' << ( flag_c (f) ? ' ' : 'N' ) << "C" <<
				' ' << ( flag_z (f) ? ' ' : 'N' ) << "Z" <<
				' ' << ( flag_s (f) ? ' ' : 'N' ) << "S";

			// Registers.
			//cerr << " A=" << hex2 (z.R1.br.A) <<
			//	" BC=" << hex4 (z.R1.wr.BC) <<
			//	" DE=" << hex4 (z.R1.wr.DE) <<
			//	" HL=" << hex4 (z.R1.wr.HL) <<
			//	crlf;
			cerr << " A=" << hex2 (get_a () ) <<
				" BC=" << hex4 (get_bc () ) <<
				" DE=" << hex4 (get_de () ) <<
				" HL=" << hex4 (get_hl () ) <<
				crlf;

			// Stack inspection.
			#if 1
			cerr <<
				" SP=" << hex4 (sp) << "->" <<
				hex2 (memread (sp + 1) ) << ' ' <<
				hex2 (memread (sp + 2) ) << ' ' <<
				hex2 (memread (sp + 3) ) << ' ' <<
				hex2 (memread (sp + 4) ) << ' ' <<
				hex2 (memread (sp + 5) ) << ' ' <<
				hex2 (memread (sp + 6) ) << ' ' <<
				hex2 (memread (sp + 7) ) << ' ' <<
				hex2 (memread (sp + 6) );
			#endif

			cerr << crlf;
			debugz80= true;

			#else

			showcurrentstate ();

			#endif
		}

		#endif

		#if 0

		//byte b= memread (z.PC);
		byte b= memread (get_pc () );
		//if (b == trapbyte && (trap= gettrapfunc (z.PC) ) != 0)
		if (b == trapbyte && (trap= gettrapfunc (get_pc () ) ) != 0)
		{
			switch ( (this->* trap) () )
			{
			case TrapEndProgram:
				return;
			case TrapRet:
				{
					//word sp= z.R1.wr.SP;
					word sp= get_sp ();
					word pc= memread (sp + 0) |
						(word (memread (sp + 1) )
							<< 8);
					//z.PC= pc;
					set_pc (pc);
					//z.R1.wr.SP= sp + 2;
					set_sp (sp + 2);
				}
				break;
			case TrapContinue:
				; // Nothing to do
			}
		}
		else
		{
			// Avoid stoping in HALT instruction.
			if (b == 0x76)
				set_pc (get_pc () + 1);
			else
				execute ();

		}
		//cin.get ();

		#else

		if (runinstruction () == TrapEndProgram)
			return;

		#endif
	}

}

void CPM::Internal::run (bool ndebugsystem, bool ndebugz80)
{
	//multi_sector_count= 1;
	//dma_addr= default_dma_addr;
	debugsystem= ndebugsystem;
	debugz80= ndebugz80;

	runner ();

	//close_all ();

	debugsystem= false;
	debugz80= false;
}

void CPM::Internal::prepareruntransient ()
{
	reset ();

	multi_sector_count= 1;
	dma_addr= default_dma_addr;
	set_pc (tpa);
	// Init stack, putting 0 as return address to return to system.
	set_sp (default_stack);
	memwrite (default_stack, 0);
	memwrite (default_stack + 1, 0);
}

void CPM::Internal::endruntransient ()
{
	close_all ();
	printer.flush ();
}

void CPM::Internal::runtransient (bool ndebugsystem, bool ndebugz80)
{
	#if 0

	reset ();

	set_pc (tpa);
	// Init stack, putting 0 as return address to return to system.
	set_sp (default_stack);
	memwrite (default_stack, 0);
	memwrite (default_stack + 1, 0);

	#else

	prepareruntransient ();

	#endif

	run (ndebugsystem, ndebugz80);

	endruntransient ();
}

void CPM::Internal::attachloader ()
{
	// Attach a pseudo loader that does nothing.
	// Some RSX may need to have a loader, and
	// it helps in deattachment.

	word old_jump= memreadw (bdos_call + 1);
	if (old_jump != bdos_jump)
	{
		// Some RSX already loaded.
		return;
	}
	word old_base= old_jump & 0xFF00;
	word new_base= old_base - 0x0100;

	// Build the RSX prefix of the loader.
	memwrite (new_base + rsx_jump_start, jp_code);
	memwritew (new_base + rsx_start, old_jump);
	memwrite (new_base + rsx_jump_next, jp_code);
	memwritew (new_base + rsx_next, old_jump);

	// Previous field indicating first RSX.
	memwritew (new_base + rsx_prev, bdos_call + 2);

	// Set remove flag to true.
	memwritew (new_base + rsx_remove, 0xFF);

	memcpy (mem + new_base + rsx_name, "LOADER  ", 8);
	memwrite (new_base + rsx_loader, 0xFF);

	// Point BDOS jump to loader.
	memwritew (bdos_call + 1, new_base + rsx_jump_start);
}

void CPM::Internal::attachrsx (word image, word len)
{
	// Image points to the PRL header of the RSX.
	cerr << "RSX header:" << crlf;
	cerr << "Start: " << hex4 (memreadw (image + rsx_start) ) <<
		crlf;
	cerr << "Next: " << hex4 (memreadw (image + rsx_next) ) <<
		crlf;
	cerr << "Remove flag: " << int (memread (image + rsx_remove) ) <<
		crlf;
	cerr << "Name: " << std::setw (8) <<
		reinterpret_cast <char *> (mem + image + rsx_name) << crlf;

	// Calculate installation address.
	word old_jump= memreadw (bdos_call + 1);
	cerr << "Old jump address: " << hex4 (old_jump) << crlf;

	word old_base= old_jump & 0xFF00;
	word new_base= (old_base - len) & 0xFF00;
	byte base_seg= new_base >> 8;

	// Move to installation location.
	for (word i= 0; i < len; ++i)
		memwrite (new_base + i, memread (image + i) );

	// Relocation.
	word reloc= image + len;
	//cerr << "Relocation map " << hex4 (reloc) << crlf;
	word pos= reloc;
	byte mask= 0;
	byte b;
	for (word i= 0; i < len; ++i)
	{
		if (mask == 0)
		{
			b= memread (pos++);
			//cerr << "Byte: " << hex2 (b) << crlf;
			mask= 0x80;
		}
		if (b & mask)
		{
			//cout << hex4 (i) <<
			//	" needs relocation" << crlf;
			memwrite (new_base + i,
				memread (new_base + i) + base_seg - 1);
		}
		mask>>= 1;
	}

	// Set previous field of old RSX (it must be one,
	// the loader at least).
	memwritew (old_base + rsx_prev, new_base + rsx_jump_next + 2);

	// Set jump to old BDOS jump.
	memwritew (new_base + rsx_next, old_jump);

	// Set previous field indicating this is the first RSX.
	memwritew (new_base + rsx_prev, bdos_call + 2);

	memwrite (new_base + rsx_loader, 0); // This is not the loader.

	// Point BDOS jump to the actual RSX.
	memwritew (bdos_call + 1, new_base + rsx_jump_start);
}

bool CPM::Internal::loadfiletpa (int handle)
{
	cerr << "Loading file in " <<
		static_cast <void *> (mem) << ", " <<
		static_cast <void *> (mem + tpa) << crlf;

	int r= read (handle, mem + tpa, bdos_base - tpa);
	if (r <= 0)
		return false;
	char c;

	if (read (handle, & c, 1) > 0) // File is too long.
		return false;

	// Check if it have header.
	if (memread (tpa) != ret_code)
		return true;

	// Pre-initialisation code.
	set_sp (default_stack);
	memwritew (default_stack, end_run_trap);
	cerr << "Running pre-intialisation code" << crlf;
	set_pc (tpa + 3);
	run (false, false);
	cerr << "End of pre-intialisation code" << crlf;

	// Relocation and activation of RSX.
	byte num_rsx= memread (tpa + 0x0F);
	if (num_rsx > 0)
		attachloader ();
	for (byte i= 0; i < num_rsx; ++i)
	{
		word recrsx= tpa + 0x10 + i * 0x10;
		cerr << "RSX record " << int (i) << crlf;
		cerr << "Offset: " << memreadw (recrsx + 0) << crlf;
		cerr << "Length: " << memreadw (recrsx + 2) << crlf;
		cerr << "NB flag: " << int (memread (recrsx + 4) ) << crlf;
		cerr << "Name: " << std::setw (8) <<
			reinterpret_cast <char *> (mem + recrsx + 6) << crlf;
		// Attach it if is not only for non-banked systems.
		if (memread (recrsx + 4) != 0xFF)
			attachrsx (memreadw (recrsx + 0) + tpa,
				memreadw (recrsx + 2) );
	}

	// Move the main program to the start of tpa.
	word len= memreadw (tpa + 1);
	for (word i= 0; i < len; ++i)
		memwrite (tpa + i, memread (tpa + 0x100 + i) );

	return true;
}

bool CPM::Internal::loadprltpa (int handle)
{
	int r= read (handle, mem + tpa, bdos_base - tpa);
	if (r <= 0)
		return false;
	char c;

	if (read (handle, & c, 1) > 0) // File is too long.
		return false;

	if (memread (tpa) != 0)
	{
		cerr << "Invalid PRL header" << crlf;
		return false;
	}
	word len= memreadw (tpa + 1);
	// Check if header + code + relocation bitmap fits in the file.
	if (len + (len + 7) / 8 + 0x100 > r)
	{
		cerr << "Invalid PRL header" << crlf;
		return false;
	}


	// Relocation not needed to run the prl in the tpa.

	// Move the code to start of tpa.
	for (word i= 0; i < len; ++i)
		memwrite (tpa + i, memread (tpa + 0x100 + i) );

	return true;
}

void CPM::Internal::tpastart (bool ndebugsystem, bool ndebugz80)
{
	console.activate ();
	runtransient (ndebugsystem, ndebugz80);
	console.deactivate ();
}

bool CPM::Internal::loadcommand (const std::string & filename)
{
	cerr << "Loading command " << filename << crlf;
	int handle= open (filename.c_str (), O_RDONLY);
	if (handle == -1)
	{
		cerr << "File not loaded" << crlf;
		return false;
	}
	bool r= loadfiletpa (handle);
	close (handle);
	return r;
}

std::string CPM::Internal::getcommandline ()
{
	memwrite (default_console_buffer, default_console_buffer_size);
	read_console_buffer (default_console_buffer);
	cout << crlf;
	return std::string (
			reinterpret_cast <char *>
				(mem + default_console_buffer + 2),
			memread (default_console_buffer + 1) );
}

namespace {

void ltrim (std::string & str)
{
	std::string::size_type l= str.find_first_not_of (" \t");
	if (l != std::string::npos)
		str.erase (0, l);
}

std::string getcommand (const std::string & str)
{
	const std::string::size_type l= str.find_first_of (" \t");
	if (l != std::string::npos)
		return str.substr (0, l);
	else
		return str;
}

bool getnumber (const std::string & str, word & n)
{
	const char * cstr= str.c_str ();
	char * aux;

	unsigned long l= (* cstr == '#') ?
		strtoul (cstr + 1, & aux, 10) :
		strtoul (cstr, & aux, 16);

	if (* aux != '\0')
		return false;
	if (l > 65535)
		return false;
	n= static_cast <word> (l);
	return true;
}

bool getbytenumber (const std::string & str, byte & n)
{
	const char * cstr= str.c_str ();
	char * aux;

	unsigned long l= (* cstr == '#') ?
		strtoul (cstr + 1, & aux, 10) :
		strtoul (cstr, & aux, 16);

	if (* aux != '\0')
		return false;
	if (l > 255)
		return false;
	n= static_cast <byte> (l);
	return true;
}

class Debugger {
public:
	Debugger (CPM::Internal & cpmi) :
		cpmi (cpmi),
		lastlistpos (tpa),
		displaypos (tpa)
	{ }
	~Debugger ()
	{
		cpmi.endruntransient ();
	}
	void doit ();
private:
	CPM::Internal & cpmi;

	typedef std::string string;
	typedef std::vector <string> vparam;
	enum Result { Ok, Bad, End };

	typedef std::map <word, byte> breakpoint_t;
	breakpoint_t breakpoint;

	typedef Result (Debugger::*func) (const vparam &);
	typedef std::map <char, func> mapfunc_t;
	static const mapfunc_t mapfunc;

	static mapfunc_t createmap ();
	static func getfunc (char letter);

	word lastlistpos;
	word displaypos;

	enum Flag { FlagC, FlagZ, FlagM, FlagE, FlagI };
	enum Reg { RegB, RegD, RegH, RegS, RegP, RegIX, RegIY };

	static vparam parseparam (string cmd);

	Result go (const vparam & vp);
	Result putbreakpoint (const vparam & vp);
	Result quit (const vparam & vp);
	void examineflag (Flag f);
	void examineA ();
	void examinereg (Reg r);
	Result examine (const vparam & vp);
	Result trace (const vparam & vp);
	Result untrace (const vparam & vp);
	word displayline (word from, word to);
	Result display (const vparam & vp);
	Result set (const vparam & vp);
	Result fill (const vparam & vp);
	Result move (const vparam & vp);
	#if defined USE_ImcZ80 && ! defined DISASM_WITH_libz80
	word disassembly (word pos);
	Result list (const vparam & vp);
	#endif
};

Debugger::mapfunc_t Debugger::createmap ()
{
	mapfunc_t m;

	m ['G']= & Debugger::go;
	m ['P']= & Debugger::putbreakpoint;
	m ['Q']= & Debugger::quit;
	m ['X']= & Debugger::examine;
	m ['T']= & Debugger::trace;
	m ['U']= & Debugger::untrace;
	m ['D']= & Debugger::display;
	m ['S']= & Debugger::set;
	m ['F']= & Debugger::fill;
	m ['M']= & Debugger::move;

	#if defined USE_ImcZ80 && ! defined DISASM_WITH_libz80
	m ['L']= & Debugger::list;
	#endif

	return m;
}

const Debugger::mapfunc_t Debugger::mapfunc (Debugger::createmap () );

Debugger::func Debugger::getfunc (char letter)
{
	mapfunc_t::const_iterator it= mapfunc.find (letter);
	if (it != mapfunc.end () )
		return it->second;
	else
		return 0;
}

Debugger::vparam Debugger::parseparam (string cmd)
{
	vparam v;
	string::size_type pos= 0;
	while ( (pos= cmd.find (',', pos) ) != string::npos)
	{
		v.push_back (cmd.substr (0, pos) );
		cmd.erase (0, pos + 1);
	}
	if (! cmd.empty () )
		v.push_back (cmd);
	return v;
}

Debugger::Result Debugger::go (const vparam & vp)
{
	typedef std::set <word> local_t;
	local_t localbreakpoint;

	word pos= cpmi.get_pc ();
	size_t nargs= vp.size ();
	if (nargs > 0)
	{
		if (! vp [0].empty () && ! getnumber (vp [0], pos) )
			return Bad;
		for (size_t i= 1; i < nargs; ++i)
		{
			word breakpos;
			if (! getnumber (vp [i], breakpos) )
				return Bad;
			localbreakpoint.insert (breakpos);
		}
	}
	cpmi.set_pc (pos);

	const local_t::iterator localend= localbreakpoint.end ();
	const breakpoint_t::iterator breakpointend= breakpoint.end ();
	for (;;)
	{
		// Check local breakpoints.
		if (localbreakpoint.find (cpmi.get_pc () ) != localend)
		{
			lastlistpos= cpmi.get_pc ();
			return Ok;
		}

		CPM::Internal::TrapReturn r= cpmi.runinstruction ();
		if (r == CPM::Internal::TrapEndProgram)
			return End;

		// Check persistent breakpoints after execution, to avoid
		// start executing after stoping in one.
		breakpoint_t::iterator it= breakpoint.find (cpmi.get_pc () );
		if (it != breakpointend)
		{
			cout << hex2 (it->second) <<
				" PASS " << hex4 (it->first) << crlf;
			if (--it->second == 0)
			{
				it->second= 1;
				lastlistpos= cpmi.get_pc ();
				return Ok;
			}
		}

	}

	return End;
}

Debugger::Result Debugger::putbreakpoint (const vparam & vp)
{
	if (vp.empty () )
	{
		for (breakpoint_t::iterator it= breakpoint.begin ();
			it != breakpoint.end ();
			++it)
		{
			cout << hex2 (it->second) << ' ' <<
				hex4 (it->first) << crlf;
		}
		return Ok;
	}
	size_t narg= vp.size ();
	word breakpos;
	if (! getnumber (vp [0], breakpos) )
		return Bad;
	byte count= 1;
	if (narg > 1)
	{
		if (! getbytenumber (vp [1], count) )
			return Bad;
	}
	if (count == 0)
		breakpoint.erase (breakpos);
	else
		breakpoint [breakpos]= count;
	return Ok;
}

Debugger::Result Debugger::quit (const vparam &)
{
	return End;
}

void Debugger::examineflag (Flag f)
{
	bool value;
	switch (f)
	{
	case FlagC: value= cpmi.flag_c (); break;
	case FlagZ: value= cpmi.flag_z (); break;
	case FlagM: value= cpmi.flag_s (); break;
	case FlagE: value= cpmi.flag_pv (); break;
	case FlagI: value= cpmi.flag_h (); break;
	}
	cout << (value ? '1' : '0') << ' '<< flush;
	string strvalue= cpmi.getcommandline ();
	if (strvalue.empty () )
		return;
	if (strvalue == "0")
		value= false;
	else if (strvalue == "1")
		value= true;
	else
	{
		cout << '?' << crlf;
		return;
	}
	switch (f)
	{
	case FlagC: cpmi.setflag_c (value); break;
	case FlagZ: cpmi.setflag_z (value); break;
	case FlagM: cpmi.setflag_s (value); break;
	case FlagE: cpmi.setflag_pv (value); break;
	case FlagI: cpmi.setflag_h (value); break;
	}
}

void Debugger::examineA ()
{
	byte value= cpmi.get_a ();
	cout << hex2 (value) << ' ' << flush;
	string strvalue= cpmi.getcommandline ();
	if (strvalue.empty () )
		return;
	byte newval;
	if (! getbytenumber (strvalue, newval) )
	{
		cout << '?' << crlf;
		return;
	}
	cpmi.set_a (newval);
}

void Debugger::examinereg (Reg r)
{
	word value;
	switch (r)
	{
	case RegB: value= cpmi.get_bc (); break;
	case RegD: value= cpmi.get_de (); break;
	case RegH: value= cpmi.get_hl (); break;
	case RegS: value= cpmi.get_sp (); break;
	case RegP: value= cpmi.get_pc (); break;
	case RegIX: value= cpmi.get_ix (); break;
	case RegIY: value= cpmi.get_iy (); break;
	}
	cout << hex4 (value) << ' ' << flush;
	string strvalue= cpmi.getcommandline ();
	if (strvalue.empty () )
		return;
	if (! getnumber (strvalue, value) )
	{
		cout << '?' << crlf;
		return;
	}
	switch (r)
	{
	case RegB: cpmi.set_bc (value); break;
	case RegD: cpmi.set_de (value); break;
	case RegH: cpmi.set_hl (value); break;
	case RegS: cpmi.set_sp (value); break;
	case RegP: cpmi.set_pc (value); break;
	case RegIX: cpmi.set_ix (value); break;
	case RegIY: cpmi.set_iy (value); break;
	}
}

Debugger::Result Debugger::examine (const vparam & vp)
{
	switch (vp.size () )
	{
	case 0:
		cpmi.showcurrentstate ();
		break;
	case 1:
		{
			const string & str= vp [0];
			if (str == "C")
				examineflag (FlagC);
			else if (str == "Z")
				examineflag (FlagZ);
			else if (str == "M")
				examineflag (FlagM);
			else if (str == "E")
				examineflag (FlagE);
			else if (str == "I")
				examineflag (FlagI);
			else if (str == "A")
				examineA ();
			else if (str == "B")
				examinereg (RegB);
			else if (str == "D")
				examinereg (RegD);
			else if (str == "H")
				examinereg (RegH);
			else if (str == "S")
				examinereg (RegS);
			else if (str == "P")
				examinereg (RegP);
			else if (str == "IX")
				examinereg (RegIX);
			else if (str == "IY")
				examinereg (RegIY);
			else
				cout << '?' << crlf;
		}
		break;
	default:
		return Bad;
	}
	return Ok;
}

Debugger::Result Debugger::trace (const vparam & vp)
{
	word n= 1;
	switch (vp.size () )
	{
	case 0:
		// Nothing.
		break;
	case 1:
		if (! getnumber (vp [0], n) )
			return Bad;
		break;
	default:
		return Bad;
	}
	for ( ; n > 0; --n)
	{
		cpmi.showcurrentstate ();
		#if 0
		cpmi.execute ();
		#else
		CPM::Internal::TrapReturn r= cpmi.runinstruction ();
		if (r == CPM::Internal::TrapEndProgram)
			return End;
		#endif

	}
	lastlistpos= cpmi.get_pc ();
	return Ok;
}

Debugger::Result Debugger::untrace (const vparam & vp)
{
	word n= 1;
	switch (vp.size () )
	{
	case 0:
		// Nothing.
		break;
	case 1:
		if (! getnumber (vp [0], n) )
			return Bad;
		break;
	default:
		return Bad;
	}
	cpmi.showcurrentstate ();
	for ( ; n > 0; --n)
	{
		CPM::Internal::TrapReturn r= cpmi.runinstruction ();
		if (r == CPM::Internal::TrapEndProgram)
			return End;
	}
	lastlistpos= cpmi.get_pc ();
	return Ok;
}

word Debugger::displayline (word from, word to)
{
	ASSERT (to > from && (to - from <= 16) );

	cout << hex4 (from) << "  ";
	for (word i= from; i < to; ++i)
		cout << hex2 (cpmi.memread (i) ) << ' ';
	for (int i= 0; i < 16 - (to - from); ++i)
		cout << "   ";

	for (word i= from; i < to; ++i)
	{
		unsigned char c= cpmi.memread (i);
		if (c < 32 || c > 127)
			c= '.';
		cout << c;
	}
	cout << crlf;
	return to;
}

Debugger::Result Debugger::display (const vparam & vp)
{
	size_t narg= vp.size ();
	if (narg > 0)
	{
		if (! vp [0].empty () )
			if (! getnumber (vp [0], displaypos) )
				return Bad;
	}
	if (narg > 1)
	{
		if (narg > 2)
			return Bad;
		word endpos;
		if (! getnumber (vp [1], endpos) )
			return Bad;
		++endpos;
		while (displaypos < endpos)
		{
			word endline= displaypos + 16;
			if (endline > endpos)
				endline= endpos;
			displaypos= displayline (displaypos, endline);
		}
		return Ok;
	}
	for (int i= 0; i < 12; ++i)
		displaypos= displayline (displaypos, displaypos + 16);
	return Ok;
}

Debugger::Result Debugger::set (const vparam & vp)
{
	if (vp.size () != 1)
		return Bad;
	word pos;
	if (! getnumber (vp [0], pos) )
		return Bad;
	for (;;)
	{
		cout << hex4 (pos) << ' ' <<
			hex2 (cpmi.memread (pos) ) << ' ' << flush;
		string strvalue= cpmi.getcommandline ();
		if (strvalue == ".")
			break;
		if (strvalue.empty () )
			++pos;
		else
		{
			byte val;
			if (! getbytenumber (strvalue, val) )
				cout << '?' << crlf;
			else
			{
				cpmi.memwrite (pos, val);
				++pos;
			}
		}
	}
	return Ok;
}

Debugger::Result Debugger::fill (const vparam & vp)
{
	if (vp.size () != 3)
		return Bad;
	word from;
	if (! getnumber (vp [0], from) )
		return Bad;
	word to;
	if (! getnumber (vp [1], to) || to == 0xFFFF)
		return Bad;
	if (to < from)
		return Bad;
	byte b;
	if (! getbytenumber (vp [2], b) )
		return Bad;
	for (word i= from; i <= to; ++i)
		cpmi.memwrite (i, b);
	return Ok;
}

Debugger::Result Debugger::move (const vparam & vp)
{
	if (vp.size () != 3)
		return Bad;
	word from;
	if (! getnumber (vp [0], from) )
		return Bad;
	word to;
	if (! getnumber (vp [1], to) || to == 0xFFFF)
		return Bad;
	if (to < from)
		return Bad;
	word dest;
	if (! getnumber (vp [2], dest) )
		return Bad;
	word size= to - from + 1;

	if (dest > from && dest <= to)
	{
		dest+= size - 1;
		for ( ; size > 0; --size)
			cpmi.memwrite (dest--, cpmi.memread (to--) );
	}
	else
	{
		for ( ; size > 0; --size)
			cpmi.memwrite (dest++, cpmi.memread (from++) );
	}
	return Ok;
}

#if defined USE_ImcZ80 && ! defined DISASM_WITH_libz80

word Debugger::disassembly (word pos)
{
	cout << hex4 (pos) << ' ';
	if (cpmi.memread (pos)  == trapbyte &&
		( cpmi.gettrapfunc (pos) ) != 0)
	{
		cout << "System trap";
		++pos;
	}
	else
		pos= cpmi.disassembly (cout, pos);
	cout << crlf;
	return pos;
}

Debugger::Result Debugger::list (const vparam & vp)
{
	//word pos= cpmi.get_pc ();
	word pos= lastlistpos;
	size_t nargs= vp.size ();
	if (nargs > 0)
	{
		if (! getnumber (vp [0], pos) )
			return Bad;
		if (nargs > 1)
		{
			word posend;
			if (! getnumber (vp [1], posend) )
				return Bad;
			if (nargs > 2)
				return Bad;
			while (pos < posend)
				pos= disassembly (pos);
			lastlistpos= pos;
			return Ok;
		}
	}
	for (int i= 0; i < 11; ++i)
		pos= disassembly (pos);
	lastlistpos= pos;
	return Ok;
}

#endif

void Debugger::doit ()
{
	std::string cmdline;
	for (;;)
	{
		cout << '#' << flush;
		cmdline= cpmi.getcommandline ();
		ltrim (cmdline);
		if (cmdline.empty () )
			continue;
		//char lettercommand= cmdline [0];
		cmdline= strupper (cmdline);
		char lettercommand= toupper (cmdline [0] );
		cmdline.erase (0, 1);
		vparam vp= parseparam (cmdline);
		Result result= Bad;

		func f= getfunc (lettercommand);
		if (f)
			result= (this->*f) (vp);

		if (result == Bad)
			cout << '?' << crlf;
		if (result == End)
			break;
	}
}

} // namespace

void CPM::Internal::debugger ()
{
	Debugger deb (* this);
	deb.doit ();
}

void CPM::Internal::poke (const std::string & strcmd)
{
	std::istringstream iss (strcmd);
	word pos;
	std::string strpos;
	iss >> strpos;
	if (! getnumber (strpos, pos) )
	{
		cout << "@POKE " << strcmd << '?' << crlf;
		return;
	}
	for (;;)
	{
		char c;
		iss >> c; // >> take care of white space.
		if (! iss)
			break;
		if (c == '"')
		{
			while ( (c= iss.get () ) != '"' && iss)
			{
				memwrite (pos, c);
				++pos;
			}
		}
		else
		{
			iss.unget ();
			std::string strnum;
			iss >> strnum;
			byte num;
			if (! getbytenumber (strnum, num) )
			{
				cout << "@POKE " << strcmd << '?' << crlf;
				return;
			}
			memwrite (pos, num);
			++pos;
		}
	}
}

void CPM::Internal::interactive (bool ndebugsystem, bool ndebugz80)
{
	static const std::string exitcommand ("@EXIT");
	static const std::string debugcommand ("@DEBUG");
	static const std::string loadcommand ("@LOAD");
	static const std::string gocommand ("@GO");
	static const std::string pokecommand ("@POKE");

	console.activate ();

	std::string cmdline;
	for (;;)
	{
		cout << "A>" << flush;

		//getline (cin, cmdline);

		#if 0
		memwrite (default_console_buffer, default_console_buffer_size);
		read_console_buffer (default_console_buffer);
		cout << crlf;
		cmdline= std::string (
			reinterpret_cast <char *>
				(mem + default_console_buffer + 2),
			memread (default_console_buffer + 1) );
		#else
		cmdline= getcommandline ();
		#endif

		#if 0
		std::string::size_type l= cmdline.find_first_not_of (" \t");
		if (l != std::string::npos)
			cmdline.erase (0, l);
		#else
		ltrim (cmdline);
		#endif

		if (cmdline.empty () )
			continue;
		if (cmdline [0] == ';')
			continue;

		//cmdline= strupper (cmdline);
		std::string cmdupper= strupper (cmdline);

		std::string command= getcommand (cmdupper);

		if (command == exitcommand)
			break;

		enum RunMode { Run, Debug, Load };
		RunMode runmode= Run;

		//bool enterdebug= false;
		if (command == debugcommand)
		{
			runmode= Debug;
			//enterdebug= true;
			cmdupper.erase (0, debugcommand.size () );
			ltrim (cmdupper);
			if (cmdupper.empty () )
			{
				prepareruntransient ();
				debugger ();
				continue;
			}
		}
		else if (command == loadcommand)
		{
			runmode= Load;
			cmdupper.erase (0, debugcommand.size () );
			ltrim (cmdupper);
			if (cmdupper.empty () )
			{
				cout << "LOAD what?" << crlf;
				continue;
			}
		}
		else if (command == gocommand)
		{
			cmdupper.erase (0, gocommand.size () );
			ltrim (cmdupper);
			word start= tpa;
			std::istringstream iss (cmdupper);
			std::string strstart;
			iss >> strstart;
			if (iss)
			{
				if (! getnumber (strstart, start) )
				{
					cout << "@GO " << strstart <<
						'?' << crlf;
					continue;
				}
				iss.get ();
				if (! iss.eof () )
				{
					cout << "@GO " << cmdupper <<
						'?' << crlf;
				}
			}

			prepareruntransient ();
			set_pc (start);
			run (ndebugsystem, ndebugz80);
			endruntransient ();
			continue;
		}
		else if (command == pokecommand)
		{
			cmdline.erase (0, pokecommand.size () );
			ltrim (cmdline);
			poke (cmdline);
			continue;
		}

		setcmd_args (cmdupper);
		word pos= cmd_args + 1;
		const word endline= pos + memread (cmd_args);
		cleardefaultfcb (default_fcb_1, default_pass_1);
		parsefilecmdline (pos, endline,
			default_fcb_1, default_pass_1);
		std::string ext= extension_from_fcb (default_fcb_1);
		if (! ext.empty () && ext != "COM" && ext != "PRL")
		{
			cerr << "Invalid extension: '" << ext << "'" << crlf;
			cout << filename_from_fcb (default_fcb_1) <<
				'?' << crlf;
			continue;
		}
		if (ext.empty () )
			setcomextension (default_fcb_1);

		word n= pos - (cmd_args + 1);
		cmdupper.erase (0, n);

		cout << "Executing " << filename_from_fcb (default_fcb_1) << 
			cmdupper << crlf;
		if (! open_file (default_fcb_1) )
		{
			cout << "Error opening command" << crlf;
			continue;
		}
		OpenFile * pof= getopenfile (default_fcb_1);

		if (ext == "PRL")
		{
			cerr << "Extension: '" << ext << '\'' << crlf;
			if (! loadprltpa (pof->gethandle () ) )
			{
				cout << "Error loading command" << crlf;
				continue;
			}
		}
		else
		{
			if (! loadfiletpa (pof->gethandle () ) )
			{
				cout << "Error loading command" << crlf;
				continue;
			}
		}

		close_all ();
		setarguments (cmdupper);
		//runtransient (ndebugsystem, ndebugz80);
		prepareruntransient ();

		switch (runmode)
		{
		case Debug:
			debugger ();
			endruntransient ();
			break;
		case Run:
			run (ndebugsystem, ndebugz80);
			endruntransient ();
			break;
		case Load:
			// Nothing to do, program is already loaded.
			break;
		}
	}

	console.deactivate ();
}

//*********************************************************
//				CPM
//*********************************************************

CPM::CPM (Console & console, Printer & printer) :
	p (new Internal (console, printer) )
{

}

CPM::~CPM ()
{
	delete p;
}

void CPM::settpaend (unsigned short tpaend)
{
	p->settpaend (tpaend);
}


void CPM::setarguments (const std::string & args)
{
	p->setarguments (args);
}

bool CPM::loadcommand (const std::string & filename)
{
	return p->loadcommand (filename);
}

void CPM::tpastart (bool ndebugsystem, bool ndebugz80)
{
	p->tpastart (ndebugsystem, ndebugz80);
}

void CPM::interactive (bool ndebugsystem, bool ndebugz80)
{
	p->interactive (ndebugsystem, ndebugz80);
}

// End of cpm.cpp
