/*
 *  Copyright (C) 2006 Ludovic Jacomme (ludovic.jacomme@gmail.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <fcntl.h>

#include <pspctrl.h>
#include <pspkernel.h>
#include <pspdebug.h>

#include "cap32.h"
#include "z80.h"
#include "psp_kbd.h"
#include "psp_danzeff.h"
#include "kbd.h"
#include "psp_menu.h"

# define KBD_MIN_ANALOG_TIME      150000
# define KBD_MIN_START_TIME       800000
# define KBD_MAX_EVENT_TIME       500000
# define KBD_MIN_PENDING_TIME     300000
# define KBD_MIN_DANZEFF_TIME     150000
# define KBD_MIN_COMMAND_TIME     100000
# define KBD_MIN_BATTCHECK_TIME 90000000 

 static SceCtrlData    loc_button_data;
 static unsigned int   loc_last_event_time = 0;
 static unsigned int   loc_last_analog_time = 0;
 static int            loc_analog_x_released = 0;
 static int            loc_analog_y_released = 0;
 static long           first_time_stamp = -1;
 static char           loc_button_press[ KBD_MAX_BUTTONS ]; 
 static char           loc_button_release[ KBD_MAX_BUTTONS ]; 
 static unsigned int   loc_button_mask[ KBD_MAX_BUTTONS ] =
 {
   PSP_CTRL_UP         , /*  KBD_UP         */
   PSP_CTRL_RIGHT      , /*  KBD_RIGHT      */
   PSP_CTRL_DOWN       , /*  KBD_DOWN       */
   PSP_CTRL_LEFT       , /*  KBD_LEFT       */
   PSP_CTRL_TRIANGLE   , /*  KBD_TRIANGLE   */
   PSP_CTRL_CIRCLE     , /*  KBD_CIRCLE     */
   PSP_CTRL_CROSS      , /*  KBD_CROSS      */
   PSP_CTRL_SQUARE     , /*  KBD_SQUARE     */
   PSP_CTRL_SELECT     , /*  KBD_SELECT     */
   PSP_CTRL_START      , /*  KBD_START      */
   PSP_CTRL_HOME       , /*  KBD_HOME       */
   PSP_CTRL_HOLD       , /*  KBD_HOLD       */
   PSP_CTRL_LTRIGGER   , /*  KBD_LTRIGGER   */
   PSP_CTRL_RTRIGGER   , /*  KBD_RTRIGGER   */
 };

 static char loc_button_name[ KBD_ALL_BUTTONS ][10] =
 {
   "Nach Oben",
   "Nach Rechts",
   "Nach Unten",
   "Nach Links",
   "Dreieck",
   "Kreis",
   "Kreuz",
   "Viereck",
   "SELECT",
   "START",
   "HOME",
   "HOLD",
   "LTRIGGER",
   "RTRIGGER",
   "JOY_HOCH",
   "JOY_RECHTS",
   "JOY_RUNTER",
   "JOY_LINKS"
 };

 static int loc_default_mapping[ KBD_ALL_BUTTONS ] = {
   CPC_CUR_UP          , /*  KBD_UP         */
   CPC_CUR_RIGHT       , /*  KBD_RIGHT      */
   CPC_CUR_DOWN        , /*  KBD_DOWN       */
   CPC_CUR_LEFT        , /*  KBD_LEFT       */
   CPC_ENTER           , /*  KBD_TRIANGLE   */
   CPC_J0_FIRE2        , /*  KBD_CIRCLE     */
   CPC_J0_FIRE1        , /*  KBD_CROSS      */
   CPC_DEL             , /*  KBD_SQUARE     */
   -1                  , /*  KBD_SELECT     */
   -1                  , /*  KBD_START      */
   -1                  , /*  KBD_HOME       */
   -1                  , /*  KBD_HOLD       */
   CPC_ESC             , /*  KBD_LTRIGGER   */
   CPC_SPACE           , /*  KBD_RTRIGGER   */
   CPC_J0_UP           , /*  KBD_JOY_UP     */
   CPC_J0_RIGHT        , /*  KBD_JOY_RIGHT  */
   CPC_J0_DOWN         , /*  KBD_JOY_DOWN   */
   CPC_J0_LEFT           /*  KBD_JOY_LEFT   */
 };

        int psp_kbd_mapping[ KBD_ALL_BUTTONS ];

 static int danzeff_cpc_key     = 0;
 static int danzeff_cpc_pending = 0;
 static int danzeff_mode        = 0;


       char command_keys[ 128 ];
 static int command_mode        = 0;
 static int command_index       = 0;
 static int command_size        = 0;
 static int command_cpc_pending = 0;
 static int command_cpc_key     = 0;

void
psp_kbd_run_command(char *Command)
{
  strncpy(command_keys, Command, 128);
  command_size  = strlen(Command);
  command_index = 0;

  command_cpc_key     = 0;
  command_cpc_pending = 0;
  command_mode        = 1;
}

int
psp_kbd_reset_mapping(void)
{
  memcpy(psp_kbd_mapping, loc_default_mapping, sizeof(loc_default_mapping));
  return 0;
}

int
psp_kbd_load_mapping(char *kbd_filename)
{
  char     Buffer[512];
  FILE    *KbdFile;
  char    *Scan;
  int      tmp_mapping[KBD_ALL_BUTTONS];
  int      cpc_key_id = 0;
  int      kbd_id = 0;
  int      error = 0;

  memcpy(tmp_mapping, loc_default_mapping, sizeof(loc_default_mapping));

  KbdFile = fopen(kbd_filename, "r");
  error   = 1;

  if (KbdFile != (FILE*)0) {

    while (fgets(Buffer,512,KbdFile) != (char *)0) {

      Scan = strchr(Buffer,'\n');
      if (Scan) *Scan = '\0';
      /* For this #@$% of windows ! */
      Scan = strchr(Buffer,'\r');
      if (Scan) *Scan = '\0';
      if (Buffer[0] == '#') continue;

      Scan = strchr(Buffer,'=');
      if (! Scan) continue;

      *Scan = '\0';
      cpc_key_id = atoi(Scan + 1);

      for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++) {
        if (!strcasecmp(Buffer,loc_button_name[kbd_id])) {
          tmp_mapping[kbd_id] = cpc_key_id;
          break;
        }
      }
    }

    error = 0;
    fclose(KbdFile);
  }

  memcpy(psp_kbd_mapping, tmp_mapping, sizeof(psp_kbd_mapping));

  return error;
}

int
psp_kbd_save_mapping(char *kbd_filename)
{
  FILE    *KbdFile;
  int      kbd_id = 0;
  int      error = 0;

  KbdFile = fopen(kbd_filename, "w");
  error   = 1;

  if (KbdFile != (FILE*)0) {

    for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++)
    {
      fprintf(KbdFile, "%s=%d\n", loc_button_name[kbd_id], psp_kbd_mapping[kbd_id]);
    }
    error = 0;
    fclose(KbdFile);
  }

  return error;
}

int
psp_kbd_enter_command()
{
  SceCtrlData  c;

  unsigned int command_key = 0;
  int          cpc_key     = 0;
  int          key_event   = 0;

  sceCtrlPeekBufferPositive(&c, 1);

  if (command_cpc_pending) 
  {
    if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_COMMAND_TIME) {
      loc_last_event_time = c.TimeStamp;
      command_cpc_pending = 0;
      cpc_key_event(command_cpc_key, 0);
    }

    return 0;
  }

  if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_COMMAND_TIME) {
    loc_last_event_time = c.TimeStamp;

    if (command_index >= command_size) {

      command_mode  = 0;
      command_index = 0;
      command_size  = 0;

      command_cpc_pending = 0;
      command_cpc_key     = 0;

      return 0;
    }
  
    command_key = command_keys[command_index++];
    cpc_key = cpc_get_key_from_ascii(command_key);

    if (cpc_key != -1) {
      command_cpc_key     = cpc_key;
      command_cpc_pending = 1;
      cpc_key_event(command_cpc_key, 1);
    }

    return 1;
  }

  return 0;
}

int 
psp_kbd_is_danzeff_mode()
{
  return danzeff_mode;
}

int
psp_kbd_enter_danzeff()
{
  unsigned int danzeff_key = 0;
  int          cpc_key     = 0;
  int          key_event   = 0;
  SceCtrlData  c;

  if (! danzeff_mode) {
    psp_init_keyboard();
    danzeff_mode = 1;
  }

  sceCtrlPeekBufferPositive(&c, 1);

  if (danzeff_cpc_pending) 
  {
    if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_PENDING_TIME) {
      loc_last_event_time = c.TimeStamp;
      danzeff_cpc_pending = 0;
      cpc_key_event(danzeff_cpc_key, 0);
    }

    return 0;
  }

  if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_DANZEFF_TIME) {
    loc_last_event_time = c.TimeStamp;
  
    sceCtrlPeekBufferPositive(&c, 1);
    danzeff_key = danzeff_readInput(c);
  }

  if (danzeff_key > DANZEFF_START) {
    cpc_key = cpc_get_key_from_ascii(danzeff_key);

    if (cpc_key != -1) {
      danzeff_cpc_key     = cpc_key;
      danzeff_cpc_pending = 1;
      cpc_key_event(danzeff_cpc_key, 1);
    }

    return 1;

  } else if (danzeff_key == DANZEFF_START) {
    danzeff_mode        = 0;
    danzeff_cpc_pending = 0;
    danzeff_cpc_key     = 0;
  } else if (danzeff_key == DANZEFF_SELECT) {
    danzeff_mode        = 0;
    danzeff_cpc_pending = 0;
    danzeff_cpc_key     = 0;
    psp_main_menu();
    psp_init_keyboard();
  }

  return 0;
}

int
cpc_decode_key(int psp_b, int button_pressed)
{
  int wake = 0;
  int reverse_analog = CPC.psp_reverse_analog;

  if (reverse_analog) {
    if ((psp_b >= KBD_JOY_UP  ) &&
        (psp_b <= KBD_JOY_LEFT)) {
      psp_b = psp_b - KBD_JOY_UP + KBD_UP;
    } else
    if ((psp_b >= KBD_UP  ) &&
        (psp_b <= KBD_LEFT)) {
      psp_b = psp_b - KBD_UP + KBD_JOY_UP;
    }
  }

  if (psp_b == KBD_START) {
     if (button_pressed) psp_kbd_enter_danzeff();
  } else
  if (psp_b == KBD_SELECT) {
    if (button_pressed) {
      psp_main_menu();
      psp_init_keyboard();
    }
  } else {
 
    if (psp_kbd_mapping[psp_b] != -1) {
      wake = 1;
      cpc_key_event(psp_kbd_mapping[psp_b], button_pressed);
    }
  }
  return 0;
}

# define ANALOG_THRESHOLD 60

void 
kbd_get_analog_direction(int Analog_x, int Analog_y, int *x, int *y)
{
  int DeltaX = 255;
  int DeltaY = 255;
  int DirX   = 0;
  int DirY   = 0;

  *x = 0;
  *y = 0;

  if (Analog_x <=        ANALOG_THRESHOLD)  { DeltaX = Analog_x; DirX = -1; }
  else 
  if (Analog_x >= (255 - ANALOG_THRESHOLD)) { DeltaX = 255 - Analog_x; DirX = 1; }

  if (Analog_y <=        ANALOG_THRESHOLD)  { DeltaY = Analog_y; DirY = -1; }
  else 
  if (Analog_y >= (255 - ANALOG_THRESHOLD)) { DeltaY = 255 - Analog_y; DirY = 1; }

  *x = DirX;
  *y = DirY;
}

static int 
kbd_reset_button_status(void)
{
  int b = 0;
  /* Reset Button status */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    loc_button_press[b]   = 0;
    loc_button_release[b] = 0;
  }
  psp_init_keyboard();
  return 0;
}

int
kbd_scan_keyboard(void)
{
  SceCtrlData c;
  long        delta_stamp;
  int         event;
  int         b;

  int new_Lx;
  int new_Ly;
  int old_Lx;
  int old_Ly;

  event = 0;
  sceCtrlPeekBufferPositive( &c, 1 );

  if ((c.Buttons & (PSP_CTRL_LTRIGGER|PSP_CTRL_RTRIGGER|PSP_CTRL_START)) ==
      (PSP_CTRL_LTRIGGER|PSP_CTRL_RTRIGGER|PSP_CTRL_START)) {
    /* Exit ! */
    SDL_Quit();
    exit(0);
  }

  delta_stamp = c.TimeStamp - first_time_stamp;
  if ((delta_stamp < 0) || (delta_stamp > KBD_MIN_BATTCHECK_TIME)) {
    first_time_stamp = c.TimeStamp;
    if (psp_is_low_battery()) {
      psp_main_menu();
      psp_init_keyboard();
      return 0;
    }
  }

  /* Check Analog Device */
  kbd_get_analog_direction(loc_button_data.Lx,loc_button_data.Ly,&old_Lx,&old_Ly);
  kbd_get_analog_direction( c.Lx, c.Ly, &new_Lx, &new_Ly);

  /* Analog device has moved */
  if (new_Lx > 0) {
    if (old_Lx  > 0) cpc_decode_key(KBD_JOY_LEFT , 0);
    cpc_decode_key(KBD_JOY_RIGHT, 1);
    loc_analog_x_released = 0;

  } else 
  if (new_Lx < 0) {
    if (old_Lx  < 0) cpc_decode_key(KBD_JOY_RIGHT, 0);
    cpc_decode_key(KBD_JOY_LEFT , 1);
    loc_analog_x_released = 0;

  } else {
    if (old_Lx  > 0) cpc_decode_key(KBD_JOY_LEFT , 0);
    else
    if (old_Lx  < 0) cpc_decode_key(KBD_JOY_RIGHT, 0);
    else if (! loc_analog_x_released) {
      cpc_decode_key(KBD_JOY_LEFT  , 0);
      cpc_decode_key(KBD_JOY_RIGHT , 0);
      loc_analog_x_released = 1;
    }
  }

  if (new_Ly < 0) {
    if (old_Ly  > 0) cpc_decode_key(KBD_JOY_DOWN , 0);
    cpc_decode_key(KBD_JOY_UP   , 1);
    loc_analog_y_released = 0;

  } else 
  if (new_Ly > 0) {
    if (old_Ly  < 0) cpc_decode_key(KBD_JOY_UP   , 0);
    cpc_decode_key(KBD_JOY_DOWN , 1);
    loc_analog_y_released = 0;

  } else {
    if (old_Ly  > 0) cpc_decode_key(KBD_JOY_DOWN , 0);
    else
    if (old_Ly  < 0) cpc_decode_key(KBD_JOY_UP   , 0);
    else if (! loc_analog_y_released) {
      cpc_decode_key(KBD_JOY_DOWN , 0);
      cpc_decode_key(KBD_JOY_UP   , 0);
      loc_analog_y_released = 1;
    }
  }

  for (b = 0; b < KBD_MAX_BUTTONS; b++) 
  {
    if (c.Buttons & loc_button_mask[b]) {
      if (!(loc_button_data.Buttons & loc_button_mask[b])) {
        loc_button_press[b] = 1;
        event = 1;
      }
    } else {
      if (loc_button_data.Buttons & loc_button_mask[b]) {
        loc_button_release[b] = 1;
        event = 1;
      }
    }
  }
  memcpy(&loc_button_data,&c,sizeof(SceCtrlData));

  return event;
}

void
kbd_wait_start(void)
{
  while (1)
  {
    SceCtrlData c;
    sceCtrlReadBufferPositive(&c, 1);
    if (c.Buttons & PSP_CTRL_START) break;
  }
}

void
psp_init_keyboard(void)
{
  cpc_kbd_reset();
}

void
psp_kbd_wait_no_button(void)
{
  SceCtrlData c;

  do {
   sceCtrlPeekBufferPositive(&c, 1);
  } while (c.Buttons != 0);
} 

int
psp_update_keys(void)
{
  int         b;

  static char first_time = 1;
  static int release_pending = 0;

  if (first_time) {

    memcpy(psp_kbd_mapping, loc_default_mapping, sizeof(loc_default_mapping));

    SceCtrlData c;
    sceCtrlPeekBufferPositive(&c, 1);

    if (first_time_stamp == -1) first_time_stamp = c.TimeStamp;
    if ((! c.Buttons) && ((c.TimeStamp - first_time_stamp) < KBD_MIN_START_TIME)) return 0;

    first_time      = 0;
    release_pending = 0;

    for (b = 0; b < KBD_MAX_BUTTONS; b++) {
      loc_button_release[b] = 0;
      loc_button_press[b] = 0;
    }
    sceCtrlPeekBufferPositive(&loc_button_data, 1);

    psp_main_menu();
    psp_init_keyboard();

    return 0;
  }

  if (command_mode) {
    return psp_kbd_enter_command();
  }

  if (danzeff_mode) {
    return psp_kbd_enter_danzeff();
  }

  if (release_pending)
  {
    release_pending = 0;
    for (b = 0; b < KBD_MAX_BUTTONS; b++) {
      if (loc_button_release[b]) {
        loc_button_release[b] = 0;
        loc_button_press[b] = 0;
        cpc_decode_key(b, 0);
      }
    }
  }

  kbd_scan_keyboard();

  /* check release event */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    if (loc_button_release[b]) {
      release_pending = 1;
      break;
    } 
  }
  /* check press event */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    if (loc_button_press[b]) {
      loc_button_press[b] = 0;
      release_pending     = 0;
      cpc_decode_key(b, 1);
    }
  }

  return 0;
}
