#include <param.h>
#include <systm.h>
#include <conf.h>
#include <inode.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <buf.h>
#include <seg.h>
#include <errno.h>

extern struct user u;
extern char b[];

/*
 * Bmap defines the structure of file system storage
 * by returning the physical block number on a device given the
 * inode and the logical block number in a file.
 * When convenient, it also leaves the physical
 * block number of the next block of the file in rablock
 * for use in read-ahead.
 */
daddr_t
bmap(ip, bn, rwflg)
register struct inode *ip;
daddr_t bn;
{
	register i;
	struct buf *bp, *nbp;
	int j, sh;
	daddr_t nb, *bap;
	dev_t dev;

	if(bn < 0)
	{
		u.u_error = EFBIG;
		return((daddr_t)0);
	}
	dev = ip->i_dev;
	rablock = 0;

	/*
	 * blocks 0..NADDR-4 are direct blocks
	 */
	if(bn < NADDR-3)
	{
		i = bn;
		nb = getbno(ip, i);
		if(nb == 0)
		{
			if(rwflg==B_READ || (bp = alloc(dev))==NULL)
				return((daddr_t)-1);
			nb = bp->b_blkno;
			bdwrite(bp);
			setbno(ip, i, nb);
			ip->i_flag |= IUPD|ICHG;
		}
		if(i < NADDR-4)
			rablock = getbno(ip, i + 1);
		return(nb);
	}

	/*
	 * addresses NADDR-3, NADDR-2, and NADDR-1
	 * have single, double, triple indirect blocks.
	 * the first step is to determine
	 * how many levels of indirection.
	 */
	sh = 0;
	nb = 1;
	bn -= NADDR-3;
	for(j=3; j>0; j--)
	{
		sh += NSHIFT;
		nb <<= NSHIFT;
		if(bn < nb)
			break;
		bn -= nb;
	}
	if(j == 0)
	{
		u.u_error = EFBIG;
		return((daddr_t)0);
	}

	/*
	 * fetch the address from the inode
	 */
	nb = getbno(ip, NADDR - j);
	if(nb == 0)
	{
		if(rwflg==B_READ || (bp = alloc(dev))==NULL)
			return((daddr_t)-1);
		nb = bp->b_blkno;
		bdwrite(bp);
		setbno(ip, NADDR - j, nb);
		ip->i_flag |= IUPD|ICHG;
	}

	/*
	 * fetch through the indirect blocks
	 */
	for(; j<=3; j++)
	{
		bp = bread(dev, nb);
		if(bp->b_flags & B_ERROR)
		{
			brelse(bp);
			return((daddr_t)0);
		}
		ka5->r[0] = baddr(bp);
		bap = &b;
		sh -= NSHIFT;
		i = (bn>>sh) & NMASK;
		nb = bap[i];
		if(nb == 0)
		{
			if(rwflg==B_READ || (nbp = alloc(dev))==NULL)
			{
				brelse(bp);
				return((daddr_t)-1);
			}
			nb = nbp->b_blkno;
			bdwrite(nbp);
			bap[i] = nb;
			bdwrite(bp);
		} else
			brelse(bp);
	}

	/*
	 * calculate read-ahead.
	 */
	if(i < NINDIR-1)
		rablock = bap[i+1];
	return(nb);
}

/*
 * Pass back  c  to the user at his location u_base;
 * update u_base, u_count, and u_offset.  Return -1
 * on the last character of the user's read.
 * u_base is in the user address space unless u_segflg is set.
 */
passc(c)
register c;
{
	register id;

	if ((id = u.u_segflg) == KERND)
		*u.u_base = c;
	else if ((id == USERD) ? subyte(u.u_base, c) : suibyte(u.u_base, c) < 0)
	{
		u.u_error = EFAULT;
		return(-1);
	}
	u.u_count--;
	u.u_offset++;
	u.u_base++;
	return(u.u_count == 0 ? -1 : 0);
}

/*
 * Pick up and return the next character from the user's
 * write call at location u_base;
 * update u_base, u_count, and u_offset.  Return -1
 * when u_count is exhausted.  u_base is in the user's
 * address space unless u_segflg is set.
 */
cpass()
{
	register c, id;

	if (u.u_count == 0)
		return(-1);
	if ((id = u.u_segflg) == KERND)
		c = *u.u_base;
	else if ((c = id == USERD ? fubyte(u.u_base) : fuibyte(u.u_base)) < 0)
	{
		u.u_error = EFAULT;
		return(-1);
	}
	u.u_count--;
	u.u_offset++;
	u.u_base++;
	return(c & 0377);
}

/*
 * Routine which sets a user error; placed in
 * illegal entries in the bdevsw and cdevsw tables.
 */
nodev()
{

	u.u_error = ENODEV;
}

/*
 * Null routine; placed in insignificant entries
 * in the bdevsw and cdevsw tables.
 */
nulldev()
{
}

/*
 * copy count bytes from from to to.
 */
bcopy(from, to, count)
caddr_t from, to;
register count;
{
	register char *f, *t;

	f = from;
	t = to;
	do
		*t++ = *f++;
	while(--count);
}
