/*
 * DZ-11 driver
 */

#include "../h/param.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/tty.h"
#ifdef MONWORD
#include "../h/mon.h"
#endif

#define NDZLINE NDZ11*8
#define CARRIER 0       /* comment out if necessary */
/* #define BREAK        0       /* comment out if necessary */
/* #define SILO         /* enables dz input silo; should be off for
                           xon/xoff to work properly            */
#define SCANRATE        5
/* with SILO enabled you might prefer a SCANRATE of 2 for smooth echoing */
#define CARRATE         HZ/2
/* for auto-baud detection it should be short.  Original: HZ*2          */

/*
 * Hardware bits
 */
#ifdef SILO
#define CSRDFLT 050140
#else
#define CSRDFLT 040140
#endif SILO

#define PRTYERR 010000
#define FRAMERR 020000

#define _8BITS     030
#define _7BITS    0120
#define OPARITY   0200
#define LPRDFLT 010040

#define DTROFF       0
#define DTRON        1
#define BREAKOFF     0
#define BREAKON      1

#define SSPEED       7

struct device *dzaddr[] = {
        (struct device *)0170500,
        (struct device *)0170510,
        (struct device *)0170520,
        (struct device *)0170620,
        (struct device *)0170670,
};

struct tty dz11[NDZLINE];

char    dzstab[] = {
        0,      /* 0 baud */
        020,    /* 50 baud */
        021,    /* 75 baud */
        022,    /* 110 baud */
        023,    /* 134.5 baud */
        024,    /* 150 baud */
        0,      /* 200 baud - XXX */
        025,    /* 300 baud */
        026,    /* 600 baud */
        027,    /* 1200 baud */
        030,    /* 1800 baud */
        032,    /* 2400 baud */
        034,    /* 4800 baud */
        036,    /* 9600 baud */
        0,      /* EXTA */
        0,      /* EXTB */
        031,    /* 2000 baud */
        033,    /* 3600 baud */
        035,    /* 7200 baud */
        037     /* 19200 baud / maintenance mode */
};

struct device {
        int     dzcsr;
        int     dzrbuf; /* r/o */
        char    dztcr;  /* r/w; transmit enable */
        char    dzdtr;  /* r/w */
        char    dztbuf; /* also ring */
        char    dzbrk;  /* also carrier */
};

#define dzlpr   dzrbuf
#define dzmsr   dzbrk

#define SCANON      01
#define SCANLOCK    02

#ifdef DZPRESENT
char dzpresent[NDZ11];
#endif DZPRESENT

char dzto[NDZLINE];
#ifdef BREAK
char dzsbrk[NDZ11];     /* software copy of break register */
#endif BREAK
char dzstat;
/* int ndz11 = NDZLINE; */
char maptab[];

/*
 * open a DZ-11 line
 */
dzopen(dev, flag)
{
        register struct tty *tp;
        register line;
        extern dzstart();

        if(dzstat == 0) {
#ifdef DZPRESENT
                for(line = 0; line < NDZ11; line++)
                        if(feelwd(dzaddr[line]) != -1)
                                dzpresent[line]++;
#endif DZPRESENT
                dzscan();
                dzstat |= SCANON;
        }

#ifdef DZPRESENT
        if((line = minor(dev)) >= NDZLINE || dzpresent[line>>3] == 0){
#else
        if((line = minor(dev)) >= NDZLINE){
#endif DZPRESENT
                u.u_error = ENXIO;
                return;
        }
        tp = &dz11[line];
        if((tp->t_state&(ISOPEN|WOPEN)) == 0){
                tp->t_oproc = dzstart;
                tp->t_iproc = NULL;
                ttychars(tp);
                tp->t_ispeed = SSPEED;
                tp->t_ospeed = SSPEED;
                tp->t_flags = XTABS|CRMOD|ECHO|LCASE;
                dzparam(line);
        }
        else if((tp->t_state&XCLUDE) && u.u_uid != 0){
                u.u_error = EBUSY;
                return;
        }
        ttyname(dev, tp);
#ifdef CARRIER
        spl6();
        while((tp->t_state&(CARR_ON|DTRPAUSE)) != CARR_ON){
                if((tp->t_state & DTRPAUSE) == 0)
                        dzmodem(line, DTRON);
                tp->t_state |= WOPEN;
                sleep((caddr_t)&tp->t_rawq, TTIPRI);
        }
        dzmodem(line, DTRON);
        ttyopen(dev, tp);
        spl0();
#else
        dzmodem(line, DTRON);
        tp->t_state |= CARR_ON;
        ttyopen(dev, tp, flag);
#endif CARRIER
}

/*
 * close a DZ-11 line
 */
dzclose(dev)
{
        register struct tty *tp;
        register line;

        line = minor(dev);
        tp = &dz11[line];
        tp->t_pgrp = 0;
#ifdef BREAK
        dzbreak(line, BREAKOFF);
#endif
/*      dzaddr[line>>3]->dzlpr= line & 07;      /* clock off to stop noise */
        wflushtty(tp);
        if(tp->t_state&HUPCLS){
                dzmodem(line, DTROFF);
        }
        tp->t_state &= CARR_ON|DTRPAUSE;
        tp->t_xstate = 0;
}

/*
 * read from a DZ-11 line
 */
dzread(dev)
{
        ttread(&dz11[minor(dev)]);
}

/*
 * write on a DZ-11 line
 */
dzwrite(dev)
{
        ttwrite(&dz11[minor(dev)]);
}

/*
 * ioctl for DZ-11
 */
dzioctl(dev, cmd, addr, flag)
{
        register struct tty *tp;
        register line;

        line = minor(dev);
        tp = &dz11[line];
        if(ttioccomm(cmd, tp, (caddr_t)addr, line)){
                if(cmd == TIOCSETP || cmd == TIOCSETN || cmd == TIOCSETA)
                        dzparam(line);
        }
        else
                u.u_error = ENOTTY;
}

/*
 * set DZ-11 hardware registers
 */
dzparam(dev)
{
        register struct tty *tp;
        register struct device *dzp;
        register lpr;

        tp = &dz11[dev];
        dzp = dzaddr[dev>>3];
        dzp->dzcsr = CSRDFLT;
        if(tp->t_ispeed == 0){
                dzmodem(dev, DTROFF);
                return;
        }
        lpr = (dzstab[tp->t_ispeed] << 8)|LPRDFLT|(dev&07);
        if((tp->t_flags&(ODDP|EVENP)) == 0 || (tp->t_flags&RAW))
                lpr |= _8BITS;
        else {
                lpr |= _7BITS;
                if((tp->t_flags&EVENP) == 0)
                        lpr |= OPARITY;
        }
        dzp->dzlpr = lpr;
}

/*
 * DZ-11 receiver interrupt
 * Warning : also called from routine dzscan
 */
dzrint(dev)
{
        register struct tty *tp;
        register struct device *dzp;
        register c, i, n;

#ifdef MONWORD
        mn_monw |= MN_DZIN;
#endif
#ifdef SILO
        dzstat |= SCANLOCK;
#endif
        spl5();
        n = minor(dev);
        for(i = 0; i < NDZ11; i++){
#ifdef DZPRESENT
                if(!dzpresent[n])
                        goto next;
#endif DZPRESENT
                dzp = dzaddr[n];
                while((c = dzp->dzrbuf) < 0){
                        tp = &dz11[((c >> 8)&07)|(n << 3)];
                        if(tp >= &dz11[NDZLINE])
                                continue;
                        if((tp->t_state&ISOPEN) == 0){
                                wakeup((caddr_t)&tp->t_rawq);
                                continue;
                        }
                        if(c&FRAMERR){
                                if(tp->t_flags&RAW) {
                                        if(tp->t_flags & PFRAME)
                                                c |= 0200;
                                        else
                                                c = 0;
                                } else
                                        c = tp->t_un.t_intrc;
                        } else if(tp->t_flags & PFRAME)
                                c &= ~0200;
                        if(c&PRTYERR){
                                if((tp->t_flags&(EVENP|ODDP)) == EVENP || (tp->t_flags&(EVENP|ODDP)) == ODDP)
                                        continue;
                        }
                        ttyinput(c, tp);
                }
next:
                if(++n >= NDZ11)
                        n = 0;
        }
#ifdef SILO
        dzstat &= ~SCANLOCK;
#endif
#ifdef MONWORD
        mn_monw &= ~MN_DZIN;
#endif
}

/*
 * DZ-11 transmitter interrupt
 */
dzxint(dev)
{
        register struct tty *tp;
        register struct device *dzp;
        register i, n;

#ifdef MONWORD
        mn_monw |= MN_DZOUT;
#endif
        n = minor(dev);
        for(i = 0; i < NDZ11; i++){
#ifdef DZPRESENT
                if(!dzpresent[n])
                        goto next;
#endif DZPRESENT
                dzp = dzaddr[n];
                while(dzp->dzcsr < 0){
                        tp = &dz11[((n << 3)|(dzp->dzcsr >> 8)&07)];
                        dzp->dztbuf = tp->t_char;
                        tp->t_state &= ~BUSY;
                        dzstart(tp);
                }
next:
                if(++n >= NDZ11)
                        n = 0;
        }
#ifdef MONWORD
        mn_monw &= ~MN_DZOUT;
#endif
}

/*
 * start routine for DZ-11
 */
dzstart(tp)
register struct tty *tp;
{
        register struct device *dzp;
        register unit, c;
        int s;

        unit = tp - dz11;
        dzp = dzaddr[unit >> 3];
        unit = (1 << (unit&07));
        s = spl5();
        if((tp->t_state&(TIMEOUT|BUSY)) || (tp->t_xstate&XPAGE1)){
                splx(s);
                return;
        }
        if(tp->t_state&TTSTOP){
                dzp->dztcr &= ~unit;
                splx(s);
                return;
        }
        if((c = getc(&tp->t_outq)) >= 0){
                if(c >= 0200 && (tp->t_flags&RAW) == 0){
                        dzp->dztcr &= ~unit;
                        if(c == 0200){
                                tp->t_xstate |= XPAGE1;
                        } else {
                                tp->t_state |= TIMEOUT;
                                dzto[tp - dz11] = (((c&0177)+6+(SCANRATE-1))/SCANRATE)+1;
                        }
                } else {
                        dzp->dztcr |= unit;
                        if((tp->t_flags&RAW) == 0)
                                c |= (maptab[c]&0200);
                        tp->t_char= c;  
                        tp->t_state |= BUSY;
                }
                if((tp->t_outq.c_cc <= TTLOWAT) && (tp->t_state&ASLEEP)){
                        tp->t_state &= ~ASLEEP;
#ifdef MX
                        if(tp->t_chan)
                                mcstart(tp->t_chan, (caddr_t)&tp->t_outq);
                        else
#endif MX
                                wakeup((caddr_t)&tp->t_outq);
                }
        }
        else
                dzp->dztcr &= ~unit;
        splx(s);
}

/*
 * set/reset DTR
 */
dzmodem(line, flag)
{
        register struct device *dzp;
        register bit;

        dzp = dzaddr[line >> 3];
        bit = (1 << (line&07));
        if(flag == DTROFF)
                dzp->dzdtr &= ~bit;
        else
                dzp->dzdtr |= bit;
}

#ifdef BREAK
/*
 * turn break on and off
 */
dzbreak(line, flag)
{
        register bit;
        register char  *breakp;

        bit = (1 << (line&07));
        breakp = &dzsbrk[line>>3];
        if(flag == BREAKOFF)
                bit = *breakp & ~bit;
        else
                bit |= *breakp;
        *breakp = bit;
        dzaddr[line>>3]->dzbrk = bit;
}
#endif

/*
 * scan lines for input, and process any timeouts
 */
dzscan()
{
        register n;
#ifdef CARRIER
        static cscan;

        if(cscan <= 0){
                dzcarrier();
                cscan = CARRATE;
        }
        else
                cscan -= SCANRATE;
#endif CARRIER
        for(n = 0; n < NDZLINE; n++){
                if((dzto[n] > 0) && (--dzto[n] <= 0)){
                        dz11[n].t_state &= ~TIMEOUT;
                        dzstart(&dz11[n]);
                }
        }
#ifdef SILO
        if((dzstat&SCANLOCK) == 0)
                dzrint(0);
#endif
        timeout(dzscan, (caddr_t)0, SCANRATE);
}

#ifdef CARRIER
/*
 * scan DZ-11 lines for carrier transitions
 */
dzcarrier()
{
        register struct tty *tp;
        register struct device *dzp;
        register i;
        char bit;

        for(i = 0; i < NDZLINE; i++){
#ifdef DZPRESENT
                if(!dzpresent[i>>3])
                        continue;
#endif DZPRESENT
                dzp = dzaddr[i >> 3];
                tp = &dz11[i];
                bit = (1 << (i&07));
                if(tp->t_state & DTRPAUSE) {
                        tp->t_state &= ~DTRPAUSE;
                        wakeup((caddr_t)&tp->t_rawq);
                }
                if(dzp->dzmsr&bit){
                        if((tp->t_state&CARR_ON) == 0){
                                wakeup((caddr_t)&tp->t_rawq);
                                tp->t_state |= CARR_ON;
                        }
                } else {
                        if(tp->t_state&CARR_ON){
                                if(tp->t_state&ISOPEN){
                                        signal(tp->t_pgrp, SIGHUP);
/* dzmodem(.., DTROFF) */               dzp->dzdtr &= ~bit;
                                        tp->t_state |= DTRPAUSE;
                                        flushtty(tp);
                                }
                                tp->t_state &= ~CARR_ON;
                        }
                }
        }
}
#endif CARRIER
