#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/reg.h"
#include "../h/signal.h"
#include "../h/stats.h"
#include "../h/370.h"
#include "../h/bitmap.h"
#include "../h/mon.h"

/*
 * Low memory locations
 */
#define SVCCODE (*(int *)136)
#define PGMCODE (*(int *)140)
#define TRANADR (*(int *)144)
#define MONCLASS (*(short *)148)
#define PERCODE (*(short *)150)
#define PERADDR (*(int *)152)
#define MONCODE (*(int *)156)

#define FORKSVC 2           /* svc code for fork system call */
#define PPFINT  20          /* pseudo page fault interruption */
#define PAGADDR 0x00fff000  /* page number bits in address */
#define HIBIT   0x80000000  /* set when pseudo page fault is resolved */

#define TRAPCNT 100         /* check clock every TRAPCNT traps */

#define PGM_PER 0x80
#define PGM_MC  0x40
#define PGM_SEG 16
#define PGM_PAGE 17

/*
 * Mapping of program interrupts to signals.
 */
static int pgmtsig [] = {
	0,              SIGINS,         SIGPRIV,        SIGXEQ,
	SIGPROT,	0,		SIGSPEC,	SIGDATA,
	SIGFIXO,	SIGFIXD,	SIGDECO,	SIGDECD,
	SIGEXPO,	SIGEXPU,	SIGSGNF,	SIGFLTD,
	SIGSEG,		SIGPAGE,	0,		0,
};

/*
 * Names of program interrupts to be printed when kernel gets them.
 */
static char *pgmname[] = {
        0,
        "Operation exception",
        "Privileged operation exception",
        "Execute exception",
        "Protection exception",
        "Addressing exception",
        "Specification exception",
        "Data exception",
        "Fixed-point overflow exception",
        "Fixed-point divide exception",
        "Decimal overflow exception",
        "Decimal divide exception",
        "Exponent overflow exception",
        "Exponent underflow exception",
        "Significance exception",
        "Floating-point divide exception",
        "Segment translation exception",
        "Page translation exception",
        "Translation specification exception",
        "Special operation exception",
	"Pseudo page fault exception",
};

/*
 * Offsets of the user's registers relative to
 * the saved r0. See reg.h
 */
char    regloc[16] =
{
	R0, R1, R2, R3, R4, R5, R6, R7,
	R8, R9, R10, R11, R12, R13, R14, R15,
};

/*
 * Psuedo-Page Fault Priority
 */
#define PPPF    PZERO

/*
 * Types of traps
 */
#define SVC     1
#define PGM     2
#define IO      3
#define EXT     4
#define MCK     5
#define INIT    6
#define PANIC   7

/*
 * First level interrupt handler:
 * All interrupts come here for timer management
 * before calling appropriate interrupt routine.
 * After interrupt handling, more timer management
 * and scheduling is done.
 *
 * Warning: for the INIT interrupt (initialization),
 * everything in the user stucture is zero except u_procp (see zero.s)
 */
trap(type, r0)
{
	int *ar0;
	int user = 0;
	int trapcode;
	static int trapcnt;
	tod_t a_stck(), a_stckc();
	tod_t ck, ckc;
	struct proc *p;
	cpu_t t;
	cpu_t trapstart, trapend;
	cpu_t cpuswap();

	/*
	 * Stop charging for previous activity
	 */
	ar0 = &r0;
	if(ar0[RPS] & UMODE) {
                t = cpuswap(2, T_HIGH);
		u.u_ar0 = ar0;
		user = 1;
		u.u_utime += t;
		u.u_procp->p_time += t;
		stats.s_utime += t;
		stats.s_ttime += t;
                if ((u.u_procp->p_flag & STRC) && btst(u.u_sigtr, SIGSLICE))
			if ((u.u_ps.ps_tslice -= t) < 0)  /* timer must not go negative */
			        u.u_ps.ps_tslice = 0L;    /* CP bug */

	} else if(ar0[RPS] & WAIT) {
                t = cpuswap(1, T_HIGH);
		stats.s_wtime += t;
		stats.s_ttime += t;
	} else {
                t = cpuswap(0, T_HIGH);
		u.u_stime += t;
		u.u_procp->p_time += t;
		stats.s_stime += t;
		stats.s_ttime += t;
	}
	if (trapcnt++ > TRAPCNT) {
		if (time + 2 < ctime(ck = a_stck())) {
			ckc = a_stckc();
		        printf("Clock stopped. ck %x %x ckc %x %x\n", ck, ckc);
			tick();   /* get clock going again */
			t = cpuswap(0, T_HIGH);  /* charge for tick */
			stats.s_ttime += t;
			stats.s_exttime += t;
		}
		trapcnt = 0;
	}
	trapstart = u.u_stime;

	/* switch out to second level handler */
	switch(type) {

	case SVC:
		trapcode = SVCCODE & 0377;
		u.u_ar0 = ar0;
		svc();
		break;

	case PGM:
		trapcode = PGMCODE & 0xff;
		u.u_ar0 = ar0;
		pgm();
		break;

	case IO:
		iointr();
		break;

	case EXT:
		extrn(ar0);
		break;

	case MCK:
		mck();
		break;

	case INIT:
		main();
		break;

	case PANIC:
		panic("console panic");
		break;
	}

	/*
	 * Charge for time used during the interrupt.
	 */
	t = cpuswap(0, T_HIGH);
	stats.s_ttime += t;
        trapend = u.u_stime + t;
	switch(type) {

	case SVC:
                if (trapcode >= N_S_SVC)
			trapcode = N_S_SVC - 1;
		if (trapcode == FORKSVC && trapend < trapstart) { /* child */
			trapstart = 0;
			trapcode = 0;
		}
		stats.s_svc[trapcode].s_calls++;
		stats.s_svc[trapcode].s_cputime += (trapend - trapstart);
		u.u_stime += t;
		u.u_procp->p_time += t;
		stats.s_stime += t;
		break;

	case PGM:
		if (trapcode > N_S_PGM)
			switch (trapcode) {
			case 0x40:              /* monitor call */
				trapcode = 21;
				break;
			case 0x80:              /* per */
				trapcode = 22;
				break;
			default:                /* unknown */
				trapcode = 23;
				break;
			}
		stats.s_pgm[trapcode].s_calls++;
		stats.s_pgm[trapcode].s_cputime += (trapend - trapstart);
		u.u_stime += t;
		u.u_procp->p_time += t;
		stats.s_stime += t;
		break;

	case IO:
		stats.s_iotime += t;
		break;

	case EXT:
		stats.s_exttime += t;
		break;
	}

	/*
	 * If returning to user, process signals and reschedule cpu.
	 * Charge for system time, and start time slice for user.
	 */
	if(user) {
		u.u_ar0 = ar0;
		p = u.u_procp;
		if ((p->p_issig || u.u_issig) && issig())
			psig();
		setpri(p);
		while(runrun)
			qswtch();
		sureg();
                t = cpuswap(2, u.u_ps.ps_tslice);
		stats.s_btu.s_calls++;
		stats.s_btu.s_cputime += t;
                u.u_stime += t;
		u.u_procp->p_time += t;
		stats.s_stime += t;
		stats.s_ttime += t;
	}
	/* returns to interrupted program */
}

/*
 * Set new CPU timer value and return time passed since last set.
 * If we're running under VM with SEPP, use the VM accounting
 * interface when possible (not timing wait state).
 * Note that the do ... while loop determines the elapsed time
 * using no privileged operations.  However, IF THIS CODE IS TRACED
 * IT WILL NOT WORK!
 */
cpu_t cpuswap(mode, newcpu)
register int mode;
cpu_t newcpu;
{
	static cpu_t lastcpu = T_HIGH;
	cpu_t t, a_stpt();
	tod_t tod, nowtod, a_stck();
	static tod_t lasttod = 0;

	if(!(vm & SEPP)) mode = 1+2;
	if(mode & 1) {
                t = a_stpt();   /* store cpu timer */
                t = lastcpu - t;
	} else {
		do {
			tod = vmai.ai_toddsp;
			nowtod = vmai.ai_vtime + (a_stck() - tod);
			t = nowtod - lasttod;
		} while(tod != vmai.ai_toddsp);
		lasttod = nowtod;
	}
	if(mode & 2) {
                a_spt(newcpu);     /* set cpu timer */
                lastcpu = newcpu;
	}
	return(t);
}

/*
 * Process SVCs (system calls)
 */
svc()
{
        int i, *r1;
        struct sysent *callp;

        if((u.u_ar0[RPS] & UMODE) == 0)
                panic("svc");

        stats.s_svcs++;
	if((u.u_procp->p_flag & STRC) && btst(u.u_sigtr, SIGSVC)) {
		u.u_ps.ps_svc = SVCCODE;
		psignal(u.u_procp, SIGSVC);
		issig();        /* notifies tracer */
		if(!btst(u.u_sig, SIGSVC)) goto out; /* svc cancel by tracer */
		bclr(u.u_sig, SIGSVC);
		SVCCODE = u.u_ps.ps_svc;
	}
        u.u_ar0[RPS] &= ~COND;
        i = SVCCODE & 0377;
	if(i < nsysent) {
                callp = &sysent[i];
                r1 = (int *)u.u_ar0[R1];
                for(i = 0; i<callp->sy_narg; i++) {
                        u.u_arg[i] = fuword((caddr_t)(r1++));
                }
		u.u_r.r_val1 = u.u_ar0[R0];
		u.u_r.r_val2 = u.u_ar0[R1];
                u.u_dirp = (caddr_t)u.u_arg[0];
                u.u_error = 0;
                u.u_intflg = 1; /* for sleep */
		u.u_ap = u.u_arg;
                if (save(u.u_qsav)) {
                        if (u.u_error==0)
                                u.u_error = EINTR;
                } else {
                        (*callp->sy_call)();
                }
                u.u_intflg = 0;
	} else {
		nosys();
	}
        if(u.u_error) {
                u.u_ar0[RPS] |= COND;     /* set overflow cc */
                u.u_ar0[R0] = u.u_error;
                switch(u.u_error) {
                case ENOSPC:    i = SIGNOSPC; break;
                case ENFILE:    i = SIGNFILE; break;
                case EPIPE:     i = SIGPIPE; break;
                case EFAULT:    i = SIGSYS; break;
                default:        i = 0; break;
                }
                if(i) psignal(u.u_procp, i);
        } else {
		u.u_ar0[R0] = u.u_r.r_val1;
		u.u_ar0[R1] = u.u_r.r_val2;
	}

out:    ;
}

/*
 * Program interrupt handling.
 */
pgm()
{
        register int i, *ip;
	register int code;

	code = PGMCODE;

	/*
	 * Check for pseudo page fault completion -
	 * This is the only asynchronous program interrupt!
	 */
	if (((code & 037) == PPFINT) && (TRANADR & HIBIT)) {
		wakeup((caddr_t)(TRANADR & PAGADDR));
		return;
	}

	/*
	 * Check for kernel program interrupt - never allowed.
	 */
	ip = u.u_ar0;
        if((ip[RPS] & UMODE) == 0) {
                printf("Kernel error -- %s\n", pgmname[code & 037]);
                printf("PSW = %8x %8x\n", ip[RPS], ip[RPC]);
		if ((code & 037) == PPFINT)
			printf("TRANADR = %8x\n", TRANADR);
                for(i=0; i<16; i++) {
                        if(i <= 9) printf("  R%d %8x", i, ip[regloc[i]]);
                        else printf(" R%d %8x", i, ip[regloc[i]]);
                        if((i%4) == 3) printf("\n");
                }
                ipanic("trap");
        }

        stats.s_pgms++;
	u.u_ps.ps_pgm = code;
        i = code & 037;
	if (i == PPFINT) {
                stats.s_ppfs++;
                sleep((caddr_t)(TRANADR & PAGADDR), PPPF);
		return;
        }
        if(i == PGM_SEG || i == PGM_PAGE) {
                if(grow(u.u_ar0[R13])) goto out;
		u.u_ps.ps_trans = TRANADR;
	}
        if(code & PGM_PER) {
		u.u_ps.ps_per = PERCODE;
		u.u_ps.ps_perad = PERADDR;
                psignal(u.u_procp, SIGPER);
	}
        if(code & PGM_MC) {
		u.u_ps.ps_moncd = MONCODE;
		u.u_ps.ps_moncl = MONCLASS;
                psignal(u.u_procp, SIGMC);
	}
        i = pgmtsig[i];

	if(i) psignal(u.u_procp, i);
out:    ;
}

/*
 * nonexistent system call-- set fatal error code.
 */
nosys()
{
	u.u_error = EINVAL;
}

/*
 * Ignored system call
 */
nullsys()
{
}
