#
/*
 * RX Floppy Disk Driver
 *
 * By: SR Wilbur
 * On: 9-Jun-77
 */

#include "../param.h"
#include "../buf.h"
#include "../conf.h"
#include "../user.h"

/*
 * Device register structure
 */

struct 
{	int rxcs;			/* control and status register */
	char rxdb;			/* data register */
};

struct
{	char rxcslo;			/* low order csr */
	char rxcshi;			/* high order csr */
	char rxta;			/* track address */
};

struct devtab rxtab;			/* device table */
int rxtrack;				/* current sector and track no */
int rxsector;
int rxoffset;				/* offset within buffer for copying */
int rxtail 0;				/* mini fork return location */


#define NRXBLK	500
#define	RXADDR	0177170

#define	GO	01
#define	FILL	00
#define	EMPTY	02
#define	WSECT	04
#define	RSECT	06
#define	UNIT1	020
#define	DONE	040
#define	IENABLE	0100
#define	TR	0200
#define	INIT	040000

#define RXDST(P)	while(RXADDR->rxcslo >= 0); RXADDR->rxdb = P

/*
 * This driver has a number of limitations because of the hardware
 * involved.  'Rxstategy' should block any such illegal
 * tranfers.  The known limitations are:
 * 	- 11/70 is not supported
 * 	- transfers from buffers in user space are not allowed
 * 	- only tranfers of exact multiples of 0400 words are allowed
 */

rxstrategy(abp)
struct buf *abp;
{	register struct buf *bp;

	bp = abp;
	if ((bp->b_blkno >= NRXBLK) || (bp->b_flags & B_PHYS)
	   || (bp->b_xmem != 0) || ((bp->b_wcount & 0377) != 0))
	{	bp->b_flags =| B_ERROR;	/* addressing error */
		iodone(bp);
		return;
	}

	/* Since request is ok add buffer to i/o request list */

	bp->av_forw = 0;		/* end of list marker */
	spl5();				/* do not allow device i/u here */
	if (rxtab.d_actf == 0)
		rxtab.d_actf = bp;	/* empty list - add to head */
	else
		rxtab.d_actl->av_forw = bp;
	rxtab.d_actl = bp;		/* adjust tail pointer */
	spl0();
	if (rxtab.d_active == 0)
		rxstart();		/* initiate i/o if not active */
}

/*
 * 'Rxempty' removes 128 bytes from floppy disk controller buffer
 * into caller's buffer after a read operation
 */

rxempty(bp)
{	register char *p;		/* pointer to temp array of bytes */
	register *dv;			/* pointer to device registers */
	int i;

	dv = RXADDR;
l0:	dv->rxcs = EMPTY | GO;		/* empty buffer command - i/us off */
	p = bp->b_addr + rxoffset;
	for (i=0; ;i++)
	{
	l1:	if ((dv->rxcslo & DONE) != 0) break;
		if (dv->rxcslo >= 0) goto l1;
		*p++ =dv->rxdb;
	}

	if (dv->rxcs < 0 || i != 128)	/* error */
	{	if (++(rxtab.d_errcnt) > 10)
		{	bp->b_flags =| B_ERROR;
			return;
		}
		goto l0;
	}
}

/*
 * 'Rxfill' adds 128 bytes to the floppy disk controller buffer
 * from the caller's buffer before a write operation
 */

rxfill(bp)
struct buf *bp;
{	register char *p;		/* pointer to segment of buffer */
	register *dv;
	int i;

	dv = RXADDR;
l0:	dv->rxcs = FILL | GO;		/* fill buffer - no i/us */
	p = bp->b_addr + rxoffset;	/* relevant section of buffer */
	for (i=0;;i++)
	{
	l1:	if ((dv->rxcslo & DONE) != 0) break;
		if (dv->rxcslo >= 0) goto l1;
		dv->rxdb = *p++;
	}

	if (dv->rxcs < 0 || i != 128)	/* error */
	{	if (++(rxtab.d_errcnt) > 10)
		{	bp->b_flags =| B_ERROR;
			return;
		}
		goto l0;
	}
}

/*
 * 'Rxstart' checks to see if more buffers are queued for tranfer. If
 * so the initial track and sector numbers are computed and the first
 * tranfer is intiated
 */

rxstart()
{	register struct buf *bp;
	register int f;

	if ((bp = rxtab.d_actf) == 0)	/* nothing to do */
	{	rxtab.d_active = 0;	/* reset flag */
		return;
	}

	/* compute sector and track numbers */

	rxtab.d_active++;		/* set active marker */
	rxtab.d_errcnt = 0;		/* zero error count */
	f = bp->b_blkno * 12;
	rxtrack = (f+1)/78;		/* 3 sector interlace */
	rxsector = (f%26) +1;
	rxoffset = (bp->b_flags & B_READ)? -128:0; /* incremented before use (READ) */
	rxnxio();			/* called from fork level as well */
}

/*
 * 'Rxnxio' organises the actual transfers and may be called at
 * mainstream level (rxstrategy) or from the tail end fork of 'rxintr'
 */

rxnxio()
{	register int f;
	register struct buf *bp;

	bp = rxtab.d_actf;		/* buffer pointer */
	f = IENABLE | GO | (bp->b_dev.d_minor?UNIT1:0);
					/* basic command */
	if (bp->b_flags & B_READ)
	{	if (rxoffset >= 0)		/* read command */
			rxempty(bp);	/* not first time round */
		if (((bp->b_wcount + bp->b_wcount + rxoffset +128) == 0) ||
			(rxtab.d_errcnt >= 10)) goto done;
		RXADDR->rxcs = f + RSECT; 	/* start transfer */
	}
	else
	{	if ((bp->b_wcount + bp->b_wcount + rxoffset) == 0) goto done;
		rxfill(bp);
		if (rxtab.d_errcnt >= 10) goto done;
		RXADDR->rxcs = f + WSECT;
	}

	RXDST(rxsector);
	RXDST(rxtrack);
	return;

	/* When transfer is finally complete come here to unhook buffer
	 * from linked list and give signal to system
	 */

done:	rxtab.d_actf = bp->av_forw;		/* unhook buffer */
	iodone(bp);
	rxstart();				/* see if more to do */
}

/*
 * Interrupt service routine.  Basically staightforward except that
 * long operations are 'mini-forked' into, ie. they are executed
 * when no other outstanding interrupts exist but before user tasks are
 * rescheduled.  This process needs a minor modification to the 'call'
 * mechanism in m40.s
 */

rxintr()
{	register struct buf *bp;

	if (rxtab.d_active == 0) return;	/* unexpected i/u */

	bp = rxtab.d_actf;
	if (RXADDR->rxcs < 0)			/* check for errors */
	{	deverror(bp, RXADDR->rxdb, 0);	/* print diagnostic */
		RXADDR->rxcs = INIT;		/* try to correct */
		while ((RXADDR->rxcs & DONE) == 0); /* nasty - should be
							 mini-fork */
		if (++rxtab.d_errcnt <= 10)	/* try it again */
		{	rxfork(&rxnxio);
			return;
		}
		bp->b_flags =| B_ERROR;		/* set error flag */
		rxtab.d_actf = bp->av_forw;	/* unhook buffer */
		iodone(bp);
		rxfork(&rxstart);		/* try for another buffer */
		return;
	}

	/* no errors occurred so increment sector, track, and offset */

	rxoffset =+ 128;
	rxsector = (rxsector+2)%26 + 1;
	if (rxsector == 1) rxtrack++;
	rxfork(&rxnxio);			/* transfer next sector */
}

/*
 * 'Rxfork' is a KLUDGE.  It allows time consuming tasks to be serviced
 * at processor priority zero, but before normal scheduled tasks
 */

rxfork(sr)
int sr;
{	rxtail = sr;				/* when this is non zero it
						 * causes sr to be forked to */
}
