#

/*
 *************************************************************************************
 *                                                                                   *
 *                                 newuser                                           *
 *                                                                                   *
 *    Usage:    newuser {username} {groupname} {parent directory}                    *
 *                                                                                   *
 *   Establishes a new user in the system by                                         *
 *    (i)   creating a directory named  {parent directory}/{username} ;              *
 *    (ii)  updating the password file accordingly ;  and                            *
 *    (iii) updating the group file accordingly .                                    *
 *                                                                                   *
 *************************************************************************************
 */

#define namegroupfile    "/etc/group"
#define namepassfile     "/etc/passwd"
#define nametmpgroupfile "/tmp/group"
#define nametmppassfile  "/tmp/passwd"

#define maxusers 256
#define minusercode 10
#define bool int
#define false 0
#define true  1
#define eot '\0'
#define eol '\n'
#define empty "\0"
#define nil 0
#define maxlength 4000
#define input  0
#define output 1


struct groupprofile
 { char text[maxlength],
        *name,
        *pass,
        *code,
        *list;
 };
struct userprofile
 { char text[maxlength],
        *name,
        *pass,
        *code,
        *groupcode,
        *funny,
        *directory,
        *processor;
 };
int groupfile, passfile, tmpgroupfile, tmppassfile;

main (argcount, arg)
   int argcount;
   char *arg[];
 {
   struct userprofile newuser;
   int newposition;
   ensure(argcount==4,
          "Usage is:  ","newuser {username} {groupname} {parent directory}");
   startup();
   initnewuserprofile(arg[1],arg[3],&newuser);
   amendgroupfile(arg[2],&newuser);
   scanpassfile(arg[3],&newposition,&newuser);
   updatepassfile(&newuser,newposition);
   printf("    %s updated\n", namepassfile);
   updategroupfile();
   printf("    %s updated\n", namegroupfile);
   createdirectory(newuser.directory,newuser.name,arg[2]);
   printf("    New user %s established, in group %s,\n    with directory %s\n",
          newuser.name, arg[2], newuser.directory);
   windup();
 }

initnewuserprofile (name, parent, newuser)
   char *name, *parent;
   struct userprofile *newuser;
 {
   newuser->name = name;
   newuser->pass = empty;
   newuser->code = newuser->text;
   newuser->groupcode = newuser->text+4;
   newuser->funny = empty;
   newuser->directory = newuser->text+8;
   newuser->processor = empty;
   copy(parent,newuser->directory); append(newuser->directory,'/',name);
 }

createdirectory (directory, owner, groupname)
   char *directory, *owner, *groupname;
 {
   int inodebuffer[34];
   ensure(execute("/bin/mkdir",directory,nil) &&
          stat(directory,inodebuffer)>=0,
          "Cannot create directory ",directory);
   ensure(execute("/bin/chown",owner,directory),
          "Cannot change ownership of ",directory);
   ensure(execute("/bin/chgrp",groupname,directory),
          "Cannot change group of ",directory);
 }

bool execute (commandname, arg1, arg2)
   char *commandname, *arg1, *arg2;
 {
   int pid, newpid;
   pid = getpid();
   newpid = fork();
   if (pid!=getpid())
      if (execl(commandname,commandname,arg1,arg2,nil)<0)
         return(false);
   while (wait()!=newpid) ;
   return(true);
 }

amendgroupfile (groupname, newuser)
   char *groupname;
   struct userprofile *newuser;
 {
   struct groupprofile group;
   bool groupfound;
   int groupno;
   groupfound = false;
   while (readgroupprofile(groupfile,&group))
    { if (same(groupname,group.name))
       { groupno = internal(group.code);
         ensure(groupno>=0 && groupno<maxusers,
                "Invalid group number:  ",group.code);
         groupfound = true;
         copy(group.code,newuser->groupcode);
         if (same(group.list,empty))
            group.list = newuser->name;
         else
            append(group.list,',',newuser->name);
       } ;
      writegroupprofile(tmpgroupfile,&group);
    };
   ensure(groupfound,"Group name non-existent:  ",groupname);
 }

updategroupfile ()
 {
   char ch;
   rewrite(&groupfile,namegroupfile);
   reset(&tmpgroupfile,nametmpgroupfile);
   while ((ch = cgetc(tmpgroupfile)) != eot)
      cputc(ch,groupfile);
 }

scanpassfile (parent, newposition, newuser)
   char *parent;
   int *newposition;
   struct userprofile *newuser;
 {
   struct userprofile user;
   bool allocated[maxusers];
   int serial, position;
   for (serial = 0; serial<maxusers; serial++)
      allocated[serial] = false;
   *newposition = 1;
   for (position = 1; readuserprofile(passfile,&user); position++)
    { ensure(!same(newuser->name,user.name),"User name duplicated:  ",newuser->name);
      serial = internal(user.code);
      if (serial>=0 && serial<maxusers)
         allocated[serial] = true;
      else
       { printf("Invalid user number:-\n");
         writeuserprofile(output,&user);
       };
      if (isparent(parent,user.directory))
         *newposition = position;
      writeuserprofile(tmppassfile,&user);
    };
   for (serial = minusercode; serial<maxusers && allocated[serial]; serial++ );
   ensure(serial<maxusers,"All user numbers allocated",empty);
   decimalise(serial,newuser->code);
 }

updatepassfile (newuser, newposition)
   struct userprofile *newuser;
   int newposition;
 {
   struct userprofile user;
   int position;
   rewrite(&passfile,namepassfile);
   reset(&tmppassfile,nametmppassfile);
   for (position = 1; readuserprofile(tmppassfile,&user); position++)
    { if (position==newposition)
         writeuserprofile(passfile,newuser);
      writeuserprofile(passfile,&user);
    };
 }

bool readgroupprofile (file, group)
   int file;
   struct groupprofile *group;
{
   int fieldcount;
   if (readprofile(file,group->text,&fieldcount))
    { ensure(fieldcount==4,"Invalid group profile:-\n",group->text);
      group->name = group->text;
      separate(group->name,&group->pass);
      separate(group->pass,&group->code);
      separate(group->code,&group->list);
      return(true);
    };
   return(false);
 }

writegroupprofile (file, group)
   int file;
   struct groupprofile *group;
 {
   writestring(file,group->name,':');
   writestring(file,group->pass,':');
   writestring(file,group->code,':');
   writestring(file,group->list,eol);
 }

bool readuserprofile (file, user)
   int file;
   struct userprofile *user;
 {
   int fieldcount;
   if (readprofile(file,user->text,&fieldcount))
    { ensure(fieldcount==7,"Invalid user profile:-\n",user->text);
      user->name = user->text;
      separate(user->name,&user->pass);
      separate(user->pass,&user->code);
      separate(user->code,&user->groupcode);
      separate(user->groupcode,&user->funny);
      separate(user->funny,&user->directory);
      separate(user->directory,&user->processor);
      return(true);
    };
   return(false);
 }

writeuserprofile (file, user)
   int file;
   struct userprofile *user;
 {
   writestring(file,user->name,':');
   writestring(file,user->pass,':');
   writestring(file,user->code,':');
   writestring(file,user->groupcode,':');
   writestring(file,user->funny,':');
   writestring(file,user->directory,':');
   writestring(file,user->processor,eol);
 }

bool readprofile (file, buffer, fieldcount)
   int file;
   char *buffer;
   int *fieldcount;
 {
   char ch;
   *fieldcount = 1;
   while ((ch = cgetc(file)) != eol)
    { if (ch==eot)
         return(false);
      else
       { *buffer++ = ch;
         if (ch==':') (*fieldcount)++;
       };
    };
   *buffer = eot;
   return(true);
 }

bool isparent (dir1, dir2)
   char *dir1, *dir2;
 {
   while (*dir1 == *dir2)
    { if (*dir1==eot)
         return(false);
      dir1++; dir2++;
    };
   return(*dir1==eot && *dir2=='/');
 }

separate (oldfield, newfield)
   char *oldfield, **newfield;
 {
   while (*oldfield!=':') oldfield++;
   *oldfield = eot;
   *newfield = ++oldfield;
 }

writestring (file, string, terminator)
   int file;
   char *string, terminator;
 {
   while (*string!=eot)
      cputc(*string++,file);
   cputc(terminator,file);
 }



reset (file, filename)
   int *file;
   char *filename;
 {
   cclose(*file);
   *file = copen(filename,'r');
   ensure(*file>=0,"Cannot re-open ",filename);
 }

rewrite (file, filename)
   int *file;
   char *filename;
 {
   cclose(*file);
   *file = copen(filename,'w');
   ensure(*file>=0,"Cannot write to ",filename);
 }

append (string1, separator, string2)
   char *string1, *string2, separator;
 {
   while (*string1!=eot) string1++;
   *string1++ = separator;
   copy(string2,string1);
 }

copy (source, destination)
   char *source, *destination;
 {
   while ((*destination = *source) != eot)
    { destination++; source++; };
 }

bool same (string1, string2)
   char *string1, *string2;
 {
   while (*string1 == *string2)
    { if (*string1==eot)
         return(true);
      string1++; string2++;
    };
   return(false);
 }

int internal (code)
   char *code;
 {
   int val;
   char ch;
   val = 0;
   while ((ch = *code++) != eot)
    { if (ch<'0' || ch>'9')
         return(-1);
      val = 10*val + ch-'0';
    };
   return(val);
 }

decimalise (val, code)
   int val;
   char *code;
 {
   char c[12], *cp;
   cp = &c[11];
   *cp = eot;
   do
    { cp--;
      *cp = val%10 + '0';
      val = val/10;
    } while (val>0);
   copy(cp,code);
 }

ensure (satisfactory, message, parameter)
   bool satisfactory;
   char *message, *parameter;
 {
   if (!satisfactory)
    { printf("**** %s%s\n**** New user not established\n", message,parameter);
      windup();
    };
 }

startup ()
 {
   groupfile = copen(namegroupfile,'r');
   ensure(groupfile>=0,"Cannot open ",namegroupfile);
   passfile = copen(namepassfile,'r');
   ensure(passfile>=0,"Cannot open ",namepassfile);
   tmpgroupfile = copen(nametmpgroupfile,'w');
   ensure(tmpgroupfile>=0,"Cannot create ",nametmpgroupfile);
   tmppassfile = copen(nametmppassfile,'w');
   ensure(tmppassfile>=0,"Cannot create ",nametmppassfile);
 }

windup ()
 {
   unlink(nametmpgroupfile);
   unlink(nametmppassfile);
   cexit();
 }
