/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)R4state.c	1.7 89/01/11
*/

/*
**	Read a SUN IV state file and structure it in memory.
*/

#define	STDIO

#include	"global.h"
#include	"address.h"
#include	"debug.h"
#include	"state.h"
#include	"node.h"
#include	"statefile.h"

#define	TOKEN_DATA
#include	"sun4state.h"

#include	<setjmp.h>


#ifndef	DEFUNCT_TIME
#define	DEFUNCT_TIME	(60*60*24*7L)	/* Statefiles time out after 1 week */
#endif

extern Address *AddressP;
extern Entry *	RSource;
extern char *	RSrcAddr;
extern Time_t	RSrcDate;

static char *	Buf;
static char *	BufEnd;
static bool	IgnoreCRC;
static int	Last_C;
static FILE *	RStateFd;
static Crc_t	StCRC;
static int	TokenLength;
static int	TokenCount;

extern jmp_buf	NameErrJmp;
extern char *	HomeNode;

extern Token	Rcheck();
extern void	SetTop();
extern void	R4setFlags();

static char *	Conv43Addr();
static void	FixComment();
static Token4	Rtoken();
static void	DomsSet();
static void	HierSet();



void
R4state(fd, foreign, sourceaddr, sourcedate, ignoreCRC)
	FILE *		fd;
	bool		foreign;
	char *		sourceaddr;
	Time_t		sourcedate;
	bool		ignoreCRC;
{
	register Node *	np;
	register Data *	dp;
	register Node *	op;
	register Entry *nep;
	register Entry *oep;
	register char *	buf;
	auto Token4	token;
	auto Token4	last_token;
	char *		hierarchy;
	char *		errs1;
	char *		errs2;
	Data		null_data;
	char		temp[TOKEN_4_SIZE+2];

	Trace4(1, "R4state %d %s %s", fileno(fd), foreign?"true":"false", sourceaddr==NULLSTR?"":sourceaddr);

	if ( TokenCount = setjmp(NameErrJmp) )
	{
		Error("Bad home nodename \"%s\" at char %d", HomeNode, TokenCount-1);
		return;
	}

	if ( Home == (Entry *)0 )
	{
		Home = Enter(HomeNode, NodeHash);

		if ( !(Home->e_states & S_FOUND) )
		{
			Home->e_states |= S_FOUND;
			NodeCount++;
			ClearDomains(Home, true);
		}
	}

	if ( foreign )
	{
		RSrcAddr = sourceaddr;
		RSrcDate = sourcedate;

		AddressP = SplitAddress(sourceaddr);

		SetHierarchy(NULLSTR, &ForeignHier);
	}
	else
		SetHierarchy(NULLSTR, &DomHier);

	Buf = temp;
	BufEnd = &temp[TOKEN_4_SIZE+1];
	Last_C = EOF;
	IgnoreCRC = ignoreCRC;
	RStateFd = fd;
	RSource = (Entry *)0;
	StCRC = 0;
	TokenCount = 0;

	token = t4_start;
	last_token = t4_start;

	np = (Node *)0;
	oep = (Entry *)0;

	if ( TokenCount = setjmp(NameErrJmp) )
	{
		errs1 = "Bad node name in %s SUN IV statefile: \"%s\" (char %d)";
		errs2 = foreign?"foreign":"home";

		if ( ignoreCRC )
		{
			Warn(errs1, errs2, &temp[1], TokenCount-1);
			token = last_token;
		}
		else
		{
			Error(errs1, errs2, &temp[1], TokenCount-1);
			return;
		}
	}

	buf = &temp[1];

	for ( ;; )
	{
		switch ( token = Rtoken(Accept4[(int)token]) )
		{
		case t4_address:
			if ( (hierarchy = Conv43Addr(buf)) != NULLSTR )
			{
				HierSet(hierarchy, foreign);
				goto node;
			}
			Error("null %sSUN IV statefile \"address\"", foreign?"foreign ":"");
			AnteState = true;
			return;

		case t4_region:
			last_token = t4_nextregion;

			if ( (hierarchy = Conv43Addr(buf)) == NULLSTR )
			{
				token = last_token;
				continue;
			}
node:
			nep = Enter(buf, NodeHash);
			np = nep->e_node;
			oep = (Entry *)0;

			Trace2(2, "NODE \"%s\"", nep->e_name);

			if ( foreign )
			{
				register Token	token3;

				if ( (token3 = Rcheck(nep)) == t_eof )
				{
					AnteState = true;
					return;
				}
				if ( token3 != t_node )
					continue;	/* Don't mark it FOUND */
			}

			last_token = token;

			if ( !(nep->e_states & S_FOUND) )
			{
				NodeCount++;
				nep->e_states |= S_FOUND;
			}

			if ( np->n_hierarchy != NULLSTR )
				free(np->n_hierarchy);
			if ( hierarchy != NULLSTR )
			{
				np->n_hierarchy = newstr(hierarchy);
				SetTop(np);
			}
			else
				np->n_hierarchy = NULLSTR;

			continue;

		case t4_visible:
			DomsSet(np, buf, (bool)(nep==Home));
			continue;

		case t4_link:
			last_token = t4_nextlink;
			dp = &null_data;

			if ( (hierarchy = Conv43Addr(buf)) == NULLSTR )
			{
				token = last_token;
				continue;
			}

			oep = Enter(buf, NodeHash);
			op = oep->e_node;

			Trace2(2, "Link \"%s\"", oep->e_name);

			if ( foreign && oep == Home )
				continue;

			if ( op == np )
			{
				errs1 = "Link to self for";
				errs2 = oep->e_name;
				break;
			}

			last_token = token;

			dp = MakeLink(nep, oep);
			dp->d_states |= S_MSG;

			if ( foreign && nep == RSource )
			{
				if ( !(oep->e_states & S_FOUND) )
				{
					NodeCount++;
					oep->e_states |= S_FOUND;
				}

				(void)MakeLink(oep, nep);

				dp->d_states &= ~FOREIGN_FLAGS;
				dp->d_cost = 0;
				dp->d_speed = 0;
			}

			if ( foreign )
			{
				/*
				**	Link hierarchy
				*/

				if
				(
					op->n_hierarchy != NULLSTR
					&&
					strccmp(hierarchy, op->n_hierarchy) != STREQUAL
				)
				{
					register Link **lpp;

					/*
					**	This node thinks that this link
					**	has a different hierarchy
					**	==> different link?
					*/

					if ( (lpp = IsLinked(nep, oep)) != (Link **)0 )
					{
						if
						(
							Links(oep) == 1
							||
							(
								op->n_domains.d_count
								==
								np->n_domains.d_count
								&&
								(
									op->n_domains.d_count == 0
									||
									op->n_domains.d_head->d_entry
									==
									np->n_domains.d_head->d_entry
								)
								&&
								(RSrcDate - op->n_state) > DEFUNCT_TIME
							)
						)
						{
							Warn
							(
								"Changed hierarchy for node \"%s\" linked to \"%s\" from \"%s\" to \"%s\"",
								oep->e_name,
								nep->e_name,
								op->n_hierarchy,
								hierarchy
							);
							free(op->n_hierarchy);
							op->n_hierarchy = NULLSTR;

						}
						else
						{
							dp = &null_data;
							Unlink(nep, lpp);
							Warn
							(
								"Broke putative link between \"%s.%s\" and \"%s.%s\" -- node of same name already exists called \"%s.%s\"",
								nep->e_name, np->n_hierarchy,
								oep->e_name, hierarchy,
								oep->e_name, op->n_hierarchy
							);
						}
					}
				}
				if ( op->n_hierarchy == NULLSTR )
					op->n_hierarchy = newstr(hierarchy);
				SetTop(op);	/* Last domain is top level */
			}

			continue;

		case t4_xalias:
			(void)Conv43Addr(buf);
			if ( buf[0] == '\0' )
				continue;
			nep = Enter(buf, AliasHash);
			if ( foreign )
			{
				if
				(
					(nep->e_states & S_FOUND) 
					||
					Lookup(buf, NodeHash) != (Entry *)0
					||
					Lookup(buf, DomainHash) != (Entry *)0
				)
				{
					if
					(
						nep->e_value == NULLSTR
						||
						strcmp(nep->e_value, RSrcAddr) != STREQUAL
					)
						Warn
						(
							"Rejected export alias \"%s\" for \"%s\" -- already exists",
							buf, RSrcAddr
						);

					continue;	/* Reject if already exists */
				}
				nep->e_value = newstr(RSrcAddr);
				AliasCount++;
				nep->e_states |= S_FOUND;
			}
			else
			{
				if ( !(nep->e_states & S_FOUND) )
				{
					nep->e_value = newstr(Home->e_name);
					AliasCount++;
					nep->e_states |= S_FOUND;
				}
				else
				if ( strcmp(nep->e_value, Home->e_name) != STREQUAL )
				{
					errs1 = "Bad export alias name";
					errs2 = buf;
					break;
				}
				nep->e_next = XAliases;
				XAliases = nep;
			}
			continue;

		case t4_cost:
			if ( foreign && nep != RSource )
				continue;
			dp->d_cost = (ulong)DecodNum(buf, TokenLength);
			continue;

		case t4_delay:
			if ( foreign && nep != RSource )
				continue;
			if ( DecodNum(buf, TokenLength) >= 3600 )
				dp->d_states |= S_INTERMITTENT;
			continue;

		case t4_restrict:
			if ( foreign && nep != RSource )
				continue;
			if ( buf[0] != '\0' )
			{
				dp->d_cost = 65535;	/* Make it expensive */
				dp->d_speed = 1;	/* Make it slow */
			}
			else
			if ( dp->d_cost == 65535 && dp->d_speed == 1 )
			{
				dp->d_cost = 0;
				dp->d_speed = 0;
			}
			continue;

		case t4_speed:
			if ( foreign && nep != RSource )
				continue;
			dp->d_speed = (ulong)DecodNum(buf, TokenLength);
			continue;

		case t4_traffic:
			continue;

		case t4_lflags:
			if ( foreign && nep != RSource )
				continue;
			R4setFlags(&dp->d_states, buf, foreign);
			dp->d_states |= S_MSG;
			continue;

		case t4_comment:
#			if	ALL_COMMENTS != 1
			if ( nep != Home )
				continue;
#			endif	ALL_COMMENTS != 1
			FixComment(buf);
			if ( np->n_comment != NULLSTR )
				free(np->n_comment);
			np->n_comment = newstr(buf);
			continue;

		case t4_eof:
			goto out;

		case t4_error:
			errs1 = "Illegal statefile token";
			errs2 = buf;
			break;

		case t4_expect:
			errs1 = "Expected field missing";
			errs2 = "";
			break;

		default:
			Warn("Unknown token [%d] in SUN IV statefile", (int)token);
			token = last_token;
			continue;
		}

		if ( foreign )
		{
			Error("%s \"%s\"", errs1, errs2);
			AnteState = true;
		}
		else
			Warn("%s \"%s\" ignored", errs1, errs2);

		token = last_token;
	}

out:
	if
	(
		!ignoreCRC
		&&
		TokenCount > 0
		&&
		(
			getc(fd) != LOCRC(StCRC)
			||
			getc(fd) != HICRC(StCRC)
		)
	)
	{
		Error("bad %sSUN IV statefile CRC", foreign?"foreign ":"");
		AnteState = true;
		return;
	}

	if ( foreign )
	{
		FreeAddress(AddressP);

		if ( RSource == (Entry *)0 )
		{
			AnteState = true;
			return;
		}

		if ( sourcedate > 0 )
			RSource->e_node->n_state = sourcedate;

		CheckHierarchy(RSource, &ForeignHier);
		CheckSourceDomains(RSource);
	}
	else
	{
		register char *	cp;

		if ( (cp = Home->e_node->n_hierarchy) != NULLSTR )
			free(cp);
		Home->e_node->n_hierarchy = DomString(&DomHier, DOMAIN_SEP);
	}
}



/*
**	Read a separator followed by a token (name or number)
**	and return token type.
*/

static Token4
Rtoken(accept)
	Token4s		accept;
{
	register char *	s = Buf;
	register int	c;
	register FILE *	fd = RStateFd;
	register char *	ep = BufEnd;
	register Token4	token = t4_null;

	Trace2(4, "Rtoken accept %#lx", accept);

	if ( (c = Last_C) != EOF )
	{
		Last_C = EOF;
		goto next;
	}

	for ( ;; )
	{
		if ( (c = getc(fd)) == EOF )
		{
			Trace1(1, "Rtoken at EOF!");
			if ( token == t4_null )
				return t4_eof;
			goto eof;
		}

next:		*s++ = (char)c;

		if ( c > 0177 )
		{
			if ( (c &= 0177) > 037 )
			{
				TokenLength = (s - Buf) - 1;
				return t4_error;
			}

			if ( token == t4_null )
			{
				if ( !(accept & (1L<<c)) )
				{
					if ( accept & T_EXPECT )
					{
						token = t4_expect;
						goto backout;
					}
					if ( c == (int)t4_eof )
					{
						token = (Token4)c;
						goto backout;
					}
					token = t4_null;
				}
				else
					token = (Token4)c;

				if ( s != &Buf[1] )
				{
					if ( !IgnoreCRC )
						StCRC = acrc(StCRC, Buf, --s - Buf);
					Trace3
					(
						3,
						"Ignore token \"%s\": \"%s\"",
						Token4Names[Buf[0]&037],
						ExpandString(Buf, s-Buf)
					);
					Buf[0] = *s;
					s = &Buf[1];
				}

				TokenCount++;
				continue;
			}
			else
			{
backout:			Last_C = *(unsigned char *)--s;
eof:				if ( (c = s - Buf) > 0 )
				{
					if ( !IgnoreCRC )
						StCRC = acrc(StCRC, Buf, c);
					TokenLength = --c;
				}
				else
					TokenLength = 0;
				*s = '\0';
				Trace3
				(
					4,
					"Rtoken \"%s\": \"%s\"",
					Token4Names[(int)token],
					ExpandString(Buf, TokenLength+1)
				);
				return token;
			}
		}

		if ( s > ep )
		{
			TokenLength = (s - Buf) - 1;
			return t4_error;
		}
	}
}



/*
**	Strip types from "string" and remove special domain.
*/

static char *
Conv43Addr(string)
	char *		string;
{
	register int	c;
	register char *	sp;
	register char *	cp;
	register char *	tp;
	register char *	hp;

	Trace2(3, "Conv43Addr(%s)", string);

	cp = tp = sp = string;
	hp = NULLSTR;

	if ( *cp != '9' )
	{
		*cp = '\0';	/* Ignore non-nodes */
		return hp;
	}

	do
		if ( (c = *sp++) == TYPE_SEP )
			cp = tp;
		else
		if ( (*cp++ = c) == DOMAIN_SEP )
		{
			if ( hp == NULLSTR )
				hp = cp;
			tp = cp;
		}
	while
		( c != '\0' );

	/** "tp" points at last domain **/

	if ( strccmp(tp, OZCC) == STREQUAL )
	{
		if ( tp == hp )
			hp = NULLSTR;
		if ( tp == string )
			tp++;
		*--tp = '\0';
	}

	Trace2(4, "Conv43Addr() ==> %s", string);

	if ( hp != NULLSTR )
		hp[-1] = '\0';	/* "buf" contains node name */

	return hp;		/* Return hierarchy */
}



/*
**	Make comment conform.
*/

static void
FixComment(buf)
	char *		buf;
{
	register char *	cp;
	register char *	np;
	register bool	again;

	for ( cp = buf ; (np = MatchString(cp, ":\t")) != NULLSTR ; )
	{
		np += 2;
		(void)strcpy(cp, np);
		do
		{
			if ( (cp = strchr(cp, '\n')) == NULLSTR )
				goto out;
			if ( cp[1] == '\0' )
				goto out;
			*cp++ = ',';
			if ( cp[1] == '\0' )
				goto out;
			if ( *cp == '\t' )
				again = true;
			else
				again = false;
			*cp++ = ' ';
		}
		while
			( again );
	}
out:
	buf[TOKEN_SIZE] = '\0';

	if ( (cp = strrchr(buf, ',')) != NULLSTR )
		*cp = '\0';

	for ( cp = buf ; (np = strchr(cp, '\t')) != NULLSTR ; cp = np )
		*np++ = ' ';
}



/*
**	Decode state flags from string.
*/

static void
R4setFlags(sp, cp, foreign)
	register States *	sp;
	register char *		cp;
	bool			foreign;
{
	register int		c;
	register States		flags = 0;
	register States		accept;

	Trace2(3, "R4setFlags(%s)", cp);

	while ( c = *cp++ )
	{
		switch ( c - 'A' )
		{
		case fl4_dead:		flags |= S_DEAD;		break;
		case fl4_down:		flags |= S_DOWN;		break;
		case fl4_filtered:	flags |= S_FILTERED;		break;
		case fl4_foreign:	flags |= S_FOREIGN;		break;
		}

		Trace2(4, "R4setFlag %s", Flag4Names[c-'A']);
	}

	if ( foreign )
	{
		accept = FOREIGN_FLAGS;

		if ( *sp & S_LAN )
			accept &= ~S_DOWN;
	}
	else
		accept = EXTERNAL_FLAGS;

	*sp &= ~accept;
	accept |= S_DOWN;
	*sp |= flags & accept;
}



/*
**	Set domains for a node up to and including visibility.
*/

static void
DomsSet(np, buf, home)
	register Node *	np;
	char *		buf;
	bool		home;
{
	register char *	hp;
	register char *	dp;
	register Entry *dep;
	register char *	vp;

	Trace4(2, "DomsSet(%s, %s%s)", np->n_hierarchy, buf, home?", home":"");

	buf[0] = '9';	/* Otherwise Conv43Addr() will reject it */

	if ( (hp = Conv43Addr(buf)) != NULLSTR )
		*--hp = DOMAIN_SEP;
	else
	if ( buf[0] == '\0' )
		return;

	Trace3(3, "DomsSet() match strings %s %s", np->n_hierarchy, buf);

	if ( (vp = MatchString(np->n_hierarchy, buf)) == NULLSTR )
	{
		/** Just set visible **/
		if ( hp != NULLSTR )
			*hp = '\0';
		else
		if ( buf[0] == '\0' )
			return;
		dep = AddDomain(buf, &np->n_domains, S_FOUND);
		if ( home )
			dep->e_states |= S_OURDOM;
		else
			dep->e_states &= ~S_OTHDOM;
		return;
	}

	if ( hp != NULLSTR )
		*hp = '\0';

	/** Set primary == visible **/

	dep = AddDomain(buf, &np->n_domains, S_FOUND);
	if ( home )
		dep->e_states |= S_OURDOM;
	else
		dep->e_states &= ~S_OTHDOM;

	/** Set remaining sub-domains **/

	hp = np->n_hierarchy;

	while ( hp < vp && (dp = strchr(hp, DOMAIN_SEP)) != NULLSTR )
	{
		*dp = '\0';
		dep = AddDomain(hp, &np->n_domains, S_FOUND);
		if ( home )
			dep->e_states |= S_OURDOM;
		else
			dep->e_states &= ~S_OTHDOM;
		*dp++ = DOMAIN_SEP;
		hp = dp;
	}
}



static void
HierSet(hier, foreign)
	char *		hier;
	bool		foreign;
{
	register char *	cp;
	register char *	np;
	DomHead *	dhp;
	register States	states;

	Trace3(1, "HierSet(%s, %s)", hier, foreign?"foreign":"home");

	if ( foreign )
	{
		dhp = &ForeignHier;
		states = 0;
	}
	else
	{
		dhp = &DomHier;
		states = S_HIERDOM;
	}

	for ( cp = hier ; ; )
	{
		if ( (np = strchr(cp, DOMAIN_SEP)) != NULLSTR )
			*np = '\0';

		(void)AddDomain(cp, dhp, states);

		if ( np == NULLSTR )
			break;

		*np++ = DOMAIN_SEP;
		cp = np;
	}
}
