/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)route.h	1.7 86/01/05
*/

/*
**	Information on routing tables.
**
**	Routing tables are kept in a binary file with the following format:
**
**	Index		nodes;			Number of nodes.
**	Index		links;			Number of links to home.
**	Index		aliases;		Number of aliases for nodes.
**	Index		domains;		Number of domains.
**	Index		levels;			Number of levels in hierarchy.
**	States		homeflags;		Flags for this node.
**	LinkEntry	linktable[links];	Description of links.
**	NodeEntry	nodetable[nodes];	Sorted list of nodes with routing information.
**	AliasEntry	aliastable[aliases];	Sorted list of aliases with true ids.
**	DomainEntry	domaintable[domains];	Sorted list of domains with routing information.
**	Index		level[levels];		Index of each successive domain in hierarchy.
**	bit		forwtable[nodes*links];	Boolean array for broadcast forwarding.
**	bit		members[domains*nodes];	Boolean array for domain membership.
**	bit		domforw[domains*links];	Boolean array for all routes to all domains.
**	char		strings[];		Names of handlers and values for aliases.
*/

typedef short		Index;

/*
**	A 'LinkEntry' contains the index to the destination node in the
**	'node_table', and indices for the names of the handlers for the
**	link in the 'strings' area.
*/

typedef struct
{
	Index	le_index;			/* Index of destination NodeEntry */
	short	le_handlers[PT_N];		/* Index of handler names in strings area */
	short	le_caller;			/* Index of caller name in strings area */
	short	le_filter;			/* Index of filter name in strings area */
	States	le_flags;			/* Flags for link and node */
}
			LinkEntry;

#define	LINK_ENTRY_SIZE	sizeof(LinkEntry)

/*
**	A 'NodeEntry' contains the name of the node, indices
**	for the 'LinkEntry's on the shortest paths for each protocol type,
**	and an index for the primary 'DomainEntry'.
**
**	NodeEntrys are stored in the 'node_table' sorted by node name,
**	thus nodes may be located by binary search in the 'node_table'.
*/

#define	MAX_LINKS	0x7ffe			/* Checked in WriteRoute.c */
#define	LINK_N_A	0x7fff			/* No Way! (and avoid -ve) */

typedef struct
{
	Index	ne_primary;			/* Index of primary domain */
	Index	ne_shortest[PT_N];		/* Index of LinkEntry on shortest paths */
	char	ne_name[NODE_NAME_SIZE+1];	/* Name of node */
}
			NodeEntry;

#define	NODE_ENTRY_SIZE	sizeof(NodeEntry)

/*
**	An 'AliasEntry' contains an alias for an address,
**	and an index for the address in the 'strings' area.
*/

typedef struct
{
	short	ae_value;			/* Index to true name in strings area */
	char	ae_name[NODE_NAME_SIZE+1];	/* An alias */
}
			AliasEntry;

#define	ALIAS_ENTRY_SIZE	sizeof(AliasEntry)

/*
**	A 'DomainEntry' contains the name of a domain,
**	an index for a handler in the 'strings' area,
**	and an index of the node on the shortest path.
*/

typedef struct
{
	Index	de_shortest;			/* Index of 'LinkEntry' on shortest path */
	short	de_hierarchy;			/* Position if in hierarchy */
	short	de_handler;			/* Index to handler name in strings area */
	char	de_name[NODE_NAME_SIZE+1];	/* A domain */
}
			DomainEntry;

#define	DOMAIN_ENTRY_SIZE	sizeof(DomainEntry)

/*
**	The hierarchy levels are stored as an array of Indices.
*/

#define	LEVEL_ENTRY_SIZE	sizeof(Index)

/*
**	The broadcast table for each link is stored as an array
**	of booleans in an vector of bytes, 8 booleans per byte.
*/

#define	SIZE_FORW_TABLE		(((NodeCount*LinkCount)+7)/8)

/*
**	The domain membership table for each node is stored as an array
**	of booleans in a vector of bytes, 8 booleans per byte.
*/

#define	SIZE_MEMBER_TABLE	(((NodeCount*DomainCount)+7)/8)

/*
**	The domain routing table for each link is stored as an array
**	of booleans in a vector of bytes, 8 booleans per byte.
*/

#define	SIZE_DOMFORW_TABLE	(((DomainCount*LinkCount)+7)/8)

/*
**	Other variables:
*/

Extern int		AliasCount;		/* Count of aliases */
Extern char *		AliasTable;		/* Pointer to 'aliastable' */
Extern char *		DomForwTable;		/* Pointer to "domforwtable" */
Extern int		DomainCount;		/* Count of domains */
Extern char *		DomainTable;		/* Pointer to 'domaintable' */
Extern char *		ForwTable;		/* Pointer to 'forwtable' */
Extern int		LinkCount;		/* Count of links from 'home' */
Extern char *		LinkTable;		/* Pointer to 'linktable' */
Extern int		LevelCount;		/* Count of levels in hierarchy */
Extern char *		LevelTable;		/* Table of hierarchy members */
Extern int		NodeCount;		/* Count of nodes in NodeTable */
Extern char *		NodeTable;		/* Pointer to 'nodetable' */
Extern char *		MemberTable;		/* Pointer to 'membertable' */
Extern char *		RouteBase;		/* Base of routing tables in memory */
Extern char *		RouteFile;		/* File name for routing tables */
Extern char *		Strings;		/* Pointer to 'strings' */

/*
**	Macros:
*/

#define	ROUTE_HEADER_SIZE	((sizeof(Index)*5)+sizeof(States))
#define	NODE_COUNT		((Index *)RouteBase)[0]
#define	LINK_COUNT		((Index *)RouteBase)[1]
#define	ALIAS_COUNT		((Index *)RouteBase)[2]
#define	DOMAIN_COUNT		((Index *)RouteBase)[3]
#define	LEVEL_COUNT		((Index *)RouteBase)[4]
#define	HOME_FLAGS		(*(States *)(&((Index *)RouteBase)[5]))

#define	LINK_TABLE_ADDR		&RouteBase[ROUTE_HEADER_SIZE]
#define	NODE_TABLE_ADDR		&LinkTable[LINK_ENTRY_SIZE*LinkCount]
#define	ALIAS_TABLE_ADDR	&NodeTable[NODE_ENTRY_SIZE*NodeCount]
#define	DOMAIN_TABLE_ADDR	&AliasTable[ALIAS_ENTRY_SIZE*AliasCount]
#define	LEVEL_TABLE_ADDR	&DomainTable[DOMAIN_ENTRY_SIZE*DomainCount]
#define	FORW_TABLE_ADDR		&LevelTable[LEVEL_ENTRY_SIZE*LevelCount]
#define	MEMBER_TABLE_ADDR	&ForwTable[SIZE_FORW_TABLE]
#define	DOMFORW_TABLE_ADDR	&MemberTable[SIZE_MEMBER_TABLE]
#define	STRINGS_ADDR		&DomForwTable[SIZE_DOMFORW_TABLE]

#define	RT_LINK(A)		((LinkEntry *)&LinkTable[(A)*LINK_ENTRY_SIZE])
#define	RT_NODE(A)		((NodeEntry *)&NodeTable[(A)*NODE_ENTRY_SIZE])
#define	RT_ALIAS(A)		((AliasEntry *)&AliasTable[(A)*ALIAS_ENTRY_SIZE])
#define	RT_DOMAIN(A)		((DomainEntry *)&DomainTable[(A)*DOMAIN_ENTRY_SIZE])
#define	RT_LEVEL(A)		((Index *)&LevelTable[(A)*LEVEL_ENTRY_SIZE])
