// Import verbsSet from the external file
import { verbsSet } from './verbs.js';

document.addEventListener('DOMContentLoaded', function() {
    const textInput = document.getElementById('vc-textInput');
    const verbCountElement = document.getElementById('vc-verbCount');
    const characterCountElement = document.getElementById('vc-characterCount');
    const verbListElement = document.getElementById('vc-verbList');
    const copyButton = document.getElementById('vc-copyButton');

    function updateCharacterCount() {
        const characters = textInput.value.length;
        characterCountElement.textContent = `Characters: ${characters}`;
        // Removed the direct call to updateVerbCount here for debounce application
    }

    function updateVerbCount() {
        const text = textInput.value.toLowerCase();
        const words = text.split(/\s+/);
        // Use verbsSet for checking verbs
        const verbs = words.filter(word => verbsSet.has(word));
        verbCountElement.textContent = verbs.length;
        updateVerbList(verbs);
    }

    function updateVerbList(verbs) {
        verbListElement.innerHTML = verbs.join(', ');
    }

    function debounce(func, timeout = 300) {
        let timer;
        return function(...args) {
            clearTimeout(timer);
            timer = setTimeout(() => { func.apply(this, args); }, timeout);
        };
    }

    // Apply debounce only to updateVerbCount
    textInput.addEventListener('input', function() {
        updateCharacterCount();
        debounceUpdateVerbCount();
    });

    const debounceUpdateVerbCount = debounce(updateVerbCount, 300);

    copyButton.addEventListener('click', async () => {
        if (navigator.clipboard) {
            try {
                await navigator.clipboard.writeText(textInput.value);
                copyButton.textContent = 'Copied!';
                setTimeout(() => {
                    copyButton.textContent = 'Copy Text';
                }, 2000);
            } catch (err) {
                console.error('Failed to copy:', err);
            }
        } else {
            textInput.select();
            document.execCommand('copy');
            copyButton.textContent = 'Copied!';
            setTimeout(() => {
                copyButton.textContent = 'Copy Text';
            }, 2000);
        }
    });

    // Initialize character and verb counts on load
    updateCharacterCount();
    updateVerbCount();
});









// Grab elements
const selectElement = (selector) => {
    const element = document.querySelector(selector);
    if(element) return element;
    throw new Error(`Something went wrong! Make sure that ${selector} exists/is typed correctly.`);  
};

const scrollHeader = () => {
    const navbarElement = selectElement('#header');
    if(window.scrollY >= 15) {
        navbarElement.classList.add('activated');
    } else {
        navbarElement.classList.remove('activated');
    }
};


window.addEventListener('scroll', scrollHeader);

// Open menu & search pop-up
const menuToggleIcon = selectElement('#menu-toggle-icon');
//const formOpenBtn = selectElement('#search-icon');
const formCloseBtn = selectElement('#form-close-btn');
const searchContainer = selectElement('#search-form-container');

const toggleMenu = () =>{
    const mobileMenu = selectElement('#menu');
    mobileMenu.classList.toggle('activated');
    menuToggleIcon.classList.toggle('activated');
}

menuToggleIcon.addEventListener('click', toggleMenu);

// Open/Close search form popup
//formOpenBtn.addEventListener('click', () => searchContainer.classList.add('activated'));
formCloseBtn.addEventListener('click', () => searchContainer.classList.remove('activated'));
// -- Close the search form popup on ESC keypress
window.addEventListener('keyup', (event) => {
    if(event.key === 'Escape') searchContainer.classList.remove('activated');
});









