// Assuming adjectives.js is correctly set up and exported
import { adjectivesSet } from './adjectives.js';

document.addEventListener('DOMContentLoaded', function() {
    const textInput = document.getElementById('vc-textInput');
    const adjectiveCountElement = document.getElementById('vc-adjectiveCount');
    const characterCountElement = document.getElementById('vc-characterCount');
    const adjectiveListElement = document.getElementById('vc-adjectiveList');
    const copyButton = document.getElementById('vc-copyButton');

    function updateCharacterCount() {
        const characters = textInput.value.length;
        characterCountElement.textContent = `Characters: ${characters}`;
    }

    function findAndUpdateAdjectives() {
        const text = textInput.value.toLowerCase();
        let foundWords = text.match(/\b\w+\b/gi) || [];
        let foundAdjectives = foundWords.filter(word => adjectivesSet.has(word));
        adjectiveCountElement.textContent = foundAdjectives.length;
        const placeholderText = "Your adjective list will be displayed here after you enter text in the text-box above.";
        adjectiveListElement.textContent = text.length === 0 ? placeholderText : foundAdjectives.length > 0 ? foundAdjectives.join(', ') : "No adjectives detected.";
    }

    function debounce(func, timeout = 300) {
        let timer;
        return function(...args) {
            clearTimeout(timer);
            timer = setTimeout(() => { func.apply(this, args); }, timeout);
        };
    }

    const debouncedFindAndUpdateAdjectives = debounce(findAndUpdateAdjectives);

    textInput.addEventListener('input', function() {
        updateCharacterCount();
        debouncedFindAndUpdateAdjectives();
    });

    copyButton.addEventListener('click', async () => {
        if (navigator.clipboard) {
            try {
                await navigator.clipboard.writeText(textInput.value);
                copyButton.textContent = 'Copied!';
                setTimeout(() => {
                    copyButton.textContent = 'Copy Text';
                }, 2000);
            } catch (err) {
                console.error('Failed to copy:', err);
            }
        } else {
            textInput.select();
            document.execCommand('copy');
            copyButton.textContent = 'Copied!';
            setTimeout(() => {
                copyButton.textContent = 'Copy Text';
            }, 2000);
        }
    });

    // Initial call to update counts
    updateCharacterCount();
    findAndUpdateAdjectives();
});







// Grab elements
const selectElement = (selector) => {
    const element = document.querySelector(selector);
    if(element) return element;
    throw new Error(`Something went wrong! Make sure that ${selector} exists/is typed correctly.`);  
};

const scrollHeader = () => {
    const navbarElement = selectElement('#header');
    if(window.scrollY >= 15) {
        navbarElement.classList.add('activated');
    } else {
        navbarElement.classList.remove('activated');
    }
};


window.addEventListener('scroll', scrollHeader);

// Open menu & search pop-up
const menuToggleIcon = selectElement('#menu-toggle-icon');
//const formOpenBtn = selectElement('#search-icon');
const formCloseBtn = selectElement('#form-close-btn');
const searchContainer = selectElement('#search-form-container');

const toggleMenu = () =>{
    const mobileMenu = selectElement('#menu');
    mobileMenu.classList.toggle('activated');
    menuToggleIcon.classList.toggle('activated');
}

menuToggleIcon.addEventListener('click', toggleMenu);

// Open/Close search form popup
//formOpenBtn.addEventListener('click', () => searchContainer.classList.add('activated'));
formCloseBtn.addEventListener('click', () => searchContainer.classList.remove('activated'));
// -- Close the search form popup on ESC keypress
window.addEventListener('keyup', (event) => {
    if(event.key === 'Escape') searchContainer.classList.remove('activated');
});









