
/*
 *@@sourcefile common.h:
 *      header file for common.c.
 *
 *      This prototypes functions that are common to all
 *      parts of XFolder. As opposed to helpers.c, these
 *      functions work with XFolder only, while the helper
 *      functions will work with any PM application.
 *
 *      This file also declares all kinds of structures, id's,
 *      flags, strings, commands etc. which are used by
 *      all XFolder parts, especially xfldr.c, xfdesk.c and
 *      common.c.
 *
 *      Additional declarations are passed through from the
 *      XFolder .IDL files into the xf*.h files.
 *
 *      As opposed to the declarations in dlgids.h, these
 *      declarations are NOT used by the NLS resource DLLs.
 *      The declarations have been separated to avoid
 *      unnecessary recompiles.
 *
 *      Note: this header automatically #include's pmprintf.h.
 */

/*
 *      Copyright (C) 1997-99 Ulrich Mller.
 *      This file is part of the XFolder source package.
 *      XFolder is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XFolder main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#ifndef COMMON_HEADER_INCLUDED
    #define COMMON_HEADER_INCLUDED

    /********************************************************************
     *                                                                  *
     *   Debugging flags                                                *
     *                                                                  *
     ********************************************************************/

    /*
     *  The following #define's determine whether additional debugging
     *  info will be compiled into xfldr.dll. These flags fall into two
     *  categories:
     *  1)  if you uncomment the DONTDEBUGATALL line,
     *      no debugging will be coded at all; this is
     *      recommended for public XFolder releases
     *  2)  otherwise, the following lines allow
     *      debugging of certain XFolder code only.
     *      This will produce additional code and slow
     *      down the system.
     *  Note that these debugging flags affect only the .C files which
     *  have #include'd common.h. That's only the files in the /MAIN
     *  directory, not in /HELPERS.
     */

    #define DONTDEBUGATALL

    #ifndef DONTDEBUGATALL

        // if the following is commented out, no PMPRINTF will be
        // used at all. XFolder uses Dennis Bareis' PMPRINTF
        // package to do this.

        // **** IMPORTANT NOTE: if you use this flag, you _must_
        // have the PMPRINTF DLLs somewhere on your LIBPATH, or
        // otherwise XFLDR.DLL cannot be loaded, because the imports
        // will fail. That is, XFolder classes can neither be registered
        // nor loaded at WPS startup. This has cost me a lot of thought
        // once, and you'll get no error message, so be warned.
            #define _PMPRINTF_

        // the following replaces the SOMMethodDebug macros with
        // a PMPRINTF version. This leads to a LOT of output for
        // each SOM method called from all the XFolder files and
        // slows down the system _very_ much if the PMPRINTF output
        // wnd is open
            // #define DEBUG_SOMMETHODS

        // the following writes xfldtrap.log even for "quiet"
        // exceptions, i.e. those handled by excHandlerQuiet
        // (xfdesk.c)
            // #define DEBUG_WRITEQUIETEXCPT

        // the following will write an extended stack dump into
        // the traplog of the exception handlers (xfdesk.c)
            // #define DEBUG_EXCPT_STACKDUMP

        // the following printfs about folder context menus
            // #define DEBUG_CONTEXT

        // the following allows debug mode for XShutdown, which
        // will be enabled by holding down SHIFT while selecting
        // "Shutdown..." from the desktop context menu. In addition,
        // you'll get some PMPRINTF info and beeps
            // #define DEBUG_SHUTDOWN

        // the following beeps when thread priorities change
            // #define DEBUG_PRIORITY

        // the following printf's each added awake object
            // #define DEBUG_AWAKEOBJECTS

        // the following gives information on ordered folder content
        // (sorting by .ICONPOS etc.)
            // #define DEBUG_ORDEREDLIST

        // the following displays internal status bar data
            // #define DEBUG_STATUSBARS

        // the following will printf all kinds of settings
        // notebook information
            // #define DEBUG_SETTINGS

        // the following will printf WM_CONTROL for WPS cnrs
            // #define DEBUG_CNRCNTRL

        // the following will printf wpAddToContent
            // #define DEBUG_CNRCONTENT

        // the following will printf lots of sort info
            // #define DEBUG_SORT

        // the following will printf language code queries and
        // NLS DLL evaluation
            #define DEBUG_LANGCODES

        // the following will printf folder hotkey info
            // #define DEBUG_KEYS

        // the following displays XFolder memory usage in the
        // "Object internals" of the Desktop settings; this
        // will produce additional debug code for malloc(),
        // so this better only be used for debugging
            // #define DEBUG_MEMORY

        // the following beeps when the Worker thread cleans up
        // the default heap
            // #define DEBUG_MEMORYBEEP

        // the following displays a lot of infos about menu
        // processing (msgs), esp. for folder content menus
            // #define DEBUG_MENUS

        // debug startup folder processing
            // #define DEBUG_STARTUP

        // debug title clash dialog
            // #define DEBUG_TITLECLASH

        // debug new system sounds
            // #define DEBUG_SOUNDS

        // debug data/program file associations/icons
            // #define DEBUG_ASSOCS

        // debug folder icon replacements
            // #define DEBUG_FLDRICONS

        // debug wpRestoreData and such
            // #define DEBUG_RESTOREDATA

    #endif

    /********************************************************************
     *                                                                  *
     *   #include's                                                     *
     *                                                                  *
     ********************************************************************/

    // the following reacts to the _PMPRINTF_ macro def'd above;
    // if that's #define'd, _Pmpf(()) commands will produce output,
    // if not, no code will be produced
    #include "pmprintf.h"

    #ifdef   SOMMethodDebug
             #undef  SOMMethodDebug
    #endif

    // SOMMethodDebug is the macro defined for all those
    // xxxMethodDebug thingies created by the SOM compiler.
    // If you have uncommended DEBUG_SOMMETHODS above, this
    // will make sure that _Pmpf is used for that. In order
    // for this to work, you'll also need _PMPRINTF_.
    #ifdef DEBUG_SOMMETHODS
        #define  SOMMethodDebug(c,m) _Pmpf(("%s::%s", c,m))
    #else
        #define  SOMMethodDebug(c,m) ;
    #endif

    /********************************************************************
     *                                                                  *
     *   INI keys                                                       *
     *                                                                  *
     ********************************************************************/

    // XFolder application in OS2.INI; you must specify the version
    // number with this, because the Prf* functions seem to have a
    // bug in that they refer to "XFolderxxx" if only "XFolder" is
    // specified here
    #define INIAPP_XFOLDER                  "XFolder"

    #define INIKEY_DEFAULTTITLE     "DefaultTitle"
    #define INIKEY_GLOBALSETTINGS   "GlobalSettings"
    // #define INIKEY_XFOLDERPATH      "XFolderPath"
    #define INIKEY_ACCELERATORS     "Accelerators"
    #define INIKEY_LANGUAGECODE     "Language"
    #define INIKEY_JUSTINSTALLED    "JustInstalled"
    // #define INIKEY_DONTDOSTARTUP    "DontDoStartup"
    // #define INIKEY_LASTPID          "LastPID"
    #define INIKEY_FAVORITEFOLDERS  "FavoriteFolders"
    #define INIKEY_QUICKOPENFOLDERS "QuickOpenFolders"

    #define INIKEY_WNDPOSSTARTUP    "WndPosStartup"
    #define INIKEY_WNDPOSNAMECLASH  "WndPosNameClash"
    #define INIKEY_NAMECLASHFOCUS   "NameClashLastFocus"

    #define INIKEY_STATUSBARFONT    "SB_Font"
    #define INIKEY_SBTEXTNONESEL    "SB_NoneSelected"
    #define INIKEY_SBTEXT_WPOBJECT  "SB_WPObject"
    #define INIKEY_SBTEXT_WPPROGRAM "SB_WPProgram"
    #define INIKEY_SBTEXT_WPFILESYSTEM "SB_WPDataFile"
    #define INIKEY_SBTEXT_WPURL        "SB_WPUrl"
    #define INIKEY_SBTEXT_WPDISK    "SB_WPDisk"
    #define INIKEY_SBTEXT_WPFOLDER  "SB_WPFolder"
    #define INIKEY_SBTEXTMULTISEL   "SB_MultiSelected"
    #define INIKEY_SB_LASTCLASS     "SB_LastClass"
    #define INIKEY_DLGFONT          "DialogFont"

    #define INIKEY_BOOTMGR          "RebootTo"
    #define INIKEY_AUTOCLOSE        "AutoClose"

    #define DEFAULT_LANGUAGECODE    "001"

    // some default INI keys
    #define WPINIAPP_FOLDERPOS      "PM_Workplace:FolderPos"
    #define WPINIAPP_TASKLISTPOS    "PM_Workplace:WindowListPos"
    #define WPINIKEY_TASKLISTPOS    "SavePos"

    // XFolder WPS object id's
    #define XFOLDER_CONFIGID        "<XFOLDER_CONFIG>"
    #define XFOLDER_STARTUPID       "<XFOLDER_STARTUP2>"
    #define XFOLDER_SHUTDOWNID      "<XFOLDER_SHUTDOWN2>"
    #define XFOLDER_MAINID          "<XFOLDER_MAIN>"
    #define XFOLDER_WPSID           "<XFOLDER_WPS>"
    #define XFOLDER_KERNELID        "<XFOLDER_KERNEL>"

    /********************************************************************
     *                                                                  *
     *   Help panels                                                    *
     *                                                                  *
     ********************************************************************/

    #define ID_XMH_VARIABLE         2
    #define ID_XMH_CONFIGFOLDER     3
    #define ID_XMH_REFRESH          4
    #define ID_XMH_SNAPTOGRID       5
    #define ID_XMH_COPYFILENAME     6
    #define ID_XMH_XSHUTDOWN        7
    #define ID_XMH_RESTARTWPS       8

    // the following defines the panel ID for the
    // first settings page in the System notebook;
    // the other help panels follow thereafter */
    // help panel IDs for XFolder context menu items

    #define ID_XSH_SETTINGS1            9
    #define ID_XSH_SETTINGS_DTP1        16
    #define ID_XSH_SETTINGS_FLDRSORT    18
    #define ID_XSH_SETTINGS_FLDR1       19
    #define ID_XSH_SETTINGS_SB2         20
    #define ID_XSH_SETTINGS_WPSCLASSES  21
    #define ID_XSH_SETTINGS_OBJINTERNALS 22
    #define ID_XSH_SETTINGS_KERNEL1     23
    #define ID_XSH_SETTINGS_KERNEL2     24
    #define ID_XMH_STARTUPSHUTDOWN      25
    #define ID_XMH_FOLDERCONTENT        26
    #define ID_XSH_SETTINGS_HPFS        27
    #define ID_XSH_SETTINGS_FAT         27              // same page as HPFS
    #define ID_XSH_SETTINGS_CFGM        28              // "Config folder menu items"
    #define ID_XSH_SETTINGS_FILEOPS     29              // "Config folder menu items"
    #define ID_XSH_SETTINGS_WPS         30
    #define ID_XSH_SETTINGS_ERRORS      31

    // the following defines the panel ID for the
    // first dlg item (ID_XSDI_FIRST == ID_XSDI_REPLICONS);
    // the other help panels follow thereafter */
    #define ID_XSH_FIRST            61            // single settings dlg items

    // help panel IDs for various dlg boxes
    #define ID_XFH_BORED            40
    #define ID_XFH_NOCONFIG         (ID_XFH_BORED+1)
    #define ID_XFH_NOOBJECT         (ID_XFH_BORED+2)
    #define ID_XFH_LIMITREACHED     (ID_XFH_BORED+3)
    #define ID_XFH_NOTEXTCLIP       (ID_XFH_BORED+4)
    #define ID_XFH_REBOOTEXT        (ID_XFH_BORED+5)
    #define ID_XFH_AUTOCLOSEDETAILS (ID_XFH_BORED+6)
    #define ID_XFH_SELECTCLASS      (ID_XFH_BORED+7)
    #define ID_XFH_REGISTERCLASS    (ID_XFH_BORED+8)
    #define ID_XFH_TITLECLASH       (ID_XFH_BORED+9)

    #define ID_XFH_SELECTSOME       87

    /********************************************************************
     *                                                                  *
     *   Various other identifiers/flag declarations                    *
     *                                                                  *
     ********************************************************************/

    // ID's of XFolder object windows
    #define ID_XFOLDEROBJECT        0x1234
    #define ID_WORKEROBJECT         0x1235
    #define ID_SOUNDOBJECT          0x1236

    // offset by which the controls should be moved
    // when converting buttons to Warp 4 notebook style
    // (using winhAssertWarp4Notebook); this is in
    // "dialog units"
    #define WARP4_NOTEBOOK_OFFSET   14

    // XFolder settings page IDs; these are used by
    // the following:
    // --  the notebook.c functions to identify open
    //     pages;
    // --  the settings functions in common.c to
    //     identify the default settings to be set.
    // Groups of settings pages:
    // 1) in "Workplace Shell"
    #define SP_1GENERIC             1
    #define SP_2REMOVEITEMS         2
    #define SP_25ADDITEMS           3
    #define SP_26CONFIGITEMS        4
    #define SP_27STATUSBAR          5
    #define SP_3SNAPTOGRID          6
    #define SP_4ACCELERATORS        7
    #define SP_5INTERNALS           8
    #define SP_DTP1                 9
    #define SP_DTP2                 10
    #define SP_28STATUSBAR2         11
    #define SP_FILEOPS              12
    // 2) in "OS/2 Kernel"
    #define SP_SCHEDULER            20
    #define SP_MEMORY               21
    #define SP_HPFS                 22
    #define SP_FAT                  23
    #define SP_ERRORS               24
    #define SP_WPS                  25
    // 3) "Sort" pages both in folder notebooks and
    //    "Workplace Shell"
    #define SP_FLDRSORT_FLDR        30
    #define SP_FLDRSORT_GLOBAL      31
    // 4) "XFolder" page in folder notebooks
    #define SP_XFOLDER_FLDR         40

    // markers for XFolder variable menu items:
    // these flags are used in the global linked list
    // of menu items which are inserted into folder
    // context menus to determine what we need to do
    // if a menu item gets selected
    #define OC_PROGRAM              2     // program object
    #define OC_FOLDER               3     // folder: insert submenu
    #define OC_ACTION               4     // not used
    #define OC_TEMPLATE             5     // template: create new object
    #define OC_OTHER                1     // all others: simply open

    #define OC_CONTENT              7     // this item is part of the "folder content" menu
    #define OC_CONTENTFOLDER        8     // the same, but it's a folder

    #define OC_SEPARATOR            10    // program object, but separator only

    #define CLIPBOARDKEY "%**C"        /* code in program object's parameter list for
                                          inserting clipboard data */

    // miscellaneae
    #define DEFAULTTITLELENGTH      50
    #define LANGUAGECODELENGTH      30

    // log file names
    #define XFOLDER_CRASHLOG        "xfldtrap.log"
    #define XFOLDER_SHUTDOWNLOG     "xshutdwn.log"

    // window class names
    #define WNDCLASS_XFLDOBJECT     "XFolderMainObjectWindow"
    #define WNDCLASS_SUPPLOBJECT    "XFolderSupplementaryObjectWindow"

    // XFolder object window in Workplace thread (XOM_xxx)
    #define XOM_POCCALLBACK             WM_USER+1100
    #define XOM_BEGINSTARTUP            WM_USER+1101
    #define XOM_EXCEPTIONCAUGHT         WM_USER+1102
    #define XOM_LIMITREACHED            WM_USER+1103
    #define XOM_QUERYXFOLDERVERSION     WM_USER+1105
    #define XOM_EXTERNALSHUTDOWN        WM_USER+1106
    #define XOM_BEGINQUICKOPEN          WM_USER+1107
    #define XOM_NEXTQUICKOPEN           WM_USER+1108

    // Supplementary object window msgs (for each
    // subclassed folder frame, xfldr.c)
    #define SOM_ACTIVATESTATUSBAR       WM_USER+1200

    // common dlg msgs for settings notebook dlg funcs
    #define WM_SETTINGS2DLG         WM_USER+200    // set controls
    #define WM_DLG2SETTINGS         WM_USER+201    // read controls
    #define WM_ENABLEITEMS          WM_USER+202    // enable/disable controls
    #define WM_DEFAULT              WM_USER+203    // set default values
    #define WM_OPTIMIZE             WM_USER+204    // propose better values (CONFIG.SYS)

    // misc
    #define WM_UPDATE               WM_USER+1006 // in dlgs
    #define WM_SETLONGTEXT          WM_USER+1007 // for cmnMessageBox

    // common value for indicating that a Global Setting
    // is to be used instead of an instance's one
    #define SET_DEFAULT             255

    // flags for xfSet/QueryStatusBarVisibility
    #define STATUSBAR_ON            1
    #define STATUSBAR_OFF           0
    #define STATUSBAR_DEFAULT       255

    // flags for xfSortViewOnce etc.
    #define SV_NAME                 0
    #define SV_TYPE                 1
    #define SV_CLASS                2
    #define SV_REALNAME             3
    #define SV_SIZE                 4
    #define SV_LASTWRITEDATE        5
    #define SV_LASTACCESSDATE       6
    #define SV_CREATIONDATE         7
    #define SV_EXT                  8
    #define SV_FOLDERSFIRST         9
    #define SV_LAST                 9


    /********************************************************************
     *                                                                  *
     *   Global structures                                              *
     *                                                                  *
     ********************************************************************/

    /* GlobalSettings structure for use in memory
       and OS2.INI; we use ULONGs for performance
       reasons. Each setting in here corresponds to
       a setting in the "System" objects and Desktop
       settings notebook.
       Use cmnQueryGlobalSettings() to get a pointer
       to the global settings. */
    typedef struct GLOBALSETTINGS_STRUCT
    {
        ULONG       ReplIcons,
                    MenuCascadeMode,
                    FullPath,
                    KeepTitle,
                    RemoveX,
                    AppdParam,
                    MoveRefreshNow;

        ULONG       MaxPathChars,
                    DefaultMenuItems;

        LONG        VarMenuOffset;

        ULONG       AddSnapToGridItem;

        LONG        GridX,
                    GridY,
                    GridCX,
                    GridCY;

        ULONG       Accelerators,
                    TemplatesOpenSettings;
    /* XFolder 0.52 */
        ULONG       RemoveLockInPlaceItem,
                    RemoveFormatDiskItem,
                    RemoveCheckDiskItem,
                    RemoveViewMenu,
                    RemovePasteItem,

                    ulRemoved1, // DebugMode,
                    AddCopyFilenameItem;
        ULONG       ulXShutdownFlags;
        ULONG       NoWorkerThread,
                    NoSubclassing,
                    TreeViewAutoScroll,
                    ShowStartupProgress;
        ULONG       ulStartupDelay;

    /* XFolder 0.70 */
        ULONG       AddFolderContentItem,
                    FCShowIcons,
                    StatusBar,
                    SBStyle;                    // status bar style
        LONG        lSBBgndColor, lSBTextColor; // status bar colors
        ULONG       TemplatesReposition;
        USHORT      usLastRebootExt;
        ULONG       AddSelectSomeItem,
                    ReplaceSort,
                    AlwaysSort,
                    DefaultSort,
                    CleanupINIs,

    /* XFolder 0.80  */
                    ShowBootupStatus,
                    WpsShowClassInfo;
        ULONG       SBForViews,                 // SBV_xxx flags
                    ReplConfirms,
                    ShowXFolderAnim,
                    FileAttribs,
                    ShowInternals,
                    ExtendFldrViewMenu;

    } GLOBALSETTINGS, *PGLOBALSETTINGS;

    /* NLSSTRINGS:
       structure which is filled upon WPS IPL with
       all language-dependent XFolder strings. Use
       cmnQueryNLSStrings() to get access to this
       structure. */
    typedef struct NLSSTRINGS_STRUCT {
        // strings from resources
        PSZ     pszNotDefined,
                pszProductInfo,
                pszRefreshNow,
                pszSnapToGrid,

                pszFldrContent,
                pszCopyFilename,
                pszBored,
                pszFldrEmpty,

                pszSelectSome,
                pszQuickStatus,

                pszCtrl,
                pszAlt,
                pszShift,

                pszBackspace,
                pszTab,
                pszBacktab,
                pszEnter,
                pszEsc,
                pszSpace,
                pszPageup,
                pszPagedown,
                pszEnd,
                pszHome,
                pszLeft,
                pszUp,
                pszRight,
                pszDown,
                pszPrintscrn,
                pszInsert,
                pszDelete,
                pszScrlLock,
                pszNumLock,

                pszWinLeft,
                pszWinRight,
                pszWinMenu,

                pszSDFlushing,
                pszSDCAD,
                pszSDRebooting,
                pszSDRestartingWPS,
                pszSDClosing,
                pszShutdown,
                pszRestartWPS,
                pszSDSavingDesktop,
                pszSDSavingProfiles,

                pszStarting,

                pszPopulating,

                psz1Generic,
                psz2RemoveItems,
                psz25AddItems,
                psz26ConfigFolderMenus,
                psz27StatusBar,
                psz3SnapToGrid,
                psz4Accelerators,
                psz5Internals,
                pszFileOps,

                pszSortByName,
                pszSortByType,
                pszSortByClass,
                pszSortByRealName,
                pszSortBySize,
                pszSortByWriteDate,
                pszSortByAccessDate,
                pszSortByCreationDate,
                pszSortByExt,
                pszSortFoldersFirst,

                pszSort,
                pszAlwaysSort,

                pszInternals,

                pszSBClassMnemonics,
                pszSBClassNotSupported,

                pszWpsClasses,
                pszWpsClassLoaded,
                pszWpsClassLoadingFailed,
                pszWpsClassReplacedBy,
                pszWpsClassOrphans,
                pszWpsClassOrphansInfo,

                pszScheduler,
                pszMemory,
                pszErrors,
                pszWPS,

                pszProcessContent,

                pszSettings,
                pszSettingsNotebook,
                pszAttributes,
                pszAttrArchived,
                pszAttrSystem,
                pszAttrHidden,
                pszAttrReadOnly,

                pszWarp3FldrView,
                pszSmallIcons,
                pszFlowed,
                pszNonFlowed,
                pszNoGrid,

                pszShowStatusBar;

    } NLSSTRINGS, *PNLSSTRINGS;

    // flags for mnuMenuItemSelected (menus.c)
    #define SEL_WHITESPACE 1
         // the context menu was opened on the
         // whitespace in an open container view
         // of somSelf (which is a folder then)
    #define SEL_SINGLESEL 2
         // the context menu was opened for a
         // single selected object: somSelf can
         // be any object then, including folders
    #define SEL_MULTISEL 3
         // the context menu was opened on one
         // of a multitude of selected objects;
         // again, somSelf can be any object
    #define SEL_SINGLEOTHER 4
         // the context menu was opened for a
         // single object _other_ than the selected objects

    #ifdef SOM_WPObject_h
        // OBJECTLISTITEM:
        // linked list structure for storing a list of
        // objects. This is used for
        // storing the currently awake WPS objects
        // on the system (in the XFolder Worker thread)
        typedef struct _OBJECTLISTITEM {
            struct _OBJECTLISTITEM
                                    *pNext, *pPrevious;
            ULONG                   ulSize;
            WPObject                *pObj;
        } OBJECTLISTITEM, *POBJECTLISTITEM;

        // VARMENULISTITEM:
        // linked list item for variable menu items
        // inserted into folder context menus; these
        // list items are created for both config folder
        // and folder content items
        typedef struct VARMENULISTITEM_STRUCT {
            struct VARMENULISTITEM_STRUCT   *pNext, *pPrevious;
            ULONG                       ulSize;
            WPObject                    *pObject;
            CHAR                        szTitle[100];
            USHORT                      usObjType;
        } VARMENULISTITEM, *PVARMENULISTITEM;

        // ORDEREDLISTITEM:
        // linked list structure for the ordered list
        // of objects in a folder (xfQueryOrderedContent).
        typedef struct _ORDEREDLISTITEM {
            struct _ORDEREDLISTITEM *pNext, *pPrevious;
            ULONG                   ulSize;
            WPObject                *pObj;
            CHAR                    szIdentity[CCHMAXPATH];
        } ORDEREDLISTITEM, *PORDEREDLISTITEM;

        // PROCESSCONTENTINFO:
        // structure needed for processing the ordered content
        // of a folder (Startup, Shutdown folders)
        typedef struct _PROCESSCONTENTINFO {
            WPObject                         *pObject;  // current object in this folder
            HWND                             hwndView;  // hwnd of this object, if opened succfly.
            ULONG                            ulObjectNow, ulObjectMax;
            ULONG                            ulTiming;
            ULONG                            ulFirstTime;
            PFNWP                            pfnwpCallback;
            ULONG                            ulCallbackParam;
            BOOL                             fCancelled;
        } PROCESSCONTENTINFO, *PPROCESSCONTENTINFO;
    #endif

    #ifdef SOM_WPFolder_h
        // CONTENTMENULISTITEM:
        // additional linked list item for
        // "folder content" menus
        typedef struct CONTENTMENULISTITEM_STRUCT {
            struct CONTENTMENULISTITEM_STRUCT   *pNext, *pPrevious;
            ULONG                       ulSize;
            WPFolder                    *pFolder;
            SHORT                       sMenuId;
        } CONTENTMENULISTITEM, *PCONTENTMENULISTITEM;
    #endif

    /*
     *  The following functions are actually present in
     *  common.c:
     */

    /********************************************************************
     *                                                                  *
     *   XFolder debugging helpers                                      *
     *                                                                  *
     ********************************************************************/

    /*
     * cmnDumpMemoryBlock:
     *      if _PMPRINTF_ has been #define'd in common.h,
     *      this will dump a block of memory to the PMPRINTF
     *      output window. Useful for debugging internal
     *      structures.
     *      If _PMPRINTF_ has been NOT #define'd in common.h,
     *      no code will be produced. :-)
     */

    #ifdef _PMPRINTF_
    void cmnDumpMemoryBlock(PBYTE pb,       // in: start address
                            ULONG ulSize,   // in: size of block
                            ULONG ulIndent);// in: how many spaces to put
                                            //     before each output line
    #else
        // _PMPRINTF not #define'd: do nothing
        #define cmnDumpMemoryBlock(pb, ulSize, ulIndent)
    #endif

    /********************************************************************
     *                                                                  *
     *   XFolder module handling                                        *
     *                                                                  *
     ********************************************************************/

    /*
     *  The following routines are for querying the XFolder
     *  installation path and similiar routines, such as
     *  querying the current NLS module, changing it, loading
     *  strings, the help file and all that sort of stuff.
     */

    /*
     * cmnQueryXFolderBasePath:
     *      this routine returns the path of where XFolder was installed,
     *      i.e. the parent directory of where the xfldr.dll file
     *      resides (e.g. "C:\XFolder"). The buffer to copy this to
     *      is assumed to be CCHMAXPATH in size.
     *      As opposed to versions before V0.81, OS2.INI is no longer
     *      needed for this to work. The path is retrieved from the
     *      DLL directly by evaluating what was passed to _DLL_InitModule
     *      (module.*).
     */

    BOOL cmnQueryXFolderBasePath(PSZ pszPath);

    /*
     * cmnQueryLanguageCode:
     *      returns PSZ to three-digit language code (e.g. "001").
     */

    PSZ cmnQueryLanguageCode(VOID);

    /*
     * cmnSetLanguageCode:
     *      changes XFolder's language to three-digit language code in
     *      pszLanguage (e.g. "001"). This does not reload the NLS DLL,
     *      but only change the setting.
     */

    BOOL cmnSetLanguageCode(PSZ pszLanguage);

    /*
     * cmnQueryHelpLibrary:
     *      returns PSZ to full help library path in XFolder directory,
     *      depending on where XFolder was installed and on the current
     *      language (e.g. "C:\XFolder\help\xfldr001.hlp").
     */

    PSZ cmnQueryHelpLibrary(VOID);

    /*
     * cmnQueryMessageFile:
     *      returns PSZ to full message file path in XFolder directory,
     *      depending on where XFolder was installed and on the current
     *      language (e.g. "C:\XFolder\help\xfldr001.msg").
     */

    PSZ cmnQueryMessageFile(VOID);

    /*
     * cmnQueryIconsDLL:
     *      this returns the HMODULE of XFolder ICONS.DLL
     *      (new with V0.84).
     *      If this is queried for the first time, the DLL
     *      is loaded from the /ICONS directory.
     */

    HMODULE cmnQueryIconsDLL(VOID);

    /*
     * cmnLoadString:
     *      pretty similar to WinLoadString, but allocates
     *      necessary memory as well. *ppsz is a pointer
     *      to a PSZ; if this PSZ is != NULL, whatever it
     *      points to will be free()d, so you should set this
     *      to NULL if you initially call this function.
     *      This is used at WPS startup and when XFolder's
     *      language is changed later to load all the strings
     *      from a NLS DLL.
     */

    void cmnLoadString(HAB habDesktop, HMODULE hmodResource, ULONG ulID, PSZ *ppsz);

    /*
     * cmnQueryNLSModuleHandle:
     *      returns the module handle of the language-dependent XFolder
     *      national language support DLL.
     *      This is called in two situations:
     *          a) with fEnforceReload == FALSE everytime some part
     *             of XFolder needs the NLS resources (e.g. for dialogs);
     *             on the first call, the NLS DLL is loaded into memory.
     *          b) with fEnforceReload == TRUE when the user changes
     *             XFolder's language in the "Workplace Shell" object.
     *      If the DLL is (re)loaded, this function also initializes
     *      all language-dependent XFolder parts such as the NLSSTRINGS
     *      structure, which can always be accessed using cmnQueryNLSStrings.
     *      This function also checks for whether the NLS DLL has a
     *      decent version level to support this XFolder version.
     */

    HMODULE cmnQueryNLSModuleHandle(BOOL fEnforceReload);

    /*
     * cmnQueryNLSStrings:
     *      returns pointer to global NLSSTRINGS structure which contains
     *      all the language-dependent XFolder strings from the resource
     *      files.
     */

    PNLSSTRINGS cmnQueryNLSStrings(VOID);

    /********************************************************************
     *                                                                  *
     *   XFolder Global Settings                                        *
     *                                                                  *
     ********************************************************************/

    /*
     * cmnQueryStatusBarSetting:
     *      returns a PSZ to a certain status bar setting, which
     *      may be:
     *              SBS_STATUSBARFONT       font (e.g. "9.WarpSans")
     *              SBS_TEXTNONESEL         mnemonics for no-object mode
     *              SBS_TEXTMULTISEL        mnemonics for multi-object mode
     *      Note that there is no key for querying the mnemonics for
     *      one-object mode, because this is now handled by the
     *      WPS classes to provide different data depending on the
     *      class of the selected object.
     */

    PSZ cmnQueryStatusBarSetting(USHORT usSetting);

    /*
     * cmnSetStatusBarSetting:
     *      sets usSetting to pszSetting. If pszSetting == NULL, the
     *      default value will be loaded from the XFolder NLS DLL.
     *      usSetting works just like in cmnQueryStatusBarSetting.
     */

    BOOL cmnSetStatusBarSetting(USHORT usSetting, PSZ pszSetting);

    /*
     * cmnQueryStatusBarHeight:
     *      returns the height of the status bars according to the
     *      current settings in pixels. This was calculated when
     *      the status bar font was set.
     */

    ULONG cmnQueryStatusBarHeight(VOID);

    /*
     * cmnLoadGlobalSettings:
     *      this loads the Global Settings from the INI files; should
     *      not be called directly, because this is done automatically
     *      by cmnQueryGlobalSettings, if necessary.
     */

    PGLOBALSETTINGS cmnLoadGlobalSettings(VOID);

    /*
     * cmnQueryGlobalSettings:
     *      returns pointer to the GLOBALSETTINGS structure which
     *      contains the XFolder Global Settings valid for all
     *      classes. Loads the settings from the INI files if this
     *      hasn't been done yet. Used all the time throughout XFolder.
     */

    PGLOBALSETTINGS cmnQueryGlobalSettings(VOID);

    /*
     * cmnStoreGlobalSettings:
     *      stores the current Global Settings back into the INI files;
     *      returns TRUE if successful.
     */

    BOOL cmnStoreGlobalSettings(VOID);

    /*
     * cmnSetDefaultSettings:
     *      resets those Global Settings which correspond to usSettingsPage
     *      in the System notebook to the default values; usSettingsPage
     *      may have 1 thru the number of XFolder Settings pages in the
     *      System notebook (SP_* flags). This is used to initialize
     *      XFolder settings at startup and by the "Default" buttons on
     *      the notebook settings pages.
     */

    BOOL cmnSetDefaultSettings(USHORT usSettingsPage);

    /********************************************************************
     *                                                                  *
     *   Folder hotkey functions                                        *
     *                                                                  *
     ********************************************************************/

    // maximum no. of folder hotkeys
    #define FLDRHOTKEYCOUNT (ID_XSSI_LB_LAST-ID_XSSI_LB_FIRST+1)

    // maximum length of folder hotkey descriptions
    #define MAXLBENTRYLENGTH 50

    // XFolder keyboard hotkeys; an array
    // of these exists in common.c
    typedef struct _XFLDHOTKEY {
        USHORT  usFlags;     //  Keyboard control codes
        USHORT  usKeyCode;   //  Hardware scan code
        USHORT  usCommand;   //  corresponding menu item id to send to container
    } XFLDHOTKEY, *PXFLDHOTKEY;

    #define FLDRHOTKEYSSIZE sizeof(XFLDHOTKEY)*FLDRHOTKEYCOUNT

    /*
     * cmnQueryFldrHotkeys:
     *      this returns the address of the static
     *      folder hotkeys array in common.c. The
     *      size of that array is FLDRHOTKEYSSIZE.
     */

    PXFLDHOTKEY cmnQueryFldrHotkeys(VOID);

    /*
     * cmnLoadDefaultFldrHotkeys:
     *      this resets the folder hotkeys to the default
     *      values.
     */

    void cmnLoadDefaultFldrHotkeys(VOID);

    /*
     * cmnLoadFolderHotkeys:
     *      this initializes the folder hotkey array with
     *      the data which was previously stored in OS2.INI.
     */

    void cmnLoadFolderHotkeys(VOID);

    /*
     * cmnStoreFldrHotkeys:
     *       this stores the folder hotkeys in OS2.INI.
     */

    void cmnStoreFldrHotkeys(VOID);

    /*
     * cmnProcessFldrHotkey:
     *      this is called by the subclassed folder frame wnd proc
     *      to check for whether a given WM_CHAR message matches
     *      one of the folder hotkeys.
     *      The parameters are those of the WM_CHAR message. This
     *      returns TRUE if the pressed key was a hotkey; in that
     *      case, the corresponding WM_COMMAND message is
     *      automatically posted to the folder frame, which will
     *      cause the defined action to occur.
     */

    BOOL cmnProcessFldrHotkey(HWND hwndFrame, MPARAM mp1, MPARAM mp2);

    /*
     * cmnDescribeKey:
     *      this stores a description of a certain
     *      key into pszBuf, using the NLS DLL strings.
     *      usFlags must contain usFlags from WM_CHAR,
     *      usKeyCode must be either usch (char code)
     *      or, with virtual keys, usvk (vk code).
     *      Returns TRUE if this was a valid key combo.
     */

    BOOL cmnDescribeKey(PSZ pszBuf, USHORT usFlags, USHORT usKeyCode);

    /********************************************************************
     *                                                                  *
     *   Folder frame window subclassing                                *
     *                                                                  *
     ********************************************************************/

    #ifdef SOM_XFolder_h

        // SUBCLASSEDLISTITEM:
        // linked list structure to remember original window
        // procedures when subclassing folder frame windows;
        // this list is maintained by wpOpen of XFolder and is also
        // used for storing various other data for an open frame view
        typedef struct _SUBCLASSEDLISTITEM {
            struct _SUBCLASSEDLISTITEM  *pNext, *pPrevious;
            ULONG       ulSize;
            HWND        hwndFrame;          // folder frame
            WPFolder    *somSelf;           // folder object
            WPObject    *pRealObject;       // "real" object; this is == somSelf
                                            // for folders, but the corresponding
                                            // disk object for WPRootFolders
            PFNWP       pfnwpOriginal;      // orig. frame wnd proc
            HWND        hwndStatusBar,      // status bar window
                        hwndCnr,            // cnr window
                        hwndSupplObject;    // supplementary object wnd
            ULONG       ulView;             // OPEN_CONTENTS
                                            //   | OPEN_TREE
                                            //   | OPEN_DETAILS
            BOOL        fNeedCnrScroll;     // scroll container after adding status bar?
            BOOL        fRemoveSrcEmphasis; // flag for whether XFolder has added
                                            // container source emphasis
            USHORT      usLastSelMenuItem;  // last selected menu item ID
        } SUBCLASSEDLISTITEM, *PSUBCLASSEDLISTITEM;


        VOID cmnInitPSLI(VOID);

        PSUBCLASSEDLISTITEM cmnQueryPSLI(HWND hwndFrame);

        PSUBCLASSEDLISTITEM cmnSubclassFolderFrame(HWND hwndFrame,
                                                   XFolder *somSelf,
                                                   WPObject *pRealObject,
                                                   ULONG ulView);

        VOID cmnRemovePSLI(PSUBCLASSEDLISTITEM psli);

    #endif

    /********************************************************************
     *                                                                  *
     *   Miscellaneous                                                  *
     *                                                                  *
     ********************************************************************/

    /*
     * cmnMessageBox:
     *      this is a new message box function; it works
     *      pretty similar to WinMessageBox, but displays
     *      a user-defined icon instead. It also adjusts
     *      itself according to font changes and such.
     *      Returns MBID_* codes like WinMessageBox.
     */

    ULONG cmnMessageBox(HWND hwndOwner,     // in: owner
                        PSZ pszTitle,       // in: msgbox title
                        PSZ pszMessage,     // in: msgbox text
                        ULONG flStyle);     // in: MB_* flags

    /*
     * cmnGetMessage:
     *      like DosGetMessage, but automatically finds the
     *      (NLS) XFolder message file.
     *      Parameters:
     *          PCHAR *pTable     array[ulTable] of PSZs with
     *                            substitutes for "%1" etc. in
     *                            the msg string
     *          ULONG ulTable     size of that array
     *          PSZ pszBuf        buffer to hold message string
     *          ULONG cbBuf       size of pszBuf
     *          ULONG ulMsgNumber msg number to retrieve
     *
     */

    APIRET cmnGetMessage(PCHAR *pTable,     // in: like DosGetMessage pTable
                         ULONG ulTable,     // in: like DosGetMessage cTable
                         PSZ pszBuf,        // out: like DosGetMessage pBuf
                         ULONG cbBuf,       // in: like DosGetMessage cbBuf
                         ULONG ulMsgNumber);// in: like DosGetMessage msgnumber

    /*
     * cmnMessageBoxMsg:
     *      calls cmnMessageBox, but this one accepts ULONG indices
     *      into the XFolder message file (XFLDRxxx.MSG) instead
     *      of real PSZs.
     */

    ULONG cmnMessageBoxMsg(HWND hwndOwner,
                           ULONG ulTitle,       // in: msg index for dlg title
                           ULONG ulMessage,     // in: msg index for message
                           ULONG flStyle);      // in: like cmnMsgBox

    /*
     * cmnMessageBoxMsgExt:
     *      like CmnMessageBoxMsg, but with string substitution
     *      (see DosGetMessage docs for more); substitution only
     *      takes place for the message specified with ulMessage,
     *      not for the title
     */

    ULONG cmnMessageBoxMsgExt(HWND hwndOwner, ULONG ulTitle,
            PCHAR *pTable, ULONG ulTable, ULONG ulMessage, ULONG flStyle);

    /*
     * cmnSetHelpPanel:
     *      sets help panel before calling fnwpDlgGeneric
     */

    VOID cmnSetHelpPanel(ULONG ulHelpPanel);

    /*
     *  fnwpDlgGeneric:
     *          this is the dlg procedure for XFolder dlg boxes;
     *          it can process WM_HELP messages
     */

    MRESULT EXPENTRY fnwpDlgGeneric(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);

    /********************************************************************
     *                                                                  *
     *   System sounds                                                  *
     *                                                                  *
     ********************************************************************/

    /*
     * cmnQuerySystemSound:
     *      this gets a system sound from the MMPM.INI file.
     *      Parameters:
     *      usIndex: index of sound to query:
     *          MMSOUND_WARNING         0
     *          MMSOUND_INFORMATION     1
     *          MMSOUND_ERROR           2
     *          MMSOUND_ANIMATEOPEN     3
     *          MMSOUND_ANIMATECLOSE    4
     *          MMSOUND_DRAG            5
     *          MMSOUND_DROP            6
     *          MMSOUND_SYSTEMSTARTUP   7
     *          MMSOUND_SHUTDOWN        8
     *          MMSOUND_SHREDDER        9
     *          MMSOUND_LOCKUP          10
     *          MMSOUND_ALARMCLOCK      11
     *          MMSOUND_PRINTERROR      12
     *      New XFolder system sounds:
     *          MMSOUND_XFLD_SHUTDOWN   555
     *          MMSOUND_XFLD_RESTARTWPS 556
     *          MMSOUND_XFLD_SORT       557
     *      pszDescr: address of buffer to where to return the
     *                sound description as in the "Sound" object;
     *                may be NULL
     *      pszFile:  address of buffer to where to return the
     *                sound file; may be NULL
     *      pulVolume: address of ULONG to where to return the
     *                 volume (0-100); if the "Global volume"
     *                 flag is set, the global volume will be
     *                 returned, otherwise the individual sound
     *                 volume
     *      The string buffers are recommended to be at least
     *      CCHMAXPATH in size.
     */

    BOOL cmnQuerySystemSound(USHORT usIndex, PSZ pszDescr, PSZ pszFile, PULONG pulVolume);

    /*
     * cmnSetSystemSound:
     *      reversely, this SETS a system sound in MMPM.INI.
     */

    BOOL cmnSetSystemSound(USHORT usIndex, PSZ pszDescr, PSZ pszFile, BYTE bVolume);

    /********************************************************************
     *                                                                  *
     *   Various other function prototypes                              *
     *                                                                  *
     ********************************************************************/

    /*
     *  The following dialog functions are not present in
     *  common.c, but in other XFolder source files.
     *  We prototype them here anyway, because this file is
     *  #include'd by most XFolder modules.
     */

    // fwd dcl: dlg procs for settings notebook pages (xfsys.c);
    // some of these need to be fwd-declared for xfdesk.c

    MRESULT EXPENTRY fnwpSettingsXFolderInternals(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
    MRESULT EXPENTRY fnwpSettingsStatusBars1(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);

    MRESULT EXPENTRY fnwpSettingsObjDetails(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);

    /*
     *  Here come miscellaneous other dialog funcs.
     */

    // fwd dcl: wnd procs for subclassed windows (xfldr.c)
    MRESULT EXPENTRY fnwpFolderContentMenu(HWND hwndMenu, ULONG msg, MPARAM mp1, MPARAM mp2);
    MRESULT EXPENTRY fnwpSubclassedFolderFrame(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);
    MRESULT EXPENTRY fnwpStatusBar(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);

    MRESULT EXPENTRY fnwpSelectSome(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);

    // callback function in xfldr.c for Worker thread (in xfdesk.c)
    // to update all status bars
    MRESULT EXPENTRY fncbUpdateStatusBars(HWND hwndView, ULONG ulActivate,
                    MPARAM mpView, MPARAM mpFolder);

    /*
     *  A few MACROS are also declared here:
     */

    /*
     * NLS_MODULE:
     *      this macro is just a shortcut to querying the
     *      language-dependent XFolder DLL handle
     */

    #define NLS_MODULE cmnQueryNLSModuleHandle(FALSE)

    /*
     * DEFAULT_SORT:
     *      returns a useable setting for the default sort criterion
     *      according to instance / global settings; before using this
     *      macro, you need to initialize the following:
     *          PGLOBALSETTINGS     pGlobalSettings
     *          XFolderData         *somThis
     */

    #define DEFAULT_SORT ((_DefaultSort == SET_DEFAULT) ? pGlobalSettings->DefaultSort : _DefaultSort)

    /*
     * ALWAYS_SORT:
     *      the same for AlwaysSort
     */

    #define ALWAYS_SORT ((_AlwaysSort == SET_DEFAULT) ? pGlobalSettings->AlwaysSort : _AlwaysSort)


#endif
