/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "mathutil.h"
#include "Printerr.h"
#include "texcache.h"
#include "util.h"

// the texture cache will automatically defragment memory
// when it has this many fragmented areas
#define AUTO_DEFRAG  666

typedef struct
{
	FxU8 blue;
	FxU8 green;
	FxU8 red;
} PaletteEntry;

typedef struct
{
	FxU8 blue;
	FxU8 green;
	FxU8 red;
	FxU8 alpha;
} ARGB32bpp;

// loads a Targa file (.tga)
// fills out the tex_info structure
// it converts the texture to have one of the following formats:
// GR_TEXFMT_ARGB_4444
// GR_TEXFMT_ALPHA_INTENSITY_88
// GR_TEXFMT_RGB_565 (chooses this format if the specified format isn't one of the above)
// if the format contains alpha, the alpha parameter is used
FxBool LoadGlideTextureTGA(char *filename, GrTexInfo *tex_info, GrTextureFormat_t tex_format, FxU8 alpha, FxU8 lods)
{
	FILE *fp;
	TGA_Header header;
	PaletteEntry palette[256];
	FxU32 i, j, image_size, total_size;
	FxU32 width, height, log_width, log_height;
	void *tmp_data;
	FxU16 *data, *curr_data;
	ARGB32bpp *texels, *curr_texel;
	FxU32 s, t, s0, t0, index, next_row, curr_width, curr_height;
	FxU32 total_alpha, total_red, total_green, total_blue;

	if (!(fp = fopen(filename, "rb")))
	{
		return FXFALSE;
	}

	if (fread(&header, sizeof(TGA_Header), 1, fp) != 1)
	{
		fclose(fp);
		return FXFALSE;
	}

	// calculate log width
	width = (header.width_hi<<8) | header.width_lo;
	log_width = 0;
	while (width >>= 1)
	{
		log_width++;
	}
	width = 1<<log_width; // width must be power of 2

	// calculate log height
	height = (header.height_hi<<8) | header.height_lo;
	log_height = 0;
	while (height >>= 1)
	{
		log_height++;
	}
	height = 1<<log_height; // height must be power of 2

	// check for bad textures
	if (header.image_type >= TGA_CMAP_RLE || // can't handle RLE compression
			log_width > 8 || log_height > 8 || // can't handle textures bigger than 256 in width or height
			(unsigned)((header.width_hi<<8) | header.width_lo) != width || // can't handle non-power of 2 width or...
			(unsigned)((header.height_hi<<8) | header.height_lo) != height) // ...non-power of 2 height
	{
		fclose(fp);
		return FXFALSE;
	}

	image_size = 1<<(log_width+log_height);

	// find the size of the whole mipmap
	total_size = image_size;
	curr_width = width;
	curr_height = height;
	for (j=1; j<=lods; j++)
	{
		curr_width = MAX(1, curr_width>>1);
		curr_height = MAX(1, curr_height>>1);

		total_size += curr_width*curr_height;
	}

	// store the texture in 32bpp format
	// to be used later for converting to other formats
	// and creating mipmaps
	texels = new ARGB32bpp[total_size];
	if (!texels)
	{
		fclose(fp);
		return FXFALSE;
	}

	// convert the texture from 8, 16, or 24 bpp to a 32 bpp argb format
	// ******** 8 bpp ********
	if (header.bpp == 8)
	{
		FxU32 num_entries;
		if (!header.b_color_mapped || header.color_map_entry_size != 24)
		{
			delete [] texels;
			fclose(fp);
			return FXFALSE;
		}
		// read in the palette
		num_entries = (header.color_map_entries_hi<<8) | header.color_map_entries_lo;
		if (fread(&palette[0], sizeof(PaletteEntry), num_entries, fp) != num_entries)
		{
			delete [] texels;
			fclose(fp);
			return FXFALSE;
		}
		tmp_data = new FxU8[image_size];
		if (!tmp_data)
		{
			delete [] texels;
			fclose(fp);
			return FXFALSE;
		}
		if (fread(tmp_data, sizeof(FxU8), image_size, fp) != image_size)
		{
			delete [] texels;
			delete [] tmp_data;
			fclose(fp);
			return FXFALSE;
		}
		// convert the 8 bpp color mapped texture to a 32 bpp ARGB-8888 texture
		for (i=0; i<image_size; i++)
		{
			texels[i].red   = palette[((FxU8 *)tmp_data)[i]].red;
			texels[i].green = palette[((FxU8 *)tmp_data)[i]].green;
			texels[i].blue  = palette[((FxU8 *)tmp_data)[i]].blue;
			texels[i].alpha = alpha;
		}
		delete [] tmp_data;
	}
	// ******** 16 bpp ********
	else if (header.bpp == 16)
	{
		tmp_data = new FxU16[image_size];
		if (!tmp_data)
		{
			delete [] texels;
			fclose(fp);
			return FXFALSE;
		}
		if (fread(tmp_data, sizeof(FxU16), image_size, fp) != image_size)
		{
			delete [] texels;
			delete [] tmp_data;
			fclose(fp);
			return FXFALSE;
		}
		// convert the 15 bpp RGB-555 texture to a 32 bpp ARGB-8888 texture
		for (i=0; i<image_size; i++)
		{
			texels[i].red   = (FxU8)( (float)((((FxU16 *)tmp_data)[i] & 0x7c00)>>10)*(256.0f/32.0f) );
			texels[i].green = (FxU8)( (float)((((FxU16 *)tmp_data)[i] & 0x03e0)>>5 )*(256.0f/32.0f) );
			texels[i].blue  = (FxU8)( (float)((((FxU16 *)tmp_data)[i] & 0x001f)    )*(256.0f/32.0f) );
			texels[i].alpha = alpha;
		}
		delete [] tmp_data;
	}
	// ******** 24 bpp ********
	else if (header.bpp == 24)
	{
		tmp_data = new PaletteEntry[image_size];
		if (!tmp_data)
		{
			delete [] texels;
			fclose(fp);
			return FXFALSE;
		}
		if (fread(tmp_data, sizeof(PaletteEntry), image_size, fp) != image_size)
		{
			delete [] texels;
			delete [] tmp_data;
			fclose(fp);
			return FXFALSE;
		}
		// convert the 24 bpp RGB-888 texture to a 32 bpp ARGB-8888 texture
		for (i=0; i<image_size; i++)
		{
			texels[i].red   = ((PaletteEntry *)tmp_data)[i].red;
			texels[i].green = ((PaletteEntry *)tmp_data)[i].green;
			texels[i].blue  = ((PaletteEntry *)tmp_data)[i].blue;
			texels[i].alpha = alpha;
		}
		delete [] tmp_data;
	}
	// ******** 32 bpp ********
	else if (header.bpp == 32)
	{
		// read in the 32 bpp BGRA-8888 texture
		if (fread(texels, sizeof(ARGB32bpp), image_size, fp) != image_size)
		{
			delete [] texels;
			fclose(fp);
			return FXFALSE;
		}
	}
	else // unknown bpp, fail!!!!
	{
		delete [] texels;
		fclose(fp);
		return FXFALSE;
	}

	// we're done with the file, close it
	fclose(fp);

	// data will be 16 bpp
	data = new FxU16[total_size];
	if (!data)
	{
		delete [] texels;
		fclose(fp);
		return FXFALSE;
	}

	// create the mipmaps while we're still in 32 bpp
	curr_texel = &texels[image_size];
	curr_width = width;
	curr_height = height;
	for (j=1; j<=lods; j++)
	{
		curr_width = MAX(1, curr_width>>1);
		curr_height = MAX(1, curr_height>>1);

		next_row = width - (1<<j);

		for (t=0; t<curr_height; t++)
		{
			for (s=0; s<curr_width; s++)
			{
				// find the average color from the texture (LOD0)
				total_alpha = 0;
				total_red = 0;
				total_green = 0;
				total_blue = 0;
				index = (t<<(j+log_width)) + (s<<j);
				for (t0=(1<<j); t0>0; t0--)
				{
					for (s0=(1<<j); s0>0; s0--)
					{
						total_alpha += texels[index].alpha;
						total_red   += texels[index].red;
						total_green += texels[index].green;
						total_blue  += texels[index].blue;
						index++;
					}
					index += next_row;
				}
				// avg_color = total_color/num_texels
				// num_texels = 1<<(j+j), so avg_color = total_color>>(j+j)
				curr_texel->alpha = (FxU8)(total_alpha>>(j+j));
				curr_texel->red   = (FxU8)(total_red>>(j+j));
				curr_texel->green = (FxU8)(total_green>>(j+j));
				curr_texel->blue  = (FxU8)(total_blue>>(j+j));

				curr_texel++;
			}
		}
	}

	// convert the 32 bpp argb to the specified format
	curr_data = data;
	curr_texel = texels;
	if (tex_format == GR_TEXFMT_ARGB_4444)
	{
		// convert to a 16 bpp ARGB-4444 texture
		for (i=total_size; i>0; i--, curr_data++, curr_texel++)
		{
			*curr_data = ((curr_texel->alpha>>4)<<12) |
									 ((curr_texel->red  >>4)<< 8) |
									 ((curr_texel->green>>4)<< 4) |
										(curr_texel->blue >>4);
		}
	}
	else if (tex_format == GR_TEXFMT_ALPHA_INTENSITY_88)
	{
		// convert to a 16 bpp AI-88 texture
		for (i=total_size; i>0; i--, curr_data++, curr_texel++)
		{
			*curr_data = (curr_texel->alpha<<8) |
									 (curr_texel->red);
		}
	}
	else // GR_TEXFMT_RGB_565
	{
		tex_format = GR_TEXFMT_RGB_565;
		// convert to a 16 bpp RGB-565 texture
		for (i=total_size; i>0; i--, curr_data++, curr_texel++)
		{
			*curr_data = ((curr_texel->red  >>3)<<11) |
									 ((curr_texel->green>>2)<< 5) |
										(curr_texel->blue >>3);
		}
	}

	// we don't need the 32 bpp texels anymore, delete them
	delete [] texels;

#ifdef USE_GLIDE3
	tex_info->largeLodLog2 = MAX(log_width, log_height);
	tex_info->smallLodLog2 = tex_info->largeLodLog2 - lods;
	tex_info->aspectRatioLog2 = (log_width - log_height);
#else
	tex_info->largeLod = GR_LOD_1 - MAX(log_width, log_height);
	tex_info->smallLod = tex_info->largeLod + lods;
	tex_info->aspectRatio = GR_ASPECT_1x1 - (log_width - log_height);
#endif // USE_GLIDE3
	tex_info->format = tex_format;
	tex_info->data = data;

	return FXTRUE;
}

// takes a square texture and maps it to a sphere
// the original texture data is replaced with the
// new spherized data (which is the same size)
FxBool SpherizeTexture(GrTexInfo *tex_info)
{
	FxU16 *src, *dest_data, *src_data;
	float dist_sqr, radius, radius_sqr, dx, dy, dist_ratio, radius_inv;
	int u, v, width;

	// must be a square texture
#ifdef USE_GLIDE3
	if (tex_info->aspectRatioLog2 != GR_ASPECT_LOG2_1x1)
	{
		return FXFALSE;
	}
	width = 1<<(tex_info->largeLodLog2);
#else
	if (tex_info->aspectRatio != GR_ASPECT_1x1)
	{
		return FXFALSE;
	}
	width = 256>>(tex_info->largeLod);
#endif // USE_GLIDE3

	src = new FxU16[width*width];
	if (!src)
	{
		return FXFALSE;
	}
	memcpy(src, tex_info->data, sizeof(FxU16)*width*width);
	src_data = src;
	dest_data = (FxU16 *)tex_info->data;

	radius = 0.5f*width;
	radius_sqr = radius*radius;
	radius_inv = 1.0f/radius;

	for (dy=-radius; dy<radius; dy+=1.0f)
	{
		for (dx=-radius; dx<radius; dx+=1.0f)
		{
			dist_sqr = (dx*dx) + (dy*dy);
			if (dist_sqr <= radius_sqr)
			{
				dist_ratio = fsqrt(dist_sqr)*radius_inv;
				u = (int)(radius + dist_ratio*dx);
				v = (int)(radius + dist_ratio*dy);
				*dest_data = src[v*width + u];
			}
			else
			{
				*dest_data = 0;
			}
			dest_data++;
		}
	}

	delete [] src;

	return FXTRUE;
}

// ******************************
// ****  class GlideTexture  ****
// ******************************
GlideTexture::GlideTexture()
{
	FxU32 i;

	memset(&tex_info, 0, sizeof(GrTexInfo));
	memory_required = 0;
	for (i=0; i<MAX_TMUS; i++)
	{
		cache_info[i-GR_TMU0].start_address = 0xffffffff;
		cache_info[i-GR_TMU0].b_cached = FXFALSE;
		cache_info[i-GR_TMU0].time_last_used = 0;
		cache_info[i-GR_TMU0].prev = NULL;
		cache_info[i-GR_TMU0].next = NULL;
	}
}

GlideTexture::~GlideTexture()
{
	FxU32 i;

	memset(&tex_info, 0, sizeof(GrTexInfo));
	memory_required = 0;
	for (i=0; i<MAX_TMUS; i++)
	{
		cache_info[i-GR_TMU0].start_address = 0xffffffff;
		cache_info[i-GR_TMU0].b_cached = FXFALSE;
		cache_info[i-GR_TMU0].time_last_used = 0;
		cache_info[i-GR_TMU0].prev = NULL;
		cache_info[i-GR_TMU0].next = NULL;
	}
}

// ******************************
// ****  class TextureCache  ****
// ******************************
TextureCache::TextureCache(GrChipID_t tmu)
{
	m_tmu = tmu;
	m_fragments = 0;
	m_start_tex_list = NULL;
	m_end_tex_list = NULL;
}

TextureCache::~TextureCache()
{
	m_fragments = 0;
	m_start_tex_list = NULL;
	m_end_tex_list = NULL;
}

// deletes the memory allocated for the texture data
// and removes texture from the texture cache list
void TextureCache::DestroyGlideTexture(GlideTexture *texture)
{
	RemoveGlideTexture(texture);
	// see if removing this texture changed the fragmentation
	m_fragments = CountFragments();
	delete [] texture->tex_info.data;
	memset(texture, 0, sizeof(GlideTexture));
}

// it adds the texture to the texture cache
// 1) if there is room at the end of the texture cache it'll put it there, otherwise...
// 2) it will see if there is any fragmented memory that can fit the texture, otherwise...
// 3) it tries to find a texture of the same size or bigger, that was least
//    recently used to be replaced by this texture, otherwise...
// 4) it will remove a few textures from the head of the texture cache list until
//    there is enough room to put in the new texture there
// * it makes sure no texture spans the 2MB boundary
void TextureCache::CacheTexture(GlideTexture *texture)
{
	FxU32 tmu_index, min_address, max_address, curr_open_address;
	FxU32 memory_required, start_address, boundary_2MB;
	FxI32 time_last_used;
	GlideTexture *tex, *lru_texture;

	tmu_index = m_tmu - GR_TMU0;

	// make sure it isn't already in the cache
	if (texture->cache_info[tmu_index].b_cached)
	{
		return;
	}

	min_address = grTexMinAddress(m_tmu);
	max_address = grTexMaxAddress(m_tmu) +
#ifdef USE_GLIDE3
		grTexCalcMemRequired(GR_LOD_LOG2_1, GR_LOD_LOG2_1, GR_ASPECT_LOG2_1x1, GR_TEXFMT_RGB_565);
#else
		grTexCalcMemRequired(GR_LOD_1, GR_LOD_1, GR_ASPECT_1x1, GR_TEXFMT_RGB_565);
#endif // USE_GLIDE3

	boundary_2MB = min_address + (1<<21);

	if (m_end_tex_list)
	{
		curr_open_address = m_end_tex_list->cache_info[tmu_index].start_address + m_end_tex_list->memory_required;
	}
	else
	{
		curr_open_address = min_address;
	}

	start_address = 0xffffffff;
	memory_required = grTexTextureMemRequired(GR_MIPMAPLEVELMASK_BOTH, &texture->tex_info);

	// see if it would cross the 2M boundary
	if (curr_open_address < boundary_2MB && curr_open_address+memory_required > boundary_2MB)
	{
		curr_open_address = boundary_2MB;
	}

	// see if it fits at the end of the texture cache
	if (memory_required <= max_address - curr_open_address)
	{
		start_address = curr_open_address;
		// append the texture to the list
		AppendGlideTexture(texture);
		PrintError("CACHE: fit just fine at the end\n");
	}
	else
	{
		PrintError("CACHE: didn't fit at end: ");

		// see if there's fragmentation, if so, see if
		// we can squeez this texture in some fragmented area
		if (m_fragments)
		{
			FxU32 memory_fragmented, address, fragments_remaining;

			fragments_remaining = m_fragments;
			for (tex = m_start_tex_list; tex->cache_info[tmu_index].next; tex = tex->cache_info[tmu_index].next)
			{
				address = tex->cache_info[tmu_index].start_address + tex->memory_required;

				// see if there's fragmentation here
				if (address == tex->cache_info[tmu_index].next->cache_info[tmu_index].start_address)
				{
					continue;
				}

				fragments_remaining--;

				// check the 2MB boundary
				if (address < boundary_2MB && address+memory_required > boundary_2MB)
				{
					// we need to use texture memory after the 2MB boundary, but
					// the next texture starts before the 2MB boundary, so...
					if (tex->cache_info[tmu_index].next->cache_info[tmu_index].start_address < boundary_2MB)
					{
						// if there are more fragments remaining, keep checking
						if (fragments_remaining)
						{
							continue;
						}
						// otherwise, we're done, no fragments big enough for this texture
						break;
					}
					address = boundary_2MB;
				}

				memory_fragmented = tex->cache_info[tmu_index].next->cache_info[tmu_index].start_address - address;

				if (memory_fragmented >= memory_required)
				{
					start_address = address;
					break;
				}
			}

			// if there was room, insert this texture into the list
			if (start_address != 0xffffffff)
			{
				texture->cache_info[tmu_index].next = tex->cache_info[tmu_index].next;
				texture->cache_info[tmu_index].prev = tex;
				tex->cache_info[tmu_index].next->cache_info[tmu_index].prev = texture;
				tex->cache_info[tmu_index].next = texture;
				// re-count the fragments, we might have gotten rid of a fragment
				m_fragments = CountFragments();
				PrintError("putting it in fragmented memory: %d - %d\n", start_address, start_address+memory_required);
			}
			else
			{
				// auto-defrag if too fragmented
				if (m_fragments >= AUTO_DEFRAG)
				{
					DefragmentTexCache();
					PrintError("DEFRAGING....\n");
				}
			}
		}

		// if there wasn't any fragmentation or if there wasn't
		// enough room in the fragmented area for this texture then
		// we need to find another texture in cache to remove
		if (start_address == 0xffffffff)
		{
			lru_texture = NULL;
			time_last_used = -1;
			// find the least recently used texture with the same size or bigger
			for (tex = m_start_tex_list; tex; tex = tex->cache_info[tmu_index].next)
			{
				if (tex->memory_required >= memory_required)
				{
					// see if it hasn't been used as recently
					if (tex->cache_info[tmu_index].time_last_used < time_last_used)
					{
						// check the 2MB boundary
						if (!(tex->cache_info[tmu_index].start_address < boundary_2MB && tex->cache_info[tmu_index].start_address+memory_required > boundary_2MB))
						{
							time_last_used = tex->cache_info[tmu_index].time_last_used;
							lru_texture = tex;
						}
					}
				}
			}

			// we found the least recently used texture to replace
			if (lru_texture)
			{
				start_address = lru_texture->cache_info[tmu_index].start_address;
				// replace texture for lru_texture in the list
				if (lru_texture->cache_info[tmu_index].prev)
				{
					lru_texture->cache_info[tmu_index].prev->cache_info[tmu_index].next = texture;
				}
				if (lru_texture->cache_info[tmu_index].next)
				{
					lru_texture->cache_info[tmu_index].next->cache_info[tmu_index].prev = texture;
				}
				texture->cache_info[tmu_index].next = lru_texture->cache_info[tmu_index].next;
				texture->cache_info[tmu_index].prev = lru_texture->cache_info[tmu_index].prev;
				if (lru_texture == m_start_tex_list)
				{
					m_start_tex_list = texture;
				}
				if (lru_texture == m_end_tex_list)
				{
					m_end_tex_list = texture;
				}

				lru_texture->cache_info[tmu_index].prev = NULL;
				lru_texture->cache_info[tmu_index].next = NULL;

				lru_texture->cache_info[tmu_index].start_address = 0xffffffff;
				lru_texture->cache_info[tmu_index].b_cached = FXFALSE;
				lru_texture->cache_info[tmu_index].time_last_used = 0;

				// see if we have caused fragmentation by removing this texture
				if (lru_texture->memory_required != memory_required)
				{
					m_fragments = CountFragments();
				}
				PrintError("replaced, caused fragmentation? %s\n", (lru_texture->memory_required != memory_required) ? "YES" : "NO");
			}
			// if we didn't find a texture of the same size or bigger
			// we'll have to remove multiple textures in continous memory
			// I'll just remove textures from the head of the list since
			// I don't want to deal with the 2MB boundary
			else
			{
				tex = m_start_tex_list;
				while (tex)
				{
					lru_texture = tex;
					tex = tex->cache_info[tmu_index].next;
					RemoveGlideTexture(lru_texture);
					// see if there's room now in the beginning of the texture cache
					if (m_start_tex_list->cache_info[tmu_index].start_address - min_address >= memory_required)
					{
						// add the texture to the front of the list
						start_address = min_address;
						texture->cache_info[tmu_index].prev = NULL;
						texture->cache_info[tmu_index].next = m_start_tex_list;
						m_start_tex_list->cache_info[tmu_index].prev = texture;
						if (m_end_tex_list == m_start_tex_list)
						{
							m_end_tex_list = texture;
						}
						m_start_tex_list = texture;
						break;
					}
				}
				// re-count the fragments
				m_fragments = CountFragments();
				PrintError("removed a bunch of textures to make this one fit\n");
			}
		} // if (start_address == 0xffffffff) // no fragmented area open
	} // !(see if it fits at the end of the texture cache)

	// now that we've made room, let's add the texture to the cache
	texture->memory_required = memory_required;
	texture->cache_info[tmu_index].start_address = start_address;
	texture->cache_info[tmu_index].b_cached = FXTRUE;
	texture->cache_info[tmu_index].time_last_used = os2_timeGetTime(); // we will time stamp it so it doesn't emmidiately get replaced
	grTexDownloadMipMap(m_tmu, texture->cache_info[tmu_index].start_address, GR_MIPMAPLEVELMASK_BOTH, &texture->tex_info);
}

// this is to be used when the texture data changes
// if the texture isn't already cached, it'll become cached
// otherwise, just the new data is downloaded
void TextureCache::ReloadTexture(GlideTexture *texture)
{
	FxU32 tmu_index;

	tmu_index = m_tmu - GR_TMU0;

	if (texture->cache_info[tmu_index].b_cached)
	{
		grTexDownloadMipMap(m_tmu, texture->cache_info[tmu_index].start_address, GR_MIPMAPLEVELMASK_BOTH, &texture->tex_info);
	}
	else
	{
		CacheTexture(texture);
	}
}

void TextureCache::TexSource(GlideTexture *texture)
{
	FxU32 tmu_index;

	tmu_index = m_tmu - GR_TMU0;

	if (texture->cache_info[tmu_index].b_cached)
	{
		grTexSource(m_tmu, texture->cache_info[tmu_index].start_address, GR_MIPMAPLEVELMASK_BOTH, &texture->tex_info);
	}
	else
	{
		PrintError("Cache miss!!!!\n");
		CacheTexture(texture);
		grTexSource(m_tmu, texture->cache_info[tmu_index].start_address, GR_MIPMAPLEVELMASK_BOTH, &texture->tex_info);
	}

	// time stamp it for lru cache
	texture->cache_info[tmu_index].time_last_used = os2_timeGetTime();
}

void TextureCache::DefragmentTexCache()
{
	GlideTexture *texture;
	FxU32 tmu_index, curr_open_address, boundary_2MB;

	if (!m_fragments)
	{
		return;
	}

	tmu_index = m_tmu - GR_TMU0;

	boundary_2MB = grTexMinAddress(m_tmu) + (1<<21);
	curr_open_address = grTexMinAddress(m_tmu);

	for (texture = m_start_tex_list; texture != NULL; texture = texture->cache_info[tmu_index].next)
	{
		// make sure no texture spans the 2MB boundary
		if (curr_open_address < boundary_2MB && curr_open_address+texture->memory_required > boundary_2MB)
		{
			curr_open_address = boundary_2MB;
		}
		texture->cache_info[tmu_index].start_address = curr_open_address;
		curr_open_address += texture->memory_required;
		grTexDownloadMipMap(m_tmu, texture->cache_info[tmu_index].start_address, GR_MIPMAPLEVELMASK_BOTH, &texture->tex_info);
	}

	// count the fragments, we might still have fragmentation at
	// the 2MB boundary
	m_fragments = CountFragments();
}

void TextureCache::AppendGlideTexture(GlideTexture *texture)
{
	FxU32 tmu_index;

	tmu_index = m_tmu - GR_TMU0;

	if (m_start_tex_list)
	{
		texture->cache_info[tmu_index].prev = m_end_tex_list;
		texture->cache_info[tmu_index].next = NULL;
		m_end_tex_list->cache_info[tmu_index].next = texture;
		m_end_tex_list = texture;
	}
	else
	{
		texture->cache_info[tmu_index].prev = NULL;
		texture->cache_info[tmu_index].next = NULL;
		m_start_tex_list = texture;
		m_end_tex_list = texture;
	}
}

void TextureCache::RemoveGlideTexture(GlideTexture *texture)
{
	FxU32 tmu_index;

	tmu_index = m_tmu - GR_TMU0;

	// if the texture isn't cached, then it's not in the list
	if (!texture->cache_info[tmu_index].b_cached)
	{
		return;
	}

	if (texture->cache_info[tmu_index].prev)
	{
		texture->cache_info[tmu_index].prev->cache_info[tmu_index].next = texture->cache_info[tmu_index].next;
	}
	if (texture->cache_info[tmu_index].next)
	{
		texture->cache_info[tmu_index].next->cache_info[tmu_index].prev = texture->cache_info[tmu_index].prev;
	}
	if (texture == m_start_tex_list)
	{
		m_start_tex_list = texture->cache_info[tmu_index].next;
	}
	if (texture == m_end_tex_list)
	{
		if (texture->cache_info[tmu_index].prev)
		{
			m_end_tex_list = texture->cache_info[tmu_index].prev;
		}
		else
		{
			m_end_tex_list = m_start_tex_list;
		}
	}

	texture->cache_info[tmu_index].prev = NULL;
	texture->cache_info[tmu_index].next = NULL;

	texture->cache_info[tmu_index].start_address = 0xffffffff;
	texture->cache_info[tmu_index].b_cached = FXFALSE;
	texture->cache_info[tmu_index].time_last_used = 0;
}

int TextureCache::CountFragments()
{
	GlideTexture *tex;
	FxU32 tmu_index, fragments;

	if (!m_start_tex_list)
	{
		return 0;
	}

	tmu_index = m_tmu - GR_TMU0;

	fragments = 0;
	for (tex = m_start_tex_list; tex->cache_info[tmu_index].next; tex = tex->cache_info[tmu_index].next)
	{
		// if there is a gap between the end of this texture memory and the beginning of the next texture memory
		// increment the number of fragments
		if (tex->cache_info[tmu_index].start_address+tex->memory_required < tex->cache_info[tmu_index].next->cache_info[tmu_index].start_address)
		{
			fragments++;
		}
	}

	return fragments;
}

void TextureCache::TexCacheDump()
{
	GlideTexture *tex;
	FxU32 tmu_index, i;

	tmu_index = m_tmu - GR_TMU0;

	PrintError("*********************************\n");
	for (tex=m_start_tex_list, i=0; tex; tex=tex->cache_info[tmu_index].next, i++)
	{
		PrintError("#%d: %d - %d\n", i, tex->cache_info[tmu_index].start_address, tex->cache_info[tmu_index].start_address+tex->memory_required);
	}
	PrintError("*********************************\n");
}
