/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "util.h"
#include "mathutil.h"
#include "texcache.h"
#include "crystal.h"
#include "torus.h"
#include "lighting.h"
#include "Printerr.h"
#include "intro.h"
#include "scrncptr.h"
#include "tlib.h"


#define INTRO_ENABLED  1
#define STEREO_VIEW_ENABLED  0

#define MOUSE_SENSITIVITY (0.5f)
#define NUM_BUMPMAPS  5  // was 6

// mouse state
static FxU8 gMouseState;
static POINTL gMousePos1, gMousePos2, gMousePos3;
#define LEFT_MOUSE_BUTTON_DOWN   0x1
#define MIDDLE_MOUSE_BUTOTN_DOWN 0x2
#define RIGHT_MOUSE_BUTTON_DOWN  0x4

// keyboard state
#define VK_MINUS  0xbd // -
#define VK_EQUAL  0xbb // =
static FxU32 gKeyboardState;
#define SHIFT_KEY_DOWN      0x1
#define CONTROL_KEY_DOWN 0x2
static int g_bExitApp = FALSE;

// global variables
int SCREEN_RES, SCREEN_REFRESH;
float SCREEN_RES_X, SCREEN_RES_Y;
int HI_SCREEN_RES, HI_SCREEN_RES_X, HI_SCREEN_RES_Y;
int LO_SCREEN_RES, LO_SCREEN_RES_X, LO_SCREEN_RES_Y;

#ifdef USE_GLIDE3
GrContext_t gContext;
#endif // USE_GLIDE3

int gNumTMUsUsed, gRenderMode, gCurrBumpmap, gSpecialBlend;
GlideTexture gTexture, gEnvmapTexture, gBackgroundTexture, gBumpmapTexture[NUM_BUMPMAPS];
int gDetail, gDisplayBackground, gWireframe, gDrawNormals, gVsynch, gExtraInfoDisplay;
float gInnerRadius, gChromatic, gDepth, gBumpmapExcessAlpha, gDefaultDepths[NUM_BUMPMAPS];
Matrix gProjMat;
CrystalBall gObjectCrystalBall, gLightCrystalBall, *gCurrCrystalBall;
char gTextureFileName[64], gBackgroundFileName[64];
FxBool gHelpMenu;
Vector gLightDir, gCameraDir;
int gNumScreenShots, gNumScreensToCapture, gScreenCaptureEnabled;

#if STEREO_VIEW_ENABLED
FxBool gStereoView;
float gStereoViewAngle;
#endif

// lighting
float gLightAmbientFactor =   32.0f; // (ambient + diffuse) must be <= 255
float gLightDiffuseFactor =  192.0f;
float gLightSpecularFactor = 128.0f;

// local function prototypes
FxBool WinOpenHighestResolution(int *resolution, int *res_x, int *res_y);
int RenderTorus();
void Render();
void InitGlideState();
FxBool LoadAndCacheTextures();
void SetTextureAlpha(GlideTexture *texture, FxU8 alpha);
void DestroyTextures();
void SetupCrystalBallRotation(CrystalBall *crystal_ball, POINTL pos1, POINTL pos2);
MRESULT APIENTRY UIMain(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);

int main(int argc, char *argv[])
{
	int i;
	FxBool set_resolution;
	char str[256];

	gNumScreenShots = 0;
	gNumScreensToCapture = 0;

	gSpecialBlend = 1;
	gVsynch = 1;
	gHelpMenu = FXFALSE;
	gExtraInfoDisplay = 1;
#if STEREO_VIEW_ENABLED
	gStereoView = FXFALSE;
	gStereoViewAngle = DEG_TO_RAD(6.0f);
#endif

	gLightDir[X] = 1.0f;
	gLightDir[Y] = 1.0f;
	gLightDir[Z] = 2.0f;
	gLightDir[W] = 0.0f;
	gCameraDir[X] = 0.0f;
	gCameraDir[Y] = 0.0f;
	gCameraDir[Z] = 1.0f;
	gCameraDir[W] = 0.0f;

	strcpy(gTextureFileName, "data/tex.dat");
	strcpy(gBackgroundFileName, "data/bg.dat");
	set_resolution = FXFALSE;

	gScreenCaptureEnabled = 0;
	SCREEN_REFRESH = GR_REFRESH_NONE;

	// parse the arguments
	for (i=1; i<argc; i++)
	{
		if (stricmp(argv[i], "-res") == 0)
		{
			if (argc <= i+1)
			{
				printf("resolution not specified");
				return 0;
			}

			i++;

			set_resolution = FXTRUE;

			if (strcmp(argv[i], "1600x1200") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_1600x1200;
				HI_SCREEN_RES_X = 1600;
				HI_SCREEN_RES_Y = 1200;
			}
			else if (strcmp(argv[i], "1280x1024") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_1280x1024;
				HI_SCREEN_RES_X = 1280;
				HI_SCREEN_RES_Y = 1024;
			}
			else if (strcmp(argv[i], "1024x768") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_1024x768;
				HI_SCREEN_RES_X = 1024;
				HI_SCREEN_RES_Y = 768;
			}
			else if (strcmp(argv[i], "800x600") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_800x600;
				HI_SCREEN_RES_X = 800;
				HI_SCREEN_RES_Y = 600;
			}
			else if (strcmp(argv[i], "640x480") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_640x480;
				HI_SCREEN_RES_X = 640;
				HI_SCREEN_RES_Y = 480;
			}
			else if (strcmp(argv[i], "512x384") == 0)
			{
				HI_SCREEN_RES = GR_RESOLUTION_512x384;
				HI_SCREEN_RES_X = 512;
				HI_SCREEN_RES_Y = 384;
			}
			else
			{
				sprintf(str, "%s is not a valid resolution.\nValid options:\n512x384\n640x480\n800x600\n1024x768\n1280x1024\n1600x1200\n", argv[i]);
				printf( str );
				return 0;
			}
		}
		else if (stricmp(argv[i], "-ref") == 0)
		{
			if (argc <= i+1)
			{
				printf("refresh rate not specified");
				return 0;
			}

			i++;

			if (strcmp(argv[i], "60") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_60Hz;
			}
			else if (strcmp(argv[i], "70") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_70Hz;
			}
			else if (strcmp(argv[i], "72") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_72Hz;
			}
			else if (strcmp(argv[i], "75") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_75Hz;
			}
			else if (strcmp(argv[i], "80") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_80Hz;
			}
			else if (strcmp(argv[i], "90") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_90Hz;
			}
			else if (strcmp(argv[i], "100") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_100Hz;
			}
			else if (strcmp(argv[i], "85") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_85Hz;
			}
			else if (strcmp(argv[i], "120") == 0)
			{
				SCREEN_REFRESH = GR_REFRESH_120Hz;
			}
			else
			{
				sprintf( str, "%s is not a valid refresh rate.\nValid options:\n60\n70\n72\n75\n80\n85\n90\n100\n120\n", argv[i] );
				printf( str );
				return 0;
			}
		}
		else if (stricmp(argv[i], "-screencapture") == 0)
		{
			if (argc <= i+1)
			{
				printf( "Number of screen captures not specified.\n  -screencapture #\nWhere # is a positive integer.\n");
				return 0;
			}
			i++;
			gScreenCaptureEnabled = atoi(argv[i]);
		}
		else if (stricmp(argv[i], "-tex") == 0)
		{
			if (argc <= i+1)
			{
				printf( "Texture was not specified.\n  -tex <texture_name>\ni.e.  -tex tex.dat\n" );
				return 0;
			}
			i++;
			strcpy(gTextureFileName, argv[i]);
		}
		else if (stricmp(argv[i], "-bg") == 0)
		{
			if (argc <= i+1)
			{
				printf( "Background was not specified.\n  -bg <texture_name>\ni.e.  -tex bg.dat\n" );
				return 0;
			}
			i++;
			strcpy(gBackgroundFileName, argv[i]);
		}
	}

	if (!GlideInitialize())
	{
		printf( "Could not initialize Glide.  Application Aborted.\n" );
		return -1;
	}

	gNumTMUsUsed = GetNumTMUs();

	// HACK: some Voodoo Graphics cards don't support
	// GR_REFRESH_NONE, and just hang, so just make it 60Hz
	if (GetNumTMUs() == 1 && SCREEN_REFRESH == GR_REFRESH_NONE)
	{
		SCREEN_REFRESH = GR_REFRESH_60Hz;
	}

	// this is the lowest resolution I'll support
	LO_SCREEN_RES = GR_RESOLUTION_640x480;
	LO_SCREEN_RES_X = 640;
	LO_SCREEN_RES_Y = 480;

	if (set_resolution)
	{
#ifdef USE_GLIDE3
		// try to get a triple-buffer with depth-buffer frame buffer
		gContext = grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1);
		if (!gContext)
		{
			// now try to get a double-buffer with depth-buffer frame buffer
			gContext = grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1);
			if (!gContext)
#else // USE_GLIDE3

		// try to get a triple-buffer with depth-buffer frame buffer
		if (!grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1))
		{
			// now try to get a double-buffer with depth-buffer frame buffer
			if (!grSstWinOpen(0, HI_SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1))
#endif // USE_GLIDE3

			{
				GlideCleanup();

				sprintf(str, "Could not open display (%dx%d)  Application Aborted.\n", (int)HI_SCREEN_RES_X, (int)HI_SCREEN_RES_Y);
				printf( str );
				return -1;
			}
		}
	}
	// find out what's the highest resolution I can get (at least 640x480)
	else if (!WinOpenHighestResolution(&HI_SCREEN_RES, &HI_SCREEN_RES_X, &HI_SCREEN_RES_Y))
	{
		GlideCleanup();
		printf( "Could not open display (640x480)  Application Aborted.\n" );
		return -1;
	}
	

	SCREEN_RES = HI_SCREEN_RES;
	SCREEN_RES_X = (float)HI_SCREEN_RES_X;
	SCREEN_RES_Y = (float)HI_SCREEN_RES_Y;
	os2_InitializeXWindow(SCREEN_RES_X, SCREEN_RES_Y);
        
	tlConSet(0.0f, 0.0f, 1.0f, 1.0f, SCREEN_RES_X, SCREEN_RES_Y, 80, 40, 0xff00007f);
	tlConClear();

	if (gScreenCaptureEnabled)
	{
		if (!SetScreenCaptureRect(0, 0, (int)SCREEN_RES_X, (int)SCREEN_RES_Y))
		{
#ifdef USE_GLIDE3
			grSstWinClose(gContext);
#else
			grSstWinClose();
#endif // USE_GLIDE3
			GlideCleanup();
			printf( "Could not setup screen capture rectangle.  Application Aborted.\n" );
			return -1;
		}
	}

	// set up glide state
	InitGlideState();


	// We've gotten this far, which means we were able in intialize Glide
	// and set the proper resolution
	// Let's start the demo fun ! ! !


#if INTRO_ENABLED
	// LINUX
	os2_SetXWindow(UIIntro);

	grTexClampMode(GR_TMU0, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);

	if (GetNumTMUs() >= 2)
	{
		grTexClampMode(GR_TMU1, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);
	}

	// initialize intro //
	if (InitIntro()) {
		while (gIntroRunning) {

			// LINUX: handle keys and mouse here //
		        os2_EventLoop();
			grSstIdle();
			RenderIntro();
		} /* while */

		CleanupIntro();
	}
	else
	{
	  PrintError("intro failed\n");
	}
#endif // INTRO_ENABLED

	// set up the font stuff
	tlConSet(0.0f, 0.0f, 1.0f, 1.0f, SCREEN_RES_X, SCREEN_RES_Y, 80, 40, 0xff00007f);
	tlConClear();

	// LINUX
	os2_GetCursorPos(&gMousePos1);

	// LINUX
	os2_SetXWindow(UIMain);
	

	grTexClampMode(GR_TMU0, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);
	if (GetNumTMUs() >= 2)
	{
		grTexClampMode(GR_TMU1, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);
	}

	g_bExitApp = FALSE;
	gMouseState = 0;
	gKeyboardState = 0;

	gDetail = 50;
	gWireframe = 0;
	gDrawNormals = 0;
	gDisplayBackground = 0;
	gChromatic = 127.0f;
	gBumpmapExcessAlpha = 0.0f;
	gInnerRadius = 0.25f;
	gCurrBumpmap = 0;
	gDefaultDepths[0] = 4.0f;
	gDefaultDepths[1] = 4.0f;
	gDefaultDepths[2] = 4.0f;
	gDefaultDepths[3] = 3.0f;
	gDefaultDepths[4] = 3.0f;
	gDefaultDepths[5] = 2.0f;
	gDepth = gDefaultDepths[gCurrBumpmap];
	gRenderMode = 3;

	// precalculate the DrawTorus vertices
	PreCalcTorusVerts(gDetail, gDetail, gInnerRadius, 1.0f);

	InitCrystalBall(&gObjectCrystalBall);
	gObjectCrystalBall.rotx = DEG_TO_RAD(30.0f);
	gObjectCrystalBall.roty = DEG_TO_RAD(10.0f);
	gObjectCrystalBall.rot_axis = X;
	InitCrystalBall(&gLightCrystalBall);
	gCurrCrystalBall = &gObjectCrystalBall;

	if (!LoadAndCacheTextures())
	{
#ifdef USE_GLIDE3
		grSstWinClose(gContext);
#else
		grSstWinClose();
#endif // USE_GLIDE3

		GlideCleanup();
		printf( "Failed to load textures in function: LoadAndCacheTexture()  Application Aborted.\n" );
		return -1;
	}

	while (!g_bExitApp)
	{
		static int last_update = 0;
		int time;
		time = os2_timeGetTime();

		// update gMousePos3 20 times per second
		// this is used just for the dynamic object rotation
		// when the mouse button is released with some momentum

		if (time - last_update > (1000/20))
		{
			last_update = time;
			gMousePos3.x = gMousePos1.x;
			gMousePos3.y = gMousePos1.y;
		}

		os2_EventLoop();
		grSstIdle();
		Render();
	}

#ifdef USE_GLIDE3
	grSstWinClose(gContext);
#else
	grSstWinClose();
#endif // USE_GLIDE3

	if (gScreenCaptureEnabled)
	{
		CleanupScreenCapture();
	}

	DestroyTextures();

	GlideCleanup();
	os2_ShutdownXWindow();

	return 0;
}

FxBool WinOpenHighestResolution(int *resolution, int *res_x, int *res_y)
{
	struct
	{
		int resolution;
		int res_x;
		int res_y;
	} res_info[3] = {
		{GR_RESOLUTION_1024x768,  1024,  768},
		{GR_RESOLUTION_800x600,    800,  600},
		{GR_RESOLUTION_640x480,    640,  480}
	};
	int i;

	// HACK: some Voodoo1s hang if I call grSstWinOpen with a resolution of 1024x768
	// so I'll just skip it for all Voodoo1s
	if (GetNumTMUs() == 1)
	{
		i = 1;
	}
	else
	{
		i = 0;
	}
	for (; i<3; i++)
	{
		// try to get a triple-buffer with depth-buffer frame buffer
		if (grSstWinOpen(0, res_info[i].resolution, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1))
		{
			*resolution = res_info[i].resolution;
			*res_x = res_info[i].res_x;
			*res_y = res_info[i].res_y;
			return FXTRUE;
		}
		// now try to get a double-buffer with depth-buffer frame buffer
		else if (grSstWinOpen(0, res_info[i].resolution, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1))
		{
			*resolution = res_info[i].resolution;
			*res_x = res_info[i].res_x;
			*res_y = res_info[i].res_y;
			return FXTRUE;
		}
	}

	return FXFALSE;
}

int RenderTorus()
{
	int num_passes;

	num_passes = 0;

	// transform and calculate the lighting for all the vertices of the torus
	// this is done separately to take advantage of the shared vertices of the torus
	XformAndLightTorusVerts(gDetail, gDetail, (gRenderMode == 0) ? FXTRUE : FXFALSE);

	// set up the alpha combine unit which won't be changed in any of the passes
	grAlphaCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);

	if (gNumTMUsUsed >= 2)
	{
		if (gRenderMode == 0)
		{
			// disable 2nd S and Ts
#ifdef USE_GLIDE3
			grVertexLayout(GR_PARAM_ST1, 0, GR_PARAM_DISABLE);
#else
			grHints(GR_HINT_STWHINT, 0);
#endif // USE_GLIDE3

			GetTexCache(GR_TMU0)->TexSource(&gTexture);
			grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
									 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
			grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER_ADD_LOCAL_ALPHA, GR_COMBINE_FACTOR_LOCAL,
										 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
			grAlphaBlendFunction(GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
			DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, 0.0f, 0.0f, 0);
			num_passes++;
		}
		else
		{
			// enable 2nd S and Ts
#ifdef USE_GLIDE3
			grVertexLayout(GR_PARAM_ST1, offsetof(GrVertex, tmuvtx[1]), GR_PARAM_ENABLE);
#else
			grHints(GR_HINT_STWHINT, GR_STWHINT_ST_DIFF_TMU1);
#endif // USE_GLIDE3

			// pass #1: bumpmap (bumpmap - shifted)
			if (gRenderMode & 1)
			{
				GetTexCache(GR_TMU0)->TexSource(&gBumpmapTexture[gCurrBumpmap]);
				grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_SCALE_OTHER_MINUS_LOCAL_ADD_LOCAL_ALPHA, GR_COMBINE_FACTOR_ONE,
										 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
				GetTexCache(GR_TMU1)->TexSource(&gBumpmapTexture[gCurrBumpmap]);
				grTexCombine(GR_TMU1, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
										 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
				grAlphaBlendFunction(GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
				if (gRenderMode & 2)
				{
					grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER_ADD_LOCAL_ALPHA, GR_COMBINE_FACTOR_ONE,
												 GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
				}
				else
				{
					grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER_ADD_LOCAL_ALPHA, GR_COMBINE_FACTOR_LOCAL,
												 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
				}

				DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, gBumpmapExcessAlpha, gDepth, TORUS_BUMPMAP);
				num_passes++;
			}

			// pass #2: envmap (texture + envmap)
			if (gRenderMode & 2)
			{
				GetTexCache(GR_TMU0)->TexSource(&gEnvmapTexture);
				grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_BLEND, GR_COMBINE_FACTOR_LOCAL_ALPHA,
										 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
				GetTexCache(GR_TMU1)->TexSource(&gTexture);
				grTexCombine(GR_TMU1, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
										 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
				grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_LOCAL,
											 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
				if (gRenderMode & 1)
				{
					if (gSpecialBlend)
					{
						grAlphaBlendFunction(GR_BLEND_DST_COLOR, GR_BLEND_SRC_COLOR, GR_BLEND_ZERO, GR_BLEND_ZERO);
					}
					else
					{
						grAlphaBlendFunction(GR_BLEND_DST_COLOR, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
					}
				}
				else
				{
					grAlphaBlendFunction(GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
				}

				DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, 0.0f, 0.0f, TORUS_ENVMAP);
				num_passes++;

			}
		}
	}
	else // 1 TMU
	{
		// set up the combine units
		grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
		if (gRenderMode == 0)
		{
			GetTexCache(GR_TMU0)->TexSource(&gTexture);
			grAlphaBlendFunction(GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
			grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER_ADD_LOCAL_ALPHA, GR_COMBINE_FACTOR_LOCAL,
										 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
			DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, 0.0f, 0.0f, TORUS_FIRST_PASS);
			num_passes++;
		}
		else
		{
			grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_LOCAL,
										 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_TEXTURE, FXFALSE);

			// pass #1-2: bumpmap (bumpmap - shifted)
			if (gRenderMode & 1)
			{
				if (gRenderMode & 2)
				{
					grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_ONE,
												 GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
				}
				// pass #1: bumpmap
				GetTexCache(GR_TMU0)->TexSource(&gBumpmapTexture[gCurrBumpmap]);
				grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
				DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, 127.0f+gBumpmapExcessAlpha, 0.0f, TORUS_BUMPMAP | TORUS_FIRST_PASS);
				num_passes++;

				// pass #2: negated shifted bumpmap
				grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO);
				// invert the colors
				grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
										 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXTRUE, FXFALSE);
				DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, 127.0f+gBumpmapExcessAlpha, gDepth, TORUS_BUMPMAP);
				num_passes++;

				// restore the normal grTexCombine unit
				grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
										 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
			}

			// enable gouraud for lighting
			grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_LOCAL,
										 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_TEXTURE, FXFALSE);

			// pass #3-4: texture + envmap
			if (gRenderMode & 2)
			{
				// pass #3: texture
				if (255.0f-gChromatic)
				{
					if (gRenderMode & 1)
					{
						if (gSpecialBlend)
						{
							grAlphaBlendFunction(GR_BLEND_DST_COLOR, GR_BLEND_SRC_COLOR, GR_BLEND_ZERO, GR_BLEND_ZERO);
						}
						else
						{
							grAlphaBlendFunction(GR_BLEND_DST_COLOR, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
						}
					}
					else
					{
						grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
					}
					GetTexCache(GR_TMU0)->TexSource(&gTexture);
					DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, 255.0f-gChromatic, 0.0f, TORUS_ENVMAP | TORUS_FIRST_PASS);
					num_passes++;
				}
				// pass #4: envmap
				if (gChromatic)
				{
					grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ONE_MINUS_SRC_ALPHA, GR_BLEND_ZERO, GR_BLEND_ZERO);
					GetTexCache(GR_TMU0)->TexSource(&gEnvmapTexture);
					DrawTorus(gDetail, gDetail, gInnerRadius, 1.0f, gChromatic, 0.0f, TORUS_ENVMAP);
					num_passes++;
				}
			}
		}
	}

	return num_passes;
}

void Render()
{
	static float fps = 0.0f;
	static int frames = 0;
	static int total_time = 0;
	int frame_time;
	int num_polys, num_passes;
	Matrix frustum_mat, viewport_mat, scale_mat, translate_mat;
	Vector light_dir;

	frame_time = os2_timeGetTime();

	switch (gDisplayBackground)
	{
		case 0:
			// clears (color-bgr, alpha, depth)
			grBufferClear(0, 0, 0xffff);
			break;

		case 1:
			// draw the background
			GetTexCache(GR_TMU0)->TexSource(&gBackgroundTexture);
			DrawBackground(SCREEN_RES_X, SCREEN_RES_Y);
			break;

		case 2:
			FadeBackground(SCREEN_RES_X, SCREEN_RES_Y, 64.0f);
			break;
	}

	num_polys = (gDetail*gDetail)<<1;
	num_passes = 0;

	if (gMouseState & LEFT_MOUSE_BUTTON_DOWN)
	{
		SetupCrystalBallRotation(gCurrCrystalBall, gMousePos1, gMousePos2);
	}
	if (gMouseState & RIGHT_MOUSE_BUTTON_DOWN)
	{
		if (gCurrCrystalBall == &gObjectCrystalBall)
		{
			gObjectCrystalBall.scale = CLAMP(gObjectCrystalBall.scale*(1.0f+0.025f*(gMousePos2.y - gMousePos1.y)), 0.01f, 1.4f);
		}
		else
		{
			gLightSpecularFactor = CLAMP(gLightSpecularFactor + 0.5f*(gMousePos2.y - gMousePos1.y), 0.0f, 255.0f);
			gLightDiffuseFactor = MIN(255.0f-gLightAmbientFactor, 1.5f*gLightSpecularFactor);
		}
	}

	gMousePos2.x = gMousePos1.x;
	gMousePos2.y = gMousePos1.y;
	
	// set up the crystal ball matrix
	UpdateCrystalBall(&gObjectCrystalBall);

	UpdateCrystalBall(&gLightCrystalBall);
	MatMultVec3x4_3(light_dir, gLightCrystalBall.rot_mat, gLightDir);

	// transform the directional light by applying
	// the opposite of the overall object rotation
	DirLightXforms(gObjectCrystalBall.rot_mat, light_dir, gCameraDir);

	ScaleMat(scale_mat, gObjectCrystalBall.scale, gObjectCrystalBall.scale, gObjectCrystalBall.scale);

	TranslateMat(translate_mat, 0.0f, 0.0f, -2.0f);

#if STEREO_VIEW_ENABLED
	if (gStereoView)
	{
		Matrix m;

		// set up the projection matrix
		ViewportMat(viewport_mat, 0.0f, 0.5f*SCREEN_RES_X, 0.0f, SCREEN_RES_Y, 1.0f, 65536.0f);
		FrustumMat(frustum_mat, -1.0f*SCREEN_RES_Y/SCREEN_RES_X, 1.0f*SCREEN_RES_Y/SCREEN_RES_X, -1.0f, 1.0f, 1.0f, 11.0f);
		MatMultMat4x4(gProjMat, viewport_mat, frustum_mat);

		// put the transformations and projection into one matrix - gProjMat
		RotateYMat(m, gStereoViewAngle);
		MatMultMat4x4(gProjMat, gProjMat, translate_mat);
		MatMultMat4x4(m, m, gObjectCrystalBall.rot_mat);
		MatMultMat4x4(gProjMat, gProjMat, m);
		MatMultMat4x4(gProjMat, gProjMat, scale_mat);

		num_passes += RenderTorus();

		// set up the projection matrix
		ViewportMat(viewport_mat, 0.5f*SCREEN_RES_X, SCREEN_RES_X, 0.0f, SCREEN_RES_Y, 1.0f, 65536.0f);
		FrustumMat(frustum_mat, -1.0f*SCREEN_RES_Y/SCREEN_RES_X, 1.0f*SCREEN_RES_Y/SCREEN_RES_X, -1.0f, 1.0f, 1.0f, 11.0f);
		MatMultMat4x4(gProjMat, viewport_mat, frustum_mat);

		// put the transformations and projection into one matrix - gProjMat
		RotateYMat(m, -2.0f*gStereoViewAngle);
		MatMultMat4x4(gProjMat, gProjMat, translate_mat);
		MatMultMat4x4(m, m, gObjectCrystalBall.rot_mat);
		MatMultMat4x4(gProjMat, gProjMat, m);
		MatMultMat4x4(gProjMat, gProjMat, scale_mat);

		num_passes += RenderTorus();
	}
	else
#endif // STEREO_VIEW_ENABLED
	{
		// set up the projection matrix
		ViewportMat(viewport_mat, 0.0f, SCREEN_RES_X, 0.0f, SCREEN_RES_Y, 1.0f, 65536.0f);
		FrustumMat(frustum_mat, -1.0f*SCREEN_RES_X/SCREEN_RES_Y, 1.0f*SCREEN_RES_X/SCREEN_RES_Y, -1.0f, 1.0f, 1.0f, 11.0f);
		MatMultMat4x4(gProjMat, viewport_mat, frustum_mat);

		// put the transformations and projection into one matrix - gProjMat
		MatMultMat4x4(gProjMat, gProjMat, translate_mat);
		MatMultMat4x4(gProjMat, gProjMat, gObjectCrystalBall.rot_mat);
		MatMultMat4x4(gProjMat, gProjMat, scale_mat);

		num_passes += RenderTorus();
	}

	// output the console stuff
	tlConClear();

	if (gExtraInfoDisplay)
	{
		tlConOutput("fps: %.2f, ppf: %6d, torus polys: %6d [%d x %d vsynch: %s]\n", fps, num_polys*num_passes, num_polys,
								(int)SCREEN_RES_X, (int)SCREEN_RES_Y, gVsynch ? " on" : "off");
	}
	if (gExtraInfoDisplay == 2)
	{
		tlConOutput("tmus: %d, passes: %d (base texture: %s, environment map: %s, bumpmap: %s)", gNumTMUsUsed, num_passes,
								(gRenderMode!=1) ? " on" : "off", (gRenderMode>>1) ? " on" : "off", (gRenderMode&1) ? " on" : "off");
	}

	// capture screen before rendering console stuff
	if (gNumScreensToCapture)
	{
		char str[64];

		gNumScreensToCapture--;

		sprintf(str, "scrn%d.tga", gNumScreenShots);
		if (ScreenCaptureTGA(str))
		{
			tlConOutput("\nscreen captured...%d more to go...\n", gNumScreensToCapture);
			gNumScreenShots++;
		}
	}

	// render the console stuff
	tlConRender();

	if (gHelpMenu)
	{
		tlConClear();
		tlConOutput("\n\n\n"
								"left mouse: rotation about x and y axes\n"
								"left mouse w/shift: rotation about z-axis)\n"
								"right mouse: scaling\n"
								"equal/minus: increase/decrease detail level ---------------------- %d\n"
								"page up/page down: increase/decrease inner radius ---------------- %.2f\n"
#if STEREO_VIEW_ENABLED
								"arrow up/down: increase/decrease stereo view angle --------------- %.2f\n"
#endif
								"B: toggle special blending (%s)\n"
								"C: increase chromatic (w/control: decrease chromatic) ------------ %.2f\n"
								"D: increase bump depth (w/control: decrease bump depth) ---------- %.2f\n"
								"E: increase excess alpha (w/control: decrease excess alpha) ------ %.2f\n"
								"G: toggle background display\n"
								"I: toggle extra info display\n"
								"H/F1: toggle help menu\n"
								"M: toggle resolution (%d x %d / %d x %d)\n"
								"N: toggle display of normals\n"
								"O: toggle object being rotated (torus / light)\n"
								"P: toggle render mode\n"
								"R: reset the object orientation\n"
#if STEREO_VIEW_ENABLED
								"S: toggle stereo view\n"
#endif
								"T: toggle texture used for bumpmapping\n"
								"U: toggle num tmus used (%s)\n"
								"V: toggle vsynch\n"
								"W: toggle wireframe\n",
								gDetail, gInnerRadius,
#if STEREO_VIEW_ENABLED
								RAD_TO_DEG(gStereoViewAngle),
#endif
								gSpecialBlend ? " on" : "off",
								gChromatic, gDepth, gBumpmapExcessAlpha, LO_SCREEN_RES_X, LO_SCREEN_RES_Y, HI_SCREEN_RES_X, HI_SCREEN_RES_Y,
								(GetNumTMUs() == 1) ? "disabled" : "enabled");

		tlConRender();
	}

	DrawCursor(gMousePos1.x, gMousePos1.y);
	grBufferSwap(gVsynch);

	total_time += os2_timeGetTime() - frame_time;
	frames++;

	// compute frame rate
	if (total_time >= 1000)
	{
		fps = 1000.0f*(float)frames/(float)total_time;
		frames = 0;
		total_time = 0;
	}

}

void InitGlideState()
{
	grTexFilterMode(GR_TMU0, GR_TEXTUREFILTER_BILINEAR, GR_TEXTUREFILTER_BILINEAR);
	grTexLodBiasValue(GR_TMU0, 0.5f);
	grTexMipMapMode(GR_TMU0, GR_MIPMAP_NEAREST, FXFALSE);
	grTexClampMode(GR_TMU0, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);
	if (GetNumTMUs() >= 2)
	{
		grTexFilterMode(GR_TMU1, GR_TEXTUREFILTER_BILINEAR, GR_TEXTUREFILTER_BILINEAR);
		grTexLodBiasValue(GR_TMU1, 0.5f);
		grTexMipMapMode(GR_TMU1, GR_MIPMAP_NEAREST, FXFALSE);
		grTexClampMode(GR_TMU1, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);
	}
	grAlphaTestReferenceValue(0xff);
	grCullMode(GR_CULL_NEGATIVE);
	grDepthBufferMode(GR_DEPTHBUFFER_WBUFFER);
	grDepthBufferFunction(GR_CMP_LEQUAL);
	grDepthMask(FXTRUE);
	grAlphaTestFunction(GR_CMP_ALWAYS);
	grClipWindow(0, 0, (int)SCREEN_RES_X, (int)SCREEN_RES_Y);

	// let glide know that we will be using different S and Ts for the 2 TMUs
#ifdef USE_GLIDE3
	grCoordinateSpace(GR_WINDOW_COORDS);
	grVertexLayout(GR_PARAM_XY, offsetof(GrVertex, x), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_Q, offsetof(GrVertex, oow), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_RGB, offsetof(GrVertex, r), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_A, offsetof(GrVertex, a), GR_PARAM_ENABLE);
	grVertexLayout(GR_PARAM_ST0, offsetof(GrVertex, tmuvtx[0]), GR_PARAM_ENABLE);
#endif // USE_GLIDE3
}

FxBool LoadAndCacheTextures()
{
	TextureCache *tex_cache;
	int i;
	char str[64];

	if (!LoadGlideTextureTGA(gTextureFileName,  &gTexture.tex_info, GR_TEXFMT_RGB_565, 0, 4))
	{
		PrintError("failed to load %s\n", gTextureFileName);
		return FXFALSE;
	}

	if (!LoadGlideTextureTGA(gBackgroundFileName,  &gEnvmapTexture.tex_info, GR_TEXFMT_ARGB_4444, 255-(FxU8)gChromatic, 0))
	{
		PrintError("failed to load %s\n", gBackgroundFileName);
		return FXFALSE;
	}
	if (!SpherizeTexture(&gEnvmapTexture.tex_info))
	{
		PrintError("failed to spherize envmap\n");
		return FXFALSE;
	}

	if (!LoadGlideTextureTGA(gBackgroundFileName,  &gBackgroundTexture.tex_info, GR_TEXFMT_RGB_565, 0, 0))
	{
		PrintError("failed to load %s\n", gBackgroundFileName);
		return FXFALSE;
	}

	for (i=0; i<NUM_BUMPMAPS; i++)
	{
		sprintf(str, "data/bump%d.dat", i);
		if (!LoadGlideTextureTGA(str,  &gBumpmapTexture[i].tex_info, GR_TEXFMT_ALPHA_INTENSITY_88, 127, 4))
		{
			PrintError("failed to load %s\n", str);
			return FXFALSE;
		}
	}

	// cache all the textures
	tex_cache = GetTexCache(GR_TMU0);
	tex_cache->CacheTexture(&gTexture);
	tex_cache->CacheTexture(&gEnvmapTexture);
	tex_cache->CacheTexture(&gBackgroundTexture);
	tex_cache->CacheTexture(&gBumpmapTexture[0]);
	tex_cache->CacheTexture(&gBumpmapTexture[1]);
	tex_cache->CacheTexture(&gBumpmapTexture[2]);
	tex_cache->CacheTexture(&gBumpmapTexture[3]);
	tex_cache->CacheTexture(&gBumpmapTexture[4]);

	if (GetNumTMUs() >= 2)
	{
		// cache all the textures on TMU1 as well
		tex_cache = GetTexCache(GR_TMU1);
		tex_cache->CacheTexture(&gTexture);
		tex_cache->CacheTexture(&gEnvmapTexture);
		tex_cache->CacheTexture(&gBackgroundTexture);
		tex_cache->CacheTexture(&gBumpmapTexture[0]);
		tex_cache->CacheTexture(&gBumpmapTexture[1]);
		tex_cache->CacheTexture(&gBumpmapTexture[2]);
		tex_cache->CacheTexture(&gBumpmapTexture[3]);
		tex_cache->CacheTexture(&gBumpmapTexture[4]);
	}

	return FXTRUE;
}

void SetTextureAlpha(GlideTexture *texture, FxU8 alpha)
{
	FxU32 i, log_width, log_height, total_size, lods, curr_width, curr_height;
	FxU16 *data, new_alpha;

#ifdef USE_GLIDE3
	log_width = texture->tex_info.largeLodLog2;
	log_height = log_width - texture->tex_info.aspectRatioLog2;
	lods = texture->tex_info.largeLodLog2 - texture->tex_info.smallLodLog2;
#else
	log_width = GR_LOD_1 - texture->tex_info.largeLod;
	log_height = log_width - (GR_ASPECT_1x1 - texture->tex_info.aspectRatio);
	lods = texture->tex_info.smallLod - texture->tex_info.largeLod;
#endif // USE_GLIDE3

	new_alpha = (alpha & 0xf0)<<8;
	// find the size of the whole mipmap
	curr_width = 1<<log_width;
	curr_height = 1<<log_height;
	total_size = curr_width*curr_height;
	for (i=1; i<=lods; i++)
	{
		curr_width = MAX(1, curr_width>>1);
		curr_height = MAX(1, curr_height>>1);

		total_size += curr_width*curr_height;
	}

	// data format is GR_TEXFMT_ARGB_4444
	data = (FxU16 *)texture->tex_info.data;
	for (i=total_size; i>0; i--)
	{
		*data = (*data & 0x0fff) | new_alpha;
		data++;
	}
	for (i=0; i<(unsigned)GetNumTMUs(); i++)
	{
		GetTexCache(i)->ReloadTexture(texture);
	}
}

void DestroyTextures()
{
	int i;
	TextureCache *tex_cache;

	for (i=0; i<GetNumTMUs(); i++)
	{
		tex_cache = GetTexCache(i);
		tex_cache->DestroyGlideTexture(&gTexture);
		tex_cache->DestroyGlideTexture(&gEnvmapTexture);
		tex_cache->DestroyGlideTexture(&gBackgroundTexture);
		tex_cache->DestroyGlideTexture(&gBumpmapTexture[0]);
		tex_cache->DestroyGlideTexture(&gBumpmapTexture[1]);
		tex_cache->DestroyGlideTexture(&gBumpmapTexture[2]);
		tex_cache->DestroyGlideTexture(&gBumpmapTexture[3]);
		tex_cache->DestroyGlideTexture(&gBumpmapTexture[4]);
	}
}

void SetupCrystalBallRotation(CrystalBall *crystal_ball, POINTL pos1, POINTL pos2)
{
	float crossz, mag_inv;

	if (gKeyboardState & SHIFT_KEY_DOWN)
	{
		// find the rotation of the mouse cursor about the z-axis
		// by computing the cross product of the mouse positions (treating
		// the center of the screen as the origin)
		crossz = ((pos1.x-(0.5f*SCREEN_RES_X))*(pos2.y-(0.5f*SCREEN_RES_Y)) -
							(pos1.y-(0.5f*SCREEN_RES_Y))*(pos2.x-(0.5f*SCREEN_RES_X)));
		// normalize the cross product
		mag_inv = fsqrt_inv(SQR(pos1.x-(0.5f*SCREEN_RES_X)) + SQR(pos1.y-(0.5f*SCREEN_RES_Y)));
		mag_inv *= fsqrt_inv(SQR(pos2.x-(0.5f*SCREEN_RES_X)) + SQR(pos2.y-(0.5f*SCREEN_RES_Y)));
		// apply the rotation of the mouse about the z-axis to the object
		crystal_ball->rotz = MOUSE_SENSITIVITY*100.0f*crossz*mag_inv;
		crystal_ball->rot_axis = Z;
	}
	else
	{
		crystal_ball->rotx = MOUSE_SENSITIVITY*(-0.5f)*(pos2.y - pos1.y);
		crystal_ball->roty = MOUSE_SENSITIVITY*(-0.5f)*(pos2.x - pos1.x);
		crystal_ball->rot_axis = X;
	}
}

MRESULT APIENTRY UIMain(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   extern unsigned int g_winWidth, g_winHeight;

	HPS hps;
	RECTL rect;
        CHAR key;

	int i;
	TextureCache *tex_cache;

	switch(msg)	{
		case WM_CHAR:
			if (SHORT1FROMMP(mp1) & KC_SHIFT)
				gKeyboardState |= SHIFT_KEY_DOWN;
			else
				gKeyboardState &= ~SHIFT_KEY_DOWN;

			if (SHORT1FROMMP(mp1) & KC_CTRL)
				gKeyboardState |= CONTROL_KEY_DOWN;
			else
				gKeyboardState &= ~CONTROL_KEY_DOWN;

			if ((SHORT1FROMMP(mp1) & KC_KEYUP)==0)
                           return 0;

                           if (SHORT1FROMMP(mp1) & KC_VIRTUALKEY)
                              {
                              key = SHORT2FROMMP(mp2);
                              switch ( key )
                                 {
                                 case VK_ESC:
                                    g_bExitApp = TRUE;
                                    break;
                                 case VK_ENTER:
				    gNumScreensToCapture = gScreenCaptureEnabled;
                                    break;
                                 case VK_PAGEUP:
                                    gInnerRadius = MAX(0.0f, gInnerRadius-0.01f);
                                    PreCalcTorusVerts(gDetail, gDetail, gInnerRadius, 1.0f);
                                    break;
                                 case VK_PAGEDOWN:
                                    gInnerRadius = MIN(1.0f, gInnerRadius+0.01f);
                                    PreCalcTorusVerts(gDetail, gDetail, gInnerRadius, 1.0f);
                                    break;
                                 case VK_F1:
 				    gHelpMenu ^= 1;
                                    break;
                                 default:
                                    break;
                                 }
                              return 0;
                              }

//                           if (SHORT1FROMMP(mp1) & KC_CHAR)
                              {
                              key = CHAR1FROMMP(mp2);

                              if ( gKeyboardState & CONTROL_KEY_DOWN )
                                 key += 'A' - 1;

                              switch( key )
                                 {
                                 case '=':
					gDetail = MIN(MAX_TORUS_DETAIL, gDetail+1);
					PreCalcTorusVerts(gDetail, gDetail, gInnerRadius, 1.0f);
                                    break;
                                 case '-':
					gDetail = MAX(4, gDetail-1);
					PreCalcTorusVerts(gDetail, gDetail, gInnerRadius, 1.0f);
                                    break;


				case 'B':
				case 'b':
					gSpecialBlend ^= 1;
				break;

				case 'C':
				case 'c':
					if (gKeyboardState & CONTROL_KEY_DOWN) {
						gChromatic = MAX(0.0f, gChromatic-1.0f);
					} else {
						gChromatic = MIN(255.0f, gChromatic+1.0f);
					}
					if (GetNumTMUs() > 1) {
						SetTextureAlpha(&gEnvmapTexture, 255-(FxU8)gChromatic);
					}
				break;

				case 'D':
				case 'd':
					if (gKeyboardState & CONTROL_KEY_DOWN) {
						gDepth = MAX(0.0f, gDepth-0.1f);
					} else {
						gDepth = MIN(10.0f, gDepth+0.1f);
					}
				break;

				case 'E':
				case 'e':
					if (gKeyboardState & CONTROL_KEY_DOWN) {
						gBumpmapExcessAlpha = MAX(0.0f, gBumpmapExcessAlpha-1.0f);
					} else {
						gBumpmapExcessAlpha = MIN(128.0f, gBumpmapExcessAlpha+1.0f);
					}
				break;

				case 'G':
				case 'g':
					gDisplayBackground = (gDisplayBackground+1) % 3;
				break;

				case 'H': 
				case 'h': 
				case '?': 
					gHelpMenu ^= 1;
				break;

				case 'I':
				case 'i':
					gExtraInfoDisplay = (gExtraInfoDisplay+1) % 3;
				break;


				case 'M':  /* resolution mode change */
				case 'm':

					if (SCREEN_RES == LO_SCREEN_RES)
					{
						SCREEN_RES = HI_SCREEN_RES;
						SCREEN_RES_X = (float)HI_SCREEN_RES_X;
						SCREEN_RES_Y = (float)HI_SCREEN_RES_Y;
					}
					else
					{
						SCREEN_RES = LO_SCREEN_RES;
						SCREEN_RES_X = (float)LO_SCREEN_RES_X;
						SCREEN_RES_Y = (float)LO_SCREEN_RES_Y;
					}

					if (gScreenCaptureEnabled)
					{
						if (!SetScreenCaptureRect(0, 0, (int)SCREEN_RES_X, (int)SCREEN_RES_Y))
						{
#ifdef USE_GLIDE3
							grSstWinClose(gContext);
#else
							grSstWinClose();
#endif // USE_GLIDE3
							GlideCleanup();
							os2_ShutdownXWindow();
							exit(0xdeadbeef);
						}
					}

					// close glide window before reopening it
#ifdef USE_GLIDE3
					grSstWinClose(gContext);
#else
					grSstWinClose();
#endif // USE_GLIDE3
					// try to get a triple-buffer with depth-buffer frame buffer
					if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1))
					{
						// now try to get a double-buffer with depth-buffer frame buffer
						if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1))
						{
							GlideCleanup();
							os2_ShutdownXWindow();
							exit(0xdeadbeef);
						}
					}

					InitGlideState();

					// reset the console
					tlConSet(0.0f, 0.0f, 1.0f, 1.0f, SCREEN_RES_X, SCREEN_RES_Y, 80, 40, 0xff00007f);
					os2_ResizeXWindow( SCREEN_RES_X, SCREEN_RES_Y );
					// reload textures
					for (i=0; i<GetNumTMUs(); i++)
					{
						tex_cache = GetTexCache(i);
						tex_cache->ReloadTexture(&gTexture);
						tex_cache->ReloadTexture(&gEnvmapTexture);
						tex_cache->ReloadTexture(&gBackgroundTexture);
						tex_cache->ReloadTexture(&gBumpmapTexture[0]);
						tex_cache->ReloadTexture(&gBumpmapTexture[1]);
						tex_cache->ReloadTexture(&gBumpmapTexture[2]);
						tex_cache->ReloadTexture(&gBumpmapTexture[3]);
						tex_cache->ReloadTexture(&gBumpmapTexture[4]);
					}
				break;

				case 'N':
				case 'n':
					gDrawNormals ^= 1;
				break;

				case 'O':
				case 'o':
					if (gCurrCrystalBall == &gObjectCrystalBall) {
						gCurrCrystalBall = &gLightCrystalBall;
					} else {
						gCurrCrystalBall = &gObjectCrystalBall;
					}
				break;

				case 'P':
				case 'p':
					gRenderMode = (gRenderMode+1) & 3;
				break;

				case 'R':
				case 'r':
					InitCrystalBall(gCurrCrystalBall);
				break;

				case 'T':
				case 't':
					if (gKeyboardState & CONTROL_KEY_DOWN) {
						gCurrBumpmap = (gCurrBumpmap+NUM_BUMPMAPS-1)%NUM_BUMPMAPS;
					} else {
						gCurrBumpmap = (gCurrBumpmap+1) % NUM_BUMPMAPS;
					}
					gDepth = gDefaultDepths[gCurrBumpmap];
				break;

				case 'U':
				case 'u':
					if (GetNumTMUs() > 1) {
						gNumTMUsUsed ^= 3;
					}
				break;

				case 'V':
				case 'v':
					gVsynch ^= 1;
				break;

				case 'W':
				case 'w':
					gWireframe ^= 1;
				break;
                                 }
                              }
		break;

      case WM_BUTTON1DOWN:
	gMouseState |= LEFT_MOUSE_BUTTON_DOWN;
         gMousePos1.x = SHORT1FROMMP(mp1);
         gMousePos1.y = g_winHeight-SHORT2FROMMP(mp1);
	gMousePos2.x = gMousePos1.x;
	gMousePos2.y = gMousePos1.y;
	break;

      case WM_BUTTON2DOWN:
	gMouseState |= RIGHT_MOUSE_BUTTON_DOWN;
         gMousePos1.x = SHORT1FROMMP(mp1);
         gMousePos1.y = g_winHeight-SHORT2FROMMP(mp1);
	gMousePos2.x = gMousePos1.x;
	gMousePos2.y = gMousePos1.y;
	break;

      case WM_BUTTON1UP:
	gMouseState &= ~LEFT_MOUSE_BUTTON_DOWN;
         gMousePos1.x = SHORT1FROMMP(mp1);
         gMousePos1.y = g_winHeight-SHORT2FROMMP(mp1);

	if (ABS(gMousePos1.x - gMousePos2.x) + ABS(gMousePos1.y - gMousePos2.y) == 0)
	{
		gCurrCrystalBall->rotx = gCurrCrystalBall->roty = gCurrCrystalBall->rotz = 0.0f;
		gCurrCrystalBall->rot_axis = W;
	}
	else
	{
		SetupCrystalBallRotation(gCurrCrystalBall, gMousePos1, gMousePos2);
	}
         break ;

      case WM_BUTTON2UP:
	gMouseState &= ~RIGHT_MOUSE_BUTTON_DOWN;
         gMousePos1.x = SHORT1FROMMP(mp1);
         gMousePos1.y = g_winHeight-SHORT2FROMMP(mp1);
         break;

      case WM_MOUSEMOVE:
         gMousePos1.x = SHORT1FROMMP(mp1);
         gMousePos1.y = g_winHeight-SHORT2FROMMP(mp1);
         break;

    case WM_DESTROY:
      g_bExitApp = TRUE;
      break;

      default: 
         return WinDefWindowProc (hwnd, msg, mp1, mp2);
      }

   return 0;
}
