/*=======================================================================*/
/* European Personal Systems Centre, Mountbatten House, Basingstoke, UK. */
/*=======================================================================*/
/* SECURITY     ===>  IBM DAPTOOLS Use Only                              */
/* PROGRAM NAME ===>  ICONEA.CMD                                         */
/* VERSION      ===>  1.16                                               */
/* RELEASE      ===>  4                                                  */
/* LANGUAGE     ===>  OS/2 Procedures Language 2/REXX                    */
/* AUTHOR       ===>  Karri Mooney                     (KARRI at WINVMK) */
/* DATE         ===>  14th July 1992                                     */
/* SUPPORT      ===>  Author                                             */
/* PURPOSE      ===>  To attach an icon, bitmap or metafile to a file or */
/*              ===>  files in Extended Attributes.                      */
/*              ===>                                                     */
/* SYNTAX       ===>  ICONEA File_Name Source_File Option                */
/*              ===>                                                     */
/*              ===>  where: File_Name is the target file (wildcards are */
/*              ===>         supported).                                 */
/*              ===>                                                     */
/*              ===>         Source_File is a .ICO, .BMP or .MET file    */
/*              ===>         (only one file can be specified).           */
/*              ===>                                                     */
/*              ===>    and: Option can be either blank or /S for        */
/*              ===>         recursive search through subdirectories.    */
/*              ===>                                                     */
/* COMMENTS     ===>  If you have any comments/suggestions, please append*/
/*                    to ICONEA FORUM on IBMPC or alternatively, send me */
/*                    a VM/PROFS note.                                   */
/* SYSTEM S/W   ===>  Operating System/2 Version 2.00                    */
/*=======================================================================*/
/* CHANGE HISTORY:                                                       */
/*                                                                       */
/* Version Release DD/MM/YY Comments                                     */
/*=======================================================================*/
/*    1.00    1    29/06/92 First Release.                               */
/*    1.05    -    01/07/92 Added support for bitmaps and metafiles.     */
/*    1.07    -    01/07/92 Added support for multiple files (wildcards).*/
/*    1.10    2    01/07/92 Added checking routines that a file is an    */
/*                          icon, bitmap or a metafile.                  */
/*    1.15    3    14/07/92 Added recursive search through all           */
/*                          subdirectories. Thanks to Richard Polisuk for*/
/*                          the suggestion.                              */
/*    1.16    4    14/07/92 Added check for REXXUTIL.DLL.                */
/*=======================================================================*/
/*-----------------------------------------------------------------------*/
/* Check if REXXUTIL.DLL exists before anything. First of all check the  */
/* current drive and if it's not there we'll try the C: drive.           */
/*-----------------------------------------------------------------------*/
  rc = stream("\OS2\DLL\REXXUTIL.DLL",'c','query exists')
  if rc = '' then do
    rc = stream("C:\OS2\DLL\REXXUTIL.DLL",'c','query exists')
    if rc = '' then do
      say
      say 'Cannot find \OS2\DLL\REXXUTIL.DLL and therefore cannot'
      say 'continue...please use the OS/2 Installation diskettes '
      say 'to find REXXUTIL.DLL.'
      say 'Sorry!'
      exit
    end
  end

/*-----------------------------------------------------------------------*/
/* Add the RexxUtil function SysPutEA.                                   */
/*                                                                       */
/* Syntax: call SysPutEA File_Name,EA_Name,Full_EA                       */
/*                                                                       */
/* where:            File_Name  -  File name of the target               */
/*                   EA_Name    -  Name of the Extended Attribute        */
/*                   Full_EA    -  Stream of EA data                     */
/*-----------------------------------------------------------------------*/
  call RxFuncAdd SysPutEA,RexxUtil,SysPutEA

/*-----------------------------------------------------------------------*/
/* Add the RexxUtil function SysFileTree                                 */
/*                                                                       */
/* Syntax: call SysFileTree Source_File,'Files','FO'                     */
/*         call SysFileTree Source_File,'Files','SO'                     */
/*                                                                       */
/* where:            Source_File -  File name of the target              */
/*                   Files       -  Array for results                    */
/*                   F           -  Indicates search for files only      */
/*                   S           -  Recursive search through directories */
/*                   O           -  Return fully qualified file names    */
/*-----------------------------------------------------------------------*/
  call RxFuncAdd SysFileTree,RexxUtil,SysFileTree

/*-----------------------------------------------------------------------*/
/* Parse the command line for the file names and the only option so far. */
/*-----------------------------------------------------------------------*/
  arg File_Name Source_File Option .

/*-----------------------------------------------------------------------*/
/* Check that both file names were entered.                              */
/*-----------------------------------------------------------------------*/
  if File_Name = '' | File_Name = '?' | Source_File = '' then do
    call Help
    exit
  end

/*-----------------------------------------------------------------------*/
/* Define some constants and variables.                                  */
/*-----------------------------------------------------------------------*/
  FALSE         = 0                /* Boolean False                      */
  TRUE          = 1                /* Boolean True                       */
  Error         = FALSE            /* Error flag                         */
  Icon_Sig      = 15               /* Icon signature position            */
  Bitmap_Sig    ="BM"              /* Bitmap signature                   */
  Metafile_Sig  ='D3A8A8'x         /* Metafile BDT signature             */
  EAT_ICON      ='F9FF'x           /* EA identifier for .ICON            */
  EAT_BITMAP    ='FBFF'x           /* EA identifier for .BITMAP          */
  EAT_METAFILE  ='FAFF'x           /* EA identifier for .METAFILE        */
  Target.       = ''               /* Target file name array             */
  Source.       = ''               /* Source file name array             */

/*-----------------------------------------------------------------------*/
/* Get all the target files.                                             */
/*-----------------------------------------------------------------------*/
  if Option = "/S" |,
     Option = "-S" then call SysFileTree File_Name,'Target','SO'
  else                  call SysFileTree File_Name,'Target','FO'

  if Target.0 = 0 then do
    say
    say 'Cannot find any files matching 'File_Name'...please try again.'
    exit
  end

/*-----------------------------------------------------------------------*/
/* Now find the size of the EA source file. We need this for the EA.     */
/*-----------------------------------------------------------------------*/
  call SysFileTree Source_File,'Source','F'

  if Source.0 = 0 then do
    say
    say 'Cannot find a file matching 'Source_File'...please try again.'
    exit
  end

  Source_Size = word(Source.1,3)

/*-----------------------------------------------------------------------*/
/* And read it in before we go and process the different formats. We need*/
/* to do it here, otherwise the following line will be repeated three    */
/* times (once for each supported format).                               */
/*-----------------------------------------------------------------------*/
  EA_Value = charin(Source_File,1,Source_Size)

/*-----------------------------------------------------------------------*/
/* Check which of the three formats it is.                               */
/*-----------------------------------------------------------------------*/
  Extension = substr(Source_File,lastpos('.',Source_File),4)

  select
    when Extension = '.ICO' then call Icon_Format
    when Extension = '.BMP' then call Bitmap_Format
    when Extension = '.MET' then call Metafile_Format
    otherwise do
      say
      say 'The EA source file is not an icon, bitmap or a metafile.'
      exit
    end
  end

/*-----------------------------------------------------------------------*/
/* If an error has occurred, a message has already been displayed by one */
/* of the three routines.                                                */
/*-----------------------------------------------------------------------*/
  if Error then exit

/*-----------------------------------------------------------------------*/
/* Loop for all the target files found.                                  */
/*-----------------------------------------------------------------------*/
  do i = 1 to Target.0

/*-----------------------------------------------------------------------*/
/* And attach the icon/bitmap/metafile into EAs.                         */
/*-----------------------------------------------------------------------*/
    call SysPutEA Target.i,EA_Name,Full_EA

    if result = 0 then do
      say
      say Source_File' was attached to 'Target.i' successfully.'
    end
    else do
      say
      say  Source_File' could not be attached to 'Target.i' .'
      say 'The return code was 'result'.'
    end
  end

  exit
/*=======================================================================*/
/* End of main program logic.                                            */
/*=======================================================================*/

/*=======================================================================*/
/* Start of procedures:                                                  */
/*=======================================================================*/
/*=======================================================================*/
/* Icon_Format: Process the .ICON Extended Attribute.                    */
/*=======================================================================*/
Icon_Format:

/*-----------------------------------------------------------------------*/
/* Check that the file really is an icon.                                */
/*-----------------------------------------------------------------------*/
  if substr(EA_Value,Icon_Sig,2) <> 'CI' &,
     substr(EA_Value,Icon_Sig,2) <> 'IC' then do
    say
    say Source_File 'is not an icon.'
    Error = TRUE
    return
  end

/*-----------------------------------------------------------------------*/
/* Build the EA.                                                         */
/*-----------------------------------------------------------------------*/
  EA_Name    = ".ICON"                        /* Icon EA name            */
  Full_EA    = EAT_ICON         ||,           /* Icon EA identifier      */
               d2c(Source_Size) ||,           /* Length                  */
               EA_Value                       /* Data stream             */

return

/*=======================================================================*/
/* Bitmap_Format: Process the .BITMAP Extended Attribute.                */
/*=======================================================================*/
Bitmap_Format:

/*-----------------------------------------------------------------------*/
/* Check that the file really is a bitmap.                               */
/*-----------------------------------------------------------------------*/
  if substr(EA_Value,1,2) <> Bitmap_Sig then do
    say
    say Source_File 'is not a bitmap.'
    Error = TRUE
    return
  end

/*-----------------------------------------------------------------------*/
/* Build the EA.                                                         */
/*-----------------------------------------------------------------------*/
  EA_Name    = ".BITMAP"                      /* Bitmap EA name          */
  Full_EA    = EAT_BITMAP       ||,           /* Bitmap EA identifier    */
               d2c(Source_Size) ||,           /* Length                  */
               EA_Value                       /* Data stream             */

return

/*=======================================================================*/
/* Metafile_Format: Process the .METAFILE Extended Attribute.            */
/*=======================================================================*/
Metafile_Format:

/*-----------------------------------------------------------------------*/
/* Check that the file really is a metafile.                             */
/*-----------------------------------------------------------------------*/
  if substr(EA_Value,3,3) <> Metafile_Sig then do
    say
    say Source_File 'is not a metafile.'
    Error = TRUE
    return
  end

/*-----------------------------------------------------------------------*/
/* Build the EA.                                                         */
/*-----------------------------------------------------------------------*/
  EA_Name    = ".METAFILE"                    /* Metafile EA name        */
  Full_EA    = EAT_METAFILE     ||,           /* Metafile EA identifie   */
               d2c(Source_Size) ||,           /* Length                  */
               EA_Value                       /* Data stream             */

return

/*=======================================================================*/
/* Help: Display little bit of information for the user.                 */
/*=======================================================================*/
Help:

say 'Ŀ'
say ' ICONEA - IBM DAPTOOLS Use Only - 14th July 1992 - by Karri Mooney '
say 'Ĵ'
say '                                                                   '
say ' This REXX Command file attaches an icon, bitmap or metafile to    '
say ' a file or files in Extended Attributes.                           '
say '                                                                   '
say ' Correct syntax: ICONEA File_Name Source_File Option               '
say '                                                                   '
say ' where: File_Name   = the target file(s) to which the EAs will be  '
say '                      attached to. Wildcards are allowed.          '
say '        Source_File = a .ICO, .BMP or .MET file.                   '
say '                                                                   '
say '        Option      = Blank or /S for recursive search through     '
say '                      subdirectories.                              '
say '                                                                   '
say ' Example: ICONEA D:\*.HLP MYHELP.ICO /S                            '
say '                                                                   '
say '   would: Attach MYHELP.ICO to all the .HLP files on drive D.      '
say '                                                                   '
say 'Ĵ'
say '                         Version 1.16                              '
say ''

return
/*=======================================================================*/
/* End of REXX .CMD file.                                                */
/*=======================================================================*/
