// Hi Emacs, this may look like C code, but it is really -*- C++ -*-
/*
 * df, yet another clone of this unix utility program.
 *
 * Approach: All drive letters from C to Z are scanned with
 * DosQueryFSInfo. This would lead to error messages at all
 * drive letters which are unused. So we turn them off with
 * the OS-function DosError and reenable them later on.
 * A call to DosQueryFSInfo in mode FSIL_ALLOC gives information
 * about the total and available size of the drive. An additional
 * call of DosQueryFSAttach returns information about the file
 * system driver attached to the drive (FAT,HPFS,NFS,LAN,CDROM)
 * and for LAN and NFS drives information about the service name.
 *
 * Compiler: emx-0.8g
 *
 * Author: Bernd Esser
 *         Physikalisches Institut der Universit"at Bonn, Germany
 *         esser@pib1.physik.uni-bonn.de
 *
 * $Log: df.cc,v $
 * Revision 1.2  1993/10/25  18:42:23  Bernd_Esser
 * style brushup
 *
 * Revision 1.1  1993/09/25  17:12:16  Bernd_Esser
 * first version
 */

/*-----------------------------------------------------------------*/

#define INCL_DOSMISC
#define INCL_DOSERRORS
#define INCL_DOSFILEMGR

#include <os2.h>
#include <string.h>
#include <stdlib.h>
#include <iostream.h>

/*-----------------------------------------------------------------*/

struct drvInfo {
  ULONG totalSize;
  ULONG usedSize;
  ULONG availSize;
  ULONG percentUsed;
  char  driveLetter;
  char  remoteName[256];
  char  driverName[256];
};

/*-----------------------------------------------------------------*/

int getDriveInfo(const int driveNum, struct drvInfo * info)
{
  struct buffer1 {
    ULONG  unitID;
    ULONG  sectorsPerUnit;
    ULONG  totalUnits;
    ULONG  availUnits;
    USHORT bytesPerUnit;
  } buf1;
  
  /* turn off all error messages */
  int rc = DosError(FERR_DISABLEHARDERR | FERR_DISABLEEXCEPTION);
  
  if (DosQueryFSInfo(driveNum, FSIL_ALLOC, &buf1, sizeof(buf1)) != NO_ERROR)
    return 0;
  else {
    info->totalSize   = ULONG(double(buf1.totalUnits) * 
			      double(buf1.bytesPerUnit) / 1024.0);
    info->availSize   = ULONG(double(buf1.availUnits) * 
			      double(buf1.bytesPerUnit) / 1024.0);
    info->usedSize    = info->totalSize - info->availSize;
    info->percentUsed = ULONG(100.0 * 
			      double(info->usedSize) / 
			      double(info->totalSize));
  }
  
  /* reenable error messages */
  rc = DosError(FERR_ENABLEHARDERR | FERR_ENABLEEXCEPTION);
  
  info->driveLetter = (driveNum - 1) + 'A';
  UCHAR drv[3];
  drv[0] = info->driveLetter;
  drv[1] = ':';
  drv[2] = 0;
  
  struct buffer2 {
    USHORT iType;
    USHORT cbName;
    USHORT cbFSDName;
    USHORT cbFSAData;
    char   buf[1024];
  } buf2;
  
  ULONG len = sizeof(buf2);
  
  if (DosQueryFSAttach(drv, 0, FSAIL_QUERYNAME, 
		       (struct FSQBUFFER2 *)&buf2, &len) == NO_ERROR) {
    strcpy(info->driverName, &buf2.buf[buf2.cbName + 1]);
    
    switch (buf2.iType) {
    case 1 :     /* should not happen at all */
      strcpy(info->remoteName, "local char device");
      break;
    case 2 :     /* ditto */
      strcpy(info->remoteName, "pseudo char device");
      break;
    case 3 :
      strcpy(info->remoteName, "local");
      break;
    case 4 :
      strcpy(info->remoteName, 
	     (char *)&buf2.buf[buf2.cbName + 1 + buf2.cbFSDName + 1]);
      break;
    default :
      strcpy(info->remoteName, "unknown device type");
      break;
    }
  }
  else {
    strcpy(info->driverName, "???");
    strcpy(info->remoteName, "???");
  }
  
  return 1;
}

/*-----------------------------------------------------------------*/

int main(int argc, char * argv[])
{
  struct drvInfo driveInfo;
  
  cout.form("Filesystem          1024-blocks       Used  Available  "
	    "Capacity  Mounted on\n");
  
  for(int i = 3; i < 27; i++)
    if (getDriveInfo(i, &driveInfo))
      cout.form("%-20s %10lu %10lu %10lu %8lu%%  %c:(%s)\n",
		driveInfo.remoteName, driveInfo.totalSize,
		driveInfo.usedSize, driveInfo.availSize,
		driveInfo.percentUsed, driveInfo.driveLetter,
		driveInfo.driverName);
  
  return 0;
}

/*
 * Local Variables:
 * compile-command: "dmake df.exe"
 * end:
 */
