/*
 * OS/2 read-ahead cache layer
 *
 * Copyleft Alex Strelnikov.
 *
 * rcache is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * rcache is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
*/


#define INCL_DOS
#define INCL_DOSERRORS

#include <os2.h>

#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <share.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "rcache.h"

#define READ_SIZE  2048
#define READ_COUNT 1024

static CacheHeader *HCache = NULL;


void APIENTRY ReadAheadThread(ULONG arg)
{
    CacheHeader *HCache;
    int BytesRead;

    HCache = (CacheHeader *)arg;

    if (HCache == NULL)
        return;

    while (HCache->ThreadStop != TRUE)
    {
        if (HCache->BufferAvail <= HCache->BufferSize - READ_SIZE &&
            HCache->BufferEOF == FALSE && HCache->CacheDisable == FALSE)
        {
            DosRequestMutexSem(HCache->BufferBusy, SEM_INDEFINITE_WAIT);

            BytesRead = read(HCache->Handle,
                             &HCache->CacheBuffer[HCache->BufferPtr],
                             READ_SIZE);

            if (BytesRead == 0) // EOF
            {
                HCache->BufferEOF = TRUE;

                DosReleaseMutexSem(HCache->BufferBusy);

                continue;
            }

            HCache->BufferPtr += BytesRead;
            HCache->BufferAvail += BytesRead;
            HCache->FilePos += BytesRead;

            if (HCache->BufferPtr + READ_SIZE > HCache->BufferSize)
                HCache->BufferPtr = 0;

            DosReleaseMutexSem(HCache->BufferBusy);
        }
        else
        {
            DosSleep(10);
        }
    }

    return;
}


int cache_open(char *patchname, int oflag)
{
    APIRET rc;

    if (HCache == NULL)
    {
        HCache = (CacheHeader *)malloc(sizeof(CacheHeader));

        HCache->Handle = open(patchname, oflag);

        if(HCache->Handle == -1)
        {
            free(HCache);

            return -1;
        }

        HCache->FilePos = 0;
        HCache->ReadFilePos = 0;
        HCache->BufferPtr = 0;
        HCache->BufferReadPtr = 0;
        HCache->BufferAvail = 0;

        HCache->BufferSize = READ_SIZE * READ_COUNT;
        HCache->CacheBuffer = (unsigned char *)malloc(HCache->BufferSize);
        HCache->BufferEOF = FALSE;
        HCache->ThreadStop = FALSE;
        HCache->CacheDisable = FALSE;

        rc = DosCreateMutexSem(NULL, &HCache->BufferBusy, 0, FALSE);

        if (rc != NO_ERROR)
        {
            free(HCache);

            return -1;
        }

        rc = DosCreateThread(&HCache->BufferThread,
                             (PFNTHREAD) ReadAheadThread,
                             (ULONG) HCache,
                             CREATE_READY,
                             65535L);
        if (rc != NO_ERROR)
        {
            free(HCache);

            return -1;
        }

        return HCache->Handle;
    }
    else
        return open(patchname, oflag);
}


int cache_close(int handle)
{
    if (HCache == NULL)
        return -1;

    if (HCache->Handle == handle)
    {
        HCache->ThreadStop = TRUE;

        DosWaitThread(&HCache->BufferThread, DCWW_WAIT);

        free(HCache);

        HCache = NULL;
    }

    return close(handle);
}


int cache_read(int handle, char *buffer, unsigned int count)
{
    int BytesRead = 0;

    if (HCache == NULL)
        return -1;

    if (HCache->Handle == handle && HCache->CacheDisable == FALSE)
    {
        while(HCache->BufferAvail < count && HCache->BufferEOF == FALSE)
            DosSleep(200);

        if (HCache->BufferAvail > 0)
        {
            DosRequestMutexSem(HCache->BufferBusy, SEM_INDEFINITE_WAIT);

            while(HCache->BufferReadPtr < HCache->BufferSize &&
                  HCache->BufferAvail > 0 && BytesRead < count)
            {
                *buffer++ = HCache->CacheBuffer[HCache->BufferReadPtr++];

                BytesRead++;
                HCache->BufferAvail--;
            }

            if (HCache->BufferReadPtr >= HCache->BufferSize)
                HCache->BufferReadPtr = 0;

            if (HCache->BufferAvail > 0 && BytesRead < count)
            {
                if (HCache->BufferReadPtr >= HCache->BufferSize)
                    HCache->BufferReadPtr = 0;

                while(HCache->BufferReadPtr < HCache->BufferSize &&
                      HCache->BufferAvail > 0 && BytesRead < count)
                {
                    *buffer++ = HCache->CacheBuffer[HCache->BufferReadPtr++];

                    BytesRead++;
                    HCache->BufferAvail--;
                }
            }

            DosReleaseMutexSem(HCache->BufferBusy);

            HCache->ReadFilePos += BytesRead;

            return BytesRead;
        }

        return 0; // EOF
    }
    else
        return read(handle, buffer, count);
}


long cache_lseek(int handle, long offset, int origin)
{
    long pos, ptr;

    if (HCache == NULL)
        return -1;

    if (HCache->Handle == handle && HCache->CacheDisable == FALSE)
    {
        DosRequestMutexSem(HCache->BufferBusy, SEM_INDEFINITE_WAIT);

        switch(origin)
        {
            case SEEK_SET:  // Beginning of file
                pos = offset;
            break;

            case SEEK_CUR:  // Current position
                pos = HCache->FilePos + offset;
            break;

            case SEEK_END:  // End of file
                pos = -1;
            break;
        }

        if (pos > HCache->ReadFilePos && pos < HCache->FilePos)
        { // In cache
            ptr = pos - HCache->ReadFilePos;

            if ((HCache->BufferSize - HCache->BufferReadPtr) > ptr)
            {
                HCache->BufferReadPtr += ptr;
                HCache->ReadFilePos += ptr;
                HCache->BufferAvail -= ptr;
            }
            else
            {
                HCache->BufferReadPtr = ptr - (HCache->BufferSize - HCache->BufferReadPtr);
                HCache->ReadFilePos += ptr;
                HCache->BufferAvail -= ptr;
            }
        }
        else
        { // Out of cache
            HCache->ReadFilePos = HCache->FilePos = lseek(handle, offset, origin);
            HCache->BufferPtr = 0;
            HCache->BufferReadPtr = 0;
            HCache->BufferAvail = 0;
            HCache->BufferEOF = FALSE;
        }

        DosReleaseMutexSem(HCache->BufferBusy);

        return HCache->FilePos;
    }
    else
        return lseek(handle, offset, origin);
}


int cache_disable(void)
{
    DosRequestMutexSem(HCache->BufferBusy, SEM_INDEFINITE_WAIT);

    HCache->CacheDisable = TRUE;

    DosReleaseMutexSem(HCache->BufferBusy);

    return 0;
}


int cache_enable(void)
{
    DosRequestMutexSem(HCache->BufferBusy, SEM_INDEFINITE_WAIT);

    HCache->CacheDisable = FALSE;

    HCache->ReadFilePos = HCache->FilePos = lseek(HCache->Handle, 0, SEEK_CUR);
    HCache->BufferPtr = 0;
    HCache->BufferReadPtr = 0;
    HCache->BufferAvail = 0;
    HCache->BufferEOF = FALSE;

    DosReleaseMutexSem(HCache->BufferBusy);

    return 0;
}
