/*
 * OS/2 read-ahead cache layer
 *
 * Copyleft Alex Strelnikov.
 *
 * cache is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cache is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
*/


#define INCL_DOS
#define INCL_DOSERRORS

#include <os2.h>

#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <share.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libdvdread/dvd_reader.h>

#include "dvd_cache.h"

#define READ_SIZE  2048
#define READ_COUNT 1024

static DVDCacheHeader *DVDCache = NULL;


void APIENTRY DVDReadAheadThread(ULONG arg)
{
    DVDCacheHeader *DVDCache;
    int BytesRead;

    DVDCache = (DVDCacheHeader *)arg;

    while (DVDCache->ThreadStop != TRUE)
    {
        DosRequestMutexSem(DVDCache->BufferBusy, SEM_INDEFINITE_WAIT);

        if (DVDCache->BufferAvail * READ_SIZE <= DVDCache->BufferSize - READ_SIZE &&
            DVDCache->BufferEOF == FALSE && DVDCache->CacheDisable == FALSE)
        {
            BytesRead = DVDReadBlocks(DVDCache->Title,
                                      DVDCache->FilePos,
                                      1,
                                      &DVDCache->CacheBuffer[DVDCache->BufferPtr]);

            if (BytesRead <= 0) // EOF
            {
                DVDCache->BufferEOF = TRUE;

                DosReleaseMutexSem(DVDCache->BufferBusy);

                continue;
            }

            DVDCache->BufferPtr += READ_SIZE;
            DVDCache->BufferAvail++;
            DVDCache->FilePos++;

            if (DVDCache->BufferPtr + READ_SIZE > DVDCache->BufferSize)
                DVDCache->BufferPtr = 0;
        }
        else
        {
            DosReleaseMutexSem(DVDCache->BufferBusy);

            DosSleep(10);
        }

        DosReleaseMutexSem(DVDCache->BufferBusy);
    }

    return;
}


int dvd_cache_open(dvd_file_t *title)
{
    APIRET rc;

    if (DVDCache == NULL)
    {
        DVDCache = (DVDCacheHeader *)malloc(sizeof(DVDCacheHeader));

        DVDCache->Title = title;
        DVDCache->FilePos = 0;
        DVDCache->ReadFilePos = 0;
        DVDCache->BufferPtr = 0;
        DVDCache->BufferReadPtr = 0;
        DVDCache->BufferAvail = 0;

        DVDCache->BufferSize = READ_SIZE * READ_COUNT;
        DVDCache->CacheBuffer = (unsigned char *)malloc(DVDCache->BufferSize);
        DVDCache->BufferEOF = FALSE;
        DVDCache->ThreadStop = FALSE;
        DVDCache->CacheDisable = FALSE;

        rc = DosCreateMutexSem(NULL, &DVDCache->BufferBusy, 0, FALSE);

        if (rc != NO_ERROR)
        {
            free(DVDCache);

            return -1;
        }

        rc = DosCreateThread(&DVDCache->BufferThread,
                             (PFNTHREAD) DVDReadAheadThread,
                             (ULONG) DVDCache,
                             CREATE_READY,
                             65535L);
        if (rc != NO_ERROR)
        {
            free(DVDCache);

            return -1;
        }

        return 0;
    }

}


int dvd_cache_close(void)
{
    if (DVDCache == NULL)
    {
        DVDCache->ThreadStop = TRUE;

        DosWaitThread(&DVDCache->BufferThread, DCWW_WAIT);

        free(DVDCache);

        DVDCache = NULL;
    }

    return 0;
}


int dvd_cache_read(int pack, unsigned char *buffer)
{
    int BytesRead = 0;

    if (DVDCache->CacheDisable == FALSE)
    {
        while(DVDCache->BufferAvail < 1 && DVDCache->BufferEOF == FALSE)
            DosSleep(200);

        if (DVDCache->BufferAvail > 0)
        {
            DosRequestMutexSem(DVDCache->BufferBusy, SEM_INDEFINITE_WAIT);

            while(DVDCache->BufferReadPtr < DVDCache->BufferSize &&
                  DVDCache->BufferAvail > 0 && BytesRead < READ_SIZE)
            {
                *buffer++ = DVDCache->CacheBuffer[DVDCache->BufferReadPtr++];

                BytesRead++;
            }

            DVDCache->BufferAvail--;

            if (DVDCache->BufferReadPtr >= DVDCache->BufferSize)
                DVDCache->BufferReadPtr = 0;

            if (DVDCache->BufferAvail > 0 && BytesRead < READ_SIZE)
            {
                if (DVDCache->BufferReadPtr >= DVDCache->BufferSize)
                    DVDCache->BufferReadPtr = 0;

                while(DVDCache->BufferReadPtr < DVDCache->BufferSize &&
                      DVDCache->BufferAvail > 0 && BytesRead < READ_SIZE)
                {
                    *buffer++ = DVDCache->CacheBuffer[DVDCache->BufferReadPtr++];

                    BytesRead++;
                }

                DVDCache->BufferAvail--;
            }

            DosReleaseMutexSem(DVDCache->BufferBusy);

            DVDCache->ReadFilePos++;

            return BytesRead / READ_SIZE;
        }

        return 0; // EOF
    }
}


long dvd_cache_lseek(long offset)
{
    long pos, ptr;

    if (DVDCache->CacheDisable == FALSE)
    {
        DosRequestMutexSem(DVDCache->BufferBusy, SEM_INDEFINITE_WAIT);

        DVDCache->FilePos = offset;
        DVDCache->ReadFilePos = 0;
        DVDCache->BufferPtr = 0;
        DVDCache->BufferReadPtr = 0;
        DVDCache->BufferAvail = 0;
        DVDCache->BufferEOF = FALSE;

        DosReleaseMutexSem(DVDCache->BufferBusy);

        return offset;
    }
}
