
/*
 *@@sourcefile xdoc.cpp:
 *      create HTML documentation from C/C++ source files.
 *
 *      See readme.txt in the same directory as this file
 *      for an introduction.
 *
 *      See the XWorkplace or WarpIN sources for how to
 *      write source code which converts into HTML.
 *
 *@@header "xdoc.h"
 *@@header "base\bs_base.h"
 */

/*      Copyright (C) 1999-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of the XFolder main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <io.h>

#include "setup.h"

#include <stdio.h>
#include <setjmp.h>

#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include "helpers\dosh.h"
#include "helpers\stringh.h"
#include "helpers\xstring.h"

#include "base\bs_base.h"           // V0.9.14 (2001-07-07) [umoeller]
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"

#include "bldlevel.h"

#include "xdoc.h"

/* ******************************************************************
 *
 *   Global variables
 *
 ********************************************************************/

list<PXDSourceFile> G_SourceFilesList(SHADOW);
            // list of all processed files, including headers added later;
            // PXDSourceFile is a REFPOINTER<XDSourceFile>
list<PXDSourceFile> G_HeaderFilesList(SHADOW);
            // list of all additional @@include headers only; these
            // items are also in SourceFilesList;
            // PXDSourceFile is a REFPOINTER<XDSourceFile>

list<PXDDefine>     G_AllDefinesList(SHADOW);
            // PXDDefine is a REFPOINTER<XDDefine>
list<PXDFunction>   G_AllFunctionsList(SHADOW);
            // PXDFunction is a REFPOINTER<XDFunction>
list<PXDClass>      G_AllClassesList(SHADOW);
            // PXDClass is a REFPOINTER<XDClass>
list<PXDClass>      G_RootClassesList(SHADOW);
            // PXDClass is a REFPOINTER<XDClass>

list<PXDDeclBase>   G_AllDeclsList(SHADOW);     // contains really everything
            // PXDDeclBase is a REFPOINTER<XDDeclBase>

list<PXDChangeLogEntry> G_ChangeLogsList(SHADOW);
list<PXDReviewEntry>    G_ReviewsList(SHADOW);

list<PXDCategory>   G_RootCategoriesList(SHADOW);
            // list of root categories only
list<PXDCategory>   G_AllCategoriesFlatList(SHADOW);
            // list of really all categories
ULONG               G_cAllCategories = 0;
            // raised with each category found

ULONG               G_ulFileCounter = 0;

list<string*>       IncludePaths(SHADOW);

BOOL                optResolveRefs = TRUE;
BOOL                optWriteHeaders = FALSE;
BOOL                optChangeLogOnly = FALSE;

ULONG               G_ulVerbosity = 1;
ULONG               ulSourceFileCount = 0;

PCSZ                G_pcszCategoryTag     = "@@category";
PCSZ                G_pcszSourcefileTag   = "@@sourcefile";
PCSZ                G_pcszIncludeTag      = "@@include ";
PCSZ                G_pcszHeaderTag       = "@@header ";
PCSZ                G_pcszSomTag          = "@@somclass ";
PCSZ                G_pcszGlossTag        = "@@gloss:";
PCSZ                G_pcszReviewTag       = "@@reviewed"; // V0.9.18 (2002-02-25) [lafaix]

// index filenames
PCSZ                G_pcszIndexCategories = "index1.htm";
PCSZ                G_pcszIndexSrcFiles = "index2.htm";
PCSZ                G_pcszIndexAll = "index3.htm";
PCSZ                G_pcszIndexFuncs = "index4.htm";
PCSZ                G_pcszIndexClasses = "index5.htm";
PCSZ                G_pcszIndexOther = "index6.htm";
PCSZ                G_pcszIndexChangesByDate = "index7.htm";
PCSZ                G_pcszIndexReviews = "index8.htm"; // V0.9.18 (2002-02-25) [lafaix]

ULONG ParseSourceFile(PXDSourceFile pSourceFile);
VOID WriteGlobalIndices();
VOID WriteChangeLogOnly();

DEFINE_CLASS(XDArgument, BSRoot);

DEFINE_CLASS(XDBase, BSRoot);
    DEFINE_CLASS(XDChangeLogEntry, BSRoot);
        DEFINE_CLASS(XDReviewEntry, XDChangeLogEntry); // V0.9.18 (2002-02-25) [lafaix]
    DEFINE_CLASS(XDDeclBase, XDBase);
        DEFINE_CLASS(XDGlossary, XDDeclBase);
        DEFINE_CLASS(XDTypedef, XDDeclBase);
        DEFINE_CLASS(XDIndexableBase, XDDeclBase);
            DEFINE_CLASS(XDDefine, XDIndexableBase);
            DEFINE_CLASS(XDFunction, XDIndexableBase);
            DEFINE_CLASS(XDClass, XDIndexableBase);
                DEFINE_CLASS(XDSomClass, XDClass);
    DEFINE_CLASS(XDCategory, XDBase);
    DEFINE_CLASS(XDSourceFile, XDBase);

/* ******************************************************************
 *
 *   Exceptions
 *
 ********************************************************************/

class Excpt
{
    public:
        APIRET  _arc;
        string  _strContext;

        Excpt(APIRET arc, const string &strContext)
            : _arc(arc), _strContext(strContext)
        {
        }

};

/* ******************************************************************
 *
 *   Comparison funcs
 *
 ********************************************************************/

short XWPENTRY CompareCategories(void *p1, void *p2, void *pStorage)
{
    if (*(PXDCategory)p1 < *(PXDCategory)p2)
        return -1;
    if (*(PXDCategory)p1 == *(PXDCategory)p2)
        return 0;
    return +1;
}

short XWPENTRY CompareChangeLogEntries(void *p1, void *p2, void *pStorage)
{
    if (*(PXDChangeLogEntry)p1 < *(PXDChangeLogEntry)p2)
        return -1;
    if (*(PXDChangeLogEntry)p1 == *(PXDChangeLogEntry)p2)
        return 0;
    return +1;
}

short XWPENTRY CompareClasses(void *p1, void *p2, void *pStorage)
{
    if (*(PXDClass)p1 < *(PXDClass)p2)
        return -1;
    if (*(PXDClass)p1 == *(PXDClass)p2)
        return 0;
    return +1;
}

short XWPENTRY CompareFunctions(void *p1, void *p2, void *pStorage)
{
    if (*(PXDFunction)p1 < *(PXDFunction)p2)
        return -1;
    if (*(PXDFunction)p1 == *(PXDFunction)p2)
        return 0;
    return +1;
}

short XWPENTRY CompareDefines(void *p1, void *p2, void *pStorage)
{
    if (*(PXDDefine)p1 < *(PXDDefine)p2)
        return -1;
    if (*(PXDDefine)p1 == *(PXDDefine)p2)
        return 0;
    return +1;
}

short XWPENTRY CompareDeclarations(void *p1, void *p2, void *pStorage)
{
    if (*(PXDDeclBase)p1 < *(PXDDeclBase)p2)
        return -1;
    if (*(PXDDeclBase)p1 == *(PXDDeclBase)p2)
        return 0;
    return +1;
}

/* ******************************************************************
 *
 *   File helpers
 *
 ********************************************************************/

/*
 *@@ WriteInfo:
 *      writes bottom HTML line.
 */

VOID WriteInfo(FILE *HTMLFile)
{
    DATETIME        dt;
    DosGetDateTime(&dt);
    fprintf(HTMLFile,
            "<FONT SIZE=-1><I>Created %04d-%02d-%02d "
            "using xdoc V"BLDLEVEL_VERSION" &copy; 1999-2002 Ulrich M&ouml;ller</I></FONT>\n",
            dt.year, dt.month, dt.day);
}

/* ******************************************************************
 *
 *   String helpers
 *
 ********************************************************************/

struct HTMLESCAPE
{
    const char *pcszFind,
               *pcszReplace;
};

HTMLESCAPE G_Escapes[] =
    {
        // there are a few things to be preserved, so
        // translate them to temporaries first
        {"&", "@@@@amp@@@@"},
        {"<B>", "@@@@BBB@@@@"},
        {"</B>", "@@@@/BBB@@@@"},
        {"<I>", "@@@@III@@@@"},
        {"</I>", "@@@@/III@@@@"},

        // replace special chars
        {"<", "&lt;"},
        {">", "&gt;"},

        {"", "&auml;"},
        {"", "&ouml;"},
        {"", "&uuml;"},
        {"", "&Auml;"},
        {"", "&Ouml;"},
        {"", "&Uuml;"},
        {"", "&szlig;"},

        {"(C)", "&copy;"},

        // last, translate preserves back
        {"@@@@BBB@@@@", "<b>"},
        {"@@@@/BBB@@@@", "</b>"},
        {"@@@@III@@@@", "<I>"},
        {"@@@@/III@@@@", "</I>"},
        {"@@@@amp@@@@", "&"}        // fixed V0.9.9 (2001-02-10) [umoeller]
    };

string  G_strEscFinds[sizeof(G_Escapes) / sizeof(G_Escapes[0])];
string  G_strEscRepls[sizeof(G_Escapes) / sizeof(G_Escapes[0])];

BOOL    G_fEscapesSet = FALSE;

/*
 * TranslateChars:
 *      translates special characters into HTML escapes.
 */

VOID TranslateChars(string &str)
{
    ULONG ul;

    if (!G_fEscapesSet)
    {
        // first run:
        // copy PSZs to strings

        for (ul = 0;
             ul < sizeof(G_Escapes) / sizeof(G_Escapes[0]);
             ul++)
        {
            G_strEscFinds[ul] = G_Escapes[ul].pcszFind;
            G_strEscRepls[ul] = G_Escapes[ul].pcszReplace;
        }

        G_fEscapesSet = TRUE;
    }

    // now replace stuff in "str"
    for (ul = 0;
         ul < sizeof(G_Escapes) / sizeof(G_Escapes[0]);
         ul++)
    {
        size_type   ulOfs = 0;
        while (str._find_replace(G_strEscFinds[ul],
                                 G_strEscRepls[ul],
                                 &ulOfs)
                != string::npos)
            ;
    }
}

/*
 * FormatLines:
 *      this formats a C-style comment in the given buffer:
 *      -- leading "*" chars are removed in each line;
 *      -- lines with leading "+" are converted to PRE;
 *      -- leading "--" characters are converted to LI tags;
 *      -- empty lines are converted to P tags.
 *
 *      The new comment starts out with the real comment tag
 *      --  not with a paragraph (P) tag.
 *
 *@@changed V0.9.3 (2000-05-19) [umoeller]: totally rewritten.
 *@@changed V0.9.7 (2001-01-07) [umoeller]: rewritten again... this was too shaky.
 */

VOID FormatLines(string &strComment)
{
    if (strComment.length())
    {
        string strNew = "\n"; // "<!-- begin comment: ---" + strComment + "--- -->\n";

        // reallocate the buffer to make room for string
        // replacements later... make sure we have at
        // least 20% more room than the length of the string.
        // This will reduce allocations for the functions in here,
        // but will also give us plenty of memory for the link
        // resolves later... since this string will be stored
        // in the declarations.
        strNew.reserve(strComment.length() * 12 / 10);

        // string strNew = "<!-- begin comment-->\n";
        BOOL fQuit = FALSE;

        PSZ pszComment = (PSZ)strComment.c_str();

        PCSZ pStartOfLine = pszComment;

        // skip leading newline...
        if (*pStartOfLine == '\n')
            pStartOfLine++;

        ULONG ulListLevel = 0;
                // this is > 0 if we're in a UL block
        ULONG ulLeadingSpacesList = 0;
                // if (ulListLevel > 0), this contains the no. of leading
                // spaces that lead to the indentation...

        BOOL fNeedsParagraph = FALSE;

        do
        {
            PCSZ pNextLine = strchr(pStartOfLine, '\n');
            if (!pNextLine)
            {
                // no next line found:
                pNextLine = pStartOfLine + strlen(pStartOfLine);
                fQuit = TRUE;
            }

            // skip leading spaces in this line
            PCSZ pCopyLineFrom = pStartOfLine;
            while ((*pCopyLineFrom) && (*pCopyLineFrom == ' '))
                pCopyLineFrom++;

            string strNewLine;
            BOOL fAddParagraph = FALSE,
                 fAddEndPre = FALSE;
            ULONG ulLeadingSpacesThis = 0;

            if (*pCopyLineFrom)
            {
                fAddParagraph = FALSE;

                switch (*pCopyLineFrom)
                {
                    case '\n':
                        // empty line:
                        fAddParagraph = TRUE;
                        pCopyLineFrom = NULL;   // skip subsequent
                    break;

                    case '*':
                    {
                        // copy after this
                        ++pCopyLineFrom;
                    break; }

                    case '+':
                        strNew += "<pre>";
                        // copy after this
                        ++pCopyLineFrom;
                        fAddEndPre = TRUE;
                        fNeedsParagraph = FALSE;
                    break;

                    // default: any other character, copy entire line
                }

                if ((pCopyLineFrom) && (*pCopyLineFrom)) // points to after "*", probably
                {
                    PCSZ pEOL = strchr(pCopyLineFrom, '\n');
                    if (!pEOL)
                    {
                        pEOL = pCopyLineFrom + strlen(pCopyLineFrom);
                        fQuit = TRUE;
                    }

                    // again, strip leading spaces... if we're not in PRE
                    if (!fAddEndPre)
                        while ((*pCopyLineFrom) && (*pCopyLineFrom == ' '))
                        {
                            ulLeadingSpacesThis++;
                            pCopyLineFrom++;
                        }

                    if (pEOL > pCopyLineFrom)
                    {
                        strNewLine.assign(pCopyLineFrom, pEOL + 1);
                    }
                    else
                        fAddParagraph = TRUE;
                }
                else
                    fQuit = TRUE;

                if (fAddParagraph)
                {
                    if (!fAddEndPre)
                        fNeedsParagraph = TRUE;
                    strNew += '\n';
                    // and nothing else at this point...
                    // on the next loop, we'll add <p>
                }

                /* string strTest;
                strTest._printf("<!-- len: %d, lead: %d p: %d-->",
                                strNewLine.length(),
                                ulLeadingSpacesThis,
                                fNeedsParagraph);
                strNew += strTest; */

                if (strNewLine.length())
                {
                    // ok, we have something for the new line...
                    // if the line starts with "--", add a list bullet

                    // check for "--" at beginning of line
                    // for list beginnings...
                    BOOL fSkipRest = FALSE;
                    if (!fAddEndPre)
                    {
                        PCSZ p = strNewLine.c_str();
                        if (    (*p == '-')
                             && (*(p+1) == '-')
                           )
                        {
                            if (ulListLevel == 0)
                            {
                                strNew += "<ul><li>";
                                ulListLevel++;
                                // store current leading spaces count
                                // so we can detect the end of the list
                                ulLeadingSpacesList = ulLeadingSpacesThis;
                            }
                            else
                                // we're already in a list:
                                strNew += "<p><li>";

                            // skip "--"
                            strNew += (p+2);
                            fNeedsParagraph = FALSE;
                            fSkipRest = TRUE;
                        }
                    }

                    if (!fSkipRest)
                    {
                        // add regular line...

                        if (fNeedsParagraph)
                        {
                            // we had an empty line on the last loop:
                            // if we're in a list, check the indentation of this line...
                            // if it's about the same as at the time the list
                            // was started, termiante the list
                            if (    (ulListLevel)
                                 && (ulLeadingSpacesList / 3 == ulLeadingSpacesThis / 3)
                               )
                            {
                                // stop list
                                ulListLevel--;
                                strNew +=  "</ul>";
                            }
                            else
                                strNew +=  "<p>";

                            fNeedsParagraph = FALSE;
                        }

                        strNew += strNewLine;
                    }
                } // end if (strNewLine.length())

                if (fAddEndPre)
                    strNew += "</pre>";

            } // end if (*pCopyLineFrom)

            // pNextLine points to \n char now
            pStartOfLine = pNextLine + 1;

        } while (!fQuit);

        // close open list levels, if any are left
        while (ulListLevel--)
            strNew += "</ul>\n";

        // now remove the duplicate PRE tags
        string strFind = "</pre><pre>";
        string strReplace = "";
        ULONG ulPos = 0;
        while (strNew._find_replace(strFind,
                                    strReplace,
                                    &ulPos)
                         != string::npos)
                ;

        // replace the original with the new string
        strComment.swap(strNew);
            // this is faster than
            // strComment = strNew;
    }


    /* PSZ pszComment = strdup(strComment.c_str());

    if (pszComment)
    {
        // overwrite leading " * " chars in each line
        PSZ p = pszComment;
        while ((p = strchr(p, '\n')))
        {
            ++p;
            while (*p == ' ')
                p++;

            if (*p == '*')
                *p = ' ';
        }

        // now replace leading " +" stuff with <pre>
        p = pszComment;
        while ((p = strstr(p, "\n +")))
        {
            ++p;
            PSZ p2 = strchr(p, '\n');
            PSZ pszTemp = (PSZ)malloc(strlen(pszComment)+20);

            PSZ pSource = pszComment;
            PSZ pTarget = pszTemp;

            while (pSource <= p)
                *pTarget++ = *pSource++;

            strcpy(pTarget, "<pre>");
            pTarget += 5;

            pSource = p + 4;
            while (pSource < p2)
                *pTarget++ = *pSource++;

            strcpy(pTarget, "</pre>\n");
            pTarget += 7;

            pSource = p2+1;
            while (*pSource)
                *pTarget++ = *pSource++;
            *pTarget = 0;

            free(pszComment);
            pszComment = pszTemp;
            p = pszTemp;
        }

        // now remove leading spaces in each line
        // and check for list tags

        if (pszComment)
            if (strlen(pszComment))
            {
                ULONG   ulLastSpaceCountLI = 0,
                        ulIndentSpacesCurrent = 0;
                LONG    lIndentationLevel = 0;

                PSZ pTemp = (PSZ)malloc(3*strlen(pszComment));

                PSZ pSource = pszComment;
                PSZ pTarget = pTemp;
                // PSZ pszInsertNext = NULL;
                BOOL fInsertParagraph = FALSE;
                while (*pSource)
                {
                    PSZ pszInsert = NULL;
                    if (*pSource == '\n')
                    {
                        // newline character:
                        // copy char and take next
                        *pTarget++ = *pSource++;

                        // skip over leading spaces in this line
                        ULONG ulSpaceCountThis = 0;
                        while (*pSource == ' ')
                        {
                            ++ulSpaceCountThis;
                            ++pSource;
                        }

                        // pSource now points to the first non-space char in the line
                        if (*pSource == '\n')
                        {
                            // another newline -- empty line:
                            // this normally inserts a <p>.
                            // However, before inserting this,
                            // we should check if maybe a "--" comes up
                            // next. So delay this until we come to the
                            // first "real" character.
                            fInsertParagraph = TRUE;
                            pszInsert = "\n";
                        }
                        else
                            // line is not empty:
                            // check for "--", which we might need to convert
                            if ((*pSource == '-') && (*(pSource+1) == '-'))
                            {
                                // OK, we got "--":
                                // at this point, pszNext might have a "<p>".
                                // We'll just ignore that and insert "<LI"> instead.
                                if ((ulSpaceCountThis / 3) > (ulLastSpaceCountLI / 3))
                                {
                                    // indentation larger:
                                    if (lIndentationLevel > 0)
                                        pszInsert = "<p><ul><li>";
                                    else
                                        pszInsert = "<ul><li>";
                                    ulIndentSpacesCurrent = ulSpaceCountThis + 3;
                                    lIndentationLevel++;
                                }
                                else if ((ulSpaceCountThis / 3) < (ulLastSpaceCountLI / 3))
                                {
                                    // indentation smaller:
                                    // rare case, must be
                                    //          -- inner
                                    //      -- outer
                                    pszInsert = "</ul><li>";
                                    lIndentationLevel--;
                                }
                                else
                                    // same:
                                    pszInsert = "<p><li>";

                                ulLastSpaceCountLI = ulSpaceCountThis;

                                // skip "--"
                                pSource += 2;

                                // and don't insert <p> later
                                fInsertParagraph = FALSE;
                            }
                            else
                                // some other character.
                                // Check if we had a "<p>" queued up before
                                if (fInsertParagraph)
                                {
                                    if (    (ulSpaceCountThis / 3)
                                          < (ulIndentSpacesCurrent / 3)
                                       )
                                    {
                                        if (lIndentationLevel > 0)
                                            pszInsert = "</ul><p>";
                                        else
                                            pszInsert = "</ul>";
                                        lIndentationLevel--;
                                        ulIndentSpacesCurrent = ulSpaceCountThis;
                                    }
                                    else
                                        pszInsert = "<p>";
                                    fInsertParagraph = FALSE;
                                }
                    }

                    if (pszInsert)
                    {
                        strcpy(pTarget, pszInsert);

                        pTarget += strlen(pszInsert);
                    }
                    else
                        // just copy char:
                        *pTarget++ = *pSource++;
                }

                *pTarget = 0;

                while (lIndentationLevel > 0)
                {
                    strcat(pTemp, "</ul>\n");
                    lIndentationLevel--;
                }

                free(pszComment);
                pszComment = pTemp;

                // remove redundant </pre>\n<pre> stuff
                p = pszComment;
                while ((p = strstr(p, "</pre>\n<pre>")))
                    memcpy(p, "           \n", 12);
            }
            */
}

/*
 * FormatComment:
 *      this formats pszComment and returns a new buffer
 *      containing the new, formatted comment.
 *
 *      This calls FormatLines in turn.
 */

string FormatComment(string &strComment)
{
    string strNewComment;

    if (G_ulVerbosity > 4)
        printf("\nFormatting comment %s\n", strComment.c_str());

    if (strComment.length())
    {
        // skip the first line if it has a ":" char in it
        ULONG   ulStart = strComment.find(":"),
                ulEnd = strComment.find("*/");
        if (ulEnd == string::npos)
        {
            printf("\nWARNING: invalid comment found.\n");
            fflush(stdout);
        }
        else
        {
            if (ulStart == string::npos)
                ulStart = 0;
            else
            {
                // we had a ":": skip to after end of line
                ulStart = strComment.find("\n", ulStart);
                if (ulStart == string::npos)
                    ulStart = 0;
                // ulStart++;
            }

            if (ulEnd > ulStart)
            {
                strNewComment = strComment.substr(ulStart, (ulEnd - ulStart));

                // create HTML escapes
                TranslateChars(strNewComment);

                // then turn everything into new stuff
                FormatLines(strNewComment);

                // capitalize first letter
                PSZ p = (PSZ)strNewComment.c_str();
                while (     (*p)
                        &&  (    (*p == ' ')
                              || (*p == '\n')
                            )
                      )
                    ++p;
                if (*p)
                    *p = toupper(*p);
            }
        }
    }

    return (strNewComment);
}

/*
 *@@ FormatPRE:
 *      this makes all 0d0a pairs single 0a (\n) chars,
 *      because Netscape will display two line breaks
 *      in <pre> blocks otherwise.
 */

VOID FormatPRE(string &str)
{
    PSZ p = (PSZ)str.c_str();

    while (p)
    {
        p = strchr(p, 0x0d);
        if (p)
            if (*(p+1) == 0x0a)
            {
                *p  = ' ';
                *(p+1) = 0x0a;
                p += 2;
            }
    }

    TranslateChars(str);
}

/*
 * ResolveRefs:
 *      this takes any text as input and turns all
 *      function and file names into proper <A...> tags.
 *
 *      This gets called once for every comment buffer
 *      which was encountered after all files have been
 *      parsed.
 *
 *      Most notably, this gets called from:
 *
 *      -- main() (on each category comment)
 *
 *      -- main() (on each sourcefile comment)
 *
 *      -- XDDeclBase::Resolve (to resolve things in the comment)
 *
 *      -- XDClass::Resolve (to additionally resolve things in
 *         the class definition)
 *
 *      Note: This goes thru G_AllDeclsList for every call. In
 *      order to successfully replace class members, that list
 *      contains class members BEFORE the containing classes.
 *      So we search for "Class::method" before searching for
 *      "Class".
 *
 *@@changed V0.9.7 (2001-01-15) [umoeller]: this function is now SIGNIFICANTLY faster
 *@@changed V0.9.14 (2001-07-14) [umoeller]: added typedefs support
 */

VOID ResolveRefs(string &strComment,
                 string *pstrCurrentFunction) // in: function name to print in bold instead; can be NULL
{
    string      strReplaceWith,
                strSearchString2;

    // loop 1: function names to search for
    list<PXDDeclBase>::iterator FuncStart2 = G_AllDeclsList.begin(),
                                FuncEnd2 = G_AllDeclsList.end();
    for (;
         FuncStart2 != FuncEnd2;
         ++FuncStart2)
    {
        PXDDeclBase ppThis = *FuncStart2;
        BOOL        fPrintBoldOnly = FALSE;

        // is this the "current function" to be printed in bold?
        if (pstrCurrentFunction)
            if (ppThis->_strSearchString == *pstrCurrentFunction)
                fPrintBoldOnly = TRUE;

        // replacement string:
        BOOL fReplacementStringCreated = FALSE;

        // replace-all loop
        ULONG ulFindPos = 0,
              ulFound = string::npos,
              cbReplaceWith = strReplaceWith.size();
        // size_t  ShiftTable[256];
        // int     fRepeat = FALSE;
        while ((ulFound = strComment._find_word(ppThis->_strSearchString,
                                                ulFindPos))
                    != string::npos)
        {
            // ulFound now has offset of strSearchString:

            // do we have a replacement string yet?
            if (!fReplacementStringCreated)
            {
                if (fPrintBoldOnly)
                    // current function:
                    // print in bold instead of anchor
                    strReplaceWith._printf("<U>%s</U>",
                                           ppThis->_strReplacementString.c_str());
                else
                    // not current function:
                    // if this is a typedef, replace it not with
                    // the typedef, but with the thing it really
                    // points to V0.9.14 (2001-07-14) [umoeller]
                    if (ppThis->IsA(XDTypedef::tXDTypedef))
                    {
                        // printf("found typedef %s\n", ppThis->_strSearchString.c_str());
                        XDDeclBase *pp2 = ((XDTypedef*)ppThis)->_pLinkedTo;
                        strReplaceWith._printf("<a href=\"%s\">%s</A>",
                                               // use the typedef's target as output
                                               // file name
                                               pp2->QueryOutputFilename(),
                                               ppThis->_strReplacementString.c_str());
                    }
                    else
                        // no typedef:
                        // replace with link
                        strReplaceWith._printf("<a href=\"%s\">%s</A>",
                                               ppThis->QueryOutputFilename(),
                                               ppThis->_strReplacementString.c_str());
                                                // now using strReplacementString
                                                // V0.9.9 (2001-02-10) [umoeller]
            }

            // replace that!
            strComment.replace(ulFound,     // first char to replace
                               // no. of chars to replace:
                               ppThis->_strSearchString.size(),
                               // replace with:
                               strReplaceWith);

            /* strComment._find_replace(*pstrSearchString,
                                     strReplaceWith,
                                     &ulFound); */

            // search on after found occurence
            ulFindPos = ulFound + cbReplaceWith;
        }
    } // for (; FuncStart2 != FuncEnd2; ...

    // loop 2: file names to search for
    list<PXDSourceFile>::iterator FileStart2 = G_SourceFilesList.begin(),
                                  FileEnd2 = G_SourceFilesList.end();
    for (;
         FileStart2 != FileEnd2;
         ++FileStart2)
    {
        PXDSourceFile pSourceThis = *FileStart2;
        if (pSourceThis->WillBeWritten())
        {
            // replacement string
            string strReplaceWith;
            BOOL fReplacementStringCreated = FALSE;

            // replace-all loop
            ULONG ulFindPos = 0,
                  ulFound = string::npos,
                  cbReplaceWith = strReplaceWith.size();
            while ((ulFound = strComment._find_word(pSourceThis->_strIndexNameOrig,
                                                    ulFindPos))
                        != string::npos)
            {
                // ulFound now has offset of pSourceThis->_strIndexName:

                // do we have a replacement string yet?
                if (!fReplacementStringCreated)
                {
                    strReplaceWith._printf("<a href=\"%s\">%s</A>",
                                           pSourceThis->QueryOutputFilename(),
                                           pSourceThis->_strIndexNameHTML.c_str());
                    fReplacementStringCreated = TRUE;
                }

                // replace that!
                strComment.replace(ulFound,     // first char to replace
                                   // no. of chars to replace:
                                   pSourceThis->_strIndexNameOrig.size(),
                                   // replace with:
                                   strReplaceWith);

                /* strComment._find_replace(pSourceThis->_strIndexName,
                                         strReplaceWith,
                                         &ulFound); */

                // search on after the part we just replaced
                ulFindPos = ulFound + cbReplaceWith;
            }

        }
    } // end for (; FileStart2 != FileEnd2; ...
}

/*
 *@@ FindNotInCppComment:
 *      like BSString::find, but makes sure the
 *      find string is not in a C++ comment.
 *
 *@@added V0.9.18 (2002-03-16) [umoeller]
 */

ULONG FindNotInCppComment(const string &str,
                          const char *pcsz,
                          ULONG ulOfs)
{
    string str2(str, ulOfs);

    ULONG ulNextComment,
          ulNextLine = 0;
    for (;;)
    {
        ulNextComment = str2.find("//", ulNextLine);
        if (ulNextComment != string::npos)
        {
            str2.reserve(0);
            ulNextLine = str2.find('\n', ulNextComment);
            PSZ psz = (PSZ)str2.c_str() + ulNextComment;
            if (ulNextLine == string::npos)
            {
                memset(psz,
                       ' ',
                       str2.size() - ulNextComment);
                break;
            }
            else
            {
                memset(psz,
                       ' ',
                       ulNextLine - ulNextComment);
                ++ulNextLine;
            }
        }
        else
            break;
    }

    ULONG ulFound = ulFound = str2.find(pcsz);
    if (ulFound != string::npos)
        return ulFound + ulOfs;

    return string::npos;

    /* ULONG ulStartOfLine = ulOfs;
    for (;;)
    {
        ULONG   ulFound = str.find(pcsz, ulOfs);

        if (ulFound != string::npos)
        {
            // found string:
            // do we have a c++ comment in this line?
            ULONG ulNextComment = str.find("//", ulStartOfLine);
            if (ulNextComment != string::npos)
            {
                ULONG ulNextLine = str.find('\n', ulNextComment);
                if (    (ulFound > ulNextComment)
                     && (ulFound < ulNextLine)
                   )
                {
                    ulOfs = ulFound + 1;
                    ulStartOfLine = ulFound + 1;
                    continue;
                }
                else
                    // pcsz is outside comment in this line
                    return ulFound;
            }
            else
                return ulFound;
        }
        else
            break;
    };

    return (string::npos); */
}

/*
 *@@ FindClass:
 *      returns the XDClass pointer for the given
 *      class name or NULL if not found.
 *
 *@@added V0.9.7 (2001-01-07) [umoeller]
 */

PXDClass FindClass(string &strClassName)
{
    list<PXDClass>::iterator ClassStart = G_AllClassesList.begin(),
                             ClassEnd = G_AllClassesList.end();
    for (;
         ClassStart != ClassEnd;
         ClassStart++)
    {
        PXDClass pClassThis = *ClassStart;
        if (    (pClassThis->_strNameForResolve == strClassName)
             || (pClassThis->_strIndexNameOrig == strClassName)
           )
            return (pClassThis);
    }

    return (NULL);
}

/*
 *@@ PrintClass:
 *
 *@@added V0.9.7 (2001-01-07) [umoeller]
 */

VOID PrintClass(FILE *ThisFile,
                XDClass *pClsThis2,
                XDClass *pClassBold)
{
    string strDescendants;
    if (pClsThis2->_cDescendants)
        if (pClsThis2->_cDescendants == 1)
            strDescendants = " (1 descendant)";
        else
            strDescendants._printf(" (%d descendants)",
                                   pClsThis2->_cDescendants);

    if (pClsThis2 == pClassBold)
        fprintf(ThisFile, "\n<li><b>%s</b>%s",
                pClsThis2->_strIndexNameHTML.c_str(),
                strDescendants.c_str());
    else
        fprintf(ThisFile, "\n<li><a href=\"%s\">%s</A>%s",
                pClsThis2->QueryOutputFilename(),
                pClsThis2->_strIndexNameHTML.c_str(),
                strDescendants.c_str());
}

/*
 *@@ DumpDescendants:
 *      writes out the descendants of a class in a
 *      bullet list. Does nothing if there are no
 *      descendants for this class.
 *
 *      Preconditions: main() must have resolved the
 *      class hierarchies first.
 *
 *      This recurses.
 *
 *@@added V0.9.7 (2001-01-07) [umoeller]
 */

VOID DumpDescendants(FILE* ThisFile,
                     XDClass *pClassRoot,
                     XDClass *pClassBold)       // in: class to print in bold
{
    if (pClassRoot->_cDescendants)
    {
        fprintf(ThisFile, "\n<ul>");

        list<PXDClass>::iterator DescStart = pClassRoot->_DescendantsList.begin(),
                                 DescEnd = pClassRoot->_DescendantsList.end();
        for (;
             DescStart != DescEnd;
             DescStart++)
        {
            PXDClass pClsThis = *DescStart;
            XDClass *pClsThis2 = pClsThis;
            PrintClass(ThisFile,
                       pClsThis2,
                       pClassBold);

            // recurse!
            DumpDescendants(ThisFile,
                            pClsThis,       // as root
                            pClassBold);
        }
        fprintf(ThisFile, "\n</ul>");
    }
}

/*
 *@@ DumpRootAndDescendants:
 *
 *@@added V0.9.7 (2001-01-07) [umoeller]
 *@@changed V0.9.14 (2001-07-14) [umoeller]: no longer dumping siblings for parent classes
 */

VOID DumpRootAndDescendants(FILE* ThisFile,
                            XDClass *pClass)
{
    // climb up the parents tree
    list<XDClass*> ParentsList(SHADOW);

    // if the class has a parent, climb up the parents
    // tree and insert the parents too; however, do
    // not dump the sibling classes of the parent classes
    XDClass *pThis = pClass;
    while (pThis->_pParent)
    {
        ParentsList.push_front(pThis->_pParent);
        pThis = pThis->_pParent;
    }

    ParentsList.push_back(pClass);

    list<PXDClass>::iterator DescStart = ParentsList.begin(),
                             DescEnd = ParentsList.end();
    for (;
         DescStart != DescEnd;
         DescStart++)
    {
        pThis = *DescStart;
        fprintf(ThisFile, "\n<ul>");
        PrintClass(ThisFile,
                   pThis,
                   pClass);
    }

    DumpDescendants(ThisFile,
                    pClass,
                    pClass);

    DescStart = ParentsList.begin();
    DescEnd = ParentsList.end();
    for (;
         DescStart != DescEnd;
         DescStart++)
    {
        fprintf(ThisFile, "\n</ul>");
    }
}

/*
 *@@ XDBase:
 *
 *@@added V0.9.18 (2002-03-16) [umoeller]
 */

XDBase::XDBase(const string &strIndexName,
               BSClassID &Class)
    : BSRoot(Class),
      _ChangeLogsList(SHADOW)
{
    _strIndexNameOrig = strIndexName;
    _strIndexNameHTML = strIndexName;
    TranslateChars(_strIndexNameHTML);
    _pSourceFile = 0;
    _pMyCategory = 0;
}

/* ******************************************************************
 *
 *   XDArgument implementation
 *
 ********************************************************************/

/*
 *@@ XDArgument:
 *      default constructor.
 */

XDArgument::XDArgument(const string &strArgument_,
                       const string &strArgComment_,
                       BOOL *pfHasNonVoidArgs)
    : BSRoot(tXDArgument)
{
    ULONG ulLastSpace = strArgument_.rfind(' ');
    if (ulLastSpace != string::npos)
    {
        // normalize
        string strTemp = strArgument_;

        // normalize args string so that we always have the
        // starts to the right

        ULONG ulOfs = 0;
        ULONG cCount = 0;
        while (strTemp._find_replace("* ", " *", &ulOfs) != string::npos)
            ++cCount;
                    // for "const char** ppcsz" we get
                    //                  ^ ulLastSpace
                    //      const char* *ppcsz
                    //      const char **ppcsz
        ulOfs = 0;
        while (strTemp._find_replace("& ", " &", &ulOfs) != string::npos)
            ++cCount;

        ULONG ulLastStar = strTemp.rfind('*');
        if (ulLastStar == string::npos)
            ulLastStar = strTemp.rfind('&');
        if (ulLastStar == string::npos)
            ulLastStar = strTemp.rfind(' ');

        _strType = strTemp.substr(0, ulLastStar + 1);
        _strType._format();
        _strIdentifier = strTemp.substr(ulLastStar + 1);
        _strIdentifier._format();
    }
    else
        _strType = strArgument_;

    if (    (_strType())
         && (_strType != "VOID")
         && (_strType != "void")
       )
        *pfHasNonVoidArgs = TRUE;

    _strArgComment = strArgComment_;
    FormatLines(_strArgComment);
}

/* ******************************************************************
 *
 *   XDChangeLogEntry
 *
 ********************************************************************/

/*
 *@@ XDChangeLogEntry:
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

XDChangeLogEntry::XDChangeLogEntry(XDDeclBase *pDecl,   // in: declaration or NULL
                                   XDSourceFile *pSourceFile, // in: source file or NULL
                                   string &strDeclName, // in: for error msgs
                                   PSZ pFound,      // start of line in comment
                                   BOOL fAdded)     // TRUE: added or changed?
    : BSRoot(tXDChangeLogEntry)
{
    PSZ pError = NULL;

    _pMyDecl = pDecl;
    _pMySourceFile = pSourceFile;

    _fAdded = fAdded;

    if (ParseEntry(pFound, &pError))
    {
        // no error:
        // store in instance list
        if (_pMyDecl)
        {
            _pMyDecl->_ChangeLogsList.push_back(this);
            if ((fAdded) && (_pMyDecl->_pAddedLog == NULL))
                // first added log for this:
                _pMyDecl->_pAddedLog = this;
        }
        if (_pMySourceFile)
        {
            _pMySourceFile->_ChangeLogsList.push_back(this);
            if ((fAdded) && (_pMySourceFile->_pAddedLog == NULL))
                // first added log for this:
                _pMySourceFile->_pAddedLog = this;
        }
        // store in global list
        G_ChangeLogsList.push_back(this);
    }
    else
    {
        // don't store this entry anywhere
        printf("\n  Skipping change log entry for \"%s\", line was:"
               "\n       %s",
               strDeclName.c_str(),
               pError);

        free(pError);
    }
}

/*
 *@@ ParseEntry:
 *      read the entry starting at pEntry and fill the corresponding
 *      members.
 *
 *      The expected format is as follows:
 *
 +        @@tag Va.b.c (yyyy-mm-dd) [author]: description
 *
 *      Returns TRUE if the entry was successfuly read, FALSE otherwise.
 *
 *      If an error occured, *ppError points to a buffer containing the
 *      incorrect entry.  The caller must free it in this case.
 *
 *@@added V0.9.18 (2002-02-25) [lafaix]
 */

BOOL XDChangeLogEntry::ParseEntry(PSZ pEntry, PSZ *ppError)
{
    BOOL ok = TRUE;

    // clear the error buffer pointer
    *ppError = NULL;

    // pEntry has start of entry tag:
    PSZ pEOL = strhFindEOL(pEntry, NULL);

    // extract line
    PSZ pLine = strhSubstr(pEntry, pEOL);
    if (pLine)
    {
        // find first space after @@ tag
        PSZ p = pLine;
        while ( (*p) && (*p != ' ') )
            p++;
        if (*p)
        {
            BOOL fContinue = TRUE;
            BOOL fVersion = FALSE,
                 fDate = FALSE,
                 fAuthor = FALSE;

            while ((fContinue) && (p) && (ok))
            {
                switch (*p)
                {
                    case 'V':
                        // version number:
                        if (sscanf(p+1, "%u.%u.%u",
                                   &_version._ulMajor,
                                   &_version._ulMinor,
                                   &_version._ulRevision)
                                != 3)
                        {
                            printf("\n  --> Warning: invalid version number in entry");
                            fflush(stdout);
                            ok = FALSE;
                        }
                        else
                            fVersion = TRUE;

                        p = strchr(p, ' ');
                    break;

                    case '(':   // date
                        if (fDate)
                            // already parsed:
                            p++;
                        else
                        {
                            if (sscanf(p+1, "%d-%d-%d",
                                       &_date._ulYear,
                                       &_date._ulMonth,
                                       &_date._ulDay)
                                   != 3)
                            {
                                printf("\n  --> Warning: invalid date in entry");
                                fflush(stdout);
                                ok = FALSE;
                            }
                            else
                            {
                                fDate = TRUE;

                                // fix two-digit dates
                                if (    (_date._ulYear > 30)
                                     && (_date._ulYear < 100)
                                   )
                                    _date._ulYear += 1900;
                            }

                            p = strchr(p, ')');
                            if (!p)
                                ok = FALSE;
                        }
                    break;

                    case '[':   // author
                        if (fAuthor)
                            // already parsed:
                            p++;
                        else
                        {
                            PSZ p2 = strchr(p, ']');
                            if (!p2)
                            {
                                printf("\n  --> Warning: invalid author in entry");
                                fflush(stdout);
                                ok = FALSE;
                            }
                            else
                            {
                                _strAuthor.assign(p+1, p2);
                                fAuthor = TRUE;
                            }

                            p = p2 + 1;
                        }
                    break;

                    case ':': // the colon terminates the head of the version string,
                              // so get the rest of the line and stop
                        p++;
                        // skip spaces
                        while ( (*p) && (*p == ' ') )
                            p++;
                        if (*p)
                            _strDescription = p;
                        fContinue = FALSE;
                    break;

                    case 0:
                        fContinue = FALSE;
                    break;

                    default:
                        // skip spaces
                        p++;
                    break;
                }
            } // while (fContinue)

        }

        while (     (*pEOL)
                 && (   (*pEOL == '\n')
                     || (*pEOL == '\r')
                    )
              )
            pEOL++;

        // clear change log in comment
        strcpy(pEntry, pEOL);

        if (ok)
            free (pLine);
        else
            *ppError = pLine;
    } // end if (pLine)

    return (ok);
}

/*
 *@@ WriteHTMLDescription:
 *      writes the long description for this entry.
 *
 *      This is a virtual method, and is currently overriden by
 *      XDReviewEntry.
 *
 *@@added V0.9.18 (2002-02-25) [lafaix]
 */

VOID XDChangeLogEntry::WriteHTMLDescription(FILE *ThisFile)
{
    if (_pMyDecl)
    {
        // log entry for declaration:
        fprintf(ThisFile,
                "<A HREF=\"%s\">%s</A> (",
                _pMyDecl->QueryOutputFilename(),
                _pMyDecl->_strIndexNameHTML.c_str()); // GetFirstIdentifier().c_str()); // short name
        // append source file name
        fprintf(ThisFile, "%s",
                _pMyDecl->_pSourceFile->_strLongFilename.c_str());

        // if it's a class method, add the class name in brackets
        /* if (_pMyDecl->_pOwningClass)
        {
            string strClassName = _pMyDecl->_pOwningClass->GetFirstIdentifier();
            fprintf(IndexFile, ", Class: %s", strClassName.c_str());
        } */
        fprintf(ThisFile,
                ")");
    }
    else if (_pMySourceFile)
        // log entry for source file:
        fprintf(ThisFile,
                "File <A HREF=\"%s\">%s</A>",
                _pMySourceFile->QueryOutputFilename(),
                _pMySourceFile->_strIndexNameHTML.c_str());

    if (_fAdded)
        fprintf(ThisFile, " added");
    else
        fprintf(ThisFile, " changed");

    if (_strAuthor())
        fprintf(ThisFile, " by <I>%s</I>", _strAuthor.c_str());

    // add date
    if (_date._ulYear)
    {
        fprintf(ThisFile,
                " (%04d-%02d-%02d)",
                _date._ulYear,
                _date._ulMonth,
                _date._ulDay);
    }

    if (_strDescription())
        fprintf(ThisFile, ":\n<BR>%s", _strDescription.c_str());
}

/* ******************************************************************
 *
 *   XDReviewEntry
 *
 ********************************************************************/

/*
 *@@ XDReviewEntry:
 *
 *@@added V0.9.18 (2002-02-25) [lafaix]
 */

XDReviewEntry::XDReviewEntry(XDSourceFile *pSourceFile, // in: source file
                             PSZ pFound)                // start of line in comment
    : XDChangeLogEntry(tXDReviewEntry)
{
    PSZ pError = NULL;

    _pMySourceFile = pSourceFile;

    if (ParseEntry(pFound, &pError))
    {
        // no error:
        _pMySourceFile->_ChangeLogsList.push_back(this);
        _pMySourceFile->_ReviewsList.push_back(this);
        _pMySourceFile->_ulReviewCount++;
        // store in global list
        G_ChangeLogsList.push_back(this);
        G_ReviewsList.push_back(this);
    }
    else
    {
        // don't store this entry anywhere
        printf("\n  Skipping entry for \"%s\", line was:"
               "\n       %s",
               "@@todo", //strDeclName.c_str(),
               pError);

        free(pError);
    }
}

/*
 *@@ WriteHTMLDescription:
 *      writes the long description for this entry.
 *
 *@@added V0.9.18 (2002-02-25) [lafaix]
 */

VOID XDReviewEntry::WriteHTMLDescription(FILE *ThisFile)
{
    // log entry for source file:
    fprintf(ThisFile,
            "File <A HREF=\"%s\">%s</A>",
            _pMySourceFile->QueryOutputFilename(),
            _pMySourceFile->_strIndexNameHTML.c_str());

    fprintf(ThisFile, " reviewed");

    if (_strAuthor())
        fprintf(ThisFile, " by <I>%s</I>", _strAuthor.c_str());

    // add date
    if (_date._ulYear)
    {
        fprintf(ThisFile,
                " (%04d-%02d-%02d)",
                _date._ulYear,
                _date._ulMonth,
                _date._ulDay);
    }

    if (_strDescription())
        fprintf(ThisFile, ":\n<BR>%s", _strDescription.c_str());
}

/* ******************************************************************
 *
 *   XDBase
 *
 ********************************************************************/

/*
 *@@ WriteHTML:
 *      creates an HTML file for this declaration.
 *
 *      This is a virtual method, but presently not overridden.
 *
 *      This calls WriteHTMLBody in turn, which in turn usually
 *      calls WriteHTMLDescription.
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

VOID XDBase::CreateHTMLFile()
{
    /*
     * write index for this source file:
     *
     */

    CHAR szThisFile[CCHMAXPATH] = "HTML\\";
    strcat(szThisFile, QueryOutputFilename());

    if (G_ulVerbosity > 1)
        printf("Writing file \"%s\"\n", szThisFile);

    FILE *File = fopen(szThisFile, "w");
    if (!File)
        printf("xdoc: Unable to create output file \"%s\", skipping.", szThisFile);
    else
    {
        string strHTMLTitle(QueryHTMLTitle());
        PSZ p = (PSZ)strHTMLTitle.c_str();
        *p = toupper(*p);
        string strTitle =  strHTMLTitle + " " +_strIndexNameHTML;
        fprintf(File, "\n<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2//EN\">"
                      "\n<html %s>"
                      "\n<head>\n<title>%s\n</title>"
                      "\n<META http-equiv=\"Content-Type\" content=\"text/html; charset=cp850\">"
                      "\n</head>",
                QueryHTMLAttributes(),
                strTitle.c_str());
        fprintf(File, "\n<body>");

        fprintf(File, "\n<h2>%s</h2>",
                strTitle.c_str());

        WriteHTMLBody(File);

        fprintf(File, "\n<p><hr>");
        WriteInfo(File);
        fprintf(File, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");

        fprintf(File, "\n</body>\n</html>");

        fclose(File);
    }
}

/*
 *@@ QueryHTMLAttributes:
 *      this must return the attributes to the HTML
 *      tag which should go into files which are
 *      produced for instances of this class.
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

const char* XDBase::QueryHTMLAttributes()
{
    return ("GROUP=1 NOSUBLINKS=0 XPOS=10% WIDTH=30%");
}

/*
 *@@ WriteHTMLBody:
 *      called by XDBase::CreateHTMLFile to write the
 *      BODY part of the HTML file.
 *
 *      This standard implementation writes the category
 *      and the header prologs from the source file.
 *      However, this is virtual and can be overridden
 *      by subclasses (e.g. XDGlossary has no use for
 *      headers).
 *
 *@@added V0.9.9 (2001-02-10) [umoeller]
 */

VOID XDBase::WriteHTMLBody(FILE *File)
{
    // category
    if (_pMyCategory)
        fprintf(File, "\n<b>Category:</b> %s\n<br>",
                _pMyCategory->_strLinkNameHTMLLong.c_str());

    // file:
    if (_pSourceFile)
        _pSourceFile->WriteHTMLSourcefileProlog(File);

    WriteHTMLDescription(File);
}

/* ******************************************************************
 *
 *   XDCategory
 *
 ********************************************************************/

/*
 *@@ XDCategory:
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

XDCategory::XDCategory(string &strName,             // category name (short)
                       XDCategory *pParent,         // parent category or NULL
                       XDSourceFile *pFirstSourceFile, // source file this ctg resides in;
                                                    // this will also be the introduction
                                                    // if not overridden
                       string &strComment)          // in: entire category comment...
            : XDBase(strName, tXDCategory),
              _ChildCategoriesList(SHADOW),
              _MemberDeclsList(SHADOW)
{
    _cDirectChildCategories = 0;
    _cTotalChildCategories = 0;
    _cDirectMemberDecls = 0;
    _cTotalMemberDecls = 0;

    // store index of this category,
    // starting with 1
    _ulIndex = ++G_cAllCategories;

    // create output filename
    _strOutputFilename._printf("xa%04lX.htm", _ulIndex);

    _pParent = pParent; // can be NULL

    // full name is same as short name;
    // this is modified if we add more
    _strLinkNameHTMLShort._printf("<a href=\"%s\">%s</A>",
                              QueryOutputFilename(),
                              _strIndexNameHTML.c_str());
    _strLinkNameHTMLLong = _strLinkNameHTMLShort;

    _strCommentOriginal = strComment;
    _strCommentNew = FormatComment(strComment);

    if (pParent)
    {
        // parent specified:
        // append to parent's list
        pParent->_ChildCategoriesList.push_back(this);
        pParent->_cDirectChildCategories++;

        // compose full name by climbing up all the parents
        XDCategory *pParentThis = pParent;
        while (pParentThis)
        {
            _strLinkNameHTMLLong = pParentThis->_strLinkNameHTMLShort + " | " + _strLinkNameHTMLLong;
            pParentThis->_cTotalChildCategories++;

            // climb up
            pParentThis = pParentThis->_pParent;
        }
    }
    else
    {
        // no parent:
        // append to global root
        G_RootCategoriesList.push_back(this);
    }

    // in any case, append to global categories list...
    // we need that for resolve later
    G_AllCategoriesFlatList.push_back(this);
}

/*
 *@@ WriteHTMLDescription:
 *      this virtual XDBase function is overridden to
 *      write the body of the category HTML text.
 *
 *@@added V0.9.4 (2000-07-29) [umoeller]
 */

VOID XDCategory::WriteHTMLDescription(FILE* ThisFile)
{
    if (_strCommentNew())
    {
        fprintf(ThisFile, "<b>Introduction:</b>\n<p>%s<p>",
                _strCommentNew.c_str());
    }

    if (_cDirectChildCategories)
    {
        // we have subcategories:
        // sort the list according to category names
        _ChildCategoriesList.sort(CompareCategories);

        fprintf(ThisFile, "<b>Subcategories:</b>\n<ul>");

        // go thru all child categories and add links
        list<PXDCategory>::iterator CtgStart = _ChildCategoriesList.begin(),
                                    CtgEnd = _ChildCategoriesList.end();
        for (;
             CtgStart != CtgEnd;
             ++CtgStart)
        {
            XDCategory *pCtgThis = *CtgStart;

            fprintf(ThisFile, "\n<li><a href=\"%s\">%s%s</A> (%d items)",
                              pCtgThis->QueryOutputFilename(),
                              (pCtgThis->_cDirectChildCategories)
                                 // if that category has child categories in turn,
                                 // add "@"... yahoo-style
                                 ? "@"
                                 : "",
                              pCtgThis->_strIndexNameHTML.c_str(),
                              pCtgThis->_cTotalMemberDecls);
        }

        fprintf(ThisFile, "\n</ul>");
    }

    if (_cDirectMemberDecls)
    {
        list<PXDDeclBase>::iterator DeclStart = _MemberDeclsList.begin(),
                                    DeclEnd = _MemberDeclsList.end();
        for (;
             DeclStart != DeclEnd;
             ++DeclStart)
        {
            XDDeclBase *pDeclThis = *DeclStart;
            fprintf(ThisFile, "\n<a href=\"%s\">%s</A><br>",
                    pDeclThis->QueryOutputFilename(),
                    pDeclThis->_strIndexNameHTML.c_str());    // first real name
        }
    }
}

/* ******************************************************************
 *
 *   XDGlossary implementation
 *
 ********************************************************************/

/*
 *@@ XDGlossary:
 *      constructor.
 *
 *      Glossary entries can be defined with the @@gloss tag.
 *
 *@@added V0.9.9 (2001-02-10) [umoeller]
 */

XDGlossary::XDGlossary(XDSourceFile *pSourceFile_,
                       const string &strDefName_,
                       const string &strReadableName,
                       string &strComment)
            : XDDeclBase(pSourceFile_,
                         // DECL_GLOSSARY,
                         'g',               // filename prefix
                         strDefName_,        // orig decl name
                         strReadableName,       // index name
                         tXDGlossary)
{
    // XDDeclBase has set _strReplacementString = _strIndexName;
    // instead, we use strReadableName for the replacement string
    _strSearchString = "@" + strDefName_;
    _strReplacementString = strReadableName;

    _strCommentOriginal = strComment;
    _strCommentNew = FormatComment(strComment);

    pSourceFile_->_AllDeclsList.push_back(this);
    G_AllDeclsList.push_back(this);
}

/*
 *@@ WriteHTMLDescription:
 *
 *@@added V0.9.9 (2001-02-10) [umoeller]
 */

VOID XDGlossary::WriteHTMLBody(FILE* ThisFile)
{
    fprintf(ThisFile, "\n%s", _strCommentNew.c_str());
}

/* ******************************************************************
 *
 *   XDDeclBase implementation
 *
 ********************************************************************/

/*
 *@@ XDDeclBase:
 *      constructor.
 */

XDDeclBase::XDDeclBase(XDSourceFile *pSourceFile_,
                       CHAR cFilenamePrefix,
                       const string &strOrigDeclName_,
                       const string &strIndexName,
                       BSClassID &Class)
                : XDBase(strIndexName, Class)
{
    _pSourceFile = pSourceFile_;

    // set search string (overridden by subclasses maybe)
    _strSearchString = _strIndexNameOrig;

    _strReplacementString = _strIndexNameHTML;

    // copy category from source file
    if (_pMyCategory = _pSourceFile->_pCurrentCategory)
    {
        // specified:
        // store this declaration in the category
        _pMyCategory->_MemberDeclsList.push_back(this);
        // raise count
        _pMyCategory->_cDirectMemberDecls++;

        // climb up parents to raise the parent counts as well
        XDCategory *pParentThis = _pMyCategory;
        while (pParentThis)
        {
            pParentThis->_cTotalMemberDecls++;
            // climb up
            pParentThis = pParentThis->_pParent;
        }
    }

    // _ulDeclType = ulDeclType_;

    // copy original name
    _strOrigDeclName = strOrigDeclName_;

    _strOutputFilename._printf("x%c%02lX%04lX.htm",
                   cFilenamePrefix,
                   _pSourceFile->_ulFileIndex,
                   G_ulFileCounter++);      // unique file name

    _pAddedLog = NULL;
}

/*
 *@@ GetFirstIdentifier:
 *      returns the first identifier from
 *      _IdentifiersList or "" if there's none.
 *
 *@@added V0.9.1 (2000-02-06) [umoeller]
 */

/* string XDDeclBase::GetFirstIdentifier(VOID)
                   const
{
    if (_IdentifiersList.begin() != _IdentifiersList.end())
        return (*_IdentifiersList.begin());

    return ("");
} */

/*
 *@@ ParseDeclChangeLogs:
 *      called by XDSourceFile::ParseDeclComment to
 *      retrieve the "added" and "changed" tags.
 *
 *      At this point, _strCommentOriginal has been
 *      set to the original (unformatted) comment and
 *      _strCommentNew has the comment as formatted by
 *      FormatComment.
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

ULONG XDDeclBase::ParseDeclChangeLogs()
{
    XDChangeLogEntry *pChangeLog;
    PSZ     pStart = (PSZ)_strCommentNew.c_str(),
            pFound;
    while ((pFound = strstr(pStart, "@@added")))
    {
        pChangeLog = new XDChangeLogEntry(this,
                                          NULL,
                                          _strOrigDeclName,
                                          pFound,
                                          TRUE);      // added
    }
    while ((pFound = strstr(pStart, "@@changed")))
    {
        pChangeLog = new XDChangeLogEntry(this,
                                          NULL,
                                          _strOrigDeclName,
                                          pFound,
                                          FALSE);      // changed
    }

    if (_pAddedLog == NULL)
        // no added log given:
        // use the one from the sourcefile
        if (_pSourceFile)
        {
            _pAddedLog = _pSourceFile->_pAddedLog;
                    // can be NULL also
            if (_pAddedLog)
                // we have one: add to the front of the list
                _ChangeLogsList.push_front(_pAddedLog);
        }

    return (1);
}

/*
 *@@ Resolve:
 *      this virtual XDDeclBase method gets called
 *      by main() on all declarations on G_AllDeclsList.
 *
 *      This method is supposed to go thru a
 *      declaration's description (in its comment)
 *      and replace all occurences of _other_ declarations
 *      with HTML links to that declaration.
 *
 *      This is overridden by subclasses which need to
 *      resolve more stuff. Presently, only XDClass
 *      overrides this to resolve declarations in the
 *      class definition as well.
 */

VOID XDDeclBase::Resolve()
{
    printf(".");
    fflush(stdout);
    ResolveRefs(_strCommentNew,
                &_strIndexNameOrig);
}

/*
 *@@ QueryHTMLAttributes:
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

const char* XDDeclBase::QueryHTMLAttributes()
{
    return ("GROUP=2 NOSUBLINKS=0 NOSUBLINKS=ctg XPOS=30% WIDTH=70%");
}

/*
 *@@ WriteHTML:
 *
 */

VOID XDDeclBase::WriteHTMLDescription(FILE* ThisFile)
{
    if (_strCommentNew())
        fprintf(ThisFile, "\n<p><b>Description:</b>\n<p>%s",
                _strCommentNew.c_str());

    // sort by version/date (XDChangeLogEntry < operator);
    // do this before using the iterators, because these
    // will become invalid
    _ChangeLogsList.sort(CompareChangeLogEntries);

    // now dump all log entries
    list<PXDChangeLogEntry>::iterator LogStart = _ChangeLogsList.begin(),
                                      LogEnd = _ChangeLogsList.end();

    if (LogStart != LogEnd)
    {
        // we have log entries:

        fprintf(ThisFile, "\n<p><b>History:</b>\n<ul>");

        for (;
             LogStart != LogEnd;
             ++LogStart)
        {
            XDChangeLogEntry *pLogThis = *LogStart;

            // write version
            fprintf(ThisFile,
                    "\n<li><b>V%d.%d.%d</b>",
                    pLogThis->_version._ulMajor,
                    pLogThis->_version._ulMinor,
                    pLogThis->_version._ulRevision);
            // write date
            if (pLogThis->_date._ulYear)
                fprintf(ThisFile,
                        " (%04d-%02d-%02d)",
                        pLogThis->_date._ulYear,
                        pLogThis->_date._ulMonth,
                        pLogThis->_date._ulDay);

            if (pLogThis->_fAdded)
            {
                // "added" log entries might be copied from the
                // source file, so check
                if (pLogThis->_pMyDecl == this)
                    fprintf(ThisFile, ": added");
                else if (pLogThis->_pMySourceFile)
                    fprintf(ThisFile, ": file \"%s\" added",
                            pLogThis->_pMySourceFile->_strIndexNameHTML.c_str());
                else
                    // shouldn't happen
                    continue;
            }
            else
                fprintf(ThisFile, ": changed");
            if (pLogThis->_strAuthor())
                fprintf(ThisFile, " by <I>%s</I>", pLogThis->_strAuthor.c_str());

            if (pLogThis->_strDescription())
                fprintf(ThisFile, ":\n<br>%s", pLogThis->_strDescription.c_str());
        }

        fprintf(ThisFile, "\n</ul>");
    }
}

/* ******************************************************************
 *
 *   XDTypedef implementation
 *
 ********************************************************************/

XDTypedef::XDTypedef(XDSourceFile *pSourceFile,
                     const string &strDefName,
                     XDDeclBase *pLinkedTo)
            : XDDeclBase(pSourceFile,
                         'y',               // filename prefix
                         strDefName,  // orig decl name
                         strDefName,     // index name
                         tXDTypedef)
{
    _pLinkedTo = pLinkedTo;

    pSourceFile->_AllDeclsList.push_back(this);
    G_AllDeclsList.push_back(this);
}

/* ******************************************************************
 *
 *   XDDefine implementation
 *
 ********************************************************************/

/*
 *@@ XDDefine:
 *
 *@@added V0.9.1 (2000-01-26) [umoeller]
 */

XDDefine::XDDefine(XDSourceFile *pSourceFile_,
                   const string &strDefName_,
                   const string &strDefDefinition)      // can be empty, e.g. with "case"
            : XDIndexableBase(pSourceFile_,
                              // DECL_DEFINE,
                              'd',               // filename prefix
                              strDefName_,       // orig decl name
                              strDefName_,       // index name
                              tXDDefine)
{
    // store in global lists
    pSourceFile_->_DefinesList.push_back(this);
    G_AllDefinesList.push_back(this);
    pSourceFile_->_AllDeclsList.push_back(this);
    G_AllDeclsList.push_back(this);
}

/*
 *@@ WriteHTML:
 *
 *@@added V0.9.1 (2000-02-06) [umoeller]
 */

VOID XDDefine::WriteHTMLDescription(FILE* ThisFile)
{
    // write stuff before parameters
    fprintf(ThisFile, "\n\n<p><pre>%s</pre>",
            _strOrigDeclName.c_str());

    XDDeclBase::WriteHTMLDescription(ThisFile);
}

/* ******************************************************************
 *
 *   XDFunction implementation
 *
 ********************************************************************/

/*
 *@@ ParseFunction:
 *      this gets called from ParseDeclComment with
 *      a declaration block to check whether this
 *      is a function.
 *
 *      If so, we return a newly created XDFunction
 *      instance.
 *
 *      If not, we return NULL.
 *
 *@@changed V0.9.18 (2002-03-16) [umoeller]: largely rewritten to fix argument parsing
 */

XDFunction* XDFunction::CreateFunction(XDSourceFile *pSourceFile,
                                       XDClass *pOwningClass,    // in: class of which this is a member or NULL
                                       string &strComment,       // in: current comment block
                                       string &strDeclBlock)     // in: decl header block up to opening bracket
{
    XDFunction *pFunction = 0;

    if (G_ulVerbosity >= 4)
    {
        printf("\n  Entering " __FUNCTION__);
        fflush(stdout);
    }

    // find parameter list
    ULONG   ulOpeningBracket = strDeclBlock.find('(');
    if (    (ulOpeningBracket == 0)
         || (ulOpeningBracket == string::npos)
       )
    {
        // no parameters: this comment is not for a function
        if (G_ulVerbosity >= 4)
        {
            printf("\n    \"%s\" has no params; leaving ParseFunction",
                    strDeclBlock.c_str());
            fflush(stdout);
        }
        return (0);
    }

    // get all the stuff before "("
    // by creating a substring
    string strBeforeBracket(strDeclBlock, 0, ulOpeningBracket);
    strBeforeBracket._format();

    if (G_ulVerbosity >= 4)
    {
        printf("\n    strBeforeBracket: \"%s\"", strBeforeBracket.c_str());
        fflush(stdout);
    }

    do      // for allowing breaks
    {
        if (    (strBeforeBracket.find(';') != string::npos)
             || (strBeforeBracket.find('#') != string::npos)
           )
        {
            printf("\n  --> Warning: cannot handle \"%s\", skipping", strBeforeBracket.c_str());
            break;
        }

        string  strOrigDeclName,
                strBeforeFunctionName;
        // find last space before "("
        ULONG   ulLastSpaceBeforeBracket = strBeforeBracket.rfind(' ');
        if (ulLastSpaceBeforeBracket != string::npos)
        {
            // space found:
            // create two substrings from the stuff
            // before and after the space
            strOrigDeclName = strBeforeBracket.substr(ulLastSpaceBeforeBracket + 1);
            strBeforeFunctionName = strBeforeBracket.substr(0,
                                                            ulLastSpaceBeforeBracket);

            if (    (strBeforeFunctionName.find("//") != string::npos)
                 || (strBeforeFunctionName.find("/*") != string::npos)
               )
            {
                // we cannot handle comments for return values yet
                printf("\n  --> Warning: cannot handle \"%s\", skipping", strBeforeBracket.c_str());
                break;
            }
        }
        else
        {
            // space not found:
            // that's a constructor probably, this has no return value
            strOrigDeclName = strBeforeBracket;
        }

        if (!strOrigDeclName())
        {
            // not found: get outta here
            printf("\n  --> Warning: cannot handle \"%s\", skipping", strBeforeBracket.c_str());
            break;
        }

        if (G_ulVerbosity)
        {
            if (G_ulVerbosity >= 3)
                printf("\n  Found function: \"%s\"", strOrigDeclName.c_str());
            else
            {
                printf(".");
            }
            fflush(stdout);
        }

        string    strIndexName(strOrigDeclName);

        if (pOwningClass)
        {
            // we're inside a "class" definition:
            // (i.e. comment in a "class { ... }" header):
            // prefix the class name
            strIndexName = pOwningClass->_strIndexNameOrig + "::" + strOrigDeclName;
        }
        else
        {
            // no owning class (i.e. we're not in a "class" block:

            // now check function name for whether
            // it has a prefix which was declared
            // using @@somclass
            list<PXDSomClass>::iterator ClassStart = pSourceFile->_SomClassesList.begin(),
                                        ClassEnd = pSourceFile->_SomClassesList.end();
            for (;
                 ClassStart != ClassEnd;
                 ++ClassStart)
            {
                PXDSomClass pClassThis = *ClassStart;
                // compare the first characters of the
                // function name with the SOM prefix
                ULONG cbFuncPrefix = pClassThis->_strSomFuncPrefix.size();
                if (strOrigDeclName.compare(0,                  // from start
                                            cbFuncPrefix,
                                            pClassThis->_strSomFuncPrefix)
                        == 0)
                {
                    // @@somclass prefix found:

                    // extract short name
                    strIndexName =  pClassThis->_strIndexNameOrig
                                    + "::"
                                    // append function name without prefix
                                    // e.g. "wpInitData"
                                    + strOrigDeclName.substr(cbFuncPrefix); // index

                    pOwningClass = pClassThis;
                }
            }

            if (G_ulVerbosity >= 4)
            {
                printf("\n    Checked @@somclass");
                fflush(stdout);
            }
        }

        /*
         * Get parameters:
         *
         */

        ULONG ulEndOfArgs = FindNotInCppComment(strDeclBlock,
                                                ")",
                                                ulOpeningBracket + 1);

        if (ulEndOfArgs == string::npos)
        {
            printf("\nError in func \"%s\": Cannot find end of parameters. Skipping.\n",
                       strOrigDeclName.c_str());
            break;
        }

        *(PSZ)(strDeclBlock.c_str() + ulEndOfArgs) = ' ';

        PCSZ p = strDeclBlock.c_str() + ulEndOfArgs;
        CHAR c;
        while ((p) && (c = *p))
        {
            if (    (c == ' ')
                 || (c == '\n')
                 || (c == '\t')
               )
                ++p;
            else
                if (    (c == '/')
                     && (p[1] == '/')
                   )
                    p = strchr(p, '\n');
            else
            {
                // any other character: this is either a C++ const keyword
                // or a C++ initializer list, so stop
                ulEndOfArgs = p - strDeclBlock.c_str();
                break;
            }
        }

        // extract parameters (temporary): all the stuff
        // after the opening bracket
        string strParams(strDeclBlock, ulOpeningBracket + 1, ulEndOfArgs - (ulOpeningBracket + 1));

        list<XDArgument*> ArgList(SHADOW);

        if (G_ulVerbosity >= 4)
        { printf("\n    Checking params \"%s\"...", strParams.c_str()); fflush(stdout); }

        BOOL fLastParm_Exit = FALSE;

        // current search position
        ULONG ulStartOfArg = 0;

        ULONG ulArgNoThis = 0;

        BOOL  fHasNonVoidArgs = FALSE;

        string strMangledArgs = "(";

        do {

            /*
             * find end of parameter:
             *
             */

            string      strThisArg,
                        strThisArgComment;

            BOOL        fCommentCont = TRUE;

            ULONG       ulArgLen,
                        ulNextArg;

            ULONG ulNextComma = FindNotInCppComment(strParams,
                                                    ",",
                                                    ulStartOfArg);
                    // closing bracket has been overwritten with space
                    // above, so if this returns npos, we have the last arg

            ULONG ulNextComment = strParams.find("//", ulStartOfArg);
            ULONG ulNextLine = strParams.find('\n', ulStartOfArg);

            // now we have:
            // -- ulStartOfArg: pointing to
            //      "XDClass *pOwningClass,    // in: class of which this is a member"
            // -- ulNextComma: pointing to
            //      ",    // in: class of which this is a member"

            if (ulNextComma == string::npos)
            {
                // no comma -> last argument:

                if (ulNextComment != string::npos)
                {
                    ulArgLen = ulNextComment - ulStartOfArg;
                    strThisArgComment.assign(strParams,
                                             ulNextComment + 2,
                                             ulNextLine - ulNextComment - 2);
                }
                else
                    // no comment for last arg:
                    ulArgLen = string::npos;

                fLastParm_Exit = TRUE;
            }
            else
            {
                // another comma coming:
                ulArgLen = ulNextComma - ulStartOfArg;

                if (    (ulNextComment != string::npos)
                     && (ulNextComment < ulNextLine)
                   )
                    strThisArgComment.assign(strParams,
                                             ulNextComment + 2,
                                             ulNextLine - ulNextComment - 2);
            }

            strThisArg.assign(strParams, ulStartOfArg, ulArgLen);
            strThisArg._format();

            ulNextArg = ulNextLine + 1;

            if (strThisArgComment())
            {
                // now check if the comment is continued on the
                // next line
                BOOL fStop = FALSE;
                while (!fStop)
                {
                    ulNextComment = strParams.find("//", ulNextArg);
                    ulNextLine = strParams.find('\n', ulNextArg);

                    if (ulNextComment != string::npos)
                    {
                        ulNextComma = FindNotInCppComment(strParams,
                                                          ",",
                                                          ulNextArg);
                        string strCont;

                        if (ulNextComma == string::npos)
                        {
                            if (fLastParm_Exit)
                            {
                                strCont.assign(strParams,
                                               ulNextComment + 2);
                                fStop = TRUE;
                            }
                            else
                                break;
                        }
                        else if (ulNextComment < ulNextComma)
                        {
                            if (ulNextLine != string::npos)
                            {
                                strCont.assign(strParams,
                                               ulNextComment + 2,
                                               ulNextLine - ulNextComment - 2);
                                ulNextArg = ulNextLine + 1;
                            }
                            else
                            {
                                strCont.assign(strParams,
                                               ulNextComment + 2);
                                fStop = TRUE;
                            }
                        }
                        else
                            break;

                        strThisArgComment += strCont;
                    }
                    else
                        break;
                }
            }

            if (strThisArg())
            {
                // create ARGUMENT structure
                XDArgument* pArg = new XDArgument(strThisArg,
                                                  strThisArgComment,
                                                  &fHasNonVoidArgs);
                ArgList.push_back(pArg);

                if (fHasNonVoidArgs)
                {
                    if (ulArgNoThis)
                        strMangledArgs += ", ";

                    strMangledArgs += pArg->_strType;

                    ++ulArgNoThis;
                }
            }

            ulStartOfArg = ulNextArg + 1;

        } while (!fLastParm_Exit);

        strMangledArgs += ')';

        strIndexName += strMangledArgs;

        // create function structure
        pFunction = new XDFunction(pSourceFile,
                                   strOrigDeclName,
                                   strIndexName,
                                   strBeforeFunctionName,
                                   strMangledArgs,
                                   ArgList);

        if (G_ulVerbosity >= 4)
        { printf("\n    Done with params."); fflush(stdout); }

        if (G_ulVerbosity >= 3) {
            printf("OK"); fflush(stdout); }

    } while (FALSE);        // end do

    if (G_ulVerbosity >= 3)
    { printf(" FuncDone"); fflush(stdout); }

    if (G_ulVerbosity >= 4)
    {  printf("\n  Leaving " __FUNCTION__); fflush(stdout); }

    return (pFunction);
}

/*
 *@@ XDFunction:
 *      the constructor.
 *
 *      Functions are a bit complex... since they can be member
 *      functions of a C++ class.
 *
 *      stOrigDeclName is the function name as it appears in the
 *      source file. By contrast, strIndexName is beautified
 *      already:
 *
 *      -- For plain C functions, this is the same as strOrigDeclname
 *         (i.e. the function name only).
 *
 *      -- For C++ member functions in the sources, this has the full
 *         name including the class (e.g. "XDFunction::XDFunction").
 *
 *      -- For C++ member functions in a header class definition,
 *         this also has the full class name; this is hacked this
 *         way by ParseFunction.
 *
 *      -- For SOM methods, this also has the class name; ParseFunction
 *         finds the SOM class for us.
 *
 *      In some cases (when functions are specified as SOM member
 *      functions with the @@somclass prefixes or when they appear
 *      in a class header definition), pOwningClass is already known,
 *      and we can set the member to that directly. For all other
 *      member functions, we have to delay that until later when
 *      we have also parsed all the classes.
 *
 *@@changed V0.9.18 (2002-03-16) [umoeller]: fixed "virtual"
 */

XDFunction::XDFunction(XDSourceFile *pSourceFile_,
                       const string &strOrigDeclName_,      // in: declaration name (with or without class name)
                       const string &strIndexName_,         // in: index name (always with class name, plus mangled args)
                       const string &strBeforeFunctionName, // in: all the stuff before "("
                       const string &strMangledArgs,        // in: mangled arguments (e.g. "(const char *, const char*)")
                       list<XDArgument*> &ArgList)          // in: arguments list (SHADOW mode)
            : XDIndexableBase(pSourceFile_,
                              'f',               // filename prefix
                              strOrigDeclName_,  // orig decl name
                              strIndexName_,     // index name
                              tXDFunction),
              _ArgList(STORE),
              _strMangledArgs(strMangledArgs),
              _OverridesList(SHADOW),
              _OverriddenByList(SHADOW)
{
    _pOwningClass = NULL;
                        // can be NULL even though this is a member function
                        // main() resolves the missing ones later

    // for very small search strings,
    // V0.9.6 (2000-11-12) [umoeller]: and if this is for a function,
    // we require an extra () for the search
                // made this a member variable V0.9.9 (2001-02-10) [umoeller]
                // and moved this here
    if (_strSearchString.size() < 6)
    {
        _strSearchString += "()";
    }

    // _IdentifiersList.push_back(strIndexName_);

    _strBeforeFunctionName = strBeforeFunctionName;
    _strFunctionReturn = strBeforeFunctionName;

    _fIsStaticFunction = FALSE;
    _fIsVirtualFunction = FALSE;

    // do we have a class prefix?
    ULONG ul = _strIndexNameOrig.find("::");
    if (ul != string::npos)
    {
        // yes: copy that as the class name...
        _strOwningClass.assign(_strIndexNameOrig,
                               0,
                               ul);
        // and the rest as the pure name
        _strPureName.assign(_strIndexNameOrig,
                            ul + 2);
    }
    else
        _strPureName == _strIndexNameOrig;

    // get rid of linkage specifiers etc.
    // and format return type
    if (strBeforeFunctionName())
    {
        ULONG ulOfs = 0;
        _strFunctionReturn._find_replace("SOM_Scope", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("SOMLINK", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("EXPENTRY", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("extern", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("inline", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("_System", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("_Optlink", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("_Pascal", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("_Far32", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("_Far16", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._find_replace("_Export", "", &ulOfs);
        ulOfs = 0;
        _strFunctionReturn._format();

        // get rid of "static", but set flag if present
        ulOfs = 0;
        if (_strFunctionReturn._find_replace("static", "", &ulOfs)
                    != string::npos)
            _fIsStaticFunction = TRUE;

        // get rid of "virtual", but set flag if present
        // V0.9.18 (2002-03-16) [umoeller]
        ulOfs = 0;
        if (_strFunctionReturn._find_replace("virtual", "", &ulOfs)
                    != string::npos)
            _fIsVirtualFunction = TRUE;
    }

    // copy args list V0.9.18 (2002-03-16) [umoeller]
    _ArgList = ArgList;

    // store in global lists
    pSourceFile_->_FunctionsList.push_back(this);
    G_AllFunctionsList.push_back(this);

    pSourceFile_->_AllDeclsList.push_back(this);
    G_AllDeclsList.push_back(this);
};

/*
 *@@ DumpList:
 *
 *@@added V0.9.18 (2002-03-16) [umoeller]
 */

VOID XDFunction::DumpList(FILE* ThisFile,
                          PCSZ pcszTitle,
                          list<PXDFunction> &List)
{
    if (!List.empty())
    {
        fprintf(ThisFile, "\n<p><b>%s</b><ul>\n", pcszTitle);
        list<PXDFunction>::iterator lBegin = List.begin(),
                                    lEnd = List.end();
        for (;
             lBegin != lEnd;
             ++lBegin)
        {
            XDFunction *pThis = *lBegin;

            fprintf(ThisFile, "<li><a href=\"%s\">%s</A>",
                pThis->QueryOutputFilename(),
                pThis->_strIndexNameHTML.c_str()); // short name
        }
        fprintf(ThisFile, "\n</ul>");
    }
}

/*
 *@@ WriteHTML:
 *
 */

VOID XDFunction::WriteHTMLDescription(FILE* ThisFile)
{
    // write stuff before parameters
    fprintf(ThisFile, "\n\n<pre>");

    if (_strBeforeFunctionName())
        fprintf(ThisFile, "%s ",
                _strBeforeFunctionName.c_str()); // APIRET SOMLINK etc.

    fprintf(ThisFile, "%s(",
                      _strOrigDeclName.c_str());        // fnwpMain(

    ULONG   ulIndent = _strBeforeFunctionName.size()
                       + _strOrigDeclName.size()
                       + 2;     // space plus opening bracket

    list<PXDArgument>::iterator ArgStart = _ArgList.begin(),
                                ArgEnd = _ArgList.end();
    ULONG ulTotal = 0, ulCount = 0;
    for (;
         ArgStart != ArgEnd;
         ++ArgStart)
        ++ulTotal;

    // add parameters
    for (ArgStart = _ArgList.begin();
         ArgStart != ArgEnd;
         ++ArgStart)
    {
        XDArgument *pArgThis = *ArgStart;
        fprintf(ThisFile, "%s %s",
                pArgThis->_strType.c_str(),
                pArgThis->_strIdentifier.c_str());

        ++ulCount;
        if (ulCount < ulTotal)
        {
            // another parameter to come: indent
            fprintf(ThisFile, ",\n");
            CHAR szSpaces[1000];
            ULONG  ul2;
            for (ul2 = 0; ul2 < ulIndent; ++ul2)
                szSpaces[ul2] = ' ';
            szSpaces[ul2] = 0;
            fprintf(ThisFile, szSpaces);
        }
    }
    fprintf(ThisFile, ")</pre>\n");

    /*
     * parameter list:
     *
     */

    if (!_ArgList.empty())
    {
        fprintf(ThisFile, "\n<p><b>Parameters:</b><dl>\n");
        for (ArgStart = _ArgList.begin();
             ArgStart != ArgEnd;
             ++ArgStart)
        {
            XDArgument *pArgThis = *ArgStart;

            fprintf(ThisFile, "<dt><CODE>%s %s</CODE>\n",
                    pArgThis->_strType.c_str(),
                    pArgThis->_strIdentifier.c_str());
            if (pArgThis->_strArgComment())
                fprintf(ThisFile, "<dd>%s\n",
                        pArgThis->_strArgComment.c_str());
        }
        fprintf(ThisFile, "\n</dl></dl>");
    }

    if (_pOwningClass)
    {
        fprintf(ThisFile, "\n<p><B>Member of:</B> <a href=\"%s\">%s</A>",
                _pOwningClass->QueryOutputFilename(),
                _pOwningClass->_strIndexNameHTML.c_str()); // short name
        DumpList(ThisFile, "Overrides:", _OverridesList);
        DumpList(ThisFile, "Overridden by:", _OverriddenByList);
    }

    if (_strFunctionReturn())
        fprintf(ThisFile, "\n<p><b>Returns:</b> <CODE>%s</CODE>\n",
                _strFunctionReturn.c_str());

    XDDeclBase::WriteHTMLDescription(ThisFile);
}

/* ******************************************************************
 *
 *   XDClass implementation
 *
 ********************************************************************/

/*
 *@@ XDClass:
 *      constructor.
 *
 *@@added V0.9.7 (2001-01-07) [umoeller]
 */

XDClass::XDClass(XDSourceFile *pSourceFile_,
                 const string &strOrigDeclName_,
                 const string &strNameForResolve,
                 const string &strDefinition,   // full definition.. must be formatted with FormatPRE!
                 BSClassID &Class)
            : XDIndexableBase(pSourceFile_,
                              'c',               // filename prefix
                              strOrigDeclName_,  // orig decl name
                              strOrigDeclName_,  // index name
                              Class),
              _llParentNames(SHADOW),
              _DescendantsList(SHADOW),
              _MemberFunctionsList(SHADOW)
{
    // _IdentifiersList.push_back(strOrigDeclName_);

    _strNameForResolve = strNameForResolve;
    _strDefinition = strDefinition;

    // store class
    pSourceFile_->_ClassesList.push_back(this);
    G_AllClassesList.push_back(this);
            // we can't resolve the root classes until we
            // have created all classes... so defer this

    // pSourceFile_->_AllDeclsList.push_back(this);
    // G_AllDeclsList.push_back(this);
            // no, defer this until later; we want the class names
            // to appear AFTER class members on the global lists
            // so that ResolveRefs replaces them correctly

    _pParent = NULL;
    _cParentNames = 0;
    _cDescendants = 0;
    _cMemberFunctions = 0;
            // all resolved later by main()
};

/*
 *@@ Resolve:
 *      this virtual XDDeclBase method gets called
 *      by main() on all declarations on G_AllDeclsList.
 *
 *      This method is supposed to go thru a
 *      declaration's description (in its comment)
 *      and replace all occurences of _other_ declarations
 *      with HTML links to that declaration.
 */

VOID XDClass::Resolve()
{
    printf("c");
    ResolveRefs(_strDefinition,
                &_strIndexNameOrig);

    // call parent, which resolves the function comment
    XDDeclBase::Resolve();
}

/*
 *@@ WriteHTML:
 *
 */

VOID XDClass::WriteHTMLDescription(FILE* ThisFile)
{
    if ((_pParent) || (_cDescendants))
    {
        // we have inheritance of some sort:
        fprintf(ThisFile, "<p><b>Class hierarchy:</b>\n");
        DumpRootAndDescendants(ThisFile,
                               this);
    }
    else
        fprintf(ThisFile, "\n<p>");

    if (_cMemberFunctions)
    {
        fprintf(ThisFile, "<b>Members:</b>");

        list<PXDFunction>::iterator FuncStart = _MemberFunctionsList.begin(),
                                    FuncEnd = _MemberFunctionsList.end();
        for (;
             FuncStart != FuncEnd;
             FuncStart++)
        {
            PXDFunction FuncThis = *FuncStart;
            fprintf(ThisFile, "\n<br><a href=\"%s\">%s</A>",
                    FuncThis->QueryOutputFilename(),
                    FuncThis->_strIndexNameHTML.c_str()); // short name
        }
    }
    else
        // fprintf(ThisFile, "<b>No members.</b>");
        fprintf(ThisFile, "<b>Definition:</b>\n<pre>%s</pre>",
                _strDefinition.c_str() );

    /* fprintf(ThisFile, "<b>Definition:</b>\n<pre>%s</pre>",
            _strDefinition.c_str() ); */

    XDDeclBase::WriteHTMLDescription(ThisFile);
}

/* ******************************************************************
 *
 *   XDSourceFile implementation
 *
 ********************************************************************/

/*
 *@@ XDSourceFile:
 *      default constructor.
 */

XDSourceFile::XDSourceFile(const string &strShortFilename,
                           const string &strFilePath,
                           ULONG ulFileIndex_,
                           PXDSourceFile pFileForWhichThisIsHeader)
            : XDBase(strShortFilename, tXDSourceFile),
              _AllDeclsList(SHADOW),
              _DefinesList(SHADOW),
              _FunctionsList(SHADOW),
              _ClassesList(SHADOW),
              _SomClassesList(SHADOW),
              _ChangeLogsList(SHADOW),
              _ReviewsList(SHADOW)
{
    _ulFileIndex = ulFileIndex_;
    _pCurrentCategory = 0;

    _ulCommentCount = 0;
    _ulTagsCount = 0;

    // copy and create source filenames
    // _strIndexName.c_str(), strShortFilename.c_str());
    _strFilePath = strFilePath;
    _strLongFilename = strFilePath;
    _strLongFilename += strShortFilename;

    // create HTML output filename
    /* CHAR    szIndexFilestem[CCHMAXPATH];
    string strTmp(_strIndexName);
    _splitpath((PSZ)strTmp.c_str(), 0, 0, szIndexFilestem, 0); */
    _strOutputFilename._printf("xi%03lX.htm",
            ulFileIndex_);

    // check header
    _pMyHeader = NULL;
    if (pFileForWhichThisIsHeader)
    {
        // this is a header file for another source:
        _fIsHeader = TRUE;
        // set that file's header pointer to ourselves
        pFileForWhichThisIsHeader->_pMyHeader = this;
    }
    else
        _fIsHeader = FALSE;

    _ulFunctionCount = 0;
    _ulReviewCount = 0;

    _pAddedLog = NULL;
}

/*
 *@@ ParseDefine:
 *      this gets called from ParseDeclComment with
 *      a declaration block to check whether this
 *      is a simple define.
 *
 *      A block is considered a "simple define" if
 *      it starts with "#define" or "case".
 *
 *      If so, we return a newly created XDDefine
 *      instance.
 *
 *      If not, we return NULL.
 *
 *@@added V0.9.1 (2000-01-26) [umoeller]
 */

XDDefine* XDSourceFile::ParseDefine(string &strComment,       // in: current comment block
                                    PCSZ pAfterComment)   // in: pointer to end of comment in the source file
{
    XDDefine *pDefine = 0;

    PCSZ pSearch = pAfterComment;

    // skip spaces
    while (     (*pSearch)
             && (   (*pSearch == ' ')
                 || (*pSearch == '\n')
                 || (*pSearch == '\r')
                )
          )
        pSearch++;

    // now we got pSearch pointing to the first non-space
    // character after the comment; check out what this
    // is

    // check "case"
    if (memcmp(pSearch, "case", 4) == 0)
    {
        // skip "case"
        pSearch += 4;
        // skip spaces
        while (     (*pSearch)
                 && (   (*pSearch == ' ')
                     || (*pSearch == '\n')
                     || (*pSearch == '\r')
                    )
              )
            pSearch++;

        // now we got pSearch pointing to the first word after "case";
        // find colon
        PSZ pColon = strchr(pSearch, ':');
        if (pColon)
        {
            // extract that
            string strDefine(pSearch, pColon);
            strDefine._format();

            if (G_ulVerbosity >= 3)
                printf("Found case: %s\n", strDefine.c_str());
            else
            {
                printf(".");
                fflush(stdout);
            }

            pDefine = new XDDefine(this,
                                   strDefine,
                                   "case statement");
        }
    }
    else if (memcmp(pSearch, "#define", 7) == 0)
    {
        string strDefine;

        // skip "define"
        pSearch += 7;
        // skip spaces
        while (     (*pSearch)
                 && (   (*pSearch == ' ')
                     || (*pSearch == '\n')
                     || (*pSearch == '\r')
                    )
              )
            pSearch++;

        // now we got pSearch pointing to the first word after "case";
        // find end of line
        PSZ pEOL = strhFindEOL(pSearch, NULL);
        if (pEOL)
        {
            // extract that line (e.g. WM_BLAH 10000)
            string strDefBlock(pSearch, pEOL);

            ULONG ulOpening = strDefBlock.find('(');
            ULONG ulNextSpace = strDefBlock.find(' ');
            if (    (ulOpening != string::npos)
                 && (   (ulNextSpace == string::npos)
                     || (ulNextSpace > ulOpening)
                    )
               )
            {
                // opening bracket found before space:
                // find closing too
                ULONG ulClosing = strDefBlock.find(')');
                if (ulClosing != string::npos)
                {
                    // OK, we got what we need:
                    // extract that
                    strDefine = string(pSearch, pSearch + ulOpening);
                }
            }
            else if (ulNextSpace != string::npos)
            {
                // no opening bracket before space:
                strDefine = string(pSearch, pSearch + ulNextSpace);
            }

            if (strDefine())
            {
                if (G_ulVerbosity >= 3)
                    printf("Found #define: %s\n", strDefine.c_str());
                else
                {
                    printf(".");
                    fflush(stdout);
                }

                strDefine._format();
                pDefine = new XDDefine(this,
                                       strDefine,
                                       "#define statement");
            }
        }
    }

    return (pDefine);
}

/*
 *@@ ParseClass:
 *      this gets called from ParseDeclComment for a
 *      declaration block if ParseFunction failed.
 *
 *      We then check whether we have a "struct" or
 *      "class" declaration.
 *
 *@@changed V0.9.18 (2002-02-13) [umoeller]: fixed hang on resolving SOM classes
 *@@changed V0.9.18 (2002-03-16) [umoeller]: fixed templates
 */

XDClass* XDSourceFile::ParseClass(string &strComment,
                                          // in: current comment block (null-terminated)
                                  string &strDeclBlock,
                                          // in: decl header block up to opening bracket
                                  PCSZ *ppDeclBlockInFile)
                                          // in: pointer to start of decl in source file;
                                          // needed for extraction
                                          // out: where to continue search
{
    XDClass *pClassNew = 0;
    if (G_ulVerbosity >= 4)
    {
        printf("\n  Entering ParseClass");
        fflush(stdout);
    }

    do
    {
        ULONG   ulStructBegin = strDeclBlock.find("struct "),
                ulStructKeywordLen = 0;
        if (ulStructBegin != string::npos)
            ulStructKeywordLen = 7;
        else
        {
            // "struct" not found: try "class"
            ulStructBegin = strDeclBlock.find("class ");
            if (ulStructBegin != string::npos)
                ulStructKeywordLen = 6;
        }

        if (ulStructKeywordLen)
        {
            // we have either "class" or "struct":
            ULONG   ulOpeningBracket = strDeclBlock.find('{',
                                                         ulStructBegin + ulStructKeywordLen);
            if (ulOpeningBracket == string::npos)
                // empty declaration:
                break;

            // check if we have a leading "template"
            // V0.9.18 (2002-03-16) [umoeller]
            string strBeforeDefinition(strDeclBlock, 0, ulOpeningBracket);
            string strTemplate;
            ULONG ulTemplate = strBeforeDefinition.find("template");
            if (ulTemplate != string::npos)
            {
                // yes: extract this because there will probably
                // be something like "template <class P>", and
                // we'll choke below
                ulStructKeywordLen = 0;
                ULONG ulStart = strBeforeDefinition.find('<'),
                      ulEnd = strBeforeDefinition.find('>');
                if (    (ulStart != string::npos)
                     && (ulEnd != string::npos)
                   )
                {
                    strTemplate.assign(strBeforeDefinition, ulStart + 1, ulEnd - ulStart - 1);

                    // reassign
                    ulStructBegin = strDeclBlock.find("struct ", ulEnd);
                    if (ulStructBegin != string::npos)
                        ulStructKeywordLen = 7;
                    else
                    {
                        // "struct" not found: try "class"
                        ulStructBegin = strDeclBlock.find("class ", ulEnd);
                        if (ulStructBegin != string::npos)
                            ulStructKeywordLen = 6;
                    }
                }

                if (!ulStructKeywordLen)
                {
                    printf("\n:Cannot handle template \"%s\"", strBeforeDefinition.c_str());
                    break;
                }

                if (G_ulVerbosity >= 2)
                    printf("\nfound template \"%s\"", strTemplate.c_str());
            }

            // find first character after "class" or "struct" and spaces;
            // this must be the class name
            ULONG   ulClassName = strDeclBlock.find_first_not_of(' ',
                                                                 ulStructBegin + ulStructKeywordLen);
            if (ulClassName > ulOpeningBracket)
            {
                printf("\n  warning: cannot handle class/struct \"%s\"",
                       strDeclBlock.c_str());
                break;
            }

            // find first space after start of the class name;
            // after the class name, we can still have stuff like " : baseclass"
            ULONG   ulSpaceAfterClassName = strDeclBlock.find(' ',
                                                              ulClassName);
            ULONG   ulNameLen = 0;

            if (    (ulSpaceAfterClassName == string::npos)
                 || (ulSpaceAfterClassName > ulOpeningBracket)
               )
                ulNameLen = (ulOpeningBracket - ulClassName);
            else
                ulNameLen = (ulSpaceAfterClassName - ulClassName);

            // extract class name
            string strClassName(strDeclBlock,       // from
                                ulClassName,        // start
                                ulNameLen);         // count
            if (!strClassName())
            {
                printf("\n  warning: cannot handle class/struct \"%s\"",
                       strDeclBlock.c_str());
                break;
            }

            strClassName._format();

            string strNameForResolve = strClassName;
                        // without template in any case
                        // V0.9.18 (2002-03-16) [umoeller]

            if (strTemplate())
            {
                strClassName.append("<");
                strClassName.append(strTemplate);
                strClassName.append(">");
            }
            if (G_ulVerbosity)
            {
                if (G_ulVerbosity >= 3)
                    printf("\n  Found struct/class: \"%s\"", strClassName.c_str());
                else
                {
                    printf(".");
                    fflush(stdout);
                }
            }

            // check if there's a ":" after the class name...
            // i.e. the class is inherited from something
            list<string*> ParentNamesList(SHADOW);

            ULONG ulColonAfterClassName = strDeclBlock.find(':',
                                                            ulClassName + ulNameLen);
            if (    (ulColonAfterClassName != string::npos)
                 && (ulColonAfterClassName < ulOpeningBracket)
               )
            {
                ULONG ulStartOfParentsList = ulColonAfterClassName + 1;
                // alright, we have inheritance:
                // skip spaces after colon
                while (strDeclBlock[ulStartOfParentsList] == ' ')
                    ulStartOfParentsList++;

                ULONG ulEndOfParentsList = strDeclBlock.find_first_of("\n{",
                                                                      ulStartOfParentsList);
                if (ulEndOfParentsList != string::npos)
                {
                    string strParents(strDeclBlock,      // from
                                      ulStartOfParentsList,
                                      (ulEndOfParentsList - ulStartOfParentsList));

                    ULONG ulIndex = 0;
                    BSString strParentThis;
                    while (TRUE)
                    {
                        if (!strParents._extract_word(ulIndex,
                                                      strParentThis))
                            break;

                        if (    (strParentThis != "public")
                             && (strParentThis != "virtual")
                           )
                        {
                            // that's a class name:
                            // this still can have a template, e.g.
                            // "template <class T> class list_reverse_iterator : public list_iterator<T>"
                            // so we remove the <> stuff first
                            ULONG ulStart = strParentThis.find('<');
                            if (ulStart != string::npos)
                            {
                                strParentThis.erase(ulStart);
                            }

                            ParentNamesList.push_back(new BSString(strParentThis));
                        }

                        ulIndex++;
                    }
                }
            }

            // find beginning of structure to extract
            PCSZ pStartToExtract = *ppDeclBlockInFile;
            while (*pStartToExtract == ' ')
                ++pStartToExtract;

            // now find end of whole structure to extract
            PSZ     pStartOfBlock = strchr(*ppDeclBlockInFile, '{');
            if (pStartOfBlock)
            {
                PSZ     pSearch = pStartOfBlock + 1,
                        pEndOfBlock = 0;
                LONG    lBlockLevel = 1;

                while (pSearch)
                {
                    if (*pSearch == 0)
                        break;

                    switch (*pSearch)
                    {
                        case '/':
                            if (*(pSearch+1) == '*')
                            {
                                // start of C-style comment:
                                pSearch = strstr(pSearch + 2, "*/");
                            }
                            else if (*(pSearch+1) == '/')
                            {
                                // start of C++-style comment:
                                pSearch = strchr(pSearch + 1, '\n');
                            }
                        break;

                        case '{':
                            lBlockLevel++;
                        break;

                        case '}':
                            lBlockLevel--;
                            if (lBlockLevel < 1)
                            {
                                pEndOfBlock = pSearch + 1;
                                pSearch = 0;
                            }
                        break;
                    }

                    if (pSearch)
                        ++pSearch;
                }

                PSZ pClosingBracket = pEndOfBlock;

                if (pEndOfBlock)
                {
                    // now that we have the end of the block,
                    // go for the next ";"
                    pEndOfBlock = strchr(pEndOfBlock, ';');
                    if (pEndOfBlock)
                        ++pEndOfBlock;
                }

                if (pEndOfBlock)
                {
                    string strDefinitionOrig(pStartToExtract, pEndOfBlock);
                            // fixed bad operators V0.9.18 (2002-03-16) [umoeller]
                    string strDefinitionPRE(pStartToExtract, pEndOfBlock);
                    FormatPRE(strDefinitionPRE);

                    // create class structure
                    pClassNew = new XDClass(this,           // sourcefile
                                            strClassName,
                                            strNameForResolve,  // V0.9.18 (2002-03-16) [umoeller]
                                            strDefinitionPRE,
                                            XDClass::tXDClass);
                    // copy the list of parent names...
                    // pClassNew->_llParentNames = ParentNamesList;
                    list<string*>::iterator ParentStart = ParentNamesList.begin(),
                                            ParentEnd = ParentNamesList.end();
                    for (;
                         ParentStart != ParentEnd;
                         ParentStart++)
                    {
                        string *p = *ParentStart;
                        pClassNew->_llParentNames.push_back(new string(*p));
                    }

                    // now check for typedefs
                    if (strDefinitionOrig.find("typedef ") != string::npos)
                    {
                        // yes, we have a typedef:
                        // find the new declaration after "}"
                        PSZ pBeginNewRealName = pClosingBracket + 1;

                        while (*pBeginNewRealName == ' ')
                            ++pBeginNewRealName;
                        PSZ pEndNewRealName = pBeginNewRealName;
                        while (*pEndNewRealName)
                        {
                            if (    (*pEndNewRealName == ' ')
                                 || (*pEndNewRealName == ',')
                                 || (*pEndNewRealName == ';')
                               )
                                break;
                            ++pEndNewRealName;
                        }

                        string strTypedef(pBeginNewRealName, pEndNewRealName);
                        // printf("\n found typedef %s", strTypedef.c_str());
                        XDTypedef *pTD = new XDTypedef(this,        // sourcefile
                                                       strTypedef,
                                                       pClassNew);
                    }

                    // NOW SEARCH THE CLASS BLOCK RECURSIVELY
                    // for more comments...
                    // the call hierachy is this:
                    //      XDSourceFile::LoadAndParse calls
                    //          XDSourceFile::ParseBlock, which calls
                    //              XDSourceFile::ParseDeclComment, which calls
                    //                  XDSourceFile::ParseClass (we're here!);
                    //                  in this block, we go for XDSourceFile::ParseBlock
                    //                  again with the class (which will pass pClassNew
                    //                  as the "owning" class to all other parse functions
                    ParseBlock(strDefinitionOrig.c_str(),       // subblock
                               pClassNew);

                    // SEARCH ON AFTER THE CLOSING BRACKET
                    *ppDeclBlockInFile = pEndOfBlock;
                }
            } // end if (pStartOfBlock)
        } // end if (cbStructKeyword)
    } while (FALSE);

    return (pClassNew);
}

/*
 *@@ ParseDeclComment:
 *      this gets called from ParseSourceFile for
 *      all comments other than those with a "@@ sourcefile"
 *      tag. This includes functions, typedefs, structs,
 *      classes.
 *
 *      This calls ParseDefine, ParseFunction, or ParseClass
 *      in turn.
 *
 *      Returns TRUE if this comment really documented
 *      something that we recognize.
 */

BOOL XDSourceFile::ParseDeclComment(XDClass *pOwningClass,    // in: class of which this is a member
                                                              //     or NULL
                                    string &strComment,      // in: current comment block (null-terminated)
                                    PCSZ *ppAfterComment)    // in: pointer to end of comment in the source file;
                                                            // out: where to continue search...
                                                            // (raised by two by caller)
{
    BOOL brc = FALSE;

    if (G_ulVerbosity >= 4)
    {
        printf("\n  Entering " __FUNCTION__);
        fflush(stdout);
    }

    // this "loop" never loops; it's only to be able
    // to do a "break" upon errors
    do
    {
        /*
         * check for declaration comments:
         *
         */

        if (strComment.find("@@") == string::npos)
            // no marker tag:
            break;

        // check if we got a "define" thing
        XDDeclBase *pDecl;
        if (!(pDecl = ParseDefine(strComment,
                                  *ppAfterComment)))
        {
            // no define:
            // extract code block then
            // get pointer to next comment
            PSZ pNextComment = strstr(*ppAfterComment, "/*");
            // and to next code block
            PSZ pNextBlock = strstr(*ppAfterComment, "{");

            if (    // check whether there's no other comment before the block
                    ((pNextComment == 0) || (pNextBlock < pNextComment))
                    // and there must be a block following
                 && (pNextBlock)
               )
            {
                // extract full header up to "{", including that
                string strDeclBlock(*ppAfterComment, pNextBlock + 1);

                // no define: try function then
                pDecl = XDFunction::CreateFunction(this,
                                                   pOwningClass,
                                                   strComment,
                                                   strDeclBlock);
                // not a function comment: try class/structure then...
                // but we can't have such a thing in a class
                if ((!pDecl) && (pOwningClass == NULL))
                    pDecl = ParseClass(strComment,
                                       strDeclBlock,
                                       ppAfterComment); // this is modified!!
            } // end if pNextBlock
        } // end if if (!pDecl)

        if (pDecl)
        {
            /*
             * Reformat the declaration comment:
             *
             */

            // store new comment with function
            pDecl->_strCommentOriginal = strComment;
            pDecl->_strCommentNew = FormatComment(strComment);

            pDecl->ParseDeclChangeLogs();

            _ulFunctionCount++;

            brc = TRUE;
        }
    } while (FALSE); // never loop

    if (G_ulVerbosity >= 4)
    {
        printf("\n  Leaving " __FUNCTION__);
        fflush(stdout);
    }

    return (brc);
}

/*
 *@@ ParseSourcefileChangeLogs:
 *      called by XDSourceFile::ParseDeclComment to
 *      retrieve the "added" and "changed" tags.
 *
 *      At this point, _strFileComment has the comment
 *      as formatted by FormatComment.
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

ULONG XDSourceFile::ParseSourcefileChangeLogs()
{
    ULONG ulCount = 0;
    XDChangeLogEntry *pChangeLog;
    string strFilename(_strFilePath);
    PSZ     pStart = (PSZ)_strCommentNew.c_str(),
            pFound;
    while ((pFound = strstr(pStart, "@@added")))
    {
        pChangeLog = new XDChangeLogEntry(NULL,
                                          this,
                                          strFilename,
                                          pFound,
                                          TRUE);        // added
        ulCount++;
    }
    while ((pFound = strstr(pStart, "@@changed")))
    {
        pChangeLog = new XDChangeLogEntry(NULL,
                                          this,
                                          strFilename,
                                          pFound,
                                          FALSE);      // changed
        ulCount++;
    }

    return (ulCount);
}

/*
 *@@ ParseSourcefileComment:
 *      this gets called from ParseSourceFile if
 *      a "@ @ sourcefile" comment has been found in
 *      the source file.
 *
 *      Warning: this adds to SourceFileList if
 *      @ @ header tags are found.
 *
 *      Returns the number of tags found.
 *
 *@@changed V0.9.18 (2002-03-16) [umoeller]: fixed endless loop on missing header
 */

ULONG XDSourceFile::ParseSourcefileComment(string &strComment)
{
    if (_strCommentNew())
        printf("\n  --> Warning: file \"%s\" has more than one @@sourcefile flag, overwriting.",
                _strLongFilename.c_str());

    string s_strIncludeTag(G_pcszIncludeTag);

    /*
     * search for include directions (@ @ include):
     *
     */

    ULONG ulIncludeTag = 0;
    while ((ulIncludeTag = strComment.find(s_strIncludeTag,    // space-terminated
                                           ulIncludeTag + 1))
                != string::npos)
    {
        // extract include file (e.g. "dosh.h"):

        // 1) go beyond "@@include " tag
        ULONG ulIncludeName = strComment.find_first_not_of(' ',
                                                     ulIncludeTag + s_strIncludeTag.size());

        // 2) extract stuff after that up to end of line
        ULONG ulEOL = strComment.find('\n', ulIncludeName);
        string strTemp(strComment, ulIncludeName, (ulEOL - ulIncludeName));
        TranslateChars(strTemp);

        // 3) store in source file (or append to previous includes)
        string strNewInclude(_strInclude); // empty the first time
        if (_strInclude())
        {
            strNewInclude += "\n<br>";
            strNewInclude += strTemp;
        }
        else
            strNewInclude = strTemp;
        _strInclude = strNewInclude;

        if (G_ulVerbosity > 2)
            printf("\n  Found include %s", _strInclude.c_str());

        // 4) delete the include line from the comment
        strComment.erase(ulIncludeTag, (ulEOL - ulIncludeTag));
    }

    /*
     * search for header directions (@ @ header):
     *
     */

    PSZ pHeaderComment = (PSZ)strComment.c_str();
    while ((pHeaderComment = strstr(pHeaderComment, G_pcszHeaderTag))) // space-terminated
    {
        if (G_ulVerbosity > 2)
            printf("\n  Found header %s", pHeaderComment);

        // extract header file (e.g. "dosh.h")
        if (_strHeader())
        {
            printf("\n  --> Warning: file \"%s\" has more than one @@header flag.",
                    _strLongFilename.c_str());
        }

        PSZ pHeaderData = strchr(pHeaderComment, ' ');
        while (*pHeaderData == ' ')
            ++pHeaderData;
        PCSZ pEOL = strchr(pHeaderData, '\n');

        string strNewHeader;

        if (*pHeaderData == '\"')
        {
            // surrounded by quotes:
            PSZ pSecondQuote;
            if (    (pSecondQuote = strchr(pHeaderData + 2, '\"'))
                 && (pSecondQuote < pEOL)
               )
                strNewHeader.assign(pHeaderData + 1, pSecondQuote);
        }
        else
            // no quotes:
            strNewHeader.assign(pHeaderData, pEOL);

        if (!strNewHeader())
            printf("\n  --> Warning: Syntax error in @@header tag (file \"%s\").\n",
                   _strLongFilename.c_str());
        else
        {
            BOOL    fFound = FALSE;

            // header extracted:
            TranslateChars(strNewHeader);

            string strHeaderDir;

            if (access(strNewHeader.c_str(), 0) != 0)
            {
                // header not found in current dir:
                // try include paths
                list<string*>::iterator IncludeStart = IncludePaths.begin(),
                                    IncludeEnd = IncludePaths.end();
                for (;
                     IncludeStart != IncludeEnd;
                     ++IncludeStart)
                {
                    string strHeaderTemp(**IncludeStart);
                    strHeaderTemp += "\\";
                    strHeaderTemp += strNewHeader;

                    if (G_ulVerbosity > 2)
                    {
                        printf("\n  Trying include dir \"%s\" for header \"%s\"",
                               strHeaderTemp.c_str(),
                               strNewHeader.c_str());
                    }

                    if (access(strHeaderTemp.c_str(), 0) == 0)
                    {
                        // found:
                        strHeaderDir = **IncludeStart;
                        fFound = TRUE;
                    }
                }

                if (!fFound)
                {
                    printf("\n  --> Warning: header \"%s\" not found; specify -i option.\n",
                            strNewHeader.c_str());
                }
            } // end if (access(pszNewHeader, 0) != 0)
            else
            {
                // header found in current directory:
                strHeaderDir = _strFilePath;
                fFound = TRUE;      // V0.9.18 (2002-03-16) [umoeller]
            }

            if (fFound)
            {
                // store in XDSourceFile
                _strHeader = strNewHeader;

                if (G_ulVerbosity > 2)
                {
                    printf("\n  Found header %s in path %s",
                            strNewHeader.c_str(), strHeaderDir.c_str());
                    fflush(stdout);
                }

                // check if this header has already been processed
                list<PXDSourceFile>::iterator SrcFirst = G_SourceFilesList.begin(),
                                              SrcEnd = G_SourceFilesList.end();
                for (; SrcFirst != SrcEnd; ++SrcFirst)
                    if (*SrcFirst == strNewHeader)
                        break;

                if (SrcFirst == SrcEnd)
                {
                    // not found:
                    // create new source file instance for this header
                    XDSourceFile *pHeaderFile
                        = new XDSourceFile(strNewHeader.c_str(),
                                           strHeaderDir.c_str(),
                                           ulSourceFileCount++,
                                           this); // source file for which this is header

                    // add to source files list
                    G_SourceFilesList.push_back(pHeaderFile);
                    // add to header files list, so we can avoid
                    // duplicate headers
                    G_HeaderFilesList.push_back(pHeaderFile);
                }
                else
                    if (G_ulVerbosity)
                    {
                        printf("\n  Header %s already parsed, ignoring",
                                strNewHeader.c_str());
                        fflush(stdout);
                    }

                // delete the line from the comment
                while (pHeaderComment < pEOL)
                    *pHeaderComment++ = ' ';
                // and search for next from this pEOL then
            }
            else
                break;      // fixed loop V0.9.18 (2002-03-16) [umoeller]
        }
    } // end while ((pHeaderComment = strstr(pHeaderComment, G_pcszHeaderTag))) // space-terminated

    // search for SOM pseudo class declarations ("@@somclass ")
    ULONG ulSomTag = 0;

    string s_cstrSomTag(G_pcszSomTag);

    while ((ulSomTag = strComment.find(s_cstrSomTag,
                                       ulSomTag + 1))
                != string::npos)
    {
        // 1) go beyond "@@somclass " tag
        ULONG ulClassName = strComment.find_first_not_of(' ',
                                                         ulSomTag + s_cstrSomTag.size());

        // 2) extract class name (e.g. "XFolder");
        ULONG ulSpaceAfterClassName = strComment.find(' ',
                                                      ulClassName);
        string strSomClassName(strComment,
                               ulClassName,
                               (ulSpaceAfterClassName - ulClassName));

        // 3) extract prefix
        ULONG ulPrefix = strComment.find_first_not_of(' ',
                                                      ulSpaceAfterClassName);
        ULONG ulEOL = strComment.find('\n', ulPrefix);
        string strPrefix(strComment,
                         ulPrefix,
                         (ulEOL - ulPrefix));

        // create SOM-class object
        XDSomClass *pClass = new XDSomClass(this,
                                            strSomClassName,
                                            strPrefix);

        if (G_ulVerbosity > 2)
            printf("\n  Found Pseudo-SOM class \"%s\" (prefix: \"%s\")",
                   strSomClassName.c_str(), pClass->_strSomFuncPrefix.c_str());

        // store in XDSourceFile
        _ClassesList.push_back(pClass);
        _SomClassesList.push_back(pClass);
        // pSourceFile->_AllDeclsList.push_back(pClass);

        // delete the line from the comment
        strComment.erase(ulSomTag, (ulEOL - ulSomTag));
    }

    _strCommentOriginal = strComment;
    _strCommentNew = FormatComment(strComment);
    ParseSourcefileChangeLogs();
    ParseSourcefileReviews();

    return (1);
}

/*
 *@@ CreateCategory:
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

XDCategory* XDSourceFile::CreateCategory(string &strCategory,
                                         XDCategory *pParentCategory,
                                         string &strCategoryComment)
{
    XDCategory *pCtgReturn = NULL;

    list<PXDCategory> *pList;

    // determine list: either root or sublist
    if (pParentCategory == NULL)
        pList = &G_RootCategoriesList;
    else
        pList = &pParentCategory->_ChildCategoriesList;

    // check that list for whether the category exists already
    list<PXDCategory>::iterator CtgStart = pList->begin(),
                                CtgEnd = pList->end();
    for (;
         CtgStart != CtgEnd;
         ++CtgStart)
    {
        XDCategory *pCtgThis = *CtgStart;
        if (pCtgThis->_strIndexNameOrig == strCategory)
        {
            // exists already:
            pCtgReturn = pCtgThis;
            break;
        }
    }

    if (!pCtgReturn)
    {
        // not found yet:
        pCtgReturn = new XDCategory(strCategory,
                                    pParentCategory,
                                    this,      // source file
                                    strCategoryComment);
    }

    return (pCtgReturn);
}

/*
 *@@ ParseCategoryComment:
 *      gets called when XDSourceFile::ParseBlock finds
 *      a "category" tag in a comment. This extracts the
 *      glossary entry and creates an XDCategory for it.
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

ULONG XDSourceFile::ParseCategoryComment(string &strComment,
                                         ULONG pos)     // in: ofs of category tag in strComment
{
    // find first char of category name
    PSZ p = (PSZ)strComment.c_str() + pos + strlen(G_pcszCategoryTag);
    // skip ":"
    while ((*p) && (*p == ':'))
        p++;
    // skip spaces afterwards
    while ((*p) && (*p == ' '))
        p++;

    if (*p)
    {
        PCSZ pEOL = strhFindEOL(p, NULL);
        string strNull("");
        string strCategoryFull(p, pEOL);
        // string strCategoryComment(pEOL + 1);

        // find backslash
        PCSZ pStart = p;
        BOOL fCont = TRUE;
        XDCategory *pParentCategory = NULL;

        while (fCont)
        {
            string strCategory;
            PCSZ pNextBackslash = strchr(pStart, '\\');
            if (pNextBackslash)
            {
                strCategory.assign(pStart, pNextBackslash);
                pStart = pNextBackslash + 1;
            }
            else
            {
                strCategory.assign(pStart, pEOL);
                fCont = FALSE;
            }

            // create or find category
            pParentCategory = CreateCategory(strCategory,
                                             pParentCategory, // initially NULL
                                             // comment for this category:
                                             // pass the comment only if this is
                                             // really the last category
                                             (fCont) ? strNull : strComment);
            // on the next loop, we take the new or
            // found category as the parent
        }

        // store current category for all subsequent declarations
        // in source file
        _pCurrentCategory = pParentCategory;
    }

    return (1);
}

/*
 *@@ ParseSourcefileReviews:
 *      called by XDSourceFile::ParseSourcefileComment to
 *      find "reviewed" tags in a comment. This extracts the
 *      review entries and creates an XDReviewEntry for
 *      them.
 *
 *@@added V0.9.18 (2002-02-25) [lafaix]
 */

ULONG XDSourceFile::ParseSourcefileReviews()
{
    // find first char of review name
    PSZ     pStart = (PSZ)_strCommentNew.c_str(),
            pFound;
    ULONG   ulCount = 0;
    while ((pFound = strstr(pStart, G_pcszReviewTag)))
    {
        XDReviewEntry* pxdr = new XDReviewEntry(this, pFound);
        ulCount++;
    }

    return (ulCount);
}

/*
 *@@ ParseGlossaryComment:
 *      gets called when XDSourceFile::ParseBlock finds
 *      a "gloss" tag in a comment. This extracts the
 *      glossary entry and creates an XDGlossary for it.
 *
 *      The syntax of "gloss" is
 *
 *          gloss: identifier full name with spaces
 *
 *      The identifier must be the first word and can
 *      afterwards be used with "@identifier" in the
 *      comments to have it replaced with the
 *      "full name with spaces", which is turned into
 *      a link to the glossary entry.
 *
 *@@added V0.9.9 (2001-02-10) [umoeller]
 */

ULONG XDSourceFile::ParseGlossaryComment(string &strComment,
                                         ULONG pos) // in: ofs of gloss tag in strComment
{
    XDGlossary *pNew  = 0;

    // find first char of category name
    string s_strGlossTag(G_pcszGlossTag);

    PSZ p = (PSZ)strComment.c_str() + pos + s_strGlossTag.size();
    // skip spaces afterwards
    while ((*p) && (*p == ' '))
        p++;

    if ((*p) && (*p != '\n'))
    {
        // p should point to the identifier now

        // get end of line
        PCSZ pEOL = strhFindEOL(p, NULL);

        PCSZ pSpaceAfterIdentifier = strchr(p, ' ');
        if (    pSpaceAfterIdentifier
             && (pSpaceAfterIdentifier < pEOL)
           )
        {
            string strIdentifier(p, pSpaceAfterIdentifier);

            // skip spaces after ID
            while (    (*pSpaceAfterIdentifier)
                    && (*pSpaceAfterIdentifier == ' ')
                  )
                pSpaceAfterIdentifier++;

            if (*pSpaceAfterIdentifier && pSpaceAfterIdentifier < pEOL)
            {
                string strReadableName(pSpaceAfterIdentifier, pEOL);

                pNew = new XDGlossary(this,        // source file
                                      strIdentifier,
                                      strReadableName,
                                      strComment);
            }
            else
                printf("\n  --> Warning: cannot get readable name for @@gloss entry %s.",
                            strIdentifier.c_str());
        }
        else
        {
            string str(p, pEOL);
            printf("\n  --> Warning: cannot get identifier for @@gloss entry %s.",
                        str.c_str());
        }
    }
    else
        printf("\n  --> Warning: invalid syntax for @@gloss entry.");

    return (1);
}

/*
 *@@ ParseBlock:
 *      gets called by LoadAndParse for an entire source file...
 *      but will get called recursively by ParseClass to parse
 *      members of a class definition. This calls all the other
 *      parse methods in turn.
 *
 *@@added V0.9.7 (2001-01-07) [umoeller]
 */

ULONG XDSourceFile::ParseBlock(PCSZ pSearchPos,
                               PXDClass pOwningClass)    // NULL if not in a class block
{
    // here comes the search-for-all-comments loop
    do
    {
        // find next comment; the first time, this
        // searches from the top
        const char *pStartOfComment = strstr(pSearchPos, "/*");
        if (pStartOfComment == NULL)
            // no more comments in file
            break;

        const char *pAfterComment = strstr(pStartOfComment + 2, "*/");
        if (pAfterComment == 0)
        {
            // no matching end of comment: that's an error
            CHAR    szStart[200];
            strhncpy0(szStart, pStartOfComment, sizeof(szStart));
            string str;
            str._printf("--> Warning: Matching end-of-comment not found for: %s\n skipping source file",
                        szStart);
            fflush(stdout);
            throw Excpt(-1, str);
        }

        pAfterComment += 2;

        // extract comment into new string
        string strComment;
        strComment.assign(pStartOfComment, pAfterComment);
        _ulCommentCount++;
            // now we have the whole comment in pszComment

        // check if this is a special comment:

        ULONG   pos = 0;

        string s_strGlossTag(G_pcszGlossTag);

        // 0) check for "@@" in general... maybe we have nothing here
        if ((pos = strComment.find("@@")) != string::npos)
        {
            // 1) @@category
            if ((pos = strComment.find(G_pcszCategoryTag)) != string::npos)
            {
                /*
                 * comment introducing a category:
                 *
                 */

                _ulTagsCount += ParseCategoryComment(strComment,
                                                     pos);
            }
            // 2) @@gloss
            else if ((pos = strComment.find(s_strGlossTag)) != string::npos)
            {
                /*
                 * comment introducing a glossary entry:
                 *
                 */

                _ulTagsCount += ParseGlossaryComment(strComment,
                                                     pos);
            }
            // 3) @@sourcefile
            else
            {
                if (strComment.find(G_pcszSourcefileTag) != string::npos)
                {
                    /*
                     * comment for whole source file:
                     *
                     */

                    _ulTagsCount += ParseSourcefileComment(strComment);
                }
                else
                {
                    /*
                     * comment not for whole source file:
                     *
                     */

                    if (ParseDeclComment(pOwningClass,     // owning class... NULL if root
                                         strComment,
                                         &pAfterComment))
                        ++_ulTagsCount;
                }
            }
        } // if ((pos = strComment.find("@@")) != string::npos)

        // go for next comment
        pSearchPos = pAfterComment;

    } while (TRUE);

    return (0);
}

/*
 * ParseSourceFile:
 *      this takes a XDSourceFile structure, reads in the
 *      corresponding source file, analyzes all the comments
 *      and functions in there and creates a list of functions
 *      in the XDSourceFile structure.
 *
 *      This function has the main parsing logic for taking
 *      a source file into all those functions, classes etc.
 *      and creating lists accordingly.
 *
 *      Returns the number of tagged comments found.
 */

ULONG XDSourceFile::LoadAndParse()
{
    if (G_ulVerbosity)
        printf("Analyzing file \"%s\"", _strLongFilename.c_str());

    // read the source file
    PSZ     pszFile0 = NULL;
    APIRET  arc = doshLoadTextFile(_strLongFilename.c_str(),
                                   &pszFile0,
                                   NULL);
    if (arc != NO_ERROR)
    {
        throw Excpt(arc, "doshLoadTextFile");
    }

    // replace all \r\n with \n only
    ULONG cFile = strlen(pszFile0) + 1;
    PSZ pszFile = (PSZ)malloc(cFile);
    PSZ pSource = pszFile0,
        pTarget = pszFile;
    while (*pSource)
    {
        if (*pSource == '\r')
            // skip
            pSource++;
        else
            *pTarget++ = *pSource++;
    }
    *pTarget = 0;
    free(pszFile0);

    PCSZ pSearchPos = pszFile;

    ParseBlock(pSearchPos,
               NULL);       // no class yet

    free(pszFile);

    if (G_ulVerbosity > 1)
        printf("\n  %d comments, %d tagged found", _ulCommentCount, _ulTagsCount);
    if (G_ulVerbosity)
        printf("\n");
    return (_ulTagsCount);
}

/*
 *@@ WillBeWritten:
 *      returns TRUE if the source file
 *      contains enough tagged items so
 *      that it will be written as HTML.
 */

BOOL XDSourceFile::WillBeWritten()
{
    return (    (   (_ulFunctionCount != 0)
                 || (_strCommentNew())
                 || (_strInclude())
                )
            && (    (!_fIsHeader)
                 || (optWriteHeaders)
               )
           );
}

/*
 *@@ WriteHTMLSourcefileProlog:
 *      called by lots of WriteHTMLDescription functions
 *      to have the prolog specifying the file and
 *      category written into a declaration's HTML
 *      file.
 *
 *@@added V0.9.4 (2000-07-29) [umoeller]
 */

VOID XDSourceFile::WriteHTMLSourcefileProlog(FILE *ThisFile)
{
    // create link only for non-headers or
    //       if headers output is on
    if (    (!_fIsHeader)
         || (optWriteHeaders)
       )
        fprintf(ThisFile, "<b>File:</b> <a href=\"%s\">%s</A>\n",
                          QueryOutputFilename(),
                          _strLongFilename.c_str());
    else
        fprintf(ThisFile, "<b>File:</b> %s\n",
                          _strLongFilename.c_str());


    /*
     * #include's from source file (@ @ include):
     *
     */

    if (_strInclude())
    {
        fprintf(ThisFile, "<p><CODE>%s</CODE>",
                _strInclude.c_str());
    }

    if (_pMyHeader)
        if (_pMyHeader->_strInclude())
        {
            fprintf(ThisFile, "<p><CODE>%s</CODE>",
                    _pMyHeader->_strInclude.c_str());
        }
}

/*
 *@@ QueryHTMLAttributes:
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

const char* XDSourceFile::QueryHTMLAttributes()
{
    return ("GROUP=1 XPOS=10% WIDTH=30%");
}

/*
 * WriteHTML:
 *      this gets called for each input source file
 *      to produce a large number of HTML files
 *      (one for each declaration).
 */

VOID XDSourceFile::WriteHTMLDescription(FILE* IndexFile)
{
    if (G_ulVerbosity)
        printf("Writing HTML for funcs in file \"%s\"", _strLongFilename.c_str());

    /*
     * subfiles loop: create one HTML file for each function
     *
     */

    list<PXDDeclBase>::iterator DeclStart = _AllDeclsList.begin(),
                                DeclEnd = _AllDeclsList.end();
    for (;
         DeclStart != DeclEnd;
         ++DeclStart)
    {
        XDDeclBase  *pDeclThis = *DeclStart;
        // string    strFirstDeclName = pDeclThis->GetFirstIdentifier();

        CHAR    szThisFile[CCHMAXPATH] = "HTML\\";
        strcat(szThisFile, pDeclThis->QueryOutputFilename());

        if (G_ulVerbosity > 1)
            printf("\n    \"%s\"", szThisFile);
        else
        {
            printf(".");
            fflush(stdout);
        }

        pDeclThis->CreateHTMLFile();
    }

    if (G_ulVerbosity)
        printf("\n");

    // is header file?
    if (WillBeWritten())
    {

        /*
         * write index for this source file:
         *
         */

        if (_strCommentNew())
            fprintf(IndexFile, "\n<p>%s", _strCommentNew.c_str());

        if (_ulReviewCount)
        {
            fprintf(IndexFile, "\n<P><B>Reviews</B>");

            _ReviewsList.sort(CompareChangeLogEntries);

            list<PXDReviewEntry>::reverse_iterator ReviewStart2 = _ReviewsList.rbegin(),
                                                   ReviewEnd2 = _ReviewsList.rend();
            for (;
                 ReviewStart2 != ReviewEnd2;
                 ++ReviewStart2)
            {
                XDReviewEntry *pLogThis = *ReviewStart2;

                fprintf(IndexFile,
                        "\n<P><B>V%d.%d.%d</B>: \n",
                            pLogThis->_version._ulMajor,
                            pLogThis->_version._ulMinor,
                            pLogThis->_version._ulRevision);

                pLogThis->WriteHTMLDescription(IndexFile);
            }
        }

        if (_ulFunctionCount > 0)
        {
            fprintf(IndexFile, "\n<p><b>Declarations index for %s</b>\n<p>\n",
                        _strLongFilename.c_str());

            // V0.9.18 (2002-02-26) [lafaix]
            _AllDeclsList.sort(CompareDeclarations);

            DeclStart = _AllDeclsList.begin();
            DeclEnd = _AllDeclsList.end();
            for (;
                 DeclStart != DeclEnd;
                 ++DeclStart)
            {
                XDDeclBase *pDeclThis = *DeclStart;
                if (pDeclThis->IsA(XDIndexableBase::tXDIndexableBase))
                    fprintf(IndexFile, "\n<a href=\"%s\">%s</A><br>",
                            pDeclThis->QueryOutputFilename(),
                            pDeclThis->_strIndexNameHTML.c_str());    // first real name
            }
        }

        /* fprintf(IndexFile, "\n<p><hr>");
        WriteInfo(IndexFile);
        fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");

        fprintf(IndexFile, "\n</body>\n</html>");

        fclose(IndexFile); */
    }
}

/* ******************************************************************
 *
 *   Program startup
 *
 ********************************************************************/

/*
 * ExplainTitle:
 *
 */

void ExplainTitle()
{
    printf("xdoc V"BLDLEVEL_VERSION" ("__DATE__") (C) 1999-2002 Ulrich Mller\n");
    printf("Part of the WarpIN source package.\n");
    printf("This is free software under the GNU General Public Licence (GPL).\n");
    printf("Refer to the COPYING file in the WarpIN installation dir for details.\n");
    printf("\nxdoc goes thru the specified C source files and writes lots of HTML files \n");
    printf("into the HTML subdirectory of the current directory. This is created if it\n");
    printf("does not exist yet.\n");
}

/*
 *@@ ExplainSyntax:
 *
 */

void ExplainSyntax(PSZ pszError)
{
    if (pszError)
        printf("\nxdoc: %s\n", pszError);
    printf("\nUsage: xdoc [-options] <filemask> ... \n");
    printf("with <filemask> being your C/C++ source files.\n");
    printf("You may specify several file masks. Example: xdoc helpers\\*.c main\\*.c\n");
    printf("Options: \n");
    printf("  -i<path[;path2...]>  specify include path for headers, if not in same dir as \n");
    printf("                       the respective source file\n");
    printf("  -H            write out header files in files index (def.: do not include).\n");
    printf("  -R            do not resolve cross-references (def.: do resolve).\n");
    printf("  -C            write plain-text CHANGELOG file only; no HTML is produced.\n");
    printf("  -v{0|1|2|3|4} verbosity level (default: 1).\n");
    printf("  -h            show this help.\n");
    exit(2);
}

/*
 *@@ mainResolveClassHierarchies:
 *
 *@@added V0.9.18 (2002-03-16) [umoeller]
 */

void mainResolveClassHierarchies()
{
    if (G_ulVerbosity) {
        printf("\nResolving class hierarchies");
        fflush(stdout);
    }

    // for each class, go thru its parents and add this
    // class as a descendant to its parent
    list<PXDClass>::iterator ClassStart = G_AllClassesList.begin(),
                             ClassEnd = G_AllClassesList.end();
    for (;
         ClassStart != ClassEnd;
         ClassStart++)
    {
        printf(".");
        fflush(stdout);

        PXDClass pClassThis = *ClassStart;

        pClassThis->_cParentNames = 0;

        list<string*>::iterator ParentNameStart = pClassThis->_llParentNames.begin(),
                               ParentNameEnd = pClassThis->_llParentNames.end();
        for (;
             ParentNameStart != ParentNameEnd;
             ParentNameStart++)
        {
            string  *pstrParentName = *ParentNameStart;
            (pClassThis->_cParentNames)++;

            if (!pstrParentName->empty())
            {
                // store the parent class in the descendant class
                pClassThis->_pParent = FindClass(*pstrParentName);
                if (pClassThis->_pParent)
                {
                    // store the current class in the parent class's
                    // descendants list
                    pClassThis->_pParent->_DescendantsList.push_back(pClassThis);
                    (pClassThis->_pParent->_cDescendants)++;
                }
                else
                    printf("\n  --> Warning: parent class \"%s\" for \"%s\" not found.",
                           pstrParentName->c_str(),
                           pClassThis->_strIndexNameOrig.c_str());
            }
        }
    }

    // now sort all descendants lists
    for (ClassStart = G_AllClassesList.begin();
         ClassStart != ClassEnd;
         ClassStart++)
    {
        PXDClass pClassThis = *ClassStart;
        if (pClassThis->_cDescendants)
            // sort the descendants list by name... this uses the "<" operator
            pClassThis->_DescendantsList.sort(CompareClasses);

        // and store each class in the global declarations lists...
        // we have deferred this in XDClass::XDClass

        XDClass *pClassThis2 = pClassThis;
                    // the list needs a plain pointer
        pClassThis->_pSourceFile->_AllDeclsList.push_back(pClassThis2);
        G_AllDeclsList.push_back(pClassThis2);
    }
}

/*
 *@@ mainResolveMethodTables:
 *
 *@@added V0.9.18 (2002-03-16) [umoeller]
 */

void mainResolveMethodTables()
{
    if (G_ulVerbosity) {
        printf("\nResolving method tables");
        fflush(stdout);
    }

    list<PXDFunction>::iterator FuncStart = G_AllFunctionsList.begin(),
                                FuncEnd = G_AllFunctionsList.end();
    for (;
         FuncStart != FuncEnd;
         FuncStart++)
    {
        PXDFunction pFuncThis = *FuncStart;
        fflush(stdout);
        // check if this function is a member of a class,
        // but the class has not yet been resolved
        if (    (pFuncThis->_strOwningClass())
             && (pFuncThis->_pOwningClass == NULL)
           )
        {
            // yes: resolve the function now...
            pFuncThis->_pOwningClass = FindClass(pFuncThis->_strOwningClass);
        }

        XDClass *pClassThis;
        if (pClassThis = pFuncThis->_pOwningClass)
        {
            // in any case, add the function to the class's
            // method table
            XDFunction *pf2 = pFuncThis;
            pClassThis->_MemberFunctionsList.push_back(pf2);
            ++(pClassThis->_cMemberFunctions);
        }
    }

    // resolve overridden methods V0.9.18 (2002-03-16) [umoeller]
    for (FuncStart = G_AllFunctionsList.begin();
         FuncStart != FuncEnd;
         FuncStart++)
    {
        PXDFunction pFuncThis = *FuncStart;
        XDClass *pClassThis;
        if (pClassThis = pFuncThis->_pOwningClass)
        {
            printf(".");
            fflush(stdout);

            // climb up the parents and check method overrides
            while (pClassThis = pClassThis->_pParent)
            {
                list<PXDFunction>::iterator thatBegin = pClassThis->_MemberFunctionsList.begin(),
                                            thatEnd = pClassThis->_MemberFunctionsList.end();
                for (;
                     thatBegin != thatEnd;
                     ++thatBegin)
                {
                    XDFunction *pThat = *thatBegin;
                    if (    (pThat->_strPureName == pFuncThis->_strPureName)
                         && (pThat->_strMangledArgs == pFuncThis->_strMangledArgs)
                       )
                    {
                        // pFuncThis overrides pThat:
                        pFuncThis->_OverridesList.push_back(pThat);
                        pThat->_OverriddenByList.push_back(pFuncThis);

                        // and stop, we won't need any more parents
                        break;
                    }
                }
            }

        }
    }
}

/*
 * main:
 *
 */

int main(int argc, char *argv[])
{
    if (argc < 2)
    {
        ExplainTitle();
        ExplainSyntax(NULL);
    }

    try
    {
        // parse parameters on cmd line
        if (argc > 1)
        {
            SHORT i = 0;
            while (i++ < argc-1)
            {
                if (G_ulVerbosity >= 2)
                {
                    printf("parsing arg \"%s\"\n", argv[i]);
                    fflush(stdout);
                }

                if (argv[i][0] == '-')
                {
                    // option found:
                    SHORT i2;
                    for (i2 = 1;
                         i2 < strlen(argv[i]);
                         i2++)
                    {
                        switch (argv[i][i2])
                        {
                            case 'i':
                            {
                                i2++;
                                PCSZ pcszToken = &argv[i][i2],
                                     pcszNext;

                                if (!*pcszToken)
                                    ExplainSyntax("Syntax error with -i parameter.");

                                while (TRUE)
                                {
                                    string str;
                                    if (pcszNext = strchr(pcszToken + 1, ';'))
                                        str.assign(pcszToken,
                                                   pcszNext);
                                    else
                                        str.assign(pcszToken);

                                    str += '\\';

                                    if (G_ulVerbosity >= 2)
                                    {
                                        printf("    include path \"%s\"\n", str.c_str());
                                        fflush(stdout);
                                    }

                                    IncludePaths.push_back(new BSString(str));

                                    if (pcszNext)
                                        pcszToken = pcszNext + 1;
                                    else
                                        break;
                                }
                                // skip rest
                                i2 = strlen(argv[i]);
                            break; }

                            case 'H':
                                optWriteHeaders = TRUE;
                            break;

                            case 'R':
                                optResolveRefs = FALSE;
                            break;

                            case 'C':
                                optResolveRefs = FALSE;
                                optChangeLogOnly = TRUE;
                            break;

                            case 'v':
                                i2++;
                                switch (argv[i][i2])
                                {
                                    case '0': G_ulVerbosity = 0; break;
                                    case '1': G_ulVerbosity = 1; break;
                                    case '2': G_ulVerbosity = 2; break;
                                    case '3': G_ulVerbosity = 3; break;
                                    case '4': G_ulVerbosity = 4; break;
                                    case '5': G_ulVerbosity = 5; break;
                                    default: ExplainSyntax("Invalid verbosity level specified.");
                                }
                            break;

                            case 'h':
                                ExplainTitle();
                                ExplainSyntax(0);
                            break;

                            default:  // unknown option
                                ExplainSyntax("Invalid option specified.");
                            break;
                        }
                    }
                } // end if (argv[i][0] == '-')
                else
                {
                    /*
                     * collect file names:
                     *
                     */

                    CHAR szFileMask[CCHMAXPATH] = "";
                    PSZ pszFilePath = 0;

                    strcpy(szFileMask, argv[i]);

                    PSZ pszLastBackslash = strrchr(szFileMask, '\\');
                    if (pszLastBackslash)
                    {
                        pszFilePath = strhSubstr(szFileMask, pszLastBackslash+1);
                        printf("%s\n", pszFilePath);
                    }
                    // no option ("-"): seems to be filemask
                    HDIR          hdirFindHandle = HDIR_SYSTEM;
                    FILEFINDBUF3  ffb3           = {0};      // returned from FindFirst/Next
                    ULONG         ulResultBufLen = sizeof(FILEFINDBUF3);
                    ULONG         ulFindCount    = 1;        // look for 1 file at a time

                    APIRET arc = DosFindFirst( szFileMask,    // file pattern
                                        &hdirFindHandle,      // directory search handle
                                        FILE_NORMAL,          // search attribute
                                        &ffb3,                // result buffer
                                        ulResultBufLen,       // result buffer length
                                        &ulFindCount,         // number of entries to find
                                        FIL_STANDARD);        // return level 1 file info

                    if (arc != NO_ERROR)
                    {
                        throw Excpt(arc, "DosFindFirst");
                    }

                    // keep finding the next file until there are no more files
                    while (arc != ERROR_NO_MORE_FILES)
                    {
                        PXDSourceFile pSourceFile = new XDSourceFile(ffb3.achName,
                                                                 pszFilePath,
                                                                 ulSourceFileCount++,
                                                                 NULL);    // no header

                        if (G_ulVerbosity > 2)
                            printf("\nFound file \"%s\"", pSourceFile->_strLongFilename.c_str());

                        G_SourceFilesList.push_back(pSourceFile);

                        ulFindCount = 1;                    // reset find count
                        arc = DosFindNext(hdirFindHandle,
                                         &ffb3,             // result buffer
                                         ulResultBufLen,
                                         &ulFindCount);     // number of entries to find

                    } // endwhile

                    arc = DosFindClose(hdirFindHandle);    // close our find handle

                } // end elseif (argv[i][0] == '-')
            }
        }

        if (ulSourceFileCount > 0)
        {
            list<PXDSourceFile>::iterator FileStart,
                                          FileEnd;

            /*
             * analyze source files:
             *
             */

            if (G_ulVerbosity)
                printf("Analyzing %d files...\n", ulSourceFileCount);

            BOOL    fFilesToGo = TRUE;
            FileStart = G_SourceFilesList.begin();
            while (fFilesToGo)
            {
                (*FileStart)->LoadAndParse();
                        // warning: this adds to SourceFileList if
                        // @@header tags are found

                // next source file
                ++FileStart;

                if (FileStart == G_SourceFilesList.end())
                                // we have to re-check end() every time because
                                // headers might have been added
                    fFilesToGo = FALSE;
            } // end while (!fExit)

            /*
             * resolve class hierarchies:
             *
             */

            mainResolveClassHierarchies();

            /*
             * resolve method tables:
             *
             */

            mainResolveMethodTables();

            /*
             * resolve cross-references:
             *
             */

            if (optResolveRefs)             // command-line option, disabled for
                                            // change-log only also
            {
                // resolve in categories
                if (G_ulVerbosity) {
                    printf("\nResolving cross-references in categories");
                    fflush(stdout);
                }

                list<PXDCategory>::iterator CtgStart = G_AllCategoriesFlatList.begin(),
                                            CtgEnd = G_AllCategoriesFlatList.end();
                for (;
                     CtgStart != CtgEnd;
                     ++CtgStart)
                {
                    PXDCategory pCtgThis = *CtgStart;
                    if (G_ulVerbosity > 1)
                        printf("\n%s", pCtgThis->_strIndexNameOrig.c_str());
                    else
                        printf(".");
                    fflush(stdout);
                    if (pCtgThis->_strCommentNew())
                        ResolveRefs(pCtgThis->_strCommentNew,
                                    NULL);
                }

                // resolve in file comments
                if (G_ulVerbosity) {
                    printf("\nResolving cross-references in file comments");
                    fflush(stdout);
                }

                FileStart = G_SourceFilesList.begin();
                FileEnd = G_SourceFilesList.end();
                for (;
                     FileStart != FileEnd;
                     ++FileStart)
                {
                    PXDSourceFile pSourceThis = *FileStart;
                    if (G_ulVerbosity > 1)
                        printf("\n%s", pSourceThis->_strIndexNameHTML.c_str());
                    else
                        printf(".");
                    fflush(stdout);
                    if (pSourceThis->_strCommentNew())
                        ResolveRefs(pSourceThis->_strCommentNew,
                                    NULL);
                }

                // resolve in declaration comments
                if (G_ulVerbosity) {
                    printf("\nResolving cross-references in declarations");
                    fflush(stdout);
                }

                list<PXDDeclBase>::iterator FuncStart = G_AllDeclsList.begin(),
                                            FuncEnd = G_AllDeclsList.end();
                for (;
                     FuncStart != FuncEnd;
                     ++FuncStart)
                {
                    (*FuncStart)->Resolve();
                }

            } // end if optResolveRefs

            if (optChangeLogOnly)
            {
                /*
                 * write change log only:
                 *
                 */

                WriteChangeLogOnly();
            }
            else
            {
                /*
                 * write HTML output:
                 *
                 */

                if (G_ulVerbosity)
                    printf("\nWriting HTML output into \".\\HTML\"...\n");

                DosCreateDir("HTML", 0);
                FileStart = G_SourceFilesList.begin();
                FileEnd = G_SourceFilesList.end();
                for (;
                     FileStart != FileEnd;
                     ++FileStart)
                {
                     (*FileStart)->CreateHTMLFile();
                }

                WriteGlobalIndices();
            }
        }
    }
    catch(Excpt &X)
    {
        printf("\n\nxdoc: DOS error %d occured.\n"
               "    Context: %s\n"
               "    Type 'help %d' to get more info.",
               X._arc,
               X._strContext.c_str(),
               X._arc );
        return (X._arc);
    }
    catch(...)
    {
        printf("\nxdoc: main() caught an undefined exception. Terminating.\n");
        return (1);
    }

    return (0);
}

/* ******************************************************************
 *
 *   HTML output
 *
 ********************************************************************/

/*
 *@@ DumpCategoriesFor:
 *
 */

VOID DumpCategoriesFor(FILE *File,
                       XDCategory *pParentCategory)
{
    list<PXDCategory> *pList;

    if (pParentCategory == NULL)
        pList = &G_RootCategoriesList;
    else
        pList = &pParentCategory->_ChildCategoriesList;

    // sort the list according to category names
    pList->sort(CompareCategories);

    if (!pParentCategory)
        fprintf(File, "\n<ul>");

    list<PXDCategory>::iterator CtgStart = pList->begin(),
                                CtgEnd = pList->end();
    for (;
         CtgStart != CtgEnd;
         ++CtgStart)
    {
        XDCategory *pCtgThis = *CtgStart;

        if (!pParentCategory)
            fprintf(File, "\n<li><a href=\"%s\">%s%s</A> (%d items)",
                          pCtgThis->QueryOutputFilename(),
                         (pCtgThis->_cDirectChildCategories)
                            // if that category has child categories in turn,
                            // add "@"... yahoo-style
                            ? "@"
                            : "",
                          pCtgThis->_strIndexNameHTML.c_str(),
                          pCtgThis->_cTotalMemberDecls);

        // and write HTML
        pCtgThis->CreateHTMLFile();

        DumpCategoriesFor(File, pCtgThis);
    }

    if (!pParentCategory)
        fprintf(File, "\n</ul>");
}

/*
 *@@ WriteCategoriesIndex:
 *      calls DumpCategoriesFor, which recurses
 *      into subcategories.
 *
 *@@added V0.9.4 (2000-07-28) [umoeller]
 */

VOID WriteCategoriesIndex(VOID)
{
    // files index
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexCategories);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<html NOSUBLINKS=index WIDTH=30%c>\n"
                       "<head>\n<title>Categories Index</title>\n</head>\n",
                       '%'
                       );
    fprintf(IndexFile, "<body>\n<h2>Index of Categories</h2>\n");

    DumpCategoriesFor(IndexFile, NULL);

    fprintf(IndexFile, "\n<p><hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);
}

/*
 *@@ WriteSourceFilesIndex:
 *
 *@@added V0.9.1 (2000-02-06) [umoeller]
 */

VOID WriteSourceFilesIndex(VOID)
{
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexSrcFiles);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<html NOSUBLINKS=index SUBLINKS=f WIDTH=30%c>\n"
                       "<head>\n<title>Files Index</title>\n</head>\n",
                       '%'
                       );
    fprintf(IndexFile, "<body>\n<h2>Index of All Source Files</h2>\n");

    list<PXDSourceFile>::iterator FileStart= G_SourceFilesList.begin(),
                                  FileEnd = G_SourceFilesList.end();
    for (;
         FileStart != FileEnd;
         ++FileStart)
    {
        PXDSourceFile pSourceFile = *FileStart;
        // is header?
        if (pSourceFile->WillBeWritten())
            fprintf(IndexFile, "\n<br><a href=\"%s\">%s</A>",
                    pSourceFile->QueryOutputFilename(),
                    pSourceFile->_strLongFilename.c_str());
    }
    fprintf(IndexFile, "\n<p><hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);
}

/*
 *@@ WriteFunctionsIndex:
 *
 *@@added V0.9.1 (2000-02-06) [umoeller]
 */

VOID WriteFunctionsIndex(VOID)
{
    // index of all classes
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexFuncs);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<html WIDTH=30%c>\n"
                       "<head>\n<title>Functions Index</title>\n</head>\n",
                       '%');
    fprintf(IndexFile, "<body>\n<h2>Index of All Functions</h2>\n");

    // sort the damn thing according to short names;
    // my list template uses the "<" operator on the member objects
    // for "sort"
    G_AllFunctionsList.sort(CompareFunctions);

    list<PXDFunction>::iterator FuncStart = G_AllFunctionsList.begin(),
                                FuncEnd = G_AllFunctionsList.end();
    for (;
         FuncStart != FuncEnd;
         ++FuncStart)
    {
        XDFunction *pDeclThis = *FuncStart;
        fprintf(IndexFile, "\n<br><a href=\"%s\">%s</A>",
                pDeclThis->QueryOutputFilename(),
                pDeclThis->_strIndexNameHTML.c_str()); // short name

        // if it's a class method, add the class name in brackets
        if (pDeclThis->_pOwningClass)
        {
            // string strClassName = pDeclThis->_pOwningClass->GetFirstIdentifier();
            // fprintf(IndexFile, " (Class: %s)", strClassName.c_str());
        }
    }

    fprintf(IndexFile, "\n<p><hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);
}

/*
 *@@ WriteClassesIndex:
 *
 *@@added V0.9.1 (2000-02-06) [umoeller]
 */

VOID WriteClassesIndex(VOID)
{
    // index of all classes
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexClasses);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<html WIDTH=30%c>\n"
                       "<head>\n<title>Class Hierarchy</title>\n</head>\n",
                       '%');
    fprintf(IndexFile, "<body>\n<h2>Hierarchy of All Classes</h2>\n<ul>");

    // sort the damn thing according to short names;
    // my list template uses the "<" operator on the member objects
    // for "sort"
    G_AllClassesList.sort(CompareClasses);

    list<PXDClass>::iterator ClsStart2 = G_AllClassesList.begin(),
                             ClsEnd2 = G_AllClassesList.end();
    for (;
         ClsStart2 != ClsEnd2;
         ++ClsStart2)
    {
        PXDClass pClsThis = *ClsStart2;
        // dump only classes without parents (root classes)
        // descendant classes are written by DumPDescendants
        if (pClsThis->_cParentNames == 0)
        {
            PrintClass(IndexFile,
                       pClsThis,
                       NULL);
            /* fprintf(IndexFile, "\n<li><a href=\"%s\">%s</A>",
                    pClsThis->QueryOutputFilename(),
                    pClsThis->_strIndexName.c_str()); // short name
                    */

            DumpDescendants(IndexFile,
                            pClsThis,
                            NULL);
        }
    }

    fprintf(IndexFile, "\n</ul>\n<hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);
}

/*
 *@@ WriteDefinesIndex:
 *
 *@@added V0.9.1 (2000-02-06) [umoeller]
 */

VOID WriteDefinesIndex(VOID)
{
    // index of all classes
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexOther);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<html WIDTH=30%c>\n"
                       "<head>\n<title>Other Definitions Index</title>\n</head>\n",
                       '%');
    fprintf(IndexFile, "<body>\n<h2>Index of Definitions:</h2>\n");

    // sort the damn thing according to short names;
    // my list template uses the "<" operator on the member objects
    // for "sort"
    G_AllDefinesList.sort(CompareDefines);

    list<PXDDefine>::iterator DefStart = G_AllDefinesList.begin(),
                              DefEnd = G_AllDefinesList.end();
    for (;
         DefStart != DefEnd;
         ++DefStart)
    {
        PXDDefine pDefThis = *DefStart;
        fprintf(IndexFile, "\n<br><a href=\"%s\">%s</A>",
                pDefThis->QueryOutputFilename(),
                pDefThis->_strIndexNameHTML.c_str()); // short name
    }

    fprintf(IndexFile, "\n<p><hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);
}

/*
 *@@ WriteAllDeclsIndex:
 *
 *@@added V0.9.1 (2000-02-06) [umoeller]
 */

VOID WriteAllDeclsIndex(VOID)
{
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexAll);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<html WIDTH=30%c>\n"
                       "<head>\n<title>Complete Index</title>\n</head>\n",
                       '%');
    fprintf(IndexFile, "<body>\n<h2>Complete Index of All Declarations</h2>\n");

    // sort the damn thing according to short names;
    // my list template uses the "<" operator on the member objects
    // for "sort"
    G_AllDeclsList.sort(CompareDeclarations);

    list<PXDDeclBase>::iterator FuncStart2 = G_AllDeclsList.begin(),
                                FuncEnd2 = G_AllDeclsList.end();
    for (;
         FuncStart2 != FuncEnd2;
         ++FuncStart2)
    {
        XDDeclBase *pDeclThis = *FuncStart2;

        if (pDeclThis->IsA(XDIndexableBase::tXDIndexableBase))   // V0.9.9 (2001-02-10) [umoeller]
        {
            fprintf(IndexFile, "\n<br><a href=\"%s\">%s</A> (",
                    pDeclThis->QueryOutputFilename(),
                    pDeclThis->_strIndexNameHTML.c_str()); // short name

            // append source file name
            fprintf(IndexFile, "%s",
                    pDeclThis->_pSourceFile->_strLongFilename.c_str());

            // if it's a class method, add the class name in brackets
            /* if (pDeclThis->_pOwningClass)
            {
                string strClassName = pDeclThis->_pOwningClass->GetFirstIdentifier();
                fprintf(IndexFile, ", Class: %s", strClassName.c_str());
            } */

            fprintf(IndexFile, ")");
        }
    }

    fprintf(IndexFile, "\n<p><hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);
}

/*
 *@@ WriteChangesByDate:
 *
 *@@added V0.9.4 (2000-07-27) [umoeller]
 */

VOID WriteChangesByDate(VOID)
{
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexChangesByDate);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<html WIDTH=30%c>\n"
                       "<head>\n<title>Index of All Changes by Date</title>\n</head>\n",
                       '%');
    fprintf(IndexFile, "<body>\n<h2>Index of All Changes By Date</h2>\n");
    fprintf(IndexFile, "\n\n<dl>");

    // sort the damn thing according to date;
    // my list template uses the "<" operator on the member objects
    // for "sort"
    G_ChangeLogsList.sort(CompareChangeLogEntries);

    // XDDate  *pLastDate = NULL;
    XDVersion  *pLastVersion = NULL;
    BOOL    fInUL = FALSE;

    // now dump all log entries
    list<PXDChangeLogEntry>::reverse_iterator LogStart2 = G_ChangeLogsList.rbegin(),
                                              LogEnd2 = G_ChangeLogsList.rend();
    for (;
         LogStart2 != LogEnd2;
         ++LogStart2)
    {
        XDChangeLogEntry *pLogThis = *LogStart2;

        if (    (pLastVersion == NULL) // first run?
             || (memcmp(pLastVersion, &pLogThis->_version, sizeof(pLogThis->_version))
                    != 0)           // version changed?
           )
        {
            if (fInUL)
                // not first run:
                fprintf(IndexFile, "\n</ul>");

            fprintf(IndexFile,
                    "\n<dt><b>V%d.%d.%d</b>: \n<dd><ul>",
                    pLogThis->_version._ulMajor,
                    pLogThis->_version._ulMinor,
                    pLogThis->_version._ulRevision);

            pLastVersion = &pLogThis->_version;

            fInUL = TRUE;
        }

#if 0
        if (pLogThis->_pMyDecl)
        {
            // log entry for declaration:
            fprintf(IndexFile,
                    "\n<li><a href=\"%s\">%s</A> (",
                    pLogThis->_pMyDecl->QueryOutputFilename(),
                    pLogThis->_pMyDecl->_strIndexName.c_str()); // GetFirstIdentifier().c_str()); // short name
            // append source file name
            fprintf(IndexFile, "%s",
                    pLogThis->_pMyDecl->_pSourceFile->_szLongFilename);

            // if it's a class method, add the class name in brackets
            /* if (pLogThis->_pMyDecl->_pOwningClass)
            {
                string strClassName = pLogThis->_pMyDecl->_pOwningClass->GetFirstIdentifier();
                fprintf(IndexFile, ", Class: %s", strClassName.c_str());
            } */
            fprintf(IndexFile,
                    ")");
        }
        else if (pLogThis->_pMySourceFile)
            // log entry for source file:
            fprintf(IndexFile,
                    "\n<li>File <a href=\"%s\">%s</A>",
                    pLogThis->_pMySourceFile->_strIndexName.c_str(),
                    pLogThis->_pMySourceFile->_strIndexName.c_str());

        if (pLogThis->_fAdded)
            fprintf(IndexFile, " added");
        else
            fprintf(IndexFile, " changed");
        if (pLogThis->_strAuthor())
            fprintf(IndexFile, " by <I>%s</I>", pLogThis->_strAuthor.c_str());

        // add date
        if (pLogThis->_date._ulYear)
        {
            fprintf(IndexFile,
                    " (%04d-%02d-%02d)",
                    pLogThis->_date._ulYear,
                    pLogThis->_date._ulMonth,
                    pLogThis->_date._ulDay);
        }

        if (pLogThis->_strDescription())
            fprintf(IndexFile, ":\n<br>%s", pLogThis->_strDescription.c_str());
#endif

        fprintf(IndexFile, "\n<LI>");

        pLogThis->WriteHTMLDescription(IndexFile);
    }

    if (fInUL)
        // not first run:
        fprintf(IndexFile, "\n</UL>");

    fprintf(IndexFile, "\n</DL><P><HR>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<BR>\n<A HREF=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</BODY>\n</HTML>");
    fclose(IndexFile);
}

/*
 *@@ WriteReviews:
 *
 *@@added V0.9.18 (2002-02-25) [lafaix]
 */

VOID WriteReviews(VOID)
{
    CHAR    szFilename[CCHMAXPATH];
    sprintf(szFilename, "HTML\\%s",
                        G_pcszIndexReviews);
    FILE *IndexFile = fopen(szFilename, "w");

    fprintf(IndexFile, "<HTML WIDTH=30%c>\n"
                       "<HEAD>\n<TITLE>Index of All Reviews by Date</TITLE>\n</HEAD>\n",
                       '%');
    fprintf(IndexFile, "<BODY>\n<H2>Index of All Reviews By Date</H2>\n");
    fprintf(IndexFile, "\n\n<DL>");

    // sort the damn thing according to date;
    // my list template uses the "<" operator on the member objects
    // for "sort"
    G_ReviewsList.sort(CompareChangeLogEntries);

    XDVersion  *pLastVersion = NULL;
    BOOL    fInUL = FALSE;

    // now dump all log entries
    list<PXDReviewEntry>::reverse_iterator ReviewStart2 = G_ReviewsList.rbegin(),
                                           ReviewEnd2 = G_ReviewsList.rend();
    for (;
         ReviewStart2 != ReviewEnd2;
         ++ReviewStart2)
    {
        XDReviewEntry *pLogThis = *ReviewStart2;

        if (    (pLastVersion == NULL) // first run?
             || (memcmp(pLastVersion, &pLogThis->_version, sizeof(pLogThis->_version))
                    != 0)           // version changed?
           )
        {
            if (fInUL)
                // not first run:
                fprintf(IndexFile, "\n</UL>");

            fprintf(IndexFile,
                    "\n<DT><B>V%d.%d.%d</B>: \n<DD><UL>",
                    pLogThis->_version._ulMajor,
                    pLogThis->_version._ulMinor,
                    pLogThis->_version._ulRevision);

            pLastVersion = &pLogThis->_version;

            fInUL = TRUE;
        }

        fprintf(IndexFile,
                "\n<LI>");

        pLogThis->WriteHTMLDescription(IndexFile);
    }

    if (fInUL)
        // not first run:
        fprintf(IndexFile, "\n</ul>");

    fprintf(IndexFile, "\n</dl><p><hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n<br>\n<a href=\"index.htm\">[Index]</A> ");
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);
}

/*
 * WriteGlobalIndices:
 *
 */

VOID WriteGlobalIndices(VOID)
{
    if (G_ulVerbosity)
        printf("Writing global indices\n");

    // main index pointing to other indices
    FILE *IndexFile = fopen("HTML\\index.htm", "w");
    fprintf(IndexFile, "<html"
                       " NOSUBLINKS=index"
                       ">\n"
                       "<head>\n<title>Main Index</title>\n</head>\n");
    fprintf(IndexFile, "<body>\n<h2>Indices:</h2>\n");

    fprintf(IndexFile, "<ul><li><a href=\"%s\">Index of categories</A>\n",
                       G_pcszIndexCategories);
    fprintf(IndexFile, "<p><li><a href=\"%s\">Index of source files</A>\n",
                       G_pcszIndexSrcFiles);
    fprintf(IndexFile, "<p><li>Declarations indices:\n");
    fprintf(IndexFile, "<ul><li><a href=\"%s\">All declarations</A>\n",
                       G_pcszIndexAll);
    fprintf(IndexFile, "<li><a href=\"%s\">Functions only</A>\n",
                       G_pcszIndexFuncs);
    fprintf(IndexFile, "<li><a href=\"%s\">Class hierarchy</A>\n",
                       G_pcszIndexClasses);
    fprintf(IndexFile, "<li><a href=\"%s\">Other definitions</A>\n",
                       G_pcszIndexOther);

//     fprintf(IndexFile, "</ul><br><li>Change logs:\n");
//     fprintf(IndexFile, "<ul><li><a href=\"%s\">All changes by version/date</A>\n",
//                        G_pcszIndexChangesByDate);
//     fprintf(IndexFile, "</ul>\n");

    fprintf(IndexFile, "</UL>"
                       "<BR><LI>Change logs:\n");
    fprintf(IndexFile, "<UL>"
                       "<LI><A HREF=\"%s\">All changes by version/date</A>\n",
                       G_pcszIndexChangesByDate);
    fprintf(IndexFile, "</UL>\n");
    fprintf(IndexFile, "<BR><LI>Reviews:\n");
    fprintf(IndexFile, "<UL><LI><A HREF=\"%s\">All reviews by version/date</A>\n",
                       G_pcszIndexReviews);
    fprintf(IndexFile, "</UL>\n");
    fprintf(IndexFile, "</UL>\n");

    fprintf(IndexFile, "\n<p><hr>");
    WriteInfo(IndexFile);
    fprintf(IndexFile, "\n</body>\n</html>");
    fclose(IndexFile);

    WriteSourceFilesIndex();
    WriteFunctionsIndex();
    WriteClassesIndex();
    WriteDefinesIndex();
    WriteAllDeclsIndex();
    WriteChangesByDate();
    WriteReviews(); // V0.9.18 (2002-02-25) [lafaix]

    if (G_ulVerbosity)
        printf("Writing categories\n");
    WriteCategoriesIndex();

    if (G_ulVerbosity)
        printf("Done!\n");
}

/*
 *@@ WriteChangeLogOnly:
 *
 *@@added V0.9.9 (2001-03-27) [umoeller]
 */

VOID WriteChangeLogOnly(VOID)
{
    if (G_ulVerbosity)
        printf("\nWriting CHANGELOG...\n");

    // main index pointing to other indices
    FILE *IndexFile = fopen("CHANGELOG", "w");

    fprintf(IndexFile, "Index of all changes by date\n");
    fprintf(IndexFile, "----------------------------\n\n");

    // sort the damn thing according to date;
    // my list template uses the "<" operator on the member objects
    // for "sort"
    G_ChangeLogsList.sort(CompareChangeLogEntries);

    // XDDate  *pLastDate = NULL;
    XDVersion  *pLastVersion = NULL;
    BOOL    fInUL = FALSE;

    // now dump all log entries
    list<PXDChangeLogEntry>::reverse_iterator LogStart2 = G_ChangeLogsList.rbegin(),
                                              LogEnd2 = G_ChangeLogsList.rend();
    for (;
         LogStart2 != LogEnd2;
         ++LogStart2)
    {
        XDChangeLogEntry *pLogThis = *LogStart2;

        if (    (pLastVersion == NULL) // first run?
             || (memcmp(pLastVersion, &pLogThis->_version, sizeof(pLogThis->_version))
                    != 0)           // version changed?
           )
        {
            if (fInUL)
                // not first run:
                fprintf(IndexFile, "\n");

            fprintf(IndexFile,
                    "\n***********************************");
            fprintf(IndexFile,
                    "\n*");

            fprintf(IndexFile,
                    "\n* V%d.%d.%d:",
                    pLogThis->_version._ulMajor,
                    pLogThis->_version._ulMinor,
                    pLogThis->_version._ulRevision);

            fprintf(IndexFile,
                    "\n*");
            fprintf(IndexFile,
                    "\n***********************************\n");

            pLastVersion = &pLogThis->_version;

            fInUL = TRUE;
        }

        if (pLogThis->_pMyDecl)
        {
            // log entry for declaration:
            fprintf(IndexFile,
                    "\n  %s (",
                    pLogThis->_pMyDecl->_strIndexNameOrig.c_str());
            // append source file name
            fprintf(IndexFile, "%s",
                    pLogThis->_pMyDecl->_pSourceFile->_strLongFilename.c_str());

            fprintf(IndexFile,
                    ")");
        }
        else if (pLogThis->_pMySourceFile)
            // log entry for source file:
            fprintf(IndexFile,
                    "\n  File %s",
                    pLogThis->_pMySourceFile->_strIndexNameHTML.c_str());


        if (pLogThis->_fAdded)
            fprintf(IndexFile, " added");
        else
            fprintf(IndexFile, " changed");
        if (pLogThis->_strAuthor())
            fprintf(IndexFile, " by %s", pLogThis->_strAuthor.c_str());

        // add date
        if (pLogThis->_date._ulYear)
        {
            fprintf(IndexFile,
                    " (%04d-%02d-%02d)",
                    pLogThis->_date._ulYear,
                    pLogThis->_date._ulMonth,
                    pLogThis->_date._ulDay);
        }

        if (pLogThis->_strDescription())
            fprintf(IndexFile, ":\n    %s", pLogThis->_strDescription.c_str());

        fprintf(IndexFile, "\n");
    }

    if (fInUL)
        // not first run:
        fprintf(IndexFile, "\n");

    fclose(IndexFile);

    if (G_ulVerbosity)
        printf("Done!\n");
}

