
/*
 * strrpl.c:
 *      go thru the files specified on the command line
 *      and replace strings therein.
 *      Type "strrpl" for more info.
 *
 *      Changes for V0.9.0:
 *      -- -t option for test mode added;
 *      -- support for escape characters;
 *      -- lotsa bugfixes against recursive replaces.
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>

#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include "setup.h"

#include "helpers\dosh.h"
#include "helpers\stringh.h"
#include "helpers\xstring.h"

#include "bldlevel.h"

PSZ         G_pszFileMask = NULL;
XSTRING     // G_strSearchForOriginal,
            G_strSearchFor,
            G_strReplaceWith;
size_t      G_ShiftTable[256];

BOOL        G_fRepeat = FALSE;
ULONG       G_ulTotalFound = 0;

BOOL        G_optTestMode = FALSE,
            G_optQuietMode = FALSE;

/*
 *  Explain:
 *
 */

void Explain(PCSZ pcszFormat,
             ...)
{
    if (pcszFormat)
    {
        va_list arg_ptr;
        printf("strrpl error: ");
        va_start(arg_ptr, pcszFormat);
        vprintf(pcszFormat, arg_ptr);
        va_end(arg_ptr);
        printf("\n");
    }
    else
    {
        printf("strrpl V"BLDLEVEL_VERSION" ("__DATE__") (C) 1998-2000 Ulrich Mller\n");
        printf("Part of the WarpIN source package.\n");
        printf("This is free software under the GNU General Public Licence (GPL).\n");
        printf("Refer to the COPYING file in the WarpIN installation dir for details.\n");
        printf("Goes thru the specified files and replaces strings therein.\n");
    }

    printf("\nUsage: strrpl [-t] <filemask> <search> <replace>\n");
    printf("with:\n");
    printf("   <filemask>  the files to search for (e.g. \"*.html\")\n");
    printf("   <search>    the string to search for\n");
    printf("   <replace>   the string to replace all occurences of <search> with\n");
    printf("Options: -t    test mode, don't write files\n");
    printf("         -q    quiet mode, don't output replaced positions\n");
    printf("\nEscape characters in search/replace strings:\n");
    printf("   \\q          double quote\n");
    printf("   \\r          carriage return (0x0d)\n");
    printf("   \\n          newline (0x0a)\n");
    printf("   \\-          dash (-)\n");
    printf("   \\\\          backslash character\n");
    printf("Note: DOS and OS/2 use \\r\\n for line breaks (0x0d, 0x0a).\n");
}

/*
 *  ReplaceEscapes:
 *
 */

VOID ReplaceEscapes(PXSTRING pxstr)
{
    ULONG ulOfs;

    // double quote
    ulOfs = 0;
    while (xstrFindReplaceC(pxstr,
                            &ulOfs,
                            "\\q",
                            "\""))
            ;

    // carriage return
    ulOfs = 0;
    while (xstrFindReplaceC(pxstr,
                            &ulOfs,
                            "\\r",
                            "\r"))
            ;

    // newline
    ulOfs = 0;
    while (xstrFindReplaceC(pxstr,
                            &ulOfs,
                            "\\n",
                            "\n"))
            ;

    // dash
    ulOfs = 0;
    while (xstrFindReplaceC(pxstr,
                            &ulOfs,
                            "\\-",
                            "-"))
            ;

    // backslash
    ulOfs = 0;
    while (xstrFindReplaceC(pxstr,
                            &ulOfs,
                            "\\\\",
                            "\\"))
            ;

}

/*
 *  ProcessFile:
 *
 */

void ProcessFile(PSZ pszPathPrefix,
                 PSZ pszFileNameOnly)
{
    PSZ     pszContent;
    BOOL    fChanged = FALSE;
    // LONG    lExtra = strlen(G_pszReplaceWith) - strlen(G_pszSearchForReplaced);
    CHAR    szFilename[2*CCHMAXPATH];
    APIRET  arc = NO_ERROR;

    sprintf(szFilename, "%s%s", pszPathPrefix, pszFileNameOnly);

    if (!(arc = doshLoadTextFile(szFilename, &pszContent, NULL)))
    {
        ULONG   ulOfs = 0;
        XSTRING strContent;

        xstrInitSet(&strContent,
                    pszContent);

        while (TRUE)
        {
            if (xstrFindReplace(&strContent,
                                &ulOfs,
                                &G_strSearchFor,
                                &G_strReplaceWith,
                                G_ShiftTable,
                                &G_fRepeat))
            {
                if (!G_optQuietMode)
                    printf("  \"%s\": replaced string @ pos %d\n",
                           szFilename,
                           ulOfs);

                // search on after replacement string
                fChanged = TRUE;
                G_ulTotalFound++;

            }
            else
                // nuttin more found:
                break;
        }

        if ((fChanged) && (!G_optTestMode))
        {
            APIRET arc = doshWriteTextFile(szFilename, strContent.psz, NULL, FALSE);
            if (arc != 0)
            {
                printf("Error %d writing file \"%s\".\n", arc, szFilename);
                exit(1);
            }
        }
    }
    else
    {
        printf("Error reading file \"%s\".\n", szFilename);
        exit(1);
    }
}

/*
 *  main:
 *
 */

int main(int argc, char *argv[])
{
    BOOL    fProceed = FALSE;
    HDIR          hdirFindHandle = HDIR_CREATE;
    FILEFINDBUF3  FindBuffer     = {0};      /* Returned from FindFirst/Next */
    ULONG         ulResultBufLen = sizeof(FILEFINDBUF3);
    ULONG         ulFindCount    = 1;        /* Look for 1 file at a time    */
    APIRET        rc             = NO_ERROR; /* Return code                  */
    ULONG         ulTotalFiles = 0;
    PSZ           p = 0;
    CHAR          szPath[CCHMAXPATH];

    xstrInit(&G_strSearchFor, 0);
    xstrInit(&G_strReplaceWith, 0);

    // parse parameters on cmd line
    if (argc > 1) {
        SHORT i = 0;
        while (i++ < argc-1)
        {
            if (argv[i][0] == '-')
            {
                SHORT i2;
                for (i2 = 1; i2 < strlen(argv[i]); i2++)
                {
                    switch (argv[i][i2])
                    {
                        case 't':
                            G_optTestMode = TRUE;
                        break;

                        case 'q':
                            G_optQuietMode = TRUE;
                        break;

                        default:  // unknown parameter
                            Explain("Unknown parameter %s", argv[i]);
                        break;
                    }
                }
            }
            else
            {
                // no option ("-"): seems to be data
                if (G_pszFileMask == NULL)
                    G_pszFileMask = strdup(argv[i]);
                else if (G_strSearchFor.ulLength == 0)
                    xstrcpy(&G_strSearchFor, argv[i], 0);
                else if (G_strReplaceWith.ulLength == 0)
                {
                    xstrcpy(&G_strReplaceWith, argv[i], 0);
                    fProceed = TRUE;
                }
                else
                    Explain("Too many arguments.");
            }
        }
    }

    if (!fProceed)
    {
        Explain("Not enough parameters given.");
        return (1);
    }

    // extract path
    p = strrchr(G_pszFileMask, '\\');
    if (p)
    {
        strncpy(szPath, G_pszFileMask, (p - G_pszFileMask + 1));
        szPath[p - G_pszFileMask + 1] = 0;
    }
    else
        szPath[0] = 0;

    // replace escape characters (V0.9.0)
    ReplaceEscapes(&G_strSearchFor);
    ReplaceEscapes(&G_strReplaceWith);

    // find files
    rc = DosFindFirst(G_pszFileMask,        // File pattern
                      &hdirFindHandle,      // Directory search handle
                      FILE_NORMAL,          // Search attribute
                      &FindBuffer,          // Result buffer
                      ulResultBufLen,       // Result buffer length
                      &ulFindCount,         // Number of entries to find
                      FIL_STANDARD);        // Return level 1 file info

    if (rc != NO_ERROR)
        printf("Error: No files found for file mask \"%s\".\n", G_pszFileMask);
    else
    {
        // no error:
        printf("Searching for \"%s\" in \"%s\"...\n",
               G_strSearchFor.psz, G_pszFileMask);

        // process first found file
        ProcessFile(szPath, FindBuffer.achName);
        ulTotalFiles++;

        // keep finding the next file until there are no more files
        while (rc != ERROR_NO_MORE_FILES)
        {
            ulFindCount = 1;                      // Reset find count.

            rc = DosFindNext(hdirFindHandle,      // Directory handle
                             &FindBuffer,         // Result buffer
                             ulResultBufLen,      // Result buffer length
                             &ulFindCount);       // Number of entries to find

            if (rc == NO_ERROR)
            {
                ProcessFile(szPath, FindBuffer.achName);
                ulTotalFiles++;
            }
        } // endwhile

        rc = DosFindClose(hdirFindHandle);    // close our find handle
        if (rc != NO_ERROR)
            printf("DosFindClose error\n");

        printf("Files processed:        %d.\n", ulTotalFiles);
        printf("Total strings replaced: %d.\n", G_ulTotalFound);
    }

    return (0);
}

