/*
 *@@sourcefile threadbase.cpp:
 *  based on Christian Wonschina's txthread sources. This file contains a base
 *  class for using of threads in a C++ class.
 *
 *  <B>Introduction:</B>
 *
 *  A class which needs threading should inherit this class. The derived class
 *  should also implement _ThdFunction member function which is declared in
 *  ThreadBase class. The function is used inside a thread created by
 *  a ThreadBase object.
 *  The thread can be started by calling execute member function. Execute function
 *  uses _begingthread to start a thread and ThreadStarter function and this pointer
 *  are passed to _begingthread. Inside ThreadStarter _ThdFunction is called.
 *
 *
 */

/*      Copyright (C) 2000-2001 Teemu Ahola, Christian Wonschina <fonz>.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOSPROCESS
#define INCL_DOSERRORS
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#include "wipengine\threadbase.h"

//#define PRINT_DEBUG_INFO

/*
 *@@ThreadBase:
 *  is a default constructor for ThreadBase class. This method initializes
 *  stack size and other internal variables. Note that this method does not
 *  start a thread.
 */
ThreadBase::ThreadBase() {
    _ulStackSize = 16000;
    _tid = 0;
    _ulStatus = 0;
    }

/*
 *@@~ThreadBase:
 *  destructor. Checks if a thread is in suspend-mode and if so resumes it.
 *  After that the thread is simply killed.
 */
ThreadBase::~ThreadBase() {
    if(_ulStatus == TS_SUSPENDED) // The thread is in suspend-mode, resume first.
        resume();

    if(_ulStatus > TS_NOTSTARTED)
        kill();
    }

/*
 *@@setStackSize:
 *  sets a new stack size. This must be done before calling execute() method
 *  since stack size is used in _begingthread. A stack size smaller that 8kB
 *  should not be used. A default value is 16kB.
 */
VOID ThreadBase::setStackSize(ULONG p_ulNewSize) {
    _ulStackSize = p_ulNewSize;
    }

VOID ThreadBase::softKill(VOID) {
    _ulStatus = TS_NOTSTARTED;
    _tid = 0;
    _endthread();
    }

/*
 *@@kill:
 *  a brutal way to kill a thread. This method simply calls DosKillThread
 *  and updates _ulStatus.
 */
VOID ThreadBase::kill(VOID) {
    DosKillThread(_tid);
    _ulStatus = TS_NOTSTARTED;
    _tid = 0;
    }

/*
 *@@waitThread:
 *  this method returns only when a thread has completed execution or
 *  a thread is not running. <I>Note:</I> this method should be used
 *  with caution because a calling thread will be blocked until
 *  the thread completes execution.
 */
VOID ThreadBase::waitThread(VOID) {
    #ifdef PRINT_DEBUG_INFO
     printf("Check status: %i\n", _ulStatus);
     fflush(stdout);
    #endif
    if (_ulStatus != TS_NOTSTARTED) {
        #ifdef PRINT_DEBUG_INFO
         printf("Waiting thread dying.\n");
         fflush(stdout);
        #endif
        DosWaitThread(&_tid, DCWW_WAIT);
        #ifdef PRINT_DEBUG_INFO
         printf("OK. Thread has died.\n");
         fflush(stdout);
        #endif
        _ulStatus = TS_NOTSTARTED;
        _tid = 0;
        }
    #ifdef PRINT_DEBUG_INFO
    else
        printf("Skip waiting. The thread was already killed.\n");
    #endif
    }


/*
 *@@execute:
 *  starts a thread execution by calling _beginthread.
 */
TID ThreadBase::execute() {
    if (_ulStatus == TS_NOTSTARTED) {
        _tid = _beginthread(ThreadStarter, 0, _ulStackSize, (void*)this);
        _ulStatus = TS_WORKING;
        return _tid;
        }
    return 0;
    }

/*
 *@@suspend:
 *  suspends a thread execution.
 */
VOID ThreadBase::suspend() {
    DosSuspendThread(_tid);
    _ulStatus = TS_SUSPENDED;
    }

/*
 *@@resume:
 *  resumes a thread execution.
 */
VOID ThreadBase::resume() {
    DosResumeThread(_tid);
    _ulStatus = TS_WORKING;
    }

/*
 * This part is not used in the WicPM back-end engine.
 */
#ifdef PM_IS_USED
VOID ThreadBase::createMQ() {
    _habAnchor = WinInitialize(0);
    _hmqQueue = WinCreateMsgQueue(_habAnchor, 0);
    }

VOID ThreadBase::deleteMQ() {
    WinDestroyMsgQueue(_hmqQueue);
    WinTerminate(_habAnchor);
    }

VOID ThreadBase::cancelShutdown(BOOL p_bCancel) {
    WinCancelShutdown(_hmqQueue, p_bCancel);
    }
#endif
/*
 *
 */


/*
 *@@returnThreadStatus:
 *  returns the status of the thread.
 *  <B>Status:</B>
 *  TS_NOTSTARTED a thread is not running
 *  TS_SUSPENDED a thread has been suspended
 *  TS_WORKING a thread is working
 */
ULONG ThreadBase::returnThreadStatus() {
    return _ulStatus;
    }


/*
 *@@ThreadStarter:
 *  this function a newly created thread will execute. Inside this function
 *  _ThdFunction is called and it must be implemented in a derived class.
 */
VOID _Optlink ThreadStarter(VOID *p_pParam) {
    ThreadBase* thread = (ThreadBase*)p_pParam;
    thread->_ThdFunction();
    // If the thread function exits normally clear TID and status and kill thread.
    thread->softKill();
    }



