/*
 *@@sourcefile mlu.cpp:
 *  The message logging unit or MLU for the WicPM back-end engine.
 *  This unit is responsible for logging messages and message handling.
 *
 *  <B>Introduction:</B>
 *  This unit basis on a thread which has a message queue which is used
 *  for gathering messages from archive units. Also messages can be queried
 *  and deleted by using of messages.
 *
 *  <B>Using:</B>
 *  Using of the MLU is quite easy. After creation of MsgLoggingUnit it can be
 *  started using 'start' method. At this point a new thread is started and
 *  a new message queue is created. 'start' method does not return until the
 *  message queue has been created. After that the message queue handler can
 *  be acquired from public _hQReceive attribute. This handler can be used for
 *  message posting. Also a call-back function can be provided afterwards by
 *  using SetCallBackData method. Also if user specified data is required in
 *  the call-back function, it can be provided by using 'SetCallBackData' method.
 *  <B>Note:</B> The call-back function is executed in MLU thread's context.
 *
 *  <I>Example:</I>
 *      HQUEUE hQReceiver;
 *      MsgLoggingUnit msgLogUnit;
 *      msgLogUnit.SetCallBackData(CallBackFunction, (PVOID)this);
 *      msgLogUnit.start();
 *      hQReceiver = msgLogUnit._hQReceiver;
 *
 *
 *  After MLU has been initialized and required information is provided messages
 *  can be sent to it. There are four different types of messages: log, find,
 *  delete and quit. The type of message is passed to the queue by using
 *  <I>request</I> parameter of DosWriteQueue. The parameter must be QUE_MSG_LOG,
 *  QUE_MSG_FIND, QUE_MSG_DELETE or QUE_MSG_QUIT. Also find, delete and quit
 *  requires special structure called MsgInfo which must include TID, a pointer
 *  to MessageVector and event semaphore (not required) in QUE_MSG_FIND case,
 *  TID and event semaphore (not required) in QUE_MSG_DELETE case and event
 *  semaphore (not required) in QUE_MSG_QUIT case. QUE_MSG_LOG needs a pointer to
 *  an allocated MsgClass which will be deleted after it is logged to the MessageMap.
 *  Also MsgInfo must not be an automatic variable but rather an allocated memory object.
 *
 *  <I>Example:</I>
 *      MsgLoggingUnit::LogMessage(aidSender,
 *                                 bssMessageString,
 *                                 20, 0, hQReceiver);
 *
 *      MessageVector *pv = new MessageVector;
 *      MsgLoggingUnit::FindMessage(aidSender, pv,
 *                                  hQReceiver);
 *
 *  Although the thread can be killed by using a 'killer' message, the killing
 *  should let rather be done by the destructor.
 *
 */
/*      This file Copyright (C) 2000-2001 Teemu Ahola.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define INCL_DOSPROCESS
#define INCL_DOSQUEUES
#define INCL_DOSERRORS
#define INCL_DOSSEMAPHORES
#include <os2.h>
#include "setup.h"

#include "helpers\stringh.h"
#include "helpers\xstring.h"
#include "base\bs_base.h"
#include "base\bs_string.h"

#include <map>
#include <vector>
#include <list>
#include <string.h>

#include "wipengine\wipebase.h"
#include "wipengine\threadbase.h"
#include "wipengine\mlu.h"

#ifdef PRINT_DEBUG_INFO
#undef PRINT_DEBUG_INFO
#endif
// Debugging switch
//#define PRINT_DEBUG_INFO


/*
 *@@compare:
 *  returns 0 if the two MsgClasses are the same,
 *  -1 if "this" is smaller, 1 if "p_msg" is smaller.
 *  Comparing bases on _aidSender's value.
 */
int MsgClass::compare(const MsgClass &p_msg // Reference to a MsgClass object
                                            // to be compared with.
                     ) const {
    if (_aidSender == p_msg._aidSender)
        return 0;
    else if (_aidSender < p_msg._aidSender)
        return -1;
    else
        return 1;
    }


/*
 *@@mluDefaultCallBack:
 *  Default callback function is used when the user has not provided
 *  an callback function. This function simply frees the data passed
 *  with the message to the logger thread. Only if the callback type
 *  is CB_TYPE_FINDINGCOMPLETED, the memory is not freed since the
 *  user wants to get that memory.
 */
LONG mluDefaultCallBack(SHORT p_sCBType, LONG, PVOID p_pData, PVOID) {

    #ifdef PRINT_DEBUG_INFO
     printf("mluDefaultCallBack(SHORT, LONG, PVOID, PVOID):\n"
            " - parameters:\n"
            "    p_sCBType = %i\n"
            "    p_pData = 0x%X\n\n",
            p_sCBType, p_pData);
     fflush(stdout);
    #endif

    // Free the data if the message was not FIND since user wants the
    // data.
    if ((p_pData) && (p_sCBType != CB_TYPE_MLU_FINDING_COMPLETED))
        delete p_pData;

    return RET_OK;
    }


/*
 *@@LogMessage:
 *  Static interface method for message logging. This helps a user
 *  to log messages since this method takes care of memory allocation
 *  and message sending. It is not necessary to free allocated memory
 *  here since it is done automatically in the message logging unit
 *  thread. Notice that this function is asynchronous i.e. this function
 *  return immediately after sending of a message. Syncronization between
 *  the logger thread and user thread must be done using semaphore, which
 *  is provided to the thread via this function, or a call back function
 *  which is provided to the message logging object.
 *
 *  Return values: NO_ERROR
 *                 ERROR_QUE_NO_MEMORY
 *                 ERROR_QUE_INVALID_HANDLE
 */
USHORT MsgLoggingUnit::LogMessage(AID p_aidSender,
                                  BSString &p_bssMessageString,
                                  USHORT p_usMessageType,
                                  LONG p_lDetailInfo,
                                  HQUEUE p_hQMLU,
                                  HEV p_hevSemaphore,
                                  eMsgPriority p_ePriority
                                  ) {
    ULONG ulPriority = p_ePriority;
    MsgClass *pMsgClass = new MsgClass;
    MsgInfo *pMsgInfo = new MsgInfo;

    #ifdef PRINT_DEBUG_INFO
     printf("LogMessage(AID, BSString, USHORT, LONG, "
            "HQUEUE, HEV, eMsgPriority):\n"
            " - parameters:\n"
            "    p_aidSender = %i\n"
            "    p_bssMessageString = %s\n"
            "    p_usMessageType = %i\n"
            "    p_lDetailInfo = %i\n"
            "    p_hQMLU = %i\n"
            "    p_hevSemaphore = %i\n"
            "    p_ePriority = %i\n\n",
            p_aidSender, p_bssMessageString.c_str(), p_usMessageType,
            p_lDetailInfo, p_hQMLU, p_hevSemaphore, p_ePriority);
     fflush(stdout);
    #endif

    pMsgClass->_aidSender = p_aidSender;
    pMsgClass->_bssMessage = p_bssMessageString;
    pMsgClass->_usMessageType = p_usMessageType;
    pMsgClass->_lDetailInfo = p_lDetailInfo;

    pMsgInfo->_aidSender = p_aidSender;
    pMsgInfo->_pData = pMsgClass;
    pMsgInfo->_hevSem = p_hevSemaphore;

    return DosWriteQueue(p_hQMLU, QUE_MSG_LOG, sizeof(MsgInfo), (PVOID)pMsgInfo,
                         ulPriority);

    }


/*
 *@@FindMessage:
 *  Static interface method for message finding. This helps a user
 *  to find messages since this method takes care of memory allocation
 *  and message sending. It is not necessary to free allocated memory
 *  here since it is done automatically in the message logging unit
 *  thread's callback. The user must provide a pointer to a newly allocated
 *  MessageVector object where all finded messages are stored.
 *  Notice that this function is asynchronous i.e. this function
 *  return immediately after sending of a message. Syncronization between
 *  the logger thread and user thread must be done using semaphore, which
 *  is provided to the thread via this function, or a call back function
 *  which is provided to the message logging object.
 *
 *  Return values: NO_ERROR
 *                 ERROR_QUE_NO_MEMORY
 *                 ERROR_QUE_INVALID_HANDLE
 */
USHORT MsgLoggingUnit::FindMessage(AID p_aidSender,
                                   MessageVector *p_msgVector,
                                   HQUEUE p_hQMLU,
                                   HEV p_hevSemaphore,
                                   eMsgPriority p_ePriority
                                  ) {
    ULONG ulPriority = p_ePriority;
    MsgInfo *pMsgInfo = new MsgInfo;

    #ifdef PRINT_DEBUG_INFO
     printf("FindMessage(AID, MessageVector, HEV, eMsgPriority):\n"
            " - parameters:\n"
            "    p_aidSender = %i\n"
            "    p_msgVector = 0x%X\n"
            "    p_hQMLU = %i\n"
            "    p_hevSemaphore = %i\n"
            "    p_ePriority = %i\n\n",
            p_aidSender, p_msgVector, p_hQMLU, p_hevSemaphore, p_ePriority);
     fflush(stdout);
    #endif

    if (!pMsgInfo)
        return RET_ERR_PARAMETER_ERROR;

    pMsgInfo->_aidSender = p_aidSender;
    pMsgInfo->_pData = p_msgVector;
    pMsgInfo->_hevSem = p_hevSemaphore;

    return DosWriteQueue(p_hQMLU, QUE_MSG_FIND, sizeof(MsgInfo), (PVOID)pMsgInfo,
                         ulPriority);

    }

/*
 *@@DeleteMessage:
 *  Static interface method for message finding. This helps a user
 *  to find messages since this method takes care of memory allocation
 *  and message sending. It is not necessary to free allocated memory
 *  here since it is done automatically in the message logging unit
 *  thread callback. The user must provide a pointer to a newly allocated
 *  MessageVector object where all finded messages are stored.
 *  Notice that this function is asynchronous i.e. this function
 *  return immediately after sending of a message. Syncronization between
 *  the logger thread and user thread must be done using semaphore, which
 *  is provided to the thread via this function, or a call back function
 *  which is provided to the message logging object.
 *
 *  Return values: NO_ERROR
 *                 ERROR_QUE_NO_MEMORY
 *                 ERROR_QUE_INVALID_HANDLE
 */
USHORT MsgLoggingUnit::DeleteMessage(AID p_aidSender,
                                     HQUEUE p_hQMLU,
                                     HEV p_hevSemaphore,
                                     eMsgPriority p_ePriority
                                    ) {
    ULONG ulPriority = p_ePriority;
    MsgInfo *pMsgInfo = new MsgInfo;

    pMsgInfo->_aidSender = p_aidSender;
    pMsgInfo->_pData = NULL;
    pMsgInfo->_hevSem = p_hevSemaphore;

    return DosWriteQueue(p_hQMLU, QUE_MSG_DELETE, sizeof(MsgInfo), (PVOID)pMsgInfo,
                         ulPriority);

    }


/*
 *@@LogMessage:
 *  inserts a message to the multimap and uses archive id as a key.
 *  This function uses mutex semaphore to control inserting because this function
 *  is public so it can be used in other threads.
 */
USHORT MsgLoggingUnit::LogMessage(MsgClass &p_pMessageToStore,
                                    // Reference to a message to be stored.
                                 const AID p_aidSender
                                    // Archive identifier used as a key.
                                 ) {
    APIRET ulrc = 0;
    USHORT usrc = RET_OK;
    // Wait until there is no one who is working with message map by
    // requesting mutex semaphore.
    ulrc = DosRequestMutexSem(_mtxSemHMap, SEM_INDEFINITE_WAIT);
    #ifdef PRINT_DEBUG_INFO
     printf("DosRequestMutexSem in LogMessage returned: %i\n", ulrc);
     fflush(stdout);
    #endif
    _Messages.insert(pair<const AID, MsgClass>(p_aidSender, p_pMessageToStore));
    // Release requested semaphore.
    ulrc = DosReleaseMutexSem(_mtxSemHMap);
    #ifdef PRINT_DEBUG_INFO
     printf("DosReleaseMutexSem in LogMessage returned: %i\n", ulrc);
     fflush(stdout);
    #endif

    return usrc;
    }


/*
 *@@InitSemaphores:
 *  initializes map mutex semaphore and init event semaphore by
 *  using DosCreateMutexSem and DosCreateEventSem. <I>Map mutex semaphore</I>
 *  is used to coordinate _Messages multimap usage. <I>Init event semaphore</I>
 *  is used during initialization phase and it is closed soon after
 *  the initialization is completed.
 */
VOID MsgLoggingUnit::InitSemaphores(VOID) {
    APIRET ulrc = 0;
    // Create mutex semaphore for coordination of _Messages multimap
    // usage.
    ulrc = DosCreateMutexSem(NULL, &_mtxSemHMap, 0L, FALSE);
    #ifdef PRINT_DEBUG_INFO
     printf("Map mutex semaphore: %i  Return value: %i\n", _mtxSemHMap, ulrc);
     fflush(stdout);
    #endif
    // Create event semaphore for initialization syncronization.
    ulrc = DosCreateEventSem(NULL, &_hevSemInit, 0L, FALSE);
    #ifdef PRINT_DEBUG_INFO
     printf("Init event semaphore: %i  Return value: %i\n", _hevSemInit, ulrc);
     fflush(stdout);
    #endif
    }


/*
 *@@_ThdFunction:
 *  this function is an implementation for a virtual function in
 *  ThreadBase class and therefore required. This function is called
 *  from inside ThreadStarted function which is called from inside
 *  _begingthread function which actually creates a new thread. In
 *  this function an unique message queue is created and after that
 *  Start function is notified by posting event semaphore. Next
 *  the function starts to wait messages in an indefinite loop.
 *  When a message arrives to the message queue needed operations
 *  are accomplishend depending the message type. After that the function
 *  calls a call-back function if provided and posts event semaphore
 *  if provided except when the operation is 'log'. Also the memory
 *  allocated for the receivced data is deleted. But memory allocated for
 *  extra data which comes with MsgInfo is not freed here rather freed
 *  in the callback function.
 *
 *  <B>Operations:</B>
 *      <I>Log</I>: Logs a received message to the message map.
 *      <I>Find</I>: Finds all messages whose key is given and creates
 *                   a new MessageVector where the messages are copied.
 *      <I>Delete</I>: Deletes all messages whose key is given.
 *      <I>Quit</I>: This function exits which eventually kills the
 *                   thread. There is no need for additional data so
 *                   pData field can be NULL in DosWriteQueue.
 */
VOID MsgLoggingUnit::_ThdFunction() {
    AID aidSender = 0;
    APIRET ulrc = ERROR_QUE_DUPLICATE, ulSemrc = NO_ERROR;
    USHORT usIndexNbr = 0;
    CHAR szQueueName[256];
    REQUESTDATA request = {0};
    MsgClass *pMsgData = NULL;
    ULONG ulDataLength = 0;
    BYTE bPriority = 0;
    ULONG ulCommand;
    PVOID pReceivedData = NULL;
    MsgInfo *pMsgInfo = NULL;
    USHORT usCB = RET_OK;


    // Create unique queue. DosCreateQueue returns ERROR_QUE_DUPLICATED, if there is
    // already a queue with same name.
    while (ulrc == ERROR_QUE_DUPLICATE) {
        // Format an unique queue name.
        sprintf(szQueueName, "\\QUEUES\\WIPENGINE\\MLU%i.QUE", usIndexNbr);
        // Create a new queue with FIFO priority-ordering and address-converting
        // flags.
        ulrc = DosCreateQueue(&_hQReceiver, QUE_FIFO | QUE_CONVERT_ADDRESS,
                              szQueueName);
        usIndexNbr++;
        }
    #ifdef PRINT_DEBUG_INFO
     printf("Message queue created with name: %s\n", szQueueName);
     fflush(stdout);
    #endif

    // Release requested semaphore.
    ulSemrc = DosPostEventSem(_hevSemInit);
    #ifdef PRINT_DEBUG_INFO
     printf("DosPostEventSem for init in _ThdFunction returned: %i\n", ulrc);
     fflush(stdout);
    #endif

    // If error occurred, return immediately. This also completes the thread.
    if (ulrc != NO_ERROR)
        return;

    while (TRUE) {
        // Wait until there is a message in the queue created earlier.
        ulrc = DosReadQueue(_hQReceiver, &request, &ulDataLength, (PPVOID)&pReceivedData,
                            0L, DCWW_WAIT, &bPriority, 0L);
        ulCommand = request.ulData;

        #ifdef PRINT_DEBUG_INFO
         printf("\nMessage of type %i received with return code %i\n", ulCommand, ulrc);
         fflush(stdout);
        #endif
        switch (ulCommand) {
            case QUE_MSG_QUIT: {
                pMsgInfo = (MsgInfo *)pReceivedData;
                #ifdef PRINT_DEBUG_INFO
                 printf("Quit\n\n");
                 fflush(stdout);
                #endif
                // Close the queue.
                DosCloseQueue(_hQReceiver);
                _hQReceiver = NULLHANDLE;
                #ifdef PRINT_DEBUG_INFO
                 printf("Message queue closed.\n");
                 fflush(stdout);
                #endif
                // Make a call-back if a call-back function has been provided.
                if (_pfnWipeCallBack)
                    (*_pfnWipeCallBack)(CB_TYPE_MLU_QUIT_COMPLETED,
                                        RET_OK, NULL, _pUserData);
                // If there is an event semaphore, post it since a thread which
                // wrote this message may want us to notify that quitting was
                // completed.
                if (pMsgInfo->_hevSem != NULLHANDLE)
                    ulSemrc = DosPostEventSem(pMsgInfo->_hevSem);
                // Free the memory received for the message.
                delete pMsgInfo;
                pMsgInfo = NULL;
                pReceivedData = NULL;
                return;
                }
            case QUE_MSG_LOG: {
                pMsgInfo = (MsgInfo *)pReceivedData;
                pMsgData = (MsgClass *)pMsgInfo->_pData;
                #ifdef PRINT_DEBUG_INFO
                 printf("Log\n");
                 printf("AID: %i  Type: %i\n\n", pMsgData->_aidSender,
                        pMsgData->_usMessageType);
                 fflush(stdout);
                #endif
                if (pMsgData)
                    // Log the received message.
                    usCB = LogMessage(*pMsgData, pMsgData->_aidSender);
                // Make a call-back.
                if (_pfnWipeCallBack)
                    (*_pfnWipeCallBack)(CB_TYPE_MLU_LOGGING_COMPLETED, usCB, pMsgInfo->_pData,
                                        _pUserData);
                // If there is a event semaphore, post it since a thread which
                // sent this message may want us to notify.
                if (pMsgInfo->_hevSem != NULLHANDLE)
                    ulSemrc = DosPostEventSem(pMsgInfo->_hevSem);
                delete pMsgInfo;
                pMsgInfo = NULL;
                pReceivedData = NULL;
                pMsgData = NULL;
                break;
                }
            case QUE_MSG_FIND: {
                pMsgInfo = (MsgInfo *)pReceivedData;
                #ifdef PRINT_DEBUG_INFO
                 printf("Find\n");
                 fflush(stdout);
                #endif
                if (pMsgInfo) {
                    #ifdef PRINT_DEBUG_INFO
                     printf("Start finding.\n");
                     fflush(stdout);
                    #endif
                    // Find messages. A new MessageVector is created in FindMessage
                    // so there is no need to do it here.
                    usCB = FindMessages(*(MessageVector*)pMsgInfo->_pData,
                                        pMsgInfo->_aidSender);
                    // Make a call-back if a call-back function has been provided.
                    if (_pfnWipeCallBack)
                        (*_pfnWipeCallBack)(CB_TYPE_MLU_FINDING_COMPLETED,
                                            usCB, pMsgInfo->_pData, _pUserData);
                    // If there is a event semaphore, post it since a thread which
                    // wrote this message may want us to notify that find was
                    // completed.
                    if (pMsgInfo->_hevSem != NULLHANDLE)
                        ulSemrc = DosPostEventSem(pMsgInfo->_hevSem);
                    // Free the memory received for the message.
                    delete pMsgInfo;
                    pMsgInfo = NULL;
                    pReceivedData = NULL;
                    }
                break;
                } // end of case QUE_MSG_FIND
            case QUE_MSG_DELETE: {
                pMsgInfo = (MsgInfo *)pReceivedData;
                #ifdef PRINT_DEBUG_INFO
                 printf("Delete\n");
                 fflush(stdout);
                #endif
                if (pMsgInfo) {
                    #ifdef PRINT_DEBUG_INFO
                     printf("Start deletion.\n");
                     fflush(stdout);
                    #endif
                    // Delete messages which key value is _tidThread.
                    usCB = DeleteMessages(pMsgInfo->_aidSender);
                    // Make a call-back if a call-back function has been provided.
                    if (_pfnWipeCallBack)
                        (*_pfnWipeCallBack)(CB_TYPE_MLU_DELETION_COMPLETED,
                                            usCB, pMsgInfo->_pData, _pUserData);
                    // If there is a event semaphore, post it since a thread which
                    // wrote this message may want us to notify that deletion was
                    // completed.
                    if (pMsgInfo->_hevSem != NULLHANDLE)
                        ulSemrc = DosPostEventSem(pMsgInfo->_hevSem);
                    // Free the memory received for the message.
                    delete pMsgInfo;
                    pMsgInfo = NULL;
                    pReceivedData = NULL;
                    }
                break;
                } // end of case QUE_MSG_DELETE
            } // end of switch (ulCommand)
        }
    }


/*
 *@@Start:
 *  this function calls 'execute' function which was inherit from
 *  threadbase class. After that this function waits until an event
 *  semaphore is posted from '_ThdFunction' in a newly created thread.
 *  Next the event semaphore is closed since there is no need for it
 *  anymore. This function should be used when starting message loggig
 *  thread.
 *
 *  Return values: RET_ERR_THREADCREATIONFAILED
 *                 RET_OK
 */
USHORT MsgLoggingUnit::Start(VOID) {
    APIRET ulrc = 0;
    ULONG ulOwned;
    PID pid;
    TID tid;
    /*if (!_pfnWipeCallBack)
        return RET_ERR_CALLBACKNOTDEFINED;*/

    if (execute() == 0)
        return RET_ERR_MLU_THREAD_CREATION_FAILED;

    // Wait until the initialization is completed.
    ulrc = DosWaitEventSem(_hevSemInit, SEM_INDEFINITE_WAIT);
    #ifdef PRINT_DEBUG_INFO
     printf("DosWaitEventSem in Start returned: %i\n", ulrc);
     fflush(stdout);
    #endif

    // Close the initialization event semaphore since there is
    // no need for it anymore.
    ulrc = DosCloseEventSem(_hevSemInit);
    #ifdef PRINT_DEBUG_INFO
     printf("DosCloseEventSem in Start returned: %i\n", ulrc);
     fflush(stdout);
    #endif
    _hevSemInit = NULLHANDLE;

    return RET_OK;
    }


/*
 *@@FindMessages:
 *  this function finds all elements in the map that have the keys
 *  with the value p_tid. All found elements are stored in
 *  MessageVector which is created if necessary. This means if
 *  p_MsgVector is NULL a new MessageVector is created.
 *
 *  Return value: RET_ERR_PARAMETERERROR
 *                RET_ERR_MSGNOTFOUND
 *                RET_OK
 */
USHORT MsgLoggingUnit::FindMessages(MessageVector &p_msgVector,
                                    // refecence to a MessageVector
                                    AID p_aid // key value
                                    ) {
    USHORT usrc = RET_OK;
    ULONG ulrc = 0;
    MessageMap::iterator p;

    // Wait until there is no one who is working with message map by
    // requesting mutex semaphore.
    DosRequestMutexSem(_mtxSemHMap, SEM_INDEFINITE_WAIT);
    #ifdef PRINT_DEBUG_INFO
     printf("DosRequestMutexSem in FindMessages returned: %i\n", ulrc);
     fflush(stdout);
    #endif
    // Find messages which key is p_tid.
    p = _Messages.find(p_aid);

    // Check that any messages are found.
    if (p != _Messages.end()) {
        #ifdef PRINT_DEBUG_INFO
         printf("Messages was found.\n");
         fflush(stdout);
        #endif
        // Go through all found messages.
        do  {
            // Push a message to the vector.
            p_msgVector.push_back((*p).second);
            p++;
            } while(p != _Messages.upper_bound(p_aid));
        }
    else {
        #ifdef PRINT_DEBUG_INFO
         printf("No messages was found.\n");
         fflush(stdout);
        #endif
        //p_MsgVector = NULL;
        usrc = RET_ERR_MSG_NOT_FOUND;
        }
    // Release requested semaphore.
    DosReleaseMutexSem(_mtxSemHMap);
    #ifdef PRINT_DEBUG_INFO
     printf("DosReleaseMutexSem in FindMessages returned: %i\n", ulrc);
     fflush(stdout);
    #endif
    return usrc;
    }


/*
 *@@DeleteMessages:
 *  deletes all messages which key value is p_aid from the
 *  message map.
 *
 *  Return values: RET_OK
 *
 */
USHORT MsgLoggingUnit::DeleteMessages(AID p_aid // a key value
                                     ) {
    USHORT usrc = RET_OK;
    ULONG ulrc = 0;

    // Wait until there is no one who is working with message map by
    // requesting mutex semaphore.
    ulrc = DosRequestMutexSem(_mtxSemHMap, SEM_INDEFINITE_WAIT);
    #ifdef PRINT_DEBUG_INFO
     printf("DosRequestMutexSem in DeleteMessages returned: %i\n", ulrc);
     fflush(stdout);
    #endif

    #ifdef PRINT_DEBUG_INFO
     ulrc = _Messages.count(p_aid);
     printf("The count of the lements to be deleted in the map: %i\n", ulrc);
     fflush(stdout);
    #endif

    // Erase elements that have keys with the value p_tid.
    _Messages.erase(p_aid);


    #ifdef PRINT_DEBUG_INFO
     ulrc = _Messages.size();
     printf("Total count of elments in the map after deletion: %i\n", ulrc);
     fflush(stdout);
    #endif

    // Release requested semaphore.
    ulrc = DosReleaseMutexSem(_mtxSemHMap);
    #ifdef PRINT_DEBUG_INFO
     printf("DosReleaseMutexSem in DeleteMessages returned: %i\n", ulrc);
     fflush(stdout);
    #endif

    return usrc;
    }

/*
 *@@~MsgLoggingUnit:
 *  destructor. Sends 'killer' message to the thread and waits it's
 *  termination.
 */
MsgLoggingUnit::~MsgLoggingUnit() {

    MsgInfo *pMsgInfo = new MsgInfo;
    pMsgInfo->_aidSender = NULLHANDLE;
    pMsgInfo->_pData = NULL;
    pMsgInfo->_hevSem = NULLHANDLE;

    // Send killer message to the thread.
    DosWriteQueue(_hQReceiver, QUE_MSG_QUIT, 0L, (PVOID)pMsgInfo,
                  QUE_MSG_PRIORITY_NORMAL);
    // Wait until the thread has died.
    waitThread();
    }


int operator<(const MsgClass &s1, const MsgClass &s2) {
    return (s1.compare(s2) < 0);
    }

int operator==(const MsgClass &s1, const MsgClass &s2) {
    return (s1.compare(s2) == 0);
    }

int operator<=(const MsgClass &s1, const MsgClass &s2) {
    return (s1.compare(s2) <= 0);
    }

int operator!=(const MsgClass &s1, const MsgClass &s2) {
    return (s1.compare(s2) != 0);
    }

int operator>(const MsgClass &s1, const MsgClass &s2) {
    return (s1.compare(s2) > 0);
    }

int operator>=(const MsgClass &s1, const MsgClass &s2) {
    return (s1.compare(s2) >= 0);
    }




