
/*
 *@@sourcefile wic.cpp:
 *      this is the WarpIN Command Interface, "wic.exe".
 *
 *      WIC used to be a simple utility just for creating
 *      archives, but it is no longer that simple. Starting
 *      with V0.9.20, this has become a full-fledged
 *      command-line installer as well.
 */

/*
 *      This file Copyright (C) 1998-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#include <stdio.h>
#include <stdarg.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader
#include <io.h>
#include <signal.h>             // added V0.9.12 (2001-06-11) [bird]

#ifdef __IBMCPP__
    #include <direct.h>
#endif

#include "setup.h"              // added V0.9.2 (2000-03-15) [umoeller]

#include "helpers\ansiscrn.h"
#include "helpers\dosh.h"
#include "helpers\linklist.h"
#include "helpers\nls.h"
#include "helpers\standards.h"
#include "helpers\stringh.h"
#include "helpers\xstring.h"

#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"

#include "bldlevel.h"

#include "wiarchive\wiarchive.h"

#include "wic.h"

#pragma hdrstop

DEFINE_CLASS(PackageMaskSpec, BSRoot);

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

char        G_szFilestem[256] = "";  // filestem given on the cmdline
char        G_szArcFile[256] = "",
            G_szStubFile[256] = "",
            G_szScriptFile[256] = "";

extern BOOL G_fRexxAllowed = FALSE;         // V0.9.20 (2002-07-19) [umoeller]

list<PackageMaskSpec*> G_listPackageMaskSpecs(STORE);
    // list of files to work on in 'Add' mode
ULONG       G_ulPackageMaskSpecsCount = 0;
    // item count on that list

// verbosity level: 1 means "-v"
ULONG       G_ulVerbosity = 0;

/* ******************************************************************
 *
 *  Help
 *
 ********************************************************************/

void PrintAddMode()
{
    ANSI_fg_color(B_WHITE);
    printf("Add files (-a):");
    ANSI_fg_color(WHITE);
    printf("\n    wic <arc> -a {<pckno> [-r] [-c<dir>] <mask>...}... [-u<exe> | -U[+]]\n"
           "    Adds the files in <mask> to decimal <pckno> in archive <arc>.\n");
}

void PrintScriptMode()
{
    ANSI_fg_color(B_WHITE);
    printf("Set installation script (-s):");
    ANSI_fg_color(WHITE);
                   printf(" wic <arc> -s <file>\n"
           "    Sets <file> as the install script for archive <arc>.\n");
}

void PrintTestMode()
{
    ANSI_fg_color(B_WHITE);
    printf("Test installation script (-t):");
    ANSI_fg_color(WHITE);
                                  printf(" wic -t <file.wis>\n"
           "    Loads and tests the specified script file for integrity.\n");
}

void PrintListMode()
{
    ANSI_fg_color(B_WHITE);
    printf("List packages (-l):");
    ANSI_fg_color(WHITE);
                        printf(" wic <arc> -l [<pckno>]\n"
           "    Lists the files in <pckno> in <arc>; without <pckno>, a summary is shown.\n");
}

void PrintExtractFilesMode()
{
    ANSI_fg_color(B_WHITE);
    printf("Extract files (-x):");
    ANSI_fg_color(WHITE);
                        printf(" wic <arc> -x {[<pckno>] [<fmask>]}...\n"
           "    Extracts files from the given archive.\n");
}

void PrintExtractScriptMode()
{
    ANSI_fg_color(B_WHITE);
    printf("Extract script (-X):");
    ANSI_fg_color(WHITE);
                        printf(" wic <arc> -X [<filename>]\n"
           "    Extracts the archive's script into the given <filename>.\n");

}

void PrintQueryMode()
{
    ANSI_fg_color(B_WHITE);
    printf("Query database (-q):");
    ANSI_fg_color(WHITE);
                        printf(" wic <arc> -q\n"
           "    Compares the given archive against the database.\n");
}

void PrintInstallMode()
{
    ANSI_fg_color(B_WHITE);
    printf("Install archive (-i):");
    ANSI_fg_color(WHITE);
                        printf(" wic -i <cid.xcd>\n"
           "    Installs an archive using an XML CID script.\n");
}

/*
 * ExplainParams:
 *      say hello if the params aren't right.
 *      Exits with arc.
 *
 *@@changed V0.9.6 (2000-10-27) [umoeller]: added "-x", "-X" modes
 *@@changed V0.9.13 (2001-06-09)    [bird]: added "@paramfile".
 *@@changed V0.9.20 (2002-07-03) [umoeller]: added arc
 *@@changed V0.9.20 (2002-07-03) [umoeller]: reorganized, extended
 */

void ExplainParams(APIRET arc)
{
    ANSI_fg_color(B_WHITE);
    printf("wic V"BLDLEVEL_VERSION" "__DATE__" - WarpIN Command Interface\n");
    ANSI_fg_color(WHITE);
    printf("(C) 1998-2002 Jens Bckman, Ulrich Mller, Yuri Dario\n");
    printf("Published under the GNU General Public Licence (see the COPYING file).\n");
    ANSI_fg_color(B_WHITE);
    printf("Syntax: ");
    ANSI_fg_color(WHITE);
    printf("wic <arc> -<mode> [-v] <params> [@paramfile]\n"
           "'-v' switches to 'verbose' mode with more output for testing.\n"
           "The different <mode>'s are (type \"wic -h<mode>\" for details):\n");

    PrintAddMode();

    PrintScriptMode();

    PrintTestMode();

    PrintListMode();

    PrintExtractFilesMode();

    PrintExtractScriptMode();

    PrintQueryMode();

    PrintInstallMode();

    exit(arc);
}

/*
 *@@ ExplainMore:
 *      called upon "wic -ha" for explaining "add mode" further.
 *      Exits with arc.
 *
 *@@added V0.9.1 (2000-02-05) [umoeller]
 *@@changed V0.9.20 (2002-07-03) [umoeller]: added arc
 */

VOID ExplainMore(CHAR cMode,     // can be 'h'
                 APIRET arc)
{
    switch (cMode)
    {
        case 'a':
            PrintAddMode();
            printf("    '-r' will recurse into subdirectories for that one <pckno> only.\n"
                   "    '-c<dir>' will change to <dir> while collecting files for <pckno>.\n"
                   "    You can specify several packages and/or several filemasks.\n"
                   "    '-u<exe>' will use <exe> as stub for self-extracting archive.\n"
                   "    '-U' will create a standard self-installing executable using stub.exe.\n"
                   "    If '+' is not specified, this will add the base WarpIN files to package\n"
                   "    30000. If '+' is specified also, this will also add the extra WarpIN files\n"
                   "    to package 30001 for a full WarpIN self-install.\n"
                   "    In both cases, the WarpIN files to be packed are assumed to be in the same\n"
                   "    directory as the WIC.EXE that was started.\n"
                   );

            ANSI_fg_color(B_WHITE);
            printf("Examples for archive mode (type \"wic -h\" to get the main help):\n");
            ANSI_fg_color(WHITE);
            printf(
                   "    wic test -a 1 *.exe\n"
                   "        adds all *.exe files in the current directory to package 1.\n"
                   "    wic test -a 1 -r -csubdir1 * 2 -r *.inf *.hlp\n"
                   "        changes to subdirectory 'subdir1' first and adds all files in that\n"
                   "        directory and subdirectories to package 1. After that, all *.inf\n"
                   "        and *.hlp files in the current (previous) directory and its\n"
                   "        subdirectories are added to package 2.\n"
                   "    wic test -a 1 -r -csubdir1 * 2 -r *.inf *.hlp -s script.wis\n"
                   "        as before, but sets script.wis as the archive script at the same time.\n"
                  );
        break;

        case 's':
            PrintScriptMode();
            printf("    'Script' mode is the only mode that can be combined with 'add' mode, e.g.:\n"
                   "        wic test.wpi -s test.wis -a 1 *.cpp *.h 2 *.exe\n");
        break;

        case 't':
            PrintTestMode();
            printf("    'Test' mode saves you from having to build a whole archive just to find\n"
                   "    that there are syntax errors. This will run the WarpIN script parser\n"
                   "    over the script, which should detect all syntax errors. However, this\n"
                   "    cannot detect wrong package indices in the script, since those will have\n"
                   "    to be compared to a binary archive.\n");
        break;

        case 'l':
            PrintListMode();
        break;

        case 'x':
            PrintExtractFilesMode();
            printf("    If <pckno> is not specified, files from all packages in the archive will\n"
                   "    be extracted.\n"
                   "    <fmask> is an optional OS/2-style filemask. For example, you can specify\n"
                   "    \"*.txt\" to extract all files ending in .txt. If not specified, \"*\" is\n"
                   "    assumed to extract all files.\n"
                   "    You can specify more than one package and more than one filemask. The\n"
                   "    extracted files go into the current directory; subdirectories will always\n"
                   "    be created if specified in the archive.\n"
                  );
        break;

        case 'X':
            PrintExtractScriptMode();
            printf("    If <filename> is not specified, the archive's fully qualified filename\n"
                   "    with a .wis extension will be used.\n"
                  );
        break;

        case 'q':
            PrintQueryMode();
            printf("    This compares each package in the archive against the database and prints\n"
                   "    out whether it is already installed and, if so, compares the versions.\n"
                  );
        break;

        case 'i':
            PrintInstallMode();
            printf("    This performs a full install of an archive, using the data specified in\n"
                   "    the given XML response file. This script can be generated by the PM\n"
                   "    WarpIN installer by using the 'Archive' menu item.\n"
                  );
        break;

        default:
            ExplainParams(arc);
    }

    throw WICExcpt(arc);
}

/* ******************************************************************
 *
 *  Miscellaneous
 *
 ********************************************************************/

/*
 *@@ Error:
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

VOID Error(APIRET arc,          // out: error code to exit with
           PCSZ pcszFormat,
           ...)
{
    va_list     args;

    printf("wic: ");

    va_start(args, pcszFormat);
    vprintf(pcszFormat, args);
    va_end(args);

    printf("\n");

    if (arc == CID_INCORRECT_INVOCATION)
        printf("Type \"wic -h\" for help.\n");

    throw WICExcpt(arc);
}

/*
 * PrintFilesCallback:
 *      callback for WIArchive::for_all_files
 *      when enumerating files in "list mode".
 *
 *@@changed V0.9.6 (2000-11-23) [umoeller]: fixed divide-by-0 with 0-byte files
 */

int XWPENTRY PrintFilesCallback(WIFileHeader* file,
                                unsigned long ulNotUsedHere)
{
    CHAR    szOriginal[20],
            szCompressed[20];

    printf(" %-40s %10s %10s   %3d%%\n",
                file->name,
                nlsThousandsULong(szOriginal,
                                  file->origsize,
                                  '.'),
                nlsThousandsULong(szCompressed,
                                  file->compsize,
                                  '.'),
                (file->origsize)    // V0.9.6 (2000-11-23) [umoeller]
                    ? 100 - ( (file->compsize * 100) / file->origsize)
                    : 0);

    return (0);
}

/* ******************************************************************
 *
 *  "Add" Mode
 *
 ********************************************************************/

/*
 * WICallbackAddMode:
 *      this is the general callback from the WIArchive class.
 *      Gets called while we're packing the files for progress
 *      display.
 *
 *@@changed V0.9.6 (2000-11-23) [umoeller]: fixed divide-by-0 with 0-byte files
 */

int WICallbackAddMode(enCallbackMode mode,
                      short s,
                      WIFileHeader *pwifh,
                      void *pvUser)
{
    static char     szLastFile[300] = "";
    static int      iLastPackage = -1;

    if (mode == CBM_PERCENTAGE)
    {
        if (iLastPackage != pwifh->package)
        {
            printf("Now going for package %d...\n", pwifh->package);
            iLastPackage = pwifh->package;
        }
        // s has the percentage
        if (strcmp(szLastFile, pwifh->name) != 0)
            // new file:
            strcpy(szLastFile, pwifh->name);
        else
            // go back in same line
            printf ("\r");
        printf ("Compressing %s (%03d%%)", pwifh->name, s);
        fflush(stdout);
    }
    else if (mode == CBM_UPDATING)
    {
        // ANSI: go right 60 chars
        printf("\r%c[%dC",
                    27,     // ESC
                    60);
        printf("%02d%% compression \n",
                (pwifh->origsize)   // V0.9.6 (2000-11-23) [umoeller]: avoid divide-by-0
                    ? 100 - ((pwifh->compsize * 100) / pwifh->origsize)
                    : 0);
        fflush(stdout);
    }
    else
    {
        printf("\nwic error: backend reported error %d. Terminating.\n", mode);
    }

    return (CBRC_ABORT);
}

/*
 *@@ CreateSelfMask:
 *      creates an automatic package mask spec for
 *      WarpIN self-install (-U and -U+ modes).
 *
 *@@added V0.9.14 (2001-08-23) [umoeller]
 */

APIRET CreateSelfMask(string &strSourcePath,        // in: directory of WIC.EXE
                      LONG lPckNo,                  // in: pck no to create (e.g. 30000)
                      const char **papcszFiles,     // in: array of file names to add
                      ULONG cFiles)                 // in: array item count
{
    PackageMaskSpec *pPackageMaskSpecThis = new(PackageMaskSpec);
    pPackageMaskSpecThis->lPackage = lPckNo;
    pPackageMaskSpecThis->strSubdir = strSourcePath;

    printf("Creating self-install package index %d...\n", lPckNo);

    ULONG ul;
    for (ul = 0;
         ul < cFiles;
         ul++)
    {
        string *pThis = new string(papcszFiles[ul]);
        pPackageMaskSpecThis->listFileMasks.push_back(pThis);
        pPackageMaskSpecThis->cFileMasks++;
    }

    G_listPackageMaskSpecs.push_back(pPackageMaskSpecThis);
    G_ulPackageMaskSpecsCount++;

    return NO_ERROR;
}

/*
 *@@ AddFileMaskToPackage:
 *      this actually processes one file mask, as specified
 *      in PackageMaskSpec, and calls WIArchive.add for each file which was
 *      found. This changes to subdirectories and eventually
 *      recurses, if desired.
 *
 *      Gets called by AddMode().
 *
 *      Throws:
 *      -- WICExcpt.
 *
 *@@added V0.9.0 (2000-01-03) [umoeller]
 *@@changed V0.9.1 (2000-02-07) [umoeller]: added "-c" subdir support
 *@@changed V0.9.9 (2001-04-06) [umoeller]: fixed root dir
 *@@changed V0.9.12 (2001-06-11) [umoeller]: various adjustments for error codes
 */

APIRET AddFileMaskToPackage(WIArchiveRW *pArc,        // in: archive to append files to
                            PackageMaskSpec *pPackageMaskSpecThis, // in: spec what to append
                            const char *pszFileMaskThis,    // in: current file mask
                            ULONG ulRecursionLevel) // initially 0
{
    APIRET arc = NO_ERROR;

    CHAR    szCurDir2Collect[2*CCHMAXPATH] = "";

    _getcwd(szCurDir2Collect, sizeof(szCurDir2Collect));

    // indent
    ULONG ul0;
    for (ul0 = 0; ul0 < ulRecursionLevel; ul0++)
        printf("  ");
    printf("  adding \"%s\" to package %d",
           pszFileMaskThis,
           pPackageMaskSpecThis->lPackage);
    if (pPackageMaskSpecThis->fRecurse)
        printf(" recursively");
    fflush(stdout);

    // DosFindFirst returns the file names only, so if the
    // user specified relative paths on the command line,
    // we'll need to manually add those again when adding
    // files to the archive.
    CHAR    szDrive[_MAX_DRIVE],        // "C:"
            szDir[_MAX_DIR],            // "\test\"
            szFilestem[_MAX_FNAME],     // "*"
            szExt[_MAX_EXT];            // ".obj"
    _splitpath((PSZ)pszFileMaskThis, szDrive, szDir, szFilestem, szExt);

    // now check if this is valid
    if (szDrive[0] != 0)
    {
        // drive specified: no!
        Error(CID_INCORRECT_INVOCATION,
              "Error in file mask \"%s\".\n"
              "You cannot store absolute drive specifications in an archive because this would\n"
              "require the user to have the same drive letters.\n",
              pszFileMaskThis);
    }
    // check if first part of the path is a backslash; that's absolute too
    if (szDir[0] == '\\')
    {
        Error(CID_INCORRECT_INVOCATION,
              "Error in file mask \"%s\".\n"
              "You cannot store absolute path specifications in an archive because this would\n"
              "prevent the user from installing to any directory.\n"
              "If you specify paths in file masks, use relative specifications only.\n",
              pszFileMaskThis);
    }

    HDIR          hdirFindHandle = HDIR_CREATE;
    FILEFINDBUF3  ffb3           = {0};      // Returned from FindFirst/Next
    ULONG         ulResultBufLen = sizeof(FILEFINDBUF3);
    ULONG         ulFindCount    = 1;        // Look for 1 file at a time

    BOOL          fAnythingFound = FALSE;

    // 1) find files

    arc = DosFindFirst(pszFileMaskThis,      // PSZ file pattern
                       &hdirFindHandle,      // directory search handle
                       FILE_NORMAL,          // search attribute
                       &ffb3,                // result buffer
                       ulResultBufLen,       // result buffer length
                       &ulFindCount,         // number of entries to find
                       FIL_STANDARD);        // return level 1 file info

    if (arc == NO_ERROR)
    {
        // keep finding the next file until there are no more files
        while (arc != ERROR_NO_MORE_FILES)
        {
            // PSZ pszFileNameInArc;
            if (    // skip directory entries
                    (strcmp(ffb3.achName, ".") != 0)
                 && (strcmp(ffb3.achName, "..") != 0)
                    // skip the archive file being created
                 && (strcmp(ffb3.achName, G_szArcFile) != 0)
               )
            {
                CHAR    szNameInArchive[2*CCHMAXPATH],
                        szNameFullPath[3*CCHMAXPATH];
                sprintf(szNameInArchive, "%s%s",
                        szDir,
                        ffb3.achName);

                // fixed root dir V0.9.9 (2001-04-06) [umoeller]
                strcpy(szNameFullPath, szCurDir2Collect);
                if (szNameFullPath[strlen(szNameFullPath) - 1] != '\\')
                    strcat(szNameFullPath, "\\");
                strcat(szNameFullPath, szNameInArchive);

                if (G_ulVerbosity)
                {
                    // indent
                    printf("\n");
                    for (ul0 = 0; ul0 < ulRecursionLevel; ul0++)
                        printf("  ");
                    printf("    adding \"%s\" -> \"%s\"",
                           szNameFullPath,
                           szNameInArchive);
                }
                else
                    printf(".");
                fflush(stdout);

                pArc->add(szNameFullPath,       // real file name
                          szNameInArchive,      // file name in archive
                          pPackageMaskSpecThis->lPackage); // package no.
                fAnythingFound = TRUE;                                                              //bird
            }

            ulFindCount = 1;                    // reset find count
            arc = DosFindNext(hdirFindHandle,   // directory handle
                             &ffb3,             // result buffer
                             ulResultBufLen,    // result buffer size
                             &ulFindCount);     // number of entries to find

        } // endwhile

        arc = DosFindClose(hdirFindHandle);     // close our find handle
    }

    printf("\n");

    // 2) recurse for subdirectories

    if (pPackageMaskSpecThis->fRecurse)
    {
        hdirFindHandle = HDIR_CREATE;
        ulResultBufLen = sizeof(FILEFINDBUF3);
        ulFindCount    = 1;        // look for 1 file at a time

        CHAR    szDirsMask[CCHMAXPATH];
        sprintf(szDirsMask,
                "%s%s*",
                szDrive, szDir);

        // recursion enabled:
        // search again, for subdirs this time
        arc = DosFindFirst(szDirsMask,          // file pattern
                           &hdirFindHandle,     // directory search handle
                           MUST_HAVE_DIRECTORY, // search attribute
                           &ffb3,               // result buffer
                           ulResultBufLen,      // result buffer length
                           &ulFindCount,        // number of entries to find
                           FIL_STANDARD);       // return level 1 file info

        if (arc == NO_ERROR)
        {
            fAnythingFound = TRUE;

            // keep finding the next file until there are no more files
            while (arc != ERROR_NO_MORE_FILES)
            {
                if (    // skip directory entries
                        (strcmp(ffb3.achName, ".") != 0)
                     && (strcmp(ffb3.achName, "..") != 0)
                   )
                {
                    // valid subdirectory found:
                    CHAR szRecursionFileMask[CCHMAXPATH];
                    sprintf(szRecursionFileMask,
                            "%s%s%s\\%s%s",
                            szDrive, szDir, ffb3.achName, szFilestem, szExt);

                    // RECURSE!!
                    arc = AddFileMaskToPackage(pArc,
                                               pPackageMaskSpecThis,
                                               szRecursionFileMask,
                                               ulRecursionLevel + 1);
                }

                ulFindCount = 1;                    // reset find count
                arc = DosFindNext(hdirFindHandle,   // directory handle
                                 &ffb3,             // result buffer
                                 ulResultBufLen,    // result buffer size
                                 &ulFindCount);     // number of entries to find
            } // endwhile

            arc = DosFindClose(hdirFindHandle);     // close our find handle
        }
    }

    if (!fAnythingFound)
    {
        printf("\nwic error: No files for search mask \"%s\" found.\n",
                pszFileMaskThis);
        if (arc)                                                                                    //bird
            printf("OS/2 reported SYS%04d: \"%s\"\n", arc, doshQuerySysErrorMsg(arc));
        printf("wic warning: archive \"%s\" is invalid now.\n", G_szArcFile);
        throw WICExcpt(ERROR_FILE_NOT_FOUND);                                                  //bird
    }

    return (arc);
}

/*
 *@@ AddMode:
 *      this gets called from main() if "-a" was
 *      specified ("add" mode). At this point,
 *      listPackageMaskSpecs has a list of specifications
 *      to add.
 *
 *      This function is responsible for adding all
 *      files to the archive and setting the script.
 *
 *      Throws:
 *      -- WICExcpt.
 *
 *@@changed V0.9.13 (2001-06-04) [umoeller]: changed header field values
 *@@changed V0.9.14 (2001-08-23) [umoeller]: this changed to the subdirs for each package mask, fixed
 *@@changed V0.9.14 (2001-08-23) [umoeller]: disabled updating archives, since back-end is still buggy
 *@@changed V0.9.19 (2002-07-01) [umoeller]: changed header field values
 */

VOID AddMode(VOID)
{
    WIArchiveRW   Arc;
    WIArcHeader ArcHeader;

    if (!access(G_szArcFile, 0))
    {
        // V0.9.14 (2001-08-23) [umoeller]: disabled updating archives,
        // since the back-end is still buggy
        Error(CID_RESOURCEINUSE,
              "Archive \"%s\" exists already.\nUpdating archives presently doesn't work.\n",
              G_szArcFile);
    }

    if (Arc.open(G_szArcFile, 1))
    {
        Error(CID_IOERROR,
              "Error reading/creating archive \"%s\".\n",
              G_szArcFile);
    }

    if (G_szStubFile[0])
    {
        if (access(G_szStubFile, 0))
        {
            // stub file doesn't exist:
            Error(CID_RESOURCENOTFOUND,
                  "Cannot find stub file \"%s\".\n");
        }

        printf("Setting stub file to \"%s\"\n", G_szStubFile);
        Arc.setStubFile(G_szStubFile);
    }
    else
        printf("No stub set, creating plain WPI.\n");

    // replaced this V0.9.19 (2002-07-01) [umoeller]
    strcpy(ArcHeader.szAppName, "WIC " BLDLEVEL_VERSION);   // V0.9.19 (2002-07-01) [umoeller]
    strcpy(ArcHeader.szDeveloper, "OS/2 Netlabs");
    strcpy(ArcHeader.szInternetAddr, "http://warpin.netlabs.org");
    ArcHeader.sAppRevision = 1;
    Arc.setArcHeader(ArcHeader);

    APIRET arc = NO_ERROR;

    if (G_ulPackageMaskSpecsCount)
    {
        printf("Collecting files:\n");
        fflush(stdout);

        // go thru the packages which were specified on the command line;
        // each of these PackageMaskSpec structures has in turn a list
        // of file masks to be added.
        // This list might be empty if only a script is to be added.
        list<PackageMaskSpec*>::iterator PckStart = G_listPackageMaskSpecs.begin(),
                                         PckEnd = G_listPackageMaskSpecs.end();
        for (; PckStart != PckEnd; PckStart++)
        {
            PackageMaskSpec* pPackageMaskSpecThis = *PckStart;

            CHAR    szCurDirSaved[2*CCHMAXPATH] = "",
                    szPackageName[20];

            const char *pcszSubdir = NULL;

            sprintf(szPackageName, "Pck%03d", pPackageMaskSpecThis->lPackage);
            Arc.setPackage(pPackageMaskSpecThis->lPackage, szPackageName);

            // change to subdir?
            if (pPackageMaskSpecThis->strSubdir())
            {
                // yes:
                pcszSubdir = pPackageMaskSpecThis->strSubdir.c_str();
                printf("  changing to subdir \"%s\"...", pcszSubdir);
                // get current directory (for switching back later)
                _getcwd(szCurDirSaved, sizeof(szCurDirSaved));

                if (arc = doshSetCurrentDir(pcszSubdir)) // V0.9.12 (2001-06-11) [umoeller]
                {
                    // error:
                    Error(CID_RESOURCENOTFOUND,
                          "Unable to change to subdir \"%s\".\nCurrent directory is: \"%s\".\n",
                           pcszSubdir,
                           szCurDirSaved);
                }
                else
                    printf(" OK.\n");
            }

            // go thru list of file masks for this package
            list<string*>::iterator FileMaskStart = pPackageMaskSpecThis->listFileMasks.begin(),
                                FileMaskEnd = pPackageMaskSpecThis->listFileMasks.end();

            for (; FileMaskStart != FileMaskEnd; FileMaskStart++)
            {
                string *pszFileMaskThis = *FileMaskStart;

                if (arc = AddFileMaskToPackage(&Arc,
                                               pPackageMaskSpecThis,
                                           pszFileMaskThis->c_str(),
                                           0))
                    break;
            } // end for FileMaskStart

            // did we change to a subdir?
            if (pcszSubdir)
            {
                if (arc = doshSetCurrentDir(szCurDirSaved))
                {
                    // error:
                    Error(CID_RESOURCENOTFOUND,
                          "Unable to change back to dir \"%s\".",
                          szCurDirSaved);
                }
            }
        } // end for PckStart...
    } // end if (ulPackageMaskSpecsCount)

    if (!arc)
        if (G_szScriptFile[0] != 0)
        {
            PSZ         pszScriptBuf = NULL;

            if ((arc = doshLoadTextFile(G_szScriptFile, &pszScriptBuf, NULL)))
            {
                Error(CID_RESOURCENOTFOUND,
                      "Script file \"%s\" not found.\n",
                      G_szScriptFile);
            }
            printf("Script \"%s\" successfully read, %d bytes.\n",
                    G_szScriptFile,
                    strlen(pszScriptBuf));
            printf("Writing script to archive... ");
            fflush(stdout);
            Arc.setScript(pszScriptBuf);
            printf("OK\n");
        }

    Arc.setCallbackFunc(WICallbackAddMode, NULL);

    Arc.close();

    printf("Done.\n");
}

/* ******************************************************************
 *
 *  "List" Mode
 *
 ********************************************************************/

/*
 *@@ ListMode:
 *
 *      Throws:
 *      -- WICExcpt.
 *
 *@@changed V0.9.6 (2000-11-23) [umoeller]: fixed divide-by-0 with 0-byte files
 */

void ListMode(LONG lListPackage)
{
    WIArchiveRW   Arc;

    CHAR szOriginal[30],
         szCompressed[30];

    if (Arc.open(G_szArcFile, 0))
    {
        Error(CID_RESOURCENOTFOUND,
              "Unable to open \"%s\" for reading.\n",
              G_szArcFile);
    }

    if (lListPackage == 0)
    {
        // "list all packages" mode:
        ULONG ulPckCount = 0,
              ulTotalFiles = 0,
              ulTotalCompressed = 0,
              ulTotalOriginal = 0;

        const WIArcHeader *pHeader = Arc.getArcHeader();

        printf("Archive: \"%s\"\n", G_szArcFile);
        printf("WPI revision: %d\n", pHeader->wi_revision_needed);
        printf("Stub size: %d bytes\n", Arc.getStubSize());

        list<WIPackHeaderLI*> *pPackageList = Arc.getPackList();
        list<WIPackHeaderLI*>::iterator pckStart = pPackageList->begin(),
                                        pckEnd = pPackageList->end();
        for (;
             pckStart != pckEnd;
             pckStart++)
        {
            WIPackHeader* pHeaderThis = (**pckStart)._p;

            printf("Package %d\n", pHeaderThis->number);
            printf("  %4d files, size: %10s bytes original\n"
                   "                    %10s bytes compressed (%d%% compression)\n",
                    pHeaderThis->files,
                    nlsThousandsULong(szOriginal,
                                      pHeaderThis->origsize,
                                      '.'),
                    nlsThousandsULong(szCompressed,
                                      pHeaderThis->compsize,
                                      '.'),
                    (pHeaderThis->origsize) // V0.9.6 (2000-11-23) [umoeller]
                        ? 100 - ( (pHeaderThis->compsize * 100 ) / pHeaderThis->origsize)
                        : 0
                );

            ulTotalFiles += pHeaderThis->files;
            ulTotalCompressed += pHeaderThis->compsize;
            ulTotalOriginal += pHeaderThis->origsize;
            ulPckCount++;
        }
        if (ulTotalFiles == 0)
            // otherwise we'd get a division by zero below
            printf("No packages found.\n");
        else
            printf("Summary: %d package(s), %d files total\n"
                   "    %10s bytes original\n"
                   "    %10s bytes compressed (%d%% compression)\n",
                    ulPckCount,
                    ulTotalFiles,
                    nlsThousandsULong(szOriginal,
                                      ulTotalOriginal,
                                      '.'),
                    nlsThousandsULong(szCompressed,
                                      ulTotalCompressed,
                                      '.'),
                    (ulTotalOriginal)   // V0.9.6 (2000-11-23) [umoeller]
                        ? 100 - ( (ulTotalCompressed * 100 ) / ulTotalOriginal )
                        : 0
                );
    }
    else
    {
        // list one package only:
        // enumerate through package headers
        BOOL fFound = FALSE;
        list<WIPackHeaderLI*> *pPackageList = Arc.getPackList();
        list<WIPackHeaderLI*>::iterator pckStart = pPackageList->begin(),
                                        pckEnd = pPackageList->end();
        WIPackHeader* pPckThis = NULL;

        for (; pckStart != pckEnd; pckStart++)
        {
            pPckThis = (**pckStart)._p;
            if (pPckThis->number == lListPackage)
            {
                fFound = TRUE;
                break;
            }
        }

        if (fFound)
        {
            // pPckThis has found package
            printf("Listing files in package %d\n", lListPackage);
            printf(" %-40s %10s %10s   compression\n",
                        "filename",
                        "origsize",
                        "compsize");
            printf(" ----------------------------------------------------------------------------\n");
            Arc.forAllFiles(pPckThis, PrintFilesCallback, 0);

            printf("Package %d summary:\n", pPckThis->number);
            printf("  %4d files, size: %10s bytes original\n"
                   "                    %10s bytes compressed (%d%% compression)\n",
                    pPckThis->files,
                    nlsThousandsULong(szOriginal,
                                      pPckThis->origsize,
                                      '.'),
                    nlsThousandsULong(szCompressed,
                                      pPckThis->compsize,
                                      '.'),
                    (pPckThis->origsize)    // V0.9.6 (2000-11-23) [umoeller]
                        ? 100 - ( (pPckThis->compsize * 100 ) / pPckThis->origsize )
                        : 0
                );
        }
        else
        {
            Error(CID_RESOURCENOTFOUND,
                  "Package %d does not exist in archive \"%s\".\n",
                  lListPackage,
                  G_szArcFile);
        }
    }
    Arc.close();
}

/* ******************************************************************
 *
 *  "Extract" Mode
 *
 ********************************************************************/

ULONG   G_ulCurrentFile = 0;

/*
 * WICallbackExtractMode:
 *      this is the general callback from the WIArchive class.
 *      Gets called while we're extracting the files for progress
 *      display.
 */

int WICallbackExtractMode(enCallbackMode mode,
                          short s,
                          WIFileHeader *pwifh,
                          void *pvUser)
{
    int rc = CBRC_ABORT;

    PackageMaskSpec* pPackageMaskSpecThis = (PackageMaskSpec*)pvUser;

    static char szLastFile[CCHMAXPATH] = "";

    switch (mode)
    {
        case CBM_NEXTFILE:
            if (szLastFile[0])
                // not first file:
                printf("\r  %s (100%%)\n", szLastFile);

            // ### check for whether file exists

            rc = CBRC_PROCEED;

            if (pwifh)
            {
                strcpy(szLastFile, pwifh->name);
                G_ulCurrentFile++;

                if (    pPackageMaskSpecThis
                     && pPackageMaskSpecThis->cFileMasks
                   )
                {
                    // we have file mask:
                    // apply filters
                    rc = CBRC_SKIP;

                    list<string*>::iterator MaskStart = pPackageMaskSpecThis->listFileMasks.begin(),
                                        MaskEnd = pPackageMaskSpecThis->listFileMasks.end();
                    for (; MaskStart != MaskEnd; MaskStart++)
                    {
                        string *pstrMask = *MaskStart;
                        if (doshMatch(pstrMask->c_str(),
                                      pwifh->name))
                        {
                            rc = CBRC_PROCEED;
                            break;
                        }
                    }

                    if (rc != CBRC_PROCEED)
                    {
                        printf("  skipping %s\n", pwifh->name);
                        szLastFile[0] = '\0';
                    }
                }
            }

        break;

        case CBM_PERCENTAGE:
            printf("\r  %s (%03d%%)", pwifh->name, (ULONG)s);
            fflush(stdout);
            rc = CBRC_PROCEED;
        break;

        default:
            printf("Error.\n");
        break;
    }

    return (rc);
}

/*
 *@@ ExtractMode:
 *
 *      Throws:
 *      -- WICExcpt.
 *
 *@@added V0.9.6 (2000-10-27) [umoeller]
 *@@changed V0.9.14 (2001-08-23) [umoeller]: fixed broken file masks and other things
 */

void ExtractMode(VOID)
{
    WIArchiveRW   Arc;

    if (Arc.open(G_szArcFile, 0))
    {
        Error(CID_RESOURCENOTFOUND,
              "Unable to open \"%s\" for reading.\n",
              G_szArcFile);
    }

    Arc.setCallbackFunc(WICallbackExtractMode, NULL);

    if (!G_ulPackageMaskSpecsCount)
    {
        // no packages specified: extract all packages,
        // so create package specs for all packages in the archive first
        // for the loop below
        ULONG ulPckCount = 0;
        list<WIPackHeaderLI*> *pPackageList = Arc.getPackList();
        list<WIPackHeaderLI*>::iterator pckStart = pPackageList->begin(),
                                        pckEnd = pPackageList->end();
        for (; pckStart != pckEnd; pckStart++)
        {
            WIPackHeader* pHeaderThis = (**pckStart)._p;

            PackageMaskSpec* pPackageMaskSpecThis = new(PackageMaskSpec);
            pPackageMaskSpecThis->lPackage = pHeaderThis->number;
            G_listPackageMaskSpecs.push_back(pPackageMaskSpecThis);
            G_ulPackageMaskSpecsCount++;

            ulPckCount++;
        }
    }

    // now go thru the packages
    list<PackageMaskSpec*>::iterator PckStart = G_listPackageMaskSpecs.begin(),
                                     PckEnd = G_listPackageMaskSpecs.end();
    for (; PckStart != PckEnd; PckStart++)
    {
        PackageMaskSpec* pPackageMaskSpecThis = *PckStart;

        printf("Extracting package %d...\n", pPackageMaskSpecThis->lPackage);

        Arc.setCallbackFunc(WICallbackExtractMode,
                            pPackageMaskSpecThis);

        WICallbackExtractMode(CBM_NEXTFILE,
                              0,
                              NULL,
                              pPackageMaskSpecThis);

        int rc = Arc.unpack(pPackageMaskSpecThis->lPackage);
        switch (rc) // V0.9.14 (2001-07-14) [umoeller]
        {
            case WIERR_INVALID_INDEX:
                Error(CID_RESOURCENOTFOUND,
                      "Package %d does not exist in archive \"%s\".\n",
                      pPackageMaskSpecThis->lPackage,
                      G_szArcFile);
            break;

            case 0:
            break;

            default:
                Error(CID_UNEXPECTED_CONDITION,
                      "Compression backend returned error %d.\n",
                      rc);

        }

        if (rc)
            break;

    } // end for PckStart...

    Arc.close();
}

/* ******************************************************************
 *
 *  "Extract Script" Mode
 *
 ********************************************************************/

/*
 *@@ ExtractScriptMode:
 *
 *      Throws:
 *      -- WICExcpt.
 *
 *@@added V0.9.6 (2000-10-27) [umoeller]
 */

void ExtractScriptMode(VOID)
{
    WIArchiveRW   Arc;

    if (G_szScriptFile[0] == 0)
    {
        // no script filename given:
        // use archive file with different extension
        CHAR    szDrive[_MAX_DRIVE],        // "C:"
                szDir[_MAX_DIR],            // "\test\"
                szFilestem[_MAX_FNAME],     // "*"
                szExt[_MAX_EXT];            // ".wis"
        _splitpath(G_szArcFile, szDrive, szDir, szFilestem, szExt);
        strcpy(szExt, ".wis");
        // strip directory information of source .WPI file;
        // write output to current directory
        sprintf(G_szScriptFile, "%s%s", szFilestem, szExt);
    }

    printf("wic: Writing script to text file \"%s\"... ", G_szScriptFile);
    fflush(stdout);

    if (Arc.open(G_szArcFile, 0))
    {
        Error(CID_RESOURCENOTFOUND,
              "Unable to open \"%s\" for reading.\n",
              G_szArcFile);
    }

    // get the installation profile from the archive
    const char *pcszProfileTemp = Arc.getScript();
    if (pcszProfileTemp == NULL)
    {
        Error(CID_RESOURCENOTFOUND,
              "Cannot read script in archive \"%s\".\n",
              G_szArcFile);
    }

    // OK, we got the script.... open a text file for writing
    ULONG ulWritten = 0;
    APIRET arc = doshWriteTextFile(G_szScriptFile,
                                   pcszProfileTemp,
                                   &ulWritten,
                                   NULL);          // no backup
    if (arc != NO_ERROR)
    {
        Error(CID_IOERROR,
              "Error %d occured writing script to target file \"%s\".\n",
              arc,
              G_szScriptFile);
    }
    else
        printf("Done, wrote %d bytes.\n", ulWritten);

    Arc.close();
}

/* ******************************************************************
 *
 *  SigHandler()
 *
 ********************************************************************/

/*
 *@@ SigHandler:
 *      signalhandler which exits the application
 *      with an non-zero exit code.
 *
 *@@added V0.9.13 (2001-06-09) [bird]
 */

void SigHandler(int sig)
{
    flushall();
    switch (sig)
    {
        case SIGBREAK:
            printf("\nSIGBREAK\n");
            exit(-1);

        case SIGINT:
            printf("\nSIGINT\n");
            exit(-1);

        case SIGTERM:
            printf("\nSIGTERM\n");
            exit(-1);
    }
}

/* ******************************************************************
 *
 *  main()
 *
 ********************************************************************/

/*
 *@@ ParseArgs:
 *      parses the WIC command line.
 *
 *      Returns TRUE if the command line is OK and
 *      WIC can continue.
 *
 *@@added V0.9.14 (2001-08-23) [umoeller]
 *@@changed V0.9.19 (2002-04-14) [kso]: fixed parsing of parameter file
 *@@changed V0.9.20 (2002-07-03) [umoeller]: error messages cleanup
 */

VOID ParseArgs(int argc,                // in: from main()
               char *argv[],            // in: from main()
               PULONG pulNextMode,      // out: wic operation mode
               PULONG pulSelfInstall,   // out: 0, if ! -U, 1 if -U, 2 if -U+
               PLONG plListPackage)     // out: package to list in list mode
{
    PackageMaskSpec*   pPackageMaskSpecThis = 0;

    ULONG       ulNextMode = 0;
    int i = 0;
    while (i++ < argc-1)
    {
        if (argv[i][0] == '-')
        {
            ULONG i2;
            for (i2 = 1; i2 < strlen(argv[i]); i2++)
            {
                BOOL fNextArg = FALSE;

                switch (argv[i][i2])
                {
                    case 'a':
                        if (ulNextMode)
                            if (ulNextMode != MODE_SETSCRIPT)
                                Error(CID_INCORRECT_INVOCATION,
                                      "'Add' mode (-a) can only be combined with 'Script' mode (-s).");
                            else
                                // script mode before
                                if (G_szScriptFile[0] == 0)
                                    Error(CID_INCORRECT_INVOCATION,
                                          "No script specified for 'Script' mode (-s).");

                        ulNextMode = MODE_ADD;
                    break;

                    case 'l':
                        if (ulNextMode)
                            Error(CID_INCORRECT_INVOCATION,
                                  "'List' mode (-l) cannot be combined with other modes.");

                        ulNextMode = MODE_LIST;
                    break;

/*                         case 'd':
                        if (ulNextMode)
                        {
                            printf("'Delete' mode cannot be combined with other modes.\n");
                            fContinue = FALSE;
                        }
                        ulNextMode = MODE_DELETE;
                    break; */

                    case 's':
                        if (ulNextMode)
                            if (ulNextMode != MODE_ADD)
                                Error(CID_INCORRECT_INVOCATION,
                                      "'Script' mode (-s) can only be combined with 'Add' mode.");

                        ulNextMode = MODE_SETSCRIPT;
                    break;

                    case 'x':
                        if (ulNextMode)
                            Error(CID_INCORRECT_INVOCATION,
                                  "'Extract files' mode (-x) cannot be combined with other modes.");

                        ulNextMode = MODE_EXTRACT;
                    break;

                    case 'X':
                        if (ulNextMode)
                            Error(CID_INCORRECT_INVOCATION,
                                  "'Extract script' mode (-X) cannot be combined with other modes.");

                        ulNextMode = MODE_EXTRACTSCRIPT;
                    break;

                    case 'r':
                        // "recursive" on:
                        // add-mode only
                        if (ulNextMode != MODE_ADD)
                            Error(CID_INCORRECT_INVOCATION,
                                  "\"-r\" can only be used in 'Add' mode (-a).");

                        // this can only appear after a package number
                        if (!pPackageMaskSpecThis)
                            Error(CID_INCORRECT_INVOCATION,
                                  "\"-r\" for recursion needs a package number first.");

                        pPackageMaskSpecThis->fRecurse = TRUE;
                    break;

                    case 'c':
                        // change to subdir:
                        // add-mode only
                        if (ulNextMode != MODE_ADD)
                            Error(CID_INCORRECT_INVOCATION,
                                  "\"-c\" can only be used in 'Add' mode.");

                        // this can only appear after a package number
                        if (!pPackageMaskSpecThis)
                            Error(CID_INCORRECT_INVOCATION,
                                  "\"-c\" for subdirs needs a package number first.");

                        pPackageMaskSpecThis->strSubdir = &argv[i][i2+1];
                        fNextArg = TRUE;
                    break;

                    case 'u':
                        if (ulNextMode != MODE_ADD)
                            Error(CID_INCORRECT_INVOCATION,
                                  "\"-u\" can only be used in 'Add' (-a) mode.");

                        strcpy(G_szStubFile, strdup(&argv[i][i2+1]));
                        fNextArg = TRUE;
                    break;

                    case 'U':
                        if (ulNextMode != MODE_ADD)
                            Error(CID_INCORRECT_INVOCATION,
                                  "\"-U\" can only be used in 'Add' (-a) mode.");

                        // self-installer support:
                        if (argv[i][i2+1] == '+')
                        {
                            // full mode:
                            *pulSelfInstall = 2;      // full self-install
                            i2++;
                        }
                        else
                            *pulSelfInstall = 1;
                    break;

                    case 't':
                        if (ulNextMode)
                            Error(CID_INCORRECT_INVOCATION,
                                  "'Test' mode (-t) cannot be combined with other modes.");

                        ulNextMode = MODE_TESTSCRIPT;
                    break;

                    case 'q':
                        if (ulNextMode)
                            Error(CID_INCORRECT_INVOCATION,
                                  "'Query' mode (-q) cannot be combined with other modes.");

                        ulNextMode = MODE_QUERY_ARC;
                    break;

                    case 'i':
                        if (ulNextMode)
                            Error(CID_INCORRECT_INVOCATION,
                                  "'Install' mode (-i) cannot be combined with other modes.");

                        ulNextMode = MODE_INSTALL;
                    break;

                    case 'R':
                        if (ulNextMode != MODE_INSTALL)
                            Error(CID_INCORRECT_INVOCATION,
                                  "'-R' can only be used in 'Install' mode (-i).");

                        G_fRexxAllowed = TRUE;
                    break;

                    case 'v':
                        G_ulVerbosity = 1;
                    break;

                    case 'h':
                        // "help":
                        ExplainMore(argv[i][i2+1],   // next character, e.g. 'a'
                                    0);
                    break;

                    default:  // unknown parameter
                        Error(CID_INCORRECT_INVOCATION,
                              "Unknown parameter \"-%c\".\n", argv[i][i2]);
                }

                if (fNextArg)
                    break; // for (i2 = 1; i2 < strlen(argv[i]); i2++)

            } // end for (i2 = 1; i2 < strlen(argv[i]); i2++)
        } // end if (argv[i][0] == '-')
        else if (argv[i][0] == '@') // V0.9.12 (2001-06-09) [bird]
        {
            /*
             * Parameter file (debugger parameter length restrictions led to this):
             *    Create a textbuffer of the entrie file.
             *    Parse the file and create a new parameter vector.
             *    Set argv to the new parameter vector, argi to 0 and argc to
             *      the parameter count.
             *    Restrictions: Parameters enclosed in "" is not implemented.
             *                  Stupid users is allowed to loop parameter files.
             */
            char *pszBuffer = NULL;
            FILE *phFile;
            if (phFile = fopen(&argv[i][1], "rb"))
            {
                fseek(phFile, 0, SEEK_END);
                signed long cbFile = ftell(phFile);
                if (cbFile >= 0)
                {
                    if (pszBuffer = (char*)malloc(cbFile + 1))
                    {
                        memset(pszBuffer, 0, cbFile + 1);
                        fseek(phFile, 0, SEEK_SET);
                        if (cbFile > 0 && fread(pszBuffer, 1, cbFile, phFile) == 0)
                        {   /* failed! */
                            free(pszBuffer);
                            pszBuffer = NULL;
                        }
                    }
                    else
                        Error(CID_UNEXPECTED_CONDITION,
                              "Ooops! out of memory. (line %d)\n",
                              __LINE__);
                }
                fclose(phFile);
            }

            if (pszBuffer)
            {
                char ** papszArgs = NULL;
                char *  psz = pszBuffer;
                int     iArg = 0;

                while (*psz != '\0')
                {
                    /* find end of parameter word */
                    char *pszEnd = psz + 1;
                    char  ch = *pszEnd;
                    while (    (ch != ' ')
                            && (ch != '\t')
                            && (ch != '\n')
                            && (ch != '\r')
                            && (ch != '\0')
                          )
                        ch = *++pszEnd;

                    /* allocate more arg array space? */
                    if ((iArg % 512) == 0)
                    {
                        papszArgs = (char**)realloc(papszArgs, sizeof(char*) * 512);
                        if (papszArgs == NULL)
                            Error(CID_UNEXPECTED_CONDITION,
                                  "Ooops! out of memory. (line %d)\n",
                                  __LINE__);
                    }
                    *pszEnd = '\0';
                    papszArgs[iArg++] = psz;

                    /* next */
                    psz = pszEnd + 1;
                    ch = *psz;
                    while (    (ch == ' ')
                            || (ch == '\t')
                            || (ch == '\n')
                            || (ch == '\r')
                          )
                        ch = *++psz;
                }

                /*
                 * Add the remaining args.
                 */
                for (i++; i < argc; i++)        // fixed V0.9.19 (2002-04-14) [kso]
                {
                    /* allocate more arg array space? */
                    if ((iArg % 512) == 0)
                    {
                        papszArgs = (char**)realloc(papszArgs, sizeof(char*) * 512);
                        if (papszArgs == NULL)
                            Error(CID_UNEXPECTED_CONDITION,
                                  "Ooops! out of memory. (line %d)\n",
                                  __LINE__);
                    }
                    papszArgs[iArg++] = argv[i];
                }

                /*
                 * Install the new argument array and start processing it.
                 */
                i = -1;
                argc = iArg;
                argv = papszArgs;
            }
            else
                Error(CID_RESOURCENOTFOUND,
                      "Could not open parameter file \"%s\".",
                      &argv[i][1]);

        } // end if (argv[i][0] == '@') V0.9.12 (2001-06-09) [bird]
        else
        {
            // no option ("-"): seems to be file
            if (G_szFilestem[0] == 0)
            {
                // first non-option: seems to be archive name (all modes)...
                // check if this has wildcards
                if (    (strchr(argv[i], '*'))
                     || (strchr(argv[i], '?'))
                   )
                    Error(CID_INCORRECT_INVOCATION,
                          "Archive name cannot contain wildcards.");
                else
                    strcpy(G_szFilestem, argv[i]);
            }
            else
                // archive name already copied:
                // second arg depends on current mode
                switch (ulNextMode)
                {
                    case MODE_SETSCRIPT:
                    case MODE_EXTRACTSCRIPT:
                        if (G_szScriptFile[0] == 0)
                            strcpy(G_szScriptFile, argv[i]);
                        else
                            Error(CID_INCORRECT_INVOCATION,
                                  "Too many arguments after script name.");
                    break;

                    // with add or extract modes, this can be a package no.
                    case MODE_ADD:
                    case MODE_EXTRACT:
                        if (strhIsDecimal(argv[i]))
                        {
                            // argument is decimal: seems to be package number;
                            // save the previous stuff then
                            if (pPackageMaskSpecThis)
                            {
                                G_listPackageMaskSpecs.push_back(pPackageMaskSpecThis);
                                G_ulPackageMaskSpecsCount++;
                            }
                            pPackageMaskSpecThis = new(PackageMaskSpec);

                            LONG lPackage;
                            if (!(lPackage = atoi(argv[i])))
                                Error(CID_INCORRECT_INVOCATION,
                                      "You have specified an invalid package number.");

                            pPackageMaskSpecThis->lPackage = lPackage;
                        }
                        else
                        {
                            // non-decimal: seems to be file mask;
                            // in add mode, this can only appear after a package number
                            if (!pPackageMaskSpecThis)
                                        //  && (ulNextMode == MODE_ADD))
                                        // V0.9.14 (2001-07-14) [umoeller]
                                Error(CID_INCORRECT_INVOCATION,
                                      "You need to specify a package number first.");
                            else
                            {
                                pPackageMaskSpecThis->listFileMasks.push_back(
                                                    new string(argv[i]));
                                pPackageMaskSpecThis->cFileMasks++;
                            }
                        }
                    break;

                    case MODE_LIST:
                        if (strhIsDecimal(argv[i]))
                            *plListPackage = atoi(argv[i]);
                        else
                            Error(CID_INCORRECT_INVOCATION,
                                  "Invalid parameter for 'list' mode.");
                    break;

                    default:
                        Error(CID_INCORRECT_INVOCATION,
                              "Invalid parameter \"%s\".\n", argv[i]);
                    break;

                } // end switch (ulNextMode)
        } // end else if (argv[i][0] == '-')
    } // end while ((i++ < argc-1) && (fContinue))

    if (ulNextMode == 0)
        Error(CID_INCORRECT_INVOCATION,
              "You have not specified a valid operation mode.");

    if (!G_szFilestem[0])
    {
        // no filename specified:
        // in most modes, this will be an error

        switch (ulNextMode)
        {
            case MODE_TESTSCRIPT:
                Error(CID_INCORRECT_INVOCATION,
                      "You have not specified a script filename.");
            break;

            default:
                Error(CID_INCORRECT_INVOCATION,
                      "You have not specified an archive filename.");
        }
    }

    if (ulNextMode == MODE_ADD) // checks for add mode only, extract can be used without this
    {
        if (pPackageMaskSpecThis == 0)
            Error(CID_INCORRECT_INVOCATION,
                  "No package number specified for \"add\" mode.");

        if (pPackageMaskSpecThis->cFileMasks == 0)
            Error(CID_INCORRECT_INVOCATION,
                  "No file mask specified for \"add\" mode.");
    }

    if ( (ulNextMode == MODE_SETSCRIPT) && (G_szScriptFile[0] == 0) )
        Error(CID_INCORRECT_INVOCATION,
              "No script specified for \"script\" mode.");

    if (pPackageMaskSpecThis)
    {
        G_listPackageMaskSpecs.push_back(pPackageMaskSpecThis);
        G_ulPackageMaskSpecsCount++;
    }

    *pulNextMode = ulNextMode;
}

/*
 *@@ main:
 *      program entry point. Parses parameters and calls
 *      AddMode or ListMode accordingly.
 *
 *@@changed V0.9.1 (2000-02-07) [umoeller]: added "-c" subdir support
 *@@changed V0.9.6 (2000-10-27) [umoeller]: added "-x", "-X" modes
 *@@changed V0.9.13 (2001-06-09)    [bird]: added "@paramfile".
 *@@changed V0.9.14 (2001-08-23) [umoeller]: added -U[+] self-stub mode
 *@@changed V0.9.14 (2001-08-23) [umoeller]: various command-line bug fixes and checks
 */

int main(int argc, char *argv[])
{
    ULONG       ulNextMode = 0;
    LONG        lListPackage = 0;       // default: list all packages
    ULONG       ulSelfInstall = 0;      // 0: -U not specified
                                        // 1: -U specified
                                        // 2: -U+ specified
    int         rc = 0;

    // install signal handlers V0.9.12 (2001-06-09) [bird]
    signal(SIGBREAK, (_SigFunc)SigHandler);
    signal(SIGINT,   (_SigFunc)SigHandler);
    signal(SIGABRT,  (_SigFunc)SigHandler);

    // get full path of WIC.EXE
    PPIB    pib = NULL;
    CHAR    szWarpINFile[CCHMAXPATH] = "";
    DosGetInfoBlocks(NULL, &pib);
    DosQueryModuleName(pib->pib_hmte,
                       sizeof(szWarpINFile),
                       szWarpINFile);

    try
    {
        // parse parameters on cmd line
        if (argc == 1)
            // no parameters:
            ExplainParams(0);

        // go parse the command line
        ParseArgs(argc,
                  argv,
                  &ulNextMode,
                  &ulSelfInstall,
                  &lListPackage);

        // extract the path component
        string strWICPath(szWarpINFile,
                          strrchr(szWarpINFile, '\\'));

        if (ulSelfInstall)
        {
            // -U or -U+ specified:
            // create package mask specs for this then...
            // V0.9.14 (2001-08-23) [umoeller]
            const char
                *apcszBase[] =
                {
                    "wpirtl.dll",
                    "warpin.exe",
                    "warpin.hlp",
                    "warpin.tmf"
                },
                *apcszExtra[] =
                {
                    "COPYING",
                    "envwic.cmd",
                    "wic.exe",
                    "wicpm.exe",
                    "wizilla.EXE",
    //                     "wpi2exe.exe",
                    "wicpm.inf",
                    "wpi_prog.inf",
                    "wpi_user.inf",
                    "warpin.sym",
                    "wic.sym",
                    "wicpm.sym",
                    "wpirtl.sym",
                    "history.txt",
                    "readme.txt",
                    "test\\apps.cmd",
                    "test\\apps.wis"
                };

            CreateSelfMask(strWICPath,
                           WARPIN_SELF_PACKAGE_FOR_STUB,  // 30000
                           apcszBase,
                           ARRAYITEMCOUNT(apcszBase));

            if (ulSelfInstall == 2)
                // full install:
                CreateSelfMask(strWICPath,
                               WARPIN_MORE_PACKAGE_FOR_STUB,  // 30001
                               apcszExtra,
                               ARRAYITEMCOUNT(apcszExtra));

            sprintf(G_szStubFile,
                    "%s\\%s",
                    strWICPath.c_str(),
                    "stub.exe");
        }

        // add file extension, if none was specified
        const char *p;
        if (!(p = strrchr(G_szFilestem, '\\')))     // V0.9.14 (2001-07-14) [umoeller]
            p = G_szFilestem;
        if (!strchr(p, '.'))
        {
            // none given:
            // if we're in stub mode, use ".exe", otherwise ".wpi"
            if (G_szStubFile[0])
                sprintf(G_szArcFile, "%s.exe", G_szFilestem);
            else
                sprintf(G_szArcFile, "%s.wpi", G_szFilestem);
        }
        else
            strcpy(G_szArcFile, G_szFilestem);

        /*
         * MODE_SETSCRIPT only:
         *
         */
        if (ulNextMode == MODE_SETSCRIPT)
            ulNextMode = MODE_ADD;

        switch (ulNextMode)
        {
            case MODE_ADD:
                AddMode();
            break;

            /* case MODE_DELETE:
            break; */

            case MODE_LIST:
                ListMode(lListPackage);
            break;

            case MODE_EXTRACT:
                ExtractMode();
            break;

            case MODE_EXTRACTSCRIPT:
                ExtractScriptMode();
            break;

            case MODE_TESTSCRIPT:
                TestScriptMode(G_szFilestem,
                               strWICPath);
            break;

            case MODE_INSTALL:
                InstallMode(G_szFilestem,
                            strWICPath);
            break;

            case MODE_QUERY_ARC:
                QueryArcMode(G_szFilestem,
                             strWICPath);
            break;

            default:
                printf("Unsupported mode parameter.\n");
                rc = -1;
            break;
        }
    }
    catch (WICExcpt& CancelExcpt)
    {
        rc = CancelExcpt._rc;
    }

    if (rc)
    {
        printf("Terminating, rc = 0x%04lX (%d).\n", rc, rc);
    }

    return rc;
}

