/**
 * $Id: test.cpp,v 1.8 2001/07/01 20:49:04 jens Exp $
 *
 * Automated test of the WarpIN WIArchive class. Any errors will (hopefully)
 * show up when running this application, but don't be to sure...
 *
 * This file Copyright (C) 1999, 2000 Jens B&auml;ckman.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, in version 2 as it comes in the COPYING
 * file of this distribution.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <iostream.h>
#include "wiarchive2/wiarchive.h"

bool compressing = true;

/****************************************************************************
 * Callback function, processing messages from any WIArchive object
 *
 * Arguments - mode: Which action mode we are supposed to work in
 *             param: Any special parameters imporant for this callback mode
 *             pwifh: Usually a pointer to the current file header, or just
 *                    a plain, boring NULL
 *    Return - Depends. Usually 0, but some modes have a different value.
 */
int callback(short mode, unsigned long param, WIFileHeader *pwifh)
{
    int returnValue = 0;

    // Select action from the mode parameter
    switch (mode) {
        case CBM_HELLO:
            // This is the first message recieved when we set the callback
            // function. Just do nothing about this - just recieve it and
            // exit with nothing changed.
            cout << "Callback is working." << endl;
            break;

        case CBM_PERCENTAGE:
            // Sent when any progress in compressing/decompressing has been
            // made. The progress value is different depending on the operational
            // mode. If we compress a file, this is the number of bytes precessed
            // of the current file. In case decompression is commencing, this is
            // the number of bytes decompressed of the current package.
            //
            // param: Progress value
            // pwifh: Current file header
            // Return: Ignored.
            if (compressing) {
                cout << " " << pwifh->name << " -- " << param << " bytes\r" ;
            } else {
                if (param == 0)  cout << endl;
                cout << " " << pwifh->name << " -- " << param << " bytes\r" ;
            }
            break;

        case CBM_UPDATING:
            // After compression has finished on a file, this message will be
            // sent to the frontend for updating of the achieved compression
            // and such things.
            //
            // param: Size of file after compression
            // pwifh: Current file header
            // Return: Ignored.
            cout << " " << pwifh->name << " -- " << param << " bytes after ";
            switch (pwifh->method) {
                case 0:
                    cout << "storing" << endl;
                    break;
                case 1:
                    cout << "compression (BZ2)" << endl;
                    break;
                default:
                    cout << "unknown compression method" << endl;
            }
            break;

        case CBM_NEXTFILE:
            // This comes just before the WIArchive class attempts to open a new
            // output file for writing. This allows the front-end to do two things:
            //   a) update the "current file" display
            //   b) check for whether that file exists already:
            //      -- if not, we return CBRC_PROCEED;
            //      -- if it does exist, we ask the user for whether the file may be
            //         overwritten; if so, we try to delete it. Otherwise, we return
            //         CBRC_SKIP, and the back-end does not touch the file.
            // This new approach (Alpha #3) is required because sometimes the target
            // file which already exists is _locked_, e.g. if it's a system DLL in use.
            // We then need to do a DosReplaceModule, which we cannot do in the back-end
            // if we want to keep it platform-independent. Sorry for the hassle.
            //
            // param: NULL
            // pwifh: Current file header
            // Return: CBRC_PROCEED or CBRC_SKIP
            returnValue = CBRC_PROCEED;
            break;

        case CBM_ERR_MEMORY:
            // We're actually out of memory?!? What is going on here, not even
            // if swapping and stuff? We have to be running on Windows 2000 or
            // something equally bad. Well, since there is nothing we can do
            // about this, give the user a strange error message and quit. Fast.
            cout << endl;
            cout << " *** We're out of memory! *** " << endl;
            cout << "Free up some memory and then run this application again." << endl;
            exit(0);
            break;

        case CBM_ERR_WRITE:
            cout << "Recieved message CBM_ERR_WRITE!" << endl;
            break;

        case CBM_ERR_READ:
            cout << "Recieved message CBM_ERR_READ!" << endl;
            break;

        case CBM_ERR_COMPLIB:
            // While compressing/decompressing, a strange error occurred. This
            // usually means the archive is broken, sometimes badly. There is
            // no use in trying to complete the operation - let's quit this and
            // take a cup of coffee. Maybe even a beer. I personally recommend
            // a very tasty ale, Usher's Ruby Ale. You just got to try this. Oh,
            // I'm getting a little bit offtopic...
            //
            // param: Error value from BZ2 compression library
            // pwifh: Current file header
            // Return: Ignored.
            cout << endl;
            cout << " *** Compression library error *** " << endl;
            cout << "The error code recieved is " << param << endl;
            cout.flush();
            break;

        case CBM_ERR_CRC:
            cout << "Recieved message CBM_ERR_CRC!" << endl;
            break;
    }

    return returnValue;
}

/****************************************************************************
 *
 *
 * Arguments - s1:
 *             s2:
 *    Return -
 */
bool compareFiles(char *s1, char *s2)
{
    fstream file1, file2;

    // TODO: Yeees, this is not implemented yet...

    return true;
}

/****************************************************************************
 * Main function. If you're lost already - stop reading. It doesn't get any
 * better from this point.
 */
int main(int argc, char *argv[])
{
    WIArchive arc;          // Test archive
    fstream file;           // Just a file object
    const char *oldScript;  // Installer script before archiving...
    const char *newScript;  // ...and afterwards

    cout << " -=-=-=-=-=-=-=-=- Testing WarpIN archive class (WIArchive) -=-=-=-=-=-=-=-=-" << endl;
    cout << "Setting callback function... ";
    arc.setCallback(callback);
    oldScript = "Test string for installation script checking. Bruahaha. Halavah. Yetrehkoimm. Err...";

    // All tests are stored inside this try loop. A single throwed integer
    // gives an error message and exists this application.
    try {
        // ----- Create the test files ------------------------------------------
        cout << "Creating test files...       ";
        file.open("testing1", ios::out);
        file << "This is test file 1";
        file.close();
        file.open("testing2", ios::out);
        file << "This is test file 2, which is slightly larger than file #1. Slightly.";
        file << "Chances are that this file will actually be compressed, something that the first";
        file << "and third file won't have the luxury to be.";
        file.close();
        file.open("testing3", ios::out);
        file.close();
        file.open("testing4", ios::out);
        file << "This is test file 4";
        file.close();
        cout << "Done" << endl;

        // ----- Create test archive, add test files and compress them ----------
        cout << "Creating a new archive...    ";
        if (arc.open("testarc.wpi", WIArchive::WRITE) == false)  throw 1;
        cout << "Done" << endl;
        cout << "Adding test files...         ";
        arc.addPackage(1, "Package #1");
        arc.addPackage(2, "Package #2");
        arc.addPackage(3, "Package #3");
        arc.addFile("testing1", "testing_1", 1);
        arc.addFile("testing2", "testing_2", 1);
        arc.addFile("testing3", "testing_3", 2);
        arc.addFile("testing4", "testing_4", 2);
        arc.addFile("test.exe", "testing_5", 3);
        cout << "Done" << endl;
        cout << "Adding install script...     ";
        arc.setScript(oldScript);
        cout << "Done" << endl;
        cout << "Closing archive..." << endl;
        if (arc.close() == false)  throw 1;
        cout << "Done!" << endl;

        // ----- Open the archive again and check if stuff is identical ---------
        cout << "Re-opening archive...        ";
        compressing = false;
        if (arc.open("testarc.wpi", WIArchive::READ) == false)  throw 1;
        cout << "Done" << endl;
        cout << "Checking install script...   ";
        newScript = arc.getScript();
        if (strcmp(newScript, oldScript) != 0)  throw 1;
        cout << "Scripts are identical" << endl;
        cout << "Extracting packages...";
        if (arc.unpack(1) == false)  throw 1;
        if (arc.unpack(2) == false)  throw 1;
        if (arc.unpack(3) == false)  throw 1;
        cout << endl << "Done" << endl;
        cout << "Comparing files...           ";
        if (compareFiles("testing1", "testing_1") == false)  throw 1;
        if (compareFiles("testing2", "testing_2") == false)  throw 1;
        if (compareFiles("testing3", "testing_3") == false)  throw 1;
        if (compareFiles("testing4", "testing_4") == false)  throw 1;
        if (compareFiles("test.exe", "testing_5") == false)  throw 1;
        cout << "Files are identical" << endl;
        cout << "Closing archive...           ";
        if (arc.close() == false)  throw 1;
        cout << "Done" << endl;

        // ----- Remove the test files ------------------------------------------
        unlink("testing1");
        unlink("testing2");
        unlink("testing3");
        unlink("testing4");
        unlink("testing_1");
        unlink("testing_2");
        unlink("testing_3");
        unlink("testing_4");
        unlink("testing_5");
        unlink("testarc.wpi");
    } catch (int i) {
        // Enter this place if something goes wrong
        cout << "Failed!" << endl << endl;
        cout << "Sorry, test of WIArchive failed. Please report which test that failed and the" << endl;
        cout << "most vital details about your system (compiler, OS version) and report it to" << endl;
        cout << "the author by email (czw@home.se)." << endl;
        exit(1);
    }

    // ----- Report the result of this test ---------------------------------
    cout << endl;
    cout << "Success! WIArchive works perfectly according to this test and will probably" << endl;
    cout << "do so in your applications too. Have a nice day, and don't get hit by a car on" << endl;
    cout << "your way to work or something like that!" << endl;
    return 0;
}













