
/*
 *@@sourcefile wiarchiverw.cpp:
 *      this has the WIArchiveRW class, derived from
 *      WIArchive, for both compression and decompression.
 *
 *@@header "wiarchive\wiarchive.h"
 *@@added V0.9.14 (2001-07-24) [umoeller]
 */

/*
 *
 *  This file Copyright (C) 1998-2001 Jens Bckman, Ulrich Mller.
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, in version 2 as it comes in the COPYING
 *  file of this distribution.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include <io.h>
#include <string.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <sys\utime.h>

#include <stdio.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

// the following is required for VAC++
#ifdef __IBMCPP__
#include <direct.h>
#endif

#ifdef __DEBUG__
    #include <os2.h>            // otherwise pmpf won't work V0.9.14 (2001-07-24) [umoeller]
#endif

#include "setup.h"              // added V0.9.2 (2000-03-15) [umoeller]

#include "bzlib.h"

#include "base\bs_base.h"
#include "base\bs_list.h"

#define WIARCHIVE_INTERNAL
#include "wiarchive.h"

#pragma hdrstop

DEFINE_CLASS(WIArchiveRW, WIArchive);

/* ******************************************************************
 *
 *  WIArchiveRW
 *
 ********************************************************************/

/*
 *@@ WIArchiveRW:
 *
 */

WIArchiveRW::WIArchiveRW()
    : WIArchive(),
      _ToDoList(STORE)
{
    _fOpenForReadOnly = 0;

    _hfTempFile = 0;
    _TempArchiveName[0] = 0;
}

/*
 *@@ ~WIArchiveRW:
 *
 */

WIArchiveRW::~WIArchiveRW()
{
    _ToDoList.clear();
}

/*
 *@@ OpenTemp:
 *      creates a temporary archive filename and opens it.
 *
 *      Returns a C-library file or NULL on errors.
 *
 *@@changed V0.9.6 (2000-11-23) [umoeller]: now also storing temp file in member data
 *@@changed V0.9.13 (2001-06-23) [paperino]: adjustments for stub support
 */

int WIArchiveRW::OpenTemp(const char *filename)  // in: archive file name
{
    char    TempName[256];
    char    *cp;

    strcpy (TempName, filename);
    cp = strrchr(TempName, '\\');
    if (cp == NULL)
    {
        if (TempName[1] == ':')
            cp = &TempName[2];
        else
            cp = TempName;
    }
    else
        cp++;
    strcpy (cp, "_witemp_");

    _hfTempFile = ::open(TempName,        // fopen "w+b" empty binary file, destroy, readwrite
                         O_CREAT | O_RDWR | O_TRUNC | O_BINARY,
                         S_IREAD | S_IWRITE);
    if (_hfTempFile != -1)
    {
        // opening of the temporary file went fine, say that we use it
        strcpy(_TempArchiveName, TempName);
    }

    return (_hfTempFile);
}

/*
 *@@ Update:
 *      this updates the archive (which must be in "write" mode)
 *      with everything that must be updated: adds and removes files
 *      and such.
 *
 *      Preconditions:
 *
 *      -- The archive must have been opened in "write" mode.
 *
 *@@changed V0.9.19 (2002-07-01) [umoeller]: now returning error code
 */

int WIArchiveRW::Update()
{
    int rc = 0;     // V0.9.19 (2002-07-01) [umoeller]

    WIFileHeader filehead;
    unsigned short s;
    long pos, pos2, packpos;

    if (-1 == lseek(_hfArchive, _ArcExtHeader4.stubSize, SEEK_SET))
        rc = WIERR_IO_SEEK;     // V0.9.19 (2002-07-01) [umoeller]
    else if (!(rc = WriteArcHeader(_hfArchive)))
        rc = WritePackHeaders(_hfArchive);

    if (!rc)        // V0.9.19 (2002-07-01) [umoeller]
    {
        unsigned short sMax = 0;

        // find the highest package index in the archive
        list<WIPackHeaderLI*>::iterator pstart = _PackList.begin(),
                                        pend = _PackList.end();
        for (; pstart != pend; pstart++)
        {
            WIPackHeader *pPckThis = (*pstart)->_p;
            if (pPckThis->number > sMax)
                sMax = pPckThis->number;
        }
        // sMax now has the highest package index in use

        // check for every package there is
        _ArcHeader.sPackages = 0;
        for (s = 1;
             s <= sMax;
             s++)
        {
            bool fFound = 0;
            // see if we have a file on the to-do list which
            // uses the package with index s
            list<WIFileLI*>::iterator start = _ToDoList.begin(),
                                      end = _ToDoList.end();
            for (; start != end; start++)
            {
                WIFile *p = (**start)._p;
                if (p->package == s)
                {
                    fFound = 1;
                    break;
                }
            }

            if (fFound == 0)
            {
                // no file on the to-do list is using the package with index s:
                // then check if maybe a file on the list
                // of existing files uses it
                start = _FileList.begin();
                end = _FileList.end();
                for (; start != end; start++)
                {
                    WIFile *p = (**start)._p;
                    if (p->package == s)
                    {
                        fFound = 1;
                        break;
                    }
                }
            }

            if (fFound == 0)
                // no file is using this package (neither on
                // to-do list nor on existing list):
                // go for next package then, we don't need this
                continue;   // for s...

            // yes, we had at least one file in this package
            // (either on to-do list or on existing list)
            ++(_ArcHeader.sPackages);
            pstart = _PackList.begin();
            pend = _PackList.end();
            for (; pstart != pend; pstart++)
            {
                WIPackHeader *p = (**pstart)._p;
                if (p->number == s)
                    break;
            }

            // **pstart now has the current WIPackHeader to work on
            WIPackHeader *pPackHeaderThis = (**pstart)._p;        // V0.9.3 (2000-05-11) [umoeller]

            packpos = _tell(_hfArchive);
            pPackHeaderThis->files = 0;
            pPackHeaderThis->origsize = pPackHeaderThis->compsize = 0;

            // Process the files on our to-do list
            start = _ToDoList.begin();
            end = _ToDoList.end();
            for (; start != end; start++)
            {
                WIFile *pFileThis = (**start)._p;    // V0.9.3 (2000-05-11) [umoeller]

                if (pFileThis->package != s)
                    // incorrect pack number
                    continue;

                pos = _tell(_hfArchive);      // remember current position

                // copy data, store file header and open the source file
                strcpy (filehead.name, pFileThis->name);
                filehead.package = pFileThis->package;
                _hfFile = ::open(pFileThis->extra, // "rb");
                                 O_RDONLY | O_BINARY,
                                 0);
                lseek(_hfFile, 0, SEEK_END);
                filehead.origsize = _tell(_hfFile);
                filehead.extended = 0;
                filehead.magic = WIFH_MAGIC;

                // get file date/time
                struct stat StatBuf;
                fstat(_hfFile, &StatBuf);     // gets file info
                // store last write date and creation date in file header
                filehead.lastwrite = StatBuf.st_mtime;
                filehead.creation  = StatBuf.st_ctime;

                // write file header
                if (rc = WriteChecked(_hfArchive,
                                      (char*)&filehead,
                                      sizeof(WIFileHeader)))
                    // V0.9.19 (2002-07-01) [umoeller]
                    break;

                // Compress this thingie!
                lseek(_hfFile, 0, SEEK_SET);      // go to beginning of source file
                if (rc = Compress(&filehead))
                    break;

                if (filehead.compsize > filehead.origsize)
                {
                    // compressed version is larger than original one:
                    // rewind then and write the file as-is
                    lseek(_hfArchive,
                          pos + sizeof(WIFileHeader),
                          SEEK_SET);
                    lseek(_hfFile, 0, SEEK_SET);
                    if (rc = Store(&filehead))
                        break;
                }
                if (_pfnCallback)
                    _pfnCallback(CBM_UPDATING,
                                 CalcPercentage(filehead.compsize, filehead.origsize),
                                 &filehead,
                                 _pvCallbackArg);

                // remember position _after_ compressed file
                // (this is for the next file header)
                pos2 = _tell(_hfArchive);
                // go to beginning of _this_ file and rewrite the
                // updated file header (compressed size etc.)
                lseek(_hfArchive, pos, SEEK_SET);
                if (rc = WriteChecked(_hfArchive,
                                      (char *)&filehead,
                                      sizeof(WIFileHeader)))
                    // V0.9.19 (2002-07-01) [umoeller]
                    break;

                // and go back to _after_ this file for the next file header
                lseek(_hfArchive,
                      pos2,
                      SEEK_SET);
                pPackHeaderThis->files++;
                pPackHeaderThis->origsize += filehead.origsize;
                pPackHeaderThis->compsize += filehead.compsize;
            } // for (; start != end; start++)

            if (rc)
                break; // V0.9.19 (2002-07-01) [umoeller]

            // copy all the old files for package index s

            // set file pointer to what the package header has
            // as the beginning of files.  Only do this if we use a temp file
            // (i.e, updating an old archive)
            if (_hfTempFile > 0)
            {
                lseek(_hfOldArchive,
                      pPackHeaderThis->pos,
                      SEEK_SET);
                for (;;)
                {
                    pos = _tell(_hfOldArchive);
                    // read the file name and see if it's in the list
                    int stRead = read(_hfOldArchive,
                                      (char*)&filehead,
                                      sizeof(WIFileHeader));
                    if (stRead < sizeof(WIFileHeader))
                        break;

                    // search the list now
                    start = _FileList.begin();
                    end = _FileList.end();
                    for (; start != end; start++)
                    {
                        WIFile *pFileThis = (**start)._p;    // V0.9.3 (2000-05-11) [umoeller]

                        if (    (strcmp(filehead.name, pFileThis->name) == 0)
                             && (filehead.package == pFileThis->package)
                             && (filehead.package == s)
                           )
                        {
                            // this is the file, let's copy it
                            // 1)   copy file header we got above
                            if (rc = WriteChecked(_hfArchive,
                                                  (char*)&filehead,
                                                  sizeof(WIFileHeader)))
                                break;

                            // 2)   copy data in chunks of BLOCKSIZE
                            unsigned char *text = (unsigned char *)malloc(BUFFERSIZE);
                            long lBytesToGo = filehead.compsize;

                            while (lBytesToGo > 0)
                            {
                                long lBytesThis = lBytesToGo;
                                if (lBytesThis > BUFFERSIZE)
                                    lBytesThis = BUFFERSIZE;

                                read(_hfOldArchive,
                                     text,
                                     lBytesThis);

                                if (rc = WriteChecked(_hfArchive,
                                                      text,
                                                      lBytesThis))  // no. of blocks
                                    // V0.9.19 (2002-07-01) [umoeller]
                                    break;

                                lBytesToGo -= BUFFERSIZE;
                            }

                            free(text);

                            // update package header
                            pPackHeaderThis->files++;
                            pPackHeaderThis->origsize += filehead.origsize;
                            pPackHeaderThis->compsize += filehead.compsize;
                            break; // for (; start != end; start++)

                            // the source file pointer is now on the next file header
                        }
                    } // for (; start != end; start++)

                    if (rc)
                        break;  // V0.9.19 (2002-07-01) [umoeller]

                    // if we didn't find the file, advance the pointer
                    if (start == end)
                    {
                        lseek(_hfOldArchive,
                              filehead.compsize,
                              SEEK_CUR);
                        break;
                    }
                }
            }
            pPackHeaderThis->pos = packpos;
        } // end for (s = 1; ...

        if (!rc)    // V0.9.19 (2002-07-01) [umoeller]
        {
            // append headers to the _end_ of this archive so
            // that we can find them even if we have a stub
            // V0.9.13 (2001-06-23) [paperino]
            if (!(rc = AppendArcHeader(_hfArchive)))
            {
                // write the correct headers (with the updated offsets)
                // to the _front_ of the archive
                lseek(_hfArchive,
                      _ArcExtHeader4.stubSize,          // != 0 if we have a stub
                      SEEK_SET);
                if (!(rc = WriteArcHeader(_hfArchive)))
                    rc = WritePackHeaders(_hfArchive);
            }

            // done now, close file
            ::close(_hfArchive);
            _hfArchive = 0;

            // Remove the old archive and replace it with the new one.
            if (_hfTempFile > 0)
            {
                ::close(_hfOldArchive);
                _hfOldArchive = 0;

                unlink(_ArchiveName);       // old archive
                rename(_TempArchiveName, _ArchiveName);

                _hfTempFile = 0;
            }
        } // if (!rc)    // V0.9.19 (2002-07-01) [umoeller]
    } // if (!rc)        // V0.9.19 (2002-07-01) [umoeller]

    return rc;
}

/*
 *@@ WriteArcHeader:
 *      writes out the current member archive header into the
 *      specified file.
 *
 *      Writes to the current position!
 *
 *@@changed V0.9.13 (2001-07-02) [umoeller]: fixed wrong "revision needed"
 *@@changed V0.9.19 (2002-07-01) [umoeller]: now returning error code
 */

int WIArchiveRW::WriteArcHeader(int file)
{
    int         rc = 0;
    bz_stream   z;
    char        *pTempBuffer = 0;

    // First of all, set the archive identifiation and such thingies
    _ArcHeader.v1 = WI_VERIFY1;
    _ArcHeader.v2 = WI_VERIFY2;
    _ArcHeader.v3 = WI_VERIFY3;
    _ArcHeader.v4 = WI_VERIFY4;

    _ArcHeader.wi_revision_needed = WIARCHIVE_REVISION_3;
                // V0.9.13 (2001-07-06) [umoeller]

    _ArcHeader.lExtended = 0;        // V0.9.14 (2001-08-23) [umoeller]

    _ArcHeader.os = WI_OS_OS2;
    if (_pszScript == NULL)
        // no script:
        _ArcHeader.usScriptCompr = _ArcHeader.usScriptOrig = 0;
    else
    {
        // we have a script:
        // write it, but compressed
        _ArcHeader.usScriptOrig = strlen(_pszScript) + 1;

        // check size of script... if it's more than 64K, issue an error
        // V0.9.19 (2002-07-01) [umoeller]
        if (_ArcHeader.usScriptOrig > 0xFFFF)
            rc = WIERR_SCRIPT_TOO_BIG;
        else
        {
            int ScriptBufSize = _ArcHeader.usScriptOrig + 100;
            if (!(pTempBuffer = (char*)malloc(ScriptBufSize)))
                rc = WIERR_NOT_ENOUGH_MEMORY;   // V0.9.19 (2002-07-01) [umoeller]
            else
            {
                z.bzalloc = 0;
                z.bzfree  = 0;
                z.opaque = NULL;
                z.bzalloc = NULL;
                z.bzfree = NULL;
                z.next_in = _pszScript;
                z.avail_in = _ArcHeader.usScriptOrig;
                z.next_out = pTempBuffer;
                z.avail_out = ScriptBufSize;
                BZ2_bzCompressInit(&z,
                                   1,          // blockSize100k
                                   0,          // verbosity
                                   60);        // workFactor
                BZ2_bzCompress(&z, BZ_FINISH);
                _ArcHeader.usScriptCompr = ScriptBufSize - z.avail_out;
                BZ2_bzCompressEnd(&z);
            }
        }
    }

    if (!rc)
    {
        // V0.9.13 (2001-06-23) [paperino]
        _ArcExtHeader4.cbSize = sizeof(_ArcExtHeader4);

        if (_StubArchiveName[0])
        {
            // we have a stub exe: read and copy
            struct stat statbuf;
            int hfStub;
            char* buffer;

            // printf("adding '%s' as stub\n", _StubArchiveName);
                    // Yuri, please don't put printf's into the backend code...
                    // V0.9.13 (2001-07-02) [umoeller]

            // read stub size
            stat(_StubArchiveName, &statbuf);
            _ArcExtHeader4.stubSize = statbuf.st_size;
            // printf("Stub size is %d\n", _ArcExtHeader4.stubSize);
            // buffer = (char*)calloc(1, _ArcExtHeader4.stubSize);
                    // this is the only calloc call in the whole WarpIN sources,
                    // so to reduce runtime, use malloc instead V0.9.14 (2001-07-24) [umoeller]
            buffer = (char*)malloc(_ArcExtHeader4.stubSize);
            memset(buffer, 0, _ArcExtHeader4.stubSize);

            // copy stub data
            hfStub = ::open(_StubArchiveName,
                            O_RDONLY | O_BINARY,
                            0); // "rb");
            if (-1 == hfStub)
                rc = WIERR_CANNOT_READ_STUB;    // V0.9.19 (2002-07-01) [umoeller]
            else
            {
                if (read(hfStub, buffer, _ArcExtHeader4.stubSize)
                    < _ArcExtHeader4.stubSize)
                    rc = WIERR_CANNOT_READ_STUB;
                else
                    rc = WriteChecked(file,
                                      buffer,
                                      _ArcExtHeader4.stubSize);

                ::close(hfStub);

                free(buffer);       // was missing, V0.9.13 (2001-07-02) [umoeller]

                // don't process stub for next header write
                _StubArchiveName[0] = '\0';

                // require WarpIN 0.9.13 or higher
                _ArcHeader.wi_revision_needed = WIARCHIVE_REVISION_4;
            }
        }

        if (    (!rc)
             && (!(rc = WriteChecked(file,
                                     (char*)&_ArcHeader,
                                     sizeof(WIArcHeader))))
           )
        {
            // write out extended header, if we have one
            if (_fHasStub)
                rc = WriteChecked(file,
                                  (char*)&_ArcExtHeader4,
                                  sizeof(WIArcExtHeader4));

            if ((!rc) && (_pszScript != NULL))
                rc = WriteChecked(file, pTempBuffer, _ArcHeader.usScriptCompr);

            if ((!rc) && (_pExtended != NULL))
                rc = WriteChecked(file, _pExtended, _ArcHeader.lExtended);
        }
    }

    if (pTempBuffer)
        free(pTempBuffer);

    return rc;
}

/*
 *@@ AppendArcHeader:
 *      appends the archive headers to the _end_ of the
 *      WPI file. For backwards compatibility, we will
 *      only do this if the archive really has a stub.
 *
 *@@added V0.9.13 (2001-06-23) [paperino]
 *@@changed V0.9.19 (2002-07-01) [umoeller]: now returning error code
 */

int WIArchiveRW::AppendArcHeader(int file)
{
    int rc = 0;         // V0.9.19 (2002-07-01) [umoeller]

    if (_fHasStub)
    {
        long pos;
        // seek end of last sector
        lseek(file, 0, SEEK_END);
        pos = _tell(file);
        if (pos % 512)
            pos = (pos / 512 + 1) * 512;  // start at beginning of last sector

        lseek(file, pos, SEEK_SET);

        _ArcHeader.wi_revision_needed = WIARCHIVE_REVISION_4;

        // append header
        if (!(rc = WriteChecked(file,
                                (char*)&_ArcHeader,
                                sizeof(WIArcHeader))))
            // append extended header
            rc = WriteChecked(file,
                              (char*)&_ArcExtHeader4,
                              sizeof(WIArcExtHeader4));
    }

    return rc;
}

/*
 *@@ WritePackHeaders:
 *      writes all package headers into the specified archive
 *      file.
 *
 *      Writes to the current position!
 *
 *@@changed V0.9.19 (2002-07-01) [umoeller]: now returning error code
 */

int WIArchiveRW::WritePackHeaders(int file)
{
    int rc = 0;

    list<WIPackHeaderLI*>::iterator start, end;
    start = _PackList.begin();
    end = _PackList.end();
    for (; start != end; start++)
    {
        WIPackHeader *p = (**start)._p;

        if (rc = WriteChecked(file,
                              (char*)p,
                              sizeof(WIPackHeader)))
            // V0.9.19 (2002-07-01) [umoeller]
            break;
    }

    return rc;
}

/*
 *@@ add:
 *      public method for adding a file to the to-do list.
 */

void WIArchiveRW::add(const char *filename,  // in: file to add
                      const char *arcname,   // in: name of file to store in archive
                      short package)         // in: package index to add to
{
    WIFile *f = new WIFile;
    f->extra = new char[strlen(filename) + 1];
    f->name = new char[strlen(arcname) + 1];
    strcpy(f->extra, filename);
    strcpy(f->name, arcname);
    f->package = package;
    _ToDoList.push_back(new WIFileLI(f));
}

/*
 *@@ setStubFile:
 *      Adds a stub exe
 *      Arguments - filename: Name of the stub file
 *
 *@@added V0.9.13 (2001-06-23) [paperino]
 */

void WIArchiveRW::setStubFile (char *filename)
{
    if (filename && strlen(filename))
    {
        strcpy(_StubArchiveName, filename);
        _fHasStub = 1;      // V0.9.13 (2001-07-06) [umoeller]
    }
}

/*
 *@@ remove:
 *      removes the specified file from the specified package.
 */

void WIArchiveRW::remove(const char *filename,
                         short package)
{
    // first of all, check if the file exists in the to-do list
    list<WIFileLI*>::iterator start, end;
    start = _ToDoList.begin();
    end = _ToDoList.end();
    for (; start != end; start++)
    {
        WIFile *pFileThis = (*start)->_p;
        if (    (strcmp(pFileThis->name, filename) == 0)
             && (pFileThis->package == package)
           )
        {
            if (pFileThis->name != NULL)
                delete[] pFileThis->name;
            if (pFileThis->extra != NULL)
                delete[] pFileThis->extra;
            delete pFileThis;
            return;
        }
    }

    // was not on to-do list:
    // OK, remove it from the file list...
    start = _FileList.begin();
    end = _FileList.end();
    for (; start != end; start++)
    {
        WIFile *pFileThis = (**start)._p;
        if (    (strcmp (pFileThis->name, filename) == 0)
             && (pFileThis->package == package)
           )
        {
            if (pFileThis->name != NULL)
                delete[] pFileThis->name;
            if (pFileThis->extra != NULL)
                delete[] pFileThis->extra;
            delete pFileThis;
            break;
        }
    }
}

/*
 *@@ setArcHeader:
 *      copies the specified archive header data
 *      to the member archive header and fixes
 *      the data, if necessary.
 */

void WIArchiveRW::setArcHeader(const WIArcHeader &head)
{
    // _ArcHeader.wi_revision_needed = WIARCHIVE_REVISION_4;

    strcpy(_ArcHeader.szInternetAddr, head.szInternetAddr);
    strcpy(_ArcHeader.szAppName, head.szAppName);
    strcpy(_ArcHeader.szDeveloper, head.szDeveloper);
    strcpy(_ArcHeader.szPath, head.szPath);
    _ArcHeader.sAppRevision = head.sAppRevision;
    _ArcHeader.os = head.os;

    _ArcHeader.lExtended = 0;        // V0.9.14 (2001-08-23) [umoeller]

    // fix archive filename;
    // if it has a trailing "\", remove that
    char *cp = _ArcHeader.szPath;
    if (*cp != 0)
    {
        while (*cp != '\0')
            cp++;
        cp--;
        if (*cp != '\\')
        {
            cp++;  *cp = '\\';
            cp++;  *cp = '\0';
        }
    }
}

/*
 *@@ setPackage:
 *      sets some basic information about a package.
 *
 *      If no package with the specified index exists
 *      yet, a new package is created. Otherwise, the
 *      existing package is updated.
 */

void WIArchiveRW::setPackage(short package,
                             const char *name)
{

    WIPackHeader *p = 0;
    int i = 0;

    // see if we already have a package with this number
    list<WIPackHeaderLI*>::iterator start = _PackList.begin(),
                                    end = _PackList.end ();
    for (; start != end; start++)
    {
        WIPackHeader *p2 = (**start)._p;
        if (p2->number == package)
        {
            p = p2;
            i++;
            break;
        }
    }
    if (i == 0)
        p = new WIPackHeader;

    // Store this information
    strcpy(p->name, name);
    p->number = package;
    p->files = p->origsize = p->compsize = 0;
    if (i == 0)
        // new package:
        _PackList.push_back(new WIPackHeaderLI(p));
}

/*
 *@@ setScript:
 *      copies an installation script to the archive.
 *      If a script exists already, it is replaced.
 */

void WIArchiveRW::setScript(const char *cp)
{
    int i = strlen(cp);
    if (_pszScript != NULL)
        // exists already:
        free(_pszScript);      // V0.9.19 (2002-07-01) [umoeller]
    _pszScript = (char*)malloc(i + 1);
    memcpy(_pszScript, cp, i + 1);
}

/*
 *@@ open:
 *      opens the archive and reads the basic stuff in.
 *
 *      This calls WIArchive::ReadArcHeader, WIArchive::ReadPackHeaders,
 *      and WIArchive::ReadFilenames in turn.
 *
 *      If (mode == 0), the archive is opened in "read-only" mode.
 *
 *      If (mode == 1), the archive is opened in "read-write" mode.
 *      In that case, this method calls WIArchive::OpenTemp to create
 *      a temporary file.
 *
 *      Returns 0 if no error. Warning, this has changed with 0.9.13.
 *
 *@@changed V0.9.13 (2001-07-02) [umoeller]: finally returning better error codes
 */

int WIArchiveRW::open(const char *filename,
                      int mode)
{
    _fOpenForReadOnly = (mode == 0);     // we need to remember
                                         // whether the file was read-only for close()

    _Pmpf(("WIArchive::open(%s)", filename));

    strcpy(_ArchiveName, filename);
    _hfArchive = ::open(filename,
                        O_RDONLY | O_BINARY,
                        0); // fopen "rb");

    if (_hfArchive > 0)
    {
        // The file exists - check if it's a valid archive
        int i;
        if (i = ReadArcHeader(_hfArchive))
            return i;               // return error code V0.9.13 (2001-07-02) [umoeller]
        else
        {
            // Read all the files and packages in this archive
            if (i = ReadPackHeaders(_hfArchive))
                return i;           // return error code V0.9.13 (2001-07-02) [umoeller]

            if (i = ReadFilenames(_hfArchive))
                return i;           // return error code V0.9.13 (2001-07-02) [umoeller]
        }
        // If we are allowed to r/w, then create a new archive
        if (mode == 1)
        {
            ::close(_hfArchive);
            _hfArchive = OpenTemp(filename);
            if (_hfArchive <= 0)
                // OK, so we couldn't create a new archive. Run away!
                return WIERR_CANNOT_CREATE;       // V0.9.13 (2001-07-02) [umoeller]
            // UsingTemp = 1;
            _hfOldArchive = ::open(filename,
                                   O_RDONLY | O_BINARY, // "rb");
                                   0);
        }
    }
    else
    {
        // file doesn't exist:
        // are we in read-only mode?
        if (mode == 0)
            // yes: that's an error then
            return WIERR_FILENOTFOUND;

        // else read-write mode: create new archive
        _hfArchive = ::open(filename,
                            O_CREAT | O_RDWR | O_TRUNC | O_BINARY,
                            S_IREAD | S_IWRITE); // "w+b")))
        if (_hfArchive <= 0)
            // cannot create:
            return WIERR_CANNOT_CREATE;       // V0.9.14 (2001-07-24) [umoeller]
    }

    return 0;               // V0.9.13 (2001-07-02) [umoeller]
}

/*
 *@@ close:
 *      closes the archive. This calls WIArchive::Update in turn.
 *
 *@@changed V0.9.19 (2002-07-01) [umoeller]: now returning error code
 */

int WIArchiveRW::close()
{
    if (!_fOpenForReadOnly)
        return Update();
            // this will in turn call fclose

    ::close(_hfArchive);
    _hfArchive = 0;
    return 0;
}

/*
 *@@ closeAbort:
 *      closes the archive without performing an archive
 *      update. Use this for signal handlers to quickly
 *      close open files.
 *
 *@@added V0.9.4 (2000-07-22) [umoeller]
 */

void WIArchiveRW::closeAbort()
{
    ::close(_hfArchive);
    _hfArchive = 0;

    if (_hfTempFile)
    {
        ::close(_hfOldArchive);
        _hfTempFile = 0;            // @@todo this looks strange
    }
}


