
/*
 *@@sourcefile warpin.cpp:
 *      this is the main source file for WARPIN.EXE.
 *
 *      The WarpIn installer source code is organized into three
 *      major layers:
 *
 *      1)  The WIArchive class ("back end"). This one is standard
 *          C++ code which should be completely independent of OS/2
 *          and handles compression/decompression on the WPI file
 *          level only.
 *
 *          This is in the ../wiarchive and ../libbz2 directories
 *          and goes into WPIRTL.DLL.
 *
 *      2)  The "engine", which is OS/2-, but not PM-specific.
 *          This handles everything the scripts can do (script
 *          parsing, packages, install jobs etc.) plus the database.
 *          Starting with V0.9.14, this is in the "engine" directory
 *          and goes into WPIRTL.DLL as well.
 *
 *          This consists of a fairly large number of C++ classes to
 *          handle scripts, archives, packages, and install jobs.
 *          For all this, see fe_engine.cpp and fe_archive.cpp for
 *          an introduction.
 *
 *          This encapsulation allows us to to completely separate
 *          installation from what the user sees with WARPIN.EXE.
 *          WARPIN.EXE is only an application of FEInstallEngine;
 *          starting with V0.9.20, WIC.EXE is another one and can
 *          do full installs from the command line as well.
 *
 *          WarpIN operates in different "modes" depending on whether
 *          an archive was passed to WARPIN.EXE on the command line:
 *
 *          --  "install" mode, if an archive was passed on the
 *              command line; this will open the archive, load the
 *              install script, parse it, and display the pages as
 *              specified in the install script.
 *
 *          --  "database" mode, if no archive was given. This opens
 *              the entire database in a different window, from where
 *              packages can be de-installed as well.
 *
 *          See main() for more details and which functions in this
 *          file are called in which order.
 *
 *      3)  The front-end GUI, which does _only_ PM-specific
 *          stuff for all of these modes. This is the code in
 *          "frontend".
 *
 *          The GUI is not implemented thru classes, but thru plain
 *          functions. main() constructs lots of instances from the
 *          engine classes and passes the GUI callbacks to GUIInstallEngine,
 *          which get called e.g. while the files are being unpacked.
 *
 *          Note that WarpIn is multithreaded. When installation
 *          actually starts, i.e. on the "Install" page, a second
 *          thread is started, which calls the WIArchive methods
 *          upon the archive to unpack the files. The GUI part must
 *          have a number of callbacks for while installation is in
 *          progress, for showing the database, etc. If you wish to
 *          rewrite the gui_*.cpp part, you must be aware of which
 *          function gets called by which thread, or everything will
 *          crash pretty soon.
 *
 *      <B>Exception handling</B>
 *
 *      The entire main() routine is enclosed in a try{} block,
 *      so that any C++ exception thrown anywhere will finally end
 *      up in main()'s catch{} blocks. This applies especially
 *      to BSCancelExcpt, which can be thrown anywhere to terminate
 *      installation. All exceptions are defined in include\base\bs_errors.h.
 *
 *@@header "frontend\warpin.h"
 *@@header "frontend\calbacks.h"
 */

/*      This file Copyright (C) 1998-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WINSHELLDATA
#define INCL_WINWORKPLACE
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader
#ifdef __IBMCPP__
    #include <direct.h>
#endif
#include <string.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h

#include <ctype.h>

#ifdef __IBMCPP__
    #include <terminat.h>
#endif

#include "setup.h"              // was missing (99-12-23) [umoeller]
#include "bldlevel.h"           // added (99-10-24) [umoeller]

// include's from helpers
#include "helpers\dosh.h"
#include "helpers\nls.h"
#include "helpers\stringh.h"
#include "helpers\threads.h"
#include "helpers\winh.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"

#include "base\bs_logger.h"
#include "base\bs_config.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_script.h"
// #include "engine\fe_rexx.h"

#include "engine\fe_package.h"
#include "engine\fe_package_arc.h"
#include "engine\fe_package_db.h"
#include "engine\fe_archive.h"
#include "engine\fe_job.h"
#include "engine\fe_database.h"

#include "engine\fe_engine.h"

#include "frontend\warpin.h"

#define MORE_INCLUDES
#include "frontend\calbacks.h"

#include "helpers\except.h"     // must be at the bottom because wiarchive includes stdio

#ifdef __KLIB_LIBS__
    typedef unsigned long KSIZE;
    typedef unsigned long KBOOL;
    typedef KBOOL *PKBOOL;
    #define KLIBCALL _Optlink
    typedef enum { enmRead, enmWrite, enmUnknown } ENMACCESS;
#endif

#pragma hdrstop

/* ******************************************************************
 *
 *  More declarations
 *
 ********************************************************************/

void EnterArchiveMode(GUILocals &Locals,
                      const ustring &ustrArchiveFile);

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

// this is the large structure of variables which
// are shared with gui.cpp, declared in warpin.h.
WPIGLOBALS          G_WpiGlobals;

// WarpIN database (created in main())
// FEDatabase*         G_pDatabase;

// NLS settings from OS2.INI
// (moved these here from gui.cpp V0.9.14 (2001-08-23) [umoeller])
ULONG               G_ulDateFormat;
ULONG               G_ulTimeFormat;
CHAR                G_szDateSep[10],
                    G_szTimeSep[10],
                    G_szThousand[10];

/* ******************************************************************
 *
 *  PART 1
 *  ------
 *  The following code is all running on thread 1 (the main PM
 *  thread). This includes main(), which will open the archive,
 *  analyze the install script, build a list of dialog pages etc.
 *
 ********************************************************************/

/*
 *@@ ExceptOpenLogFile:
 *      hook for exceptions, registered with excRegisterHooks
 *      (src\helpers\except.c).
 *
 *@@changed V0.9.2 (2000-03-10) [umoeller]: switched date format to ISO
 */

FILE* APIENTRY ExceptOpenLogFile(VOID)
{
    CHAR        szFileName[CCHMAXPATH];
    FILE        *file;
    DATETIME    DT;

    // ULONG ulCopied;

    sprintf(szFileName, "%c:\\%s", doshQueryBootDrive(), "wpitrap.log");
    file = fopen(szFileName, "a");

    if (file)
    {
        DosGetDateTime(&DT);
        fprintf(file, "\nWarpIN trap message -- Date: %04d-%02d-%02d, Time: %02d:%02d:%02d\n",
            DT.year, DT.month, DT.day,
            DT.hours, DT.minutes, DT.seconds);
        fprintf(file, "-------------------------------------------------------\n"
                      "\nAn internal error occurred in WarpIN.\n"
                      "Please contact the author so that this error may be removed\n"
                      "in future WarpIN versions. The author's e-mail address may\n"
                      "be obtained from the WarpIN User Guide. Please supply this\n"
                      "file (?:\\WPITRAP.LOG) with your e-mail and describe as\n"
                      "exactly as possible the conditions under which the error\n"
                      "occured.\n"
                      "\nRunning WarpIN version: " BLDLEVEL_VERSION " built " __DATE__ "\n");

    }
    return (file);
}

/*
 *@@ ErrorBox:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

VOID ErrorBox(PCSZ pcsz)
{
    WinMessageBox(HWND_DESKTOP,
                  NULLHANDLE,
                  pcsz,
                  "WarpIN: Error",
                  0,
                  MB_MOVEABLE | MB_ICONEXCLAMATION | MB_CANCEL);
}

/*
 *@@ WarpINTerminate:
 *      function used with C++ set_terminate().
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.18 (2002-03-08) [umoeller]: added thread info
 */

void WarpINTerminate()
{
    char sz[500];
    ULONG tid = doshMyTID();
    PSZ psz = sz;
    psz += sprintf(psz, "WARPIN.EXE: Unexpected C++ exception caught on thread %d",
                   tid);
    if (tid != 1)
    {
        THREADINFO ti;
        if (thrFindThread(&ti, tid))
            sprintf(psz, " (thread name: %s)", ti.pcszThreadName);
    }
    ErrorBox(sz);
}

/* ******************************************************************
 *
 *  Program entry point
 *
 ********************************************************************/

/*
 *@@ ParseArgs:
 *      called from main() to parse the parameters
 *      line given to WarpIN.
 *
 *      Sets pszArchiveName to the archive name, if
 *      any was given on the command line. In that
 *      case, TRUE is returned.
 *
 *      If no archive name is given, FALSE is returned.
 *
 *      If more than one archive name was given, an
 *      exception is thrown since we can't handle this
 *      at this point.
 *
 *@@added V0.9.14 (2001-07-22) [umoeller]
 *@@changed V0.9.14 (2001-08-09) [umoeller]: added -d for database path override
 */

BOOL ParseArgs(GUILocals *pLocals,
               ustring &ustrArchiveName,
               ustring &ustrDatabasePath,
               int argc,
               char *argv[])
{
    BOOL rc = FALSE;

    if (argc > 1)
    {
        int i = 0;
        while (i++ < argc - 1)
        {
            if (argv[i][0] == '-')
            {
                BOOL fNextArg = FALSE;
                ULONG i2;
                for (i2 = 1; i2 < strlen(argv[i]); i2++)
                {
                    switch (argv[i][i2])
                    {
                        case 'U':       // 'U': force update of existing installation
                            G_WpiGlobals.SetupMode = WPIGLOBALS::UPGRADEEXISTING;
                        break;

                        case 'F':       // 'F': force WarpIN install
                            G_WpiGlobals.SetupMode = WPIGLOBALS::DISABLEEXISTING;
                        break;

                        case 'I':       // 'I': ignore self-setup
                            G_WpiGlobals.SetupMode = WPIGLOBALS::IGNORE;
                        break;

                        case 'S':       // 'S': called from stub.exe
                            // G_WpiGlobals.SetupMode = WPIGLOBALS::DISABLEEXISTING;
                        break;

                        case 'd':       // '-d<path>': database path
                        {
                            ustrDatabasePath.assignCP(pLocals->_pCodecProcess,
                                                      &argv[i][i2+1]);
                            fNextArg = TRUE;
                        }
                        break;

                        default:
                        {
                            ustring str;
                            str._printf("-%c", argv[i][i2]);
                            throw FEFatalErrorExcpt(*pLocals,
                                                    210,        // unknown option "%1"
                                                    &str,
                                                    1);
                        }
                        break;
                    }

                    if (fNextArg)
                        break; // for (i2 = 1; i2 < strlen(argv[i]); i2++)
                }
            }
            else
            {
                // no "-" option: seems to be archive name then...
                // we allow only one
                if (!rc)
                {
                    ustrArchiveName.assignCP(pLocals->_pCodecProcess, argv[i]);
                    rc = TRUE;
                }
                else
                    // more than one archive name:
                    throw FEFatalErrorExcpt(*pLocals, 209);       // more than one archive
            }
        }
    }

    return rc;
}

#ifdef __KLIB_LIBS__
KBOOL  KLIBCALL     kHeapDbgException(void *    pvAccess,
                                      ENMACCESS enmAccess,
                                      void *    pvIP,
                                      void *    pvOS);
#endif

/*
 *@@ main:
 *      program entry point.
 *
 *      This first initializes the GUI front-end, branches
 *      into several subroutines, depending on the command
 *      line parameters:
 *
 *      --  EnterArchiveMode(), if a WarpIN archive was
 *          passed on the command line; that function
 *          checks whether archives have already been
 *          installed and will then enter "install" mode;
 *
 *      --  guiBeginDatabaseMode(), if no archive was
 *          passed and we should switch into "database"
 *          mode.
 *
 *@@changed V0.9.2 (2000-03-11) [umoeller]: added REXX support
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added auto-restart WPS option
 *@@changed V0.9.6 (2000-10-27) [umoeller]: disabled white-error popups
 *@@changed V0.9.9 (2001-03-30) [umoeller]: disabled stupid finish message in CID mode
 *@@changed V0.9.14 (2001-07-22) [umoeller]: reworked self-setup with command-line args
 *@@changed V0.9.14 (2001-07-28) [umoeller]: lotsa changes for FELocals and stuff
 *@@changed V0.9.14 (2001-08-03) [umoeller]: now using DosQueryModuleName for getting our own exe
 *@@changed V0.9.14 (2001-08-09) [umoeller]: fixed try/catch crashes
 *@@changed V0.9.18 (2002-03-08) [umoeller]: major error handling rework; added restart
 */

int main(int argc,
         char *argv[])
{
    // initialize the PM environment (this is
    // needed for the profile functions)
    HAB  habThread1 = WinInitialize(0);
    HMQ  hmqThread1 = WinCreateMsgQueue(habThread1, 0);

    // "please wait, initializing"
    gshrCreateSplash(hmqThread1);

    // register exception hooks for helpers\except.c
    excRegisterHooks(ExceptOpenLogFile, NULL, NULL, TRUE);

    set_terminate(WarpINTerminate);

    DosError(FERR_DISABLEHARDERR | FERR_ENABLEEXCEPTION);
        // V0.9.6 (2000-10-27) [umoeller]

    // install the LOUD exception handler (except.h);
    // all the string parsing might crash...
    TRY_LOUD(excpt1)
    {
        // allow the GUI to restart WarpIN when the codepage changes
        BOOL    fRestart = TRUE;        // reset by GUILocals
        while (fRestart)            // V0.9.18 (2002-03-08) [umoeller]
        {
            // query all kinds of NLS data from OS2.INI
            // which we will use to display data
            G_ulDateFormat = PrfQueryProfileInt(HINI_USER, "PM_National", "iDate", 0);
            G_ulTimeFormat = PrfQueryProfileInt(HINI_USER, "PM_National", "iTime", 0);
            PrfQueryProfileString(HINI_USER,
                                  "PM_National",
                                  "sDate",
                                  "/",
                                  G_szDateSep,
                                  sizeof(G_szDateSep)-1);
            PrfQueryProfileString(HINI_USER,
                                  "PM_National",
                                  "sTime",
                                  ":",
                                  G_szTimeSep,
                                  sizeof(G_szTimeSep)-1);
            PrfQueryProfileString(HINI_USER,
                                  "PM_Default_National",
                                  "sThousand",
                                  ",",
                                  G_szThousand,
                                  sizeof(G_szThousand)-1);

            // create an instance of GUILocals, which
            // holds changes to config.sys etc. V0.9.14 (2001-07-26) [umoeller]
            // (GUILocals is now derived from the abstract base class FELocals
            // and implements gui-specific things)
            // V0.9.18 (2002-03-08) [umoeller]
            try
            {
                GUILocals Locals(habThread1,
                                 hmqThread1,
                                 G_szThousand[0],
                                 &fRestart);
                        // note: the destructor deletes all temp files again!
                        // V0.9.14 (2001-07-26) [umoeller]

                /*
                 * initialize
                 *
                 */

                // open generic try block for exceptions
                // which are not caught in the subroutines;
                // this especially applies to BSCancelExcpt

                // moved this down because exception handlers
                // need access to FELocals instace V0.9.14 (2001-08-09) [umoeller]
                try
                {
                    ustring ustrArchiveName,
                           // for the database, use WarpIN dir, unless
                           // overridden in ParseArgs V0.9.14 (2001-08-09) [umoeller]
                           ustrDatabasePath = Locals.QueryWarpINPath();

                    // parse command line V0.9.14 (2001-07-22) [umoeller]
                    BOOL fArchiveMode = ParseArgs(&Locals,
                                                  ustrArchiveName,
                                                  ustrDatabasePath,
                                                  argc,
                                                  argv);

                    // set log file name V0.9.9 (2001-03-30) [umoeller]
                    ustring ustrLogFilename(ustrDatabasePath);
                                        // use database path also
                    ustrLogFilename.appendUtf8("\\install.log");
                    Locals.SetLogFilename(ustrLogFilename);

                    // initialize the database
                    Locals._pDatabase = new FEDatabase(Locals,
                                                       ustrDatabasePath);

                    // initialize REXX support (global variable is in src\engine\fe_rexx.cpp)
                    // G_pFERexx = new FERexx(G_aRexxFunctions,
                    //                        G_cRexxFunctions);
                    // moved this to GUIInstallEngine V0.9.20 (2002-07-03) [umoeller]

                    // now go check self-setup V0.9.14 (2001-07-22) [umoeller]
                    if (!guiCheckInstall(&Locals))
                        // we should not continue:
                        throw BSCancelExcpt();

                    if (fArchiveMode)
                    {
                        /*
                         *  "Archive" mode:
                         *      an archive was given on the command line.
                         */

                        EnterArchiveMode(Locals, ustrArchiveName);

                    } // end if (argc > 1)
                    else
                    {
                        /*
                         *  "Database" mode:
                         *      if no archive given on the command line.
                         */

                        // GetGeneralEnvironment(&Locals);

                        if (Locals._pDatabase->_DBPackagesList.empty())
                        {
                            // no packages found (first run or
                            // database not found):
                            guiProductInfo(NULLHANDLE, &Locals, TRUE);
                        }
                        else
                            guiBeginDatabaseMode(&Locals);
                    }

                    // done: check what we've done and display an according message
                    if (    (G_WpiGlobals.fInstallComplete)
                         || (G_WpiGlobals.fDeinstallComplete)
                       )
                    {
                        ustring ustrDone,
                                ustr2;

                        if (G_WpiGlobals.fInstallComplete)
                        {
                            Locals.GetMessage(ustrDone,
                                              143      // installation has completed
                                              );
                            ustrDone.appendUtf8(" ");
                            if (Locals._fWPSObjectsCreated)
                            {
                                // created objects:
                                Locals.GetMessage(ustr2,
                                                  144);  // "objects created"
                                ustrDone += ustr2;
                            }
                        }

                        if (    (Locals._fConfigSysChanged)
                             && (!Locals._fWPSClassesChanged)
                           )
                        {
                            ustring aStrings;
                            aStrings.assignCP(Locals._pCodecProcess,
                                              Locals._strConfigSysBackup);
                            Locals.GetMessage(ustr2,
                                              145, // "CONFIG.SYS changed"
                                              &aStrings,      // replacement for %1
                                              1);
                            ustrDone += ustr2;
                        }
                        else if (    (!Locals._fConfigSysChanged)
                                  && (Locals._fWPSClassesChanged)
                                )
                        {
                            Locals.GetMessage(ustr2,
                                              146);  // "WPS class list changed",
                            ustrDone += ustr2;
                        }
                        else if (    (Locals._fConfigSysChanged)
                                  && (Locals._fWPSClassesChanged)
                                )
                        {
                            Locals.GetMessage(ustr2,
                                              147);  // "both CONFIG.SYS and WPS class list changed",
                            ustrDone += ustr2;
                        }

                        if (ustrDone())
                        {
                            // we got something to say:
                            ustring title;
                            title.assignUtf8("WarpIN");
                            Locals.ShowMessage(title,
                                               ustrDone,
                                               MSG_WARNING_OK);
                        }

                        // offer restart WPS if only the WPS class list,
                        // but not CONFIG.SYS has changed.
                        if (    (!Locals._fConfigSysChanged)
                             && (Locals._fWPSClassesChanged)
                           )
                        {
                            if (Locals.MessageBox(192,
                                                  193,  // restart WPS?
                                                  MSG_CONFIRM_YESNO_DEFYES)
                                == MBID_YES)
                            winhResetWPS(Locals._habThread1);
                        }
                    }
                }
                catch(BSCancelExcpt &X)
                {       // do nothing, this is for exiting only;
                }
                /*
                disabled V0.9.18 (2002-03-08) [umoeller]

                catch (FEDatabaseExcpt &X)
                {
                    ustring str2 = X._strDescription;

                    if (X._fRecover)
                    {
                        // recoverable: delete that key
                        FEDatabase Database2(Locals,
                                             strDatabasePath,
                                             X._ustrFailingPckID.GetBuffer());     // bugfixing
                        str2.appendUtf8("\nThe offending database entry has been deleted. Please restart WarpIN.");
                    }

                    string str;
                    str.assignUtf8(Locals._pCodecGui, str2);
                    guiShowMessage2("WarpIN: Database Error",
                                    str,
                                    MB_ICONEXCLAMATION | MB_CANCEL);
                } */

                // now, the GUILocals stack instance is being destroyed;
                // this will invoke the destructor and clean up properly
                // V0.9.18 (2002-03-03) [umoeller]
            }
            catch (BSUnsupportedCPExcpt &X)
            {
                ULONG acp[8];
                ULONG cb = 0;
                DosQueryCp(sizeof(acp),
                           acp,
                           &cb);

                string str;
                str._printf("Unsupported codepage %u.\n"
                            "WarpIN was unable to create a conversion table between that codepage "
                            "and Unicode. This probably means that the current system codepage "
                            "(which is %u) is not supported. Try switching to one of the supported codepages, such as 850 or 1004.\n",
                            X._usCodepage,
                            acp[0]);
                ErrorBox(str.c_str());
            }
            catch (BSExcptBase &X)
            {
                BSUniCodec codec(850);
                string str;
                str.assignUtf8(&codec, X._ustrDescription);
                ErrorBox(str.c_str());
            }
        } // end while fRestart
    }
    CATCH(excpt1)
    {
        // exception occured:
        // kill Install thread if it's running
        if (thrQueryID(&G_WpiGlobals.tiInstallThread))
            thrKill(&G_WpiGlobals.tiInstallThread);

        WinMessageBox(HWND_DESKTOP,
                      NULLHANDLE,
                      "Exception occured on main thread.",
                      "WarpIN: Internal Error",
                      0,
                      MB_ICONEXCLAMATION | MB_CANCEL | MB_MOVEABLE);

        #ifdef __KLIB_LIBS__
            if (    excpt1.RegRec2.err.ExceptionNum == XCPT_ACCESS_VIOLATION
                &&  (   excpt1.RegRec2.err.ExceptionInfo[0] == XCPT_READ_ACCESS
                     || excpt1.RegRec2.err.ExceptionInfo[0] == XCPT_WRITE_ACCESS
                     || excpt1.RegRec2.err.ExceptionInfo[0] == XCPT_EXECUTE_ACCESS
                     || excpt1.RegRec2.err.ExceptionInfo[0] == XCPT_UNKNOWN_ACCESS)
                )
            {
                ENMACCESS enmAccess = enmRead;
                switch (excpt1.RegRec2.err.ExceptionInfo[0])
                {
                    case XCPT_WRITE_ACCESS:     enmAccess = enmWrite; break;
                    case XCPT_UNKNOWN_ACCESS:   enmAccess = enmUnknown; break;
                }

                kHeapDbgException((void*)excpt1.RegRec2.err.ExceptionInfo[1],
                                  enmAccess,
                                  excpt1.RegRec2.err.ExceptionAddress,
                                  &excpt1.RegRec2.err);
            }
        #endif

    } END_CATCH();

    // clean up on the way out
    WinDestroyMsgQueue(hmqThread1);
    WinTerminate(habThread1);

    return (0);         // no error
}

/*
 *@@ ARCHIVETHREADDATA:
 *
 *@@added V0.9.20 (2002-07-06) [umoeller]
 */

typedef struct _ARCHIVETHREADDATA
{
    GUIInstallEngine    *pEngine;
    ustring             ustrArchiveFile;
} ARCHIVETHREADDATA, *PARCHIVETHREADDATA;

/*
 *@@ fntArchiveThread:
 *      transient thread started from EnterArchiveMode
 *      via thrRunSync in order not to block the SIQ
 *      while doing the potentially expensive REXX
 *      parsing.
 *
 *@@added V0.9.12 (2001-05-31) [umoeller]
 *@@changed V0.9.16 (2001-12-08) [umoeller]: fixed hang in thrRunSync on "Cancel" in varprompt dlg
 *@@changed V0.9.18 (2002-03-08) [umoeller]: added exception handling
 */

VOID _Optlink fntArchiveThread(PTHREADINFO ptiMyself)
{
    APIRET arc = 0;
    TRY_LOUD(excpt1)
    {
        PARCHIVETHREADDATA patd = (PARCHIVETHREADDATA)ptiMyself->ulData;
        GUILocals *pLocals = (GUILocals *)&patd->pEngine->_Locals;

        try
        {
            patd->pEngine->AppendArchive(patd->ustrArchiveFile);
        }
        catch(BSCancelExcpt &X)
        {       // do nothing, this is for exiting only;
            arc = 1;
        }
        catch (BSExcptBase &X)
        {
            // base exception class: all exception classes
            // are derived from this
            string str;
            str.assignUtf8(pLocals->_pCodecGui, X._ustrDescription);
            guiShowMessage2("WarpIN: Error",
                            str,
                            MB_ICONEXCLAMATION | MB_CANCEL);
            arc = 1;
        }
    }
    CATCH(excpt1)
    {
        arc = ERROR_PROTECTION_VIOLATION;
    } END_CATCH();

    // notify "done"
    WinPostMsg(ptiMyself->hwndNotify,
               WM_USER,
               (MPARAM)arc,
               0);
}

/*
 *@@ EnterArchiveMode:
 *      this gets called from main() when WarpIN was
 *      started with an archive as a parameter.
 *
 *      We create an instance of GUIInstallEngine
 *      from the file name which was given to us here.
 *      GUIInstallEngine will then parse
 *      the entire install script in the archive and
 *      create instances of FEInstallJob accordingly,
 *      each containing an instance of FEArcPackageBase.
 *
 *      We then call guiBeginInstallMode to have the GUI
 *      display the various pages in the script. That
 *      function only returns after the user has gone
 *      thru all pages.
 *
 *      If any packages have been selected for removal
 *      from the GUI (or thru environment variables),
 *      we call guiStartRemove.
 *
 *      If any packages must be installed, we then call
 *      guiStartInstall and invoke FEInstallJob::Store
 *      on all jobs which have been installed.
 *
 *      In any case, afterwards we call wpiConfigure to have
 *      CONFIG.SYS and the WPS changed, if necessary.
 *
 *      So this function only returns after everything has
 *      been done, or installation has been cancelled, or
 *      an error occured.
 *
 *@@changed V0.9.0 (99-11-01) [umoeller]: adjusted for FEInstallJob
 *@@changed V0.9.0 (99-11-02) [umoeller]: completely reworked add/remove mode
 *@@changed V0.9.3 (2000-04-28) [umoeller]: removed add/remove mode
 *@@changed V0.9.3 (2000-05-12) [umoeller]: added pszCfgSysBackup
 *@@changed V0.9.4 (2000-07-28) [umoeller]: fixed environment problems
 *@@changed V0.9.6 (2000-10-27) [umoeller]: made CONFIG.SYS backup part of WPIGLOBALS
 *@@changed V0.9.6 (2000-11-23) [umoeller]: added "no display pages" support
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added log file
 *@@changed V0.9.12 (2001-05-31) [umoeller]: moved script parsing to fntArchiveThread thread
 *@@changed V0.9.14 (2001-07-26) [umoeller]: moved most of the code to GUIInstallEngine
 */

void EnterArchiveMode(GUILocals &Locals,
                      const ustring &ustrArchiveFile)
{
    // GetGeneralEnvironment(Locals);

    /*
     *  "Archive" mode:
     *      archive given on command line
     */

    // create an instance of the install engine V0.9.14 (2001-07-26) [umoeller]
    GUIInstallEngine Engine(Locals);

    ARCHIVETHREADDATA atd;
    atd.pEngine = &Engine;
    atd.ustrArchiveFile = ustrArchiveFile;

    // parse script; this executes REXX code etc. and
    // might take a while, so use a second transient thread
    // for this V0.9.12 (2001-05-31) [umoeller]
    if (!thrRunSync(Locals._habThread1,
                    fntArchiveThread,
                    "ArchiveThread",
                    (ULONG)&atd))        // thread param
    {
        // no error parsing script, and not cancelled:

        // now parse all the environment variables to
        // override certain settings
        // GetArchiveEnvironment(Engine);       // install mode
                // removed V0.9.20 (2002-07-03) [umoeller]

        // enter "full install" mode
        BOOL fStartInstall;

        // display GUI pages
        if (fStartInstall = guiBeginInstallMode(Engine))
        {
            // 1) process KILLPROCESS attributes
            Engine.ConfigKillProcesses();

            // 2) deinstall packages, if any

            // create a list of FEDBPackage instances
            // from all the FEArcPackages which have
            // been selected for de-installation
            list<FEDBPackage*> DeinstallPackagesList(STORE);

            if (Engine.CreateDeinstallPackagesList(DeinstallPackagesList))
            {
                // anything to be deinstalled:
                Engine.UpdateGlobalStatus(GSTAT_REMOVING);

                // anything selected for removal:
                guiStartRemove(&Locals,
                               &DeinstallPackagesList,
                               // fixed flags
                               DBT_DELETEFILES | DBT_UNDOCONFIG,
                               TRUE);       // modal
            }

            // clean up list
            DeinstallPackagesList.clear();

            // 2) unpack files

            // TRUE means go ahead:
            guiStartInstall(Engine);

            FEInstallJob *pWarpINSelfInstallJob = NULL;

            if (G_WpiGlobals.fInstallComplete)
            {
                // success so far:
                // configure (CONFIG.SYS, WPS classes, objects)
                Engine.Configure(Locals._habThread1);

                // store packages in database
                Engine.Store(&pWarpINSelfInstallJob);

                Engine.UpdateGlobalStatus(GSTAT_DONEWITHALL);
            }

            if (pWarpINSelfInstallJob)
            {
                guiStoreWarpINPath(&Locals);

                // go create the WarpIN objects
                guiCreateWarpINObjects(&Locals,
                                       pWarpINSelfInstallJob->QueryTargetPath(),
                                       TRUE);       // force
            }
        } // end if (fStartInstall)
    }
}

/* ******************************************************************
 *
 *  PART 2
 *  ------
 *  The following code is started by part 1 when installation
 *  is actually about to begin, i.e. the packages shall be
 *  extracted from the archive. All this is running on a separate
 *  thread, the Install thread.
 *
 ********************************************************************/

/*
 *@@ wpiStartInstallThread:
 *      this starts the Install thread.
 *
 *      This returns immediately. From now on,
 *      the GUI gets lots of callbacks on the
 *      Install thread.
 *
 *@@changed V0.9.12 (2001-05-22) [umoeller]: now creating flag with PM message queue
 */

VOID wpiStartInstallThread(GUIInstallEngine &Engine)
{
    thrCreate(&G_WpiGlobals.tiInstallThread,
              wpiInstallThread,
              NULL,
              "Install",
              THRF_PMMSGQUEUE | THRF_WAIT,
              (ULONG)&Engine);
}

/*
 *@@ wpiInstallThread:
 *      thread function for the Install thread.
 *
 *      This gets started by wpiStartInstallThread and
 *      does the actual unpacking of the archive
 *      packages.
 *
 *      This is pretty short, because it only creates
 *      the target directories and then calls
 *      WIArchive::unpack, which will in turn call
 *      WICallback above.
 *
 *@@changed V0.9.0 (99-11-01) [umoeller]: adjusted for FEInstallJob
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added log file
 *@@changed V0.9.12 (2001-05-31) [umoeller]: fixed log file crash on exit
 *@@changed V0.9.13 (2001-06-21) [umoeller]: fixed bad DosQueryCurrentDir, thanks Yuri Dario
 */

void _Optlink wpiInstallThread(PTHREADINFO pti)
{
    GUIInstallEngine *pEngine = (GUIInstallEngine*)pti->ulData;

    FELocals &Locals = pEngine->_Locals;

    Locals.Log("Install thread started");

    pEngine->Install(NULL);              // use GUIInstallEngine's WICallback

    G_WpiGlobals.fInstallComplete = TRUE;

    Locals.Log("Install thread is exiting");
}


