
/*
 *@@sourcefile gui_db.cpp:
 *      this contains the PM-specific stuff for
 *      "database mode".
 *
 *      This has been separated from gui.cpp (99-10-24) [umoeller].
 *
 *      This has the required guiBeginDatabaseMode
 *      callback, which gets called from warpin.cpp
 *      when database mode is entered to have the
 *      database displayed.
 *
 *      Also, this has the required guiStartRemove and
 *      guiStartVerify callbacks.
 *
 *@@header "frontend\gui.h"
 */

/*
 *
 *      This file Copyright (C) 1998-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_WINWORKPLACE
#define INCL_GPILOGCOLORTABLE
#define INCL_GPIPRIMITIVES
#include <os2.h>

#include <stdio.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h
#include <io.h>

#include "setup.h"

// include's from helpers
#include "helpers\configsys.h"
#include "helpers\datetime.h"
#include "helpers\dosh.h"
#include "helpers\cnrh.h"
#include "helpers\comctl.h"
#include "helpers\gpih.h"
#include "helpers\nls.h"
#include "helpers\nlscache.h"
#include "helpers\winh.h"
#include "helpers\standards.h"
#include "helpers\stringh.h"
#include "helpers\threads.h"
#include "helpers\xstring.h"

#include "encodings\base.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_package.h"
#include "engine\fe_package_db.h"
#include "engine\fe_database.h"

#include "frontend\warpin.h"
#define MORE_INCLUDES
#include "frontend\calbacks.h"

#include "frontend\dlgids.h"
#include "frontend\gui.h"

#include "helpers\except.h"

#pragma hdrstop

/* ******************************************************************
 *
 *  Private declarations
 *
 ********************************************************************/

/*
 *@@ DatabaseView:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

struct DatabaseView
{
    GUILocals               &_Locals;

    HWND                    _hwndDBPackagesDlg,
                            _hwndDBFileListDlg,
                            _hwndDBSplitMain;

    // context menus
    HWND                    _hmenuMain,
                            _hmenuPckWhitespace,
                            _hmenuPckApplication,
                            _hmenuPckPackage,
                            _hmenuFilesWhitespace;

    // currently selected database record (fnwpDBPackagesDlg)
    PDATABASERECORD         _preccDBSelected;

    // source record core for context menus
    PDATABASERECORD         _preccLeftSource;
    PFILERECORDCORE         _preccRightSource;

    // list of all record cores for applications (tree view only)
    LINKLIST                _AppRecordsList;
    // list of all record cores for packages (all views)
    LINKLIST                _PckRecordsList;

    XADJUSTCTRLS            _xacPackagesDlg,
                            _xacFileListDlg;        // for winhAdjustControls

    list<string*>           _ConfigStrings;

    BOOL                    _fClearingContainers;
                // while this is TRUE, we do not update the files container
                // V0.9.9 (2001-02-28) [umoeller]

    /*
     *@@ DatabaseView:
     *      constructor.
     */

    DatabaseView(GUILocals &Locals)
        : _Locals(Locals),
          _ConfigStrings(STORE)
    {
        _hwndDBPackagesDlg = NULLHANDLE;
        _hwndDBFileListDlg = NULLHANDLE;
        _hwndDBSplitMain = NULLHANDLE;
        _hmenuMain = NULLHANDLE;
        _hmenuPckWhitespace = NULLHANDLE;
        _hmenuPckApplication = NULLHANDLE;
        _hmenuPckPackage = NULLHANDLE;
        _hmenuFilesWhitespace = NULLHANDLE;
        _preccDBSelected = NULL;
        _preccLeftSource = NULL;
        _preccRightSource = NULL;

        lstInit(&_AppRecordsList, FALSE);
        lstInit(&_PckRecordsList, FALSE);

        _fClearingContainers = FALSE;
    }

    ~DatabaseView()
    {
        WinDestroyWindow(_hwndDBPackagesDlg);
        WinDestroyWindow(_hwndDBPackagesDlg);
        WinDestroyWindow(_hwndDBFileListDlg);
        WinDestroyWindow(_hwndDBSplitMain);
        WinDestroyWindow(_hmenuMain);
        WinDestroyWindow(_hmenuPckWhitespace);
        WinDestroyWindow(_hmenuPckApplication);
        WinDestroyWindow(_hmenuPckPackage);
        WinDestroyWindow(_hmenuFilesWhitespace);

        WinDestroyWindow(G_hwndMainFrame);

        G_hwndMainFrame = NULLHANDLE;
    }
};

/*
 *@@ DBCREATESTRUCT:
 *
 *@@added V0.9.14 (2001-07-26) [umoeller]
 */

typedef struct _DBCREATESTRUCT
{
    USHORT          cb;
    DatabaseView    *pView;
} DBCREATESTRUCT, *PDBCREATESTRUCT;


// window class for database main frame
#define WC_DATABASEMAINFRAME    "WPI_DatabaseMainFrameClass"

MRESULT EXPENTRY fnwpDatabaseStatus(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
VOID UpdateListCnr(DatabaseView *pView, PDATABASERECORD precc);    // in: package record core

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

PFNWP                       G_pfnwpListboxOrig;

DatabaseView                *G_pView;       // needed for guiStartVerify only

// controls list for file-list window (winhAdjustControls)
const static MPARAM G_ampPackagesDlg[] =
            {
                MPFROM2SHORT(ID_WIDI_HEADING, XAC_MOVEY | XAC_SIZEX),
                MPFROM2SHORT(ID_WIDI_PACKAGESCNR, XAC_SIZEX | XAC_SIZEY),
                MPFROM2SHORT(ID_WIDI_CANCELBUTTON, XAC_MOVEX)
            };

const static MPARAM G_ampFileListDlg[] =
            {
                MPFROM2SHORT(ID_WIDI_HEADING, XAC_MOVEY | XAC_SIZEX),
                MPFROM2SHORT(ID_WIDI_TARGETPATHTXT, XAC_MOVEY),
                MPFROM2SHORT(ID_WIDI_TARGETPATHDIR, XAC_MOVEY | XAC_SIZEX),
                MPFROM2SHORT(ID_WIDI_PACKAGESCNR, XAC_SIZEX | XAC_SIZEY),
                MPFROM2SHORT(ID_WIDI_CANCELBUTTON, XAC_MOVEX)
            };

/* ******************************************************************
 *
 *  Helper funcs
 *
 ********************************************************************/

/*
 *@@ CreateDBAppRecord:
 *      creates a DATABASERECORD for the given application ID.
 *      Used with the database container subwindow.
 */

PDATABASERECORD CreateDBAppRecord(GUILocals *pLocals,
                                  HWND hwndCnrAlloc,
                                  FEDBPackage *pDBPackage)
{
    PDATABASERECORD preccApplication = // new GuiDatabaseRecord;
        (PDATABASERECORD)cnrhAllocRecords(hwndCnrAlloc,
                                    sizeof(DATABASERECORD),
                                    1);

    if (preccApplication)
    {
        // set package info to this package so we
        // can do string comparisons
        // V0.9.18 (2002-03-08) [umoeller]
        preccApplication->pDBPackage = NULL;

        preccApplication->recc.hptrIcon
            = preccApplication->recc.hptrMiniIcon
            = pLocals->_hptrApp;

        // store other fields
        preccApplication->pcszAuthor = pDBPackage->_PckID._strAuthorGui.c_str();
        preccApplication->pcszApplication = pDBPackage->_PckID._strApplicationGui.c_str();

        // make a copy of the utf-8 application string for string comparisons
        // V0.9.18 (2002-03-08) [umoeller]
        preccApplication->pszApplicationU = strdup(pDBPackage->_PckID._ustrApplication.GetBuffer());
    }

    return (preccApplication);
}

/*
 *@@ CreateDBPckRecord:
 *      creates a DATABASERECORD for the given package info.
 *      Used with the database container subwindow.
 */

PDATABASERECORD CreateDBPckRecord(GUILocals *pLocals,
                                  HWND hwndCnrAlloc,
                                  FEDBPackage* pDBPackage)
{
    PDATABASERECORD preccPck;

    if (preccPck = (PDATABASERECORD)cnrhAllocRecords(hwndCnrAlloc,
                                            sizeof(DATABASERECORD),
                                            1))
    {
        // store package info in recc
        preccPck->pDBPackage = pDBPackage;
        // and recc in package info
        pDBPackage->_pvGuiData = (PRECORDCORE)preccPck;

        preccPck->recc.hptrIcon
            = preccPck->recc.hptrMiniIcon
            = pLocals->_hptrQMark;

        // store other fields
        preccPck->pcszAuthor = pDBPackage->_PckID._strAuthorGui.c_str();
        preccPck->pcszApplication = pDBPackage->_PckID._strApplicationGui.c_str();
        preccPck->pcszPackageName = pDBPackage->_PckID._strPackageNameGui.c_str();

        // make a copy of the utf-8 application string for string comparisons
        // V0.9.18 (2002-03-08) [umoeller]
        preccPck->pszApplicationU = strdup(pDBPackage->_PckID._ustrApplication.GetBuffer());

        preccPck->pcszVersion = pDBPackage->_PckID._strVersion.c_str();
        preccPck->ulFiles = pDBPackage->_PackHeader.files;
        preccPck->ulTotalSize = pDBPackage->_PackHeader.origsize;
        string strTargetPath;
        strTargetPath.assignUtf8(pLocals->_pCodecGui, pDBPackage->QueryTargetPath());
        preccPck->pcszTargetPath = strdup(strTargetPath.c_str());

        // install date/time
        cnrhDateTimeDos2Win(&pDBPackage->_InstallDateTime,
                            &preccPck->cdateInstall,
                            &preccPck->ctimeInstall);
    }

    return (preccPck);
}

/*
 *@@ CheckLeftViewMenuItems:
 *      checks the view items for the left view
 *      (tree or details) in the given menu.
 *
 *      This can either be the "View" pulldown or
 *      a context menu.
 *
 *@@added V0.9.5 (2000-08-10) [umoeller]
 */

VOID CheckLeftViewMenuItems(GUILocals *pLocals,
                            HWND hmenu)
{
    WinCheckMenuItem(hmenu,
                     ID_WIMI_DATABASE_TREEVIEW,
                     pLocals->_GuiSettings.ulLeftView == ID_WIMI_DATABASE_TREEVIEW);
    WinCheckMenuItem(hmenu,
                     ID_WIMI_DATABASE_DETAILSVIEW,
                     pLocals->_GuiSettings.ulLeftView != ID_WIMI_DATABASE_TREEVIEW);
}

/*
 *@@ CheckRightViewMenuItems:
 *      checks the view items for the right view
 *      (files or config) in the given menu.
 *
 *      This can either be the "View" pulldown or
 *      a context menu.
 *
 *@@added V0.9.5 (2000-08-10) [umoeller]
 */

VOID CheckRightViewMenuItems(GUILocals *pLocals,
                             HWND hmenu)
{
    WinCheckMenuItem(hmenu,
                     ID_WIMI_VIEW_FILES,
                     pLocals->_GuiSettings.ulRightView == ID_WIMI_VIEW_FILES);
    WinCheckMenuItem(hmenu,
                     ID_WIMI_VIEW_CONFIG,
                     pLocals->_GuiSettings.ulRightView != ID_WIMI_VIEW_FILES);
}

/*
 *@@ ProcessWMChar:
 *      switches the focus between the two DB dialogs.
 *
 *@@added V0.9.2 (2000-03-10) [umoeller]
 */

VOID ProcessWMChar(DatabaseView *pView, MPARAM mp1, MPARAM mp2)
{
    USHORT usFlags    = SHORT1FROMMP(mp1);
    // USHORT usch       = SHORT1FROMMP(mp2);
    USHORT usvk       = SHORT2FROMMP(mp2);

    if (    (usFlags & KC_VIRTUALKEY)
         && (usvk == VK_TAB)
       )
    {
        if ((usFlags & KC_KEYUP) == 0)
        {
            HWND hwndFocus = WinQueryFocus(HWND_DESKTOP);
            if (WinIsChild(hwndFocus, pView->_hwndDBPackagesDlg))
                WinSetFocus(HWND_DESKTOP, WinWindowFromID(pView->_hwndDBFileListDlg, ID_WIDI_PACKAGESCNR));
            else
                WinSetFocus(HWND_DESKTOP, WinWindowFromID(pView->_hwndDBPackagesDlg, ID_WIDI_PACKAGESCNR));
        }
    }
}

/*
 *@@ InsertFilesIntoCnr:
 *      switches the right container to "files" mode
 *      and adds all the files for the specified package.
 *      This has been extracted from UpdateListCnr.
 *
 *@@added V0.9.2 (2000-02-19) [umoeller]
 */

VOID InsertFilesIntoCnr(PDATABASERECORD precc,  // in: database record selected on the left
                        HWND hwndFilesCnr)      // in: right cnr window
{
    // set up data for Details view columns
    XFIELDINFO      xfi[4];

    xfi[0].ulFieldOffset = FIELDOFFSET(RECORDCORE, pszIcon);
    xfi[0].pszColumnTitle = nlsGetString(WPSI_FILENAME); // "File name";
    xfi[0].ulDataType = CFA_STRING;
    xfi[0].ulOrientation = CFA_LEFT;

    xfi[1].ulFieldOffset = FIELDOFFSET(FILERECORDCORE, ulSize);
    xfi[1].pszColumnTitle = nlsGetString(WPSI_SIZE); // "Size";
    xfi[1].ulDataType = CFA_ULONG;
    xfi[1].ulOrientation = CFA_RIGHT;

    xfi[2].ulFieldOffset = FIELDOFFSET(FILERECORDCORE, cdateCreation);
    xfi[2].pszColumnTitle = nlsGetString(WPSI_CREATIONDATE); // "Creation date";
    xfi[2].ulDataType = CFA_DATE;
    xfi[2].ulOrientation = CFA_RIGHT;

    xfi[3].ulFieldOffset = FIELDOFFSET(FILERECORDCORE, cdateLastWrite);
    xfi[3].pszColumnTitle = nlsGetString(WPSI_LASTWRITEDATE); // "Last write date";
    xfi[3].ulDataType = CFA_DATE;
    xfi[3].ulOrientation = CFA_RIGHT;

    cnrhClearFieldInfos(hwndFilesCnr, FALSE);

    PFIELDINFO pFieldInfoLast =
        cnrhSetFieldInfos(hwndFilesCnr,
                          &xfi[0],
                          (sizeof(xfi) / sizeof(XFIELDINFO)),   // array item count
                          TRUE,          // draw lines
                          0);            // column index to return
    BEGIN_CNRINFO()
    {
        // set split bar
        cnrhSetSplitBarAfter(pFieldInfoLast);
        cnrhSetSplitBarPos(100);
        // switch view
        cnrhSetView(CV_DETAIL | CA_DETAILSVIEWTITLES);
    } END_CNRINFO(hwndFilesCnr);

    // now fill the container
    if (precc->pDBPackage)
    {
        ULONG           ulHeaderCount = 0;

        list<WIFileHeaderLI*>* pHeaderList;
        if (pHeaderList = precc->pDBPackage->GetFileHeaders(&ulHeaderCount))
        {
            // allocate memory for file record cores;
            // the number of items has been returned by datGetFileHeaders
            PFILERECORDCORE preccFileFirst;

            if (preccFileFirst = (PFILERECORDCORE)cnrhAllocRecords(
                                           hwndFilesCnr,
                                           sizeof(FILERECORDCORE),
                                           ulHeaderCount))
            {
                // preccFile will be used for filling the current recc;
                // the record cores are linked using the preccNextRecord fields
                PFILERECORDCORE preccFile = preccFileFirst;

                list<WIFileHeaderLI*>::iterator HeaderStart = pHeaderList->begin(),
                                                HeaderEnd = pHeaderList->end();

                for (; (HeaderStart != HeaderEnd); HeaderStart++)
                {
                    WIFileHeader *pHeaderThis = (**HeaderStart)._p;

                    strcpy(preccFile->szFilename, pHeaderThis->name);
                    preccFile->recc.pszIcon = preccFile->szFilename;
                    preccFile->ulSize = pHeaderThis->origsize;
                    // convert time_t to human-readable format
                    time_t lastwrite = pHeaderThis->lastwrite;
                    struct tm* pTimeBuf = localtime(&lastwrite);
                    preccFile->cdateLastWrite.day    = pTimeBuf->tm_mday;
                    preccFile->cdateLastWrite.month  = pTimeBuf->tm_mon + 1;
                    preccFile->cdateLastWrite.year   = pTimeBuf->tm_year + 1900;
                    time_t creation = pHeaderThis->creation;
                    pTimeBuf = localtime(&creation);
                    preccFile->cdateCreation.day   = pTimeBuf->tm_mday;
                    preccFile->cdateCreation.month = pTimeBuf->tm_mon + 1;
                    preccFile->cdateCreation.year  = pTimeBuf->tm_year + 1900;

                    // delete pHeaderThis;
                            // removed V0.9.14 (2001-07-07) [umoeller],
                            // list.clear() does this automatically

                    // next record
                    preccFile = (PFILERECORDCORE)(preccFile->recc.preccNextRecord);
                }

                // insert the record core linked list
                RECORDINSERT ri;
                memset(&ri, 0, sizeof(ri));
                ri.cb = sizeof(ri);
                ri.pRecordOrder = (PRECORDCORE)CMA_END;
                ri.pRecordParent = 0;
                ri.fInvalidateRecord = TRUE;
                ri.zOrder = CMA_TOP;
                ri.cRecordsInsert = ulHeaderCount;

                WinSendMsg(hwndFilesCnr,
                           CM_INSERTRECORD,
                           (MPARAM)preccFileFirst,
                           (MPARAM)&ri);
            }

            pHeaderList->clear();
            delete pHeaderList;
        }
    }
}

/*
 *@@ SetCnrConfigFieldInfos:
 *
 *@@added V0.9.2 (2000-02-19) [umoeller]
 */

VOID SetCnrConfigFieldInfos(HWND hwndConfigCnr)
{
    // set up config cnr details view
    XFIELDINFO      xfi[2];

    xfi[0].ulFieldOffset = FIELDOFFSET(DBCONFIGRECORD, pszConfigType);
    xfi[0].pszColumnTitle = "Type";
    xfi[0].ulDataType = CFA_STRING;
    xfi[0].ulOrientation = CFA_LEFT;

    xfi[1].ulFieldOffset = FIELDOFFSET(DBCONFIGRECORD, pszConfigData);
    xfi[1].pszColumnTitle = "Configuration data";
    xfi[1].ulDataType = CFA_STRING;
    xfi[1].ulOrientation = CFA_LEFT;

    cnrhClearFieldInfos(hwndConfigCnr, FALSE);

    PFIELDINFO pFieldInfoLast =
        cnrhSetFieldInfos(hwndConfigCnr,
                          &xfi[0],
                          (sizeof(xfi) / sizeof(XFIELDINFO)),   // array item count
                          TRUE,          // draw lines
                          0);            // column index to return
    BEGIN_CNRINFO()
    {
        // set split bar
        cnrhSetSplitBarAfter(pFieldInfoLast);
        cnrhSetSplitBarPos(100);
        // switch view
        cnrhSetView(CV_DETAIL | CA_DETAILSVIEWTITLES);
    } END_CNRINFO(hwndConfigCnr);
}

/*
 *@@ InsertConfigIntoCnr:
 *      this lists the configuration of the specified
 *      FEDBPackage by inserting DBCONFIGRECORD records
 *      into the specified container.
 *
 *      Returns the no. of records inserted or 0 if none.
 *
 *      Gets called both from UpdateListCnr when the right
 *      container is in "config" mode and from ConfirmDeinstall
 *      to list the package configuration to be de-installed.
 *
 *      This function creates a lot of strings which are used
 *      by the container record cores. All those strings are
 *      appended to the given Strings2FreeList. You should
 *      invoke FreeStringsList on that list later when the
 *      DBCONFIGRECORD's are no longer used (and only then,
 *      or otherwise the container control will crash).
 *
 *@@added V0.9.1 (2000-02-12) [umoeller]
 *@@changed V0.9.2 (2000-02-19) [umoeller]: sped up display
 *@@changed V0.9.2 (2000-02-19) [umoeller]: much shorter now since we're using virtual BSConfigBase methods
 */

ULONG InsertConfigIntoCnr(GUILocals *pLocals,
                          HWND hwndConfigCnr,       // in: cnr to receive DBCONFIGRECORD's
                          FEDBPackage *pDBPackage,  // in: package to add info for
                          list<BSString*> &Strings2FreeList) // in: list receiving strings to be freed later
{
    ULONG   ulrc = 0;

    list<BSConfigBase*>::iterator CfgStart = pDBPackage->_listUndoConfig.begin(),
                                  CfgEnd   = pDBPackage->_listUndoConfig.end();
    for (; CfgStart != CfgEnd; CfgStart++)
    {
        BSConfigBase *pManipThis = *CfgStart;

        PDBCONFIGRECORD pcrec
            = (PDBCONFIGRECORD)cnrhAllocRecords(hwndConfigCnr,
                                                sizeof(DBCONFIGRECORD),
                                                1);
        if (pcrec)
        {
            pcrec->pszConfigType = pManipThis->DescribeType();

            BSString *pstrConfig = new BSString;
            pstrConfig->assignUtf8(pLocals->_pCodecGui, pManipThis->DescribeData());
            Strings2FreeList.push_back(pstrConfig);
            pcrec->pszConfigData = pstrConfig->c_str();
            pcrec->pConfig = pManipThis;
            cnrhInsertRecords(hwndConfigCnr,
                              (PRECORDCORE)NULL,       // root level
                              (PRECORDCORE)pcrec,
                              FALSE,
                              NULL,
                              CRA_RECORDREADONLY,
                              1);
        }
        ulrc++;
    }

    // invalidate container
    cnrhInvalidateAll(hwndConfigCnr);

    return (ulrc);
}

/*
 *@@ fnwpListboxNoselect:
 *
 *@@added V0.9.5 (2000-08-26) [umoeller]
 */

MRESULT EXPENTRY fnwpListboxNoselect(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    switch (msg)
    {
        case WM_BUTTON1DOWN:
            WinSetFocus(HWND_DESKTOP, hwnd);
            mrc = (MPARAM)TRUE;
        break;

        case WM_BUTTON1UP:
        case WM_BUTTON1CLICK:
            mrc = (MPARAM)TRUE;
        break;

        default:
            mrc = G_pfnwpListboxOrig(hwnd, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ fnwpConfirmDeinstall:
 *      dialog proc for "Confirm deinstall" dlg (see
 *      ConfirmDeinstall).
 *
 *@@added V0.9.4 (2000-07-01) [umoeller]
 */

MRESULT EXPENTRY fnwpConfirmDeinstall(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    switch (msg)
    {
        case WM_INITDLG:        // mp2 has DatabaseView*
            WinSetWindowPtr(hwndDlg, QWL_USER, mp2);
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
        break;

        case WM_CONTROL:
        {
            USHORT usID = SHORT1FROMMP(mp1);
            USHORT usNotifyCode = SHORT2FROMMP(mp1);

            switch(usID)
            {
                case ID_WIDI_CFD_UNDOCONFIG:
                    if (    (usNotifyCode == BN_CLICKED)
                         || (usNotifyCode == BN_DBLCLICKED) // added V0.9.0
                       )
                    {
                        HWND hwndCnr = WinWindowFromID(hwndDlg, ID_WIDI_CFD_CONFIGCNR);
                        if (winhIsDlgItemChecked(hwndDlg, usID))
                        {
                            cnrhSelectAll(hwndCnr, TRUE);
                        }
                        else
                            cnrhSelectAll(hwndCnr, FALSE);
                    }
                break;

                case ID_WIDI_CFD_CONFIGCNR:
                {
                    HWND hwndCnr = WinWindowFromID(hwndDlg, ID_WIDI_CFD_CONFIGCNR);
                    if (usNotifyCode == CN_EMPHASIS)
                    {
                        PRECORDCORE pSel
                            = (PRECORDCORE)WinSendMsg(hwndCnr,
                                                      CM_QUERYRECORDEMPHASIS,
                                                      (MPARAM)CMA_FIRST,
                                                      (MPARAM)CRA_SELECTED);
                        winhSetDlgItemChecked(hwndDlg,
                                              ID_WIDI_CFD_UNDOCONFIG,
                                              ((pSel) && ((LONG)pSel != -1))
                                              );
                    }
                }
                break;
            }
        }
        break;

        case WM_HELP:
        {
            GUILocals *pLocals;
            if (pLocals = (GUILocals*)WinQueryWindowPtr(hwndDlg, QWL_USER))
                gshrDisplayHelp(pLocals, IDHI_CONFIRM_DEINSTALL);
        }
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ DEPENDENCYRECORD:
 *      extended record structure for dependency
 *      violation dialog (ConfirmDeinstall).
 *
 *@@added V0.9.12 (2001-05-17) [umoeller]
 */

typedef struct _DEPENDENCYRECORD
{
    RECORDCORE      recc;

            // "package"
    PCSZ    pszDepApplication,
            pszDepName,
            pszDepVersion,      // points to szDepVersion
            pszDepAuthor,

            // "requires"
            pszReqApplication,
            pszReqName,
            pszReqVersion,      // points to szReqVersion
            pszReqAuthor;

    // string buffers
    /* CHAR    szDepVersion[100],
            szReqVersion[100]; */

} DEPENDENCYRECORD, *PDEPENDENCYRECORD;

/*
 *@@ CreateBSU:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

string* CreateBSU(const string &str,
                  list<string*> &Strings2FreeList)
{
    string *pstr = new string(str);
    Strings2FreeList.push_back(pstr);
    return pstr;
}

/*
 *@@ ConfirmDeinstall:
 *      this displays the "Confirm Deinstall" dialog.
 *      pPackagesList has the packages to deinstall,
 *      depending on which the dialog will be filled
 *      with data and such.
 *
 *      This gets called in two situations:
 *      1)  from main() after the user has selected
 *          "Deinstall" in the "Archive installed" dialog;
 *      2)  from fnwpDatabaseMain also before deinstalling
 *          any packages.
 *
 *      This returns -1 if the user pressed Cancel.
 *      Otherwise a value >= 0 is returned, that is 0 ORed
 *      with the DBT_* flags in fe_database.h:
 *      -- DBT_DELETEFILES     0x01
 *      -- DBT_UNDOCONFIG      0x02
 *
 *@@changed V0.9.0 (99-11-02) [umoeller]: renamed function
 *@@changed V0.9.0 (99-11-02) [umoeller]: no longer required callback, only used in this file
 *@@changed V0.9.1 (2000-01-05) [umoeller]: "Files" checkbox wasn't working; fixed
 *@@changed V0.9.1 (2000-01-05) [umoeller]: reworked configuration flags
 *@@changed V0.9.1 (2000-01-05) [umoeller]: reworked configuration display; now using container
 *@@changed V0.9.2 (2000-03-10) [umoeller]: "undo config" wasn't checked right for multiple packages; fixed
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added individual undo-config selection
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added dependency checks
 *@@changed V0.9.12 (2001-05-17) [umoeller]: now displaying detailed dependencies, allowing ignore
 */

LONG ConfirmDeinstall(HWND hwndOwner,                   // in: dialog owner
                      GUILocals *pLocals,
                      list<FEDBPackage*>* pPackagesList) // in: package list to deinstall.
{
    LONG   lrc = -1;

    // list of strings to be freed
    list<BSString*> Strings2FreeList(STORE);

    // first, load dependency violations dialog;
    // while we're checking the dependencies, we add
    // records to the violations container...

    // note, the dialog is only shown if any violations
    // were actually found!

    HWND hwndConfirm = gshrLoadDlg(HWND_DESKTOP,
                                   G_hwndMainFrame,
                                   WinDefDlgProc,
                                   NULLHANDLE,
                                   ID_WID_DEPENDENCIES,
                                   NULL);
    HWND hwndCnr = WinWindowFromID(hwndConfirm, ID_WIDI_PACKAGESCNR);

    winhSetControlsFont(hwndConfirm,
                        0,
                        1000,
                        gshrQueryDefaultFont());

    // set up cnr details view
    XFIELDINFO      xfi[10];
    memset(&xfi, 0, sizeof(xfi));

    int i = 0;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszDepApplication);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_APPLICATION), // application
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszDepName);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_PACKAGENAME); // "Package name";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszDepVersion);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_VERSION); // "Version";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_RIGHT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszDepAuthor);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_AUTHOR); // "Author";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszReqApplication);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_APPLICATION), // application
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszReqName);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_PACKAGENAME); // "Package name";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszReqVersion);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_VERSION); // "Version";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_RIGHT;

    xfi[i].ulFieldOffset = FIELDOFFSET(DEPENDENCYRECORD, pszReqAuthor);
    xfi[i].pszColumnTitle = nlsGetString(WPSI_AUTHOR); // "Author";
    xfi[i].ulDataType = CFA_STRING;
    xfi[i++].ulOrientation = CFA_LEFT;

    PFIELDINFO pFieldInfoLast =
             cnrhSetFieldInfos(hwndCnr,
                               &xfi[0],
                               i,        // array item count
                               TRUE,     // draw lines
                               3);       // PFIELDINFO index to return

    BEGIN_CNRINFO()
    {
        SWP swpCnr;
        WinQueryWindowPos(hwndCnr, &swpCnr);
        // set split bar
        cnrhSetSplitBarAfter(pFieldInfoLast);
        cnrhSetSplitBarPos(swpCnr.cx / 2);
        // switch view
        cnrhSetView(CV_DETAIL | CA_DETAILSVIEWTITLES);
    } END_CNRINFO(hwndCnr);

    /*
     * check dependencies
     *
     */

    ULONG cRequiring = 0;
    list<FEDBPackage*>::iterator PckStart = pPackagesList->begin(),
                                 PckEnd = pPackagesList->end();

    string strDependencies;

    for (; PckStart != PckEnd; PckStart++)
    {
        FEDBPackage *pPackageThis = *PckStart;
        /* FEPackageID PckIDThis(pLocals,
                              pPackageThis, __FILE__, __LINE__,
                              FALSE);    // don't allow short format
           */

        FEPackageID &PckIDThis = pPackageThis->_PckID;

        list<FEDBPackage*> RequiringList(SHADOW);       // do not free!!

        ULONG cThis = pLocals->_pDatabase->IsPackageRequired(PckIDThis,
                                                     RequiringList,
                                                     // packages to be ignored:
                                                     pPackagesList);
        if (cThis)
        {
            // dependency violations:

            cRequiring += cThis;

            // create temporary strings
            BSString *pstrDepApplication
                = CreateBSU(PckIDThis._strApplicationGui,
                            Strings2FreeList);
            BSString *pstrDepName
                = CreateBSU(PckIDThis._strPackageNameGui,
                            Strings2FreeList);
            BSString *pstrDepAuthor
                = CreateBSU(PckIDThis._strAuthorGui,
                            Strings2FreeList);

            // allocate records for this package

            PDEPENDENCYRECORD precFirst
                = (PDEPENDENCYRECORD)cnrhAllocRecords(hwndCnr,
                                                      sizeof(DEPENDENCYRECORD),
                                                      cThis);
            if (precFirst)
            {
                PDEPENDENCYRECORD precThis = precFirst;
                ULONG ul;

                list<FEDBPackage*>::iterator ReqStart = RequiringList.begin(),
                                             ReqEnd = RequiringList.end();
                for (;
                     ReqStart != ReqEnd;
                     ReqStart++)
                {
                    FEDBPackage *pRequires = *ReqStart;

                    precThis->pszDepApplication = pstrDepApplication->c_str();
                    precThis->pszDepName = pstrDepName->c_str();
                    precThis->pszDepVersion = PckIDThis._strVersion.c_str(); // precThis->szDepVersion;
                    precThis->pszDepAuthor = pstrDepAuthor->c_str();

                    FEPackageID &ReqID = pRequires->_PckID;

                    // create temporary strings
                    BSString *pstrReqApplication
                         = CreateBSU(ReqID._strApplicationGui,
                                     Strings2FreeList);
                    BSString *pstrReqName
                         = CreateBSU(ReqID._strPackageNameGui,
                                     Strings2FreeList);
                    BSString *pstrReqAuthor
                         = CreateBSU(ReqID._strAuthorGui,
                                     Strings2FreeList);

                    precThis->pszReqApplication = pstrReqApplication->c_str();
                    precThis->pszReqName = pstrReqName->c_str();
                    precThis->pszReqVersion = PckIDThis._strVersion.c_str(); // precThis->szReqVersion;
                    precThis->pszReqAuthor = pstrReqAuthor->c_str();

                    // next record in allocated list
                    precThis = (PDEPENDENCYRECORD)precFirst->recc.preccNextRecord;
                }

                // now insert the records
                cnrhInsertRecords(hwndCnr,
                                  NULL,
                                  (PRECORDCORE)precFirst,
                                  TRUE,
                                  NULL,
                                  CRA_RECORDREADONLY,
                                  cThis);
            }
        }
    }

    if (cRequiring)
    {
        // any dependencies found:
        // ask user if these violations should be ignored

        if (WinProcessDlg(hwndConfirm) == DID_OK)
            // deinstall anyway:
            cRequiring = 0;
    }

    WinDestroyWindow(hwndConfirm);

    if (!cRequiring)
    {
        // no dependencies, or deinstall anyway:

        if (pPackagesList)
        {
            HWND hwndDlg = gshrLoadDlg(HWND_DESKTOP,
                                       hwndOwner,
                                       fnwpConfirmDeinstall,
                                       NULLHANDLE, ID_WID_CONFIRMDEINSTALL,
                                       pLocals);
            winhCenterWindow(hwndDlg);
            winhSetControlsFont(hwndDlg, 0, 2000,
                        gshrQueryDefaultFont());

            BOOL        fHasFiles = FALSE;
            ULONG       ulConfigItems = 0;
            HWND        hwndPckLbox = WinWindowFromID(hwndDlg, ID_WIDI_CFD_PACKAGESLIST),
                        hwndFilesLbox = WinWindowFromID(hwndDlg, ID_WIDI_CFD_FILESLIST),
                        hwndConfigCnr = WinWindowFromID(hwndDlg, ID_WIDI_CFD_CONFIGCNR);

            // subclass listboxes
            G_pfnwpListboxOrig = WinSubclassWindow(hwndPckLbox, fnwpListboxNoselect);
            G_pfnwpListboxOrig = WinSubclassWindow(hwndFilesLbox, fnwpListboxNoselect);

            SetCnrConfigFieldInfos(hwndConfigCnr);

            // go thru all packages to collect files and config data
            list<FEDBPackage*>::iterator PckStart = pPackagesList->begin(),
                                         PckEnd = pPackagesList->end();
            for (; PckStart != PckEnd; PckStart++)
            {
                FEDBPackage* ppi = *PckStart;
                string *pstr = new string;
                pstr->assignUtf8(pLocals->_pCodecGui, ppi->QueryPackageID());
                Strings2FreeList.push_back(pstr);

                WinInsertLboxItem(hwndPckLbox, LIT_END, pstr->c_str());

                // check files
                if (ppi->_pszFilesList)
                {
                    ULONG ulHeaderCount = 0;
                    list<WIFileHeaderLI*>* pFileHeadersList = ppi->GetFileHeaders(
                                                                          &ulHeaderCount);
                    if (pFileHeadersList)
                    {
                        list<WIFileHeaderLI*>::iterator HeaderFirst = pFileHeadersList->begin(),
                                                        HeaderLast = pFileHeadersList->end();
                        for (; HeaderFirst != HeaderLast; HeaderFirst++)
                        {
                            WIFileHeader *p = (**HeaderFirst)._p;
                            string strFullFile;
                            strFullFile.assignUtf8(pLocals->_pCodecGui, ppi->QueryTargetPath());
                            if (strFullFile[strFullFile.length() - 1] != '\\')
                                strFullFile += "\\";
                            strFullFile += p->name;
                            WinInsertLboxItem(hwndFilesLbox, LIT_END, strFullFile.c_str());
                            // delete *HeaderFirst;
                        }

                        pFileHeadersList->clear();
                        delete pFileHeadersList;
                        fHasFiles = TRUE;
                    }
                }

                ulConfigItems += InsertConfigIntoCnr(pLocals,
                                                     hwndConfigCnr,
                                                     ppi,
                                                     Strings2FreeList);
                cnrhSelectAll(hwndConfigCnr, TRUE);
            }

            winhEnableDlgItem(hwndDlg, ID_WIDI_CFD_DELETEFILES, fHasFiles);
            winhSetDlgItemChecked(hwndDlg, ID_WIDI_CFD_DELETEFILES, fHasFiles);

            winhEnableDlgItem(hwndDlg, ID_WIDI_CFD_UNDOCONFIG, (ulConfigItems != 0));
            winhSetDlgItemChecked(hwndDlg, ID_WIDI_CFD_UNDOCONFIG, (ulConfigItems != 0));

            // go!!!
            if (WinProcessDlg(hwndDlg) == DID_OK)
            {
                lrc = 0;
                if (winhIsDlgItemChecked(hwndDlg, ID_WIDI_CFD_DELETEFILES))
                    lrc |= DBT_DELETEFILES;
                if (winhIsDlgItemChecked(hwndDlg, ID_WIDI_CFD_UNDOCONFIG))
                {
                    // undo config (any items selected in container):
                    lrc |= DBT_UNDOCONFIG;

                    // go thru all container records and select config items
                    // accordingly

                    PDBCONFIGRECORD pcrec = (PDBCONFIGRECORD)0;
                    USHORT fsCmd = CMA_FIRST;
                    do
                    {
                        pcrec = (PDBCONFIGRECORD)WinSendMsg(hwndConfigCnr,
                                                            CM_QUERYRECORD,
                                                            (MPARAM)pcrec,
                                                                    // null on first loop
                                                            MPFROM2SHORT(fsCmd,
                                                                         CMA_ITEMORDER));
                        fsCmd = CMA_NEXT;
                        if ((pcrec) && ((LONG)pcrec != -1))
                        {
                            // first or next record found:
                            // mark config item as selected or not selected
                            // according to recc selection
                            pcrec->pConfig->Select((pcrec->recc.flRecordAttr & CRA_SELECTED)
                                                       != 0);
                        }
                        else
                            break;
                    } while (TRUE);
                }
            }
            WinDestroyWindow(hwndDlg);
        }
    }

    Strings2FreeList.clear();

    return (lrc);
}

/* ******************************************************************
 *
 *   "Packages" subwindow
 *
 ********************************************************************/

/*
 *@@ FreeDatabaseRecord:
 *      this removes preccRemove from the given container
 *      and frees all associated data. This is used during
 *      de-installation to really remove a record.
 */

VOID FreeDatabaseRecord(DatabaseView *pView,
                        HWND hwndCnr,
                        PDATABASERECORD preccRemove)
{
    if (preccRemove->pDBPackage)
        // is package:
        lstRemoveItem(&pView->_PckRecordsList, preccRemove);
    else
        // is application:
        lstRemoveItem(&pView->_AppRecordsList, preccRemove);

    if (preccRemove->pszApplicationU)
        free(preccRemove->pszApplicationU);

    WinSendMsg(hwndCnr,
               CM_REMOVERECORD,
               (MPARAM)&preccRemove,
               MPFROM2SHORT(1,
                            CMA_FREE | CMA_INVALIDATE));
}

/*
 *@@ ClearDatabaseContainer:
 *      this removes all record cores from the given
 *      container. It is assumed that the container
 *      uses DATABASERECORDs for its record cores.
 *      The records are removed only, not freed.
 *
 *      This is used when the display changes, but the
 *      record cores will still be used.
 *
 *@@changed V0.9.0 (99-11-04) [umoeller]: fixed the age-old display bug when switching between views
 *@@changed V0.9.2 (2000-02-19) [umoeller]: sped up display
 */

VOID ClearDatabaseContainer(DatabaseView *pView,
                            HWND hwndCnr,
                            BOOL fFreeAppsToo)
{
    PLISTNODE pNode;
    for (pNode = lstQueryFirstNode(&pView->_PckRecordsList);
         pNode;
         pNode = pNode->pNext)
    {
        PDATABASERECORD preccRemove = (PDATABASERECORD)pNode->pItemData; // *RecStart;
        WinSendMsg(hwndCnr,
                   CM_REMOVERECORD,
                   (MPARAM)&preccRemove,
                   MPFROM2SHORT(1,
                                0)); // CMA_INVALIDATE));  // no CMA_FREE, we still need the record
    }

    if (fFreeAppsToo)
    {
        for (pNode = lstQueryFirstNode(&pView->_AppRecordsList);
             pNode;
             pNode = pNode->pNext)
        {
            PDATABASERECORD preccRemove = (PDATABASERECORD)pNode->pItemData; // *RecStart;
            WinSendMsg(hwndCnr,
                       CM_REMOVERECORD,
                       (MPARAM)&preccRemove,
                       MPFROM2SHORT(1,
                                    0)); // CMA_INVALIDATE));  // no CMA_FREE, we still need the record
        }
    }

    cnrhInvalidateAll(hwndCnr);
}

#define WM_FILLCNR      WM_USER

/*
 *@@ FillDatabaseContainer:
 *      this fills the container with all applications/packages.
 *
 *      Depending on pLocals->_GuiSettings.ulLeftView, this is either
 *      a Tree view or a Details view.
 *
 *      This gets called both after the main window has been
 *      created and if the database view changed.
 *
 *@@changed V0.9.2 (2000-02-19) [umoeller]: sped up display
 *@@changed V0.9.9 (2001-02-28) [umoeller]: apps tree was missing items if same app had differing authors
 */

VOID FillDatabaseContainer(DatabaseView *pView,
                           HWND hwndCnr,        // in: cnr to fill
                           ULONG ulNewView)     // in: either ID_WIMI_DATABASE_TREEVIEW or
                                                // ID_WIMI_DATABASE_DETAILSVIEW
{
    GUIWaitPointer wp;

    GUILocals *pLocals = &pView->_Locals;
    ClearDatabaseContainer(pView,
                           hwndCnr,
                           // free apps too?
                           pLocals->_GuiSettings.ulLeftView == ID_WIMI_DATABASE_TREEVIEW);

    pLocals->_GuiSettings.ulLeftView = ulNewView;

    /*
     * ID_WIMI_DATABASE_TREEVIEW:
     *
     */

    if (pLocals->_GuiSettings.ulLeftView == ID_WIMI_DATABASE_TREEVIEW)
    {
        // CnrInfo.cxTreeIndent = 15;

        // switch cnr to Details view
        BEGIN_CNRINFO()
        {
            // set bitmap size
            cnrhSetBmpOrIconSize(16, 16);
            // switch view
            cnrhSetView(CV_TREE | CV_ICON | CA_TREELINE | CV_MINI);
                                // | CA_DRAWBITMAP);
        } END_CNRINFO(hwndCnr);

        // insert all application record cores to root level
        PLISTNODE pNode;
        FOR_ALL_NODES(&pView->_AppRecordsList, pNode)
        {
            PDATABASERECORD precc = (PDATABASERECORD)pNode->pItemData; // *AppsStart;
            cnrhInsertRecords(hwndCnr,
                              (PRECORDCORE)NULL,       // root level
                              (PRECORDCORE)precc,
                              FALSE,
                              precc->pcszApplication,
                              CRA_RECORDREADONLY | CRA_COLLAPSED,
                              1);
        }

        // insert all packages as children of the applications
        FOR_ALL_NODES(&pView->_AppRecordsList, pNode)
        {
            PDATABASERECORD preccApplication = (PDATABASERECORD)pNode->pItemData; // (*AppsStart);

            // AppsStart now has the current application recc;
            // go thru the package recc list and add all package
            // records which have this application

            PLISTNODE pRecNode;
            FOR_ALL_NODES(&pView->_PckRecordsList, pRecNode)
            {
                PDATABASERECORD preccPck = (PDATABASERECORD)pRecNode->pItemData; // (*ReccStart);

                if (!strcmp(preccPck->pcszApplication, preccApplication->pcszApplication))
                {
                    cnrhInsertRecords(hwndCnr,
                                      (PRECORDCORE)preccApplication,   // parent
                                      (PRECORDCORE)preccPck,
                                      FALSE,
                                      preccPck->pcszPackageName,
                                      CRA_RECORDREADONLY | CRA_COLLAPSED,
                                      1);
                }
            }
        }
    }

    /*
     * ID_WIMI_DATABASE_DETAILSVIEW:
     *
     */

    else if (pLocals->_GuiSettings.ulLeftView == ID_WIMI_DATABASE_DETAILSVIEW)
    {
        // switch cnr to Details view
        BEGIN_CNRINFO()
        {
            // set bitmap size
            cnrhSetBmpOrIconSize(16, 16);
            // switch view
            cnrhSetView(CV_DETAIL | CA_DETAILSVIEWTITLES);
        } END_CNRINFO(hwndCnr);

        ULONG cRecords = lstCountItems(&pView->_PckRecordsList);
        PRECORDCORE *pArray = (PRECORDCORE *)malloc(   cRecords
                                                     * sizeof(PRECORDCORE));
        PRECORDCORE *pThis = pArray;

        // the container already has the details fields we need,
        // so just insert the records now
        PLISTNODE pNode;
        FOR_ALL_NODES(&pView->_PckRecordsList, pNode)
        {
            PDATABASERECORD preccPck = (PDATABASERECORD)pNode->pItemData; // (*ReccStart);
            *pThis = (PRECORDCORE)preccPck;
            pThis++;
            /* cnrhInsertRecords(hwndCnr,
                              (PRECORDCORE)NULL,       // no parent for Details view
                              (PRECORDCORE)preccPck,
                              FALSE,
                              preccPck->pcszPackageName,
                              CRA_RECORDREADONLY | CRA_COLLAPSED,
                              1); */
        }

        RECORDINSERT ri;
        ri.cb = sizeof(RECORDINSERT);
        ri.pRecordOrder = (PRECORDCORE)CMA_FIRST;
        ri.pRecordParent = NULL;
        ri.fInvalidateRecord = TRUE;
        ri.zOrder = CMA_TOP;
        ri.cRecordsInsert = cRecords;
        WinSendMsg(hwndCnr,
                   CM_INSERTRECORDARRAY,
                   (MPARAM)pArray,
                   (MPARAM)&ri);

        free(pArray);
    }

    cnrhInvalidateAll(hwndCnr);
}

#define CONVERT_DB_REC(p) ((GuiDatabaseRecord*)((char*)p - (char*)FIELDOFFSET(GuiDatabaseRecord, recc)))

/*
 *@@ DBPackagesEmphasis:
 *      implementation for WM_CONTROL and CN_EMPHASIS
 *      in fnwpDBPackagesDlg.
 *
 *@@added V0.9.12 (2001-05-17) [umoeller]
 */

VOID DBPackagesEmphasis(DatabaseView *pView,
                        HWND hwndDlg,
                        PNOTIFYRECORDEMPHASIS pnre, // in: mp2 of WM_CONTROL
                        BOOL fClearingContainers)
{
    GUILocals *pLocals = &pView->_Locals;
    PDATABASERECORD precc;

    if (    (pnre)
         && (pnre->fEmphasisMask & CRA_SELECTED)
         && (precc = (PDATABASERECORD)pnre->pRecord)
       )
    {
        // set the currently selected record
        // CHAR    szVersion[40] = "";
        PCSZ    pcszVersion = NULL;
        BSString strInstallDateTime;
        const char *pcszTargetDir = 0;
        CHAR    szFiles[200] = "";
        string strTarget;

        // set the following for both apps and pcks
        WinSetDlgItemText(hwndDlg, ID_WIDI_DB_AUTHOR,
                          precc->pcszAuthor);

        // rule out app records
        FEDBPackage*    pDBPackage;
        if (pDBPackage = precc->pDBPackage)
        {
            /* sprintf(szVersion, "%d.%d.%d",
                    precc->ulVersionMajor,
                    precc->ulVersionMinor,
                    precc->ulVersionRevision); */
            pcszVersion = precc->pcszVersion;
            strTarget.assignUtf8(pLocals->_pCodecGui, pDBPackage->QueryTargetPath());
            pcszTargetDir = strTarget.c_str();
            sprintf(szFiles, "%d",
                    pDBPackage->_PackHeader.files);

            CHAR szDate[50], szTime[50];
            nlsDateTime(szDate, szTime,
                        &pDBPackage->_InstallDateTime,
                        G_ulDateFormat,
                        G_szDateSep[0],
                        G_ulTimeFormat,
                        G_szTimeSep[0]);
            strInstallDateTime = szDate;
            strInstallDateTime += " ";
            strInstallDateTime += szTime;
        }

        WinSetDlgItemText(hwndDlg, ID_WIDI_DB_VERSION,
                          pcszVersion);
        WinSetDlgItemText(hwndDlg, ID_WIDI_DB_INSTALLDATE,
                          strInstallDateTime.c_str());
        WinSetDlgItemText(hwndDlg, ID_WIDI_DB_TARGETDIR,
                          pcszTargetDir);
        WinSetDlgItemText(hwndDlg, ID_WIDI_DB_FILES,
                          szFiles);

        if (!fClearingContainers) // V0.9.9 (2001-02-28) [umoeller]
        {
            HWND hwndCnr = WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR);

            // count the no. of selected records in
            // the left cnr
            PDATABASERECORD precSel;
            if (    (precSel = (PDATABASERECORD)WinSendMsg(hwndCnr,
                                                     CM_QUERYRECORDEMPHASIS,
                                                     (MPARAM)CMA_FIRST,
                                                     (MPARAM)CRA_SELECTED))
                 && ((ULONG)precSel != -1)
                    // get next record to check if we have more
                    // than one selected
                 && (NULL == WinSendMsg(hwndCnr,
                                        CM_QUERYRECORDEMPHASIS,
                                        (MPARAM)precSel,
                                        (MPARAM)CRA_SELECTED))
               )
            {
                // ok, exactly one record selected:
                if (pView->_preccDBSelected != precSel)
                {
                    // store record
                    pView->_preccDBSelected = precSel;
                    UpdateListCnr(pView, precSel);
                }
            }
            else
            {
                // zero or multiple:
                // clear right container
                pView->_preccDBSelected = NULL;
                UpdateListCnr(pView, NULL);
            }
        }
    }
}

/*
 *@@ DBVerifyOrDeinstall:
 *      implementation for WM_COMMAND in fnwpDBPackagesDlg
 *      with the verify or deinstall apps or pcks commands.
 *
 *@@added V0.9.12 (2001-05-17) [umoeller]
 */

VOID DBVerifyOrDeinstall(DatabaseView *pView,
                         ULONG ulCmd)
{
    GUILocals *pLocals = &pView->_Locals;

    // list of packages to remove or verify
    // create a new list V0.9.14 (2001-07-07) [umoeller]
    list<FEDBPackage*> *pPckInfoList = new list<FEDBPackage*>(SHADOW);
    BOOL fDelete = TRUE;

    // build list
    switch (ulCmd)
    {
        // work on application:
        case ID_WIMI_DATABASE_VERIFYAPP:
        case ID_WIMI_DATABASE_DEINSTALLAPP:
        {
            // find package infos which match author
            // and application by building a comparison
            // string

            const char *pcszApp = pView->_preccLeftSource->pszApplicationU;
            list<FEDBPackage*>::iterator PckStart = pLocals->_pDatabase->_DBPackagesList.begin(),
                                         PckEnd = pLocals->_pDatabase->_DBPackagesList.end();
            for (; PckStart != PckEnd; PckStart++)
            {
                FEDBPackage* ppiThis = *PckStart;
                const char *pcszThis;
                if (    (pcszThis = ppiThis->_PckID._ustrApplication.GetBuffer())
                     && (!strcmp(pcszApp, pcszThis))
                   )
                    // matches: store this package in the list
                    pPckInfoList->push_back(ppiThis);
            }
        }
        break;

        // work on packages:
        case ID_WIMI_DATABASE_DEINSTALLPCK:
        case ID_WIMI_DATABASE_VERIFYPCK:
        {
            // V0.9.12 (2001-05-17) [umoeller]: go thru all
            // selected records
            HWND hwndCnr = WinWindowFromID(pView->_hwndDBPackagesDlg, ID_WIDI_PACKAGESCNR);
            ULONG ulSelection = 0;
            PDATABASERECORD precSel
                = (PDATABASERECORD)cnrhQuerySourceRecord(hwndCnr,
                                                   (PRECORDCORE)pView->_preccLeftSource,
                                                   &ulSelection);
            if (    (precSel)
                 && (precSel->pDBPackage)
               )
            {
                pPckInfoList->push_back(precSel->pDBPackage);

                if (ulSelection == SEL_MULTISEL)
                {
                    while (precSel = (PDATABASERECORD)cnrhQueryNextSelectedRecord(hwndCnr,
                                                                    (PRECORDCORE)precSel))
                    {
                        if (precSel->pDBPackage)
                            pPckInfoList->push_back(precSel->pDBPackage);
                    }
                }
            }
        }
    } // end switch (ulCmd)

    // confirmations/setup

    switch (ulCmd)
    {
        // work on application:
        case ID_WIMI_DATABASE_VERIFYAPP:
        case ID_WIMI_DATABASE_VERIFYPCK:
            guiStartVerify(pLocals,
                           pPckInfoList,
                           FALSE);      // non-modal
            fDelete = FALSE;
        break;

        case ID_WIMI_DATABASE_DEINSTALLAPP:
        case ID_WIMI_DATABASE_DEINSTALLPCK:
        {
            LONG lrc = ConfirmDeinstall(G_hwndMainFrame, pLocals, pPckInfoList);
            if (lrc >= 0)
            {
                guiStartRemove(pLocals,
                               pPckInfoList,
                               lrc,
                               FALSE);      // non-modal
                fDelete = FALSE;
            }
        }
        break;
    }

    if (fDelete)
        delete pPckInfoList;
}

/*
 *@@ fnCompareApps:
 *      linklist.c sort function.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 *@@changed V0.9.20 (2002-07-16) [umoeller]: now using encicmp for case-insensitivity, this sorts the database as Unicode
 */

signed short XWPENTRY fnCompareApps(void* pItem1, void* pItem2, void* pStorage)
{
    if (    (pItem1 = ((PDATABASERECORD)pItem1)->pszApplicationU)
         && (pItem2 = ((PDATABASERECORD)pItem2)->pszApplicationU)
       )
    {
        /* switch (WinCompareStrings((HAB)pStorage,
                                  0,
                                  0,
                                  (((PDATABASERECORD)pItem1)->pcszApplication),
                                  (((PDATABASERECORD)pItem2)->pcszApplication),
                                  0))
        {
            case WCS_LT: return (-1);
            case WCS_GT: return (1);
        } */

        // WinCompareStrings is sooooooooooooooooooooo slow....
        int i = encicmp((PCSZ)pItem1,       // was: strcmp
                        (PCSZ)pItem2);
        if (i < 0)
            return -1;
        if (i > 0)
            return +1;
    }

    return (0);
}

/*
 *@@ fnwpDBPackagesDlg:
 *      window procedure for the left "Packages" subdialog.
 *
 *      This one processes all the commands from the
 *      context menus (deinstall, verify etc.).
 *
 *@@changed V0.9.0 (99-10-24) [umoeller]: reworked completely for split view
 *@@changed V0.9.0 (99-11-02) [umoeller]: reworked database thread callbacks
 *@@changed V0.9.2 (2000-02-19) [umoeller]: added "View" menu support
 *@@changed V0.9.2 (2000-03-10) [umoeller]: added keyboard focus support
 *@@changed V0.9.4 (2000-06-30) [umoeller]: right split view was empty initially
 *@@changed V0.9.5 (2000-08-10) [umoeller]: F1 displays help now
 *@@changed V0.9.9 (2001-02-28) [umoeller]: now suppressing flicker on cnr view change
 *@@changed V0.9.12 (2001-05-17) [umoeller]: extracted DBPackagesEmphasis and DBVerifyOrDeinstall for support of multiple selections
 *@@changed V0.9.18 (2002-03-08) [umoeller]: finally sorting the packages...
 */

MRESULT EXPENTRY fnwpDBPackagesDlg(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT     mrc = 0;

    DatabaseView *pView = (DatabaseView *)WinQueryWindowPtr(hwndDlg, QWL_USER);
    GUILocals*   pLocals;
    if (pView)
        pLocals = &pView->_Locals;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *      mp2 has DatabaseView*.
         */

        case WM_INITDLG:
        {
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            WinSetWindowPtr(hwndDlg, QWL_USER, mp2);
            pView = (DatabaseView*)mp2;

            // set font for all the dialog controls
            winhSetControlsFont(hwndDlg, -2, 2000,
                        gshrQueryDefaultFont());

            gshrMakeHeading(hwndDlg, ID_WIDI_HEADING);

            // prepare winhAdjustControls
            memset(&pView->_xacPackagesDlg, 0, sizeof(XADJUSTCTRLS));
            winhAdjustControls(hwndDlg,             // dialog
                               G_ampPackagesDlg,    // MPARAMs array
                               sizeof(G_ampPackagesDlg) / sizeof(MPARAM), // items count
                               NULL,                // init
                               &pView->_xacPackagesDlg);  // storage area

            // set up the container for Details view
            HWND hwndDatabaseCnr = WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR);

            // 1) set up data for Details view columns
            XFIELDINFO      xfi[10];
            memset(&xfi, 0, sizeof(xfi));

            int i = 0;

            xfi[i].ulFieldOffset = FIELDOFFSET(RECORDCORE, hptrIcon);
            xfi[i].pszColumnTitle = "";
            xfi[i].ulDataType = CFA_BITMAPORICON;       // it's an icon, really
            xfi[i++].ulOrientation = CFA_LEFT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, pcszApplication);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_APPLICATION), // application
            xfi[i].ulDataType = CFA_STRING;
            xfi[i++].ulOrientation = CFA_LEFT;

            xfi[i].ulFieldOffset = FIELDOFFSET(DATABASERECORD, pcszPackageName);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_PACKAGENAME); // "Package name";
            xfi[i].ulDataType = CFA_STRING;
            xfi[i++].ulOrientation = CFA_LEFT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, pcszVersion);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_VERSION); // "Version";
            xfi[i].ulDataType = CFA_STRING;
            xfi[i++].ulOrientation = CFA_RIGHT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, pcszAuthor);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_AUTHOR); // "Author";
            xfi[i].ulDataType = CFA_STRING;
            xfi[i++].ulOrientation = CFA_LEFT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, pcszTargetPath);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_TARGETPATH); // "Target directory";
            xfi[i].ulDataType = CFA_STRING;
            xfi[i++].ulOrientation = CFA_LEFT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, ulFiles);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_FILES); // "Files";
            xfi[i].ulDataType = CFA_ULONG;
            xfi[i++].ulOrientation = CFA_RIGHT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, ulTotalSize);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_SIZE); // "Size";
            xfi[i].ulDataType = CFA_ULONG;
            xfi[i++].ulOrientation = CFA_RIGHT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, cdateInstall);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_INSTALLDATE); // "Install date";
            xfi[i].ulDataType = CFA_DATE;
            xfi[i++].ulOrientation = CFA_RIGHT;

            xfi[i].ulFieldOffset =   FIELDOFFSET(DATABASERECORD, ctimeInstall);
            xfi[i].pszColumnTitle = nlsGetString(WPSI_INSTALLTIME); // "Install time";
            xfi[i].ulDataType = CFA_TIME;
            xfi[i++].ulOrientation = CFA_RIGHT;

            PFIELDINFO pFieldInfoLast =
                     cnrhSetFieldInfos(hwndDatabaseCnr,
                                       &xfi[0],
                                       i,        // array item count
                                       TRUE,     // draw lines
                                       3);       // PFIELDINFO index to return

            // set split bar
            BEGIN_CNRINFO()
            {
                cnrhSetSplitBarAfter(pFieldInfoLast);
                cnrhSetSplitBarPos(200);
                // cnrhSetSortFunc(fnSortDBPackages);
            } END_CNRINFO(hwndDatabaseCnr);

            WinPostMsg(hwndDlg,
                       WM_FILLCNR,
                       0, 0);
        }
        break;

        /*
         * WM_FILLCNR:
         *      posted by WM_INITDLG to fill the left container.
         *      This is delayed because at the point WM_INITDLG
         *      comes in, we have no message loop yet and updating
         *      the right split view doesn't work.
         *      V0.9.4 (2000-06-30) [umoeller]
         */

        case WM_FILLCNR:
        {
            HWND hwndDatabaseCnr = WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR);

            // create record cores for all applications
            list<FEDBPackage*>::iterator PckStart = pLocals->_pDatabase->_DBPackagesList.begin(),
                                         PckEnd = pLocals->_pDatabase->_DBPackagesList.end();

            for (; (PckStart != PckEnd); PckStart++)
            {
                FEDBPackage *pDBPackage = *PckStart;
                const char *pcszApp;

                // check if this application exists in the list already
                if (pcszApp = pDBPackage->_PckID._ustrApplication.GetBuffer())
                {
                    BOOL fFound = FALSE;
                    PLISTNODE pNode;
                    FOR_ALL_NODES(&pView->_AppRecordsList, pNode)
                    {
                        PDATABASERECORD pAppThis = (PDATABASERECORD)pNode->pItemData; // *AppsStart;
                        if (!strcmp(pAppThis->pszApplicationU, pcszApp))
                        {
                            fFound = TRUE;
                            break;
                        }
                    }

                    if (!fFound)
                    {
                        // not found:
                        // create application record core
                        PDATABASERECORD preccApplication = CreateDBAppRecord(pLocals,
                                                                             hwndDatabaseCnr,
                                                                             pDBPackage);

                        // store in global list
                        lstAppendItem(&pView->_AppRecordsList, preccApplication);

                    } // end if (!fFound)
                }
            } // for (; (PckStart != PckEnd); PckStart++)

            // 3) create records for all packages
            PckStart = pLocals->_pDatabase->_DBPackagesList.begin();
            PckEnd = pLocals->_pDatabase->_DBPackagesList.end();
            for (; (PckStart != PckEnd); PckStart++)
            {
                PDATABASERECORD preccPck = CreateDBPckRecord(pLocals,
                                                             hwndDatabaseCnr,
                                                             *PckStart);
                // store in global list
                lstAppendItem(&pView->_PckRecordsList, preccPck);
            } // for (; (PckStart != PckEnd); PckStart++)

            // now sort the lists according to names... let's not
            // let the container do the sorting, it is so damn slow
            lstQuickSort(&pView->_AppRecordsList,
                         fnCompareApps,
                         (PVOID)pLocals->_habThread1);
            lstQuickSort(&pView->_PckRecordsList,
                         fnCompareApps,
                         (PVOID)pLocals->_habThread1);

            FillDatabaseContainer(pView,
                                  hwndDatabaseCnr,
                                  pLocals->_GuiSettings.ulLeftView);
        }
        break;

        /*
         * WM_CONTROL:
         *
         */

        case WM_CONTROL:
        {
            USHORT   usID = SHORT1FROMMP(mp1);
            USHORT   usNotifyCode = SHORT2FROMMP(mp1);
            switch (usID)
            {
                case ID_WIDI_PACKAGESCNR:
                    switch (usNotifyCode)
                    {
                        /*
                         * CN_EMPHASIS:
                         *      selection changed: update display
                         */

                        case CN_EMPHASIS:
                            DBPackagesEmphasis(pView,
                                               hwndDlg,
                                               (PNOTIFYRECORDEMPHASIS)mp2,
                                               pView->_fClearingContainers);
                        break;

                        /*
                         * CN_CONTEXTMENU:
                         *      context menu requested for item;
                         *      mp2 has record core or NULL for
                         *      whitespace
                         */

                        case CN_CONTEXTMENU:
                        {
                            HWND    hPopupMenu = NULLHANDLE;

                            pView->_preccLeftSource = (PDATABASERECORD)mp2;
                            if (pView->_preccLeftSource)
                            {
                                // on context record:
                                if (pView->_preccLeftSource->pDBPackage)
                                    // on package:
                                    hPopupMenu = pView->_hmenuPckPackage;
                                else
                                    // on application:
                                    hPopupMenu = pView->_hmenuPckApplication;
                            }
                            else
                            {
                                // cnr whitespace:
                                hPopupMenu = pView->_hmenuPckWhitespace;
                                CheckLeftViewMenuItems(pLocals, hPopupMenu);
                            }

                            cnrhShowContextMenu(WinWindowFromID(hwndDlg, usID),
                                                (PRECORDCORE)pView->_preccLeftSource,
                                                hPopupMenu,
                                                hwndDlg);
                        }
                        break;

                        case CN_HELP:
                            gshrDisplayHelp(pLocals, IDHI_DATABASE);
                        break;

                    } // end switch (usNotifyCode)

                break;      // case ID_WIDI_PACKAGESCNR
            }
        }
        break;

        /*
         * WM_MENUEND:
         *      if the context menu is dismissed, we'll need
         *      to remove the cnr source emphasis which was
         *      set above when showing the context menu.
         */

        case WM_MENUEND:
            cnrhSetSourceEmphasis(WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR),
                                  pView->_preccLeftSource,
                                  FALSE);
        break;

        /*
         * WM_CHAR:
         *      switch focus on "Tab" key.
         */

        case WM_CHAR:
            ProcessWMChar(pView, mp1, mp2);
            mrc = (MPARAM)TRUE;
        break;

        /*
         * WM_COMMAND:
         *
         */

        case WM_COMMAND:
        {
            ULONG       ulCmd = (ULONG)mp1;

            switch (ulCmd)
            {
                /*
                 * ID_WIMI_DATABASE_VERIFYAPP etc.:
                 *      open "Status" window, which
                 *      starts the Database thread
                 */

                case ID_WIMI_DATABASE_VERIFYAPP:
                case ID_WIMI_DATABASE_VERIFYPCK:
                case ID_WIMI_DATABASE_DEINSTALLAPP:
                case ID_WIMI_DATABASE_DEINSTALLPCK:
                    DBVerifyOrDeinstall(pView,
                                        ulCmd);
                break;

                case ID_WIMI_DATABASE_REMOVEPCK:
                {
                    FEDBPackage* pDBPackage;
                    if (    (pView->_preccLeftSource)
                         && (pDBPackage = pView->_preccLeftSource->pDBPackage)
                       )
                    {
                        ustring ustr;
                        pDBPackage->_PckID.CreateStringSix(ustr);
                        if (pLocals->MessageBox(108,       // "Warning"
                                                122,
                                                MSG_CONFIRM_YESNO_DEFYES,
                                                &ustr, 1)
                                    == MBID_YES)
                            if (pLocals->_pDatabase->RemovePackage(pDBPackage))
                            {
                                WinSendMsg(hwndDlg,
                                           WPIM_PACKAGEREMOVED,
                                           (MPARAM)pDBPackage,
                                           (MPARAM)NULL);
                                // create a temp database status for callback
                                /*
                                GUIDatabaseStatus Status(*pLocals,
                                                         hwndDlg);
                                // and call the GUI callback
                                Status.Callback(DBC_PACKAGEREMOVED,
                                                (ULONG)pDBPackage);
                                */
                            }
                    }
                }
                break;

                /*
                 * ID_WIMI_DATABASE_TREEVIEW:
                 * ID_WIMI_DATABASE_DETAILSVIEW:
                 *      switch container view
                 */

                case ID_WIMI_DATABASE_TREEVIEW:
                case ID_WIMI_DATABASE_DETAILSVIEW:
                    if (pLocals->_GuiSettings.ulLeftView != ulCmd)
                    {
                        // view changed:

                        // set the static flag in this proc to prevent flicker
                        // in the right half V0.9.9 (2001-02-28) [umoeller]
                        pView->_fClearingContainers = TRUE;

                        FillDatabaseContainer(pView,
                                              WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR),
                                              ulCmd);
                        pLocals->SaveSettings(0);
                        CheckLeftViewMenuItems(pLocals, pView->_hmenuMain);

                        pView->_fClearingContainers = FALSE;

                        // now refresh the right
                        UpdateListCnr(pView, pView->_preccDBSelected);
                    }
                break;

                case ID_WIDI_CANCELBUTTON:
                    WinDismissDlg(hwndDlg, ulCmd);
                break;
            }
        }
        break;

        /*
         * WPIM_PACKAGEREMOVED:
         *      this gets posted by guiDatabaseCallback.
         *      mp1 has the FEPackage* which has been
         *      removed by the Database thread.
         */

        case WPIM_PACKAGEREMOVED:
        {
            FEDBPackage* pPckInfoRemove = (FEDBPackage*)mp1;
            if (pPckInfoRemove)
            {
                // get the record core from the package info
                PDATABASERECORD preccRemove = (PDATABASERECORD)pPckInfoRemove->_pvGuiData;
                if (preccRemove)
                {
                    HWND hwndDatabaseCnr = WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR);
                    PDATABASERECORD preccApplication =
                                (PDATABASERECORD)cnrhQueryParentRecord(hwndDatabaseCnr,
                                                                 preccRemove);

                    FreeDatabaseRecord(pView, hwndDatabaseCnr, preccRemove);

                    // now check if the application record has
                    // any more children (i.e. any more packages)
                    if (preccApplication)
                        if (cnrhQueryFirstChildRecord(hwndDatabaseCnr, preccApplication)
                                        == NULL)
                            FreeDatabaseRecord(pView, hwndDatabaseCnr,
                                               preccApplication);
                }
            }
        }
        break;

        /*
         * WM_WINDOWPOSCHANGED:
         *
         */

        case WM_WINDOWPOSCHANGED:
            if (pView)
            {
                // this msg is passed two SWP structs:
                // one for the old, one for the new data
                // (from PM docs)
                PSWP pswpNew = (PSWP)(mp1);

                // resizing?
                if (pswpNew->fl & SWP_SIZE)
                {
                    winhAdjustControls(hwndDlg,             // dialog
                                       G_ampPackagesDlg,    // MPARAMs array
                                       sizeof(G_ampPackagesDlg) / sizeof(MPARAM), // items count
                                       pswpNew,             // mp1
                                       &pView->_xacPackagesDlg);  // storage area
                }
                mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
            }
        break;

        /*
         * WM_SYSCOMMAND:
         *      pass on to frame
         */

        case WM_SYSCOMMAND:
        case WM_HELP:
            WinPostMsg(WinQueryWindow(hwndDlg, QW_OWNER),
                       msg, mp1, mp2);
        break;

        case WM_DESTROY:
            winhAdjustControls(hwndDlg,             // dialog
                               NULL,
                               NULL,
                               NULL,
                               &pView->_xacPackagesDlg);  // storage area
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }
    return (mrc);
}

/* ******************************************************************
 *
 *   "File list" subwindow
 *
 ********************************************************************/

/*
 *@@ fnwpDBFileListDlg:
 *      window procedure for the "File list" subdialog
 *      in the database view (on the right).
 *
 *@@changed V0.9.0 (99-10-24) [umoeller]: reworked completely for split view
 *@@changed V0.9.2 (2000-02-19) [umoeller]: added right view support
 *@@changed V0.9.2 (2000-03-10) [umoeller]: added keyboard focus support
 *@@changed V0.9.5 (2000-08-10) [umoeller]: added context menu for right view
 *@@changed V0.9.5 (2000-08-10) [umoeller]: F1 displays help now
 */

MRESULT EXPENTRY fnwpDBFileListDlg(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    DatabaseView *pView = (DatabaseView *)WinQueryWindowPtr(hwndDlg, QWL_USER);
    GUILocals*   pLocals;
    if (pView)
        pLocals = &pView->_Locals;

    switch (msg)
    {
        /*
         * WM_INITDLG:
         *
         */

        case WM_INITDLG:
        {
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);

            WinSetWindowPtr(hwndDlg, QWL_USER, mp2);
            pView = (DatabaseView*)mp2;

            // make the thing properly sizeable (99-10-24) [umoeller]
            memset(&pView->_xacFileListDlg, 0, sizeof(XADJUSTCTRLS));
            winhAdjustControls(hwndDlg,
                               G_ampFileListDlg,         // global struct
                               sizeof(G_ampFileListDlg) / sizeof(MPARAM),
                               NULL,                // init
                               &pView->_xacFileListDlg);

            // set font
            winhSetControlsFont(hwndDlg, -2, 2000,
                        gshrQueryDefaultFont());

            gshrMakeHeading(hwndDlg, ID_WIDI_HEADING);

        }
        break;

        case WM_WINDOWPOSCHANGED:   // added (99-10-24) [umoeller]
            if (pView)
            {
                // this msg is passed two SWP structs:
                // one for the old, one for the new data
                // (from PM docs)
                PSWP pswpNew = (PSWP)mp1;

                // resizing?
                if (pswpNew->fl & SWP_SIZE)
                {
                    winhAdjustControls(hwndDlg,             // dialog
                                       G_ampFileListDlg,     // MPARAMs array
                                       sizeof(G_ampFileListDlg) / sizeof(MPARAM), // items count
                                       pswpNew,             // mp1
                                       &pView->_xacFileListDlg);               // storage area
                }
                mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
            }
        break;

        /*
         * WM_CHAR:
         *      switch focus on "Tab" key.
         */

        case WM_CHAR:
        {
            ProcessWMChar(pView, mp1, mp2);
            mrc = (MPARAM)TRUE;
        }
        break;

        /*
         * WM_SYSCOMMAND:
         *      pass on to frame
         */

        case WM_SYSCOMMAND:
        case WM_HELP:
            WinPostMsg(WinQueryWindow(hwndDlg, QW_OWNER),
                       msg, mp1, mp2);
        break;

        /*
         * WM_CONTROL:
         *
         */

        case WM_CONTROL:
        {
            USHORT   usID = SHORT1FROMMP(mp1);
            USHORT   usNotifyCode = SHORT2FROMMP(mp1);
            switch (usID)
            {
                case ID_WIDI_PACKAGESCNR:
                    switch (usNotifyCode)
                    {
                        /*
                         * CN_CONTEXTMENU:
                         *      context menu requested for item;
                         *      mp2 has record core or NULL for
                         *      whitespace
                         */

                        case CN_CONTEXTMENU:
                        {
                            HWND    hPopupMenu = NULLHANDLE;
                            pView->_preccRightSource = (PFILERECORDCORE)mp2;
                            if (pView->_preccRightSource)
                            {
                                /* // on context record:
                                if (G_preccDBSource->pPckInfo)
                                    // on package:
                                    hPopupMenu = pView->_hmenuPckPackage;
                                else
                                    // on application:
                                    hPopupMenu = pView->_hmenuPckApplication;
                                */
                            }
                            else
                            {
                                // cnr whitespace:
                                hPopupMenu = pView->_hmenuFilesWhitespace;
                                CheckRightViewMenuItems(pLocals, hPopupMenu);
                            }

                            if (hPopupMenu)
                                cnrhShowContextMenu(WinWindowFromID(hwndDlg, usID),
                                                    (PRECORDCORE)pView->_preccRightSource,
                                                    hPopupMenu,
                                                    hwndDlg);
                        }
                        break;

                        case CN_HELP:
                            gshrDisplayHelp(pLocals, IDHI_DATABASE);
                        break;

                    } // end switch (usNotifyCode)

                break;      // case ID_WIDI_PACKAGESCNR
            }
        }
        break;

        /*
         * WM_MENUEND:
         *      if the context menu is dismissed, we'll need
         *      to remove the cnr source emphasis which was
         *      set above when showing the context menu.
         */

        case WM_MENUEND:
            cnrhSetSourceEmphasis(WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR),
                                  pView->_preccRightSource,
                                  FALSE);
        break;

        case WM_COMMAND:
        {
            USHORT  usCmd = SHORT1FROMMP(mp1);
            switch (usCmd)
            {
                case ID_WIMI_VIEW_FILES:
                case ID_WIMI_VIEW_CONFIG:
                    if (pLocals->_GuiSettings.ulRightView != usCmd)
                    {
                        pLocals->_GuiSettings.ulRightView = usCmd;
                        // view changed:
                        /* FillDatabaseContainer(WinWindowFromID(hwndDlg, ID_WIDI_PACKAGESCNR),
                                              usCmd); */
                        UpdateListCnr(pView, pView->_preccDBSelected);
                        pLocals->SaveSettings(0);
                        CheckRightViewMenuItems(pLocals, pView->_hmenuMain);
                    }
            }
        }
        break;

        case WM_DESTROY:
        {
            // cleanup adjust-controls data
            winhAdjustControls(hwndDlg,
                               NULL,
                               0,
                               NULL,
                               &pView->_xacFileListDlg);

            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
        }
        break;

        default:
            mrc = WinDefDlgProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ UpdateListCnr:
 *      this gets called from fnwpDBPackagesDlg when
 *      a new application/package has been selected
 *      in the packages list subdialog. This then
 *      updates the file list.
 *
 *      Note that precc must be a package record core
 *      from the packages  container (not an application
 *      record core, because applications have no
 *      file lists).
 *
 *@@changed V0.9.0 (99-10-24) [umoeller]: reworked completely for split view
 *@@changed V0.9.0 (99-10-24) [umoeller]: changed name from OpenFileList
 *@@changed V0.9.0 (99-11-04) [umoeller]: fixed baaaad memory leak
 *@@changed V0.9.1 (2000-01-06) [umoeller]: now displaying target path on top of file list
 *@@changed V0.9.2 (2000-02-19) [umoeller]: reworked, can now display config too
 *@@changed V0.9.12 (2001-05-17) [umoeller]: added NULL check
 */

VOID UpdateListCnr(DatabaseView *pView,
                   PDATABASERECORD precdb)    // in: package record core in left view
{
    GUIWaitPointer wp;

    GUILocals *pLocals = &pView->_Locals;
    HWND    hwndFilesCnr = WinWindowFromID(pView->_hwndDBFileListDlg, ID_WIDI_PACKAGESCNR);

    // clear container
    cnrhRemoveAll(hwndFilesCnr);

    // nuke all the config strings we have created last time
    // V0.9.14 (2001-07-07) [umoeller]
    pView->_ConfigStrings.clear();

    string strTargetPath;

    FEDBPackage*    pDBPackage;
    if (    (precdb)      // V0.9.12 (2001-05-17) [umoeller]
            // is precc a package record core?
         && (pDBPackage = precdb->pDBPackage)
       )
    {
        // set up info text (99-10-24) [umoeller]
        strTargetPath.assignUtf8(pLocals->_pCodecGui, pDBPackage->QueryTargetPath());

        if (pLocals->_GuiSettings.ulRightView == ID_WIMI_VIEW_FILES)
            InsertFilesIntoCnr(precdb,
                               hwndFilesCnr);
        else
        {
            SetCnrConfigFieldInfos(hwndFilesCnr);
            InsertConfigIntoCnr(pLocals,
                                hwndFilesCnr,
                                pDBPackage,
                                pView->_ConfigStrings);
        }
    }

    WinSetWindowText(WinWindowFromID(pView->_hwndDBFileListDlg,
                                     ID_WIDI_TARGETPATHDIR),
                     strTargetPath.c_str());

}

/* ******************************************************************
 *
 *   main window
 *
 ********************************************************************/

/*
 *@@ fnwpDBMainClient:
 *      window proc for the client of the database window.
 *      This initializes all the subwindows in turn and
 *      manages the split windows upon resizes etc.
 *
 *@@added V0.9.0 (99-10-24) [umoeller]
 *@@changed V0.9.2 (2000-02-19) [umoeller]: added "view" menu support
 *@@changed V0.9.2 (2000-02-19) [umoeller]: "Exit" didn't work; fixed
 *@@changed V0.9.14 (2001-07-31) [umoeller]: fixed bad FELocals* ptr on WM_CREATE
 */

MRESULT EXPENTRY fnwpDBMainClient(HWND hwndClient, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    DatabaseView *pView = (DatabaseView *)WinQueryWindowPtr(hwndClient, QWL_USER);
    GUILocals*   pLocals;
    if (pView)
        pLocals = &pView->_Locals;

    switch (msg)
    {
        /*
         * WM_CREATE:
         *
         */

        case WM_CREATE:
        {
            PDBCREATESTRUCT pdbcs = (PDBCREATESTRUCT)mp1;
            pView = pdbcs->pView;       // V0.9.18 (2002-03-08) [umoeller]
            WinSetWindowPtr(hwndClient, QWL_USER, pView);

            /*
             * left window: packages dialog
             *
             */

            pView->_hwndDBPackagesDlg = gshrLoadDlg(hwndClient, hwndClient, // parent and owner
                                              fnwpDBPackagesDlg,
                                              NULLHANDLE,      // our own EXE
                                              ID_WID_DB_LEFT_PACKAGES,
                                              // create param:
                                              pView);
            WinSetWindowBits(pView->_hwndDBPackagesDlg,
                             QWL_STYLE,
                             WS_CLIPCHILDREN,         // set bit
                             WS_CLIPCHILDREN);

            /*
             * right window: files list
             *
             */

            pView->_hwndDBFileListDlg = gshrLoadDlg(hwndClient, hwndClient,
                                             fnwpDBFileListDlg,
                                             NULLHANDLE,      // our own EXE
                                             ID_WID_DB_RIGHT_LIST,
                                             // create param:
                                             pView);
            WinSetWindowBits(pView->_hwndDBFileListDlg,
                             QWL_STYLE,
                             WS_CLIPCHILDREN,         // set bit
                             WS_CLIPCHILDREN);

            SPLITBARCDATA   sbcd;
            memset(&sbcd, 0, sizeof(SPLITBARCDATA));
            sbcd.ulCreateFlags = SBCF_VERTICAL          // vertical split bar
                                    | SBCF_PERCENTAGE   // lPos has split bar pos
                                                        // in percent of the client
                                    | SBCF_3DSUNK       // draw 3D "sunk" frame
                                    | SBCF_MOVEABLE;    // moveable split bar
            sbcd.lPos = 50;     // percent
            sbcd.hwndParentAndOwner = hwndClient;
            sbcd.ulLeftOrBottomLimit = 100;
            sbcd.ulRightOrTopLimit = 100;
            sbcd.ulSplitWindowID = 1000;
            pView->_hwndDBSplitMain = ctlCreateSplitWindow(pLocals->_habThread1,
                                                     &sbcd);

            // now link the split windows
            WinSendMsg(pView->_hwndDBSplitMain,
                       SPLM_SETLINKS,
                       // left window
                       (MPARAM)pView->_hwndDBPackagesDlg,
                       // right window:
                       (MPARAM)pView->_hwndDBFileListDlg);
            ctlUpdateSplitWindow(pView->_hwndDBSplitMain);

            WinShowWindow(pView->_hwndDBPackagesDlg, TRUE);
            WinShowWindow(pView->_hwndDBFileListDlg, TRUE);
        }
        break;

        /*
         * WM_WINDOWPOSCHANGED:
         *
         */

        case WM_WINDOWPOSCHANGED:
        {
            // this msg is passed two SWP structs:
            // one for the old, one for the new data
            // (from PM docs)
            PSWP pswpNew = (PSWP)(mp1);

            // resizing?
            if (pswpNew->fl & SWP_SIZE)
            {
                WinSetWindowPos(pView->_hwndDBSplitMain, HWND_TOP,
                                0, 0,
                                pswpNew->cx, pswpNew->cy,
                                SWP_SIZE);
            }

            // return default NULL
        }
        break;

        case WM_PAINT:
        {
            HPS     hps = WinBeginPaint(hwndClient, (HPS)0, NULL);
            // do nothing
            WinEndPaint(hps);

            // return default NULL
        }
        break;

        /*
         * WM_MINMAXFRAME:
         *      when minimizing, we hide the "split window",
         *      because otherwise the child dialogs will
         *      display garbage
         */

        case WM_MINMAXFRAME:
        {
            PSWP pswp = (PSWP)mp1;
            if (pswp->fl & SWP_MINIMIZE)
                WinShowWindow(pView->_hwndDBSplitMain, FALSE);
            else if (pswp->fl & SWP_RESTORE)
                WinShowWindow(pView->_hwndDBSplitMain, TRUE);
        }
        break;

        /*
         * WM_CHAR:
         *      switch focus on "Tab" key.
         */

        case WM_CHAR:
            ProcessWMChar(pView, mp1, mp2);
            mrc = (MPARAM)TRUE;
        break;

        /*
         * WM_COMMAND:
         *
         */

        case WM_COMMAND:
        {
            USHORT  usCmd = SHORT1FROMMP(mp1);

            switch (usCmd)
            {
                case ID_WIMI_DATABASE_TREEVIEW:
                case ID_WIMI_DATABASE_DETAILSVIEW:
                    WinPostMsg(pView->_hwndDBPackagesDlg, WM_COMMAND, mp1, mp2);
                    break;

                case ID_WIMI_VIEW_FILES:
                case ID_WIMI_VIEW_CONFIG:
                    WinPostMsg(pView->_hwndDBFileListDlg, WM_COMMAND, mp1, mp2);

                default:
                    gshrDefWindowProc(pLocals,
                                      hwndClient, msg, mp1, mp2,
                                      IDHI_DATABASE);
            }
        }
        break;

        /*
         * WM_CLOSE:
         *      window list is being closed:
         *      store the window position
         */

        case WM_CLOSE:
        {
            HWND hwndFrame = WinQueryWindow(hwndClient, QW_PARENT);
            // get the object pointer and the use list item from the window

            // save window position
            winhSaveWindowPos(hwndFrame,
                              pLocals->_hiniWarpIN,
                              "GUI", "DBWndPos");

            mrc = WinDefWindowProc(hwndClient, msg, mp1, mp2);
                    // posts WM_QUIT
        }
        break;

        default:
            if (!gshrDefWindowProc(pLocals,
                                   hwndClient, msg, mp1, mp2,
                                   IDHI_DATABASE))
                mrc = WinDefWindowProc(hwndClient, msg, mp1, mp2);
    }

    return (mrc);
}

/*
 *@@ guiBeginDatabaseMode:
 *      this gets called from warpin.cpp when WarpIN is
 *      started without any archive as a parameter,
 *      and the database is not empty.
 *
 *      The global variable pDatabase points to the
 *      instance of FEDatabase which has been created
 *      upon startup.
 *
 *      This routine is responsible for displaying
 *      the installed packages and calling the database
 *      methods for package removal and verification,
 *      if the user desires this.
 *
 *      For the PM version of this function, this
 *      is done from the Database Status dialog window
 *      (fnwpDatabaseStatus), which is created from
 *      this function.
 *
 *      Required callback. After this returns, WarpIN
 *      terminates.
 *
 *@@changed V0.9.0 (99-10-24) [umoeller]: reworked completely for split view
 *@@changed V0.9.2 (2000-02-19) [umoeller]: added "view" menu support
 *@@changed V0.9.2 (2000-03-10) [umoeller]: added keyboard focus support
 */

BOOL guiBeginDatabaseMode(GUILocals *pLocals)
{
    BOOL brc;

    // register the frame class, adding a user word to the window data
    WinRegisterClass(pLocals->_habThread1,
                     WC_DATABASEMAINFRAME,
                     fnwpDBMainClient,
                     CS_SIZEREDRAW | CS_SYNCPAINT,
                     sizeof(PVOID)); // additional bytes to reserve

    // create the frame window
    ULONG flCreate = FCF_SYSMENU
                        | FCF_SIZEBORDER
                        | FCF_TITLEBAR
                        | FCF_MINMAX
                        | FCF_NOBYTEALIGN;

    SWP swpFrame;
    swpFrame.x = 100;
    swpFrame.y = 100;
    swpFrame.cx = 500;
    swpFrame.cy = 500;
    swpFrame.hwndInsertBehind = HWND_TOP;
    swpFrame.fl = SWP_MOVE | SWP_SIZE;

    // create the view struct V0.9.18 (2002-03-08) [umoeller]
    DatabaseView    View(*pLocals);

    G_pView = &View;

    HWND        hwndClient = NULLHANDLE;
    DBCREATESTRUCT dbcs;
    dbcs.cb = sizeof(dbcs);
    dbcs.pView = &View;

    G_hwndMainFrame = winhCreateStdWindow(HWND_DESKTOP,           // frame parent
                                          &swpFrame,
                                          flCreate,
                                          WS_ANIMATE,
                                          nlsGetString(WPSI_DB_MAINWINDOWTITLE), // title bar
                                          0,                      // res IDs
                                          WC_DATABASEMAINFRAME,     // client class
                                          0L,                     // client wnd style
                                          1,          // ID
                                          &dbcs,
                                          &hwndClient);
            // this cascades into all the window procs to initialize
            // the subwindows

    if (G_hwndMainFrame)
    {
        // add to tasklist
        winhAddToTasklist(G_hwndMainFrame,
                          G_hptrMain);

        // load main menu
        View._hmenuMain = WinLoadMenu(G_hwndMainFrame, NULLHANDLE, ID_WIM_MAINMENU);
        // remove "Archive" menu V0.9.9 (2001-03-27) [umoeller]
        WinSendMsg(View._hmenuMain,
                   MM_REMOVEITEM,
                   MPFROM2SHORT(ID_WIM_ARCHIVE, TRUE),
                   0);
        // remove "Selections" menu (99-11-01) [umoeller]
        WinSendMsg(View._hmenuMain,
                   MM_REMOVEITEM,
                   MPFROM2SHORT(ID_WIM_SELECT, TRUE),
                   0);
        WinSendMsg(G_hwndMainFrame, WM_UPDATEFRAME, (MPARAM)FCF_MENU, 0);
        CheckLeftViewMenuItems(pLocals, View._hmenuMain);
        CheckRightViewMenuItems(pLocals, View._hmenuMain);

        // load context menus
        View._hmenuPckWhitespace = WinLoadMenu(HWND_OBJECT,
                                           NULL,     // our EXE
                                           ID_WIM_DATABASE_LEFT_WHTSPACE);
        View._hmenuPckApplication = WinLoadMenu(HWND_OBJECT,
                                            NULL,     // our EXE
                                            ID_WIM_DATABASE_APP);
        View._hmenuPckPackage = WinLoadMenu(HWND_OBJECT,
                                        NULL,     // our EXE
                                        ID_WIM_DATABASE_PACKAGE);
        View._hmenuFilesWhitespace = WinLoadMenu(HWND_OBJECT,
                                             NULL,     // our EXE
                                             ID_WIM_DATABASE_RIGHT_WHTSPACE);

        // now position the frame and the client:
        // 1) frame
        if (!winhRestoreWindowPos(G_hwndMainFrame,
                                  pLocals->_hiniWarpIN,
                                  "GUI", "DBWndPos",
                                  SWP_MOVE | SWP_SIZE))
            // INI data not found:
            WinSetWindowPos(G_hwndMainFrame,
                            HWND_TOP,
                            100, 100,
                            500, 500,
                            SWP_MOVE | SWP_SIZE);

        WinShowWindow(G_hwndMainFrame, TRUE);

        WinSetFocus(HWND_DESKTOP,
                    WinWindowFromID(View._hwndDBPackagesDlg,
                                    ID_WIDI_PACKAGESCNR));

        // kill the startup window
        gshrDestroySplash();

        QMSG qmsg;
        while (WinGetMsg(pLocals->_habThread1,
                         &qmsg, 0, 0, 0))
            WinDispatchMsg(pLocals->_habThread1, &qmsg);

        // window is destroyed in destructor

        brc = TRUE;
    }

    return (brc);
}

/*
 *@@ guiStartVerify:
 *      this is supposed to display a dialog for the
 *      database progress and call pDatabase->VerifyPackages.
 *
 *      This only gets called from the GUI in "Database"
 *      mode.
 *
 *      This creates the database status window. See
 *      fnwpDatabaseStatus for additional remarks.
 *
 *      Required callback.
 *
 *@@added V0.9.0 (99-11-02) [umoeller]
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added log file
 */

BOOL guiStartVerify(GUILocals *pLocals,
                    list<FEDBPackage*> *pPackagesList,
                    BOOL fModal)
                            // if TRUE, this function does not return
                            // until we're done (modal dialog)
{
    GUIDatabaseStatus *pStatus = new GUIDatabaseStatus(
                            *pLocals,
                            DBT_VERIFY,
                            0, // deinstall flags, don't care
                            nlsGetString(WPSI_DB_VERIFYINGPACKAGE),
                            (fModal)
                               ? NULLHANDLE
                               : G_pView->_hwndDBPackagesDlg);

    // load the database thread status window
    // (fnwpDatabaseStatus)
    pStatus->_hwndStatus = gshrLoadDlg(HWND_DESKTOP,   // parent
                                       G_hwndMainFrame,        // owner
                                       fnwpDatabaseStatus,
                                       NULLHANDLE,      // our own EXE
                                       ID_WID_DATABASETHREAD,
                                       // pass the command to the dlg
                                       (PVOID)pStatus);
    winhCenterWindow(pStatus->_hwndStatus);
    WinShowWindow(pStatus->_hwndStatus, TRUE);

    // start database thread
    // now we have the package info which we need to verify;
    // create a thread info for the database thread
    GUIDatabaseThread
            *pdbti = new GUIDatabaseThread(pPackagesList, // list
                                           *pStatus);

    if (fModal)
    {
        WinProcessDlg(pStatus->_hwndStatus);
        WinDestroyWindow(pStatus->_hwndStatus);
    }
    else
        G_pView->_preccLeftSource->hwndStatus = pStatus->_hwndStatus;

    return (TRUE);
}

/*
 *@@ guiStartRemove:
 *      this is supposed to display a dialog for the
 *      database progress and call pDatabase->DeinstallPackages.
 *
 *      warpin.cpp calls this callback two situations:
 *
 *      -- in "install" mode, when packages need to be removed;
 *         in that case, this dialog runs modally;
 *
 *      -- from the GUI in "Database" mode; in that case, this
 *         dialog runs modelessly.
 *
 *      This creates the database status window. See
 *      fnwpDatabaseStatus for additional remarks.
 *
 *      Required callback.
 *
 *@@added V0.9.0 (99-11-02) [umoeller]
 *@@changed V0.9.4 (2000-07-01) [umoeller]: added process kills
 */

BOOL guiStartRemove(GUILocals *pLocals,
                    list<FEDBPackage*> *pPackagesList,
                            // list of packages to be removed
                    ULONG flDeinstall, // in: DBT_DELETEFILES | DBT_UNDOCONFIG
                    BOOL fModal)
                            // if TRUE, this function does not return
                            // until we're done (modal dialog)
{
    BOOL brc = FALSE;

    // kill processes
    list<FEDBPackage*>::iterator PckStart = pPackagesList->begin(),
                                 PckEnd = pPackagesList->end();
    for (; PckStart != PckEnd; PckStart++)
    {
        FEDBPackage *pPackageThis = *PckStart;

        // go thru the list of all config objects; this
        // list holds all objects as base objects, so
        // we need to type check
        list<BSConfigBase*>::iterator
                CfgStart = pPackageThis->_listUndoConfig.begin(),
                CfgEnd = pPackageThis->_listUndoConfig.end();
        for (; CfgStart != CfgEnd; CfgStart++)
        {
            BSConfigBase* pCfgThis = *CfgStart;
            if (pCfgThis)   // V0.9.7 (2000-12-16) [umoeller]
            {
                // check if this is our class
                DYNAMIC_CAST(BSKillProcess, pKill, pCfgThis);
                // BSKillProcess *pKill = pCfgThis->IsKillProcess();
                if (pKill)
                {
                    BOOL fReloop = FALSE;

                    do
                    {
                        // OK, we got a KILLPROCESS attribute:
                        fReloop = pLocals->KillOneProcess(pKill);
                    } while (fReloop);
                }
                // else
                    // not the class we need: take next then
            }
        }
    }

    pLocals->OpenLogger();

    GUIDatabaseStatus *pStatus = new GUIDatabaseStatus(
                            *pLocals,
                            DBT_DEINSTALL,
                            flDeinstall,
                            nlsGetString(WPSI_DB_DEINSTALLINGPACKAGE),
                            (fModal)
                               ? NULLHANDLE
                               : G_pView->_hwndDBPackagesDlg);

    // load the database thread status window
    // (fnwpDatabaseStatus)
    pStatus->_hwndStatus = gshrLoadDlg(HWND_DESKTOP,   // parent
                                       G_hwndMainFrame,        // owner
                                       fnwpDatabaseStatus,
                                       NULLHANDLE,      // our own EXE
                                       ID_WID_DATABASETHREAD,
                                       // pass the command to the dlg
                                       (PVOID)pStatus);
    winhCenterWindow(pStatus->_hwndStatus);
    WinShowWindow(pStatus->_hwndStatus, TRUE);

    // start database thread
    // now we have the package info which we need to verify;
    // create a thread info for the database thread
    GUIDatabaseThread
            *pdbti = new GUIDatabaseThread(pPackagesList, // list
                                           *pStatus);

    if (fModal)
    {
        WinProcessDlg(pStatus->_hwndStatus);
        WinDestroyWindow(pStatus->_hwndStatus);
    }
    else
        G_pView->_preccLeftSource->hwndStatus = pStatus->_hwndStatus;

    return (TRUE);
}

/* ******************************************************************
 *
 *  GUIDatabaseThread implementation
 *
 ********************************************************************/

static THREADINFO       G_tiDatabaseThread = {0};


/*
 *@@ GUIDatabaseThread:
 *      constructor with parameters, all of which
 *      _must_ be specified.
 *
 *      The given list must specify all packages
 *      to work on. Preconditions:
 *
 *      --  The list must be in SHADOW mode (i.e.
 *          contain pointers to the FEDBPackage's,
 *          but not copies of the FEDBPackage's
 *          themselves) because FEDatabase will
 *          remove the packages from the database.
 *
 *      --  The list itself will be freed when the
 *          database thread is done.
 *
 *      This constructor initializes _all_ data
 *      for the database thread and starts the
 *      database thread (datDatabaseThread)
 *      automatically, which keeps running after
 *      this constructor returns.
 *
 *      If you pass in a BSFileLogger* for a log
 *      file, you must close that log file (by
 *      invoking delete() on the logger) yourself
 *      when the database thread has finished its
 *      work. The database thread will only write
 *      into the logger.
 *
 *@@changed V0.9.0 (99-11-02) [umoeller]: reworked completely
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added log file
 *@@changed V0.9.12 (2001-05-22) [umoeller]: now creating flag with PM message queue
 *@@changed V0.9.14 (2001-07-07) [umoeller]: no longer copying list
 */

GUIDatabaseThread::GUIDatabaseThread(list<FEDBPackage*> *pPackagesList,
                                     GUIDatabaseStatus &Status)
    : _pPackagesList(pPackagesList),
      _Status(Status)
{
    // start database thread
    thrCreate(&G_tiDatabaseThread ,      // global PTHREADINFO at top
              datDatabaseThread,        // thread func
              NULL,
              "Database",
              THRF_PMMSGQUEUE | THRF_WAIT,    // changed V0.9.12 (2001-05-22) [umoeller]
              (ULONG)this);             // data passed to thread
    // our instance will be deleted by the database thread
}

/*
 *@@ ~GUIDatabaseThread:
 *
 *@@added V0.9.14 (2001-07-28) [umoeller]
 */

GUIDatabaseThread::~GUIDatabaseThread()
{
    if (_pPackagesList)
        delete _pPackagesList;
}

/* ******************************************************************
 *
 *  Database thread
 *
 ********************************************************************/

/*
 *@@ datDatabaseThread:
 *      the actual database thread function.
 *
 *      ((PGUIDatabaseThreadInfo)ptiMyself)->ulData is a
 *      pointer to the GUIDatabaseThread instance
 *      which specifies what needs to be done.
 *      It is the responsibility of this thread to free
 *      this structure.
 *
 *      All code has been moved to FEDatabase::DeinstallOrVerify.
 */

void _Optlink datDatabaseThread(PTHREADINFO pti)
{
    GUIDatabaseThread *pdbti
        = (GUIDatabaseThread*)(pti->ulData);

    // BSFileLogger *pLogFile = pdbti->_Status._Locals._pLogFile;
    FELocals &Locals = pdbti->_Status._Locals;

    Locals.Log("WarpIN Database thread started");

    // run!
    pdbti->_Status._Database.DeinstallOrVerify(pti->hab,
                                               pdbti->_Status,
                                               *pdbti->_pPackagesList,
                                               pdbti->_Status._ulTask,
                                               pdbti->_Status._flDeinstall);

    Locals.Log("WarpIN Database thread is exiting");

    delete pdbti;       // this has a destructor, which
                        // also closes the log file

    G_WpiGlobals.fDeinstallComplete = TRUE;
            // provoke exit message
}




