
/*
 *@@sourcefile gui.cpp:
 *      this implements the graphical user interface (GUI)
 *      of WarpIN, at least the main stuff (there is more
 *      GUI code in the other gui_* source files).
 *
 *      The functions in here get called from main() in warpin.cpp
 *      thru the required callbacks listed in include\frontend\calbacks.h.
 *
 *      See warpin.cpp for details about this concept.
 *
 *      Note that all functions which get called from outside
 *      this file (i.e. either warpin.cpp or database.cpp)
 *      have the gui* prefix. These functions are all prototyped
 *      in warpin.h.
 *
 *@@header "frontend\gui.h"
 */

/*
 *
 *      This file Copyright (C) 1998-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#define INCL_WINCOUNTRY
#define INCL_WINWORKPLACE
#define INCL_GPILOGCOLORTABLE
#define INCL_GPIPRIMITIVES
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#ifdef __IBMCPP__
    #include <direct.h>
#endif
#include <string.h>
#include <ctype.h>
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h
#include <io.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader

#include "setup.h"
#include "bldlevel.h"

// include's from helpers
#include "helpers\comctl.h"             // common controls (window procs)
#include "helpers\configsys.h"
#include "helpers\datetime.h"
#include "helpers\dialog.h"
#include "helpers\dosh.h"
#include "helpers\exeh.h"
#include "helpers\cnrh.h"
#include "helpers\comctl.h"
#include "helpers\gpih.h"
#include "helpers\nls.h"
#include "helpers\nlscache.h"
#include "helpers\prfh.h"
#include "helpers\winh.h"
#include "helpers\standards.h"
#include "helpers\stringh.h"
#include "helpers\textview.h"
#include "helpers\threads.h"
#include "helpers\textv_html.h"
#include "helpers\xstring.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"

#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

// back-end includes
#include "wiarchive\wiarchive.h"

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_script.h"

#include "engine\fe_package.h"
#include "engine\fe_package_arc.h"
#include "engine\fe_package_db.h"
#include "engine\fe_archive.h"
#include "engine\fe_job.h"
#include "engine\fe_database.h"

#include "engine\fe_engine.h"
#include "engine\fe_cid.h"

#include "frontend\warpin.h"

#include "frontend\calbacks.h"

#include "frontend\dlgids.h"
#include "frontend\gui.h"

#include "helpers\except.h"

#pragma hdrstop

/* ******************************************************************
 *
 *  Private declarations
 *
 ********************************************************************/

// forward declarations
MRESULT EXPENTRY fnwpPages(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
MRESULT EXPENTRY fnwpDriveInfo(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
MRESULT EXPENTRY fnwpDatabaseStatus(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2);
VOID TurnToPage(GUIInstallEngine &Engine,
                LONG lSearchPage,
                BOOL fStoreThis);

DEFINE_CLASS(GUIVisited, BSRoot);

/* ******************************************************************
 *
 *  Global variables
 *
 ********************************************************************/

// this always has the main window, in all modes;
// this is used as the owner for message boxes, so this
// better be a valid window at all times (99-11-01) [umoeller]
HWND                G_hwndMainFrame = NULLHANDLE,
                    G_hwndPagesClient = NULLHANDLE;

// handle of main menu bar
HWND                G_hmenuMain = NULLHANDLE;

// "Drive info" window while cnr page is showing
HWND                G_hwndDriveInfo = NULLHANDLE;

// WarpIN icon; this must be here or guiShowMessage2
// can't see it
HPOINTER            G_hptrMain = NULLHANDLE;

// tooltip control
HWND                G_hwndCnrTooltip = NULLHANDLE;

// help instance
HWND                G_hwndHelpInstance = NULLHANDLE;
ULONG               G_ulPageHelp = 0;

PPACKAGERECORD      G_preccSelected = NULL;

FEPageInfo*         G_pCurrentPageInfo = NULL;

// list of files to copy on WarpIN upgrade
PCSZ G_apcszUpgradeFiles[] =
        {
            "COPYING",
            "warpin.exe",
            "wizilla.exe",
            "wic.exe",
            "wicpm.exe",
            "wpi2exe.exe",
            "wpirtl.dll",
            "wicpm.inf",
            "wpi_prog.inf",
            "wpi_user.inf",
            "warpin.sym",
            "wic.sym",
            "wicpm.sym",
            "warpin.hlp",
            "warpin.tmf",
            "readme.txt",
            "history.txt",
            "test\\apps.cmd"
        };

/* ******************************************************************
 *
 *  GUIWaitPointer implementation
 *
 ********************************************************************/

/*
 *@@ GUIWaitPointer:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

GUIWaitPointer::GUIWaitPointer()
{
    _hptrOld = winhSetWaitPointer();
}

/*
 *@@ ~GUIWaitPointer:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

GUIWaitPointer::~GUIWaitPointer()
{
    WinSetPointer(HWND_DESKTOP, _hptrOld);
}

/* ******************************************************************
 *
 *  GUILocals implementation
 *
 ********************************************************************/

VOID LoadNLSStrings(FELocals *pLocals);

static PCSZ     G_pcszBldlevel = BLDLEVEL_VERSION,
                G_pcszBldDate = __DATE__,
                G_pcszNewLine = "\n",
                G_pcszCopyright = "(C)"; // "\xb8";
                            // doesn't work with CP437
                            // V0.9.19 (2001-04-13) [umoeller]

static const STRINGENTITY G_aEntities[] =
    {
        "&version;", &G_pcszBldlevel,
        "&date;", &G_pcszBldDate,
        "&copy;", &G_pcszCopyright
    };

/*
 *@@ GUILocals:
 *      constructor.
 *
 *@@added V0.9.18 (2002-03-03) [umoeller]
 */

GUILocals::GUILocals(HAB habThread1,
                     HMQ hmqThread1,
                     CHAR cThousands,
                     BOOL *pfRestart)
           : FELocals(habThread1,
                      cThousands,
                      (void*)G_aEntities,
                      ARRAYITEMCOUNT(G_aEntities)),
             _hmqThread1(hmqThread1),
             _pfRestart(pfRestart)
{
    _pDatabase = NULL;

    *_pfRestart = FALSE;

    // open the WARPIN.INI file
    string strINIFile(_pCodecProcess, _ustrWarpINPath);
    strINIFile += "\\warpin.ini";
    _hiniWarpIN = PrfOpenProfile(habThread1,
                                 strINIFile.c_str());

    LoadSettings();

    // load icons which we'll need; these must be
    // destroyed in guiCleanupBeforeExit
    G_hptrMain   = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, ID_ICON);
    _hptrSelect   = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_SELECTED);
    _hptrDeselect = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DESELECTED);
    _hptrSomeselect = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_SOMESELECTED);
    _hptrDeinstall = WinLoadPointer(HWND_DESKTOP,
                                   NULLHANDLE, IDP_DEINSTALL);
    _hptrError    = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_ERROR);
    _hptrQMark    = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_QMARK);
    _hptrOK       = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_OK);
    _hptrNewer    = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_NEWER);
    _hptrApp      = WinLoadPointer(HWND_DESKTOP,
                                  NULLHANDLE, IDP_DB_APP);

    // system icon size
    _ulMiniIconSize = WinQuerySysValue(HWND_DESKTOP, SV_CYICON) / 2;

    // load NLS strings
    // LoadNLSStrings(this);

    nlsInitStrings(_habThread1,
                   NULLHANDLE,      // load from exe
                   G_aEntities,
                   ARRAYITEMCOUNT(G_aEntities));

    BOOL fRetry = TRUE;
    while (fRetry)
    {
        fRetry = FALSE;

        ULONG ulCodepage;

        if (!(ulCodepage = _ulDisplayCodepage))
            ulCodepage = _pCodecProcess->QueryCodepage();

        if (!WinSetCp(_hmqThread1, _ulDisplayCodepage))
            // failed: set to default then
            ulCodepage = _ulDisplayCodepage = 0;

        try
        {
            _pCodecGui = new BSUniCodec(ulCodepage);
        }
        catch (BSUnsupportedCPExcpt &X)
        {
            string str;
            str._printf("The display codepage %u that you have selected is not supported by WarpIN. "
                        "WarpIN was unable to create a conversion table between that codepage "
                        "and Unicode. The display codepage was reset to the process codepage %u.",
                        X._usCodepage,
                        _pCodecProcess->QueryCodepage());
            guiShowMessage2("WarpIN",
                            str,
                            MB_OK);

            _ulDisplayCodepage = 0;
            fRetry = TRUE;
        }
    }

    // register comctl classes
    txvRegisterTextView(_habThread1);
    ctlRegisterTooltip(_habThread1);

}

/*
 *@@ ~GUILocals:
 *      GUI desctructor. Gets called automatically
 *      at the end of main().
 *
 *@@added V0.9.18 (2002-03-03) [umoeller]
 */

GUILocals::~GUILocals()
{
    if (_pDatabase)
    {
        delete _pDatabase;
        _pDatabase = NULL;
    }

    if (_pCodecGui)
    {
        delete _pCodecGui;
        _pCodecGui = NULL;
    }

    winhDestroyHelp(G_hwndHelpInstance, NULLHANDLE);
    G_hwndHelpInstance = NULLHANDLE;

    WinDestroyPointer(G_hptrMain);
    WinDestroyPointer(_hptrSelect);
    WinDestroyPointer(_hptrDeselect);
    WinDestroyPointer(_hptrSomeselect);
    WinDestroyPointer(_hptrDeinstall);
    WinDestroyPointer(_hptrError);
    WinDestroyPointer(_hptrQMark);
    WinDestroyPointer(_hptrOK);
    WinDestroyPointer(_hptrNewer);
    WinDestroyPointer(_hptrApp);

    PrfCloseProfile(_hiniWarpIN);
}

/*
 *@@ ShowMessage:
 *      this should display a message box
 *      with the given strings.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *@@changed V0.9.0 (99-11-01) [umoeller]: renamed, the other GUI message box prototypes have been removed.
 *@@changed V0.9.12 (2001-05-21) [umoeller]: now doing our own messagebox
 *@@changed V0.9.16 (2001-09-20) [umoeller]: adjusted with enShowMessage enum
 *@@changed V0.9.18 (2002-03-08) [umoeller]: turned this into a GUILocals method
 */

ULONG GUILocals::ShowMessage(const ustring &ustrTitle,
                             const ustring &ustrMessage,
                             ULONG fl)
{
    string  strTitle2(_pCodecGui, ustrTitle),
            strMessage2(_pCodecGui, ustrMessage);
    return guiShowMessage2(strTitle2, strMessage2, fl);
}

/*
 *@@ guiOnJobSelectionChanged:
 *      this gets called from the FEJobBase::Select
 *      method for every package selection that has
 *      changed.
 *
 *      The GUI should update its display for that
 *      package then.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *@@changed V0.9.4 (2000-07-26) [umoeller]: renamed from guiJobSelectionChanged
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUILocals method
 */

VOID GUILocals::OnJobSelectionChanged(FEJobBase *pJob)
{
    HWND hwndPackagesCnr;
    if (hwndPackagesCnr = WinWindowFromID(G_hwndPagesClient, ID_WIDI_PACKAGESCNR))
    {
        // get the record core which we created for
        // this package
        PPACKAGERECORD precc = (PPACKAGERECORD)(pJob->_pvGuiData);

        // repaint the record core
        WinSendMsg(hwndPackagesCnr,
                   CM_INVALIDATERECORD,
                   (MPARAM)&precc,
                   MPFROM2SHORT(1, CMA_ERASE));
    }
}

/*
 *@@ fnwpConfirmRexxDlg:
 *      window proc for GUILocals::ConfirmRexxAllowed.
 *
 *@@added V0.9.12 (2001-05-31) [umoeller]
 */

MRESULT EXPENTRY fnwpConfirmRexxDlg(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    BOOL fCallDefault = TRUE;
    MRESULT mrc = 0;

    switch (msg)
    {
        case WM_COMMAND:
            if (SHORT1FROMMP(mp1) == ID_WIMI_VIEW_CONFIG)
            {
                XTEXTVIEWCDATA xtxvCData;
                memset(&xtxvCData, 0, sizeof(xtxvCData));
                xtxvCData.cbData = sizeof(xtxvCData);
                xtxvCData.ulXBorder = 5;
                xtxvCData.ulYBorder = 5;

                CONTROLDEF
                    TextView =
                            {
                                WC_XTEXTVIEW,
                                NULL,       // text, set below
                                WS_VISIBLE | WS_TABSTOP | XS_VSCROLL | XS_AUTOVHIDE | XS_HSCROLL | XS_AUTOHHIDE,
                                        // updated V0.9.20 (2002-08-10) [umoeller]
                                999,          // ID
                                "4.System VIO",
                                XAC_SIZEX | XAC_SIZEY,
                                { 300,
                                    200 },
                                COMMON_SPACING,
                                &xtxvCData
                            },
                    CancelButton =  {
                                     WC_BUTTON,
                                     nlsGetString(WPSI_CLOSE), // "~Close",
                                     WS_VISIBLE | WS_TABSTOP | BS_PUSHBUTTON | BS_DEFAULT,
                                     DID_OK,          // ID
                                     CTL_COMMON_FONT,  // no font
                                     0,
                                     { STD_BUTTON_WIDTH, STD_BUTTON_HEIGHT },
                                     COMMON_SPACING
                                 };

                DLGHITEM ViewDlg[] =
                             {
                                START_TABLE,
                                    START_ROW(ROW_VALIGN_CENTER),
                                        CONTROL_DEF(&TextView),
                                    START_ROW(ROW_VALIGN_CENTER),
                                        CONTROL_DEF(&CancelButton),
                                END_TABLE
                             };

                HWND hwndDlg;
                APIRET arc = dlghCreateDlg(&hwndDlg,
                                           hwnd,
                                           FCF_TITLEBAR | FCF_SYSMENU | FCF_SIZEBORDER | FCF_NOBYTEALIGN,
                                           WinDefDlgProc,
                                           nlsGetString(WPSI_SCRIPT), // "Script",
                                           ViewDlg,
                                           ARRAYITEMCOUNT(ViewDlg),
                                           NULL,
                                           gshrQueryDefaultFont());

                PSZ psz = strdup((const char*)WinQueryWindowPtr(hwnd, QWL_USER));
                txvStripLinefeeds(&psz, 4);
                WinSetDlgItemText(hwndDlg, 999, psz);

                winhCenterWindow(hwndDlg);
                WinProcessDlg(hwndDlg);
                WinDestroyWindow(hwndDlg);

                free(psz);

                fCallDefault = FALSE;
            }
    }

    if (fCallDefault)
        mrc = WinDefDlgProc(hwnd, msg, mp1, mp2);

    return (mrc);
}

/*
 *@@ guiConfirmRexxAllowed:
 *      confirmation callback for the script
 *      classes which gets called if any REXX
 *      code was found in the script.
 *
 *      If this returns TRUE, parsing continues,
 *      otherwise installation is aborted. This
 *      allows paranoid users to stop install
 *      if they are worrying that evil, evil
 *      WarpIN might overwrite their master boot
 *      record or something.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *@@added V0.9.12 (2001-05-31) [umoeller]
 *@@changed V0.9.20 (2002-07-03) [umoeller]: turned this into a GUILocals method
 */

BOOL GUILocals::ConfirmRexxAllowed(const ustring &ustrScript)
{
    static CONTROLDEF
        Icon = {
                        WC_STATIC,
                        NULL,           // text, set below
                        WS_VISIBLE | SS_ICON,
                        0,          // ID
                        NULL,       // no font
                        0,
                        { SZL_AUTOSIZE, SZL_AUTOSIZE },
                        COMMON_SPACING
                    },
        InfoText =
                    {
                        WC_STATIC,
                        NULL,       // text, set below
                        WS_VISIBLE | SS_TEXT | DT_WORDBREAK | DT_LEFT | DT_TOP,
                        -1,          // ID
                        CTL_COMMON_FONT,
                        0,
                        { 200, SZL_AUTOSIZE },
                        COMMON_SPACING
                    },
        InfoText2 =
                    {
                        WC_STATIC,
                        NULL,       // text, set below
                        WS_VISIBLE | SS_TEXT | DT_WORDBREAK | DT_LEFT | DT_TOP,
                        -1,          // ID
                        CTL_COMMON_FONT,
                        0,
                        { 200, SZL_AUTOSIZE },
                        COMMON_SPACING
                    },
        OKButton =  {
                        WC_BUTTON,
                        nlsGetString(WPSI_OK), // "~OK",
                        WS_VISIBLE | WS_TABSTOP | BS_PUSHBUTTON | BS_DEFAULT,
                        DID_OK,          // ID
                        CTL_COMMON_FONT,  // no font
                        0,
                        { STD_BUTTON_WIDTH, STD_BUTTON_HEIGHT },
                        COMMON_SPACING
                    },
        CancelButton =  {
                        WC_BUTTON,
                        nlsGetString(WPSI_CANCEL), // "~Cancel",
                        WS_VISIBLE | WS_TABSTOP | BS_PUSHBUTTON,
                        DID_CANCEL,          // ID
                        CTL_COMMON_FONT,  // no font
                        0,
                        { STD_BUTTON_WIDTH, STD_BUTTON_HEIGHT },
                        COMMON_SPACING
                    },
        ViewButton =  {
                        WC_BUTTON,
                        nlsGetString(WPSI_VIEWSCRIPT), // "~View script...",
                        WS_VISIBLE | WS_TABSTOP | BS_PUSHBUTTON,
                        ID_WIMI_VIEW_CONFIG,   // ID
                        CTL_COMMON_FONT,  // no font
                        0,
                        { STD_BUTTON_WIDTH, STD_BUTTON_HEIGHT },
                        COMMON_SPACING
                    },
        ShowAgainCheckbox = {
                        WC_BUTTON,
                        nlsGetString(WPSI_SHOWMSGAGAIN), // "~Show this message again",
                        WS_VISIBLE | WS_TABSTOP | BS_AUTOCHECKBOX,
                        1000,          // ID
                        CTL_COMMON_FONT,  // no font
                        0,
                        { -1, -1 },
                        COMMON_SPACING
                    };

    static DLGHITEM ConfirmDlg[] =
                 {
                    START_TABLE,
                        START_ROW(ROW_VALIGN_CENTER),
                            CONTROL_DEF(&Icon),
                        START_TABLE,
                            START_ROW(ROW_VALIGN_CENTER),
                                CONTROL_DEF(&InfoText),
                            START_ROW(ROW_VALIGN_CENTER),
                                CONTROL_DEF(&ViewButton),
                            START_ROW(ROW_VALIGN_CENTER),
                                CONTROL_DEF(&InfoText2),
                            START_ROW(ROW_VALIGN_CENTER),
                                CONTROL_DEF(&OKButton),
                                CONTROL_DEF(&CancelButton),
                            START_ROW(ROW_VALIGN_CENTER),
                                CONTROL_DEF(&ShowAgainCheckbox),
                        END_TABLE,
                    END_TABLE
                 };

    if (G_WpiGlobals.fSuppressREXXConfirm)
        // user has previously selected to disable this:
        return TRUE;

    HWND hwndDlg;
    string strTitle, strMsg, strMsg2;
    GetMessageA(strTitle, 108);       // warning
    GetMessageA(strMsg, 206);         // explanation
    GetMessageA(strMsg2, 207);         // "ok = continue, "cancel" = paranoid
    /* V0.9.20 (2002-07-03) [umoeller]
    string strTitle, strMsg, strMsg2;
    strTitle.assignUtf8(((GUILocals *)pLocals)->_pCodecGui, ustrTitle);
    strMsg.assignUtf8(((GUILocals *)pLocals)->_pCodecGui, ustrMsg);
    strMsg2.assignUtf8(((GUILocals *)pLocals)->_pCodecGui, ustrMsg2);
    */

    Icon.pcszText = (const char *)(G_hptrMain);
    InfoText.pcszText = strMsg.c_str();
    InfoText2.pcszText = strMsg2.c_str();
    APIRET arc = dlghCreateDlg(&hwndDlg,
                               NULLHANDLE,
                               FCF_TITLEBAR | FCF_SYSMENU | FCF_DLGBORDER | FCF_NOBYTEALIGN,
                               fnwpConfirmRexxDlg,
                               strTitle.c_str(),
                               ConfirmDlg,
                               ARRAYITEMCOUNT(ConfirmDlg),
                               NULL,
                               gshrQueryDefaultFont());

    ULONG ulrc = DID_CANCEL;

    if (!arc && hwndDlg)
    {
        winhCenterWindow(hwndDlg);
        winhSetDlgItemChecked(hwndDlg, 1000, TRUE);

        string strScript(_pCodecGui, ustrScript);
        WinSetWindowPtr(hwndDlg, QWL_USER, (PVOID)strScript.c_str());

        ulrc = WinProcessDlg(hwndDlg);

        if (!winhIsDlgItemChecked(hwndDlg, 1000))
        {
            G_WpiGlobals.fSuppressREXXConfirm = TRUE;
            SaveSettings(0);
        }

        WinDestroyWindow(hwndDlg);
    }

    return (ulrc == DID_OK);
}

/*
 *@@ Sleep:
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *@@added V0.9.2 (2000-03-05) [umoeller]
 *@@changed V0.9.18 (2002-03-08) [umoeller]: turned this into a GUILocals method
 */

VOID GUILocals::Sleep(ULONG ulSleep)    // in: sleep time in milliseconds
{
    ULONG ul = 0;
    QMSG qmsg;
    for (ul = 0;
         ul < (ulSleep / 50);
         ul++)
    {
        DosSleep(50);
        while (WinPeekMsg(_habThread1,
                          &qmsg, 0, 0, 0,
                          PM_REMOVE))
            WinDispatchMsg(_habThread1, &qmsg);
    }
}

/*
 *@@ GUIEXECUTE:
 *      temporary structure created for fntExecute
 *      parameters.
 *
 *@@added V0.9.9 (2001-03-30) [umoeller]
 */

typedef struct _GUIEXECUTE
{
    GUILocals       *pLocals;
    BSExecute       *pExec;     // executable instance
    BSFileLogger    *pLogFile;  // log file or NULL
} GUIEXECUTE, *PGUIEXECUTE;

/*
 *@@ fntExecute:
 *      synchronous PM thread for executing a
 *      program. This has a PM msg queue and
 *      is run using thrRunSync from guiExecute.
 *
 *@@added V0.9.5 (2000-08-26) [umoeller]
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added log file
 */

void _Optlink fntExecute(PTHREADINFO ptiMyself)
{
    ULONG   ulrc = 0;
    PGUIEXECUTE pge = (PGUIEXECUTE)(ptiMyself->ulData);

    try
    {
        pge->pExec->Execute(*pge->pLocals->_pCodecProcess,
                            pge->pLogFile);
    }
    catch(BSConfigExcpt& X)
    {
        // error occured:
        ulrc = X._iData;        // APIRET
    }

    WinPostMsg(ptiMyself->hwndNotify,
               WM_USER,
               (MPARAM)ulrc,
               0);
}

/*
 *@@ guiExecute:
 *      this gets called from the engine for each
 *      executable which is run because of an EXECUTE
 *      tag.
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 *
 *      We run a second thread for executing and keep
 *      processing thread 1's message queue.
 *
 *@@added V0.9.5 (2000-08-26) [umoeller]
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added log file
 *@@changed V0.9.18 (2002-03-08) [umoeller]: turned this into a GUILocals method
 */

APIRET GUILocals::Execute(BSExecute *pExec)     // in: executable instance
{
    BOOL brc = FALSE;

    GUIEXECUTE ge;
    ge.pLocals = this;
    ge.pExec = pExec;
    ge.pLogFile = _pLogFile;

    ULONG iError = thrRunSync(_habThread1,
                              fntExecute,
                              "Execute",
                              (ULONG)&ge);

    if (iError)
    {
        PSZ pszSysError = doshQuerySysErrorMsg(iError);
        ustring aStrings[2];
        aStrings[0] = pExec->_ustrExecutable;
        aStrings[1].assignCP(_pCodecProcess, pszSysError);

        brc = FALSE;
        MessageBox(102, // "WarpIN: Error"
                   177,   // "error executing program"
                   MSG_WARNING_OK,
                   aStrings, 2);
    }

    return (iError == 0);
}

/*
 *@@ QueryGuiCodec:
 *
 *      Required GUI implementation of the respective
 *      pure virtual FELocals method.
 */

BSUniCodec& GUILocals::QueryGuiCodec()
{
    return *_pCodecGui;
}

/*
 *@@ LoadSettings:
 *      this loads the user settings from WARPIN.INI.
 *
 *@@changed V0.9.18 (2002-03-08) [umoeller]: turned this into a method
 *@@changed V0.9.18 (2002-03-08) [umoeller]: added display codepage
 *@@changed V0.9.19 (2002-04-14) [umoeller]: fixed missing initialization for default paths
 */

VOID GUILocals::LoadSettings()
{
    _GuiSettings.ulLeftView = ID_WIMI_DATABASE_DETAILSVIEW;
    _GuiSettings.ulRightView = ID_WIMI_VIEW_FILES;

    ULONG cbData = sizeof(_GuiSettings);
    PrfQueryProfileData(_hiniWarpIN,
                        "GUI", "Database",
                        &_GuiSettings, &cbData);
    cbData = sizeof(_ulIfSameDate);
    PrfQueryProfileData(_hiniWarpIN,
                        "Settings", "ulIfSameDate",
                        &_ulIfSameDate, &cbData);
    cbData = sizeof(_ulIfExistingOlder);
    PrfQueryProfileData(_hiniWarpIN,
                        "Settings", "ulIfExistingOlder",
                        &_ulIfExistingOlder, &cbData);
    cbData = sizeof(_ulIfExistingNewer);
    PrfQueryProfileData(_hiniWarpIN,
                        "Settings", "ulIfExistingNewer",
                        &_ulIfExistingNewer, &cbData);

    cbData = sizeof(G_WpiGlobals.fSuppressREXXConfirm);
    PrfQueryProfileData(_hiniWarpIN,
                        "Settings", "SuppressRexxConfirm",
                        &G_WpiGlobals.fSuppressREXXConfirm, &cbData);

    // removed environment checks here,
    // we get the environment later
    // V0.9.19 (2002-04-14) [umoeller]
    CHAR    szTemp[300];
    cbData = sizeof(szTemp);
    if (PrfQueryProfileData(_hiniWarpIN,
                            "Settings", "DefaultAppsPath",
                            szTemp, &cbData))
    {
        // found:
        _strDefaultAppsPath._printf(
                "WARPIN_DEFAULTAPPSPATH=%s",
                szTemp);
        putenv(_strDefaultAppsPath.c_str());
    }

    cbData = sizeof(szTemp);
    if (PrfQueryProfileData(_hiniWarpIN,
                            "Settings", "DefaultToolsPath",
                            szTemp, &cbData))
    {
        // found:
        _strDefaultToolsPath._printf(
                "WARPIN_DEFAULTTOOLSPATH=%s",
                szTemp);
        putenv(_strDefaultToolsPath.c_str());
    }

    // override settings via environment, if present
    // V0.9.19 (2002-04-14) [umoeller]
    FELocals::GetSettingsFromEnvironment();

    PSZ pszLogFilename;
    if (pszLogFilename = prfhQueryProfileData(_hiniWarpIN,
                                              "Settings",
                                              "LogFile",
                                              NULL))
    {
        // assume this setting is in UTF now V0.9.20 (2002-07-03) [umoeller]
        _ustrLogFilename.assignUtf8(pszLogFilename);
        free(pszLogFilename);
    }

    CHAR sz[10];
    PrfQueryProfileString(_hiniWarpIN,
                          "Settings", "DisplayCodepage",
                          "0",
                          sz,
                          sizeof(sz));
    _ulDisplayCodepage = atoi(sz);
}

/*
 *@@ SaveSettings:
 *      this saves the user settings back into WARPIN.INI.
 *
 *      If fl has the SETFL_SAVELOGFILEPATH bit set, the
 *      log file name is also saved.
 *
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added fl
 *@@changed V0.9.18 (2002-03-08) [umoeller]: turned this into a method
 *@@changed V0.9.18 (2002-03-08) [umoeller]: added display codepage
 */

VOID GUILocals::SaveSettings(ULONG fl)
{
    PrfWriteProfileData(_hiniWarpIN, "GUI", "Database",
                        &_GuiSettings, sizeof(_GuiSettings));
    PrfWriteProfileData(_hiniWarpIN, "Settings", "ulIfSameDate",
                        &_ulIfSameDate, sizeof(_ulIfSameDate));
    PrfWriteProfileData(_hiniWarpIN, "Settings", "ulIfExistingOlder",
                        &_ulIfExistingOlder, sizeof(_ulIfExistingOlder));
    PrfWriteProfileData(_hiniWarpIN, "Settings", "ulIfExistingNewer",
                        &_ulIfExistingNewer, sizeof(_ulIfExistingNewer));

    PrfWriteProfileData(_hiniWarpIN,
                        "Settings", "SuppressRexxConfirm",
                        &G_WpiGlobals.fSuppressREXXConfirm, sizeof(G_WpiGlobals.fSuppressREXXConfirm));


    PSZ pszCommand = "WARPIN_DEFAULTAPPSPATH";
    PSZ pszValue;
    if ((pszValue = getenv(pszCommand)))
    {
        // environment variable set:
        PrfWriteProfileString(_hiniWarpIN,
                              "Settings", "DefaultAppsPath",
                              pszValue);
    }

    pszCommand = "WARPIN_DEFAULTTOOLSPATH";
    if ((pszValue = getenv(pszCommand)))
    {
        // environment variable set:
        PrfWriteProfileString(_hiniWarpIN,
                              "Settings", "DefaultToolsPath",
                              pszValue);
    }

    if (fl & SETFL_SAVELOGFILEPATH)
        PrfWriteProfileString(_hiniWarpIN,
                              "Settings", "LogFile",
                              (_ustrLogFilename())
                                 ? _ustrLogFilename.GetBuffer()
                                 : NULL);

    CHAR sz[10];
    _itoa(_ulDisplayCodepage, sz, 10);
    PrfWriteProfileString(_hiniWarpIN,
                          "Settings", "DisplayCodepage",
                          sz);
}

/*
 *@@ SetWindowTextU:
 *      sneaky helper to set a window text
 *      while converting to the GUI codepage
 *      at the same time.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

BOOL GUILocals::SetWindowTextU(HWND hwnd,
                               const ustring &ustr)
{
    string str(_pCodecGui, ustr);
    return WinSetWindowText(hwnd, str.c_str());
}

/*
 *@@ SetDlgItemTextU:
 *      like GUILocals::SetWindowTextU, but
 *      with a dialog ID instead.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

BOOL GUILocals::SetDlgItemTextU(HWND hwnd,
                                ULONG id,
                                const ustring &ustr)
{
    return SetWindowTextU(WinWindowFromID(hwnd, id),
                          ustr);
}

/*
 *@@ GetMessageA:
 *      sneaky shortcut for getting a message (this
 *      calls FELocals::GetMessage with the given
 *      parameters) and converting it to the GUI
 *      codepage.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

APIRET GUILocals::GetMessageA(string &strBuf,
                              ULONG ulMsgNumber,
                              ustring *paStrings,
                              ULONG ulTable)
{
    APIRET arc;
    ustring ustrBuf;
    if (!(arc = GetMessage(ustrBuf,
                           ulMsgNumber,
                           paStrings,
                           ulTable)))
        strBuf.assignUtf8(_pCodecGui, ustrBuf);

    return arc;
}

/* ******************************************************************
 *
 *  Various helper functions
 *
 ********************************************************************/

/*
 *@@ DoCreateWarpINObjects:
 *      this gets called by guiInitialize if the
 *      WarpIN program object does not exist yet
 *      to create the WarpIN objects.
 *
 *      Returns TRUE only if all objects could be
 *      created.
 *
 *@@added (99-10-22) [umoeller]
 *@@changed V0.9.3 (2000-04-28) [umoeller]: added readme shadow
 *@@changed V0.9.3 (2000-05-23) [umoeller]: changed to "replace" objects mode
 *@@changed V0.9.14 (2001-07-24) [umoeller]: added history shadow
 */

BOOL DoCreateWarpINObjects(PCSZ pcszWarpINPath)
{
    HOBJECT     hobjFolder = NULLHANDLE;
    if ((hobjFolder = WinCreateObject("WPFolder",
                                      "WarpIN",
                                      "OBJECTID="WPOBJID_WARPINFOLDER";",
                                      "<WP_DESKTOP>",
                                      CO_REPLACEIFEXISTS))
                == NULLHANDLE)
        return (FALSE);

    WinSetObjectData(hobjFolder, "OPEN=DEFAULT");

    BSString    strSetupString;

    BSString    strWarpINFile = pcszWarpINPath;
    strWarpINFile += "\\WARPIN.EXE";

    strSetupString._printf(
            "ASSOCFILTER=*.WPI;ASSOCTYPE=WarpIN Archive;EXENAME=%s;PROGTYPE=PM;STARTUPDIR=%s;"
                "OBJECTID=" WPOBJID_WARPINEXE";",
            strWarpINFile.c_str(),        // EXEFILE
            pcszWarpINPath);       // startup dir

    if (!WinCreateObject("WPProgram",
                         "WarpIN",
                         strSetupString.c_str(),
                         WPOBJID_WARPINFOLDER,
                         CO_REPLACEIFEXISTS))
        return (FALSE);

    // OK, successfull install:
    // store our path in OS2.INI so that future instances of
    // WarpIN can know if they're not being started from the
    // initial location
    PrfWriteProfileString(HINI_USER,
                          INIAPP_WARPIN,
                          INIKEY_WARPINPATH,
                          pcszWarpINPath);

    // WarpIN User's Guide
    strSetupString._printf(
            "EXENAME=VIEW.EXE;PROGTYPE=PM;PARAMETERS=WPI_USER.INF;STARTUPDIR=%s;"
                "OBJECTID=" WPOBJID_WARPINPROGGUIDE";",
            pcszWarpINPath);       // startup dir
    WinCreateObject("WPProgram",
                    "WarpIN User's Guide",
                    strSetupString.c_str(),
                    WPOBJID_WARPINFOLDER,
                    CO_REPLACEIFEXISTS);

    // WarpIN Programmer's Guide and Reference
    strSetupString._printf(
            "EXENAME=VIEW.EXE;PROGTYPE=PM;PARAMETERS=WPI_PROG.INF;STARTUPDIR=%s;"
                "OBJECTID=" WPOBJID_WARPINUSERGUIDE";",
            pcszWarpINPath);       // startup dir
    WinCreateObject("WPProgram",
                    "WarpIN Programmer's Guide and Reference",
                    strSetupString.c_str(),
                    WPOBJID_WARPINFOLDER,
                    CO_REPLACEIFEXISTS);

    // README (shadow)
    strSetupString._printf("SHADOWID=%s\\readme.txt;",
                           pcszWarpINPath);
    WinCreateObject("WPShadow",
                    "readme.txt",
                    strSetupString.c_str(),
                    WPOBJID_WARPINFOLDER,
                    CO_REPLACEIFEXISTS);

    // history (shadow) V0.9.14 (2001-07-24) [umoeller]
    strSetupString._printf("SHADOWID=%s\\history.txt;",
                           pcszWarpINPath);
    WinCreateObject("WPShadow",
                    "history.txt",
                    strSetupString.c_str(),
                    WPOBJID_WARPINFOLDER,
                    CO_REPLACEIFEXISTS);

    return (TRUE);
}

/*
 *@@ guiCreateWarpINObjects:
 *
 *      This also gets called from EnterInstallMode
 *      if WarpIN performed a stub-controlled self-install.
 *
 *@@added V0.9.14 (2001-08-09) [umoeller]
 *@@changed V0.9.20 (2002-07-03) [umoeller]: now using ustring
 */

BOOL guiCreateWarpINObjects(GUILocals *pLocals,
                            const ustring &ustrWarpINPath,
                            BOOL fForceRecreateObjects)
{
    BOOL brc = TRUE;

    string strWarpINPath(pLocals->_pCodecProcess, ustrWarpINPath);
    PCSZ pcszWarpINPath = strWarpINPath.c_str();

    if (WinQueryObject("<WP_DESKTOP>"))
    {
        // WPS running:
        // check installation (99-10-22) [umoeller]
        HOBJECT hobjWarpinExe = WinQueryObject(WPOBJID_WARPINEXE);
        if (    (!hobjWarpinExe)
             || (fForceRecreateObjects)
           )
        {
            ULONG ulrc = DID_OK;
            // does not exist yet:
            // prompt with install text,
            // but only if we haven't forced update above
            if (!fForceRecreateObjects)
            {
                ustring str;
                str.assignUtf8(BLDLEVEL_VERSION);
                ulrc = gshrMLEMessageBox(pLocals,
                                         149,      // title
                                         148,       // create objects?
                                         TRUE,     // show cancel
                                         &str, 1);  // string replacements
            }

            if (ulrc == DID_OK)
            {
                if (DoCreateWarpINObjects(pcszWarpINPath))
                    // success:
                    pLocals->MessageBox(100,
                                        152,
                                        MSG_WARNING_OK);
                else
                    // error creating objects:
                    pLocals->MessageBox(102,
                                        271, // An error occured creating the WarpIN WPS objects.
                                        MSG_WARNING_OK);
            }

            brc = FALSE;
                // in any case, don't continue
        }
    }

    return (brc);
}

typedef struct _UPGRADEINFO
{
    PCSZ        pcszPathToUpgrade;
    PCSZ        pcszPathThis;
    PCSZ        pcszFailed;         // out: file that failed.
    APIRET      arc;
    GUILocals    *pLocals;
} UPGRADEINFO, *PUPGRADEINFO;

/*
 *@@ fntUpgradeExisting:
 *      synchronous thread started by CheckInstall
 *      to copy objects over to the existing installation.
 *
 *@@added V0.9.7 (2001-01-07) [umoeller]
 */

VOID _Optlink fntUpgradeExisting(PTHREADINFO ptiMyself)
{
    BOOL brc = TRUE;

    TRY_LOUD(excpt1)
    {
        // copy info
        PUPGRADEINFO pInfo = (PUPGRADEINFO)(ptiMyself->ulData);

        HWND hwndInfo = WinWindowFromID(G_hwndSplashDlg, ID_WIDI_INFOTEXT);

        WinSetWindowULong(hwndInfo,
                          QWL_STYLE,
                          WS_VISIBLE | SS_TEXT | DT_LEFT | DT_TOP | DT_WORDBREAK);
        winhSetWindowFont(hwndInfo, "8.Helv");

        ustring strTemplate;
        pInfo->pLocals->GetMessage(strTemplate,
                                   197);         // "Copying "%s" to "%s""

        ULONG ul;
        for (ul = 0;
             ul < sizeof(G_apcszUpgradeFiles) / sizeof(G_apcszUpgradeFiles[0]);   // array item count
             ul++)
        {
            PCSZ pcszFileThis = G_apcszUpgradeFiles[ul];

            string   strPathThis = pInfo->pcszPathThis,
                     strPathToUpgrade = pInfo->pcszPathToUpgrade,
                     strFileThis = pcszFileThis;
            string   strInfo;
            string   str1 = strPathThis + "\\" + strFileThis,
                     str2 = strPathToUpgrade + "\\" + strFileThis;
            strInfo._printf(strTemplate.GetBuffer(), // "Copying "%s" to "%s""
                            str1.c_str(),
                            str2.c_str());
            WinSetWindowText(hwndInfo, strInfo.c_str());

            APIRET arc = DosCopy(str1.c_str(),
                                 str2.c_str(),
                                 DCPY_EXISTING);        // overwrite
            if (arc != NO_ERROR)
            {
                pInfo->pcszFailed = pcszFileThis;
                pInfo->arc = arc;
                // return FALSE
                brc = FALSE;
            }

        }
    }
    CATCH(excpt1)
    {
        brc = FALSE;
    } END_CATCH();

    WinPostMsg(ptiMyself->hwndNotify, WM_USER, (MPARAM)brc, 0);
}

/*
 *@@ guiStoreWarpINPath:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

VOID guiStoreWarpINPath(GUILocals *pLocals)
{
    string str(pLocals->_pCodecProcess, pLocals->QueryWarpINPath());
    PrfWriteProfileString(HINI_USER,
                          INIAPP_WARPIN,
                          INIKEY_WARPINPATH,
                          str.c_str());
}

/*
 *@@ CheckInstall:
 *      called by main() to check if WarpIN is installed
 *      or should be now.
 *      If not, CreateWarpINObjects() gets called in turn.
 *
 *      Returns TRUE if installation can continue. If FALSE is
 *      returned, WarpIN should terminate.
 *
 *      Required callback (V0.9.14).
 *
 *@@added V0.9.7 (2001-01-06) [umoeller]
 *@@changed V0.9.9 (2001-04-06) [umoeller]: now creating "test" in target
 *@@changed V0.9.14 (2001-07-22) [umoeller]: reworked, now accepting command-line options
 *@@changed V0.9.14 (2001-08-09) [umoeller]: fixed first-time install
 *@@changed V0.9.20 (2002-07-06) [umoeller]: fixed wrong prompts with install dirs with non-ascii chars
 */

BOOL guiCheckInstall(GUILocals *pLocals)
{
    BOOL brc = TRUE,
         fForceRecreateObjects = FALSE,
         fUserPrompted = FALSE;

    BSUniCodec *pCodecProcess = pLocals->_pCodecProcess;

    /*
     * step 1: check setup mode and set it if
     *         required
     *
     */

    CHAR szLastPath[CCHMAXPATH] = "";
    // check if WARPIN.EXE has been installed in a different
    // directory before
    BOOL fAlreadyInstalled = PrfQueryProfileString(HINI_USER,
                                                   INIAPP_WARPIN,
                                                   INIKEY_WARPINPATH,
                                                   NULL,     // default string
                                                   szLastPath,
                                                   sizeof(szLastPath));

    // only do this if user hasn't specified anything on
    // the command line
    if (G_WpiGlobals.SetupMode == WPIGLOBALS::UNSPECIFIED)
    {
        if (!fAlreadyInstalled)
            // not installed at all:
            G_WpiGlobals.SetupMode = WPIGLOBALS::DISABLEEXISTING;
                    // V0.9.14 (2001-08-09) [umoeller]
        else
        {
            // OK, WarpIN has been installed already:

            // string strCurrent(pCodecProcess, pLocals->QueryWarpINPath());
            // if (stricmp(strCurrent.c_str(), szLastPath))

            // this didn't work with non-ASCII characters...
            // this kept prompting if WarpIN was is a directory with those
            // so do it with the new Unicode thing!
            // V0.9.20 (2002-07-03) [umoeller]
            ustring ustrLastPath(pCodecProcess, szLastPath);
            if (ustrLastPath.compareI(pLocals->QueryWarpINPath()))
            {
                // different: give warning
                BOOL    fAllowUpgrade = TRUE;
                CHAR    szOtherVersion[100] = "?";

                // get version number of the other installation
                PEXECUTABLE pOtherExec = {0};
                CHAR        szOtherExec[2*CCHMAXPATH];
                sprintf(szOtherExec, "%s\\WARPIN.EXE", szLastPath);

                APIRET arc;
                if (!(arc = exehOpen(szOtherExec, &pOtherExec)))
                {
                    if (!(arc = exehQueryBldLevel(pOtherExec)))
                    {
                        if (pOtherExec->pszVersion)
                        {
                            ULONG   ulMajorThis,
                                    ulMinorThis,
                                    ulRevThis,
                                    ulMajorOther,
                                    ulMinorOther,
                                    ulRevOther;
                            strcpy(szOtherVersion, pOtherExec->pszVersion);
                            // compare versions...
                            sscanf(szOtherVersion,
                                   "%d.%d.%d",
                                   &ulMajorOther,
                                   &ulMinorOther,
                                   &ulRevOther);
                            sscanf(BLDLEVEL_VERSION,
                                   "%d.%d.%d",
                                   &ulMajorThis,
                                   &ulMinorThis,
                                   &ulRevThis);

                            if (    (ulMajorOther > ulMajorThis)
                                 || (   (ulMajorOther == ulMajorThis)
                                     && (ulMinorOther > ulMinorThis)
                                    )
                                 || (   (ulMajorOther == ulMajorThis)
                                     && (ulMinorOther == ulMinorThis)
                                     && (ulRevOther > ulRevThis)
                                    )
                               )
                                fAllowUpgrade = FALSE;
                            // else: OK, allow upgrade
                        }
                        else
                            fAllowUpgrade = FALSE;
                    }
                    else
                        fAllowUpgrade = FALSE;

                    exehClose(&pOtherExec);

                } // if (arc == NO_ERROR)
                else
                    fAllowUpgrade = FALSE;

                // show msg box
                ustring strUpgrade;
                ustring aStrings[4];
                aStrings[0].assignCP(pCodecProcess, szLastPath);
                aStrings[1].assignCP(pCodecProcess, szOtherVersion);
                aStrings[2].assignCP(pCodecProcess, BLDLEVEL_VERSION);
                aStrings[3] = pLocals->QueryWarpINPath();
                pLocals->GetMessage(strUpgrade,
                                    151, // "previously installed -- update?"
                                    aStrings, 4); // string replacements: directory
                HWND hwndUpgrade = gshrLoadMLE(pLocals,
                                               ID_WID_UPGRADE,
                                               strUpgrade,
                                               TRUE);       // HTML
                if (!fAllowUpgrade)
                    // existing is newer:
                    // disable "upgrade existing"
                    WinEnableControl(hwndUpgrade, 98, FALSE);

                // set window title
                ustring strTitle;
                pLocals->GetMessage(strTitle,
                                    108); // "warning"
                pLocals->SetWindowTextU(hwndUpgrade, strTitle);

                winhCenterWindow(hwndUpgrade);
                // go!!
                ULONG ulrc = WinProcessDlg(hwndUpgrade);
                switch (ulrc)
                {
                    case 98:        // upgrade existing
                        G_WpiGlobals.SetupMode = WPIGLOBALS::UPGRADEEXISTING;
                        fUserPrompted = TRUE;
                    break;

                    case 99:        // disable existing
                        G_WpiGlobals.SetupMode = WPIGLOBALS::DISABLEEXISTING;
                        fUserPrompted = TRUE;
                    break;

                    default:
                        // no continue: get outta here
                        brc = FALSE;
                }

                WinDestroyWindow(hwndUpgrade);
            } // end if (stricmp(szLastPath, G_WpiGlobals.szWarpINPath) != 0)
        } // end if (PrfQueryProfileString(HINI_USER, INIAPP_WARPIN, INIKEY_WARPINPATH,
    } // end if (G_WpiGlobals.SetupMode == WPIGLOBALS::UNSPECIFIED)

    /*
     * step 2: do self-install according to setup mode
     *
     */

    switch (G_WpiGlobals.SetupMode)
    {
        /*
         * UPGRADEEXISTING:
         *      copy executables to existing WarpIN path.
         */

        case WPIGLOBALS::UPGRADEEXISTING:
        {
            string strWarpINCurrent(pCodecProcess, pLocals->QueryWarpINPath());

            if (!szLastPath[0])
                throw FEFatalErrorExcpt(*pLocals, 211);       // no existing WarpIN
            else
            {
                // make sure the two paths are different V0.9.14 (2001-07-22) [umoeller]
                if (strWarpINCurrent == szLastPath)
                    throw FEFatalErrorExcpt(*pLocals, 212);       // two paths are equal
            }

            // create "test" subdirectory in target path
            // V0.9.9 (2001-04-06) [umoeller]
            string strTemp = szLastPath;
            if (strTemp[strTemp.length() - 1] != '\\')
                strTemp += "\\";
            strTemp += "test";
            DosCreateDir(strTemp.c_str(), NULL);
            // end V0.9.9 (2001-04-06) [umoeller]

            UPGRADEINFO ui;
            ui.pcszPathToUpgrade = szLastPath;
            ui.pcszPathThis = strWarpINCurrent.c_str();
            ui.pLocals = pLocals;
            if (thrRunSync(pLocals->_habThread1,
                           fntUpgradeExisting,
                           "UpgradeExisting",
                           (ULONG)&ui)       // no param...
                == FALSE)
            {
                // error copying:
                PSZ pszDescription = doshQuerySysErrorMsg(ui.arc);
                ustring aStrings[3];
                aStrings[0]._itoa10(ui.arc, 0);
                aStrings[1].assignCP(pCodecProcess,
                                     ui.pcszFailed);
                aStrings[2].assignCP(pCodecProcess, pszDescription);
                pLocals->MessageBox(100,
                                  198,      // error %1 copying %2
                                  MSG_ERROR_ABORT,
                                  aStrings,
                                  3);
                free(pszDescription);
            }
            else
            {
                ustring str(pCodecProcess, szLastPath);
                pLocals->MessageBox(100,
                              199,      // success
                              MSG_ERROR_ABORT,
                              &str, 1);
            }

            // in any case, do not continue
            brc = FALSE;
        }
        break;

        case WPIGLOBALS::DISABLEEXISTING:
        {
            // update profile data
            guiStoreWarpINPath(pLocals);

            ustring str;
            str = pLocals->QueryWarpINPath();        // V0.9.14 (2001-07-22) [umoeller]
            if (pLocals->MessageBox(100,
                                    // update prog objects?
                                    200,
                                    MSG_CONFIRM_YESNO_DEFYES,
                                    &str,
                                    1)
                        == MBID_YES)
                fForceRecreateObjects = TRUE;
        }
        break;
    }

    /*
     * step 3: create WarpIN WPS objects
     *
     */

    if (brc && G_WpiGlobals.SetupMode != WPIGLOBALS::IGNORE)
    {
        brc = guiCreateWarpINObjects(pLocals,
                                     pLocals->QueryWarpINPath(),
                                     fForceRecreateObjects);
    }

    return (brc);
}

/* ******************************************************************
 *
 *  Helpers for "Full install" and "Add/remove" modes
 *
 ********************************************************************/

/*
 *@@ UpdatePackageDisplay:
 *      this updates the display on the "Container"
 *      page.
 *
 *      This is called in two situations: when a
 *      package has been (de)selected or when the
 *      emphasis changes.
 *
 *@@changed V0.9.4 (2000-07-26) [umoeller]: fixed outdated dlg items
 *@@changed V0.9.18 (2002-03-08) [umoeller]: fixed bad groups display
 */

VOID UpdatePackageDisplay(GUIInstallEngine &Engine)
{
    GUILocals *pLocals = (GUILocals *)&Engine._Locals;
    FEInstallJob *pInstallJob = G_preccSelected->pJob->IsInstallJob();

    HWND    G_hwndPageContainer = G_hwndPagesClient;      // hack

    BOOL    fEnable = FALSE;

    // update container page
    winhShowDlgItem(G_hwndPageContainer, ID_WIDI_PATHENTRY,
                    (pInstallJob != NULL));
    winhShowDlgItem(G_hwndPageContainer, ID_WIDI_PATHTEXT,
                    (pInstallJob != NULL));
    // added these two
    // V0.9.18 (2002-03-08) [umoeller]
    winhShowDlgItem(G_hwndPageContainer, ID_WIDI_PCKINSTALLED_TXT,
                    (pInstallJob != NULL));
    winhShowDlgItem(G_hwndPageContainer, ID_WIDI_PCKINSTALLED,
                    (pInstallJob != NULL));
    if (pInstallJob)
    {
        const FEArcPackagePck *pPackage = &pInstallJob->_ArcPackage;

        // enable target path entry field if...
        BOOL fEnable = (
                            // package has been selected
                            (pInstallJob->Is2BeInstalled())
                            // and path is not fixed
                         && (pPackage->_pDecl->_ulPathType & PATH_VARIABLE)
                            // and package is not installed yet
                         && (pInstallJob->_ulInstallStatus == INSTALLED_NO)
                       );

        winhEnableDlgItem(G_hwndPageContainer, ID_WIDI_PATHENTRY,
                          fEnable);
        winhEnableDlgItem(G_hwndPageContainer, ID_WIDI_PATHTEXT,
                          fEnable);

        // and set target path
        pLocals->SetDlgItemTextU(G_hwndPageContainer,
                                 ID_WIDI_PATHENTRY,
                                 pInstallJob->QueryTargetPath());

        // set installed version
        if (pInstallJob->_ulInstallStatus == INSTALLED_NO)
        {
            // not installed:
            WinEnableControl(G_hwndPageContainer, ID_WIDI_PCKINSTALLED_TXT, FALSE);
            WinSetDlgItemText(G_hwndPageContainer, ID_WIDI_PCKINSTALLED,
                              "");
        }
        else
        {
            WinEnableControl(G_hwndPageContainer, ID_WIDI_PCKINSTALLED_TXT, TRUE);
            WinSetDlgItemText(G_hwndPageContainer, ID_WIDI_PCKINSTALLED,
                              pInstallJob->_pDBPackageInstalled->_PckID._strVersion.c_str());
        }
    }

    Engine.UpdateGlobalStatus(GSTAT_COLLECTING);
}

/*
 *@@ CnrPackageOwnerDraw:
 *      helper function for container owner draw. This
 *      is shared between several window procedures,
 *      used with package record cores when WM_DRAWITEM
 *      comes in.
 *
 *      We need container owner draw to draw the package
 *      icons ourselves.
 *
 *      This returns the BOOL which must be returned
 *      from WM_DRAWITEM.
 *
 *@@added V0.9.0 (99-11-01) [umoeller]
 */

MRESULT CnrPackageOwnerDraw(GUILocals *pLocals,
                            POWNERITEM poi)
{
    MRESULT mrc = (MPARAM)FALSE;

    // check if we're to draw the text or the icon
    if (poi->idItem == CMA_ICON)
    {
        // icon:

        PCNRDRAWITEMINFO pcdii = (PCNRDRAWITEMINFO)poi->hItem;
        HPOINTER hptr;

        // 1) draw current install status to the left
        FEJobBase *pJob =  ((PPACKAGERECORD)(pcdii->pRecord))->pJob;
        FEInstallJob *pInstallJob = pJob->IsInstallJob();
        if (pInstallJob)
            if (pInstallJob->_ulInstallStatus != INSTALLED_NO)
            {
                HPOINTER hptrDraw;
                if (pInstallJob->_ulInstallStatus == INSTALLED_IS_NEWER)
                    hptrDraw = pLocals->_hptrNewer;
                else
                    hptrDraw = pLocals->_hptrOK;
                WinDrawPointer(poi->hps,
                               poi->rclItem.xLeft,
                               poi->rclItem.yBottom,
                               hptrDraw,
                               DP_MINI);
            }

        // 2) draw selection status (for installation)
        switch (pJob->QuerySelection())
        {
            case JOB_INSTALL:
                hptr = pLocals->_hptrSelect;
            break;

            case JOB_GROUP_MIXED:
                hptr = pLocals->_hptrSomeselect;
            break;

            case JOB_DEINSTALL:
                hptr = pLocals->_hptrDeinstall;
            break;

            default:    // JOB_INSTALL_IGNORE
                hptr = pLocals->_hptrDeselect;
            break;

        }

        WinDrawPointer(poi->hps,
                       poi->rclItem.xLeft + pLocals->_ulMiniIconSize + 5,
                       poi->rclItem.yBottom,
                       hptr,
                       DP_MINI);

        // mrc = (MPARAM)TRUE;     // we've drawn the item
    }

    return (mrc);
}

/*
 *@@ Packages2Cnr:
 *      helper function which inserts all packages into
 *      the given container window, which should be in
 *      Tree view to give a meaningful display.
 *
 *@@added V0.9.0 (99-11-01) [umoeller]
 *@@changed V0.9.12 (2001-05-17) [umoeller]: fixed crashes with group jobs
 *@@changed V0.9.19 (2002-04-14) [umoeller]: fixed stupid crash introduced with 0.9.18
 */

ULONG Packages2Cnr(GUIInstallEngine &Engine,
                   HWND hwndCnr)
{
    GUILocals       *pLocals = (GUILocals *)&Engine._Locals;

    ULONG           ulCount = 0;
    PPACKAGERECORD  preccThis;
    ustring         ustrTitleWithSize;

    // go thru the list of packages
    // (this has both "groups" and real packages)
    list<FEJobBase*>::iterator JobStart = Engine._pAllJobsList->begin(),
                               JobEnd = Engine._pAllJobsList->end();
    for (; JobStart != JobEnd; JobStart++)
    {
        FEJobBase *pJobThis = *JobStart;

        // create record core for this package
        preccThis = (PPACKAGERECORD)cnrhAllocRecords(hwndCnr,
                                                     sizeof(PACKAGERECORD),
                                                     1);
        ULONG ulAttrs = CRA_RECORDREADONLY;

        // store job in recc
        preccThis->pJob = pJobThis;
        // and recc in job
        pJobThis->_pvGuiData = (PRECORDCORE)preccThis;

        const FEArcPackageBase *pPackageThis = NULL;

        FEGroupJob *pGroupJob;

        if (pGroupJob = pJobThis->IsGroupJob())
        {
            // is group:
            FEGroupDecl *pDecl = (FEGroupDecl*)pGroupJob->_pGroupPackage->_pDecl;
            ustrTitleWithSize = pDecl->_ustrTitle;
            if (pDecl->_fInitiallyExpanded)
                ulAttrs |= CRA_EXPANDED;
            else
                ulAttrs |= CRA_COLLAPSED;

            pPackageThis = pGroupJob->_pGroupPackage;
        }
        else
        {
            // is package:
            FEInstallJob *pInstallJob;
            if (pInstallJob = pJobThis->IsInstallJob())
                    // fixed crash here V0.9.19 (2002-04-14) [umoeller]
            {
                const FEArcPackagePck *pPckPckThis = &pInstallJob->_ArcPackage;
                pPackageThis = pPckPckThis;
                // only for packages: sum up size
                CHAR szSizeRounded[50];
                nlsThousandsULong(szSizeRounded,
                                  pPackageThis->_ulSizeRounded,
                                  G_szThousand[0]);

                ustrTitleWithSize = pPckPckThis->_pDecl->_ustrTitle;
                ustrTitleWithSize.appendUtf8("\n(V");
                ustrTitleWithSize.appendUtf8(pPckPckThis->_PckID._strVersion.c_str());
                ustrTitleWithSize.appendUtf8(", ");
                ustrTitleWithSize.appendUtf8(szSizeRounded);
                ustrTitleWithSize.appendUtf8(" KB)");
            }
        }

        PRECORDCORE preccParent = NULL;
        // if the package belongs to a group, find
        // the group to get the parent record
        if (pPackageThis->_pGroup)
        {
            FEGroupJob *pGroupJob;
            if (pGroupJob = Engine.FindGroupJob(pPackageThis->_pGroup))
                preccParent = (PRECORDCORE)(pGroupJob->_pvGuiData);
        }

        // insert record core into cnr
        string str2(pLocals->_pCodecGui, ustrTitleWithSize);
        cnrhInsertRecords(hwndCnr,
                          // parent record:
                          preccParent,
                          (PRECORDCORE)preccThis,
                          TRUE,
                          strdup(str2.c_str()),
                          ulAttrs,
                          1);
        ulCount++;
    }

    return (ulCount);
}

/*
 *@@ CreateCnrTooltip:
 *      creates a tooltip control for the packages
 *      container. The tooltip window handle is
 *      stored in G_hwndCnrTooltip.
 *
 *@@added V0.9.2 (2000-03-10) [umoeller]
 */

VOID CreateCnrTooltip(HWND hwndOwner,
                      HWND hwndCnr)
{
    G_hwndCnrTooltip = WinCreateWindow(HWND_DESKTOP,  // parent
                                       COMCTL_TOOLTIP_CLASS, // wnd class
                                       "",            // window text
                                       TTS_SHADOW | TTF_SHYMOUSE | TTS_ALWAYSTIP,
                                            // tooltip window style
                                       10, 10, 10, 10,    // window pos and size, ignored
                                       hwndOwner, // owner window -- important!
                                       HWND_TOP,      // hwndInsertBehind, ignored
                                       19999, // window ID, optional
                                       NULL,          // control data
                                       NULL);         // presparams
    TOOLINFO    ti;
    ti.ulFlags = TTF_SUBCLASS;
    ti.hwndToolOwner = hwndOwner;
    ti.pszText = PSZ_TEXTCALLBACK;  // send TTN_NEEDTEXT
    ti.hwndTool = hwndCnr;
    // add container as tool
    WinSendMsg(G_hwndCnrTooltip,
               TTM_ADDTOOL,
               (MPARAM)0,
               &ti);
    // set timers
    WinSendMsg(G_hwndCnrTooltip,
               TTM_SETDELAYTIME,
               (MPARAM)TTDT_AUTOPOP,
               (MPARAM)(20*1000));        // 20 secs for autopop (hide)
}

static string G_strTooltip;

/*
 *@@ GetTooltipText:
 *      implementation for tooltip's TTN_NEEDTEXT.
 *
 *@@added V0.9.2 (2000-03-10) [umoeller]
 */

VOID GetTooltipText(GUILocals *pLocals,
                    HWND hwndOwner,
                    HWND hwndCnr,
                    MPARAM mp2)
{
    // find record under mouse
    POINTL ptlMouse;
    WinQueryMsgPos(WinQueryAnchorBlock(hwndOwner),
                   &ptlMouse);
    PPACKAGERECORD precc
        = (PPACKAGERECORD)cnrhFindRecordFromPoint(hwndCnr,
                                                  &ptlMouse,
                                                  NULL,
                                                  CMA_ICON | CMA_TEXT,
                                                  FRFP_SCREENCOORDS);
    if (precc)
        if (precc->pJob)
        {
            FEInstallJob *pJob;
            if (pJob = precc->pJob->IsInstallJob())
            {
                PTOOLTIPTEXT pttt = (PTOOLTIPTEXT)mp2;
                pttt->ulFormat = TTFMT_PSZ;
                G_strTooltip.assignUtf8(pLocals->_pCodecGui,
                                        pJob->_ArcPackage._pDecl->_ustrDescription);
                pttt->pszText = (PSZ)G_strTooltip.c_str();
            }
        }
}

/* ******************************************************************
 *
 *  Page management
 *
 ********************************************************************/

#define CONTROLDEF_TEXTVIEW(p, id, cx, cy) { WC_XTEXTVIEW, NULL, \
            WS_VISIBLE | WS_TABSTOP | XS_VSCROLL | XS_AUTOVHIDE | XS_HSCROLL | XS_AUTOHHIDE, \
            id, CTL_COMMON_FONT, 0, { cx, cy }, COMMON_SPACING, p }

#define CONTROLDEF_BOLDTEXT(pcsz, id, cx, cy) { WC_STATIC, pcsz, \
            WS_VISIBLE | SS_TEXT | DT_LEFT | DT_VCENTER | DT_MNEMONIC, \
            id, "9.WarpSans Bold",  0, {cx, cy}, COMMON_SPACING }

#define DEFAULT_WIDTH       144

#define DEFAULT_HEIGHT      100
#define TOP_HEIGHT           30
#define TOP_CONFIG_HEIGHT    60
#define EF_HEIGHT             8

#define HEADING_ID 0x7fff

XTEXTVIEWCDATA txvcd =
    {
        sizeof(XTEXTVIEWCDATA),
        2,
        2,
        15,
        15
    };

CONTROLDEF
    BigTextView = CONTROLDEF_TEXTVIEW(&txvcd, ID_WIDI_INFOTEXT,
                        DEFAULT_WIDTH,
                        DEFAULT_HEIGHT),
    TopTextView = CONTROLDEF_TEXTVIEW(&txvcd, ID_WIDI_INFOTEXT,
                        DEFAULT_WIDTH,
                        TOP_HEIGHT),
    MiddleTextView = CONTROLDEF_TEXTVIEW(&txvcd, ID_WIDI_READMEMLE,
                        DEFAULT_WIDTH,
                        DEFAULT_HEIGHT - TOP_HEIGHT - COMMON_SPACING),
    PackagesText = CONTROLDEF_TEXT("Packages", HEADING_ID,
                        DEFAULT_WIDTH,
                        SZL_AUTOSIZE),
    ConfigText = CONTROLDEF_TEXT("System Configuration", HEADING_ID,
                        DEFAULT_WIDTH,
                        SZL_AUTOSIZE),
    TopConfigTextView = CONTROLDEF_TEXTVIEW(&txvcd, ID_WIDI_INFOTEXT,
                        DEFAULT_WIDTH,
                        TOP_CONFIG_HEIGHT),
    PckContainer =
        {
            WC_CONTAINER,
            NULL,
            WS_VISIBLE | WS_TABSTOP | CCS_EXTENDSEL,
            ID_WIDI_PACKAGESCNR,
            CTL_COMMON_FONT,
            0,
            { DEFAULT_WIDTH,
                    DEFAULT_HEIGHT - TOP_HEIGHT - COMMON_SPACING - 2 * COMMON_SPACING - 2 * EF_HEIGHT},
            COMMON_SPACING,
            NULL
        },
    CfgContainer =
        {
            WC_CONTAINER,
            NULL,
            WS_VISIBLE | WS_TABSTOP | CCS_EXTENDSEL,
            ID_WIDI_CONFIGCNR,
            CTL_COMMON_FONT,
            0,
            { DEFAULT_WIDTH,
                DEFAULT_HEIGHT - TOP_CONFIG_HEIGHT - COMMON_SPACING},
            COMMON_SPACING,
            NULL
        },
    PathText = CONTROLDEF_TEXT(
                        "Install ~path:",
                        ID_WIDI_PATHTEXT,
                        STD_BUTTON_WIDTH,
                        -1),
    PathEntry = CONTROLDEF_ENTRYFIELD(
                        "M",
                        ID_WIDI_PATHENTRY,
                        DEFAULT_WIDTH - STD_BUTTON_WIDTH - COMMON_SPACING,
                        EF_HEIGHT),
    InstalledText = CONTROLDEF_TEXT(
                        "~Installed version:",
                        ID_WIDI_PCKINSTALLED_TXT,
                        STD_BUTTON_WIDTH,
                        EF_HEIGHT),
    InstalledValue = CONTROLDEF_TEXT(
                        "M",
                        ID_WIDI_PCKINSTALLED,
                        DEFAULT_WIDTH - STD_BUTTON_WIDTH - COMMON_SPACING,
                        EF_HEIGHT),
    BackButton = CONTROLDEF_PUSHBUTTON(" ~Back",
                        ID_WIDI_BACKBUTTON,
                        STD_BUTTON_WIDTH,
                        STD_BUTTON_HEIGHT),
    NextButton = CONTROLDEF_DEFPUSHBUTTON(
                        "~Next ",
                        ID_WIDI_NEXTBUTTON,
                        STD_BUTTON_WIDTH,
                        STD_BUTTON_HEIGHT),
    CancelButton = CONTROLDEF_PUSHBUTTON(
                        "~Cancel",
                        ID_WIDI_CANCELBUTTON,
                        STD_BUTTON_WIDTH,
                        STD_BUTTON_HEIGHT);

DLGHITEM dlgText[] =
        {
            START_TABLE,
                START_ROW(0),
                    CONTROL_DEF(&BigTextView),
                START_ROW(0),
                    CONTROL_DEF(&BackButton),
                    CONTROL_DEF(&NextButton),
                    CONTROL_DEF(&CancelButton),
            END_TABLE
        };
MPARAM G_ampText[] =
        {
            MPFROM2SHORT(ID_WIDI_INFOTEXT, XAC_SIZEX | XAC_SIZEY)
        };

DLGHITEM dlgReadme[] =
        {
            START_TABLE,
                START_ROW(0),
                    CONTROL_DEF(&TopTextView),
                START_ROW(0),
                    CONTROL_DEF(&MiddleTextView),
                START_ROW(0),
                    CONTROL_DEF(&BackButton),
                    CONTROL_DEF(&NextButton),
                    CONTROL_DEF(&CancelButton),
            END_TABLE
        };
MPARAM G_ampReadme[] =
        {
            MPFROM2SHORT(ID_WIDI_INFOTEXT, XAC_SIZEX | XAC_MOVEY),
            MPFROM2SHORT(ID_WIDI_READMEMLE, XAC_SIZEX | XAC_SIZEY)
        };

DLGHITEM dlgContainer[] =
        {
            START_TABLE,
                START_ROW(0),
                    CONTROL_DEF(&PackagesText),
                START_ROW(0),
                    CONTROL_DEF(&TopTextView),
                START_ROW(0),
                    CONTROL_DEF(&PckContainer),
                START_ROW(ROW_VALIGN_CENTER),
                    CONTROL_DEF(&PathText),
                    CONTROL_DEF(&PathEntry),
                START_ROW(ROW_VALIGN_CENTER),
                    CONTROL_DEF(&InstalledText),
                    CONTROL_DEF(&InstalledValue),
                START_ROW(ROW_VALIGN_CENTER),
                    CONTROL_DEF(&BackButton),
                    CONTROL_DEF(&NextButton),
                    CONTROL_DEF(&CancelButton),
            END_TABLE
        };
MPARAM G_ampContainer[] =
        {
            MPFROM2SHORT(HEADING_ID, XAC_SIZEX | XAC_MOVEY),
            MPFROM2SHORT(ID_WIDI_INFOTEXT, XAC_SIZEX | XAC_MOVEY),
            MPFROM2SHORT(ID_WIDI_PACKAGESCNR, XAC_SIZEX | XAC_SIZEY),
            MPFROM2SHORT(ID_WIDI_PATHENTRY, XAC_SIZEX)
        };

DLGHITEM dlgConfigure[] =
        {
            START_TABLE,
                START_ROW(0),
                    CONTROL_DEF(&ConfigText),
                START_ROW(0),
                    CONTROL_DEF(&TopConfigTextView),
                START_ROW(0),
                    CONTROL_DEF(&CfgContainer),
                START_ROW(ROW_VALIGN_CENTER),
                    CONTROL_DEF(&BackButton),
                    CONTROL_DEF(&NextButton),
                    CONTROL_DEF(&CancelButton),
            END_TABLE
        };
MPARAM G_ampConfigure[] =
        {
            MPFROM2SHORT(HEADING_ID, XAC_SIZEX | XAC_MOVEY),
            MPFROM2SHORT(ID_WIDI_INFOTEXT, XAC_SIZEX | XAC_MOVEY),
            MPFROM2SHORT(ID_WIDI_CONFIGCNR, XAC_SIZEX | XAC_SIZEY)
        };

XADJUSTCTRLS    G_xac;
BOOL            G_fXacInited = FALSE;

VOID AdjustControls(PSWP pswpNew)
{
    MPARAM  *pamp;
    ULONG   c = 0;

    switch (G_pCurrentPageInfo->_PageMode)
    {
        case FEPageInfo::MODE_TEXT:
            pamp = G_ampText;
            c = ARRAYITEMCOUNT(G_ampText);
        break;

        case FEPageInfo::MODE_README:
            pamp = G_ampReadme;
            c = ARRAYITEMCOUNT(G_ampReadme);
        break;

        case FEPageInfo::MODE_CONTAINER:
            pamp = G_ampContainer;
            c = ARRAYITEMCOUNT(G_ampContainer);
        break;

        case FEPageInfo::MODE_CONFIGURE:
            pamp = G_ampConfigure;
            c = ARRAYITEMCOUNT(G_ampConfigure);
        break;
    }

    if (c)
        winhAdjustControls(G_hwndPagesClient,
                           pamp,
                           c,
                           pswpNew,
                           &G_xac);
}

/*
 *@@ TurnToPageReadme:
 *      implementation for FEPageInfo::MODE_README in
 *      TurnToPage.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

VOID TurnToPageReadme(GUIInstallEngine &Engine,
                      GUILocals *pLocals)
{
    HWND    hwndMLE = WinWindowFromID(G_hwndPagesClient, ID_WIDI_READMEMLE);

    BSString str2Insert;

    if (!G_pCurrentPageInfo->_ulExtractFromPck)
        str2Insert.assignUtf8(pLocals->_pCodecGui,
                              G_pCurrentPageInfo->_ustrReadmeSrc);
    else
    {
        // use _strReadmeSrc as a file name:
        BSString strTempFileName;
        APIRET arc;

        if (!(arc = Engine.ExtractTempFile(G_pCurrentPageInfo->_ulExtractFromPck,
                                           G_pCurrentPageInfo->_ustrReadmeSrc.GetBuffer(),
                                           &strTempFileName)))
        {
            // successfully extracted:
            PSZ pszContent = NULL;
            if (!(arc = doshLoadTextFile(strTempFileName.c_str(),
                                         &pszContent,
                                         NULL)))
            {
                // check what codepage the script was created in...
                // we assume that the "readme" file was written in
                // the same codepage. If the codepage is different
                // from our current one, we'll need to convert:
                ULONG cpSrc = Engine._pCurrentArchive->_pScript->_ulCodepage;
                if (cpSrc == pLocals->_pCodecGui->QueryCodepage())
                    // easy
                    str2Insert = pszContent;
                else
                {
                    // alright, different:
                    // convert file contents to Unicode
                    BSUniCodec codecSrc(cpSrc);
                    ustring ustr(&codecSrc, pszContent);
                    // convert Unicode to display codepage
                    str2Insert.assignUtf8(pLocals->_pCodecGui, ustr);
                }
                free(pszContent);
            }
            else
                str2Insert._printf("Error %d reading file \"%s\" from package %d.\n",
                                   arc,
                                   strTempFileName.c_str(),
                                   G_pCurrentPageInfo->_ulExtractFromPck);
        }
        else
            str2Insert._printf("Error %d extracting file \"%s\" from package %d.\n",
                               arc,
                               strTempFileName.c_str(),
                               G_pCurrentPageInfo->_ulExtractFromPck);
    }

    // get format
    XFMTPARAGRAPH xfmtp;
    WinSendMsg(hwndMLE,
               TXM_QUERYPARFORMAT,
               (MPARAM)0,
               (MPARAM)(&xfmtp));

    switch (G_pCurrentPageInfo->_ReadmeFormat)
    {
        /*
         * README_PLAIN:
         *
         */

        case FEPageInfo::README_PLAIN:
        {
            xfmtp.fWordWrap = FALSE;
            xfmtp.lLeftMargin = 0;
            xfmtp.lRightMargin = 0;
            xfmtp.lFirstLineMargin = 0;
            xfmtp.lSpaceBefore = 0;
            xfmtp.lSpaceAfter = 0;

            PSZ     pszNew = strdup(str2Insert.c_str());
            txvStripLinefeeds(&pszNew,
                              4);       // tab size
            str2Insert = pszNew;
            free(pszNew);
        }
        break;

        /*
         * README_FLOW:
         *
         */

        case FEPageInfo::README_FLOW:
        {
            xfmtp.fWordWrap = TRUE;
            xfmtp.lLeftMargin = 0;
            xfmtp.lRightMargin = 0;
            xfmtp.lFirstLineMargin = 0;
            xfmtp.lSpaceBefore = 5;
            xfmtp.lSpaceAfter = 5;

            PSZ     pszNew = strdup(str2Insert.c_str()),
                    pTarget = pszNew;
            const char *pSource = str2Insert.c_str();

            // step 1:
            // remove all line feeds
            while (*pSource)
            {
                if (*pSource == '\r')
                    pSource++;
                else
                    *pTarget++ = *pSource++;
            }
            *pTarget = 0;

            str2Insert = pszNew;
            free(pszNew);

            // step 2:
            // remove all simple newlines; keep double newlines
            pszNew = strdup(str2Insert.c_str()),
            pSource = str2Insert.c_str(),
            pTarget = pszNew;
            while (*pSource)
            {
                if (*pSource == '\n')
                {
                    // newline:
                    if (*(pSource+1) == '\n')
                    {
                        // empty line:
                        // keep one \n
                        *pTarget++ = *pSource++;
                        // *pTarget++ = *pSource++;
                        // skip all the following
                        while ((*pSource) && (*pSource == '\n'))
                            pSource++;
                        // skip leading spaces
                        while ((*pSource) && (*pSource == ' '))
                            pSource++;
                    }
                    else
                    {
                        BOOL fAddSpace = FALSE;
                        // single newline:
                        // skip
                        pSource++;
                        fAddSpace = TRUE;
                        // and skip leading spaces after newline
                        while ((*pSource) && (*pSource == ' '))
                            pSource++;

                        // check if we have "---" or "===" next
                        if (*(pSource))
                            if (*(pSource+1))
                                if (*(pSource+2))
                                {
                                    if (   (     (*(pSource) == '=')
                                              && (*(pSource+1) == '=')
                                              && (*(pSource+2) == '=')
                                           )
                                        || (     (*(pSource) == '-')
                                              && (*(pSource+1) == '-')
                                              && (*(pSource+2) == '-')
                                           )
                                        )
                                    {
                                        // add linefeed
                                        *pTarget++ = '\r';
                                        fAddSpace = FALSE;
                                    }
                                }
                        if (fAddSpace)
                            *pTarget++ = ' ';
                    }
                }
                else if (*pSource == ' ')
                {
                    // space: keep one
                    *pTarget++ = *pSource++;
                    // skip all the following
                    while ((*pSource) && (*pSource == ' '))
                        pSource++;
                }
                else
                    // any other character: copy
                    *pTarget++ = *pSource++;
            }
            *pTarget = 0;

            str2Insert = pszNew;
            free(pszNew);
        }
        break;

        /*
         * README_HTML:
         *
         */

        case FEPageInfo::README_HTML:
        {
            xfmtp.fWordWrap = TRUE;
            xfmtp.lLeftMargin = 0;
            xfmtp.lRightMargin = 0;
            xfmtp.lFirstLineMargin = 0;
            xfmtp.lSpaceBefore = 5;
            xfmtp.lSpaceAfter = 5;

            PSZ     pszNew = strdup(str2Insert.c_str());
            txvConvertFromHTML(&pszNew,
                               NULL,
                               NULL,
                               NULL);
            str2Insert = pszNew;
            free(pszNew);
        }
        break;
    } // end switch

    // set format
    WinSendMsg(hwndMLE,
               TXM_SETPARFORMAT,
               (MPARAM)0,
               (MPARAM)(&xfmtp));
    WinSetWindowText(hwndMLE, str2Insert.c_str());

    G_ulPageHelp = IDHI_PAGE_TEXTONLY;
}

PLINKLIST   G_pllControls = NULL;

/*
 *@@ TurnToPage:
 *      this gets called by fnwpPages when the "Next" or
 *      "Back" buttons are pressed (in response to the
 *      WPIM_TURNTOPAGE message in fnwpPages).
 *
 *      We then update the dialogs according to the page
 *      data which was prepared by the script classes.
 *
 *      This func checks G_pCurrentPageInfo for whether
 *      it already contains a current page. If so, it is
 *      cleaned out.
 *
 *      We then turn to the first or next page given in
 *      lSearchPage and format it according to the page
 *      data.
 *
 *      Throws:
 *      -- FEFatalErrorExcpt.
 *
 *@@changed 0.9.0: path entry field on cnr page now accepts 254 chars
 *@@changed V0.9.1 (2000-01-12) [umoeller]: README MLE was read/write; fixed
 *@@changed V0.9.1 (2000-02-03) [umoeller]: README MLE wasn't always scrolled to top
 *@@changed V0.9.3 (2000-04-28) [umoeller]: configure checkboxes were disabled, but checked; fixed
 *@@changed V0.9.3 (2000-05-05) [umoeller]: added FORMAT=PLAIN support
 *@@changed V0.9.3 (2000-05-06) [umoeller]: much better formatting now
 *@@changed V0.9.18 (2002-03-03) [umoeller]: largely rewritten for new controls support (sizeable, dlg formatter)
 *@@changed V0.9.18 (2002-03-03) [umoeller]: made configure page use checkbox containers now
 *@@changed V0.9.18 (2002-03-03) [umoeller]: now setting "wait" pointer since this may take a second
 */

VOID TurnToPage(GUIInstallEngine &Engine,
                LONG lSearchPage,   // in: page index to turn to
                BOOL fStoreThis)    // in: store current index in G_WpiGlobals.lPagesList?
{
    // all the formatting can take a second, so set wait pointer
    // V0.9.18 (2002-03-03) [umoeller]
    GUIWaitPointer wp;

    GUILocals *pLocals = (GUILocals *)&Engine._Locals;

    /*
     * 1) handle "leaving page" stuff:
     *
     */

    // did we have a previous page?
    if (G_pCurrentPageInfo)
    {
        // yes: G_pCurrentPageInfo still has the old page
        switch (G_pCurrentPageInfo->_PageMode)
        {
            /*
             * MODE_CONTAINER:
             *      leaving container page
             */

            case FEPageInfo::MODE_CONTAINER:
            {
                // PmPrintf("  Leaving container page");

                // if lSearchPage comes _after_ the current
                // page, we must check the validity of package
                // selections (we don't want to do this if
                // the user presses the "Back" button)
                if (lSearchPage > G_pCurrentPageInfo->_lPageIndex)
                {
                    // prompt for missing dirs
                    // and check dependencies
                    if (!Engine.VerifySelections(TRUE))
                        // error found: do not leave container page
                        return;

                    // check if disk space on target drives is sufficient
                    // by evaluating the global flag which is maintained
                    // by the "Drive info" window (fnwpDriveInfo)
                    // moved this here from engine V0.9.20 (2002-07-03) [umoeller]
                    if (!G_WpiGlobals.fPlentyOfSpace)
                        if (Engine._Locals.MessageBox(108,   // "WarpIN: Warning"
                                                      110,   // "not enough space"
                                                      MSG_CONFIRM_YESNO_DEFYES)
                                != MBID_YES)
                            // do not leave container page
                            return;

                }

                // finally, leaving container page:
                // hide "drive info" window
                WinPostMsg(G_hwndDriveInfo, WPIM_UPDATE, (MPARAM)0, NULL);

                // and destroy the tooltip
                // V0.9.14 (2001-08-03) [umoeller]
                if (G_hwndCnrTooltip)
                {
                    WinDestroyWindow(G_hwndCnrTooltip);
                    G_hwndCnrTooltip = NULLHANDLE;
                }

            }
            break;  // case MODE_CONTAINER

        } // end switch (pCurrentPageInfo->lPageMode)

        // store the index of the page that we're leaving
        // in the list in G_WpiGlobals so we can use the
        // "Back" button later
        if (fStoreThis)
        {
            GUIVisited *pl = new GUIVisited(G_pCurrentPageInfo->_lPageIndex);
            G_WpiGlobals.lPagesList.push_back(pl);
        }

        if (G_fXacInited)
            // we've been here before: then clean out old XADJUSTCONTROLS
            winhAdjustControls(G_hwndPagesClient,
                               NULL,
                               0,
                               NULL,
                               &G_xac);

        // since we no longer use separate sub-frames for the various
        // pages but create and destroy the controls on the same
        // client dynamically, kill all the old controls from the
        // page we're leaving
        // V0.9.18 (2002-03-03) [umoeller]

        if (G_pllControls)
        {
            // prevent flicker
            WinEnableWindowUpdate(G_hwndPagesClient, FALSE);

            PLISTNODE pNode = lstQueryFirstNode(G_pllControls);
            while (pNode)
            {
                WinDestroyWindow((HWND)pNode->pItemData);
                pNode = pNode->pNext;
            }

            lstFree(&G_pllControls);
        }

    } // end if (pCurrentPageInfo)

    /*
     * 2) find the page we're looking for
     *
     */

    if (!(G_pCurrentPageInfo = Engine._pCurrentArchive->FindPageInfo(lSearchPage)))
    {
        ustring ustr;
        ustr._itoa10(lSearchPage, 0);
        throw FEFatalErrorExcpt(Engine._Locals,
                                272, //  Unable to find page %1 in the installation profile.
                                     // The "<PAGE ...>" tags always need to have an INDEX attribute, and there needs to be at least one page.
                                &ustr, 1);
    }

    /*
     * 3) create new controls
     *
     */

    // now using dialog formatter
    // V0.9.18 (2002-03-03) [umoeller]
    PDLGHITEM   pDlg;
    ULONG       cItems = 0;
    ULONG       idWhiteBackground = 0,
                idFocus = 0,
                idHeading = 0;

    switch (G_pCurrentPageInfo->_PageMode)
    {
        case FEPageInfo::MODE_TEXT:
            pDlg = dlgText;
            cItems = ARRAYITEMCOUNT(dlgText);
        break;

        case FEPageInfo::MODE_README:
            pDlg = dlgReadme;
            cItems = ARRAYITEMCOUNT(dlgReadme);
            idWhiteBackground = ID_WIDI_READMEMLE;
            idFocus = ID_WIDI_READMEMLE;
        break;

        case FEPageInfo::MODE_CONTAINER:
            pDlg = dlgContainer;
            cItems = ARRAYITEMCOUNT(dlgContainer);
            idFocus = ID_WIDI_PACKAGESCNR;
            idHeading = HEADING_ID;
        break;

        case FEPageInfo::MODE_CONFIGURE:
            pDlg = dlgConfigure;
            cItems = ARRAYITEMCOUNT(dlgConfigure);
            idFocus = ID_WIDI_CONFIGCNR;
            idHeading = HEADING_ID;
        break;
    }

    if (!cItems)
        throw (BSExcptBase("Unknown page type, cannot create controls."));

    RECTL rclClient;
    SIZEL szlClient;

    // prevent flicker
    WinEnableWindowUpdate(G_hwndPagesClient, FALSE);

    // go create the controls from the dlg array selected above
    dlghFormatDlg(G_hwndPagesClient,
                  pDlg,
                  cItems,
                  gshrQueryDefaultFont(),
                  DFFL_CREATECONTROLS,
                  &szlClient,
                  // store the new controls in this list
                  (PVOID*)&G_pllControls);

    // scale the client to fill up the frame;
    // this is a two-step process to get the resizing
    // correct (fnwpPages intercepts WM_WINDOWPOSCHANGED
    // and calls winhAdjustControls accordingly):

    // a) temporarily set the client to the required size
    //    (rectangle surrounding the created controls)
    WinQueryWindowRect(G_hwndMainFrame, &rclClient);
    WinCalcFrameRect(G_hwndMainFrame,
                     &rclClient,
                     TRUE);            // client from frame

    // prevent winhAdjustControls while we set the client's temp size
    G_fXacInited = FALSE;
    WinSetWindowPos(G_hwndPagesClient,
                    HWND_BOTTOM,
                    rclClient.xLeft,
                    rclClient.yBottom,
                    rclClient.xLeft + szlClient.cx,
                    rclClient.yBottom + szlClient.cy,
                    SWP_ZORDER | SWP_MOVE | SWP_SIZE);

    if (idWhiteBackground)
        // set above if a control needs a white background
        winhSetPresColor(WinWindowFromID(G_hwndPagesClient, idWhiteBackground),
                         PP_BACKGROUNDCOLOR,
                         RGBCOL_WHITE);

    if (idFocus)
        // set above if we switch focus
        WinSetFocus(HWND_DESKTOP, WinWindowFromID(G_hwndPagesClient, idFocus));

    if (idHeading)
        gshrMakeHeading(G_hwndPagesClient, idHeading);

    // now init XADJUSTCONTROLS so that winhAdjustControls
    // will pick up the original size of the dlg
    memset(&G_xac, 0, sizeof(G_xac));
    // and make sure we clean up next time
    G_fXacInited = TRUE;

    // initialize winhAdjustControls, or it won't work
    AdjustControls(NULL); // null SWP

    // b) now resize client to correct size;
    //    fnwpPages will then scale the controls on
    //    WM_WINDOWPOSCHANGED
    WinQueryWindowRect(G_hwndMainFrame, &rclClient);
    WinCalcFrameRect(G_hwndMainFrame,
                     &rclClient,
                     TRUE);            // client from frame
    // re-enable updates
    WinEnableWindowUpdate(G_hwndPagesClient, TRUE);
    // go!
    WinSetWindowPos(G_hwndPagesClient,
                    HWND_BOTTOM,
                    rclClient.xLeft,
                    rclClient.yBottom,
                    rclClient.xRight - rclClient.xLeft,
                    rclClient.yTop - rclClient.yBottom,
                    SWP_ZORDER | SWP_MOVE | SWP_SIZE);

    // set page text
    // each of the four subdialogs has a static text with ID_WIDI_INFOTEXT ID
    HWND hwndTextView = WinWindowFromID(G_hwndPagesClient, ID_WIDI_INFOTEXT);
    WinSendMsg(hwndTextView, TXM_SETWORDWRAP, (MPARAM)TRUE, 0);

    string strNew(pLocals->_pCodecGui, G_pCurrentPageInfo->_ustrInfoText);
    PSZ pszNew = strdup(strNew.c_str());
    strNew.erase();
    txvStripLinefeeds(&pszNew,
                      4);       // tab size
    WinSetWindowText(hwndTextView, pszNew);
    free(pszNew);

    /*
     * 3) initialize control values
     *
     */

    // disable "Selections" in main menu when not on container page
    WinSendMsg(G_hmenuMain,
               MM_SETITEMATTR,
               MPFROM2SHORT(ID_WIM_SELECT, TRUE),
               MPFROM2SHORT(MIA_DISABLED,
                            (G_pCurrentPageInfo->_PageMode == FEPageInfo::MODE_CONTAINER)
                                    ? 0
                                    : MIA_DISABLED));

    switch (G_pCurrentPageInfo->_PageMode)
    {

        /*
         * MODE_TEXT:
         *      text-only page.
         */

        case FEPageInfo::MODE_TEXT:
            G_ulPageHelp = IDHI_PAGE_TEXTONLY;
        break;

        /*
         * MODE_README:
         *      set the MLE with text
         */

        case FEPageInfo::MODE_README:
            TurnToPageReadme(Engine, pLocals);
        break;

        /*
         * MODE_CONTAINER:
         *      entering container page:
         *      create record cores etc.
         */

        case FEPageInfo::MODE_CONTAINER:
        {
            // set up the container
            HWND hwndPackagesCnr = WinWindowFromID(G_hwndPagesClient, ID_WIDI_PACKAGESCNR);
            BEGIN_CNRINFO()
            {
                cnrhSetView(CV_TREE | CV_ICON | CA_TREELINE | CA_OWNERDRAW);
                // CnrInfo.cxTreeIndent = 15;
                // set icon sizes to system mini-icon size
                cnrhSetBmpOrIconSize((pLocals->_ulMiniIconSize * 2 + 5),
                                     pLocals->_ulMiniIconSize);
            } END_CNRINFO(hwndPackagesCnr);

            // set entry field path limit V0.9.12 (2001-05-17) [umoeller]
            winhSetEntryFieldLimit(WinWindowFromID(G_hwndPagesClient, ID_WIDI_PATHENTRY),
                                   254);

            Packages2Cnr(Engine,
                         hwndPackagesCnr);

            // create tooltip for packages
            // V0.9.14 (2001-08-03) [umoeller]: now always creating this
            CreateCnrTooltip(G_hwndPagesClient,
                             hwndPackagesCnr);

            // show and update the drive info window
            WinPostMsg(G_hwndDriveInfo, WPIM_UPDATE, (MPARAM)1, NULL);

            G_ulPageHelp = IDHI_PAGE_CONTAINER;
        }
        break;  // case MODE_CONTAINER:

        /*
         * MODE_CONFIGURE:
         *      entering configure page
         */

        case FEPageInfo::MODE_CONFIGURE:
        {
            // enable/disable the checkboxes;
            // ULONG   ulConfigureFlags = 0;
            /* Engine.CheckJobs(NULL,
                             NULL,
                             &ulConfigureFlags); */

            HWND G_hwndPageConfigure = G_hwndPagesClient;

            // ulConfigureFlags now has the configure flags according
            // to the current package selections.
            // By contrast, G_WpiGlobals.ulAllowConfig has the configure
            // flags of the sum of all packages, that is, as if all
            // packages had been selected.

            // We _enable_ the dlg items according to ulConfigureFlags,
            // we _set_ the dlg items according to ulDoConfiguration.

            // V0.9.18 (2002-03-03) [umoeller]
            // rewrote this completely, since we now use a
            // checkbox container

            HWND hwndCnr = WinWindowFromID(G_hwndPagesClient, ID_WIDI_CONFIGCNR);

            ctlMakeCheckboxContainer(G_hwndPagesClient,
                                     ID_WIDI_CONFIGCNR);


            list<FEInstallVar*>   listVariables(SHADOW);
            ULONG cRecords = Engine.GetInstallVars(listVariables,
                                                   TRUE);       // publics only

            list<FEInstallVar*>::iterator varBegin = listVariables.begin(),
                                          varEnd = listVariables.end();
            for (;
                 varBegin != varEnd;
                 ++varBegin)
            {
                FEInstallVar *pVar = *varBegin;

                PCHECKBOXRECORDCORE preccThis = (PCHECKBOXRECORDCORE)
                                            cnrhAllocRecords(hwndCnr,
                                                             sizeof(CHECKBOXRECORDCORE),
                                                             1);

                preccThis->ulStyle = WS_VISIBLE | BS_AUTOCHECKBOX;
                preccThis->ulItemID = (ULONG)pVar;
                preccThis->usCheckState = pVar->GetValue();
                string str(pLocals->_pCodecGui, pVar->GetDescription());
                preccThis->recc.pszTree = strdup(str.c_str());

                cnrhInsertRecords(hwndCnr,
                                  (PRECORDCORE)NULL,        // parent
                                  (PRECORDCORE)preccThis,
                                  TRUE, // invalidate
                                  NULL,
                                  CRA_RECORDREADONLY,
                                  1);
            }

            /*
            winhEnableDlgItem(G_hwndPageConfigure,
                              ID_WIDI_UPDATECONFIGSYS,
                              ((ulConfigureFlags & PCK_CONFIGSYS) != 0)
                             );
            winhSetDlgItemChecked(G_hwndPageConfigure,
                                  ID_WIDI_UPDATECONFIGSYS,
                                  (     ((Engine._ulAllowConfig & PCK_CONFIGSYS) != 0)
                                     && ((ulConfigureFlags & PCK_CONFIGSYS) != 0)
                                  ));
            winhEnableDlgItem(G_hwndPageConfigure,
                              ID_WIDI_INSTALLWPSCLASSES,
                              ((ulConfigureFlags & PCK_WPSCLASSES) != 0));
            winhSetDlgItemChecked(G_hwndPageConfigure,
                                  ID_WIDI_INSTALLWPSCLASSES,
                                  (    ((Engine._ulAllowConfig & PCK_WPSCLASSES) != 0)
                                    && ((ulConfigureFlags & PCK_WPSCLASSES) != 0)
                                  ));
            winhEnableDlgItem(G_hwndPageConfigure,
                              ID_WIDI_CREATEWPSOBJECTS,
                              ((ulConfigureFlags & PCK_WPSOBJECTS) != 0));
            winhSetDlgItemChecked(G_hwndPageConfigure,
                                  ID_WIDI_CREATEWPSOBJECTS,
                                  (     ((Engine._ulAllowConfig & PCK_WPSOBJECTS) != 0)
                                     && ((ulConfigureFlags & PCK_WPSOBJECTS) != 0)
                                  ));
            */

            G_ulPageHelp = IDHI_PAGE_CONFIGURE;
        }
        break;

    } // end switch (pCurrentPageInfo->lPageMode)

    HWND    hwndFocusButton = WinWindowFromID(G_hwndPagesClient, ID_WIDI_CANCELBUTTON);

    // set "Back" button (all pages)
    winhShowDlgItem(G_hwndPagesClient, ID_WIDI_BACKBUTTON,
                    !(G_WpiGlobals.lPagesList.empty()));

    // set "Next" button (all pages)
    BOOL fShow =   (G_pCurrentPageInfo->_lNextButton >= 0);
    HWND hwndNextButton = WinWindowFromID(G_hwndPagesClient, ID_WIDI_NEXTBUTTON);
    WinShowWindow(hwndNextButton, fShow);
    if (fShow)
    {
        string sNextButtonTitle;
        if (G_pCurrentPageInfo->_ustrNextButtonTitle())
            sNextButtonTitle.assignUtf8(pLocals->_pCodecGui,
                                        G_pCurrentPageInfo->_ustrNextButtonTitle);
        else
            // use default string:
            sNextButtonTitle = nlsGetString(WPSI_NEXTBUTTON);

        sNextButtonTitle.append(" ");

        WinSetWindowText(hwndNextButton, sNextButtonTitle.c_str());
        if (fShow)
            hwndFocusButton = hwndNextButton;
    }

    if (idFocus)
        // set above if we switch focus
        WinSetFocus(HWND_DESKTOP, WinWindowFromID(G_hwndPagesClient, idFocus));
    else
        WinSetFocus(HWND_DESKTOP, hwndFocusButton);

    // show/hide controls
    /* WinShowWindow(G_hwndPageText,
                  G_pCurrentPageInfo->_PageMode == FEPageInfo::MODE_TEXT);
    WinShowWindow(G_hwndPageReadme,
                  G_pCurrentPageInfo->_PageMode == FEPageInfo::MODE_README);
    WinShowWindow(G_hwndPageContainer,
                  G_pCurrentPageInfo->_PageMode == FEPageInfo::MODE_CONTAINER);
    WinShowWindow(G_hwndPageConfigure,
                  G_pCurrentPageInfo->_PageMode == FEPageInfo::MODE_CONFIGURE); */

    // WinSetActiveWindow(HWND_DESKTOP, G_hwndPagesClient);
}

/* ******************************************************************
 *
 *  Dialog procs for "Install" mode
 *
 ********************************************************************/

/*
 *@@ ARCHIVEVIEW:
 *
 *@@added V0.9.9 (2001-03-27) [umoeller]
 */

struct ARCHIVEVIEW
{
    HWND        hwndFrame,
                hwndClient;             // container

    PRECORDCORE precFilesCurrent;

    // list of strings to be freed
    list<string*>       FreeList;

    ARCHIVEVIEW() : FreeList(STORE)
    {
    };

};

/*
 *@@ ViewArchives_ForAllFilesCallback:
 *
 *@@added V0.9.14 (2001-08-23) [umoeller]
 */

int XWPENTRY ViewArchives_ForAllFilesCallback(WIFileHeader *pwifh,
                                              unsigned long ulUser)
{
    ARCHIVEVIEW *pView = (ARCHIVEVIEW*)ulUser;

    CHAR szTemp[500];
    string *pstr = new string();
    pstr->_printf("\"%s\" (%s bytes)",
                  pwifh->name,
                  nlsThousandsULong(szTemp,
                                    pwifh->origsize,
                                    G_szThousand[0]));
    pView->FreeList.push_back(pstr);

    PRECORDCORE precFileThis = cnrhAllocRecords(pView->hwndClient,
                                                sizeof(RECORDCORE),
                                                1);

    cnrhInsertRecords(pView->hwndClient,
                      pView->precFilesCurrent, // parent
                      precFileThis,
                      TRUE,
                      pstr->c_str(),
                      CRA_RECORDREADONLY | CRA_EXPANDED,
                      1);

    return 0;
}

/*
 *@@ ViewArchives:
 *      implementation for "View archives" menu command.
 *      Displays all packages and archives.
 *
 *@@added V0.9.9 (2001-03-27) [umoeller]
 *@@changed V0.9.13 (2001-07-06) [umoeller]: added more display data
 *@@changed V0.9.14 (2001-08-23) [umoeller]: added yet more display data
 */

VOID ViewArchives(GUIInstallEngine &Engine,
                  HWND hwndOwner)
{
    GUILocals *pLocals = (GUILocals *)&Engine._Locals;

    SWP swpFrame;
    swpFrame.x = 100;
    swpFrame.y = 100;
    swpFrame.cx = 500;
    swpFrame.cy = 500;
    swpFrame.hwndInsertBehind = HWND_TOP;
    swpFrame.fl = SWP_MOVE | SWP_SIZE;

    ARCHIVEVIEW View;
    View.hwndFrame = winhCreateStdWindow(HWND_DESKTOP,           // frame parent
                                         &swpFrame,
                                         FCF_SYSMENU
                                              | FCF_SIZEBORDER
                                              | FCF_TITLEBAR
                                              | FCF_MAXBUTTON
                                              | FCF_NOBYTEALIGN,
                                         WS_ANIMATE,
                                         "WarpIN",         // title bar
                                         0,                // res IDs
                                         WC_CONTAINER,   // client class
                                         WS_VISIBLE,       // client wnd style
                                         1,                // ID
                                         0,
                                         &View.hwndClient);
    WinSetWindowPos(View.hwndFrame,
                    HWND_TOP,
                    0, 0, 0, 0,
                    SWP_SHOW | SWP_ZORDER | SWP_ACTIVATE);

    winhSetWindowFont(View.hwndClient, gshrQueryDefaultFont());

    BEGIN_CNRINFO()
    {
        cnrhSetView(CV_TREE | CV_TEXT | CA_TREELINE);
    } END_CNRINFO(View.hwndClient);

    // go for all archives
    list<FEArchive*>::iterator ArcStart = Engine._pArchivesList->begin(),
                               ArcEnd = Engine._pArchivesList->end();
    for (;
         ArcStart != ArcEnd;
         ArcStart++)
    {
        FEArchive *parc = *ArcStart;

        PRECORDCORE precArchive = cnrhAllocRecords(View.hwndClient,
                                                   sizeof(RECORDCORE),
                                                   1);

        string *pstr = new string;
        string strArc(pLocals->_pCodecGui, parc->_ustrArcFilename);
        pstr->_printf("Archive: \"%s\"\n"
                      "WPI revision: %d",
                      strArc.c_str(),
                      parc->_Arc.getArcHeader()->wi_revision_needed);
        View.FreeList.push_back(pstr);

        cnrhInsertRecords(View.hwndClient,
                          NULL,     // parent
                          precArchive,
                          TRUE,
                          pstr->c_str(),
                          CRA_RECORDREADONLY | CRA_EXPANDED,
                          1);

        // insert packages for archive
        list<FEArcPackageBase*>::iterator PckStart = parc->_pPackagesList->begin(),
                                          PckEnd = parc->_pPackagesList->end();
        for (;
             PckStart != PckEnd;
             PckStart++)
        {
            FEArcPackageBase *ppck = *PckStart;
            // FEArcPackagePck *ppckpck = ppck->IsPackage();

            DYNAMIC_CAST(FEArcPackagePck, ppckpck, ppck);

            if (ppckpck)
            {
                CHAR        szIndex[200];
                sprintf(szIndex,
                        "Package %d:\n\"",
                        ppckpck->_PackHeader.number);
                pstr = new string(szIndex);
                string str(pLocals->_pCodecGui, ppckpck->_pDecl->_ustrID);
                pstr->append(str);
                pstr->append("\"");
                View.FreeList.push_back(pstr);

                PRECORDCORE precPck = cnrhAllocRecords(View.hwndClient,
                                                       sizeof(RECORDCORE),
                                                       1);

                cnrhInsertRecords(View.hwndClient,
                                  precArchive,     // parent
                                  precPck,
                                  TRUE,
                                  pstr->c_str(),
                                  CRA_RECORDREADONLY | CRA_EXPANDED,
                                  1);

                // files list
                pstr = new string(szIndex);
                pstr->_printf("%s KB of uncompressed files",
                              nlsThousandsULong(szIndex,
                                                ppckpck->_ulSizeRounded,
                                                G_szThousand[0]));
                View.FreeList.push_back(pstr);
                PRECORDCORE precFiles = cnrhAllocRecords(View.hwndClient,
                                                         sizeof(RECORDCORE),
                                                         1);
                cnrhInsertRecords(View.hwndClient,
                                  precPck,          // parent
                                  precFiles,
                                  TRUE,
                                  pstr->c_str(),
                                  CRA_RECORDREADONLY,
                                  1);

                View.precFilesCurrent = precFiles;

                ppckpck->_Archive._Arc.forAllFiles(&ppckpck->_PackHeader,
                                                   ViewArchives_ForAllFilesCallback,
                                                   (ULONG)&View);

                // does this package have configuration?
                if (!ppckpck->_pDecl->_listConfigObjects.empty())
                {
                    PRECORDCORE precConfig = cnrhAllocRecords(View.hwndClient,
                                                              sizeof(RECORDCORE),
                                                              1);
                    cnrhInsertRecords(View.hwndClient,
                                      precPck,          // parent
                                      precConfig,
                                      TRUE,
                                      "System configuration",
                                      CRA_RECORDREADONLY,
                                      1);

                    // go thru config list for this package
                    list<BSConfigBase*>::iterator
                        CfgStart = ppckpck->_pDecl->_listConfigObjects.begin(),
                        CfgEnd = ppckpck->_pDecl->_listConfigObjects.end();
                    for (;
                         CfgStart != CfgEnd;
                         CfgStart++)
                    {
                        BSConfigBase *pcfg = *CfgStart;

                        pstr = new string(pcfg->DescribeType());
                        pstr->append(": ");
                        pstr->appendUtf8(pLocals->_pCodecGui,
                                         pcfg->DescribeData());

                        PRECORDCORE precCfgThis = cnrhAllocRecords(View.hwndClient,
                                                               sizeof(RECORDCORE),
                                                               1);

                        View.FreeList.push_back(pstr);

                        cnrhInsertRecords(View.hwndClient,
                                          precConfig,     // parent
                                          precCfgThis,
                                          TRUE,
                                          pstr->c_str(),
                                          CRA_RECORDREADONLY,
                                          1);
                    }
                }
            }
        }
    }

    HAB hab = WinQueryAnchorBlock(View.hwndFrame);
    QMSG qmsg;

    while (WinGetMsg(hab, &qmsg, 0, 0, 0))
    {
        if (    (qmsg.msg == WM_CLOSE)
             && (    (qmsg.hwnd == View.hwndClient)
                  || (qmsg.hwnd == View.hwndFrame)
                )
           )
            // do not dispatch this, this produces WM_QUIT
            break;

        WinDispatchMsg(hab, &qmsg);
    }

    WinDestroyWindow(View.hwndFrame);

    View.FreeList.clear();
}

/* ******************************************************************
 *
 *  Dialog procs for "Install" mode
 *
 ********************************************************************/

/*
 *@@ PagesControl:
 *      implementation for WM_CONTROL in fnwpPages.
 *
 *@@added V0.9.14 (2001-08-23) [umoeller]
 *@@changed V0.9.18 (2002-03-03) [umoeller]: adjustments for new config page
 */

MRESULT PagesControl(GUIInstallEngine *pEngine,
                     HWND hwndDlg,
                     MPARAM mp1,
                     MPARAM mp2)
{
    MRESULT mrc = 0;

    USHORT  usControl = SHORT1FROMMP(mp1);
    USHORT  usNotifyCode = SHORT2FROMMP(mp1);

    GUILocals *pLocals = (GUILocals *)&pEngine->_Locals;

    switch (usControl)
    {
        /*
         * ID_WIDI_PACKAGESCNR:
         *      selection container
         *      on "Container" page
         */

        case ID_WIDI_PACKAGESCNR:
        {
            switch (usNotifyCode)
            {
                /*
                 * CN_ENTER:
                 *      double-click or enter on a
                 *      container record core; update
                 *      package selections then.
                 */

                case CN_ENTER:
                {
                    PNOTIFYRECORDENTER  pnre;
                    PPACKAGERECORD      precc;
                    FEJobBase           *pJob;

                    if (    (pnre = (PNOTIFYRECORDENTER)mp2)
                            // double-click on container, not whitespace:
                         && (precc = (PPACKAGERECORD)pnre->pRecord)
                         && (pJob = precc->pJob)
                       )
                    {
                        pJob->ToggleSelection();
                            // this calls guiOnJobSelectionChanged in turn

                        // update drive info
                        WinPostMsg(G_hwndDriveInfo, WPIM_UPDATE, (MPARAM)1, NULL);

                        UpdatePackageDisplay(*pEngine);
                    }
                }
                break;

                /*
                 * CN_EMPHASIS:
                 *      package selection changed; update
                 *      dependent info in dialog
                 */

                case CN_EMPHASIS:
                {
                    PNOTIFYRECORDEMPHASIS pnre = (PNOTIFYRECORDEMPHASIS)mp2;
                    G_preccSelected = (PPACKAGERECORD)pnre->pRecord;
                    UpdatePackageDisplay(*pEngine);
                }
                break;

                case CN_HELP:
                    gshrDisplayHelp(pLocals, G_ulPageHelp);
                break;

            }
        }
        break;

        /*
         * ID_WIDI_PATHENTRY:
         *      "Path" entry field
         *      on "Container" page
         */

        case ID_WIDI_PATHENTRY:
        {
            switch (usNotifyCode)
            {
                /*
                 * EN_CHANGE:
                 *      PATH entry field has some new content;
                 *      we check for whether a new drive was enterd,
                 *      and if so, we need to update the recc data and,
                 *      if this is the "base" recc, examine all
                 *      other record cores too
                 */

                case EN_CHANGE:
                {
                    FEInstallJob *pInstallJob;
                    if (pInstallJob = G_preccSelected->pJob->IsInstallJob())
                    {
                        CHAR    szNewTargetPath[CCHMAXPATH];
                        WinQueryDlgItemText(hwndDlg, ID_WIDI_PATHENTRY,
                                            sizeof(szNewTargetPath),
                                            szNewTargetPath);
                        ustring ustrTargetPath(pLocals->_pCodecGui, szNewTargetPath);
                        if (pInstallJob->SetTargetPath(ustrTargetPath,
                                                       pEngine->_pAllJobsList))
                            // drive letter changed:
                            WinPostMsg(G_hwndDriveInfo, WPIM_UPDATE, (MPARAM)1, NULL);
                    }
                }
                break;
            }
        } break;

        /*
         * ID_WIDI_CONFIGCNR:
         *      checkboxes on "Configure" page.
         */

        case ID_WIDI_CONFIGCNR:
            if (usNotifyCode == CN_RECORDCHECKED)
            {
                PCHECKBOXRECORDCORE precc = (PCHECKBOXRECORDCORE)mp2;
                ULONG ulFlag = 0;
                FEInstallVar *pVar = (FEInstallVar *)precc->ulItemID;
                if (pVar->GetValue())
                    pVar->SetValue(FALSE);
                else
                    pVar->SetValue(TRUE);
            }

            pEngine->UpdateGlobalStatus(GSTAT_COLLECTING);
        break;

        // tooltip control
        case 19999:
            if (usNotifyCode == TTN_NEEDTEXT)
            {
                HWND hwndPackagesCnr;
                if (hwndPackagesCnr = WinWindowFromID(G_hwndPagesClient, ID_WIDI_PACKAGESCNR))
                    GetTooltipText(pLocals,
                                   hwndDlg,
                                   hwndPackagesCnr,
                                   mp2);
            }
        break;

    } // end switch(usControl)

    return (mrc);
}

/*
 *@@ PagesCommand:
 *      implementation for WM_COMMAND in fnwpPages.
 *
 *@@added V0.9.14 (2001-08-23) [umoeller]
 *@@changed V0.9.14 (2001-08-25) [umoeller]: fixed missing drive infos update on "selections" menu
 */

MRESULT PagesCommand(GUIInstallEngine *pEngine,
                     HWND hwndDlg,
                     MPARAM mp1,
                     MPARAM mp2)
{
    MRESULT mrc = 0;

    USHORT  usCommand = SHORT1FROMMP(mp1);
    switch (usCommand)
    {
        /*
         * ID_WIDI_NEXTBUTTON:
         *      "Next" button pressed.
         */

        case ID_WIDI_NEXTBUTTON:
            if (G_pCurrentPageInfo->_lNextButton > 0)
                WinPostMsg(G_hwndPagesClient,
                           WPIM_TURNTOPAGE,
                           (MPARAM)G_pCurrentPageInfo->_lNextButton,
                           (MPARAM)TRUE);  // store in "visited" list
            else
            {
                // button index == 0:
                // this means start installation.
                // Display summary:

                ULONG   ulToAdd = 0, ulToRemove = 0;
                BOOL    fContinue = FALSE;
                ustring aStrings[2];

                pEngine->CheckJobs(&ulToAdd, &ulToRemove);

                if ((ulToAdd) && (ulToRemove))
                {
                    // both install and de-install:
                    aStrings[0]._printf("%u", ulToAdd);
                    aStrings[1]._printf("%u", ulToRemove);
                    if (pEngine->_Locals.MessageBox(100,
                                      175,
                                      MSG_CONFIRM_YESNO_DEFYES,
                                      aStrings, 2)
                            == MBID_YES)
                        fContinue = TRUE;
                }
                else if (ulToAdd)
                {
                    // install only:
                    aStrings[0]._printf("%u", ulToAdd);
                    if (pEngine->_Locals.MessageBox(100,
                                      173,
                                      MSG_CONFIRM_YESNO_DEFYES,
                                      aStrings, 1)
                            == MBID_YES)
                        fContinue = TRUE;
                }
                else if (ulToRemove)
                {
                    // install only:
                    aStrings[0]._printf("%u", ulToRemove);
                    if (pEngine->_Locals.MessageBox(100,
                                      174,
                                      MSG_CONFIRM_YESNO_DEFYES,
                                      aStrings, 1)
                            == MBID_YES)
                        fContinue = TRUE;
                }

                if (fContinue)
                    WinPostMsg(hwndDlg,
                               WPIM_DISMISS,
                               (MPARAM)TRUE,        // exit pages and start install
                               0);
            }
        break;

        /*
         * ID_WIDI_BACKBUTTON:
         *      "Back" button pressed.
         */

        case ID_WIDI_BACKBUTTON:
            // TurnToPage has stored the last page in G_WpiGlobals
            if (!G_WpiGlobals.lPagesList.empty())
            {
                // get the LONG before calling pop_back, which
                // will delete the node V0.9.14 (2001-07-07) [umoeller]
                GUIVisited *lPrev = G_WpiGlobals.lPagesList.back();
                long l = lPrev->lPageNo;
                // delete last item in list
                G_WpiGlobals.lPagesList.pop_back();
                WinPostMsg(G_hwndPagesClient,
                           WPIM_TURNTOPAGE,
                           (MPARAM)l,
                           (MPARAM)FALSE);  // do _not_ store in "visited" list
            }
        break;

        /*
         * ID_WIMI_ARCHIVE_CREATE_CID:
         *      "Create CID file" button pressed.
         */

        case ID_WIMI_ARCHIVE_CREATE_CID:
        {
            FILEDLG fd;
            GUILocals *pLocals = (GUILocals *)&pEngine->_Locals;
            memset(&fd, 0, sizeof(FILEDLG));
            fd.cbSize = sizeof(FILEDLG);
            fd.fl = FDS_SAVEAS_DIALOG
                      | FDS_ENABLEFILELB
                      | FDS_CENTER;

            // file name of cid file: like archive,
            // but replace extension with .xcd
            string strTemp(pLocals->_pCodecGui,
                           pEngine->_pCurrentArchive->_ustrArcFilename);
            strcpy(fd.szFullFile,
                   strTemp.c_str());
            PSZ p;
            if (p = strrchr(fd.szFullFile, '.'))
                strcpy(p, ".xcd");

            if (    (WinFileDlg(HWND_DESKTOP,    // parent
                                hwndDlg,
                                &fd))
                 && (fd.lReturn == DID_OK)
               )
            {
                ustring ustrFullFile(pLocals->_pCodecGui, fd.szFullFile);

                if (!access(fd.szFullFile, 0))
                {
                    // file exists:
                    if (pEngine->_Locals.MessageBox(108,
                                                    125,
                                                    MSG_CONFIRM_YESNO_DEFYES,
                                                    &ustrFullFile, 1)
                            != MBID_YES)
                        break;
                }

                // else:
                FECID cid(*pEngine, ustrFullFile);
                cid.Write();
            }
        }
        break;

        /*
         * ID_WIMI_ARCHIVE_VIEW:
         *      "view archive" menu item selected.
         */

        case ID_WIMI_ARCHIVE_VIEW:
            ViewArchives(*pEngine,
                         hwndDlg);
        break;

        /*
         * ID_WIMI_SELECT_DEFAULT etc.:
         *      menu items from "selections" menu selected.
         */

        case ID_WIMI_SELECT_DEFAULT:
        case ID_WIMI_SELECT_REPAIR:
        case ID_WIMI_SELECT_UPDATE:
        case ID_WIMI_SELECT_FULL:
        case ID_WIMI_SELECT_DEINSTALL:
        {
            // items from "Selections" menu:
            // go thru all jobs
            list<FEJobBase*>::iterator JobStart = pEngine->_pAllJobsList->begin(),
                                       JobEnd = pEngine->_pAllJobsList->end();
            for (; JobStart != JobEnd; JobStart++)
            {
                FEJobBase *pTemp = *JobStart;
                FEInstallJob *pJobThis;
                if (pJobThis = pTemp->IsInstallJob())
                {
                    // install job, not group:

                    // this is not needed V0.9.18 (2002-02-06) [umoeller]
                    // const FEArcPackageBase* pPackageThis = pJobThis->_pArcPackage;
                    // if (pPackageThis->IsPackage())

                    switch (usCommand)
                    {
                        case ID_WIMI_SELECT_DEFAULT:
                            pJobThis->DefaultSelect();
                        break;

                        case ID_WIMI_SELECT_REPAIR:
                            pJobThis->RepairSelect();
                        break;

                        case ID_WIMI_SELECT_UPDATE:
                            pJobThis->UpdateSelect();
                        break;

                        case ID_WIMI_SELECT_FULL:
                            pJobThis->Select(JOB_INSTALL);
                        break;

                        case ID_WIMI_SELECT_DEINSTALL:
                            pJobThis->Select(JOB_DEINSTALL);
                        break;
                    }
                }
            } // end for (; JobStart != JobEnd; JobStart++)

            // update drive info V0.9.14 (2001-08-25) [umoeller]
            WinPostMsg(G_hwndDriveInfo, WPIM_UPDATE, (MPARAM)1, NULL);
        }
        break;

        default:
        {
            // other command: check default menu items (gui_shared.cpp)
            GUILocals *pLocals = (GUILocals *)&pEngine->_Locals;
            if (!gshrDefWindowProc(pLocals,
                                   hwndDlg, WM_COMMAND, mp1, mp2,
                                   G_ulPageHelp))
                // not processed:
                if (pEngine->_Locals.MessageBox(100,
                                  172,  // "Sure you wanna cancel?"
                                  MSG_CONFIRM_YESNO_DEFYES)
                        == MBID_YES)
                    WinPostMsg(hwndDlg, WM_QUIT, 0, 0);
        }
    }

    return (mrc);
}


/*
 *@@ fnwpPages:
 *      shared window proc for the main window and the
 *      sub-dialogs. Yes, they all share this window proc.
 *
 *      All dialogs also have a pointer to the GUIInstallEngine
 *      that handles the install in QWL_USER since this
 *      is passed as the create param with WM_INITDLG
 *      from guiBeginInstallMode.
 *
 *      This is responsible for almost everything that the
 *      front-end can do in "full install" mode.
 *
 *      NOTE: This function is temporarily subclassed when
 *      we are on the "Configure" page since we now use
 *      a checkbox container and ctlMakeCheckboxContainer
 *      subclasses the container owner too. See TurnToPage
 *      which is responsible for this.
 *
 *@@changed V0.9.3 (2000-05-11) [umoeller]: "Cancel" didn't close main window; fixed
 *@@changed V0.9.4 (2000-07-26) [umoeller]: fixed CID filename
 *@@changed V0.9.18 (2002-03-03) [umoeller]: turned this into window proc for new GUI client class
 *@@changed V0.9.18 (2002-03-03) [umoeller]: added keyboard support
 *@@changed V0.9.20 (2002-07-19) [umoeller]: made close confirmation more consistent
 */

MRESULT EXPENTRY fnwpPages(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    MRESULT mrc = 0;

    GUIInstallEngine *pEngine = (GUIInstallEngine*)WinQueryWindowPtr(hwndDlg, QWL_USER);
            // V0.9.14 (2001-07-26) [umoeller]

    switch (msg)
    {
        /*
         * WPIM_TURNTOPAGE:
         *      this use message gets posted when
         *      we need to switch to another page.
         *      The new page is in (ULONG)mp1;
         *      if ((BOOL)mp2 == TRUE), the new page
         *      will be stored in the "visited" list.
         */

        case WPIM_TURNTOPAGE:
            if (pEngine)
                // set up the dialog controls for the page
                // that we want
                TurnToPage(*pEngine,
                           (LONG)mp1,
                           (BOOL)mp2);
        break;

        case WM_WINDOWPOSCHANGED:
            // resized?
            if (    (((PSWP)mp1)->fl & SWP_SIZE)
                 && (G_fXacInited)
               )
                AdjustControls(((PSWP)mp1));
        break;

        case WM_PAINT:
        {
            RECTL rcl;
            HPS hps = WinBeginPaint(hwndDlg, NULLHANDLE, &rcl);
            GpiCreateLogColorTable(hps, 0, LCOLF_RGB, 0, 0, NULL);
            WinFillRect(hps,
                        &rcl,
                        winhQueryPresColor(hwndDlg,
                                           PP_BACKGROUNDCOLOR,
                                           FALSE,
                                           SYSCLR_BACKGROUND));
            WinEndPaint(hps);
        }
        break;

        /*
         * WM_CONTROL:
         *
         */

        case WM_CONTROL:
            if (pEngine)
                mrc = PagesControl(pEngine, hwndDlg, mp1, mp2);
        break;

        /*
         * WM_DRAWITEM:
         *      container record core owner-draw
         */

        case WM_DRAWITEM:
            switch (SHORT1FROMMP(mp1))
            {
                case ID_WIDI_PACKAGESCNR:
                    mrc = CnrPackageOwnerDraw((GUILocals*)&pEngine->_Locals,
                                              (POWNERITEM)mp2);
                break;
            }
        break;

        /*
         * WM_COMMAND:
         *
         */

        case WM_COMMAND:
            if (pEngine)
                mrc = PagesCommand(pEngine, hwndDlg, mp1, mp2);
        break;

        case WM_CHAR:
        {
            USHORT usFlags    = SHORT1FROMMP(mp1);
            USHORT usch       = SHORT1FROMMP(mp2);
            USHORT usvk       = SHORT2FROMMP(mp2);

            // key down msg?
            if ((usFlags & KC_KEYUP) == 0)
            {
                _Pmpf((__FUNCTION__ ": usFlags = 0x%lX, usch = %d, usvk = %d",
                            usFlags, usch, usvk));

                if (usFlags & KC_VIRTUALKEY)
                {
                    switch (usvk)
                    {
                        case VK_TAB:
                            // find next focus window
                            dlghSetNextFocus(G_pllControls);
                        break;

                        case VK_BACKTAB:
                            // note: shift+tab produces this!!
                            dlghSetPrevFocus(G_pllControls);
                        break;

                        case VK_ESC:
                            WinPostMsg(hwndDlg,
                                       WM_COMMAND,
                                       (MPARAM)DID_CANCEL,
                                       0);
                        break;

                        case VK_NEWLINE:        // this comes from the main key
                        case VK_ENTER:          // this comes from the numeric keypad
                            dlghEnter(G_pllControls);
                        break;
                    } // end switch
                } // end if (usFlags & KC_VIRTUALKEY)
                else
                {
                    // no virtual key:
                    // find the control for the keypress
                    dlghProcessMnemonic(G_pllControls, usch);
                }
            }
        }
        break;

        /*
         * WM_CLOSE:
         *      install window is to be closed: this comes in
         *      both when the user hits the close button, as
         *      well as a "Cancel" button, as well as the
         *      "Exit" menu item in the "WarpIN" menu. Display
         *      a confirmation for all these cases now.
         *      V0.9.20 (2002-07-19) [umoeller]
         */

        case WM_CLOSE:
            // exit:
            if (((GUILocals*)&pEngine->_Locals)->MessageBox(100,
                                                            172,  // "Sure you wanna cancel?"
                                                            MSG_CONFIRM_YESNO_DEFYES)
                    == MBID_YES)
                WinPostMsg(hwndDlg, WM_QUIT, 0, 0);
        break;

        default:
            if (pEngine)
            {
                if (!gshrDefWindowProc((GUILocals*)&pEngine->_Locals,
                                        hwndDlg, msg, mp1, mp2,
                                        G_ulPageHelp))   // help panel
                    mrc = WinDefWindowProc(hwndDlg, msg, mp1, mp2);
            }
            else
                mrc = WinDefWindowProc(hwndDlg, msg, mp1, mp2);
    }

    return (mrc);
}

/* ******************************************************************
 *
 *  PM implementation part 2:
 *      Required callbacks on thread 1.
 *
 ********************************************************************/

/*
 *@@ guiShowMessage2:
 *      this should display a message box
 *      with the given strings.
 *
 *@@added V0.9.16 (2001-09-20) [umoeller]
 */

ULONG guiShowMessage2(const BSString &strTitle,
                      const BSString &strMessage,
                      ULONG flFlags)
{
    HWND hwndOwner = G_hwndMainFrame;
    if (!WinIsWindowVisible(hwndOwner))
        hwndOwner = NULLHANDLE; // HWND_DESKTOP; V0.9.9 (2001-02-01) [umoeller]

    static MSGBOXSTRINGS s_MsgBoxStrings =
            {
                nlsGetString(WPSI_YES),
                nlsGetString(WPSI_NO),
                nlsGetString(WPSI_OK),
                nlsGetString(WPSI_CANCEL),
                nlsGetString(WPSI_ABORT),
                nlsGetString(WPSI_RETRY),
                nlsGetString(WPSI_IGNORE),
                nlsGetString(WPSI_ENTER),
                nlsGetString(WPSI_YESTOALL)
            };

    return (dlghMessageBox(hwndOwner,
                           G_hptrMain,
                           strTitle.c_str(),
                           strMessage.c_str(),
                           NULL,
                           flFlags,
                           gshrQueryDefaultFont(),
                           &s_MsgBoxStrings));
}

/*
 *@@ guiBeginInstallMode:
 *      this gets called from EnterArchiveMode for preparing
 *      "install" mode, that is, when all the following are
 *      true:
 *
 *      --  an archive was passed on the command line,
 *
 *      --  the archive was successfully loaded,
 *
 *      --  the install script was successfully parsed,
 *
 *      --  the WARPIN_DISPLAYPAGES environment variable
 *          has not been set to "NO".
 *
 *      Required callback.
 *
 *      This gets called no matter if packages from the
 *      archive have already been installed.
 *
 *      This func has no arguments, because all the
 *      needed data has been stored in WPIGLOBALS.
 *
 *      It is the responsibility of this function to start
 *      displaying all the pages described in the install
 *      script. After the last page has been turned to
 *      (i.e. installation is about to begin), this function
 *      must return with TRUE. If the user chooses to cancel
 *      the install process beforehand, this must return FALSE.
 *
 *      If TRUE is returned, the front end will process the list
 *      of packages and build a job list accordingly and start
 *      installing.
 *
 *      If FALSE is returned, WarpIN is terminated.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: renamed from guiBeginAddRemove
 *@@changed V0.9.4 (2000-07-26) [umoeller]: renamed from guiPrepareFullInstall
 *@@changed V0.9.12 (2001-05-17) [umoeller]: path entry field still had truncated paths, fixed
 *@@changed V0.9.14 (2001-08-09) [umoeller]: fixed crashes if something threw an exception while in here
 *@@changed V0.9.18 (2002-03-03) [umoeller]: adjusted for new GUI handling (client class etc.)
 *@@changed V0.9.18 (2002-03-03) [umoeller]: fixed splash dlg which would never disappear
 */

BOOL guiBeginInstallMode(GUIInstallEngine &Engine)
{

    GUILocals *pLocals = (GUILocals *)&Engine._Locals;

    /*
     * Initialize GUI:
     *
     */

    PCSZ WC_WARPINPAGES = "WarpINPagesClient";

    HAB habThread1 = Engine._Locals._habThread1;

    // register our new client class V0.9.18 (2002-03-03) [umoeller]
    WinRegisterClass(habThread1,
                     WC_WARPINPAGES,
                     fnwpPages,
                     CS_SIZEREDRAW | CS_CLIPCHILDREN,
                     sizeof(PVOID));

    // set main window title
    BSString    strWinTitle = "WarpIN";
    if (Engine._ustrAppTitle())
    {
        strWinTitle += ": ";
        strWinTitle.appendUtf8(pLocals->_pCodecGui, Engine._ustrAppTitle);
    }

    G_hwndMainFrame = winhCreateStdWindow(HWND_DESKTOP,
                                          NULL,
                                          FCF_TITLEBAR | FCF_MAXBUTTON | FCF_SYSMENU | FCF_SIZEBORDER | FCF_NOBYTEALIGN,
                                          WS_ANIMATE,
                                          strWinTitle.c_str(),
                                          0,
                                          WC_WARPINPAGES,
                                          WS_VISIBLE,
                                          0,
                                          NULL,
                                          &G_hwndPagesClient);

    // the main window is still hidden!

    // store engine pointer so the pages can find that
    WinSetWindowPtr(G_hwndPagesClient, QWL_USER, &Engine);

    // load menu -- the ol' menu in dlg trick
    G_hmenuMain = WinLoadMenu(G_hwndMainFrame, NULLHANDLE, ID_WIM_MAINMENU);
    // remove "View" menu V0.9.2 (2000-02-19) [umoeller]
    WinSendMsg(G_hmenuMain,
               MM_REMOVEITEM,
               MPFROM2SHORT(ID_WIM_VIEW, TRUE),
               0);
    WinSendMsg(G_hwndMainFrame, WM_UPDATEFRAME, (MPARAM)FCF_MENU, 0);

    // set a background presparam on the client which will be
    // inherited by all the controls... otherwise we keep getting
    // white backgrounds on statics
    // V0.9.18 (2002-03-03) [umoeller]
    winhSetPresColor(G_hwndPagesClient,
                     PP_BACKGROUNDCOLOR,
                     // RGBCOL_RED);
                     WinQuerySysColor(HWND_DESKTOP, SYSCLR_APPWORKSPACE, 0));

    // now load the four subdialogs, which will be
    // children of the main window. They all use
    // fnwpPages as their window procedure also,
    // since we have all the message handling in
    // there.
    /*
    G_hwndPageText = gshrLoadDlg(G_hwndMain,     // parent
                                 G_hwndMain,
                                 fnwpPages,
                                 NULLHANDLE,
                                 ID_WID_PAGE_TEXT,
                                 &Engine);      // create param
    // replace static text with XTextView control
    HWND hwndTextView = txvReplaceWithTextView(G_hwndPageText,
                                               ID_WIDI_INFOTEXT,
                                               WS_VISIBLE | WS_TABSTOP,
                                               XTXF_VSCROLL | XTXF_AUTOVHIDE,
                                               2);
    WinSendMsg(hwndTextView, TXM_SETWORDWRAP, (MPARAM)TRUE, 0);

    // load readme page subdialog
    G_hwndPageReadme = gshrLoadDlg(G_hwndMain,     // parent
                                   G_hwndMain,
                                   fnwpPages,
                                   NULLHANDLE,
                                   ID_WID_PAGE_README,
                                   &Engine);      // create param
    // replace static text with XTextView control
    hwndTextView = txvReplaceWithTextView(G_hwndPageReadme,
                                          ID_WIDI_INFOTEXT,
                                          WS_VISIBLE | WS_TABSTOP,
                                          XTXF_VSCROLL | XTXF_AUTOVHIDE,
                                          2);
    WinSendMsg(hwndTextView, TXM_SETWORDWRAP, (MPARAM)TRUE, 0);

    // replace readme text with XTextView control
    hwndTextView = txvReplaceWithTextView(G_hwndPageReadme,
                                          ID_WIDI_READMEMLE,
                                          WS_VISIBLE | WS_TABSTOP,
                                          XTXF_VSCROLL | XTXF_AUTOVHIDE
                                            | XTXF_HSCROLL | XTXF_AUTOHHIDE,
                                          2);

    // load container page subdialog
    G_hwndPageContainer = gshrLoadDlg(G_hwndMain,     // parent
                                      G_hwndMain,
                                      fnwpPages,
                                      NULLHANDLE,
                                      ID_WID_PAGE_CONTAINER,
                                      &Engine);      // create param
    hwndTextView = txvReplaceWithTextView(G_hwndPageContainer,
                                          ID_WIDI_INFOTEXT,
                                          WS_VISIBLE | WS_TABSTOP,
                                          XTXF_VSCROLL | XTXF_AUTOVHIDE,
                                          2);
    WinSendMsg(hwndTextView, TXM_SETWORDWRAP, (MPARAM)TRUE, 0);
    // set entry field path limit V0.9.12 (2001-05-17) [umoeller]
    winhSetEntryFieldLimit(WinWindowFromID(G_hwndPageContainer, ID_WIDI_PATHENTRY),
                           254);

    // load configure page subdialog
    G_hwndPageConfigure = gshrLoadDlg(G_hwndMain,     // parent
                                     G_hwndMain,
                                     fnwpPages,
                                     NULLHANDLE,
                                     ID_WID_PAGE_CONFIGURE,
                                     &Engine);      // create param
    hwndTextView = txvReplaceWithTextView(G_hwndPageConfigure,
                                          ID_WIDI_INFOTEXT,
                                          WS_VISIBLE | WS_TABSTOP,
                                          XTXF_VSCROLL | XTXF_AUTOVHIDE,
                                          2);
    WinSendMsg(hwndTextView, TXM_SETWORDWRAP, (MPARAM)TRUE, 0);
    */

    // this loads the drive-info window, which just
    // has the cnr for displaying drives.
    G_hwndDriveInfo = gshrLoadDlg(HWND_DESKTOP,        // parent
                                  G_hwndMainFrame,         // owner is the main window
                                  fnwpDriveInfo,
                                  0, ID_WID_DRIVEINFO,
                                  // pass engine
                                  &Engine);

    // add ourselves to the tasklist
    winhAddToTasklist(G_hwndMainFrame,
                      G_hptrMain);

    // hide the startup window
    WinDestroyWindow(G_hwndSplashDlg);
    G_hwndSplashDlg = NULLHANDLE;

    /*
     * Start user interaction:
     *
     */

    // start with page 1
    WinPostMsg(G_hwndPagesClient, WPIM_TURNTOPAGE, (MPARAM)1, MPNULL);

    WinSetWindowPos(G_hwndMainFrame,
                    HWND_TOP,
                    100, 100, 500, 500,
                    SWP_SIZE);

    winhCenterWindow(G_hwndMainFrame); // invisibly

    WinSetWindowPos(G_hwndMainFrame,
                    HWND_TOP,
                    100, 100, 500, 500,
                    SWP_SHOW | SWP_ACTIVATE);

    BOOL    fStartInstall = FALSE;

    try
    {
        // go!
                // fnwpPages takes over;
                // this does not return until we either
                // get to the last page and the button was
                // pressed there (then DID_OK is returned)
                // or everything has been cancelled
        QMSG qmsg;
        BOOL fQuit = FALSE;
        while (WinGetMsg(habThread1,
                         &qmsg, 0, 0, 0))
        {
            if (    (qmsg.msg == WPIM_DISMISS)
                 && (qmsg.hwnd == G_hwndPagesClient)
               )
            {
                fStartInstall = (BOOL)qmsg.mp1;
                fQuit = TRUE;
            }

            WinDispatchMsg(habThread1, &qmsg);

            if (fQuit)
                break;
        }
    }
    catch(...)
    {
        // we must destroy the main window or the
        // windowing system gets into solid trouble
        // V0.9.14 (2001-08-09) [umoeller]
        WinDestroyWindow(G_hwndMainFrame);
        G_hwndMainFrame = NULLHANDLE;

        throw;      // rethrow to main handler
    }

    WinDestroyWindow(G_hwndMainFrame);
    G_hwndMainFrame = NULLHANDLE;

    // return FALSE if Cancel was pressed or something went wrong
    return (fStartInstall);
}

/*
 *@@ guiProductInfo:
 *      this should display the WarpIN product
 *      info window.
 *
 *      Required callback.
 */

VOID guiProductInfo(HWND hwndOwner,
                    GUILocals *pLocals,
                    BOOL fLongVersion)
{
    ustring ustr;
    pLocals->GetCopyrightInfo(ustr,
                              pLocals->_pDatabase->QueryDatabaseFile());
    gshrMLEMessageBox(pLocals,
                      150,               // title
                      (fLongVersion) ? 156 : 101,
                      FALSE,             // hide cancel
                      &ustr, 1);     // string replacements
}


