
/*
 *@@sourcefile fe_script_old.cpp:
 *      implementation of script parsing for old-style scripts.
 *      See FEScriptBase for an introduction about script
 *      parsing.
 *
 *      This file has the following implementations:
 *
 *      -- FEOldScript as an implementation for FEScriptBase
 *         (for parsing old-style scripts);
 *
 *      -- FEOldPckDecl as an implementation for FEPckDeclBase
 *         (for packages in an old-style script).
 *
 *      This has exactly the code which used to be in FEArchive...
 *      except that a lot of code had to be adjusted to this
 *      abstraction.
 *
 *@@header "engine\fe_script_old.h"
 *@@added V0.9.9 (2001-02-19) [umoeller]
 */

/*
 *      This file Copyright (C) 1999-2001 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_REXXSAA
#define INCL_DOSERRORS
#define INCL_DOSMISC
#define INCL_WINSHELLDATA
#include <os2.h>

#ifdef __IBMCPP__
    #include <rexxsaa.h>    // EMX has this in os2.h already
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>               // needed for WIFileHeader

#include "setup.h"

// #include <list>
// #include <map>

#include "bldlevel.h"           // needed for version checks

#include "helpers\configsys.h"
#include "helpers\dosh.h"
#include "helpers\nls.h"
#include "helpers\stringh.h"
#include "helpers\xstring.h"

#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"

#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

#include "wiarchive\wiarchive.h"

#include "engine\fe_base.h"

#include "engine\fe_script.h"
#include "engine\fe_script_old.h"

#include "engine\fe_rexx.h"

#pragma hdrstop

DEFINE_CLASS(FEOldPckDecl, FEPckDeclBase);
DEFINE_CLASS(FEOldScript, FEScriptBase);

/* ******************************************************************
 *
 *  FEOldScript implementation
 *
 ********************************************************************/

/*
 *@@ Parse:
 *      virtual FEScriptBase method which is overridden
 *      here to parse old-style scripts.
 *
 *      This calls FEOldScript::ParseWARPINTag,
 *      FEOldScript::ParseHeader, and FEOldScript::ParsePages
 *      (in this order).
 *
 *      About Codepage support:
 *
 *      FEOldScript::ParseWARPINTag checks the CODEPAGE
 *      attribute and creates a BSUniCodec in the instance
 *      data accordingly, which is then used for properly
 *      setting the script data in UTF-8.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 *@@changed V0.9.20 (2002-07-03) [umoeller]: removed rexx allowed callback
 */

VOID FEOldScript::Parse(const char *pcszScript)
{
    // make a copy of the script while we're parsing
    _strScript = pcszScript;

    // replace all tabs with single spaces;
    // XML doesn't care, but the old parser does
    // V0.9.3 (2000-04-28) [umoeller]
    PSZ p = (PSZ)_strScript.c_str();
    while (*p)
    {
        if (*p == '\t')
            *p = ' ';
        ++p;
    }

    ParseWARPINTag();

    ParseHeader();

    ParsePages();

    _strScript.erase();     // free the buffer
}

/*
 *@@ ParseError:
 *      throws FEScriptExcpt when an error occured parsing
 *      the installation script. This will terminate WarpIN.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this was wpiParseError (warpin.cpp)
 *@@changed V0.9.0 (99-10-31) [umoeller]: now throws a FEScriptExcpt
 *@@changed V0.9.1 (2000-01-07) [umoeller]: now throws an FEFatalErrorExcpt
 *@@changed V0.9.9 (2001-02-28) [umoeller]: added vargs
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 *@@changed V0.9.15 (2001-08-26) [umoeller]: fixed bad error reports
 *@@changed V0.9.18 (2002-03-08) [umoeller]: rewritten
 */

VOID FEOldScript::ParseError(ULONG ulOfs,
                             int iMessage,
                             ustring *paStrings, //  = NULL,
                             int iCount) //  = 0)
{
    ULONG   ulLine = 0;

    if (ulOfs)
    {
        ulLine = 1;

        // count lines up to offset
        PCSZ    p = _strScript.c_str(),
                pEnd = p + ulOfs;
        while (p < pEnd)
        {
            if (*p == '\n')
                ++ulLine;
            ++p;
        }
    }

    throw FEScriptExcpt(ulLine,
                        0,          // column
                        _Locals,
                        iMessage,
                        paStrings,
                        iCount);
}

/*
 *@@ ParseErrorMissingAttrib:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

VOID FEOldScript::ParseErrorMissingAttrib(ULONG ulOfs,
                                          const char *pcszAttribute,
                                          const char *pcszTag)
{
    ustring astr[2];
    astr[0].assignUtf8(pcszAttribute);
    astr[1].assignUtf8(pcszTag);
    ParseError(ulOfs,
               233, // Missing %1 attribute in %2 tag.
               astr, 2);
}

/*
 *@@ GetBlock:
 *      wrapper function for strhGetBlock, which
 *      works on string's and can also display an
 *      error message.
 *
 *      strhGetBlock returns 1 if pszTag was not
 *      found and 2 if the opening, but no closing
 *      tag was found.
 *
 *      This is only a private helper method.
 *
 *      This now has full REXX support if the flREXX
 *      flags are set to anything != 0. If so,
 *      FERexx::ExecuteRexxMacros gets called
 *      with the contents of *pstrBlock and/or
 *      *pstrAttribs.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this was wpiGetBlock (warpin.cpp)
 *@@changed V0.9.0 (99-11-07) [umoeller]: now using string's
 *@@changed V0.9.1 (2000-01-06) [umoeller]: updated prototype because line numbers were always wrong
 *@@changed V0.9.2 (2000-03-11) [umoeller]: added REXX support
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 *@@changed V0.9.18 (2002-03-08) [umoeller]: now using const string& for input
 */

BOOL FEOldScript::GetBlock(const string &strSearchIn,
                                    // in: buffer to search
                           PULONG pulSearchOfs,
                                      // in/out: where to begin search (or 0 for start)
                           const string &strTag,
                                      // in: tag to search for (without brackets)
                           ULONG flREXX,
                                      // in: REXX flags; any or none of the following:
                                      // -- GBLREXX_ATTRIBS: execute REXX code in attribs
                                      // -- GBLREXX_BLOCK: execute REXX code in block
                           string *pstrBlock,
                                      // out: stuff between opening and closing tag;
                                      // pointer can be NULL if not desired
                           string *pstrAttribs,
                                      // out: attributes of opening tag;
                                      // pointer can be NULL if not desired
                           PULONG pulOfsBeginTag,
                           PULONG pulOfsBeginBlock)
{
    PSZ     pszBlock = 0;
    PSZ     pszAttribs = 0;

    ULONG ulrc  = strhGetBlock(strSearchIn.c_str(),
                               pulSearchOfs,
                               strTag.c_str(),
                               &pszBlock,
                               &pszAttribs,
                               pulOfsBeginTag,
                               pulOfsBeginBlock);

    if (ulrc == 2)
    {
        ustring ustr;
        ustr.assignCP(_pCodec, strTag);
        ParseError(*pulSearchOfs,
                   248, // Missing closing tag for "<%1>" tag.
                   &ustr, 1);
            // this terminates WarpIN
    }

    if (pstrBlock)
        *pstrBlock = pszBlock;
    if (pstrAttribs)
        *pstrAttribs = pszAttribs;

    if (pszBlock)
        free(pszBlock);
    if (pszAttribs)
        free(pszAttribs);

    if (flREXX & GBLREXX_ATTRIBS)
    {
        string strRexxMacroName;
        ULONG ulRexxOfs;

        if (!(_Rexx.ExecuteRexxMacros(*pstrAttribs,
                                      &ulRexxOfs,
                                      strRexxMacroName)))
        {
            ustring aStrings[2];
            aStrings[0].assignCP(_pCodec, strTag);
            aStrings[1].assignCP(_pCodec, strRexxMacroName);
            ParseError(*pulSearchOfs + ulRexxOfs,
                       249, // The REXX code in the <%1> attributes failed.
                            // The macro call was: =(\"%2\")
                       aStrings, 2);
        }
    }

    if (flREXX & GBLREXX_BLOCK)
    {
        string strRexxMacroName;
        ULONG ulRexxOfs;

        if (!(_Rexx.ExecuteRexxMacros(*pstrBlock,
                                      &ulRexxOfs,
                                      strRexxMacroName)))
        {
            ustring aStrings[2];
            aStrings[0].assignCP(_pCodec, strTag);
            aStrings[1].assignCP(_pCodec, strRexxMacroName);
            ParseError(*pulSearchOfs + ulRexxOfs,
                       250, // The REXX code within the <%1>...</%1> block failed.
                            // The macro call was: =(\"%s\")
                       aStrings, 2);
        }
    }

    return (ulrc == 0);     // TRUE if tag was found
}

/*
 *@@ ParseWARPINTag:
 *      this gets called by FEOldScript::Parse to parse
 *      the WARPIN tag attributes. This terminates WarpIN
 *      if the requirements are not met.
 *
 *      Note: This receives the legacy script, which is
 *      not yet converted to Unicode.
 *
 *      Throws:
 *      -- FEFatalErrorExcpt.
 *      -- BSCancelExcpt.
 *
 *@@added V0.9.0 (99-11-06) [umoeller]
 *@@changed V0.9.2 (2000-03-11) [umoeller]: added CODEPAGE attribute
 *@@changed V0.9.4 (2000-07-26) [umoeller]: added three-part WARPIN version checks
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 *@@changed V0.9.14 (2001-07-28) [umoeller]: re-enabled codepage support, which was broken
 *@@changed V0.9.16 (2001-10-19) [umoeller]: added OS attribute, which was never implemented
 */

VOID FEOldScript::ParseWARPINTag()
{
    ULONG       ulOfs = 0;
    string      strWarpINBlock,
                strWarpINAttrs;
    ULONG       ulWarpINTagOfs = 0,
                ulWarINBlockOfs = 0;

    if (!GetBlock(_strScript, &ulOfs,
                  "WARPIN",
                  0, // no REXX
                  &strWarpINBlock, &strWarpINAttrs,
                  &ulWarpINTagOfs, &ulWarINBlockOfs))
    {
        throw FEScriptExcpt(0, 0, _Locals, 170);
    }

    // codepage V0.9.14 (2001-07-26) [umoeller]
    if (!strhGetNumAttribValue(strWarpINAttrs.c_str(),
                               "CODEPAGE", (PLONG)&_ulCodepage))
        // FEScriptBase has set this to 0 previously for Unicode,
        // so if not specified, set 850 as default
        _ulCodepage = 850;

    _pCodec = new BSUniCodec(_ulCodepage);

    // VERSION attribute
    PSZ pszOSVersion;
    if (pszOSVersion = strhGetTextAttr(strWarpINAttrs.c_str(), "OS", NULL))
    {
        ULONG       aulBuf[3];

        DosQuerySysInfo(QSV_VERSION_MAJOR,      // 11
                        QSV_VERSION_MINOR,      // 12
                        &aulBuf, sizeof(aulBuf));
            // Warp 3 is reported as 20.30
            // Warp 4 is reported as 20.40
            // Aurora is reported as 20.45

        if (aulBuf[0] == 20)
        {
            PCSZ pcszRequires = NULL;

            strupr(pszOSVersion);
            if (!strcmp(pszOSVersion, "OS2_2X"))
                // this is OK always
                ;
            else if (!strcmp(pszOSVersion, "OS2_3X"))
            {
                if (aulBuf[1] < 30)
                    pcszRequires = "OS/2 Warp 3";
            }
            else if (!strcmp(pszOSVersion, "OS2_4X"))
            {
                if (aulBuf[1] < 40)
                    pcszRequires = "OS/2 Warp 4";
            }
            else if (!strcmp(pszOSVersion, "OS2_45X"))
            {
                if (aulBuf[1] < 45)
                    pcszRequires = "OS/2 Warp 4.5 (Warp 4 FP 13, WSeB, MCP/ACP, eComStation)";
            }
            else
                // invalid version string:
                throw FEScriptExcpt(0, 0, _Locals, 218);

            if (pcszRequires)
            {
                // requirement not met:
                ustring ustr(_pCodec, pcszRequires);
                throw FEFatalErrorExcpt(_Locals, 219, &ustr, 1);
            }
        }

        free(pszOSVersion);
    }

    // VERSION attribute
    PSZ pszReqVersion;
    if (pszReqVersion = strhGetTextAttr(strWarpINAttrs.c_str(), "VERSION", NULL))
    {
        ULONG   ulReqMajor,
                ulReqMinor,
                ulReqRevision = 0;
        ULONG ulScanned = sscanf(pszReqVersion,
                                 "%u.%u.%u",
                                 &ulReqMajor,
                                 &ulReqMinor,
                                 &ulReqRevision);
        ULONG   ulRunMajor, ulRunMinor, ulRunRevision;
        sscanf(BLDLEVEL_VERSION,
               "%u.%u.%u",
               &ulRunMajor,
               &ulRunMinor,
               &ulRunRevision);

        if (    // major higher
                (ulReqMajor > ulRunMajor)
             || (   // major equal:
                    (ulReqMajor == ulRunMajor)
                    // then minor must be higher
                 && (   (ulReqMinor > ulRunMinor)
                        // minor equal as well:
                     || (   (ulReqMinor == ulRunMinor)
                            // then revision must be higher
                         && (ulReqRevision > ulRunRevision)
                        )
                    )
                )
           )
        {
            ustring ustr(_pCodec, pszReqVersion);
            throw FEFatalErrorExcpt(_Locals,
                                    171,
                                    &ustr,
                                    1);
        }

        free(pszReqVersion);
    }
}

/*
 *@@ ParseRexxTags:
 *      pass the contents of a <HEAD>...</HEAD> tag to this function and
 *      it will find and interpret all contained <REXX NAME=xyz>code</REXX>
 *      tags. That means: the "code" is stored under the name "xyz" in the
 *      REXX code store within this REXX support class.
 *      The code is not veryfied at all in this function.
 *
 *      This calls FERexx::StoreRexxCode for each code block found.
 *
 *      Returns the no. of REXX tags found in the header.
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 *@@changed V0.9.12 (2001-05-31) [umoeller]: fixed return value
 *@@changed V0.9.20 (2002-07-03) [umoeller]: added codec param
 *@@changed V0.9.20 (2002-07-03) [umoeller]: moved this here from FERexx
 */

ULONG FEOldScript::ParseRexxTags(const char *pcszHeaderString)
{
    ULONG   ulrc = 0,           // 0 was missing V0.9.12 (2001-05-31) [umoeller]
            rc,
            ofs,
            ulStart,
            ulEnd;
    PSZ     pszBlock, pszAttribs, pszName;

    // loop through the input string
    for (ofs = 0; 1; ofs = ulEnd)
    {
        // see if there is a next REXX tag, terminate if not
        rc = strhGetBlock(pcszHeaderString,
                          &ofs,
                          "REXX",
                          &pszBlock,
                          &pszAttribs,
                          &ulStart,
                          &ulEnd);
        if (0 != rc)
            break;

        // if no name attribute found: ignore the rest: We cannot store it!
        if (    (NULL == pszAttribs)
             || (NULL == (pszName = strhGetTextAttr(pszAttribs, "NAME", &ofs)))
           )
        {
            free(pszBlock);
            free(pszAttribs);
            continue;
        }

        // now we have all we need to store the code
        strupr(pszName);
        _Rexx.StoreRexxCode(pszName, pszBlock, _pCodec);
        ++ulrc;

        // free all strings
        free(pszBlock);
        free(pszAttribs);
        free(pszName);
    }

    return ulrc;
}

/*
 *@@ ParseHeader:
 *      this private method gets called by FEOldScript::Parse
 *      to parse the HEAD.../HEAD block. This calls the
 *      other Parse* methods in turn.
 *
 *      Returns the no. of tags parsed.
 *
 *      Throws:
 *      -- BSCancelExcpt.
 *
 *@@added V0.9.0 (99-11-07) [umoeller]
 *@@changed V0.9.2 (2000-03-10) [cbo]: added bExternalCall
 *@@changed V0.9.2 (2000-03-10) [cbo]: added REXX support
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

ULONG FEOldScript::ParseHeader()
{
    ULONG   ulrc = 0;

    // first, extract the <HEAD>...</HEAD> part
    // from the profile

    ULONG       ulOfs = 0;
    string      strHeadBlock,
                strHeadAttrs,
                strRexxMacroName; // (cbo 6.3.00)
    ULONG       ulHeadTagOfs = 0,
                ulHeadBlockOfs = 0,
                ulBlockOfs = 0; // (cbo 6.3.00)
    if (!GetBlock(_strScript, &ulOfs,
                  "HEAD",
                  0, // no REXX
                  &strHeadBlock, &strHeadAttrs,
                  &ulHeadTagOfs, &ulHeadBlockOfs))
    {
        throw FEScriptExcpt(0, 0, _Locals, 104);  // "no <HEAD>"
    }

    // parse the block for <REXX>...</REXX> blocks and store the code.
    // If a "name" attribute occurs twice, the second occurrence will
    // overwrite the stored code from the first time.
    // V0.9.2 (2000-03-10) [cbo]
    ULONG cRexxTags;

    if (cRexxTags = ParseRexxTags(strHeadBlock.c_str()))
    {
        // there are REXX tags, and caller wants confirmation:
        // pass script to GUI in unicode
        ustring ustrScript(_pCodec, _strScript);
        if (!_Locals.ConfirmRexxAllowed(ustrScript))
            // confirmation func returned FALSE;
            throw(BSCancelExcpt());
    }

    ulrc += cRexxTags;
    // check whether a <MSG>...</MSG> block
    // exists in the HEAD block; if so, display
    // message and stop
    ulOfs = 0;
    string     strMessage;
    if (    (GetBlock(strHeadBlock,
                      &ulOfs,
                      "MSG",
                      GBLREXX_BLOCK,
                      &strMessage, 0, 0, 0))
       )
    {
        ustring ustr(_pCodec, strMessage);
        throw FEFatalErrorExcpt(_Locals, ustr);
    }

    // parse all VARPROMPT blocks
    ulrc += ParseVarPrompts(strHeadBlock,
                            ulHeadBlockOfs);

    // check if we have a <TITLE>...</TITLE> block;
    // this is optional and stored in the instance data
    ulOfs = 0;
    string strTitle;
    GetBlock(strHeadBlock,
             &ulOfs,
             "TITLE",
             GBLREXX_BLOCK,
             &strTitle, 0, 0, &ulBlockOfs);
    // convert
    _ustrTitle.assignCP(_pCodec, strTitle);

    // set up the list of FEPackages
    ulrc += ParsePackages(strHeadBlock,
                          NULL,             // for first call (this might recurse)
                          ulHeadBlockOfs);

    return ulrc;
}

/*
 *@@ ParseVarPrompts:
 *      this gets called by FEOldScript::ParseHeader to parse
 *      all VARPROMPT tags in the HEAD block.
 *
 *@@added V0.9.2 (2000-03-11) [umoeller]
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

ULONG FEOldScript::ParseVarPrompts(string strBlock, // in: HEAD block to process (first call)
                                   ULONG ulTotalBlockOfs) // in: offset of strBlock in profile (for error messages)
{
    ULONG   ulrc = 0;
    // PSZ     pszBlock = (PSZ)strBlock.c_str();  // pointer gets modified as we search on
    ULONG   ulSearchOfs = 0;

    do
    {
        string      strVarBlock,
                    strVarAttribs;
        ULONG       ulVarTagOfs = 0,
                    ulVarBlockOfs = 0;
        // extract new block
        if (GetBlock(strBlock, // pszSearch,
                     &ulSearchOfs,      // advances search offset to after VARPROMPT
                     "VARPROMPT",
                     GBLREXX_BLOCK,
                     &strVarBlock,
                     &strVarAttribs,
                     &ulVarTagOfs, &ulVarBlockOfs))
        {
            // found: get name
            PSZ pszVarNameAttr;
            if (!(pszVarNameAttr = strhGetTextAttr(strVarAttribs.c_str(), "NAME", NULL)))
            {
                ParseErrorMissingAttrib(ulTotalBlockOfs + ulVarTagOfs,
                                        "NAME",
                                        "VARPROMPT");
            }

            PSZ     pszVarTypeAttr;
            ULONG   ulType = 0;
            ULONG   ulMin = 0,
                    ulMax = 0;
            if (!(pszVarTypeAttr = strhGetTextAttr(strVarAttribs.c_str(), "TYPE", NULL)))
            {
                ParseErrorMissingAttrib(ulTotalBlockOfs + ulVarTagOfs,
                                        "TYPE",
                                        "VARPROMPT");
            }
            if (stricmp(pszVarTypeAttr, "ALPHA") == 0)
                ulType = VPT_ALPHANUM;
            else if (stricmp(pszVarTypeAttr, "PATH") == 0)
                ulType = VPT_PATH;
            else if (stricmp(pszVarTypeAttr, "FAIL") == 0)
                ulType = VPT_FAIL;
            else if (strnicmp(pszVarTypeAttr, "NUM", 3) == 0)
            {
                ulType = VPT_NUM;
                PSZ psz2;
                if (psz2 = strhExtract(pszVarTypeAttr, '(', ')', NULL))
                {
                    sscanf(psz2, "%u,%u", &ulMin, &ulMax);
                }
                if (ulMax == 0)
                {
                    ParseError(ulTotalBlockOfs + ulVarTagOfs,
                               251); // The VARPROMPT tag has an invalid TYPE syntax.
                }
            }
            else
                ParseError(ulTotalBlockOfs + ulVarTagOfs,
                           251); // The VARPROMPT tag has an invalid TYPE syntax.

            // var prompt map is in FEScriptBase,
            // so call the base class's method
            StoreVarPrompt(pszVarNameAttr,
                           strVarBlock,
                           ulType,
                           ulMin,
                           ulMax);

            free(pszVarNameAttr);
            free(pszVarTypeAttr);

            ++ulrc;
        }
        else
            // no more VARPROMPTs:
            break;

    } while (TRUE);
        // loop forever; there's a break in the loop body

    return ulrc;
}

/*
 *@@ ParsePackages:
 *      this gets called by FEOldScript::ParseHeader to set up
 *      the list of FEPackages. To do this, we analyze the
 *      PCK tags in the HEAD block of the installation profile.
 *
 *      This routine possibly calls itself recursively
 *      if GROUP tags are found.
 *
 *      It is initially called with pszBlock containing
 *      the HEAD block and will then call itself with a
 *      new block if GROUP tags are found therein.
 *
 *      This is a private method which gets called by the
 *      constructor and calls the other Parse* methods in turn.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this used to be a global function in warpin.cpp
 *@@changed V0.9.1 (2000-01-06) [umoeller]: fixed wrong line numbers in parse error msgs
 *@@changed V0.9.2 (2000-03-11) [umoeller]: added EXTERNAL support
 *@@changed V0.9.3 (2000-03-24) [umoeller]: added EXTERNAL REQUIRED support
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

ULONG FEOldScript::ParsePackages(string strBlock,
                                        // in: HEAD block to process (first call)
                                        // or GROUP block (recursive calls)
                                 FEGroupDecl *pGroup,
                                        // in: group package to append
                                        // new packages to; initially NULL
                                 ULONG ulTotalBlockOfs)
                                        // in: offset of strBlock in profile (for error messages)
{
    ULONG   ulrc = 0;
    PSZ     pszBlock = (PSZ)strBlock.c_str(),  // pointer gets modified as we search on
            pszPackage,
            pszGroup;
    ULONG   ulSearchOfs = 0;
            // ulRexxOfs = 0; // (cbo 7.3.00)
    string strRexxMacroName; // (cbo 7.3.00)

    _Locals.IncIndent(+4);

    // while we have packages to go
    while ((pszPackage = strhistr(pszBlock + ulSearchOfs, "<PCK")))
    {
        // check if we have a <GROUP> in this block
        if (    // <GROUP> found?
                (pszGroup = strhistr(pszBlock + ulSearchOfs, "<GROUP"))
                // <GROUP> before next <PCK>?
             && (pszGroup < pszPackage)
           )
        {
            // if so: extract the text between <GROUP> tags
            // and recurse!
            string      strGroupBlock,
                        strGroupAttribs;
            ULONG       ulGroupTagOfs = 0,
                        ulGroupBlockOfs = 0;

            // extract new block
            GetBlock(strBlock, // pszSearch,
                     &ulSearchOfs,      // advances search offset to after GROUP
                     "GROUP",
                     0, // no REXX
                     &strGroupBlock,
                     &strGroupAttribs,
                     &ulGroupTagOfs, &ulGroupBlockOfs);

            // get group attributes
            PSZ pszTitle;
            if (!(pszTitle = strhGetTextAttr(strGroupAttribs.c_str(), "TITLE", NULL)))
            {
                ParseErrorMissingAttrib(ulTotalBlockOfs + ulGroupBlockOfs,
                                        "TITLE",
                                        "GROUP");
            }

            BOOL fInitiallyExpanded
                = (strhFindAttribValue(strGroupAttribs.c_str(), "EXPANDED") != 0);

            ustring ustrTitle(_pCodec, pszTitle);
            _Locals.Log("found group declaration \"%s\"",
                        ustrTitle.GetBuffer());

            FEGroupDecl *pGroupNew = new FEGroupDecl(pGroup,
                                                     ustrTitle,
                                                     fInitiallyExpanded);
            free(pszTitle);

            // add this to the packages list
            if (pGroup)
                // we're in a subgroup:
                pGroup->_DeclarationsList.push_back(pGroupNew);
            else
                // we're on root level: store in script then
                _DeclarationsList.push_back(pGroupNew);

            // recurse with the group buffer
            ParsePackages(strGroupBlock,
                          pGroupNew,
                          ulTotalBlockOfs + ulGroupBlockOfs);
        }
        else
        {
            // no group, but package next:
            // extract text between <PCK> and </PCK> tags
            string      strPckBlock,
                        strPckAttribs;
            ULONG       ulPckTagOfsInBlock = 0,
                        ulPckBlockOfsInBlock = 0;
            if (!GetBlock(strBlock, // &pszSearch,
                          &ulSearchOfs,
                          "PCK",
                          GBLREXX_BLOCK | GBLREXX_ATTRIBS,
                          &strPckBlock,
                          &strPckAttribs,
                          &ulPckTagOfsInBlock, &ulPckBlockOfsInBlock))
            {
                ParseError(ulTotalBlockOfs + ulPckTagOfsInBlock,
                           252); // You have specified no information within the PCK block.
            }

            // so now we have:
            //      -- pszInfo: the "..." stuff between the <PCK>...</PCK> tags
            //      -- pszPckAttribs: the "..." stuff after the <PCK ...> tag

            // convert to utf
            ustring ustrBlock(_pCodec, strPckBlock),
                    ustrAttribs(_pCodec, strPckAttribs);

            // create package from that; the constructor
            // will parse that string data by itself
            FEOldPckDecl *pPackageNew = new FEOldPckDecl(this, // pArchive,      // archive
                                                         pGroup,
                                                         ustrBlock,
                                                         ustrAttribs,
                                                         ulTotalBlockOfs + ulPckTagOfsInBlock);

            // add this to the packages list
            if (pGroup)
                // we're in a subgroup:
                pGroup->_DeclarationsList.push_back(pPackageNew);
            else
                // we're on root level: store in script then
                _DeclarationsList.push_back(pPackageNew);

            ++ulrc;
        }
    }

    _Locals.IncIndent(-4);

    return ulrc;
}

/*
 *@@ ParsePages:
 *      this gets called by FEOldScript::Parse
 *      to set up the list containing all the FEPageInfo
 *      instances. To do this, we analyze the
 *      PAGE blocks in the BODY block of the installation
 *      profile (in our instance data).
 *
 *      Throws:
 *      -- BSCancelExcpt.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this used to be a global function in warpin.cpp
 *@@changed V0.9.1 (2000-01-06) [umoeller]: fixed wrong line numbers in parse error msgs
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 *@@changed V0.9.9 (2001-04-04) [umoeller]: added EXTRACTFROMPCK attribute
 */

ULONG FEOldScript::ParsePages()
{
    // extract the <BODY>...</BODY> part from the script
    ULONG       ulSearchOfsInProfile = 0;
    string      strBodyBlock,
                strBodyAttrs,
                strRexxMacroName; // (cbo 7.3.00)
    ULONG       ulBodyTagOfs = 0,
                ulBodyBlockOfs = 0,
                ulBlockOfs = 0; // (cbo 7.3.00)
    if (!GetBlock(_strScript,
                  &ulSearchOfsInProfile,
                  "BODY",
                  0, // no REXX
                  &strBodyBlock, &strBodyAttrs,
                  &ulBodyTagOfs, &ulBodyBlockOfs))
    {
        throw FEScriptExcpt(0, 0, _Locals, 105); // "no <BODY>"
    }

    // start with page one
    LONG    lCurrentPage = 1;

    do {
        // const char  *pcszBodyBlock = strBodyBlock.c_str();
                        // start search at the beginning; pointer gets modified
        ULONG       ulBodySearchOfs = 0;
        string      strPageBlock, // this will receive the <PAGE>...</PAGE> block
                    strPageAttrs;
        ULONG       ulPageTagOfs = 0,       // offset in entire profile (for errors)
                    ulPageBlockOfs = 0;     // offset in entire profile (for errors)

        BOOL        fExit = FALSE;

        // find the page we're looking for (lCurrentPage)
        do
        {
            // get the next <PAGE> block in <BODY> block
            if (GetBlock(strBodyBlock,
                         &ulBodySearchOfs,  // advanced after </PAGE> if found
                         "PAGE",
                         GBLREXX_BLOCK,
                         &strPageBlock,
                         &strPageAttrs,
                         &ulPageTagOfs, &ulPageBlockOfs))
            {
                // PAGE block found:
                LONG lPageFound = 0;
                // another <PAGE> block found:
                // query page index
                if (!(strhGetNumAttribValue(strPageAttrs.c_str(),
                                            "INDEX", &lPageFound)))
                {
                    ParseErrorMissingAttrib(ulBodyBlockOfs + ulPageTagOfs,
                                            "INDEX",
                                            "PAGE");
                }

                if (lPageFound == lCurrentPage)
                {
                    // it's the page we're looking for:
                    // get out of here
                    break;
                }
            }
            else
                // no other page block found:
                fExit = TRUE;
        } while (!fExit);

        if (fExit)
            // page not found: probably the
            // last page; exit the outer do loop
            break;

        // else page found: create a new WPIPAGEINFO structure
        FEPageInfo*  pPageInfoNew = new FEPageInfo();
                    // this now has a constructor

        pPageInfoNew->_lPageIndex = lCurrentPage;

        // determine page type
        PSZ pszTypeAttr;
        if (!(pszTypeAttr = strhGetTextAttr(strPageAttrs.c_str(), "TYPE", NULL)))
        {
            ParseErrorMissingAttrib(ulBodyBlockOfs + ulPageTagOfs,
                                    "TYPE",
                                    "PAGE");
        }
        else if (stricmp(pszTypeAttr, "TEXT") == 0)
        {
            pPageInfoNew->_PageMode = FEPageInfo::MODE_TEXT;
        }
        else if (stricmp(pszTypeAttr, "README") == 0)
        {
            pPageInfoNew->_PageMode = FEPageInfo::MODE_README;
        }
        else if (stricmp(pszTypeAttr, "CONTAINER") == 0)
        {
            pPageInfoNew->_PageMode = FEPageInfo::MODE_CONTAINER;
        }
        else if (stricmp(pszTypeAttr, "CONFIGURE") == 0)
        {
            pPageInfoNew->_PageMode = FEPageInfo::MODE_CONFIGURE;
        }
        else
        {
            ustring ustr(_pCodec, pszTypeAttr);
            ParseError(ulBodyBlockOfs + ulPageTagOfs,
                       253, // The PAGE tag contains an invalid TYPE attribute ("%1").,
                       &ustr, 1);
        }
        free(pszTypeAttr);

        // page intro text
        ULONG       ulPageSearchOfs = 0;
        string      strTextBody;
        if (!GetBlock(strPageBlock,
                      &ulPageSearchOfs,
                      "TEXT",
                      GBLREXX_BLOCK,
                      &strTextBody, 0, 0, &ulBlockOfs))
        {
            ustring ustr;
            ustr._printf("%u", lCurrentPage);
            ParseError(ulBodyBlockOfs + ulPageTagOfs,
                       253, // Every page must contain exactly one "<TEXT ...>" tag. Page %1 does not.
                       &ustr, 1);
        }

        pPageInfoNew->_ustrInfoText.assignCP(_pCodec, strTextBody);

        // get the buttons
        LONG        lTarget = -1;
        string      strButtonTitle,
                    strButtonAttrs;
        ULONG       ulButtonTagOfs = 0;

        ulPageSearchOfs = 0;
        if (GetBlock(strPageBlock,
                     &ulPageSearchOfs,
                     "NEXTBUTTON",
                     GBLREXX_ATTRIBS | GBLREXX_BLOCK,
                     &strButtonTitle,
                     &strButtonAttrs,
                     &ulButtonTagOfs, &ulBlockOfs))
        {
            // query button target page
            if (!(strhGetNumAttribValue(strButtonAttrs.c_str(), "TARGET", &lTarget)))
                lTarget = -1;

            pPageInfoNew->_lNextButton = lTarget;
            if (strButtonTitle())
            {
                pPageInfoNew->_ustrNextButtonTitle.assignCP(_pCodec, strButtonTitle);
            }
            else
                ParseError(ulBodyBlockOfs + ulPageBlockOfs + ulButtonTagOfs,
                           254); // The NEXTBUTTON tag has no title.
        }
        else
        {
            // no NEXTBUTTON specified:
            // use a default V0.9.2 (2000-03-11) [umoeller]

            // next page index: current page index + 1 (fits for most pages)
            pPageInfoNew->_lNextButton = pPageInfoNew->_lPageIndex + 1;

            pPageInfoNew->_ustrNextButtonTitle.erase(); // the GUI will replace this
        }

        /*
         * MODE_README:
         *
         */

        if (pPageInfoNew->_PageMode == FEPageInfo::MODE_README)
        {
            string      strReadmeBody,
                        strReadmeAttrs;
            ULONG       ulReadmeTagOfsInPageBlock = 0,
                        ulReadmeBlockOfsInPageBlock = 0;
            ulPageSearchOfs = 0;
            if (!GetBlock(strPageBlock,
                           &ulPageSearchOfs,
                          "README",
                          GBLREXX_BLOCK,
                          &strReadmeBody, &strReadmeAttrs,
                          &ulReadmeTagOfsInPageBlock, &ulReadmeBlockOfsInPageBlock))
            {
                ustring ustr;
                ustr._printf("%u", lCurrentPage);
                ParseError(ulBodyBlockOfs + ulPageTagOfs,
                           255, // You have specified that page %1 should be a
                                // README page, but the <README>...</README> block is missing.
                           &ustr, 1);
            }

            PSZ pszFormatAttr;
            if ((pszFormatAttr = strhGetTextAttr(strReadmeAttrs.c_str(), "FORMAT", NULL)))
            {
                // FORMAT specified:
                if (stricmp(pszFormatAttr, "PLAIN") == 0)
                    pPageInfoNew->_ReadmeFormat = FEPageInfo::README_PLAIN;    // default anyway
                else if (stricmp(pszFormatAttr, "FLOW") == 0)
                    pPageInfoNew->_ReadmeFormat = FEPageInfo::README_FLOW;
                else if (stricmp(pszFormatAttr, "HTML") == 0)
                    pPageInfoNew->_ReadmeFormat = FEPageInfo::README_HTML;
                else
                {
                    ustring ustr(_pCodec, pszFormatAttr);
                    ParseError(ulBodyBlockOfs + ulPageBlockOfs + ulReadmeTagOfsInPageBlock,
                               256, // The README tag contains an invalid FORMAT attribute ("%1").
                               &ustr, 1);
                }
            }

            LONG ulPckIndex = 0;
            if ((strhGetNumAttribValue(strReadmeAttrs.c_str(),
                                       "EXTRACTFROMPCK",
                                       &ulPckIndex)))
            {
                // use _strReadmeSrc as a file name:
                pPageInfoNew->_ulExtractFromPck = ulPckIndex;
            }

            pPageInfoNew->_ustrReadmeSrc.assignCP(_pCodec, strReadmeBody);
        }

        // alright, done with this page info;
        // store the page info in the list
        _PageInfoList.push_back(pPageInfoNew);
        // go for next
        ++lCurrentPage;
    } while (TRUE);
        // loop forever; there's a break in the loop

    return lCurrentPage;
}

/* ******************************************************************
 *
 *  FEOldPckDecl implementation
 *
 ********************************************************************/

/*
 *@@ FEOldPckDecl:
 *      constructor, which gets called from
 *      FEOldScript::ParsePackages for each PCK
 *      declaration found in the script.
 *
 *      Note that this now takes UTF-8 strings as
 *      input, which have been properly converted
 *      by FEOldScript::ParsePackages already.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

FEOldPckDecl::FEOldPckDecl(FEOldScript *pScript,
                           FEGroupDecl *pGroup_,            // in: "parent" group; can be NULL
                           const ustring &ustrInfo,         // in: stuff between PCK and /PCK (must be copied)
                           const ustring &ustrPckAttribs,   // in: attributes of PCK tag
                           ULONG ulPckOfs)                  // in: offset of PCK tag in script (for errors)
              : FEPckDeclBase(pGroup_,
                              tFEOldPckDecl),
                _pScript(pScript)
{
    // "INDEX="
    if (!strhGetNumAttribValue(ustrPckAttribs.GetBuffer(),
                               "INDEX",
                               (PLONG)&_ulIndexInArchive))
    {
        pScript->ParseErrorMissingAttrib(ulPckOfs,
                                         "INDEX",
                                         "PCK");
    }

    _ustrDescription = ustrInfo;

    ParsePckAttribsMain(ustrPckAttribs,
                        ulPckOfs);
}

/*
 *@@ ParsePckAttribsMain:
 *      this private method gets called by the contructor
 *      (FEOldPckDecl::FEOldPckDecl)
 *      to parse the package tag attributes. This calls
 *      the other private ParsePckAttribs* methods in
 *      turn.
 *
 *@@added V0.9.1 (2000-02-03) [umoeller]
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 *@@changed V0.9.9 (2001-02-28) [umoeller]: fixed crash with EXTERNAL REQUIRED
 *@@changed V0.9.14 (2001-08-09) [umoeller]: added auto-packageIDs for warpin self-install
 *@@changed V0.9.18 (2002-03-03) [umoeller]: added LONGFILENAMES
 *@@changed V0.9.19 (2002-07-01) [umoeller]: REQUIRED was taken as part of filename, fixed
 */

VOID FEOldPckDecl::ParsePckAttribsMain(const ustring &ustrPckAttribs,
                                       ULONG ulAttrOfs)
{
    const char *pszPckAttribs = ustrPckAttribs.GetBuffer();

    PSZ pszTitle;
    if (!(pszTitle = strhGetTextAttr(pszPckAttribs,
                                     "TITLE",
                                     NULL)))
    {
        _pScript->ParseErrorMissingAttrib(ulAttrOfs,
                                          "TITLE",
                                          "PCK");
    }
    else
    {
        _ustrTitle.assignUtf8(pszTitle);
        free(pszTitle);     // was missing V0.9.18 (2002-03-08) [umoeller]
    }

    // "TARGET="
    PSZ pszTargetPath;
    if (!(pszTargetPath = strhGetTextAttr(pszPckAttribs, "TARGET", NULL)))
    {
        _pScript->ParseErrorMissingAttrib(ulAttrOfs,
                                          "TARGET",
                                          "PCK");
    }
    else
    {
        _ustrTargetPath.assignUtf8(pszTargetPath);
        free(pszTargetPath);
    }

    BOOL fOverrideTarget = FALSE;

    switch (_ulIndexInArchive) // V0.9.14 (2001-08-09) [umoeller]
    {
        case WARPIN_SELF_PACKAGE_FOR_STUB:      // 30000, wiarchive.h
            _ustrID._printf("OS/2 Netlabs\\WarpIN\\Base\\%d\\%d\\%d",
                            VERSION_MAJOR,
                            VERSION_MINOR,
                            VERSION_REVISION);       // bldlevel.h
            fOverrideTarget = TRUE;
        break;

        case WARPIN_MORE_PACKAGE_FOR_STUB:      // 30001, wiarchive.h
            _ustrID._printf("OS/2 Netlabs\\WarpIN\\Extras\\%d\\%d\\%d",
                            VERSION_MAJOR,
                            VERSION_MINOR,
                            VERSION_REVISION);       // bldlevel.h
            fOverrideTarget = TRUE;
        break;

        default:
        {
            // "PACKAGEID="
            PSZ pszID;
            if (!(pszID = strhGetTextAttr(pszPckAttribs, "PACKAGEID", NULL)))
            {
                _pScript->ParseErrorMissingAttrib(ulAttrOfs,
                                                  "PACKAGEID",
                                                  "PCK");
            }
            else
            {
                _ustrID.assignUtf8(pszID);

                // check ID; it must contain four or five backslashes
                int cBackslashes = strhCount(pszID, '\\');
                if (    (cBackslashes != 4)
                     && (cBackslashes != 5)
                   )
                {
                    _pScript->ParseError(ulAttrOfs,
                                         237, // Package ID for package \"%1\" does not contain four or five backslashes.
                                         &_ustrID, 1);
                }

                free(pszID);
            }
        }
    } // end // V0.9.14 (2001-08-09) [umoeller]

    if (fOverrideTarget)
    {
        // TRUE if we found index 30000 or 30001 above
        // (WarpIN self-install): check if WarpIN is
        // currently registered in OS2.INI, and if so,
        // use that target path instead
        CHAR szWarpIN[CCHMAXPATH];
        if (3 < PrfQueryProfileString(HINI_USER,
                                      "WarpIN",
                                      "Path",
                                      "",
                                      szWarpIN,
                                      sizeof(szWarpIN)))
            // yes:
            _ustrTargetPath.assignCP(_pScript->_Locals._pCodecProcess,
                                     szWarpIN);
    }

    _pScript->_Locals.Log("found package declaration \"%s\"",
                          _ustrID.GetBuffer());
    _pScript->_Locals.Log("    declared target path is \"%s\"",
                          _ustrTargetPath.GetBuffer());

    // "REQUIRES="; this is optional
    ParsePckAttribsRequiresStrings(ustrPckAttribs,
                                   ulAttrOfs);

    // CONFIGSYS attributes (errors are handled by that method)
    ParsePckAttribsConfigSys(ustrPckAttribs,
                             ulAttrOfs);

    // "REGISTERCLASS=" attributes (errors are handled by that method)
    ParsePckAttribsWPSClasses(ustrPckAttribs,
                              ulAttrOfs);

    // "CREATEOBJECT=" attributes (errors are handled by that method)
    ParsePckAttribsWPSObjects(ustrPckAttribs,
                              ulAttrOfs);

    // "CLEARPROFILE=", "WRITEPROFILE=" attributes (errors are handled by that method)
    ParsePckAttribsProfiles(ustrPckAttribs,
                            ulAttrOfs);

    // "EXECUTE=" attributes (errors are handled by that method)
    ParsePckAttribsExecutes(ustrPckAttribs,
                            ulAttrOfs);

    // "KILLPROCESS=" attributes (errors are handled by that method)
    ParsePckAttribsKillProcesses(ustrPckAttribs,
                                 ulAttrOfs);

    if (strhFindAttribValue(pszPckAttribs, "NODESELECT") != 0)
        _Selected = NODESELECT;
    else if (strhFindAttribValue(pszPckAttribs, "SELECT") != 0)
        _Selected = SELECTED;
    else
        _Selected = DESELECTED;

    _ulPathType = 0;
    if (strhFindAttribValue(pszPckAttribs, "BASE") != 0)
        _ulPathType |= PATH_BASE;
    if (strhFindAttribValue(pszPckAttribs, "FIXED") == 0)
        _ulPathType |= PATH_VARIABLE;

    // V0.9.18 (2002-03-03) [umoeller]
    if (strhFindAttribValue(pszPckAttribs, "LONGFILENAMES") != 0)
        _fLongFilenames = TRUE;

    // handle external packages
    // (totally rewritten with V0.9.9 (2001-02-28) [umoeller])

    PSZ         pszExternal = NULL;
    BOOL        fRequired = FALSE;
    if ((pszExternal = strhGetTextAttr(pszPckAttribs,
                                       "EXTERNAL",
                                       NULL)))
    {
        // EXTERNAL attribute specified:
        // pszExternal now has the EXTERNAL data, which is:
        // "[REQUIRED|]archive"

        PSZ     pArchiveName = pszExternal,
                pSep = 0;
        if (pSep = strchr(pszExternal, '|'))
        {
            // separator found:
            if (strstr(pszExternal, "REQUIRED"))
            {
                // REQUIRED found:
                pArchiveName = pSep + 1;
                while (    (*pArchiveName)
                        && (*pArchiveName == ' ')
                      )
                    ++pArchiveName;

                fRequired = TRUE;
            }
            else
                pArchiveName = NULL;    // error
        }

        if (    (pArchiveName)     // V0.9.9 (2001-02-28) [umoeller]
             && (*pArchiveName)
           )
        {
            // valid archive name, apparently:
            _ustrExternalArchive.assignUtf8(pArchiveName); // pszExternal);
                                    // fixed V0.9.19 (2002-07-01) [umoeller]

            if (fRequired)
                _Type = REQEXTERNAL;
            else
                _Type = EXTERNAL;
        }
        else
            _pScript->ParseError(ulAttrOfs,
                                 238);  // Syntax error in EXTERNAL attribute.

        free(pszExternal);

    } // end if strhGetTextAttr(pszPckAttribs," EXTERNAL",
    // else: we still have the INTERNAL default from the constructor
}

/*
 *@@ ParsePckAttribsRequiresStrings:
 *      this private method gets called by
 *      FEOldPckDecl::ParsePckAttribsMain to parse
 *      the REQUIRES strings.
 *
 *      This terminates WarpIN upon errors (by calling
 *      FEArchive::ParseError), giving the user a
 *      meaningful error message.
 *
 *@@added V0.9.1 (2000-01-07) [umoeller]
 *@@changed V0.9.1 (2000-02-07) [umoeller]: fixed memory leak
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

LONG FEOldPckDecl::ParsePckAttribsRequiresStrings(const ustring &ustrPckAttribs,  // in: attributes block of PCK tag
                                                  ULONG ulAttrOfs)    // in: offset of pszPckAttribs in script (for errors)
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;
    const char  *pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszRequiresThis;
        // this can either be a decimal index
        // or a five- or six-part package ID
        if (!(pszRequiresThis = strhGetTextAttr(pszSearchIn,
                                                "REQUIRES",
                                                &ulOfs)))
            // no more attribs found:
            break;  // while

        // else: pszAttr now has the whole attribute

        if (    (atoi(pszRequiresThis) == 0)
             && (strhCount(pszRequiresThis, '\\') != 4)
             && (strhCount(pszRequiresThis, '\\') != 5)
           )
            _pScript->ParseError(ulAttrOfs,
                                 239);  // "REQUIRES" ID has invalid syntax.
        // else:
        // store in FEPckDeclBase logger;
        // this will be re-parsed after all archives
        // have been parsed to resolve indices
        // (FEArchive::ResolveRequirements)
        ustring ustr;
        ustr.assignUtf8(pszRequiresThis);
        _logRequiresStrings.Append(ustr);
        ++lrc;

        free(pszRequiresThis);

        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    return -1;
}

/*
 *@@ ParsePckAttribsConfigSys:
 *      this private method gets called by
 *      FEOldPckDecl::ParsePckAttribsMain to parse
 *      the CONFIGSYS attributes of a given PCK tag
 *      and create a list of BSConfigSys instances (bs_config.h)
 *      in pConfigSysList.
 *
 *      Returns the number of BSConfigSys instances created
 *      (0 if none found).
 *
 *      This terminates WarpIN upon errors (by calling
 *      FEArchive::ParseError), giving the user a
 *      meaningful error message.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this used to be a global function in warpin.cpp
 *@@changed V0.9.1 (2000-02-07) [umoeller]: fixed memory leak
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

LONG FEOldPckDecl::ParsePckAttribsConfigSys(const ustring &ustrPckAttribs,  // in: attributes block of PCK tag
                                            ULONG ulAttrOfs)    // in: offset of pszPckAttribs in script (for errors)
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;
    const char  *pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = NULL;

        if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                            "CONFIGSYS",
                                            &ulOfs)))
            // no more attribs found:
            break;  // while

        // else: pszAttr now has the whole attribute

        // we now create an instance of BSCfgSysManip (cfgsys.h),
        // whose constructor is capable of parsing this attribute

        try
        {
            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            BSCfgSysManip* pBSCfgSysManip = new BSCfgSysManip(ustr);
                            // this may throw a BSConfigExcpt, caught below
            _listConfigObjects.push_back(pBSCfgSysManip);

            ++lrc;

            _listVariables.push_back(&_pScript->_Locals._varConfigSys);
            // _ulConfigData |= PCK_CONFIGSYS;

            // append this attribute string to the log string
            _logCfgSysAttrs.Append(ustr);
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSCfgSysManip constructor:
            ustring strError;

            switch (X._iErrorCode)
            {
                case CFGEXCPT_DOUBLEREPLACEMODE:
                    strError.assignUtf8("Replace mode specified twice");
                break;

                case CFGEXCPT_INVALIDVERTICAL:
                    strError.assignUtf8("Invalid modifier combination; vertical modifier not allowed");
                break;

                case CFGEXCPT_INVALIDSEARCH:
                    strError.assignUtf8("Invalid search string specified");
                break;

                case CFGEXCPT_NOTIMPLEMENTED:
                    strError.assignUtf8("Keyword not implemented yet");
                break;
            }

            ustring aStrings[2] =
                {
                    strError,
                    X._ustrDescription
                };
            _pScript->ParseError(ulAttrOfs,
                                 240,       // Error parsing CONFIGSYS attribute of PCK tag:
                                 aStrings, 2);
                // this exits
        }

        free(pszAttrThis);

        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    return lrc;
}

/*
 *@@ ParsePckAttribsWPSClasses:
 *      this private method gets called by
 *      FEOldPckDecl::ParsePckAttribsMain to parse
 *      the REGISTERCLASS and REPLACECLASS attributes
 *      of a given PCK tag and create two lists of
 *      BSRegisterClass and BSReplaceClass instances (bs_config.h)
 *      in listRegisterClassAttrs/listReplaceClass.
 *
 *      Returns the total number of instances created (0 if none found).
 *
 *      This terminates WarpIN upon errors (by calling
 *      FEArchive::ParseError), giving the user a
 *      meaningful error message.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this used to be a global function in warpin.cpp
 *@@changed V0.9.1 (2000-02-07) [umoeller]: fixed memory leak
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

LONG FEOldPckDecl::ParsePckAttribsWPSClasses(const ustring &ustrPckAttribs,  // in: attributes block of PCK tag
                                             ULONG ulAttrOfs)    // in: offset of pszPckAttribs in script (for errors)
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;

    // step 1: get the REGISTERCLASS attributes
    const char  *pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = NULL;

        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "REGISTERCLASS",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "classname|classmodule"
            // (without quotes)

            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            BSRegisterClass *pRegisterClass = new BSRegisterClass(ustr);
                            // this may throw a BSConfigExcpt, caught below
            _listConfigObjects.push_back(pRegisterClass);
            // _listRegisterClasses
            ++lrc;

            _listVariables.push_back(&_pScript->_Locals._varWPSClasses);
            // _ulConfigData |= PCK_WPSCLASSES;

            // append this attribute string to the log string
            _logRegisterClassAttrs.Append(ustr);

            // advance search pointer for next search
            pszSearchIn += ulOfs;
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSCreateWPSObject constructor
            _pScript->ParseError(ulAttrOfs,
                                 241,       // Error parsing REGISTERCLASS attribute of PCK tag:
                                 &X._ustrDescription, 1);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);
    }

    // step 2: get the REPLACECLASS attributes

    // reset the search pointer
    pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = NULL;

        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "REPLACECLASS",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "oldclassname|newclassname"
            // (without quotes)

            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            BSReplaceClass *pReplaceClass = new BSReplaceClass(ustr);
                            // this may throw a BSConfigExcpt, caught below
            // _listReplaceClasses
            _listConfigObjects.push_back(pReplaceClass);
            ++lrc;

            _listVariables.push_back(&_pScript->_Locals._varWPSClasses);
            // _ulConfigData |= PCK_WPSCLASSES;

            // append this attribute string to the log string
            _logReplaceClassAttrs.Append(ustr);

            // advance search pointer for next search
            pszSearchIn += ulOfs;
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSCreateWPSObject constructor
            _pScript->ParseError(ulAttrOfs,
                                 242, // Error parsing REPLACECLASS attribute of PCK tag:
                                 &X._ustrDescription, 1);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);
    }

    return lrc;
}

/*
 *@@ ParsePckAttribsWPSObjects:
 *      this private method gets called by
 *      FEOldPckDecl::ParsePckAttribsMain to parse
 *      the CREATEOBJECT attributes of a given PCK tag
 *      and create a list of BSCreateWPSObject instances (bs_config.h)
 *      in listWPSObjectAttrs.
 *
 *      Returns the number of BSCreateWPSObject instances created (0 if none found).
 *
 *      This terminates WarpIN upon errors (by calling
 *      FEArchive::ParseError), giving the user a
 *      meaningful error message.
 *
 *@@changed V0.9.0 (99-10-31) [umoeller]: this used to be a global function in warpin.cpp
 *@@changed V0.9.1 (2000-02-07) [umoeller]: fixed memory leak
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

LONG FEOldPckDecl::ParsePckAttribsWPSObjects(const ustring &ustrPckAttribs,  // in: attributes block of PCK tag
                                             ULONG ulAttrOfs)    // in: offset of pszPckAttribs in script (for errors)
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;
    const char  *pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = NULL;

        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "CREATEOBJECT",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "[REPLACE] classname|title|location[|config]"
            // (without quotes)

            ustring ustr;
            ustr.assignUtf8(pszAttrThis);        // fixed, use codepage

            BSCreateWPSObject *pCreateWPSObject = new BSCreateWPSObject(ustr);
                            // this may throw a BSConfigExcpt, caught below
            // _listWPSObjects
            _listConfigObjects.push_back(pCreateWPSObject);
            ++lrc;

            _listVariables.push_back(&_pScript->_Locals._varWPSObjects);
            // _ulConfigData |= PCK_WPSOBJECTS;

            // append this attribute string to the log string
            _logWPSObjectAttrs.Append(ustr);
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSCreateWPSObject constructor (ParseCreateObject)
            ustring strError;

            switch (X._iErrorCode)
            {
                case WPOEXCPT_NOLOCATION:
                    strError.assignUtf8(
                            "No location specified");
                break;

                case WPOEXCPT_NOTITLE:
                    strError.assignUtf8(
                            "No title specified");
                break;

                case WPOEXCPT_NOCLASS:
                    strError.assignUtf8(
                            "No class specified");
                break;

                case WPOEXCPT_INVALIDLOCATIONSTRING:
                    // V0.9.15 (2001-08-26) [umoeller]
                    strError.assignUtf8(
                            "Location string is invalid, must be in angle brackets");
                break;
            }

            ustring aStrings[2] =
                {
                    strError,
                    X._ustrDescription
                };
            _pScript->ParseError(ulAttrOfs,
                                 243, // Error parsing CREATEOBJECT attribute of PCK tag:
                                 aStrings, 2);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);

        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    return lrc;
}

/*
 *@@ ParsePckAttribsProfiles:
 *      this private method gets called by
 *      FEOldPckDecl::ParsePckAttribsMain
 *      to parse the CLEARPROFILE and WRITEPROFILE strings.
 *
 *      Returns the no. of matching attributes found.
 *
 *      This terminates WarpIN upon errors (by calling
 *      FEArchive::ParseError), giving the user a
 *      meaningful error message.
 *
 *@@added V0.9.1 (2000-02-07) [umoeller]
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

LONG FEOldPckDecl::ParsePckAttribsProfiles(const ustring &ustrPckAttribs,
                                           ULONG ulAttrOfs)
{
    LONG        lrc = 0;
    const char  *pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        ULONG   ulOfs = 0;
        PSZ     pszAttrThis = NULL;
        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "CLEARPROFILE",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute
            ++lrc;

            // we just store this in the logger, it's only
            // for de-installation
            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            _logClearPrfAttrs.Append(ustr);

            _listVariables.push_back(&_pScript->_Locals._varWriteProfiles);
            // _ulConfigData |= PCK_CLEARPROFILES;
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSCreateWPSObject constructor (ParseCreateObject)
            /* ustring strError;

            switch (X._iErrorCode)
            {
                case PRFEXCPT_SYNTAX:
                    strError._printf(
                            "Invalid syntax (%s)",
                            X._strDescription.c_str());
                    break;
            } */

            ustring astr[2];
            astr[0].assignUtf8("CLEARPROFILE");
            astr[1] = X._ustrDescription;
            _pScript->ParseError(ulAttrOfs,
                                 247, // Error parsing %1 attribute of PCK tag: Invalid syntax (%2)
                                 astr, 2);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);

        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    // reset search buffer
    pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        ULONG   ulOfs = 0;
        PSZ     pszAttrThis = NULL;
        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "WRITEPROFILE",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute
            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            BSWriteProfile *pWriteProfile = new BSWriteProfile(ustr);
                            // this may throw a BSConfigExcpt, caught below
            // _listWriteProfiles
            _listConfigObjects.push_back(pWriteProfile);
            ++lrc;

            // store this in the logger for de-installation
            _logWritePrfAttrs.Append(ustr);

            _listVariables.push_back(&_pScript->_Locals._varClearProfiles);
            // _ulConfigData |= PCK_WRITEPROFILES;
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSCreateWPSObject constructor (ParseCreateObject)
            /* CHAR        szError[1000] = "unknown error";

            switch (X._iErrorCode)
            {
                case KILLEXCPT_SYNTAX:
                    sprintf(szError,
                            "Invalid syntax (%s)",
                            X._strDescription.c_str());
                    break;
            }
            */

            ustring astr[2];
            astr[0].assignUtf8("WRITEPROFILE");
            astr[1] = X._ustrDescription;
            _pScript->ParseError(ulAttrOfs,
                                 247, // Error parsing %1 attribute of PCK tag: Invalid syntax (%2)
                                 astr, 2);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);

        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    return lrc;
}

/*
 *@@ ParsePckAttribsExecutes:
 *      this private method gets called by
 *      FEOldPckDecl::ParsePckAttribsMain to parse
 *      the EXECUTE and DEEXECUTE strings.
 *
 *      Returns the no. of EXECUTE/DEEXECUTE attributes found.
 *
 *      This terminates WarpIN upon errors (by calling
 *      FEArchive::ParseError), giving the user a
 *      meaningful error message.
 *
 *@@added V0.9.1 (2000-02-03) [umoeller]
 *@@changed V0.9.1 (2000-02-07) [umoeller]: fixed memory leak
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 *@@changed V0.9.9 (2001-03-27) [umoeller]: added DEEXECUTE support
 */

LONG FEOldPckDecl::ParsePckAttribsExecutes(const ustring &ustrPckAttribs,
                                           ULONG ulAttrOfs)
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;
    const char  *pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = NULL;
        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "EXECUTE",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "[context | ]execfile [params]"
            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            BSExecute *pExecute = new BSExecute(ustr);
                            // this may throw a BSConfigExcpt, caught below
            _listConfigObjects.push_back(pExecute);
            ++lrc;

            // append this attribute string to the log string
            _logExecuteAttrs.Append(ustr);

            _listVariables.push_back(&_pScript->_Locals._varExecutes);
            // _ulConfigData |= PCK_EXECUTES;

            // copy the execute's config stuff to the package's one
            if (pExecute->_ulExecType == CFGT_CFGSYS)
                // _ulConfigData |= PCK_CONFIGSYS;
                _listVariables.push_back(&_pScript->_Locals._varConfigSys);
            if (pExecute->_ulExecType == CFGT_REGISTERCLASS)
                // _ulConfigData |= PCK_WPSCLASSES;
                _listVariables.push_back(&_pScript->_Locals._varWPSClasses);
            if (pExecute->_ulExecType == CFGT_CREATEOBJECT)
                // _ulConfigData |= PCK_WPSOBJECTS;
                _listVariables.push_back(&_pScript->_Locals._varWPSObjects);
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSExecute constructor
            /* CHAR        szError[1000] = "unknown error";

            switch (X._iErrorCode)
            {
                case EXEEXCPT_SYNTAX:
                    sprintf(szError,
                            "Invalid syntax (%s)",
                            X._strDescription.c_str());
                    break;
            }
            */

            ustring astr[2];
            astr[0].assignUtf8("EXECUTE");
            astr[1] = X._ustrDescription;
            _pScript->ParseError(ulAttrOfs,
                                 247, // Error parsing %1 attribute of PCK tag: Invalid syntax (%2)
                                 astr, 2);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);
        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    // restart for DEEXECUTE
    ulOfs = 0;
    pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = NULL;
        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "DEEXECUTE",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "execfile [params]"
            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            BSDeExecute *pExecute = new BSDeExecute(ustr);
                            // this may throw a BSConfigExcpt, caught below
            _listConfigObjects.push_back(pExecute);
            ++lrc;

            // append this attribute string to the log string
            _logDeExecuteAttrs.Append(ustr);

            // _ulConfigData |= PCK_DEEXECUTES;
            _listVariables.push_back(&_pScript->_Locals._varDeExecutes);
        }
        catch (BSConfigExcpt& X)
        {
            // thrown by BSCreateWPSObject constructor (ParseCreateObject)
            /* CHAR        szError[1000] = "unknown error";

            switch (X._iErrorCode)
            {
                case EXEEXCPT_SYNTAX:
                    sprintf(szError,
                            "Invalid syntax (%s)",
                            X._strDescription.c_str());
                    break;
            }
            */

            ustring astr[2];
            astr[0].assignUtf8("DEEXECUTE");
            astr[1] = X._ustrDescription;
            _pScript->ParseError(ulAttrOfs,
                                 247, // Error parsing %1 attribute of PCK tag: Invalid syntax (%2)
                                 astr, 2);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);

        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    return lrc;
}

/*
 *@@ ParsePckAttribsKillProcesses:
 *      this private method gets called by
 *      FEOldPckDecl::ParsePckAttribsMain to parse
 *      the KILLPROCESS strings.
 *
 *      Returns the no. of KILLPROCESS attributes found.
 *
 *      This terminates WarpIN upon errors (by calling
 *      FEArchive::ParseError), giving the user a
 *      meaningful error message.
 *
 *@@added V0.9.1 (2000-02-12) [umoeller]
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 *@@changed V0.9.15 (2001-08-26) [umoeller]: fixed garbage error messages
 */

LONG FEOldPckDecl::ParsePckAttribsKillProcesses(const ustring &ustrPckAttribs,
                                                ULONG ulAttrOfs)
{
    LONG        lrc = 0;
    ULONG       ulOfs = 0;
    const char  *pszSearchIn = ustrPckAttribs.GetBuffer();

    while (lrc != -1)   // no error?
    {
        PSZ     pszAttrThis = NULL;

        try
        {
            if (!(pszAttrThis = strhGetTextAttr(pszSearchIn,
                                                "KILLPROCESS",
                                                &ulOfs)))
                // no more attribs found:
                break;

            // else: pszAttr now has the whole attribute, which is like this:
            // "[REPLACE] classname|title|location[|config]"
            // (without quotes)

            ustring ustr;
            ustr.assignUtf8(pszAttrThis);

            BSKillProcess *pKillProcess = new BSKillProcess(ustr);
                            // this may throw a BSConfigExcpt, caught below
            _listConfigObjects.push_back(pKillProcess);
            ++lrc;

            // _ulConfigData |= PCK_KILLPROCESS;
            _listVariables.push_back(&_pScript->_Locals._varKillProcesses);

            // append this attribute string to the log string
            _logKillProcessAttrs.Append(ustr);
        }
        catch (BSConfigExcpt& X)
        {
            /* CHAR        szError[1000] = "unknown error";

            switch (X._iErrorCode)
            {
                // fixed error messages V0.9.15 (2001-08-26) [umoeller]
                case KILLEXCPT_SYNTAX:
                    sprintf(szError,
                            "Invalid KILLPROCESS syntax (\"%s\")",
                            X._strDescription.c_str());
                break;
            }
            _pScript->ParseError(ulAttrOfs, "Error parsing CREATEOBJECT attribute of PCK tag:\n%s",
                                 szError); */

            ustring astr[2];
            astr[0].assignUtf8("KILLPROCESS");
            astr[1] = X._ustrDescription;
            _pScript->ParseError(ulAttrOfs,
                                 247, // Error parsing %1 attribute of PCK tag: Invalid syntax (%2)
                                 astr, 2);
                // this exits
        }

        if (pszAttrThis)
            free(pszAttrThis);

        // advance search pointer for next search
        pszSearchIn += ulOfs;
    }

    return lrc;
}


