
/*
 *@@sourcefile fe_script.cpp:
 *      FEScriptBase and subclasses implementation.
 *
 *      This has been extracted from FEArchive and FEPackageBase
 *      with V0.9.9 and moved to this file to allow for a
 *      completely different implementation of script parsing.
 *      See FEArchive for an overview how scripts are now
 *      implemented.
 *
 *@@header "engine\fe_script.h"
 *@@added V0.9.9 (2001-02-19) [umoeller]
 */

/*
 *      This file Copyright (C) 1999-2001 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOSERRORS
#include <os2.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "setup.h"

#include "bldlevel.h"           // needed for version checks

#include "helpers\configsys.h"
#include "helpers\dosh.h"
#include "helpers\nls.h"
#include "helpers\stringh.h"
#include "helpers\tree.h"
#include "helpers\xstring.h"

#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_map.h"
#include "base\bs_errors.h"

#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

#include "engine\fe_base.h"
#include "engine\fe_script.h"
#include "engine\fe_script_old.h"

// #include "engine\fe_rexx.h"

/* ******************************************************************
 *
 *  Globals
 *
 ********************************************************************/

DEFINE_CLASS(FEPageInfo, BSRoot);
DEFINE_CLASS(FEVarPrompt, BSRoot);
DEFINE_CLASS(FEPckDeclBase, BSRoot);
DEFINE_CLASS(FEDBPckDecl, FEPckDeclBase);
DEFINE_CLASS(FEGroupDecl, FEPckDeclBase);

DEFINE_CLASS(FEScriptBase, BSRoot);

/*
 *@@ VarPromptEntry:
 *
 *@@added V0.9.14 (2001-07-07) [umoeller]
 */

struct VarPromptEntry
{
    TREE        _Tree;

    FEVarPrompt *_pPrompt;

    VarPromptEntry(FEVarPrompt *pPrompt)
    {
        _pPrompt = pPrompt;
        _Tree.ulKey = (ULONG)pPrompt->_strVarName.c_str();
    }
};

int TREEENTRY CompareIStrings(ULONG ul1, ULONG ul2);        // in fe_rexx.cpp

static BSMap <VarPromptEntry*, CompareIStrings>
                G_VarPromptsMap;

/* ******************************************************************
 *
 *  Helper functions
 *
 ********************************************************************/

/* ******************************************************************
 *
 *  FEScriptExcpt implementation
 *
 ********************************************************************/

/*
 *@@ FEScriptExcpt:
 *      first constructor with a plain string message.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

/* FEScriptExcpt::FEScriptExcpt(const BSString &str,
                             unsigned long ulLine,      // in: line, defaults to 0
                             unsigned long ulColumn)    // in: column, defaults to 0
{
    _strDescription.assign(str);
    _ulLine = ulLine;
    _ulColumn = ulColumn;
}

*/

/*
 *@@ FEScriptExcpt:
 *      second constructor for retrieving a TMF message.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

FEScriptExcpt::FEScriptExcpt(ULONG ulLine,
                             ULONG ulColumn,
                             FELocals &Locals,
                             int iMessage,
                             ustring *paStrings,
                             int iCount)
{
    ustring aStrings[2];
    if (ulLine && ulColumn)
    {
        aStrings[0]._itoa10(ulLine, Locals._cThousands);
        aStrings[1]._itoa10(ulColumn, Locals._cThousands);
        Locals.GetMessage(_ustrDescription,
                          263, // Script error in line %1, column %1:
                          aStrings,
                          2);
    }
    else if (ulLine)
    {
        aStrings[0]._itoa10(ulLine, Locals._cThousands);
        Locals.GetMessage(_ustrDescription,
                          264, // Script error in line %1:
                          aStrings,
                          1);
    }
    else
    {
        Locals.GetMessage(_ustrDescription,
                          265); // Script error:
    }

    ustring ustrMsg2;
    Locals.GetMessage(ustrMsg2,
                      iMessage,
                      paStrings, iCount);
    _ustrDescription.append(ustrMsg2);
}

/* ******************************************************************
 *
 *  FEVarPrompt implementation
 *
 ********************************************************************/

/*
 *@@ Validate:
 *      returns TRUE only if the specified value is
 *      valid.
 *
 *@@added V0.9.2 (2000-03-11) [umoeller]
 *@@changed V0.9.3 (2000-04-28) [umoeller]: didn't check for pcszValue == NULL; fixed
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved this here from fe_archive.cpp
 */

bool FEVarPrompt::Validate(const char *pcszValue)
     const
{
    bool brc = FALSE;

    if (pcszValue)
        switch (_ulType)
        {
            case VPT_NUM:
            break;

            case VPT_ALPHANUM:
                brc = TRUE;
            break;

            case VPT_PATH:
                brc = (doshIsValidFileName(pcszValue, TRUE) == NO_ERROR);
            break;

            case VPT_FAIL:
            break;
        }

    return brc;
}

int operator==(const FEVarPrompt &v1, const FEVarPrompt &v2)
{
    return (v1._strVarName == v2._strVarName);
}

int operator<(const FEVarPrompt &v1, const FEVarPrompt &v2)
{
    return (v1._strVarName < v2._strVarName);
}

/* ******************************************************************
 *
 *  FEInstallVar implementation
 *
 ********************************************************************/

DEFINE_CLASS(FEInstallVar, BSRoot);

/* ******************************************************************
 *
 *  FEPckDeclBase implementation
 *
 ********************************************************************/

/*
 *@@ FEPckDeclBase:
 *      constructor for the base class.
 *
 *      This constructor is protected to make sure that
 *      instances of FEPckDeclBase can only be created
 *      as instances of one of the subclasses.
 *
 *@@added V0.9.9 (2001-02-28) [umoeller]
 */

FEPckDeclBase::FEPckDeclBase(FEGroupDecl *pGroup,   // in: the group this belongs to or NULL
                             BSClassID &Class)
               : BSRoot(Class),
                 _listConfigObjects(STORE),          // switch lists to store mode
                 _listVariables(SHADOW)
{
    _pGroup = pGroup;
    _Type = INTERNAL;

    _ulIndexInArchive = 0;

    // _External = INTERNAL;

    _Selected = UNKNOWN;

    _ulPathType = 0;

    _fLongFilenames = FALSE; // V0.9.18 (2002-03-03) [umoeller]

    // _ulConfigData = 0;
}

/* ******************************************************************
 *
 *  FEScriptBase implementation
 *
 ********************************************************************/

/*
 *@@ FEScriptBase:
 *
 *@@added V0.9.14 (2001-07-07) [umoeller]
 */

FEScriptBase::FEScriptBase(FELocals &Locals,
                           FERexx &Rexx,
                           BSClassID &Class)
    : BSRoot(Class),
      _DeclarationsList(STORE),
      _PageInfoList(STORE),
      _Locals(Locals),
      _Rexx(Rexx)
{
    _ulCodepage = 0;        // Unicode per default, but this is
                            // overwritten in FEOldScript
}

/*
 *@@ StoreVarPrompt:
 *      static function for storing a new varprompt.
 *
 *@@added V0.9.14 (2001-07-07) [umoeller]
 */

void FEScriptBase::StoreVarPrompt(const char *pcszVarNameAttr,
                                  string &strVarBlock,
                                  ULONG ulType,
                                  ULONG ulMin,
                                  ULONG ulMax)
{
    FEVarPrompt *pVarPrompt = new FEVarPrompt(pcszVarNameAttr);
    pVarPrompt->_strDescription = strVarBlock;
    pVarPrompt->_ulType = ulType;
    pVarPrompt->_ulMin = ulMin;
    pVarPrompt->_ulMax = ulMax;

    // append to static string map
    VarPromptEntry *pEntry = new VarPromptEntry(pVarPrompt);
    treeInsert(&G_VarPromptsMap._TreeRoot,
               &G_VarPromptsMap._cEntries,
               &pEntry->_Tree,
               CompareIStrings);
    // G_VarPromptsMap.insert(pEntry);

    // [BSString(pszVarNameAttr)] = VarPrompt;
}

/*
 *@@ FindVarPrompt:
 *      static function to find a varprompt from
 *      the given variable name. Returns NULL if
 *      not found.
 *
 *@@added V0.9.14 (2001-07-07) [umoeller]
 *@@changed V0.9.16 (2001-12-08) [umoeller]: fixed bad pointer... the tree contains a VarPromptEntry
 *@@changed V0.9.19 (2002-04-14) [umoeller]: fixed crash if var prompt was not found
 */

FEVarPrompt* FEScriptBase::FindVarPrompt(const char *pcszVarNameAttr)
{
    VarPromptEntry *p;

    // added check here, this crashed V0.9.19 (2002-04-14) [umoeller]
    if (p = (VarPromptEntry*)treeFind(G_VarPromptsMap._TreeRoot,
                                      (ULONG)pcszVarNameAttr,
                                      CompareIStrings))
        return p->_pPrompt;

    return NULL;
}

/*
 *@@ Validate:
 *      validates a few things in the script after
 *      parsing.
 *
 *      Note that this does _not_ get called automatically
 *      by FEScriptBase::CreateScript, in order not to
 *      break any old archives that might have specified
 *      evil things in the script. However, this is now
 *      used by wic -t to validate from the command line.
 *
 *@@added V0.9.15 (2001-08-26) [umoeller]
 */

VOID FEScriptBase::Validate()
{
    if (_PageInfoList.empty())
        throw FEScriptExcpt(0, 0, _Locals, 215);     // pages

    list<FEPageInfo*>::iterator PageStart = _PageInfoList.begin(),
                                PageEnd = _PageInfoList.end();
    BOOL    fLastFound = FALSE;
    for (;
         PageStart != PageEnd;
         ++PageStart)
    {
        FEPageInfo *pThis = *PageStart;
        if (!pThis->_lNextButton)
        {
            if (fLastFound)
                throw FEScriptExcpt(0, 0, _Locals, 216);     // more than one nextbutton=0
            else
                fLastFound = TRUE;
        }
        else
        {
            // find the page with that index
            BOOL fFoundThis = FALSE;
            list<FEPageInfo*>::iterator PageStart2 = _PageInfoList.begin(),
                                        PageEnd2 = _PageInfoList.end();
            for (;
                 PageStart2 != PageEnd2;
                 ++PageStart2)
            {
                FEPageInfo *pThis2 = *PageStart2;
                if (pThis2->_lPageIndex == pThis->_lNextButton)
                {
                    fFoundThis = TRUE;
                    break;
                }
            }

            if (!fFoundThis)
            {
                ustring str[2];
                str[0]._itoa10(pThis->_lPageIndex, _Locals._cThousands);
                str[1]._itoa10(pThis->_lNextButton, _Locals._cThousands);
                throw FEScriptExcpt(0, 0, _Locals,
                                    217,
                                    str,
                                    2);
                        // page %1 specifies nextbutton %2, but that page does not exist
            }
        }
    }
}

/*
 *@@ CreateScript:
 *      this static method is the magic entry point into
 *      script parsing. This creates a script instance
 *      and parses the script.
 *
 *      Once FEArchive has retrieved the script from the
 *      archive, it calls this function with the script
 *      as the only parameter.
 *
 *      This function then does a few quick checks on
 *      the script to determine its format.
 *
 *      --  If the script starts with "<?xml ", it is
 *          assumed to be a new-style XML script. In that
 *          case, an instance of BSXmlScript is created
 *          and returned.
 *
 *      --  Otherwise, it is assumed to be an old-style
 *          script. In that case, an instance of FEOldScript
 *          is created and returned.
 *
 *      Before returning, this function calls the virtual
 *      "Parse" method on the new script instance, which
 *      does the actual work of filling all the fields.
 *      As a result, the caller doesn't even need to know
 *      any more what format the script is in. It receives
 *      an instance of FEScriptBase (actually, of one of
 *      the subclasses) with all the members nicely set...
 *      how the parsing works doesn't matter.
 *
 *      FELocals::ConfirmRexxAllowed gets called if any
 *      REXX code is encountered in the script.
 *      That function must then return TRUE if parsing should
 *      continue or FALSE if not (paranoid user selected "no"
 *      then).
 *
 *      Starting with V0.9.18, codepage management is now
 *      handled by FELocals, which converts the script to
 *      UTF-8 automatically.
 *
 *@@added V0.9.9 (2001-02-19) [umoeller]
 *@@changed V0.9.15 (2001-08-26) [umoeller]: added page checking here
 *@@changed V0.9.20 (2002-07-03) [umoeller]: removed rexx allowed callback
 */

FEScriptBase* FEScriptBase::CreateScript(const char *pcszScript, // in: entire install script
                                         FELocals &Locals,
                                         FERexx &Rexx)
{
    FEScriptBase *pScript = NULL;

    // check the script string for the magic <?xml string; if we
    // find it, we assume it's a new-style string, otherwise
    // we invoke the old parser
    PCSZ p;
    if (p = strchr(pcszScript, '<'))
    {
        if (!strcmp(p, "<?xml "))
        {
            // new script:
            Locals.Log("Parsing XML script...");
        }
        else
        {
            // old script:
            Locals.Log("Parsing old-style script...");
            pScript = new FEOldScript(Locals, Rexx);
        }

        pScript->Parse(pcszScript);
    }
    else
        // we don't even have an angle bracket in the script:
        // that's neither old nor new style...
        throw FEScriptExcpt(0, 0, Locals, 201);

    return pScript;
}


