
/*
 *@@sourcefile fe_rexx.cpp:
 *      REXX support for WarpIN scripts.
 *
 *      This was added with V0.9.2 by Cornelis Bockemhl.
 *
 *      WarpIN REXX support has been redesigned (once
 *      again) with V0.9.20. This file has all the REXX
 *      support now, and we call into FELocals for GUI
 *      support (presently for the message box only).
 *
 *@@header "engine\fe_rexx.h"
 *@@added V0.9.2 (2000-03-10) [cbo]
 */

/*
 *      Copyright (C) 2000 Cornelis Bockemhl.
 *      Copyright (C) 2000-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

// REXX support
#define INCL_REXXSAA
#include <os2.h>
#ifdef __IBMCPP__
    #include <rexxsaa.h>    // EMX has this in os2.h already
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>

// include all these headers because we need calbacks.h
#include "setup.h"

// include's from helpers
#include "helpers\standards.h"
#include "helpers\stringh.h"
#include "helpers\tree.h"
#include "helpers\xstring.h"

// base includes
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_errors.h"
#include "base\bs_map.h"

#include "wiarchive\wiarchive.h"

#include "engine\fe_base.h"
#include "engine\fe_rexx.h"

/* ******************************************************************
 *
 *  Private declarations
 *
 ********************************************************************/

////////////////////////////////////////////////////////////////////
// Functions available to REXX scripts
//
// Remark: These C style functions are only wrappers for the WarpIN
// static member functions with the same name: I somehow could not
// get static class member functions declared as APIENTRY which is
// necessary for declaring functions for REXX...
extern "C" {

ULONG APIENTRY WirexxPutEnv(PCSZ name, ULONG argc, RXSTRING argv[],
                            PCSZ queuename, PRXSTRING retstr);

ULONG APIENTRY WirexxGetEnv(PCSZ name, ULONG argc, RXSTRING argv[],
                            PCSZ queuename, PRXSTRING retstr);

ULONG APIENTRY WirexxShowMessage(PCSZ name, ULONG argc, RXSTRING argv[],
                                 PCSZ queuename, PRXSTRING retstr);

} // end extern "C"

int TREEENTRY CompareIStrings(ULONG ul1, ULONG ul2)
{
    return stricmp((const char*)ul1,
                   (const char*)ul2);
}

/* ******************************************************************
 *
 *  Global variables
 *
 *******************************************************************/

static BSMap<StringMapEntry*, CompareIStrings>
                G_EnvStrings;

DEFINE_CLASS(FERexx, BSRoot);

static BSMap<StringMapEntry*, CompareIStrings>
                G_RexxCodeStore;

FELocals *G_pLocals = NULL;
        // pointer to locals so that we can display
        // a message box from the APIENTRY C functions;
        // set by constructor
        // V0.9.20 (2002-07-03) [umoeller]

/*
 *@@ G_aRexxFunctions:
 *      global array of all WarpIN functions which
 *      are made available to REXX code. These are
 *      registered with the REXX interpreter in the
 *      FERexx::FERexx constructor and deregistered
 *      in the destructor.
 *
 *      To add a new WarpIN REXX function, just add
 *      the two fields to this array. The size of the
 *      array is automatically taken into account in
 *      the constructor/destructor, so there's nothing
 *      to adjust.
 *
 *@@added V0.9.2 (2000-03-10) [umoeller]
 */

WIREXXFUNCTION  G_aRexxFunctions[]
        = {
            "WirexxPutEnv", ::WirexxPutEnv,
            "WirexxGetEnv", ::WirexxGetEnv,
            "WirexxShowMessage", ::WirexxShowMessage
          };

// WarpIN REXX support; this is implemented thru a
// global object. We use new() to create the FERexx
// instance, because if we did this statically and
// errors occured, WarpIN would crash even before
// main() is reached, which makes debugging impossible.
// So instead, this is created in main() now.
// FERexx          *G_pFERexx = 0;
        // moved to engine V0.9.20 (2002-07-03) [umoeller]

/* ******************************************************************
 *
 *  Helper funcs
 *
 ********************************************************************/

/*
 *@@ SetRexxReturn:
 *      helper function for REXX functions to correctly return a REXX
 *      return string (for implementation details see docs extract
 *      at the end of this file).
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 */

void SetRexxReturn(char *retstr,
                   PRXSTRING rxstr)
{
    ULONG rc;

    if (255 < strlen(retstr))
    {
        // enlarge the REXX string storage
        rc = DosAllocMem((PPVOID)rxstr->strptr,
                         strlen(retstr),
                         PAG_READ | PAG_WRITE | PAG_COMMIT);

        // if an error occurs we return a string indicating that
        if (0 != rc)
        {
            strcpy(rxstr->strptr, "ERROR allocating memory for REXX return");
            rxstr->strlength = strlen(rxstr->strptr);
            return;
        }
    }

    // copy the string to it's place
    strcpy(rxstr->strptr, retstr);
    rxstr->strlength = strlen(retstr);
}

/* ******************************************************************
 *
 *  Exported REXX functions
 *
 *******************************************************************/

//////////////////////////////////////////////////////////////////
// Implement functions that can be called by REXX scripts
//
// Each function has to be defined twice:
// - once as a static member function of FERexx. This allows e.g.
//   for storing static data there (like the EnvStrings map).
// - then as a C style stand-alone function as a wrapper: This
//   is needed because a member function cannot be exported with
//   APIENTRY linkage.
// The C function is registered in the FERexx constructor using
// the G_aFERexxFunctions array.

/*
 * V0.9.9 (2001-02-28) [umoeller]:
 *      the above is no longer true. For each REXX function,
 *      we now have a single C APIENTRY function, an array
 *      of which is passed to the FERexx constructor to
 *      register the functions with the REXX interpreter
 *      when the FERexx instance gets created on WarpIN
 *      startup.
 *
 *      See src\engine\fe_rexx.cpp for this separation now.
 */

/*
 *@@ WirexxPutEnv:
 *      sets an environment value in the WarpIN environment.
 *
 *      REXX syntax: rc=WirexxPutEnv("name","value");
 *
 *      If the function is called without a "value" parameter, the variable
 *      "name" is removed from the environment.
 *
 *      The implementation is in FERexx::WirexxPutEnv.
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 */

ULONG WirexxPutEnv(PCSZ name,
                   ULONG argc,
                   RXSTRING argv[],
                   PCSZ queuename,
                   PRXSTRING retstr)
{
    int rc;
    BSString strKey, strExpression;

    // BSStringMap::const_iterator cit;

    // no name -> nothing to do
    // Remark: ALL parameters of a REXX script have to be checked for
    // validity: Unlike for C/C++ programs you cannot assume that if
    // one argument exists also all the arguments "before" this do
    // exist!
    if ((1 > argc) || !RXVALIDSTRING(argv[0]))
        return 1;

    // create environment set string
    strupr(RXSTRPTR(argv[0]));
    strKey = RXSTRPTR(argv[0]);
    strExpression = RXSTRPTR(argv[0]);
    if ((2 <= argc) && RXVALIDSTRING(argv[1]))
    {
        strExpression += "=";
        strExpression += RXSTRPTR(argv[1]);
    }
    // if argv[1] doesn't exist the variable is removed

    // the environment string is put into this static map structure
    // in order to keep the memory valid after leaving this function
    StringMapEntry *pEntry = new StringMapEntry(strKey, strExpression);
    G_EnvStrings.insert(pEntry); // [strKey] = strExpression;

    // Set the _stored_ string pointer which is persistent!
    // Remark: The putenv() function seems to store only a _pointer_ to the
    // passed string, i.e. the string storage must not be local, but
    // persistent! That's why we have the static string map...
    rc = putenv(pEntry->_strValue.c_str());

    // Now comes the return value for the REXX caller
    // Remark: You should always return _something_, and if it is an
    // empty string (explicitly, i.e. with SetRexxReturn("",retstr),
    // otherwise your function will just return the garbage that was
    // accidentally in retstr on calling!
    if (0 != rc)
        SetRexxReturn("1", retstr);
    else
        SetRexxReturn("0", retstr);

    // Here you should always return 0, otherwise the REXX interpreter
    // just stops because it means that some kind of "internal error"
    // occurred!
    // (Return error codes to the REXX caller using retstr: See above)
    return 0;
}

/*
 *@@ WirexxGetEnv:
 *      gets an environment value in the WarpIN environment.
 *
 *      remark: this is necessary for retrieving values set by WarpIN,
 *      because these are not in the inherited environment of the REXX
 *      function calls!
 *
 *      Example: WI_ARCHIVE_PATH contains the path of the WarpIN archives
 *
 *      REXX syntax: value=WirexxPutEnv("name");
 *
 *      The implementation is in FERexx::WirexxGetEnv.
 *
 *@@added V0.9.3 (2000-05-26) [cbo]
 */

ULONG WirexxGetEnv(PCSZ name,
                   ULONG argc,
                   RXSTRING argv[],
                   PCSZ queuename,
                   PRXSTRING retstr)
{
    BSString strKey, strExpression;

    // no name -> nothing to do
    // Remark: ALL parameters of a REXX script have to be checked for
    // validity: Unlike for C/C++ programs you cannot assume that if
    // one argument exists also all the arguments "before" this do
    // exist!
    if ((1 > argc) || !RXVALIDSTRING(argv[0]))
        return 1;

    // get environment value string
    PSZ pszVal = getenv(RXSTRPTR(argv[0]));

    // return a value, if not NULL
    if (NULL != pszVal)
        SetRexxReturn(pszVal, retstr);
    else
        SetRexxReturn("", retstr);

    // Here you should always return 0, otherwise the REXX interpreter
    // just stops because it means that some kind of "internal error"
    // occurred!
    // (Return error codes to the REXX caller using retstr: See above)
    return 0;
}

/*
 *@@ WirexxShowMessage:
 *      shows a message using the WarpIN callback function
 *      guiShowMessage.
 *
 *      REXX syntax: rc=WirexxShowMessage("title","message",flags);
 *
 *      The flags are those that are passed to guiShowMessage directly,
 *      i.e. you may "or" the following values together. The default
 *      value is 0.
 *
 *      -- 0x0000 MB_OK               Message box contains an OK push button.
 *      -- 0x0001 MB_OKCANCEL         Message box contains both OK and CANCEL push
 *                                    buttons.
 *      -- 0x0006 MB_CANCEL           Message box contains a CANCEL push button.
 *      -- 0x0007 MB_ENTER            Message box contains an ENTER push button.
 *      -- 0x0008 MB_ENTERCANCEL      Message box contains both ENTER and CANCEL push
 *                                    buttons.
 *      -- 0x0002 MB_RETRYCANCEL      Message box contains both RETRY and CANCEL
 *                                    push buttons.
 *      -- 0x0003 MB_ABORTRETRYIGNORE Message box contains ABORT, RETRY, and
 *                                    IGNORE push buttons.
 *      -- 0x0004 MB_YESNO            Message box contains both YES and NO push
 *                                    buttons.
 *      -- 0x0005 MB_YESNOCANCEL      Message box contains YES, NO, and CANCEL push
 *                                    buttons.
 *
 *      -- 0x0030 MB_ERROR            Message box contains a small red circle with a red
 *                                    line across it.
 *      -- 0x0020 MB_ICONASTERISK     Message box contains an information (i) icon.
 *      -- 0x0040 MB_ICONEXCLAMATION  Message box contains an exclamation point (!)
 *                                    icon.
 *      -- 0x0030 MB_ICONHAND         Message box contains a small red circle with a red
 *                                    line across it.
 *      -- 0x0000 MB_ICONQUESTION     Message box contains a question mark (?) icon.
 *      -- 0x0010 MB_INFORMATION      Message box contains an information (i) icon.
 *      -- 0x0020 MB_NOICON           Message box is not to contain an icon.
 *      -- 0x0010 MB_QUERY            Message box contains a question mark (?) icon.
 *      -- 0x0020 MB_WARNING          Message box contains an exclamation point (!) icon.
 *
 *      -- 0x0000 MB_DEFBUTTON1       The first button is the default selection. This is the
 *                                    default case, if none of MB_DEFBUTTON1,
 *                                    MB_DEFBUTTON2, and MB_DEFBUTTON3 is
 *                                    specified.
 *      -- 0x0100 MB_DEFBUTTON2       The second button is the default selection.
 *      -- 0x0200 MB_DEFBUTTON3       The third button is the default selection.
 *
 *      -- 0x0000 MB_APPLMODAL        Message box is application modal. This is the
 *                                    default case. Its owner is disabled; therefore, do
 *                                    not specify the owner as the parent if this option
 *                                    is used.
 *      -- 0x1000 MB_SYSTEMMODAL      Message box is system modal.
 *      -- 0x4000 MB_MOVEABLE         Message box is moveable.
 *
 *      With rc you get feedback about what with which button the user terminated
 *      the message display:
 *      -- 9      MBID_ENTER  ENTER push button was selected
 *      -- 1      MBID_OK     OK push button was selected
 *      -- 2      MBID_CANCEL CANCEL push button was selected
 *      -- 3      MBID_ABORT  ABORT push button was selected
 *      -- 4      MBID_RETRY  RETRY push button was selected
 *      -- 5      MBID_IGNORE IGNORE push button was selected
 *      -- 6      MBID_YES    YES push button was selected
 *      -- 7      MBID_NO     NO push button was selected
 *      -- 0xffff MBID_ERROR  Function not successful; an error occurred.
 *
 *      The implementation is in FERexx::WirexxShowMessage.
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 *@@changed V0.9.20 (2002-07-03) [umoeller]: adjusted for unicode conversions
 */

ULONG WirexxShowMessage(PCSZ name,
                        ULONG argc,
                        RXSTRING argv[],
                        PCSZ queuename,
                        PRXSTRING retstr)
{
    ustring ustrTitle, ustrMessage;
    char str[32];
    ULONG flags, rc;

    // title
    if ((1 > argc) || !RXVALIDSTRING(argv[0]))
        ustrTitle.assignUtf8("");
    else
        ustrTitle.assignCP(G_pLocals->_pCodecProcess,
                           RXSTRPTR(argv[0]));

    // message
    if ((2 > argc) || !RXVALIDSTRING(argv[1]))
        ustrMessage.assignUtf8("");
    else
        ustrMessage.assignCP(G_pLocals->_pCodecProcess,
                             RXSTRPTR(argv[1]));

    // flags
    if ((3 > argc) || !RXVALIDSTRING(argv[2]))
        flags = 0;
    else
        flags = atoi(RXSTRPTR(argv[2]));

    // display the message
    rc = G_pLocals->ShowMessage(ustrTitle,
                                ustrMessage,
                                flags);

    // put the result code into retstr
    _itoa(rc, str, sizeof(str) - 1);
    SetRexxReturn(str, retstr);

    // Here you should always return 0, otherwise the REXX interpreter
    // just stops because it means that some kind of "internal error"
    // occurred!
    // (Return error codes to the REXX caller using retstr: See above)
    return 0;
}

//////////////////////////////////////////////////////////////////
//   From the docs about writing REXX functions:
//
//   The following is a sample external function definition:
//
//   ULONG SysLoadFuncs(
//      PSZ       Name,                   // name of the function
//      LONG      Argc,                   // number of arguments
//      RXSTRING  Argv[],                 // list of argument strings
//      PSZ       Queuename,              // current queue name
//      PRXSTRING Retstr)                 // returned result string
//
//   Where
//
//    Name        Address of ASCIIZ external function name.
//
//    Argc        The size of the argument list. Argv will contain Argc RXSTRINGs.
//
//    Argv        An array of null-terminated RXSTRINGs for the function arguments.
//
//    Queue       The name of the currently defined REXX external data queue.
//
//    Retstr      Address of an RXSTRING for the returned value. Retstr is a
//                character string function or subroutine return value. When a
//                REXX program calls an external function with the REXX CALL
//                instruction, Retstr is assigned to the REXX special variable
//                RESULT. When the REXX program calls an external function as a
//                function, Retstr is used directly within the REXX expression.
//
//                The REXX interpreter provides a default 256-byte RXSTRING in
//                Retstr. A longer RXSTRING can allocated with "DosAllocMem" if
//                the returned string is longer name 256 bytes. The REXX interpreter
//                releases Retstr with "DosFreeMem" when the external function
//                completes.
//
//    Returns     An integer return code from the function. When the external
//                function returns 0, the function completed successfully. Retstr
//                contains the function return value. When external function return
//                code is not 0, the REXX interpreter raises REXX error 40
//                ("Invalid call to routine").  The Retstr value is ignored.
//
//                If the external function does not have a return value, the
//                function must set Retstr to an an empty RXSTRING (NULL strptr).
//                When a function does not return a value, the interpreter raises
//                error 44, "Function did not return data". When an external
//                function invoked with the REXX CALL instruction does not return
//                a value, the REXX interpreter drops (unassigns) the special
//                variable RESULT.

/* ******************************************************************
 *
 *  FERexx implementation
 *
 ********************************************************************/

/*
 *@@ FERexx:
 *      constructor.
 *
 *      Registers the functions in the
 *      rexx functions array passed to it.
 *
 *      This implementation has been changed to allow moving
 *      REXX support to the WPI runtime (src\base). The REXX
 *      functions array is now passed to this constructor
 *      from main() when REXX is initialized.
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 *@@changed V0.9.9 (2001-02-28) [umoeller]: added array parameter
 *@@changed V0.9.20 (2002-07-03) [umoeller]: no longer using array param, this is now here
 */

FERexx::FERexx(FELocals &Locals)
    : BSRoot(tFERexx)
{
    ULONG   rc,
            ul;

    // set global variable for message box
    G_pLocals = &Locals;

    // register REXX functions
    for (ul = 0;
         ul < ARRAYITEMCOUNT(G_aRexxFunctions);
         ++ul)
    {
        rc = RexxRegisterFunctionExe(G_aRexxFunctions[ul].pcszFunctionName,
#ifdef __IBMCPP__
                                     (PFN)G_aRexxFunctions[ul].pfn);
#else
                                     G_aRexxFunctions[ul].pfn);
#endif
    }
}

/*
 *@@ ~FERexx:
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 */

FERexx::~FERexx()
{
    ULONG rc,
          ul;

    // deregister REXX functions
    for (ul = 0;
         ul < ARRAYITEMCOUNT(G_aRexxFunctions);
         ++ul)
    {
        rc = RexxDeregisterFunction(G_aRexxFunctions[ul].pcszFunctionName);
    }
}

/*
 *@@ StoreRexxCode:
 *      store a REXX code chunk in the WarpIN REXX codestore
 *      under the given function name.
 *
 *      The code is not verified for validity in any way.
 *      However, since in this context, the REXX interpreter
 *      only allows semicola as command separators and the
 *      whole code must not contain any line breaks, some
 *      reformatting is done in order to allow for "usual"
 *      REXX code (line ends separating commands, comma as
 *      line continuation mark).
 *
 *      The code that comes in here uses the codepage of the
 *      installation script. This imposes several problems:
 *
 *      --  Actual REXX syntax should not be a problem,
 *          since that is probably forced to use ASCII
 *          anyway.
 *
 *      --  The code might however use filenames and NLS
 *          strings for message boxes. We can't store the
 *          code as Unicode here because the code might
 *          pass such a string to a REXX api that we
 *          won't intercept, and that might choke on UTF-8.
 *
 *      So the temporary solution (until I come up with
 *      something better) is that we convert all code
 *      to the process codepage before storing it. Since
 *      the REXX interpreter probably assumes that anyway,
 *      this looks good to me. V0.9.20 (2002-07-03) [umoeller]
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 *@@changed V0.9.2 (2000-03-13) [cbo]: allows line breaks and line continuation
 *@@changed V0.9.14 (2001-07-12) [umoeller]: fixed bug in xstrrpl which caused ',' continuation to break here (PmAs)
 *@@changed V0.9.20 (2002-07-03) [umoeller]: added pCodec param, added Unicode support
 */

void FERexx::StoreRexxCode(const char *pcszFuncName,    // in: REXX function name
                           const char *pcszCode,        // in: REXX code
                           BSUniCodec *pCodec)          // in: codec representing the codepage the code is in
{
    // convert code chunk to Unicode
    ustring ustrTemp(pCodec, pcszCode);
    // convert unicode to process codepage
    string rxcode(G_pLocals->_pCodecProcess, ustrTemp);

    size_type pos;
    static string   crlf("\r\n"),
                    lf("\n"),
                    splf(" \n"),
                    lfspspspsp("\n    "),
                    lfsp("\n "),
                    commalf(",\n"),
                    empty(""),
                    semicolon(";");

    // change all CRLF to LF codes
    ULONG ulOfs = 0;
    ULONG c = 0;
    while (rxcode._find_replace(crlf, lf, &ulOfs)
                != BSString::npos)
            ++c;

    // trim all lines (remove spaces)
    ulOfs = 0;
    c = 0;
    while (rxcode._find_replace(splf, lf, &ulOfs)
                != BSString::npos)
            ++c; // moved this up V0.9.14 (2001-07-07) [umoeller]

    ulOfs = 0;
    c = 0;
    while (rxcode._find_replace(lfspspspsp, lf, &ulOfs)
                != BSString::npos)
            ++c;

    ulOfs = 0;
    c = 0;
    while (rxcode._find_replace(lfsp, lf, &ulOfs)
                != BSString::npos)
            ++c;

    // remove commas at line ends and merge lines
    ulOfs = 0;
    c = 0;
    while (rxcode._find_replace(commalf, empty, &ulOfs)
                != BSString::npos)
            ++c;

    // replace line ends by semocolons
    ulOfs = 0;
    c = 0;
    while (rxcode._find_replace(lf, semicolon, &ulOfs)
                != BSString::npos)
            ++c;

    // this should run now; create a string
    // map entry for this code
    G_RexxCodeStore.insert(new StringMapEntry(pcszFuncName,
                                              rxcode));
}

/*
 *@@ ExecuteRexxCode:
 *      pass the name of a REXX code segment that was stored.
 *
 *      The return code is 0 (WIREXX_NOERROR) if no error
 *      occurred or one of the other WIREXX_* values to
 *      indicate a problem or error.
 *
 *      If WIREXX_NOERROR is returned, the function return
 *      value is stored in a new buffer in **ppszRet, which
 *      the caller must free() when it's no longer needed.
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 */

int FERexx::ExecuteRexxCode(const char *pcszFuncName,   // in: function to execute
                            const char *pcszArgs,       // in: function parameters
                            char **ppszRet)             // out: function return value
{
    RXSTRING        rxproc[2],
                    rxargv[1],
                    rxret;
    char            retbuf[256];
    SHORT           rexxrc;
    LONG            rc;

    // make sure we return a valid ret pointer
    *ppszRet = NULL;

    // no code - no call -> probably not what the caller intended to do!
    if ((NULL == pcszFuncName) || (0 == strlen(pcszFuncName)))
        return WIREXX_INVALIDNAME;

    // go find the respective code node in the global tree
    // according to the function name
    StringMapEntry *pNode;
    if (!(pNode = (StringMapEntry*)treeFind(G_RexxCodeStore._TreeRoot,
                                            (ULONG)pcszFuncName,
                                            CompareIStrings)))
        // not found:
        return WIREXX_NOCODE;

    // rxProc is an array of two RXSTRINGS which is needed
    // for RexxStart below; the second field would be the
    // tokenized code, but we don't have that
    MAKERXSTRING(rxproc[0],
                 pNode->_strValue.c_str(), // (*icode).second.c_str(),
                 pNode->_strValue.size()); // (*icode).second.size());
    MAKERXSTRING(rxproc[1],
                 NULL,
                 0);

    // argument string, if present
    if (    (NULL == pcszArgs)
         || (0 == strlen(pcszArgs))
       )
    {
        MAKERXSTRING(rxargv[0], NULL, 0);
    }
    else
    {
        MAKERXSTRING(rxargv[0], pcszArgs, strlen(pcszArgs));
    }

    // prepare a return buffer
    memset(retbuf, 0, sizeof(retbuf));
    MAKERXSTRING(rxret, retbuf, sizeof(retbuf));

    // do the call
    if (rc = RexxStart(1,                   // no. of arguments
                       rxargv,              // rxstring arglist
                       pcszFuncName,        // rexx procedure name
                       rxproc,              // instore: if != NULL, rexx code is assumed
                                            // to be in here (array of two RXSTRINGS)
                       NULL,                // initial ADDRESS environment
                       RXCOMMAND,           // call type...
                       NULL,                // exits
                       &rexxrc,             // out: return code
                       &rxret))             // out: return value from function
        return WIREXX_REXXFAILED;

    // allocate a new buffer for the result
    *ppszRet = (char *)malloc(RXSTRLEN(rxret) + 1);

    // copy the string to the result, even allowing for 0 characters!
    memcpy(*ppszRet, RXSTRPTR(rxret), RXSTRLEN(rxret));
    (*ppszRet)[RXSTRLEN(rxret)] = 0;

    // free extra allocated memory
    if (RXSTRPTR(rxret) != retbuf)
        DosFreeMem(RXSTRPTR(rxret));

    return WIREXX_NOERROR;
}

/*
 *@@ ExecuteRexxMacros:
 *      pass the pointer to a string containing =("xyz arg1 arg2...")
 *      tags to this and it will call the REXX function with the name
 *      "xyz" and the argument string "arg1 arg2...". After that the
 *      tag is removed and replaced with the return value of the function,
 *      which may also be empty.
 *
 *      Since the string length can change with this call, the string
 *      storage may be reallocated using realloc(), i.e. it is assumed
 *      that it was originally allocated with alloc() and must be
 *      freed with free().
 *
 *      The call returns TRUE if the REXX call succeeded, otherwise
 *      FALSE. In the latter case the return string is set to empty
 *      so possibly an erroneous call to REXX can still return a
 *      useable string - but better catch these cases!
 *
 *      After an error occurred, the function still tries to handle
 *      the remaining string, so "an error occurred" could mean that
 *      actually several occurred.
 *
 *      If pulErrorOffset is not NULL (which is allowed), it returns the
 *      position of the first error within the passed string.
 *      If errorName is not NULL, it returns the name of the first
 *      offending REXX macro call.
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 *@@changed V0.9.18 (2002-03-03) [umoeller]: fixed bug introduced by earlier 0.9.18 optimization
 */

BOOL FERexx::ExecuteRexxMacros(char **ppszStr,
                               PULONG pulErrorOffset,
                               char **ppszErrorCall)
{
    PSZ     ps, pe, macro, args, ret;
    ULONG   os, oe, oldtlen, newtlen, oldlen, newlen, rc;
    BOOL    ok = TRUE;

#define STAG "=(\""
#define ETAG "\")"
#define NO_ERROROFS 0xFFFFFFFF

    // initialize error return
    if (NULL != pulErrorOffset)
        *pulErrorOffset = NO_ERROROFS;
    if (NULL != ppszErrorCall)
        *ppszErrorCall = NULL;

    // do while we find start and end of a REXX macro
    for (ps = strstr(*ppszStr, STAG), pe = (NULL == ps) ? NULL : strstr(ps, ETAG);
         (NULL != ps) && (NULL != pe);
         ps = strstr(pe, STAG), pe = (NULL == ps) ? NULL : strstr(ps, ETAG))
    {
        // get the total code
        ps += strlen(STAG);
        macro = (char *)malloc(pe - ps + 1);
        memcpy(macro, ps, pe - ps);
        macro[pe - ps] = 0;

        // separate uppercase "macro" from "args"
        args = strchr(macro, ' ');
        if (NULL != args)
            *args++ = 0;            // fixed V0.9.18 (2002-03-03) [umoeller]
        strupr(macro);

        // call the macro
        if (WIREXX_NOERROR != (rc = ExecuteRexxCode(macro, args, &ret)))
        {
            // the call didn't succeed (e.g. because of errors in the REXX code)
            // create an empty return string: This will result in the erroneous
            // call just being deleted from the string
            ret = (char *)malloc(1);
            *ret = 0;
            ok = FALSE;
            if ((NULL != pulErrorOffset) && (NO_ERROROFS == *pulErrorOffset))
                *pulErrorOffset = ps - *ppszStr;
            if ((NULL != ppszErrorCall) && (NULL == *ppszErrorCall))
            {
                *ppszErrorCall = (char *)malloc(pe - ps + 1);
                strncpy(*ppszErrorCall, ps, pe - ps);
                (*ppszErrorCall)[pe - ps] = 0;
            }
        }

        // calculate all the needed lengths
        oldlen = pe - ps + strlen(STAG) + strlen(ETAG);
        newlen = strlen(ret);
        oldtlen = strlen(*ppszStr);
        newtlen = oldtlen + newlen - oldlen + 1;

        // now we need offsets instead of pointers
        os = ps - *ppszStr - strlen(STAG);
        oe = pe - *ppszStr + strlen(ETAG);

        // reallocate the string if it is growing
        if (newtlen > oldtlen)
            *ppszStr = (char *)realloc(*ppszStr, newtlen);

        // move the tail of the string
        memmove(*ppszStr + oe - oldlen + newlen, *ppszStr + oe, strlen(*ppszStr + oe) + 1);

        // insert the returned string
        memcpy(*ppszStr + os, ret, newlen);

        // reallocate the string if it is shrinking
        if (newtlen < oldtlen)
            *ppszStr = (char *)realloc(*ppszStr, newtlen);

        // free the returned string from REXX
        free(ret);

        // set pointers for further searching
        pe = ps = *ppszStr + os;
    }

    // TRUE or FALSE: success indicator
    return ok;
}

/*
 *@@ ExecuteRexxMacros:
 *      second overloaded version that uses BSStrings.
 *
 *@@added V0.9.2 (2000-03-10) [cbo]
 */

BOOL FERexx::ExecuteRexxMacros(BSString &str,
                               PULONG pulErrorOffset,
                               BSString &strErrorCall)
{
    char    *pstr,
            *perr = NULL;
    BOOL ok = TRUE;

    // copy string to a common C string
    if (pstr = strhdup(str.c_str(), NULL))      // V0.9.16 (2001-10-24) [umoeller]
    {
        // call the first overload version
        ok = ExecuteRexxMacros(&pstr, pulErrorOffset, &perr);

        // change back to BSStrings
        str = pstr;
        strErrorCall = perr;

        // free the temporary memory
        free(pstr);
        if (perr != NULL)
            free(perr);
    }

    return ok;
}


