
/*
 *@@sourcefile fe_engine.cpp:
 *      the actual WarpIN installer engine. This
 *      now (V0.9.14) implements the FEInstallEngine class.
 *
 *      FEInstallEngine now encapsulates the messy code that
 *      used to be all over the place in src\frontend.
 *      As a result, the actual installer engine is
 *      no longer procedural.
 *
 *      Besides being prettier, this has the following
 *      advantages:
 *
 *      --  The whole WarpIN installer engine is now
 *          in WPIRTL.DLL.
 *
 *      --  Because of this, it should be much easier
 *          now to create a command-line WarpIn that
 *          can also install. Mainly all this was
 *          done for a future CID install that works
 *          properly.
 *
 *      See FEInstallEngine for details.
 *
 *      To find out how this can be used, take a look
 *      at main() in src\frontend\warpin.cpp. The rest
 *      of the code in src\frontend is now exclusively
 *      GUI code, i.e. PM-specific stuff.
 *
 *@@added V0.9.14 (2001-07-26) [umoeller]
 *@@header "engine\fe_engine.h"
 */

/*
 *      This file Copyright (C) 1999-2001 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_REXXSAA
#include <os2.h>
#ifdef __IBMCPP__
    #include <rexxsaa.h>    // EMX has this in os2.h already
#endif

#include <direct.h>

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h

#include "setup.h"

#include "helpers\configsys.h"
#include "helpers\dosh.h"
#include "helpers\except.h"
#include "helpers\nls.h"
#include "helpers\procstat.h"
#include "helpers\stringh.h"
#include "helpers\threads.h"
#include "helpers\tree.h"
#include "helpers\xstring.h"

// base includes
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
#include "base\bs_map.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

// back-end includes
#include "wiarchive\wiarchive.h"

#include "libbz2\bzlib.h"               // for the error codes V0.9.12 (2001-04-29) [umoeller]

// front-end includes
#include "engine\fe_base.h"

#include "engine\fe_script.h"

#include "engine\fe_job.h"
#include "engine\fe_package.h"
#include "engine\fe_package_arc.h"
#include "engine\fe_package_db.h"
#include "engine\fe_archive.h"
#include "engine\fe_database.h"

#include "engine\fe_engine.h"

#include "engine\fe_rexx.h"

int TREEENTRY CompareIStrings(ULONG ul1, ULONG ul2);        // in fe_rexx.cpp

#pragma hdrstop

DEFINE_CLASS(FEInstallEngine, BSRoot);

/* ******************************************************************
 *
 *  Documentation
 *
 ********************************************************************/

/*
 *@@gloss: class_rels class relationships
 *
 *      The following diagram illustrates how the front-end
 *      classes typically reference each other after the engine
 *      (FEInstallEngine) has fully initialized from an
 *      WPI file:
 *
 +        FELocals ͻ
 +                                                                             
 +       ͼ
 +
 +        FEDatabase ͻ
 +                                       oo  oo                                
 +       ͳͳͼ
 +                                          
 +        FERexx ͳͳͻ
 +                                                                         
 +       ͳͳͼ
 +                                          
 +        FEInstallEngine ͳͳͻ
 +                                                                         
 +           FEInstallJob (pck 1) ٳ                                  
 +           FEInstallJob (pck 2)                                   
 +           FEGroupJob                                                
 +           FEInstallJob (pck 3) ٳ                                
 +           FEInstallJob (pck 4)                                 
 +                                                                        
 +             FEArchive ͻ
 +                                                                      
 +          > FEArcPackagePck Ŀ      
 +           > FEArcPackagePck ſ     
 +            > FEArcPackageGroup ſ    
 +             > FEArcPackagePck ſ   
 +              > FEArcPackagePck ſ  
 +                                                                      
 +                     WIArchive ͻ   
 +                       (physical WPI file)      #                   
 +                    #ͼ   
 +                                                 #                    
 +                     FEScriptBase (subclass) #ͻ   
 +                                                #                   
 +                           FEPckDeclBase (subclass) (pck 1) <ٳ  
 +                           FEPckDeclBase (subclass) (pck 2) <ٳ  
 +                           FEGroupDecl                      <ٳ  
 +                               FEPckDeclBase (subclass) (pck 3) <ٳ  
 +                               FEPckDeclBase (subclass) (pck 4) <  
 +                                                                         
 +                           FEPageInfo                                    
 +                           FEPageInfo                                    
 +                           FEPageInfo                                    
 +                           FEPageInfo                                    
 +                                                                         
 +                    ͼ        
 +                 ͼ
 +       ͼ
 *
 *      This now gives us four layers in "install" mode:
 *
 *      -- FEInstallEngine is the encapsulation of the WarpIN
 *         installer engine. One such instance gets created
 *         from EnterArchiveMode (which gets created from main())
 *         in warpin.cpp.
 *
 *         The constructor (FEInstallEngine::FEInstallEngine)
 *         takes the file name
 *         of the WPI file as input and will create a member
 *         instance of FEArchive with that file name.
 *
 *         FEArchive then runs WIArchive::open on its member
 *         back-end archive class to open the archive file.
 *
 *         EnterArchiveMode then calls FEInstallEngine::AppendArchive,
 *         which, among other things, calls FEArchive::ParseScript.
 *
 *      -- FEArchive::ParseScript in turn creates a script
 *         instance. This has been completely encapsulated
 *         in FEScriptBase and the various subclasses with V0.9.9
 *         to allow for replacing the script language entirely
 *         (e.g. for XML support).
 *
 *         The magic of this is in the static FEScriptBase::CreateScript
 *         method, which creates an instance of a FEScriptBase subclass
 *         depending on the script format. This is magic because
 *         that method will find out what subclass needs to be used
 *         and call the proper parsing methods automatically, while
 *         the caller only needs to know the fields in the FEScriptBase
 *         instance which is returned. It therefore no longer needs
 *         to bother about the parsing itself.
 *
 *         FEScriptBase contains package declarations and page
 *         declarations, represented by the FEPckDeclBase and FEPageInfo
 *         classes. These represent the "pure" package and page
 *         declarations from the script after syntax checking.
 *         However, at this point, they are not yet checked
 *         against the database.
 *
 *      -- After FEScriptBase::CreateScript has parsed the script,
 *         FEArchive::ParseScript then constructs instances of
 *         the various package classes from the package declarations
 *         in the script.
 *
 *         Basically, FEPackageBase is the base of the package
 *         classes, the most important of which is FEArcPackageBase.
 *         So in short, FEArchive (the archive) contains
 *         FEArcPackageBase's (the packages), which in turn point
 *         to the FEPckDeclBase package declarations.
 *
 *      -- The fourth layer is "install jobs" finally, which form
 *         another wrapper around the package classes to hold data
 *         which can be changed by the user. While the data in
 *         the package classes is assumed to be non-variable (because
 *         it comes from the script), the install jobs hold things
 *         like target paths and package selections, which can change.
 *
 *         The FEInstallJob instances are created by FEInstallEngine
 *         after all archives and packages have been created
 *         according to the script.
 *         See fe_job.cpp for details.
 *
 *      To summarize:
 *
 *      1)  FEArchive creates FEScriptBase.
 *
 *      2)  FEScriptBase parses the script and creates FEPckDeclBase
 *          and FEPageInfo instances.
 *
 *      3)  FEArchive then constructs FEPackageBase instances
 *          from the FEPckDeclBase instances and checks them against
 *          the database.
 *
 *      4)  FEInstallJob's are then created from the FEPackageBase
 *          before the install pages are displayed to the user.
 *
 *      All this is in turn encapsulated by FEInstallEngine.
 */

/*
 +
 +       ** FELocals *****************************************
 +       *                                                   *----------------+
 +       *****************************************************        |       |
 +                                                                    |       |
 +       ** FEDatabase ***************************************        |       |
 +       *                                oo  oo             *--------+       |
 +       *********************************||**||**************                |
 +                                        ||  ||                              |
 +       ** FERexx ***********************||**||**************                |
 +       *                                ||  ||             *-+              |
 +       *********************************||**||************** |              |
 +                                        ||  ||               |              |
 +       ** FEInstallEngine **************||**||***************|******************
 +       *                                ||  ||               |                 *
 +       *   /----- FEInstallJob (pck 1) -/|  ||               |                 *
 +       *   |/---- FEInstallJob (pck 2) --/  ||               |                 *
 +       *   ||/--- FEGroupJob                ||               |                 *
 +       *   |||/------ FEInstallJob (pck 3) -/|               |                 *
 +       *   ||||/----- FEInstallJob (pck 4) --/               |                 *
 +       *   |||||                                             |                 *
 +       *   |||||  * FEArchive *******************************|**************** *
 +       *   |||||  *                                          |               * *
 +       *   \---------> FEArcPackagePck ----------------------|--------\      * *
 +       *    \--------> FEArcPackagePck ----------------------|--------|\     * *
 +       *     \-------> FEArcPackageGroup --------------------|--------||\    * *
 +       *      \----------> FEArcPackagePck ------------------|--------|||\   * *
 +       *       \---------> FEArcPackagePck ------------------|--------||||\  * *
 +       *          *                                          |        |||||  * *
 +       *          *   * WIArchive ***************************|******* |||||  * *
 +       *          *   *   (physical WPI file)      #         |      * |||||  * *
 +       *          *   *****************************#*********|******* |||||  * *
 +       *          *                                #         |        |||||  * *
 +       *          *   * FEScriptBase (subclass) ***#***************** |||||  * *
 +       *          *   *                            #                * |||||  * *
 +       *          *   *       FEPckDeclBase (subclass) (pck 1) <------/||||  * *
 +       *          *   *       FEPckDeclBase (subclass) (pck 2) <-------/|||  * *
 +       *          *   *       FEGroupDecl                      <--------/||  * *
 +       *          *   *           FEPckDeclBase (subclass) (pck 3) <-----/|  * *
 +       *          *   *           FEPckDeclBase (subclass) (pck 4) <------/  * *
 +       *          *   *                                             *        * *
 +       *          *   *       FEPageInfo                            *        * *
 +       *          *   *       FEPageInfo                            *        * *
 +       *          *   *       FEPageInfo                            *        * *
 +       *          *   *       FEPageInfo                            *        * *
 +       *          *   *                                             *        * *
 +       *          *   ***********************************************        * *
 +       *          *                                                          * *
 +       *          ************************************************************ *
 +       *                                                                       *
 +       *************************************************************************
 *
 */

/* ******************************************************************
 *
 *  FEFileError
 *
 ********************************************************************/

/*
 *@@ FEFileError:
 *      constructor for the FEFileError helper class.
 *
 *@@added V0.9.0 (99-10-26) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: made this a constructor, this was wpiPrepareGUIFILEERROR
 */

FEFileError::FEFileError(FELocals &Locals,
                         const ustring *pstrBasePath, // in: if != NULL, base path;
                                                     // if NULL, we get the curdir
                         WIFileHeader* pwifh)        // in: file header from archive
        :   BSRoot(tFEFileError),
            _Locals(Locals)
{
    memset(&_fs3Existing, 0, sizeof(_fs3Existing));
    _pwifh = pwifh;

    if (!pstrBasePath)
    {
        // get file name (in current working directory)
        CHAR sz[CCHMAXPATH];
        _getcwd(sz, sizeof(sz));
        _strFilename = sz;
    }
    else
        _strFilename.assignUtf8(Locals._pCodecProcess, *pstrBasePath);

    if (_strFilename[_strFilename.length() - 1] != '\\')
        _strFilename += '\\';
    _strFilename += pwifh->name;

    // 1) get file info for file on disk
    //    (current directory)
    DosQueryPathInfo(_strFilename.c_str(),
                     FIL_STANDARD,
                     &_fs3Existing,
                     sizeof(_fs3Existing));

    // 2) info for file in archive:
    // WIFileHeader has two time_t fields which we need
    // to convert to human-readable format first.
    _iComp = wpiCompareFileDates(_pwifh,
                                 &_fs3Existing,
                                 &_fDateHeader,
                                 &_fTimeHeader);
        // this returns:
        // < 0:  existing file is newer
        // == 0: existing file is same write date
        // > 0:  existing file is older
}

/* ******************************************************************
 *
 *  FEInstallEngine
 *
 ********************************************************************/

/*
 *@@ FEInstallEngine:
 *      constructor for the WarpIN installer engine,
 *      FEInstallEngine. This is used only in install mode.
 *
 *      This takes the following required parameters:
 *
 *      -- Database must reference the one and only
 *         instance of FEDatabase that must have been
 *         created beforehand. The engine constantly
 *         requires access to the database.
 *
 *      -- System must reference the one and only
 *         instance of FELocals, which encapsulates
 *         basic system state information and functions.
 *
 *      After this, call FEInstallEngine::AppendArchive.
 *
 *      This also opens the logger, so if you want install
 *      logging, call FELocals::SetLogFilename beforehand.
 *
 *@@changed V0.9.19 (2002-04-14) [umoeller]: now using ustring for archive name
 *@@changed V0.9.20 (2002-07-03) [umoeller]: removed callback args
 *@@changed V0.9.20 (2002-07-03) [umoeller]: now instantiating FERexx here
 */

FEInstallEngine::FEInstallEngine(FEDatabase &Database,                // in: ref to database object
                                 FELocals &Locals)                    // in: ref to FELocals object
    : BSRoot(tFEInstallEngine),
      _Database(Database),
      _Locals(Locals)
{
    _pCurrentArchive = NULL;

    _pArchivesList = new list<FEArchive*>(STORE);

    _pAllJobsList = new list<FEJobBase*>(STORE);

    _fAnyAlreadyInstalled = FALSE;

    _pRexx = new FERexx(Locals);

    _Locals.OpenLogger();
}

/*
 *@@ ~FEInstallEngine:
 *      the destructor.
 *
 */

FEInstallEngine::~FEInstallEngine()
{
    _Locals.CloseLogger();

    delete _pRexx;

    delete _pAllJobsList;

    delete _pArchivesList;

    // @@todo nuke archive and stuff;
    // we still have a problem... the destructor crashes
    // in list.clear() if the lists are in STORE mode,
    // but only with certain archives... whatever this is
    // V0.9.15 (2001-08-26) [umoeller]
}

/*
 *@@ AppendArchive:
 *      second step when using FEInstallEngine. After creating
 *      an instance of FEInstallEngine, call this method to have
 *      an archive opened, its script parsed, external archives
 *      opened, default selections set, etc.
 *
 *      This calls FEArchive::ParseScript on the member
 *      archive that was opened in the constructor.
 *
 *      After this, the caller GUI should display the
 *      pages from FEInstallEngine::_pPageInfoList.
 *
 *      The GUI should then invoke the FEInstallJob methods
 *      on the install jobs in the engine's member lists
 *      (FEInstallEngine::_AllJobsList) to select packages
 *      and set target paths.
 *
 *      After that, call FEInstallEngine::Install to start
 *      the installation.
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.3 (2000-04-20) [umoeller]: finally made macros work in target paths
 *@@changed V0.9.3 (2000-06-05) [umoeller]: added dependency checks
 *@@changed V0.9.7 (2001-01-07) [umoeller]: fixed problems with target paths and other checks...
 *@@changed V0.9.7 (2001-01-07) [umoeller]: version checks were missing for requirements
 *@@changed V0.9.14 (2001-07-26) [umoeller]: put this together from the code in warpin.cpp
 *@@changed V0.9.20 (2002-07-03) [umoeller]: moved opening the archive here from constructor
 *@@changed V0.9.20 (2002-07-06) [umoeller]: renamed from ParseAndSetup
 */

VOID FEInstallEngine::AppendArchive(const ustring &ustrArchiveName)      // in: archive to open
{
    UpdateGlobalStatus(GSTAT_LOADINGARCHIVE);

    _Locals.Log("Opening archive \"%s\"",
                ustrArchiveName.GetBuffer());

    // now go open the archive!
    _pCurrentArchive = new FEArchive(_Locals,
                                     ustrArchiveName);

    UpdateGlobalStatus(GSTAT_COLLECTING);

    /*
     * step 1: go parse the script and create packages
     *
     */

    _pCurrentArchive->ParseScript(*_pRexx);

    // copy archive title for GUI
    _ustrAppTitle = _pCurrentArchive->_pScript->_ustrTitle;

    // we now have the list of FEPackages in
    // pArchive->PackagesList

    /*
     * step 2: create jobs for this and sync the packages
     *         with the database
     *
     */

    _Locals.Log("Creating jobs...");
    _Locals.IncIndent(+4);

    BOOL fJobsAdded = FALSE;

    list<FEArcPackageBase*>::iterator PckStart = _pCurrentArchive->_pPackagesList->begin(),
                                      PckEnd = _pCurrentArchive->_pPackagesList->end();
    for (; PckStart != PckEnd; ++PckStart)
    {
        FEArcPackageBase *pPackageThis = *PckStart;

        // first find if we're part of a group ourselves
        FEGroupJob *pGroupJob = NULL;
        if (pPackageThis->_pGroup)
            // yes: find group job for that (must exist already)
            pGroupJob = FindGroupJob(pPackageThis->_pGroup);

        DYNAMIC_CAST(FEArcPackagePck, pPckPckThis, pPackageThis);

        if (!pPckPckThis)
        {
            // group: create group job
            FEGroupJob *pJobNew = new FEGroupJob((FEArcPackageGroup*)pPackageThis,
                                                 pGroupJob,
                                                 _Locals);
            _pAllJobsList->push_back(pJobNew);
            fJobsAdded = TRUE;
        }
        else
        {
            // add config info to global config allow flags
            // (moved this here, _pDecl is NULL for groups!)
            // V0.9.12 (2001-05-17) [umoeller]
            // _ulAllowConfig |= pPckPckThis->_pDecl->_ulConfigData;

            // create job for package
            FEInstallJob *pJobNew = new FEInstallJob(*pPckPckThis,
                                                     pGroupJob,
                                                     _Locals);

            WIVersion v;
            if (pJobNew->CheckInstall(&_Database, v))
            {
                _fAnyAlreadyInstalled = TRUE;
                _Locals.Log("job \"%s\": version %u.%u.%u.%u installed",
                            pPckPckThis->_pDecl->_ustrID.GetBuffer(),
                            v._ulMajor,
                            v._ulMinor,
                            v._ulRevision,
                            v._ulFixlevel);
            }
            else
                _Locals.Log("job \"%s\": not installed",
                            pPckPckThis->_pDecl->_ustrID.GetBuffer());

            _pAllJobsList->push_back(pJobNew);
            fJobsAdded = TRUE;
        }

    } // end for (; PckStart != PckEnd; ++PckStart)

    if (fJobsAdded)
        // anything added: store archive in global archives list
        _pArchivesList->push_back(_pCurrentArchive);

    _Locals.IncIndent(-4);

    /*
     * step 3: set default selections for all jobs
     *
     */

    ustring ustrMissingPackages;

    // STEP 1... check requirements

    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (;
         JobStart != JobEnd;
         ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            // check dependencies: V0.9.3 (2000-06-05) [umoeller]
            // go thru the package ID's which have been stored in
            // the package's logger
            const FEArcPackagePck &PckPckThis = pJobThis->_ArcPackage;

            // replaced all the following with the package IDs list
            // in the package for speed V0.9.18 (2002-02-06) [umoeller]

            // PSZ     pszLogStart = PckPckThis._logRequiresIDs._pabLogString,
            //         pszLogThis = pszLogStart;
            // while (pszLogThis < pszLogStart + PckPckThis._logRequiresIDs._cbLogString)

            list<FEPackageID*>::iterator pckidStart = PckPckThis._RequiresIDs.begin(),
                                         pckidEnd = PckPckThis._RequiresIDs.end();

            for (;
                 pckidStart != pckidEnd;
                 ++pckidStart)
            {
                // ULONG   cbLogThis = strlen(pszLogThis);
                BOOL    fSufficient = FALSE;
                // pszLogThis now points to the package ID which is required
                // by the job
                // FEPackageID pckidReq(_Locals,
                //                      pszLogThis, __FILE__, __LINE__,
                //                      FALSE);   // do not allow short format

                FEPackageID *pPckidReq = *pckidStart;

                const FEDBPackage *pDBPackage;

                if (pDBPackage = _Database.FindDBPackage(*pPckidReq))
                {
                    // check version
                    // FEPackageID pckidDB(_Locals,
                    //                     pDBPackage,
                    //                     __FILE__, __LINE__,
                    //                     FALSE);   // do not allow short format
                    // check version V0.9.7 (2001-01-07) [umoeller]
                    if (    (pDBPackage->_PckID.Compare(*pPckidReq)
                                   & IDCOMP_VERSIONMASK)
                            != IDCOMP_THISOLDER
                       )
                        fSufficient = TRUE;
                }

                // else: fSufficient still FALSE

                if (!fSufficient)
                {
                    // required package not found in database:
                    // is it on the install list then?
                    ULONG ulComp = 0;
                    if (!FindPackageInArchives(*pPckidReq,
                                               &ulComp))
                    {
                        // no: add to list of missing packages, which will
                        // be reported below
                        ustring ustrIDSix;
                        pPckidReq->CreateStringSix(ustrIDSix);
                        ustrMissingPackages.append(ustrIDSix);
                        ustrMissingPackages.appendUtf8("\n");
                    }
                }

                // pszLogThis += cbLogThis + 1;    // go beyond null byte
            } // while (pszLogThis < pszLogStart + PckPckThis._logRequiresIDs._cbLogString)
        } // if (pJobThis)
    } // for (; JobStart != JobEnd; ++JobStart)

    if (ustrMissingPackages())
    {
        // dependencies failed:
        _Locals.MessageBox(102,  // error
                           190,  // missing packages
                           MSG_ERROR_ABORT,
                           &ustrMissingPackages,
                           1);
        // terminate!
        throw BSCancelExcpt();
    }

    // STEP 2... set job target paths according to package defaults

    for (JobStart = _pAllJobsList->begin();
         JobStart != JobEnd;
         ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            // set job's target path:
            // first use default target path from package
            pJobThis->SetTargetPath(pJobThis->_ArcPackage.QueryTargetPath(),
                                    _pAllJobsList);
        }
    }

    _Locals.Log("Fixing job target paths...");
    _Locals.IncIndent(+4);

    // STEP 3... if a package is already installed,
    // set the target path to that of the database package
    // or resolve macros...
    // we have to do a third loop because otherwise BASE
    // paths can't get updated
    for (JobStart = _pAllJobsList->begin();
         JobStart != JobEnd;
         ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            if (pJobThis->_pDBPackageInstalled)
            {
                // package already installed:
                // reset target path to that of the installed package;
                // we must set this twice because otherwise non-base
                // packages won't get updated
                /* const FEDBPackage *pPackageInstalled
                        = _Database.FindDBPackage(*pJobThis->_pPackageIDInstalled); */

                pJobThis->SetTargetPath(pJobThis->_pDBPackageInstalled->QueryTargetPath(),
                                        _pAllJobsList);
            }
            else
            {
                // not installed:
                // resolve macros for target path
                ustring ustrTemp(pJobThis->QueryTargetPath());
                ResolveMacros(ustrTemp);
                pJobThis->SetTargetPath(ustrTemp,
                                        _pAllJobsList);
            }

            _Locals.Log("job \"%s\": target path is \"%s\"",
                        pJobThis->QueryPckPackageID().GetBuffer(),
                        pJobThis->QueryTargetPath().GetBuffer());

            // use default selection for job;
            // this will update the parent group too
            pJobThis->DefaultSelect();
        } // end if (!strMissingPackages())
    }

    _Locals.IncIndent(-4);

    // store page info list for GUI
    // _pPageInfoList = &_pCurrentArchive->_pScript->_PageInfoList;
            // removed V0.9.20 (2002-07-06) [umoeller]
}

/*
 * CreateDeinstallPackagesList:
 *      creates a list of packages (based on the current
 *      selections) which are to be de-installed. To be
 *      precise, those packages (from the database) are
 *      newly created and appended to the given list,
 *      which should therefore be in STORE mode.
 *
 *      Returns the no. of packages that were added (or
 *      0 if none).
 *
 *@@added V0.9.16 (2001-09-20) [umoeller]
 */

ULONG FEInstallEngine::CreateDeinstallPackagesList(list<FEDBPackage*> &DeinstallPackagesList)
{
    // go thru jobs
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    ULONG   c2BeRemoved = 0;
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJob;
        if (pJob = pTemp->IsInstallJob())
            if (pJob->Is2BeRemoved())
            {
                /* FEDBPackage *pDBPck = new FEDBPackage(&_Database,
                                                      *pJob->_pPackageIDInstalled);
                DeinstallPackagesList.push_back(pDBPck); */

                //@@todo this isn't working right, get the damn interfaces
                // to the database right
                DeinstallPackagesList.push_back((FEDBPackage*)pJob->_pDBPackageInstalled);

                ++c2BeRemoved;
            }
    }

    return c2BeRemoved;
}

/*
 *@@ LogRC:
 *
 *@@added V0.9.9 (2001-03-30) [umoeller]
 */

VOID LogRC(FELocals &Locals,
           int rc)
{
    switch (rc)
    {
        case CBRC_ABORT:
            Locals.Log("  abort requested!");
        break;

        case CBRC_SKIP:
            Locals.Log("  skipping existing");
        break;

        case CBRC_PROCEED:
            Locals.Log("  overwriting existing");
        break;

        case CBRC_RETRY:
            Locals.Log("  retrying");
        break;

        case CBRC_UNLOCK:
            Locals.Log("  unlocking");
        break;

        case CBRC_DEFER:
            Locals.Log("  deferring through CONFIG.SYS");
        break;

        default:
            Locals.Log("  unknown user return code %d", rc);
        break;
    }
}

/*
 *@@ ReportFileError:
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

int ReportFileError(FEInstallEngine *pEngine,
                    ULONG ulMsg,        // in: TMF msg no.
                    ustring *paStrings, // in: array of UTF-8 strings
                    ULONG ulTable,      // in: array item count
                    int iErrorRsp)      // in: possible response (CBREC_* flags)
{
    ustring ustrErrorMsg;
    pEngine->_Locals.GetMessage(ustrErrorMsg,
                                ulMsg,
                                paStrings,
                                ulTable);
    return pEngine->OnFileError(pEngine->_pCurrentInstallJob,
                                ustrErrorMsg,
                                iErrorRsp);
}

/*
 *@@ CBNextFile:
 *      implementation for CBM_NEXTFILE in EngineWICallback.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

int CBNextFile(FEInstallEngine *pEngine,
               WIFileHeader* pwifh)         // in: file header of current file
{
    int rc = CBRC_PROCEED;

    ++(pEngine->_ulCurrentFile);

    // store the new current file for later
    pEngine->_strLastFile.assign(pwifh->name);

    // increase the bytes we've done
    pEngine->_dBytesDone += pEngine->_dBytesOfLastFile;
    // remember the size of this file for later,
    // so we can calculate the total percentage
    pEngine->_dBytesOfLastFile = pwifh->origsize;

    // update logger
    CHAR    szPacked[30],
            szUnpacked[30];
    pEngine->_Locals.Log("Unpacking \"%s\" (%s -> %s bytes)",
                         pEngine->_strLastFile.c_str(),
                         nlsThousandsULong(szPacked,
                                           pwifh->compsize,
                                           pEngine->_Locals._cThousands),
                         nlsThousandsULong(szUnpacked,
                                           pwifh->origsize,
                                           pEngine->_Locals._cThousands));

    // update display
    pEngine->UpdateFile(pwifh,
                        pEngine->_ulCurrentFile,
                        pEngine->_ulFilesTotal);

    // check for whether the file exists
    /* FILE *TestFile = fopen (pwifh->name, "rb");
    if (TestFile != NULL)
    christ this is expensive V0.9.20 (2002-07-03) [umoeller]
    */

    APIRET arc;
    ULONG cb;
    if (!(arc = doshQueryPathSize(pwifh->name, &cb)))
    {
        // file exists:
        if (cb == 0)
        {
            pEngine->_Locals.Log("  overwriting existing 0-byte file");
        }
        else
        {
            // file exists and has more than 0 bytes:
            FEFileError fi(pEngine->_Locals,
                           NULL,        // get current dir
                           pwifh);

            // now check WPIGLOBALS for confirmation settings
            BOOL    fPrompt = FALSE;
            PULONG  pulQuery = 0;
            if (fi._iComp < 0)
                // existing is newer:
                pulQuery = &pEngine->_Locals._ulIfExistingNewer;
            else if (fi._iComp == 0)
                // same date:
                pulQuery = &pEngine->_Locals._ulIfSameDate;
            else
                // existing is older:
                pulQuery = &pEngine->_Locals._ulIfExistingOlder;

            switch (*pulQuery)
            {
                // 0: prompt
                // 1: skip
                // 2: overwrite
                case 0: fPrompt = TRUE; break;
                case 1: rc = CBRC_SKIP; break;
                case 2: rc = CBRC_PROCEED; break;
            }

            if (fPrompt)
            {
                pEngine->_Locals.Log("  file exists, prompting user");

                // OK, we should prompt the user:
                rc = pEngine->OnFileExists(&fi);
                        // this returns:
                        //      ERROR_PROCEED    replace this file
                        //      ERROR_SKIP       skip this file
            }

            LogRC(pEngine->_Locals, rc);
        }

        // fclose(TestFile);
    }

    // store the new current file in the
    // current package's PSZ file list for
    // the global database later
    if (pEngine->_pCurrentInstallJob)
        pEngine->_pCurrentInstallJob->AddFileHeader(pwifh);

    return rc;
}

/*
 *@@ CBWriteError:
 *      implementation for CBM_ERR_WRITE in EngineWICallback.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

int CBWriteError(FEInstallEngine *pEngine,
                 WIFileHeader* pwifh,         // in: file header of current file
                 short s)                     // in: error code
{
    int rc = CBRC_ABORT;

    APIRET  arc;

    // first check if maybe the write error was due
    // to a full disk, because in that case, it's
    // not ERROR_DISK_FULL which is returned, so
    // we better do this
    ULONG   ulDisk, ulMap;
    DosQueryCurrentDisk(&ulDisk, &ulMap);
    double d = 0;
    if (    (!(arc = doshQueryDiskFree(ulDisk, &d)))
         && (d < pwifh->origsize)
       )
    {
        // disk is full:
        // added this callback V0.9.20 (2002-07-03) [umoeller]
        CHAR cDisk = ulDisk + 'A' - 1;
        pEngine->_Locals.Log("  disk %c: is full", cDisk);
        pEngine->OnDiskFull(cDisk);
    }
    else
    {
        // get the DOS error number; since the back end
        // is OS-independent, we attempt to recreate the
        // error by opening the file for writing (99-10-26) [umoeller]
        CHAR    szFailingFile[2 * CCHMAXPATH];
        _getcwd(szFailingFile, sizeof(szFailingFile));
        strcat(szFailingFile, "\\");
        strcat(szFailingFile, pwifh->name);

        HFILE       hfFailing = 0;
        ULONG       ulAction = 0;
        if (!(arc = DosOpen(szFailingFile,
                            &hfFailing,
                            &ulAction,
                            0,             // new file size
                            FILE_NORMAL,
                            OPEN_ACTION_REPLACE_IF_EXISTS,
                            OPEN_FLAGS_FAIL_ON_ERROR
                               | OPEN_SHARE_DENYREADWRITE
                               | OPEN_ACCESS_READWRITE,
                            NULL)))
        {
            // no error: shouldn't happen, but just to make sure
            DosClose(hfFailing);
            // the GUI needs an error code, so we invent one
            arc = ERROR_WRITE_FAULT;
        }

        if (arc == ERROR_SHARING_VIOLATION)
        {
            // "file in use by another process":

            pEngine->_Locals.Log("  file is locked");

            // use special callback for this
            FEFileError fi(pEngine->_Locals,
                           NULL,        // get current dir
                           pwifh);

            rc = pEngine->OnFileLockedError(&fi,
                                            arc);

            LogRC(pEngine->_Locals, rc);

            switch (rc)
            {
                case CBRC_SKIP:
                    // ignore the error, go on (skip the file).
                    rc = CBRC_SKIP;
                break;

                case CBRC_RETRY:
                    // have the back end retry
                    rc = CBRC_RETRY;
                break;

                case CBRC_UNLOCK:
                    // unlock the file:
                    #ifndef __EMX__
                        // we use the undocumented DosReplaceModule API.
                        DosReplaceModule(fi._strFilename.c_str(), // old module
                                         NULL,              // new module
                                         NULL);             // backup module
                    #endif

                    // and have the back end retry now
                    rc = CBRC_RETRY;
                break;

                case CBRC_DEFER:
                    // have the install thread defer processing thru CONFIG.SYS
                    DosBeep(2000, 1000);
                    rc = CBRC_SKIP; break;
            }
        }
        else
        {
            pEngine->_Locals.Log("  write error %d occured", arc);

            // error other than "locked":
            PSZ pszSysError;
            ustring astr[2];
            if (pszSysError = doshQuerySysErrorMsg(s))
            {
                astr[0].assignCP(pEngine->_Locals._pCodecProcess,
                                 pszSysError);
                free(pszSysError);
            }

            astr[1].assignUtf8(pwifh->name);

            /*
                An error occured unpacking the current archive.
                Current file: "%2"
                OS/2 reported the following error message:
                %1
            */

            rc = ReportFileError(pEngine,
                                 118,
                                 astr,
                                 2,
                                 s);        // CBREC_*

            /* V0.9.20 (2002-07-03) [umoeller]
            rc = CallFileError(pEngine,
                               pwifh,
                               ECLS_DOS_WRITE,
                               arc,
                               s);
            */

            LogRC(pEngine->_Locals, rc);

            // if the user presses "Cancel" there,
            // that function will terminate everthing
        }
    }

    return rc;
}

/*
 *@@ CBZlibError:
 *      implementation for CBM_ERR_ZLIB in EngineWICallback.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

VOID CBZlibError(FEInstallEngine *pEngine,
                 short s)                     // in: error code
{
    pEngine->_Locals.Log("  ZLIB error occured");

    PSZ pszExpl = "Unknown error";

    switch (s)
    {
        case BZ_SEQUENCE_ERROR:
                pszExpl = "Invalid sequence"; break;
        case BZ_PARAM_ERROR:
                pszExpl = "Invalid parameter"; break;
        case BZ_MEM_ERROR:
                pszExpl = "Not enough memory"; break;
        case BZ_DATA_ERROR:         // -4
                pszExpl = "Invalid data (CRC mismatch)"; break;
        case BZ_DATA_ERROR_MAGIC:
                pszExpl = "Invalid data (magic bytes didn't match)"; break;
        case BZ_IO_ERROR:
                pszExpl = "I/O error"; break;
        case BZ_UNEXPECTED_EOF:
                pszExpl = "Unexpected end of file"; break;
        case BZ_OUTBUFF_FULL:
                pszExpl = "Not enough room in output buffer"; break;
        case BZ_CONFIG_ERROR:
                pszExpl = "Configuration error"; break;
        // the following are added by WIArchive
        case ZLIBERR_NO_WIFH_MAGIC:
                pszExpl = "Invalid file header format in archive; probably outdated archive format"; break;
    }

    CHAR szCode[20];
    ustring aStrings[2];
    aStrings[0].assignUtf8(pszExpl);
    aStrings[1]._itoa10(s, pEngine->_Locals._cThousands);

    ReportFileError(pEngine,
                    119,       // "Decompression engine reported..."
                    aStrings,
                    2,
                    CBREC_CANCEL);

    /* V0.9.20 (2002-07-03) [umoeller]
    rc = CallFileError(pEngine,
                       NULL,              // WIFileHeader, unknown
                       ECLS_DECOMP,
                       s,
                       CBREC_CANCEL);
                    // and stop!
    */
}


/*
 *@@ EngineWICallback:
 *      this is the standard callback function which is
 *      used with Install(), unless a different callback
 *      was specified.
 *
 *      This includes progress of the installation and all
 *      kinds of errors. This func is therefore pretty
 *      complex. Here we analyze the parameters and call the
 *      GUI callbacks accordingly.
 *
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this was WICallback in wpi_install_thread.cpp
 *@@changed V0.9.20 (2002-07-03) [umoeller]: reworked completely, great speed optimization
 */

int EngineWICallback(enCallbackMode mode,   // in: CBM_* flags def'd in wiarchive.h
                     short s,               // in: percentage of current file
                     WIFileHeader* pwifh,   // in: file header of current file
                     void *pvUser)          // in: user param (points to FEInstallEngine*)
{
    int rc = CBRC_ABORT;

    FEInstallEngine *pEngine = (FEInstallEngine*)pvUser;

    switch (mode)
    {
        /*
         * CBM_NEXTFILE:
         *      this comes just before the WIArchive class attempts to open
         *      a new output file for writing. "param" is always 0, "pwifh"
         *      points to the file header which is about to be opened.
         *      This allows the front-end to do two things:
         *
         *      a)   update the "current file" display by calling
         *           guiUpdateFile();
         *
         *      b)   check for whether that file exists already:
         *           -- if not, we return CBRC_PROCEED;
         *           -- if it does exist, we compare the existing date
         *              with the one in the archive and call guiFileExists()
         *              if G_WpiGlobals says we should prompt. In any case,
         *              we need to return CBRC_PROCEED (overwrite) or
         *              CBRC_SKIP.
         */

        case CBM_NEXTFILE:
            rc = CBNextFile(pEngine,
                            pwifh);
        break;

        /*
         * CBM_PERCENTAGE:
         *      in this case, "param" contains the percentage of the current
         *      file being processed, and pwifh the current file header.
         *      The return value doesn't matter.
         */

        case CBM_PERCENTAGE:
            // calculate the bytes we've really done so far
            // and call percentage callback (for progress bar)
            pEngine->UpdateBytes(
                        pEngine->_dBytesDone + ((pwifh->origsize * s) / 100),
                        pEngine->_dTotalBytesToDo);
        break;

        /*
         * CBM_ERR_WRITE:
         *      error writing into output file; this is most probable when the
         *      target disk is full.
         *      "param" then has one of the CBREC_* values, which identify
         *      whether the error is recoverable. Depending on that value,
         *      we may return one of the CBRC_* values.
         */

        case CBM_ERR_WRITE:
            rc = CBWriteError((FEInstallEngine*)pvUser,
                              pwifh,
                              s);
        break;

        /*
         * CBM_ERR_READ:
         *      error reading from input file; maybe the download is corrupt.
         *      "param" then has one of the CBREC_* values below, which identify
         *      whether the error is recoverable. Depending on that value,
         *      we may return one of the CBRC_* values.
         */

        case CBM_ERR_READ:
        {
            pEngine->_Locals.Log("  read error occured");

            ReportFileError(pEngine,
                            289, // The installation engine encountered a read error for an unknown reason. Installation cannot continue.
                            NULL, 0,
                            CBREC_CANCEL);

            /* V0.9.20 (2002-07-03) [umoeller]
                this never worked anyway
            rc = CallFileError(pEngine,
                               pwifh,
                               ECLS_DOS_READ,
                               _doserrno,
                               s); // CBREC_* param passed from WIArchive
                // if the user presses "Cancel" there,
                // that function will terminate everthing
            */
        }
        break;

        /*
         * CBM_ERR_ZLIB:
         *      error somewhere in the zlib decompression part of WIArchive;
         *      this usually signifies that the archive is broken. We then
         *      terminate installation, since there's no way to recover.
         *      "param" then contains one of the ZLIBERR_* flags below.
         */

        case CBM_ERR_ZLIB:
            CBZlibError(pEngine,
                        s);
        break;

        /*
         * default:
         *      some other error occured. This is probably
         *      just as serious, so we'll stop.
         */

        default:
            pEngine->_Locals.Log("  unknown error occured");
            ReportFileError(pEngine,
                            290, // The installation engine encountered an undefined error. Installation cannot continue. Please contact the WarpIN authors for a fix.
                            NULL,
                            0,
                            CBREC_CANCEL);
    }

    return rc;
}

/*
 *@@ Install:
 *      installs all packages that are on our install
 *      jobs list and have been selected for installation.
 *      If you call this from PM, put this on a separate
 *      thread, this can take a long time.
 *
 *      This can be used in two ways. If you set pWICallback
 *      to NULL, FEInstallEngine will use its own callback and
 *      call all the other callbacks instead, which must
 *      all be specified.
 *
 *      Otherwise, if pWICallback is non-NULL, all other
 *      callbacks are ignored, and only pWICallback is
 *      called.
 *
 *      In any case, you will get tons of callbacks. The
 *      callbacks run on the same thread that called this
 *      method. If this is not thread 1, the caller must
 *      watch out for synchronization himself.
 *
 *      After this, call FEInstallEngine::Configure.
 *
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_install_thread.cpp mostly
 *@@changed V0.9.20 (2002-07-03) [umoeller]: updates for callbacks as virtual methods
 */

VOID FEInstallEngine::Install(PFNWICALLBACK pWICallback)    // in: user callback or NULL
{
    // moved this here V0.9.20 (2002-07-03) [umoeller]
    UpdateGlobalStatus(GSTAT_UNPACKING);

    TRY_LOUD(excpt1)
    {
        if (pWICallback == NULL)
            pWICallback = EngineWICallback;

        // store those callbacks that are also needed by WICallback
        /* _pfnUpdateFile = pfnUpdateFile;
        _pfnUpdateBytes = pfnUpdateBytes;
        _pfnFileExists = pfnFileExists;
        _pfnFileError = pfnFileError;
        _pfnFileLockedError = pfnFileLockedError;
        */

        _ulPackagesTotal = 0;
        _ulFilesTotal = 0;
        _dTotalBytesToDo = 0;

        _ulCurrentPackage = 0;
        _dBytesThisPackage = 0;
        _pCurrentInstallJob = 0;

        _ulCurrentFile = 0;
        _dBytesOfLastFile = 0;
        _dBytesDone = 0;

        // first go thru the list of packages which are to
        // be installed to find out the total bytes
        // that will be installed
        // changed (99-10-31) [umoeller]
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis;
            if (pJobThis = pTemp->IsInstallJob())
            {
                if (pJobThis->Is2BeInstalled())
                {
                    ++_ulPackagesTotal;
                    _ulFilesTotal += pJobThis->QueryPckFilesCount();
                    _dTotalBytesToDo += pJobThis->QueryPckOrigSize();
                }
            }
        }

        if (    (_ulPackagesTotal)
             && (_dTotalBytesToDo)
           )
        {
            // alright, now let's go:
            // get current directory, which we need to switch
            // back to later
            CHAR    szStartDir[CCHMAXPATH];
            ULONG   cbStartDir = sizeof(szStartDir);        // fixed V0.9.13 (2001-06-21) [umoeller]
            DosQueryCurrentDir(0, szStartDir, &cbStartDir);

            CHAR sz[50];
            _Locals.Log("%d packages to install, %s uncompressed bytes total",
                                          _ulPackagesTotal,
                                          nlsThousandsDouble(sz,
                                                             _dTotalBytesToDo,
                                                             _Locals._cThousands));

            // go for installing the packages!
            JobStart = _pAllJobsList->begin();
            JobEnd = _pAllJobsList->end();
            for (; JobStart != JobEnd; ++JobStart)
            {
                FEJobBase *pTemp = *JobStart;
                FEInstallJob *pJobThis;
                if (pJobThis = pTemp->IsInstallJob())
                {
                    // to be installed, package, and selected?
                    if (pJobThis->Is2BeInstalled())
                    {
                        FEArcPackagePck *pPackageThis = (FEArcPackagePck*)&pJobThis->_ArcPackage;
                        string strTargetPath;
                        strTargetPath.assignUtf8(_Locals._pCodecProcess, pJobThis->QueryTargetPath());
                        PCSZ pcszTargetPath = strTargetPath.c_str();

                        // set global variables for this package
                        ++_ulCurrentPackage;
                        _dBytesThisPackage = pJobThis->QueryPckOrigSize();

                        CHAR sz[50];
                        _Locals.Log("Unpacking files in package \"%s\"...",
                                    pJobThis->QueryPckPackageID().GetBuffer());
                        _Locals.IncIndent(4);
                        _Locals.Log("Target path: \"%s\"",
                                    pcszTargetPath);
                        _Locals.Log("Uncompressed bytes: %s",
                                    nlsThousandsDouble(sz,
                                                       _dBytesThisPackage,
                                                       _Locals._cThousands));

                        // update gui status
                        UpdatePackage(pJobThis,
                                      _ulCurrentPackage,
                                      _ulPackagesTotal);

                        // create target directory
                        APIRET arc = NO_ERROR;

                        do
                        {
                            // create path,
                            // but only if it doesn't exist yet
                            if (!doshQueryDirExist(pcszTargetPath))
                            {
                                if (!(arc = doshCreatePath(pcszTargetPath,
                                                           FALSE)))        // not hidden
                                {
                                    // created:
                                    // append to list to be stored in database
                                    // V0.9.20 (2002-07-22) [umoeller]
                                    _Locals.Log("Created directory \"%s\"",
                                                pcszTargetPath);
                                    pJobThis->_ArcPackage._logDirsCreated.Append(pJobThis->QueryTargetPath());
                                }
                            }

                            if (    (!arc)
                                 && (!(arc = doshSetCurrentDir(pcszTargetPath)))
                               )
                            {
                                // set global variable to current package
                                // G_pCurrentArcPck = pPackageThis;
                                _pCurrentInstallJob = pJobThis;

                                // UNPACK THIS PACKAGE;
                                // this will keep calling the callback above
                                pJobThis->Unpack(pWICallback,
                                                 this);         // user param
                            }
                            else
                            {
                                _Locals.Log("Error creating directory \"%s\"",
                                            pcszTargetPath);

                                ustring ustrErrorMsg,
                                        str(pJobThis->QueryTargetPath());
                                _Locals.GetMessage(ustrErrorMsg,
                                                   134,   // The directory \"%s\" could not be created.
                                                   &str, 1);

                                OnFileError(_pCurrentInstallJob,
                                            ustrErrorMsg,
                                            CBREC_RETRYCANCEL);
                                    // if the user presses "Cancel" there,
                                    // that will terminate everthing
                            }
                        } while (arc != NO_ERROR);

                        _Locals.IncIndent(-4);
                        _Locals.Log("Done unpacking files in package \"%s\"",
                                             pJobThis->QueryPckPackageID().GetBuffer());

                    } // end if ((**PackageStart).ulType == PCK_PCK)
                }
            } // end for (; PackageStart != PackageEnd; ++PackageStart)

            // finally, set percentage to 100
            UpdateBytes(_dTotalBytesToDo,
                        _dTotalBytesToDo);

            DosSleep(300);
        }
    }
    CATCH (excpt1)
    {
        // exception occured:
        ustring str;
        str.assignUtf8("An exception occured in the WarpIn install thread. "
                       "A file named \"WPITRAP.LOG\" was created in the root "
                       "directory of your boot drive, which contains more information.");
        OnFileError(NULL,
                    str,
                    CBREC_CANCEL); // terminate only
    } END_CATCH();

    // moved this here from gui V0.9.20 (2002-07-03) [umoeller]
    DoneWithInstall();
}

/*
 *@@ Configure:
 *      performs system configuration after all files
 *      have been unpacked (FEInstallEngine::Install).
 *
 *      This calls FEInstallEngine::ConfigConfigSys, FEInstallEngine::ConfigWPSClasses,
 *      FEInstallEngine::ConfigCreateObjects, FEInstallEngine::ConfigExecutes, and
 *      FEInstallEngine::ConfigDeExecutes (in this order), if
 *      required by the packages being installed.
 *
 *      This can take a while and might therefore block
 *      the PM input queue.
 *      We probably should do this while creating WPS
 *      objects etc. in the first place. ;-)
 *
 *      After this, call FEInstallEngine::Store.
 *
 *@@changed V0.9.3 (2000-05-12) [umoeller]: added pszBackup
 *@@changed V0.9.5 (2000-08-26) [umoeller]: executes wasn't executed if ulAllowConfig was 0
 *@@changed V0.9.6 (2000-10-27) [umoeller]: made CONFIG.SYS backup part of WPIGLOBALS
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 */

BOOL FEInstallEngine::Configure(HAB hab)
{
    BOOL        brc = TRUE;

    ConfigConfigSys();
    ConfigWPSClasses();
    ConfigCreateObjects();
    ConfigWriteProfiles(hab);

    // run thru executes in any case
    // this checks G_WpiGlobals.ulAllowConfig for every exec
    ConfigExecutes();

    // and run thru de-executes, just to resolve the macros
    // V0.9.9 (2001-03-27) [umoeller]
    ConfigDeExecutes();

    return brc;
}

/*
 *@@ Store:
 *      after install is complete, this stores all
 *      new packages in the database. This is the
 *      final step in installation.
 *
 *      Returns the no. of packages stored.
 *
 *      If the special package index 30001 for
 *      WarpIN self-install was encountered,
 *      *ppSelfWarpINJob is set to the install
 *      job corresponding to that package.
 *
 *@@added V0.9.14 (2001-07-26) [umoeller]
 *@@changed V0.9.14 (2001-08-09) [umoeller]: added pfWarpINSelfInstalled
 */

ULONG FEInstallEngine::Store(FEInstallJob **ppSelfWarpINJob)
{
    ULONG ulrc = 0;

    UpdateGlobalStatus(GSTAT_UPDATINGDATABASE); // V0.9.20 (2002-07-03) [umoeller]

    *ppSelfWarpINJob = NULL;

    // new packages installed:
    // store all packages in the database
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
            if (pJobThis->Is2BeInstalled())
            {
                pJobThis->Store(_Database);
                ++ulrc;

                if (pJobThis->_ArcPackage._pDecl->_ulIndexInArchive
                        == WARPIN_MORE_PACKAGE_FOR_STUB)
                    // this is the special package index for
                    // WarpIN self-install:
                    // V0.9.14 (2001-08-09) [umoeller]
                    *ppSelfWarpINJob = pJobThis;
            }
    }

    UpdateGlobalStatus(GSTAT_DONEWITHALL);

    return ulrc;
}

/* ******************************************************************
 *
 *  Tons of helper methods
 *
 ********************************************************************/

/*
 *@@ wpiResolveMacros:
 *      this resolves the macros that many script tag attributes support.
 *      The given UTF-8 string was changed if something > 0 is returned.
 *
 *      This returns the number of macros resolved or 0 if no macros
 *      were found. In that case, ppszString is not changed.
 *
 *      The following macros can be resolved:
 *
 *      --  "?:\":      "?" is replaced with the OS/2 boot drive letter.
 *      --  "$(xxx)" (with "xxx" being a decimal number):
 *                      "xxx" is replaced with the target path of the package
 *                      with the index "xxx".
 *      --  "$(vendor\application\package)" is replaced with the target path
 *                      of the specified package, which must be installed.
 *      --  "$(env)" (with "env" being an environment variable):
 *                      "env" is replaced with the value of an environment variable.
 *                      If that variable is not found, the user is prompted to
 *                      enter one.
 *
 *@@added V0.9.2 (2000-03-11) [umoeller]
 *@@changed V0.9.3 (2000-04-28) [umoeller]: finally made "?:\" stuff work
 *@@changed V0.9.3 (2000-06-04) [umoeller]: added package ID support
 *@@changed V0.9.6 (2000-11-01) [umoeller]: added UNIX slash support
 *@@changed V0.9.9 (2001-04-06) [umoeller]: fixed broken ?:\
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be wpiResolveMacros
 *@@changed V0.9.16 (2001-10-25) [umoeller]: fixed bad macro resolution with "DEVICE=?:\" config.sys strings
 *@@changed V0.9.16 (2001-12-08) [umoeller]: fixed multiple VARPROMPTs for the same variable
 *@@changed V0.9.18 (2002-03-08) [umoeller]: now using ustring
 */

ULONG FEInstallEngine::ResolveMacros(ustring &ustr)
{
    ULONG   ulrc = 0;
    PSZ     pMacroBegin = NULL;

    if (ustr.size() > 2)        // was 3, fixed; this broke ?:\ macros
                                // V0.9.9 (2001-04-06) [umoeller]
    {
        // boot drive letter?
        /* if (memcmp(str.c_str(), "?:\\", 3) == 0)
            *((CHAR*)str.c_str()) = doshQueryBootDrive(); */
        // the above failed with CONFIG.SYS macro resolution because
        // I was assuming that ?:\ could only be present at the start
        // of the string. However, with CONFIG.SYS lines, we will
        // have "DEVICE=?:\", so this didn't work. Now using this:
        // V0.9.16 (2001-10-25) [umoeller]

        PSZ _p;
        if (_p = strstr(ustr.GetBuffer(), "?:\\"))
            *_p = doshQueryBootDrive();

        // environment variables or package paths
        while ((pMacroBegin = strstr(ustr.GetBuffer(), "$(")))
        {
            // macro found:
            PSZ     pEndOfMacro;
            if (pEndOfMacro = strchr(pMacroBegin, ')'))
            {
                // extract stuff from "$(" to ")"; this will be replaced
                ustring ustrFind;
                ustrFind.assignUtf8(pMacroBegin,
                                    pEndOfMacro + 1); // include ')'
                // extract stuff in brackets (between "$(" and ")")
                PSZ     pszInBrackets1 = strhSubstr(pMacroBegin + 2,
                                                    pEndOfMacro);  // exclude ')'

                BOOL    fResolved = FALSE;

                PSZ pInBrackets2 = pszInBrackets1;
                BOOL fUnixSlashes = FALSE;
                // check if first char is '/' --> UNIX slashes
                if (*pInBrackets2 == '/')
                {
                    fUnixSlashes = TRUE;
                    // skip that char
                    ++pInBrackets2;
                }

                // do we have a '\'? --> package path from database
                if (strchr(pInBrackets2, '\\'))
                {
                    // yes: it's a package ID
                    ustring ustr;
                    ustr.assignUtf8(pInBrackets2);
                    FEPackageID PckIDThis(_Locals,
                                          ustr, __FILE__, __LINE__,
                                          TRUE);    // allow short format!

                    const FEDBPackage *pDBPackage;
                    if (pDBPackage = _Database.FindDBPackage(PckIDThis))
                    {
                        // DB package found:

                        ustring ustrTargetPath(pDBPackage->QueryTargetPath());

                        if (fUnixSlashes)
                        {
                            // UNIX slashes: convert all "\" to "/"
                            ULONG ulPos = 0;
                            while (ustrTargetPath._find_replace('\\', '/', &ulPos)
                                    != string::npos)
                                ;
                        }

                        ULONG ulPos = 0;
                        ustr._find_replace(ustrFind,     // search string
                                           ustrTargetPath,    // replacement string
                                           &ulPos);

                        ++ulrc;     // replacement count

                        fResolved = TRUE;
                    }
                    else
                    {
                        // DB package not found:

                        ustring ustrID;
                        ustrID._printf("%s\\%s\\%s",
                                       PckIDThis._ustrAuthor.GetBuffer(),
                                       PckIDThis._ustrApplication.GetBuffer(),
                                       PckIDThis._ustrPackageName.GetBuffer());
                        throw FEFatalErrorExcpt(_Locals,
                                                189,
                                                &ustrID,
                                                1);
                    }
                }

                // not found yet?
                if (!fResolved)
                {
                    // all decimals? --> package number in archive
                    if (strhIsDecimal(pInBrackets2))
                    {
                        // OK, we appear to have a package index:
                        ULONG   ulIndex = 0;
                        sscanf(pInBrackets2, "%d", &ulIndex);
                        // find that package in G_WpiGlobals
                        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                                   JobEnd = _pAllJobsList->end();
                        for (; JobStart != JobEnd; ++JobStart)
                        {
                            FEJobBase *pTemp = *JobStart;
                            FEInstallJob *pJobThis;
                            if (pJobThis = pTemp->IsInstallJob())
                            {
                                if (pJobThis->_ArcPackage._PackHeader.number == ulIndex)
                                {
                                    ustring ustrTargetPath(pJobThis->QueryTargetPath());
                                    ULONG ulPos = 0;

                                    if (fUnixSlashes)
                                    {
                                        // UNIX slashes: convert all "\" to "/"
                                        while (ustrTargetPath._find_replace('\\', '/', &ulPos)
                                                    != string::npos)
                                            ;
                                        ulPos = 0;
                                    }

                                    // replace macro with target path
                                    ustr._find_replace(ustrFind,     // search string
                                                       ustrTargetPath,   // replacement string
                                                       &ulPos);

                                    fResolved = TRUE;

                                    ++ulrc;     // replacement count

                                    break;  // for PckStart
                                }
                            }
                        }
                    }
                }

                // @@todo Christian: this displays a varprompt if $(1) is
                // not defined. that is not really helpful.

                if (!fResolved)
                {
                    // finally, attempt to resolve an environment variable
                    PCSZ pszValue;
                    if (!(pszValue = getenv(pszInBrackets1)))
                    {
                        // variable not defined:
                        // check if we have a VARPROMPT for this
                        FEVarPrompt VP(pszInBrackets1);
                        FEVarPrompt *pVP;
                        if (pVP = FEScriptBase::FindVarPrompt(pszInBrackets1))
                            VP = *pVP;

                        // now prompt
                        pszValue = VarPrompt(VP);
                                // can be NULL

                        // V0.9.16 (2001-12-08) [umoeller]:
                        // set this in the environment so we won't prompt again
                        // if the variable is used the next time; we must
                        // allocate a new string on the heap or the variable
                        // will not be stored properly
                        string *pstrNew = new string(pszInBrackets1);
                        pstrNew->append("=");
                        pstrNew->append(pszValue);
                        putenv(pstrNew->c_str());
                    }

                    // replace macro with value
                    ustring ustrValue(_Locals._pCodecProcess,
                                      pszValue);
                    ULONG ulPos = 0;
                    ustr._find_replace(ustrFind,        // search string
                                       ustrValue,       // replacement string
                                       &ulPos);

                    ++ulrc;     // replacement count
                }

                free(pszInBrackets1);
            } // end if (pEndOfMacro)
            else
                break;

            // OK, search next macro;
            // since the macro search string has been replaced
            // with real text, we can safely search the whole
            // buffer again
            // (and we must, because pszCurrent points to an
            // all new buffer)

        } // end while (pMacro = strstr(*ppszString, "$(");)
    }

    return ulrc;
}

/*
 *@@ ResolveMacros:
 *      overloaded version for freeable string buffers.
 *
 *      WARNING: It is assumed that the given string is
 *      a UTF-8 string. No check is made against this.
 *
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be wpiResolveMacros
 */

ULONG FEInstallEngine::ResolveMacros(char **ppsz)
{
    ULONG ulrc = 0;
    ustring strTemp;
    strTemp.assignUtf8(*ppsz);
    free(*ppsz);
    ulrc = ResolveMacros(strTemp);
    *ppsz = strdup(strTemp.GetBuffer());

    return ulrc;
}

/*
 *@@ FindPackageInArchives:
 *      this goes thru all currently open archives
 *      to find a package with the specified ID.
 *      See FEPackage::IsInArchive for the parameters,
 *      which gets called for all currently open
 *      archives by this helper function.
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 *@@changed V0.9.18 (2002-02-06) [umoeller]: now using FEPackageID
 */

FEArcPackagePck* FEInstallEngine::FindPackageInArchives(const FEPackageID &PckID,
                                                        PULONG pulComp)
                                  const
{
    FEArcPackagePck *pArcReturn = 0;

    list<FEArchive*>::iterator ArcStart = _pArchivesList->begin(),
                               ArcEnd = _pArchivesList->end();
    for (; ArcStart != ArcEnd; ++ArcStart)
    {
        FEArchive *pArcThis = *ArcStart;

        if ( (pArcReturn = pArcThis->IsInArchive(PckID, pulComp)) )
            break;
    }

    return pArcReturn;
}

/*
 *@@ FindInstallJob:
 *      since there is no reverse linkage from "package"
 *      packages to install jobs, this helper function
 *      can be used to find the install job for a
 *      given "package" package.
 *
 *      Returns NULL if not found.
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 */

FEInstallJob* FEInstallEngine::FindInstallJob(const FEArcPackagePck *pArcPackage)
                               const
{
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis = pTemp->IsInstallJob();
        if (pJobThis)
        {
            if (&pJobThis->_ArcPackage == pArcPackage)
            {
                return pJobThis;
            }
        }
    }

    return NULL;
}

/*
 *@@ FindInstallJob:
 *      second version of wpiFindInstallJob which
 *      takes a five- or six-part package ID as
 *      input instead.
 *
 *      Also returns NULL if not found.
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 *@@changed V0.9.18 (2002-03-08) [umoeller]: rewritten
 */

FEInstallJob* FEInstallEngine::FindInstallJob(const FEPackageID &PckID)
                               const
{
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            if (pJobThis->_ArcPackage._PckID.Compare(PckID))
                return pJobThis;
        }
    }

    return NULL;

    /*
    FEInstallJob    *pJobReturn = NULL;
    ULONG           ulComp = 0;
    // 1) find package with that ID in open archives
    FEArcPackagePck *pPckInArc;
    if (pPckInArc = FindPackageInArchives(PckID,
                                          &ulComp))
    {
        // found: check all deinstall jobs whether they
        // match that package
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis = pTemp->IsInstallJob();
            if (pJobThis)
            {
                if (pJobThis->_pArcPackage == pPckInArc)
                {
                    // found: return that
                    pJobReturn = pJobThis;
                    break;
                }
            }
        }
    }

    return pJobReturn; */
}

/*
 *@@ FindGroupJob:
 *      since there is no reverse linkage from group
 *      packages to group jobs, this helper function
 *      can be used to find the group job for a group's
 *      member package.
 *
 *      Returns NULL if not found.
 *
 *@@added V0.9.0 (99-11-06) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 */

FEGroupJob* FEInstallEngine::FindGroupJob(const FEArcPackageGroup *pGroupPackage)
                             const
{
    list<FEJobBase*>::iterator JobStart2 = _pAllJobsList->begin(),
                               JobEnd2 = _pAllJobsList->end();
    for (; JobStart2 != JobEnd2; ++JobStart2)
    {
        FEJobBase *pTemp = *JobStart2;
        FEGroupJob *pJobThis2;
        if (pJobThis2 = pTemp->IsGroupJob())
            if (pJobThis2->_pGroupPackage == pGroupPackage)
                return pJobThis2;
    }

    return NULL;
}

/*
 *@@ Is2BeRemoved:
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 *@@changed V0.9.18 (2002-02-06) [umoeller]: now using FEPackageID for speed
 */

FEInstallJob* FEInstallEngine::Is2BeRemoved(const FEPackageID &PckID)
                               const
{
    FEInstallJob *pJob;
    if (pJob = FindInstallJob(PckID))
        if (pJob->Is2BeRemoved())
            return pJob;

    return NULL;
}

/*
 *@@ CheckJobs:
 *      finds out what needs to be done for the actual
 *      install.
 *
 *      This returns the no. of packages which
 *      are selected for installation or
 *      deinstallation as well as the system
 *      configuration which needs to be done.
 *
 *      The config data is the sum of all packages,
 *      masked by WPIGLOBALS.ulDoConfiguration.
 *
 *      Any of the pointers can be NULL if you don't
 *      need that info.
 *
 *      NOTE: This function does _not_ initialize
 *      the ULONGs. Set these to 0 yourself before
 *      calling this.
 *
 *@@added V0.9.0 (99-11-02) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 */

VOID FEInstallEngine::CheckJobs(PULONG pulToAdd,      // out: packages to install
                                PULONG pulToRemove)   // out: packages to remove
{
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            if (pJobThis->Is2BeInstalled())
            {
                if (pulToAdd)
                    ++(*pulToAdd);
                // if (plistVariables)
                    /* (*pulConfig) |= ((pJobThis->_ArcPackage._pDecl->_ulConfigData)
                                     & _ulAllowConfig); */
                   //  *plistVariables = &_listVariables;
            }
            else if (pJobThis->Is2BeRemoved())
                if (pulToRemove)
                    ++(*pulToRemove);
        }
    }
}

/*
 *@@ FindInstallVar:
 *      returns a pointer to the install variable for
 *      the given key name.
 *
 *      It returns NULL
 *
 *      --  if no such variable exists, or
 *
 +      --  if the variable exists, but is not currently
 *          used by any job's package (which can only be
 *          true for the system install variables), or
 *
 *      --  if it is used, but the job is not currently
 *          selected for installation.
 *
 *@@added V0.9.20 (2002-07-03) [umoeller]
 */

FEInstallVar* FEInstallEngine::FindInstallVar(PCSZ pcszKeyNameA)
{
     // variable is set: then check if it is
     // used by any job that is currently selected
     list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                JobEnd = _pAllJobsList->end();
     for (; JobStart != JobEnd; ++JobStart)
     {
         FEJobBase *pTemp = *JobStart;
         FEInstallJob *pJobThis;
         if (pJobThis = pTemp->IsInstallJob())
         {
             if (pJobThis->Is2BeInstalled())
             {
                 FEPckDeclBase *pDecl = pJobThis->_ArcPackage._pDecl;
                 list<FEInstallVar*>::iterator
                                     declBegin = pDecl->_listVariables.begin(),
                                     declEnd = pDecl->_listVariables.end();
                 for (;
                      declBegin != declEnd;
                      ++declBegin)
                 {
                     FEInstallVar *pVar = *declBegin;
                     if (pVar->_strVarName == pcszKeyNameA)
                         // found: return it (the list only holds
                         // a pointer to the global instance)
                         return pVar;
                 }
             }
         }
     }

     return NULL;
}

/*
 *@@ GetInstallVars:
 *      fills the given list with all install
 *      variables that are currently "in use",
 *      i.e. touched by at least one package
 *      that is currently selected.
 *
 *      The list must be in SHADOW mode.
 *
 *      This returns both "set" and "clear"
 *      variables.
 *
 *      Returns the no. of items appended to
 *      the list.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

ULONG FEInstallEngine::GetInstallVars(list<FEInstallVar*> &listVariables,
                                      BOOL fPublicsOnly)
{
    ULONG ulrc = 0;

    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            if (pJobThis->Is2BeInstalled())
            {
                FEPckDeclBase *pDecl = pJobThis->_ArcPackage._pDecl;
                list<FEInstallVar*>::iterator
                                    declBegin = pDecl->_listVariables.begin(),
                                    declEnd = pDecl->_listVariables.end();
                for (;
                     declBegin != declEnd;
                     ++declBegin)
                {
                    FEInstallVar *pVar = *declBegin;

                    if (    (!fPublicsOnly)
                         || (pVar->IsPublic())
                       )
                    {
                        // append this to the list, but only once
                        BOOL fFound = FALSE;
                        list<FEInstallVar*>::iterator
                                   listBegin = listVariables.begin(),
                                   listEnd = listVariables.end();
                        for (;
                             listBegin != listEnd;
                             ++listBegin)
                        {
                            FEInstallVar *pVarThis = *listBegin;
                            if (pVar->_strVarName == pVarThis->_strVarName)
                            {
                                fFound = TRUE;
                                break;
                            }
                        }

                        if (!fFound)
                        {
                            listVariables.push_back(pVar);
                            ++ulrc;
                        }
                    }
                }
            }
        }
    }

    return ulrc;
}

/*
 *@@ QueryInstallVar:
 *      returns TRUE only if
 *
 *      1)  the given install variable is used by
 *          any of the packages which are currently
 *          selected for installation _and_
 *
 *      2)  the variable value is TRUE.
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

BOOL FEInstallEngine::QueryInstallVar(FEInstallVar &var)
{
    // if the variable isn't set, we can stop
    if (var.GetValue())
    {
        // variable is set: then check if it is
        // used by any job that is currently selected
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis;
            if (pJobThis = pTemp->IsInstallJob())
            {
                if (pJobThis->Is2BeInstalled())
                {
                    FEPckDeclBase *pDecl = pJobThis->_ArcPackage._pDecl;
                    list<FEInstallVar*>::iterator
                                        declBegin = pDecl->_listVariables.begin(),
                                        declEnd = pDecl->_listVariables.end();
                    for (;
                         declBegin != declEnd;
                         ++declBegin)
                    {
                        FEInstallVar *pVar = *declBegin;
                        if (pVar->_strVarName == var._strVarName)
                            // found:
                            return TRUE;
                    }
                }
            }
        }
    }

    return FALSE;
}

/*
 *@@ CheckDependencies:
 *      this goes thru the install jobs list to
 *      check if all package requirements are met.
 *      This prompts for all missing packages:
 *
 *      -- if a missing package is in the archive,
 *         the user is prompted whether it should
 *         be selected;
 *
 *      -- if it's not, an error message is displayed
 *         that the package could not be found.
 *
 *      This returns TRUE only if all requirements
 *      are met.
 *
 *      This gets called automatically from FEInstallEngine::VerifySelections,
 *      i.e. when the "Container" page is left, so if FALSE
 *      is returned, the Container page is not left.
 *
 *      Throws:
 *      -- FEFatalErrorExcpt.
 *
 *@@added V0.9.1 (2000-01-08) [umoeller]
 *@@changed V0.9.6 (2000-11-09) [umoeller]: version checks were missing, fixed.
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 *@@changed V0.9.18 (2002-02-06) [umoeller]: major speed optimizations
 */

BOOL FEInstallEngine::CheckDependencies()
{
    BOOL    rc = TRUE;
    BOOL    fCheckAgain = FALSE;

    do // this loops if any package selections are changed
    {
        fCheckAgain = FALSE;

        // results:
        ustring strMissing;
                // string of packages which are required by archive,
                // but not accessible (error)
        list<FEJobBase*> listJobsRequiredByArchive(SHADOW);
                // list of archive packages which are required by other
                // packages in the archive
        ustring strRequiredByArchive;
                // string list for that
        list<FEJobBase*> listJobsRequiredByDatabase(SHADOW);
                // list of archive packages which are required by the
                // database
        ustring strRequiredByDatabase;
                // string list for that

        // go thru all jobs
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis;
            if (pJobThis = pTemp->IsInstallJob())
            {
                // 1) if package has been selected for installation,
                //    check if requirements are met for that package

                if (pJobThis->Is2BeInstalled())
                {
                    // job is to be installed (selected):
                    // check required packages for that job
                    const FEPackageBase *pPckThis = &pJobThis->_ArcPackage;
                    // decode "requires" logger strings

                    // replaced all the following with the PackageIDs
                    // from the list in the package
                    // V0.9.18 (2002-02-06) [umoeller]

                    // PSZ     pszLogStart = pPckThis->_logRequiresIDs._pabLogString,
                    //         pszRequiresIDThis = pszLogStart;

                    // while (pszRequiresIDThis
                       //     < (pszLogStart + pPckThis->_logRequiresIDs._cbLogString))

                    list<FEPackageID*>::iterator pckidStart = pPckThis->_RequiresIDs.begin(),
                                                 pckidEnd = pPckThis->_RequiresIDs.end();

                    for (;
                         pckidStart != pckidEnd;
                         ++pckidStart)
                    {
                        ULONG   ulComp = 0;
                        BOOL    fNeeds2BeSelected = FALSE;

                        FEPackageID *pRequiredPckIDThis = *pckidStart;

                        // pszRequiresIDThis now has the ID of the package
                        // which is required:
                        // first check if it's in the database maybe

                        // FEPackageID RequiredPckIDThis(_Locals,
                        //                               pszRequiresIDThis, __FILE__, __LINE__,
                        //                               FALSE);    // don't allow short format

                        /* FEPackageID *_pPackageIDInstalled
                                = _Database.FindPackageID(pRequiredPckIDThis->_strAuthor.c_str(),
                                                          pRequiredPckIDThis->_strApplication.c_str(),
                                                          pRequiredPckIDThis->_strPackageName.c_str()); */

                        const FEDBPackage *pDBPckInstalled;

                        if (!(pDBPckInstalled = _Database.FindDBPackage(*pRequiredPckIDThis)))
                        // if (!_pPackageIDInstalled)
                            // required, but not installed:
                            fNeeds2BeSelected = TRUE;
                        else
                        {
                            // installed: check if maybe it's selected for
                            // de-installation, which ain't good either
                            if (Is2BeRemoved(*pRequiredPckIDThis))
                                fNeeds2BeSelected = TRUE;
                            else
                            {
                                // check if installed version level is sufficient
                                // V0.9.6 (2000-11-06) [umoeller]
                                if (    (pDBPckInstalled->_PckID.Compare(*pRequiredPckIDThis)
                                               & IDCOMP_VERSIONMASK)
                                        == IDCOMP_THISOLDER
                                   )
                                {
                                    // installed package is too old:
                                    fNeeds2BeSelected = TRUE;
                                }
                            }
                        }

                        if (fNeeds2BeSelected)
                        {
                            // not installed or too old:
                            FEArcPackagePck *pPckInArc;
                            if (pPckInArc = FindPackageInArchives(*pRequiredPckIDThis,
                                                                  &ulComp))
                            {
                                // required package is in archive:
                                // find job for that
                                FEInstallJob *pJob2BeSelected = FindInstallJob(pPckInArc->_PckID);
                                /* if (!pJob2BeSelected)
                                    throw FEFatalErrorExcpt("job not found");
                                */

                                // is job selected?
                                if (!pJob2BeSelected->Is2BeInstalled())
                                {
                                    // no: add to list
                                    if (strRequiredByArchive())
                                        strRequiredByArchive.appendUtf8("\n");
                                    ustring strTemp;
                                    ustring astr[2] =
                                        {
                                            pJobThis->_ArcPackage.QueryTitle(),
                                            pPckInArc->QueryTitle()
                                        };
                                    _Locals.GetMessage(strTemp,
                                                         259, // Package "%1" requires "%2"
                                                         astr,
                                                         2);
                                    strRequiredByArchive += strTemp;
                                    // store the job of the missing package
                                    // in the list of items to be selected
                                    listJobsRequiredByArchive.push_back(pJob2BeSelected);
                                }
                            }
                            else
                            {
                                // required package is NOT in archive:
                                // this will cause an error below
                                ustring strID;
                                pRequiredPckIDThis->CreateStringSix(strID);
                                if (strMissing())
                                    strMissing.appendUtf8("\n");
                                ustring strTemp;
                                ustring astr[2] =
                                    {
                                        pJobThis->_ArcPackage.QueryTitle(),
                                        strID
                                    };
                                _Locals.GetMessage(strTemp,
                                                   259, // Package "%1" requires "%2"
                                                   astr,
                                                   2);
                                strMissing += strTemp;
                            }
                        } // end if (!_pPackageIDInstalled)

                        // pszRequiresIDThis += cbLogThis + 1;    // go beyond null byte
                    } // end while (pszRequiresIDThis < pszLogStart + pPckThis->_logRequiresIDs._cbLogString)
                } // end if (pJobThis->Is2BeInstalled())

                // 2) if job has been selected for de-installation,
                //    check if maybe some other package requires this

                else if (pJobThis->Is2BeRemoved())
                {
                    // check if a package in the database
                    // requires package which is to be deinstalled
                    /* FEPackageID PckIDThis(_Locals,
                                          pJobThis->_pArcPackage, __FILE__, __LINE__,
                                          FALSE);    // don't allow short format
                       */
                    FEPackageID &PckIDThis = pJobThis->_ArcPackage._PckID;

                    // enumerate all packages in the database
                    // const string *pstrDBPckIDThis = _Database.EnumPackages(NULL);
                    // while (pstrDBPckIDThis)

                    list<FEDBPackage*>::iterator dbBegin = _Database._DBPackagesList.begin(),
                                                 dbEnd = _Database._DBPackagesList.end();

                    for (;
                         dbBegin != dbEnd;
                         ++dbBegin)
                    {
                        // get database package from that ID
                        /* FEPackageID DBPckIDThis(_Locals,
                                                pstrDBPckIDThis->c_str(), __FILE__, __LINE__,
                                                FALSE);    // don't allow short format
                        const FEDBPackage *pDBPckThis
                            = _Database.FindDBPackage(DBPckIDThis);
                        if (!pDBPckThis)
                            throw FEFatalErrorExcpt("database package not found");
                        */

                        const FEDBPackage *pDBPckThis = *dbBegin;
                        const FEPackageID &DBPckIDThis = pDBPckThis->_PckID;

                        // decode "requires" logger strings for that DBPackage
                        // PSZ     pszLogStart = pDBPckThis->_logRequiresIDs._pabLogString,
                        //         pszDBRequiresIDThis = pszLogStart;
                        // int     iCount = 0;

                        // while (pszDBRequiresIDThis < pszLogStart + pDBPckThis->_logRequiresIDs._cbLogString)

                        list<FEPackageID*>::iterator pckidStart = pDBPckThis->_RequiresIDs.begin(),
                                                     pckidEnd = pDBPckThis->_RequiresIDs.end();

                        for (;
                             pckidStart != pckidEnd;
                             ++pckidStart)
                        {
                            // ULONG   cbLogThis = strlen(pszDBRequiresIDThis);

                            // pszDBRequiresIDThis now has a requirement for the DB package;
                            // check if that ID is the same as that of the job which is
                            // to be removed
                            FEPackageID *pDBRequiresIDThis = *pckidStart;

                            // FEPackageID DBRequiresIDThis(_Locals,
                            //                              pszDBRequiresIDThis, __FILE__, __LINE__,
                            //                              FALSE);    // don't allow short format

                            if (pDBRequiresIDThis->Compare(PckIDThis))
                            {
                                // matches:
                                // check if the DB package which requires that package
                                // has also been selected for removal
                                if (!Is2BeRemoved(DBPckIDThis))
                                {
                                    // no: this means that
                                    // required package is to be removed:
                                    FEInstallJob *pJob2BeRemoved = Is2BeRemoved(*pDBRequiresIDThis);

                                    // check if the requiring database package
                                    // is also in the archive
                                    FEInstallJob *pRemovalJobInArchive;
                                    if (pRemovalJobInArchive = FindInstallJob(DBPckIDThis))
                                    {
                                        // yes: add this to the regular strRequiredByArchive
                                        // strings
                                        if (strRequiredByArchive())
                                            strRequiredByArchive.appendUtf8("\n");

                                        ustring strTemp;
                                        ustring astr[2] =
                                            {
                                                pRemovalJobInArchive->_ArcPackage.QueryTitle(),
                                                pJob2BeRemoved->_ArcPackage.QueryTitle()
                                            };
                                        _Locals.GetMessage(strTemp,
                                                             259, // Package "%1" requires "%2"
                                                             astr,
                                                             2);
                                        strRequiredByArchive += strTemp;
                                        // store the job of the missing package
                                        // in the list of items to be selected
                                        listJobsRequiredByArchive.push_back(pJob2BeRemoved);
                                    }
                                    else
                                    {
                                        // not in archive:
                                        // add to "required by database" strings
                                        if (strRequiredByDatabase())
                                            strRequiredByDatabase.appendUtf8("\n");
                                        ustring strTemp;
                                        ustring astr[2] =
                                            {
                                                pDBPckThis->_pDecl->_ustrID,
                                                pJob2BeRemoved->_ArcPackage.QueryTitle(),
                                            };
                                        _Locals.GetMessage(strTemp,
                                                             260, // Database package "%1" requires "%2"
                                                             astr,
                                                             2);
                                        strRequiredByDatabase += strTemp;
                                        // store the job of the missing package
                                        // in the list of items to be selected
                                        listJobsRequiredByDatabase.push_back(pJob2BeRemoved);
                                    }
                                }
                            }

                            // pszDBRequiresIDThis += cbLogThis + 1;    // go beyond null byte
                        }

                        // next package in the database
                        // pstrDBPckIDThis = _Database.EnumPackages(pstrDBPckIDThis);
                    }
                }
            }
        } // end for (; JobStart != JobEnd; ++JobStart)

        // 3) report results

        // check missing packages first; this is an error
        if (strMissing())
        {
            _Locals.MessageBox(102,       // error
                                 258,
// There are packages in the archive which require other packages to be installed:
// %1
// Those packages were not found in the archive. WarpIN cannot continue.
                                 MSG_ERROR_ABORT,
                                 &strMissing,
                                 1);
            /* BSString strMsg("There are packages in the archive which require other packages to be installed:\n");
            strMsg += strMissing
                      + "\n\nThose packages were not found in the archive. WarpIN cannot continue.";
            _Locals.ShowMessage("WarpIN",
                                strMsg,
                                MSG_WARNING_OK);
                    // @@todo this isn't good
            */
            rc = FALSE;
        }

        if (rc)
        {
            // report archive packages which are required within archive
            if (strRequiredByArchive())
            {
                /* BSString strMsg("There are packages in the archive which require other packages to be installed:\n");
                strMsg += strRequiredByArchive
                          + "\n\nShould WarpIN select these packages for you?";
                if (_Locals.ShowMessage("WarpIN",
                                        strMsg,
                                        MSG_CONFIRM_YESNO_DEFYES)
                        == MBID_YES)
                */

                if (_Locals.MessageBox(102,       // error
                                         261,
// There are packages in the archive which require other packages to be installed:
// %1
// Should WarpIN select these packages for you?
                                         MSG_CONFIRM_YESNO_DEFYES,
                                         &strRequiredByArchive,
                                         1)
                        == MBID_YES)
                {
                    JobStart = listJobsRequiredByArchive.begin();
                    JobEnd = listJobsRequiredByArchive.end();
                    for (; JobStart != JobEnd; ++JobStart)
                    {
                        (**JobStart).Select(JOB_INSTALL);
                        fCheckAgain = TRUE; // loop again, because selections have changed
                    }
                }

                // report "errors found"
                rc = FALSE;
            }
        }

        if (rc)
        {
            // report archive packages which are required by database
            if (strRequiredByDatabase())
            {
                /* BSString strMsg("You have selected packages for removal which are required by packages in the database:\n");
                strMsg += strRequiredByDatabase
                          + "\n\nShould WarpIN deselect removal of these packages for you?";
                if (_Locals.ShowMessage("WarpIN", strMsg, MSG_CONFIRM_YESNO_DEFYES)
                        == MBID_YES)
                */

                if (_Locals.MessageBox(102,
                                         262,
// You have selected packages for removal which are required by packages in the database:
// %1
// Should WarpIN deselect removal of these packages for you?
                                         MSG_CONFIRM_YESNO_DEFYES,
                                         &strRequiredByDatabase,
                                         1)
                        == MBID_YES)
                {
                    JobStart = listJobsRequiredByDatabase.begin();
                    JobEnd = listJobsRequiredByDatabase.end();
                    for (; JobStart != JobEnd; ++JobStart)
                    {
                        (**JobStart).Select(JOB_INSTALL_IGNORE);
                        fCheckAgain = TRUE; // loop again, because selections have changed
                    }
                }

                // report "errors found"
                rc = FALSE;
            }
        }
    } while (fCheckAgain);

    return rc;
}

/*
 *@@ VerifySelections:
 *      helper function which should get called from
 *      the GUI when leaving the "Container" page to
 *      check if all package selections are valid.
 *
 *      If WARPIN_DISPLAYPAGES=NO, this gets called
 *      from EnterArchiveMode instead.
 *
 *      This calls FEInstallEngine::CheckDependencies in turn.
 *
 *      This also prompts for the directories to be
 *      created, if (fPromptCreateDirs == TRUE).
 *
 *      This returns TRUE if no errors were found
 *      and the GUI should continue, or FALSE
 *      if the GUI should stay on the "Container"
 *      page.
 *
 *@@changed V0.9.0 (99-11-01) [umoeller]: this was guiCheckPackages
 *@@changed V0.9.0 (99-11-04) [umoeller]: complained about missing directories even with deselected packages; fixed
 *@@changed V0.9.1 (2000-01-08) [umoeller]: added call to wpiCheckDependencies
 *@@changed V0.9.1 (2000-01-08) [umoeller]: fixed duplicate directory reports
 *@@changed V0.9.14 (2001-07-14) [umoeller]: now using string maps for dirs
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 *@@changed V0.9.20 (2002-07-03) [umoeller]: moved check for "plenty of space" to GUI
 */

BOOL FEInstallEngine::VerifySelections(BOOL fPromptCreateDirs)
{
    // string maps for invalid and missing dirs;
    // by using a map, we automatically make
    // sure that each dir only gets added once
    // V0.9.14 (2001-07-14) [umoeller]
    BSMap<StringMapEntry*, CompareIStrings>
            mapInvalidDirs,
            mapMissingDirs;

    BOOL    fInvalidDirsFound = FALSE;
    BOOL    fPackagesToAdd = FALSE,
            fPackagesToRemove = FALSE,
            fDirsMissing = FALSE,
            fAnyDowngrades = FALSE;

    if (!CheckDependencies())
        return FALSE;

    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis = pTemp->IsInstallJob();
        if (pJobThis)
        {
            if (pJobThis->Is2BeInstalled())
            {
                fPackagesToAdd = TRUE;

                BOOL    fInvalidThis = FALSE;
                string strTargetPath(_Locals._pCodecProcess,
                                     pJobThis->QueryTargetPath());

                // check target paths
                if (strTargetPath.size() < 3) // V0.9.9 (2001-03-23) [umoeller]
                    fInvalidThis = TRUE;
                else if (strTargetPath[1] != ':')
                    fInvalidThis = TRUE;
                else if (strTargetPath[2] != '\\')
                    fInvalidThis = TRUE;
                else if (doshIsValidFileName(strTargetPath.c_str(), TRUE))
                    fInvalidThis = TRUE;

                if (fInvalidThis)
                {
                    mapInvalidDirs.insert(new StringMapEntry(strTargetPath, ""));
                    fInvalidDirsFound = TRUE;
                }
                else if (fPromptCreateDirs)
                {
                    // dir OK, but missing:
                    if (!doshQueryDirExist(strTargetPath.c_str()))
                    {
                        mapMissingDirs.insert(new StringMapEntry(strTargetPath, ""));
                        fDirsMissing = TRUE;
                    }
                }

                if (pJobThis->_ulInstallStatus == INSTALLED_IS_NEWER)
                    fAnyDowngrades = TRUE;
            }
            else if (pJobThis->Is2BeRemoved())
                fPackagesToRemove = TRUE;
        }
    } // end for

    if (fAnyDowngrades)
    {
        // selected packages are old than the installed versions:
        if (_Locals.MessageBox(108,       // "Warning"
                               187,       // "downgrading!"
                               MSG_CONFIRM_YESNO_DEFYES)
                != MBID_YES)
            // report error
            return FALSE;

    }

    if ( (!fPackagesToAdd) && (!fPackagesToRemove) )
    {
        // no packages selected for installation/deinstallation:
        // this doesn't make any sense
        _Locals.MessageBox(102,       // "Error"
                           126,       // "no packages"
                           MSG_WARNING_OK);
        // report error
        return FALSE;
    }

    if (fInvalidDirsFound)
    {
        // compose message
        ustring strInvalidDirs;
        StringMapEntry *p;
        for (p = mapInvalidDirs.first();
             p;
             p = mapInvalidDirs.next(p))
        {
            strInvalidDirs.appendUtf8("\n    ");
            ustring ustr;
            ustr.assignCP(_Locals._pCodecProcess,
                          p->_strKey);
            strInvalidDirs += ustr;
        }

        _Locals.MessageBox(114,    // "invalid directories"
                             115,
                             MSG_WARNING_OK,
                             &strInvalidDirs, 1);
        // report error
        return FALSE;
    }

    if (fDirsMissing)
    {
        // compose message
        ustring strMissingDirs;
        StringMapEntry *p;
        for (p = mapMissingDirs.first();
             p;
             p = mapMissingDirs.next(p))
        {
            strMissingDirs.appendUtf8("\n    ");
            ustring ustr;
            ustr.assignCP(_Locals._pCodecProcess,
                          p->_strKey);
            strMissingDirs.append(ustr);
        }

        if (_Locals.MessageBox(116,    // "create directories"
                               117,
                               MSG_CONFIRM_YESNO_DEFYES,
                               &strMissingDirs, 1)
                    != MBID_YES)
            // report error
            return FALSE;
    }

    return TRUE;
}

/*
 *@@ TEMPFILETEMP:
 *
 *@@added V0.9.9 (2001-04-04) [umoeller]
 */

typedef struct _TEMPFILETEMP
{
    const char  *pcszFile2Check;
    BOOL        fFound;
    ULONG       ulError;
} TEMPFILETEMP, *PTEMPFILETEMP;

/*
 *@@ TempFileCallback:
 *      WIArchive callback for unpack while
 *      FEInstallEngine::ExtractTempFile is running.
 *
 *@@added V0.9.9 (2001-04-04) [umoeller]
 */

int TempFileCallback(enCallbackMode mode,   // in: CBM_* flags def'd in wiarchive.h
                     short s,               // in: percentage of current file
                     WIFileHeader* pwifh,   // in: file header of current file
                     void *pvUser)          // in: user param; we use the single file name here
{
    int rc = CBRC_ABORT;

    PTEMPFILETEMP ptft = (PTEMPFILETEMP)pvUser;

    switch (mode)
    {
        case CBM_NEXTFILE:
            if (ptft->fFound)
                // found the file on the last loop:
                // stop now
                rc = CBRC_ABORT;
            else if (stricmp(pwifh->name,
                             ptft->pcszFile2Check))
                // no match:
                rc = CBRC_SKIP;
            else
            {
                // only for this one file:
                rc = CBRC_PROCEED;
                // report "abort" on the next file
                ptft->fFound = TRUE;
            }
        break;

        case CBM_PERCENTAGE:
            rc = CBRC_PROCEED;
        break;

        case CBM_ERR_WRITE:
        case CBM_ERR_READ:
        case CBM_ERR_ZLIB:
        default:
            ptft->ulError = mode;
            rc = CBRC_ABORT;
    }

    return rc;
}

/*
 *@@ wpiExtractTempFile:
 *      attempts to extract the single file with the
 *      given file name in the given package into
 *      the system's %TEMP% directory.
 *
 *      Returns:
 *
 *      --  NO_ERROR: file was extracted, and pstrTempFile
 *          name receives the full path of the temp file
 *          so the caller can play with it.
 *
 *      --  ERROR_FILE_NOT_FOUND: file doesn't exist in
 *          the specified package.
 *
 *      --  ERROR_INVALID_DATA: cannot write to disk, or
 *          archive is broken.
 *
 *@@added V0.9.9 (2001-04-04) [umoeller]
 *@@changed V0.9.13 (2001-06-21) [umoeller]: fixed bad DosQueryCurrentDir, thanks Yuri Dario
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_helpers.cpp
 */

APIRET FEInstallEngine::ExtractTempFile(ULONG ulPckIndex,     // in: package to extract from
                                        const char *pcszFileInPck, // in: file name in pck
                                        BSString *pstrTempFileName)   // out: temporary file created
{
    APIRET arc = NO_ERROR;

    // create temp dir... the backend is too dumb to
    // allow us to specify an output file name, so
    // we first create

    CHAR    szDir[CCHMAXPATH];

    if (!(arc = doshCreateTempFileName(szDir,
                                       NULL,        // $(TEMP)
                                       "wpi",       // prefix
                                       "tmp")))     // extension
    {
        if (!(arc = DosCreateDir(szDir,
                                 NULL)))
        {
            CHAR    szStartDir[CCHMAXPATH];
            ULONG   cbStartDir = sizeof(szStartDir);        // V0.9.13 (2001-06-21) [umoeller]
            doshQueryCurrentDir(szStartDir);

            if (!(arc = doshSetCurrentDir(szDir)))
            {
                TEMPFILETEMP tft = {
                                        pcszFileInPck,
                                        FALSE,      // not found
                                        0           // no error yet
                                   };
                _pCurrentArchive->_Arc.setCallbackFunc(TempFileCallback,
                                                       // user param: file name 2 check
                                                       (VOID*)&tft);

                _pCurrentArchive->_Arc.unpack(ulPckIndex);

                if (!tft.fFound)
                    arc = ERROR_FILE_NOT_FOUND;
                else
                    if (tft.ulError)
                        arc = ERROR_INVALID_DATA;
                    else
                    {
                        // else: OK, we got the file. Cute...
                        // create another temp file name in
                        // the parent directory and move the
                        // file there
                        CHAR    szTempFinal[CCHMAXPATH];
                        string  strTempTemp(szDir);
                        if (strTempTemp[strTempTemp.length() - 1] != '\\')
                            strTempTemp += '\\';
                        strTempTemp += pcszFileInPck;

                        if (!(arc = doshCreateTempFileName(szTempFinal,
                                                           NULL,        // $(TEMP)
                                                           "wpi",       // prefix
                                                           "tmp")))     // extension
                        {
                            // move to parent now
                            if (!(arc = DosMove(strTempTemp.c_str(),
                                                szTempFinal)))
                            {
                                // move to parent directory
                                DosSetCurrentDir("..");
                                // remove the empty dir
                                DosDeleteDir(szDir);

                                // give the file name to the parent,
                                // if he wants us to
                                if (pstrTempFileName)
                                    pstrTempFileName->assign(szTempFinal);

                                // store the file in the list of
                                // temp files to be deleted on exit
                                /* _Locals._strTempFiles += szTempFinal;
                                _Locals._strTempFiles += "|"; */
                                _Locals._TempFilenamesList.push_back(new string(szTempFinal));
                            }
                        }
                    }
            }

            // reset directory
            doshSetCurrentDir(szStartDir);
        }
    }

    return arc;
}

/* ******************************************************************
 *
 *  Killing processes
 *
 ********************************************************************/

/*
 *@@ ConfigKillProcesses:
 *      goes thru all install jobs, checking for BSKillProcess
 *      configuration data, and prompts the user for each
 *      process which needs to be killed.
 *
 *      If such an attribute is found and that process
 *      is currently running (which is checked for here, by
 *      calling prc32FindProcessFromName), the user is offered
 *      to have the process killed.
 *
 *      Gets called from FEInstallEngine::Configure.
 *      Calls FELocals::KillOneProcess in turn.
 *
 *@@added V0.9.2 (2000-03-05) [umoeller]
 *@@changed V0.9.3 (2000-05-05) [umoeller]: switched to 16-bit DosQProcStat
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 */

ULONG FEInstallEngine::ConfigKillProcesses()
{
    ULONG   ulrc = 0;

#ifndef __EMX__
    // go thru the list of all packages that were installed
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            if (pJobThis->Is2BeInstalled())
            {
                FEArcPackagePck *pPackageThis = &pJobThis->_ArcPackage;

                // go thru the list of all config objects; this
                // list holds all objects as base objects, so
                // we need to type check
                list<BSConfigBase*>::iterator
                        CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                        CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                for (; CfgStart != CfgEnd; ++CfgStart)
                {
                    // check if this is our class
                    DYNAMIC_CAST(BSKillProcess, pKill, *CfgStart);
                    // BSKillProcess *pKill = (**CfgStart).IsKillProcess();
                    if (!pKill)
                        // not the class we need: take next then
                        continue;

                    BOOL fReloop = FALSE;

                    do
                    {
                        // OK, we got a KILLPROCESS attribute:
                        fReloop = _Locals.KillOneProcess(pKill);
                    } while (fReloop);
                }
            }
        }
    }
#endif

    return ulrc;
}

/* ******************************************************************
 *
 *  System configuration
 *
 ********************************************************************/

/*
 *@@ ConfigConfigSys:
 *      goes thru all install jobs, checking for BSCfgSysManip
 *      configuration data, and hacks CONFIG.SYS according
 *      to that data.
 *
 *      This resolves macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      Gets called from FEInstallEngine::Configure.
 *
 *@@changed V0.9.0 (99-11-01) [umoeller]: adjusted for FEInstallJob
 *@@changed V0.9.0 (99-11-01) [umoeller]: added "global install status" window
 *@@changed V0.9.3 (2000-05-12) [umoeller]: added pszBackup
 *@@changed V0.9.6 (2000-10-27) [umoeller]: made CONFIG.SYS backup part of WPIGLOBALS
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 */

BOOL FEInstallEngine::ConfigConfigSys()
{
    BOOL    brc = TRUE;

    if (QueryInstallVar(_Locals._varConfigSys))
    {
        UpdateGlobalStatus(GSTAT_CONFIGSYS);

        // create instance of BSConfigSys; this will load CONFIG.SYS
        BSConfigSys* pConfigSys = new BSConfigSys;

        // go thru the list of all packages that were installed
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis;
            if (pJobThis = pTemp->IsInstallJob())
            {
                if (pJobThis->Is2BeInstalled())
                {
                    FEArcPackagePck *pPackageThis = &pJobThis->_ArcPackage;
                    // now go thru the CONFIG.SYS manipulators in the
                    // FEPackage and apply them to the BSConfigSys object;
                    // there may be none of these
                    /* list<BSCfgSysManip*>::iterator ManipFirst = pPackageThis->_listCfgSysManips.begin(),
                                                 ManipEnd = pPackageThis->_listCfgSysManips.end(); */

                    // go thru the list of all config objects; this
                    // list holds all objects as base objects, so
                    // we need to type check
                    list<BSConfigBase*>::iterator
                            CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                            CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                    for (; CfgStart != CfgEnd; ++CfgStart)
                    {
                        // check if this is our class
                        DYNAMIC_CAST(BSCfgSysManip, pManip, *CfgStart);
                        // BSCfgSysManip *pManip = (**CfgStart).IsCfgSysManip();
                        if (!pManip)
                            // not the class we need: take next then
                            continue;

                        // resolve macros
                        ResolveMacros(pManip->_ustrNewLine);
                        /*
                        if (pManip->_ConfigManip.pszNewLine)
                        {
                            ustring strNewLine;
                            strNewLine.assignUtf8(pManip->_ConfigManip.pszNewLine);
                            if (ResolveMacros(strNewLine))
                            {
                                free(pManip->_ConfigManip.pszNewLine);
                                pManip->_ConfigManip.pszNewLine = strdup(strNewLine.GetBuffer());
                            }
                        }
                        */

                        // apply to CONFIG.SYS text
                        try
                        {
                            pConfigSys->Manipulate(*_Locals._pCodecProcess,
                                                   *pManip,
                                                        // the manipulator
                                                   &pPackageThis->_logCfgSysDone,
                                                        // CONFIG.SYS logger for database
                                                   _Locals._pLogFile);
                                // this may throw BSConfigExcpt
                            _Locals._fConfigSysChanged = TRUE;
                        }
                        catch(BSConfigExcpt&)
                        {
                            _Locals.MessageBox(102,
                                               124, // "error"
                                               MSG_WARNING_OK);
                            brc = FALSE;
                        }
                        catch(...)
                        {
                            throw;      // rethrow
                        }
                    }
                }
            }
        }

        // rewrite CONFIG.SYS
        if (pConfigSys->Flush(&_Locals._strConfigSysBackup,
                              _Locals._pLogFile) == 0)
            _Locals._fConfigSysChanged = TRUE;

        delete pConfigSys;
    } // end if (G_WpiGlobals.ulDoConfiguration | PCK_CONFIGSYS)

    return brc;
}

/*
 *@@ ConfigWPSClasses:
 *      goes thru all install jobs, checking for BSRegisterClass
 *      and BSReplaceClass  configuration data, and hacks the
 *      WPS class list according to that data.
 *
 *      This invokes BSRegisterClass::Register for each
 *      class to be registered.
 *
 *      This resolves macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      Gets called from FEInstallEngine::Configure.
 *
 *@@changed V0.9.0 (99-11-01) [umoeller]: adjusted for FEInstallJob
 *@@changed V0.9.0 (99-11-01) [umoeller]: added "global install status" window
 *@@changed V0.9.1 (2000-01-06) [umoeller]: "already registered" prompt now only comes up if DLL is different
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 */

BOOL FEInstallEngine::ConfigWPSClasses()
{
    BOOL    brc = TRUE;

    if (QueryInstallVar(_Locals._varWPSClasses))
    {
        UpdateGlobalStatus(GSTAT_WPSCLASSES);

        // go thru the list of all packages that were installed
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis;
            if (pJobThis = pTemp->IsInstallJob())
            {
                if (pJobThis->Is2BeInstalled())
                {
                    FEArcPackagePck *pPackageThis = &pJobThis->_ArcPackage;
                    // now go thru the BSRegisterClass objects in the list in
                    // FEPackage and invoke Register() on them;
                    // there may be none of these

                    // go thru the list of all config objects; this
                    // list holds all objects as base objects, so
                    // we need to type check
                    list<BSConfigBase*>::iterator
                            CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                            CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                    for (; CfgStart != CfgEnd; ++CfgStart)
                    {
                        // check if this is our class
                        DYNAMIC_CAST(BSRegisterClass, pRegThis, *CfgStart);
                        // BSRegisterClass *pRegThis = (**CfgStart).IsRegisterClass();
                        if (!pRegThis)
                            // not the class we need: take next then
                            continue;

                        // resolve macros
                        ResolveMacros(pRegThis->_ustrModule);

                        BOOL    fRetry = FALSE;
                        BOOL    fReplace = FALSE;

                        do {
                            fRetry = FALSE;

                            try
                            {
                                pRegThis->Register(*_Locals._pCodecProcess,
                                                   fReplace, // initially FALSE
                                                   &pPackageThis->_logRegisterClassesDone,
                                                        // classes logger for database
                                                   _Locals._pLogFile);
                                _Locals._fWPSClassesChanged = TRUE;
                            }
                            catch(BSConfigExcpt& X)
                            {
                                brc = FALSE;
                                ustring aStrings[2];
                                ULONG   ulRplCount = 0;
                                ULONG   ulMsg = 0;
                                ULONG   Type = MSG_WARNING_OK;
                                PSZ     pszSysError = 0;

                                switch(X._iErrorCode)
                                {
                                    case REGEXCPT_QUERYCLASSLIST:
                                        ulMsg = 135; // error querying WPS class list
                                    break;

                                    case REGEXCPT_ALREADYREGISTERED:
                                        // BSRegisterClass::Register(FALSE) has determined
                                        // that the class is already registered;
                                        // BSConfigExcpt.pszSubstring has the currently
                                        // registered DLL; if the two are different,
                                        // ask user
                                        if (!(pRegThis->_ustrModule == X._ustrDescription))
                                        /*( if (stricmp(pRegThis->_pszModule, // new DLL
                                                    X._strDescription.GetBuffer()) // registered DLL
                                               != 0)
                                        */
                                        {
                                            // different:
                                            aStrings[0] = pRegThis->_ustrClassName;
                                            aStrings[1] = X._ustrDescription;
                                            ulRplCount = 2;
                                            ulMsg = 136; // The class "%1" is already registered with the DLL "%2".
                                            Type = MSG_CONFIRM_YESNO_DEFYES;
                                        }
                                        else
                                        {
                                            // the two are the same:
                                            fRetry = TRUE;
                                            fReplace = TRUE;
                                        }
                                    break;

                                    case REGEXCPT_REGISTER:
                                        // BSRegisterClass::Register failed;
                                        // BSConfigExcpt.iData has the APIRET of winhRegisterClass
                                        pszSysError = doshQuerySysErrorMsg(X._iData);
                                        aStrings[0] = pRegThis->_ustrClassName;
                                        aStrings[1].assignCP(_Locals._pCodecProcess,
                                                             pszSysError);
                                        ulRplCount = 2;
                                        ulMsg = 137; // The class "%1" could not be registered. The following error was reported:
                                    break;
                                } // end switch

                                if (ulMsg)
                                    if (_Locals.MessageBox(138,  // "WarpIN: Class Registering Error"
                                                           ulMsg,
                                                           Type,
                                                           aStrings,
                                                           ulRplCount)
                                            == MBID_YES)
                                    {
                                        fRetry = TRUE;
                                        fReplace = TRUE;
                                    }

                                if (pszSysError)
                                    free(pszSysError);
                            }
                        } while (fRetry);
                    }

                    // now go thru the BSReplaceClass objects in the list in
                    // FEPackage and invoke Replace() on them;
                    // there may be none of these

                    // go thru the list of all config objects; this
                    // list holds all objects as base objects, so
                    // we need to type check
                    // list<BSConfigBase*>::iterator
                            CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                            CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                    for (; CfgStart != CfgEnd; ++CfgStart)
                    {
                        // check if this is our class
                        DYNAMIC_CAST(BSReplaceClass, pReplThis, *CfgStart);
                        // BSReplaceClass *pReplThis = (**CfgStart).IsReplaceClass();
                        if (!pReplThis)
                            // not the class we need: take next then
                            continue;

                        try
                        {
                            pReplThis->Replace(*_Locals._pCodecProcess,
                                               &pPackageThis->_logReplaceClassDone,
                                                    // classes logger for database
                                               _Locals._pLogFile);
                            _Locals._fWPSClassesChanged = TRUE;
                        }
                        catch(BSConfigExcpt& X)
                        {
                            brc = FALSE;
                            ustring aStrings[2];
                            ULONG   ulRplCount = 0,
                                    ulMsg = 142;    // unknown error

                            switch(X._iErrorCode)
                            {
                                case REGEXCPT_REPLACE:
                                                // error querying WPS class list
                                    aStrings[0] = pReplThis->_ustrOldClassName;
                                    aStrings[1] = pReplThis->_ustrNewClassName;
                                    ulRplCount = 2;
                                    ulMsg = 140;    // Error replacing WPS class "%1" with "%2".
                                break;
                            } // end switch

                            _Locals.MessageBox(139,   // WarpIN: Class Replacement Error
                                               ulMsg,
                                               MSG_WARNING_OK,
                                               aStrings, ulRplCount);
                        }
                    }
                } // end if (ppiThis->ulSelection)
            }
        }
    } // end if (G_WpiGlobals.ulDoConfiguration | PCK_WPSCLASSES)

    return brc;
}

/*
 *@@ ConfigCreateObjects:
 *      goes thru all install jobs, checking for BSCreateObject
 *      configuration data, and creates WPS objects
 *      according to that data.
 *
 *      This invokes BSCreateWPSObject::CreateObject for each
 *      object to be created, which in turn stores the
 *      HOBJECT in its instance data, after having replaced macros.
 *
 *      This resolves macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      Gets called from FEInstallEngine::Configure.
 *
 *@@changed V0.9.0 (99-11-01) [umoeller]: adjusted for FEInstallJob
 *@@changed V0.9.0 (99-11-01) [umoeller]: added "global install status" window
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 *@@changed V0.9.19 (2002-06-15) [umoeller]: now showing "abort, retry, ignore" on failure
 */

BOOL FEInstallEngine::ConfigCreateObjects()
{
    BOOL    brc = TRUE;

    if (QueryInstallVar(_Locals._varWPSObjects))
    {
        UpdateGlobalStatus(GSTAT_WPSOBJECTS);

        // go thru the list of all packages that were installed
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis;
            if (pJobThis = pTemp->IsInstallJob())
            {

                if (pJobThis->Is2BeInstalled())
                {
                    FEArcPackagePck *pPackageThis = &pJobThis->_ArcPackage;
                    // now go thru the CreateWPSObjects objects in the list in
                    // FEPackage and invoke CreateObject() on them;
                    // there may be none of these

                    // go thru the list of all config objects; this
                    // list holds all objects as base objects, so
                    // we need to type check
                    list<BSConfigBase*>::iterator
                            CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                            CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                    for (; CfgStart != CfgEnd; ++CfgStart)
                    {
                        // check if this is our class
                        DYNAMIC_CAST(BSCreateWPSObject, pcroThis, *CfgStart);
                        // BSCreateWPSObject *pcroThis = (**CfgStart).IsCreateWPSObject();
                        if (!pcroThis)
                            // not the class we need: take next then
                            continue;

                        // resolve macros
                        ResolveMacros(pcroThis->_ustrSetupString);
                        ResolveMacros(pcroThis->_ustrLocation);

                        BOOL fDo = TRUE;

                        while (fDo)
                        {
                            fDo = FALSE;
                            try
                            {
                                pcroThis->CreateObject(*_Locals._pCodecProcess,
                                                       &pPackageThis->_logWPSObjectsDone,
                                                    // WPS objects logger for database
                                                       _Locals._pLogFile);
                                        // this may throw BSConfigExcpt
                                _Locals._fWPSObjectsCreated = TRUE;
                            }
                            catch(BSConfigExcpt &X)
                            {
                                // error occured:
                                ustring aStrings[4];
                                aStrings[0] = pcroThis->_ustrTitle;
                                aStrings[1] = pcroThis->_ustrClassName;
                                aStrings[2] = pcroThis->_ustrLocation;
                                aStrings[3] = pcroThis->_ustrSetupString;

                                brc = FALSE;
                                switch (_Locals.MessageBox(102, // "WarpIN: Error"
                                                           (X._iErrorCode == WPOEXCPT_CREATE_BADLOCATION)
                                                               ? 278    // "invalid location"
                                                               : 141,   // "WPS object could not be created"
                                                           MSG_ERROR_ABORTRETRYIGNORE,
                                                                // V0.9.19 (2002-06-15) [umoeller]
                                                           aStrings, 4))
                                {
                                    case MBID_ABORT:
                                        throw BSCancelExcpt();
                                    break;

                                    case MBID_RETRY:
                                        fDo = TRUE;
                                    break;

                                    // on MBID_IGNORE, just continue
                                }
                            }
                        }
                    }
                }
            }
        } // end for (; JobStart != JobEnd; ++JobStart)


    } // end if (G_WpiGlobals.ulDoConfiguration | PCK_WPSOBJECTS)

    return brc;
}

/*
 *@@ ConfigWriteProfiles:
 *      goes thru all install jobs, checking for BSWriteProfile
 *      configuration data, and manipulates INI files
 *      according to that data.
 *
 *      This resolves macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      Gets called from FEInstallEngine::Configure.
 *
 *@@added V0.9.1 (2000-02-07) [umoeller]
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 */

BOOL FEInstallEngine::ConfigWriteProfiles(HAB hab)
{
    BOOL    brc = TRUE;

    // if (G_WpiGlobals.ulAllowConfig & PCK_EXECUTES)
    {
        // guiUpdateGlobalStatus(GSTAT_WPSOBJECTS);

        // go thru the list of all packages that were installed
        list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                                   JobEnd = _pAllJobsList->end();
        for (; JobStart != JobEnd; ++JobStart)
        {
            FEJobBase *pTemp = *JobStart;
            FEInstallJob *pJobThis;
            if (pJobThis = pTemp->IsInstallJob())
            {
                if (pJobThis->Is2BeInstalled())
                {
                    FEArcPackagePck *pPackageThis = &pJobThis->_ArcPackage;
                    // now go thru the BSWriteProfile objects in the list in
                    // FEPackage and invoke Write() on them;
                    // there may be none of these

                    // go thru the list of all config objects; this
                    // list holds all objects as base objects, so
                    // we need to type check
                    list<BSConfigBase*>::iterator
                            CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                            CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                    for (; CfgStart != CfgEnd; ++CfgStart)
                    {
                        // check if this is our class
                        BSConfigBase *pThis = *CfgStart;
                        DYNAMIC_CAST(BSWriteProfile, pPrfThis, pThis);
                        if (!pPrfThis)
                            // not the class we need: take next then
                            continue;

                        // resolve macros
                        ResolveMacros(pPrfThis->_ustrProfile);
                        ResolveMacros(pPrfThis->_ustrWriteString);

                        try
                        {
                            pPrfThis->Write(*_Locals._pCodecProcess,
                                            hab,
                                            &pPackageThis->_logWritePrfsDone,
                                            _Locals._pLogFile);
                                    // this may throw BSConfigExcpt

                            /* if (ulConfigThis & PCK_CONFIGSYS)
                                G_WpiGlobals.fConfigSysChanged = TRUE;
                            if (ulConfigThis & PCK_WPSCLASSES)
                                G_WpiGlobals.fWPSObjectsCreated = TRUE;
                            if (ulConfigThis & PCK_WPSOBJECTS)
                                G_WpiGlobals.fWPSClassesChanged = TRUE; */
                        }
                        catch(BSConfigExcpt& X)
                        {
                            brc = FALSE;
                            _Locals.MessageBox(102, // "WarpIN: Error"
                                               178,   // "error writing to profile"
                                               MSG_WARNING_OK,
                                               0, 0);
                        }
                    }
                }
            }
        } // end for (; JobStart != JobEnd; ++JobStart)
    } // end if (G_WpiGlobals.ulDoConfiguration | PCK_EXECUTES)

    return brc;
}

/*
 *@@ ConfigExecutes:
 *      goes thru all install jobs, checking for BSExecute
 *      configuration data, and executes programs
 *      according to that data.
 *
 *      The pfnExecute callback gets called for each executable.
 *      It is assumed that this callback will operate synchronously;
 *      since this may cause problems with PM callers, a callback
 *      has been implemented for this.
 *
 *      This resolves macros.
 *
 *      Returns FALSE if any error occured.
 *
 *      Gets called from FEInstallEngine::Configure.
 *
 *@@added V0.9.1 (2000-02-07) [umoeller]
 *@@changed V0.9.3 (2000-04-08) [umoeller]: G_WpiGlobals.ulAllowConfig wasn't always set properly, so we ignore this
 *@@changed V0.9.9 (2001-03-27) [umoeller]: reported the wrong config flags, fixed
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 */

BOOL FEInstallEngine::ConfigExecutes()
{
    BOOL    brc = TRUE;

    // go thru the list of all packages that were installed
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            if (pJobThis->Is2BeInstalled())
            {
                FEArcPackagePck *pPackageThis = &pJobThis->_ArcPackage;
                // now go thru the BSExecute objects in the list in
                // FEPackage and invoke CreateObject() on them;
                // there may be none of these

                // go thru the list of all config objects; this
                // list holds all objects as base objects, so
                // we need to type check
                list<BSConfigBase*>::iterator
                        CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                        CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                for (; CfgStart != CfgEnd; ++CfgStart)
                {
                    // check if this is our class
                    DYNAMIC_CAST(BSExecute, pExecThis, *CfgStart);
                    if (!pExecThis)
                        // not the class we need: take next then
                        continue;

                    // V0.9.16 (2001-11-25) [umoeller]
                    // check if maybe this is a BSDeExecute; the
                    // above check worked for those too! we don't
                    // want to run these here
                    if (pExecThis->IsA(BSDeExecute::tBSDeExecute))
                        // this is a de-execute: take next too
                        continue;

                    // ULONG       ulConfigThis = 0;
                    BOOL        fStartAllowed = TRUE;
                    // get execute's configure flags and check
                    // if they're allowed by the user
                    if (pExecThis->_ulExecType & CFGT_CFGSYS)
                    {
                        if (!QueryInstallVar(_Locals._varConfigSys))
                            // not allowed:
                            fStartAllowed = FALSE;
                    }
                    if (pExecThis->_ulExecType & CFGT_REGISTERCLASS)
                    {
                        if (!QueryInstallVar(_Locals._varWPSClasses))
                            // not allowed:
                            fStartAllowed = FALSE;
                    }
                    if (pExecThis->_ulExecType & CFGT_CREATEOBJECT)
                    {
                        if (!QueryInstallVar(_Locals._varWPSObjects))
                            // not allowed:
                            fStartAllowed = FALSE;
                    }
                    // ulConfigThis is still 0 if no flags exist
                    // in the executable

                    if (fStartAllowed)
                    {
                        // resolve macros
                        ResolveMacros(pExecThis->_ustrExecutable);
                        ResolveMacros(pExecThis->_ustrParams);

                        /* if (ulConfigThis & PCK_WPSOBJECTS)
                            _pfnUpdateGlobalStatus(*this, GSTAT_WPSOBJECTS);
                        else if (ulConfigThis & PCK_WPSCLASSES)
                            _pfnUpdateGlobalStatus(*this, GSTAT_WPSCLASSES);
                        else if (ulConfigThis & PCK_CONFIGSYS)
                            _pfnUpdateGlobalStatus(*this, GSTAT_CONFIGSYS);
                        */

                        // use GUI callback for this...
                        // we must not block PM thread 1!
                        _Locals.Execute(pExecThis);

                        if (pExecThis->_ulExecType & CFGT_CFGSYS)
                            _Locals._fConfigSysChanged = TRUE;
                        // fixed the following V0.9.9 (2001-03-27) [umoeller]
                        if (pExecThis->_ulExecType & CFGT_REGISTERCLASS)
                            _Locals._fWPSClassesChanged = TRUE;
                        if (pExecThis->_ulExecType & CFGT_CREATEOBJECT)
                            _Locals._fWPSObjectsCreated = TRUE;
                    }
                }
            }
        }
    } // end for (; JobStart != JobEnd; ++JobStart)

    return brc;
}

/*
 *@@ ConfigDeExecutes:
 *      goes thru all install jobs, checking for BSDeExecute
 *      configuration data, and stores the de-execute data
 *      in each package's de-execute logger.
 *
 *      Note that the executables are _not_ started. This
 *      simply runs through al BSDeExecute instances to resolve
 *      the macros, so that these can be safely stored in the
 *      database. The executables will only be started on
 *      de-install.
 *
 *      Gets called from FEInstallEngine::Configure.
 *
 *@@added V0.9.9 (2001-03-27) [umoeller]
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.14 (2001-07-26) [umoeller]: this used to be in wpi_configure.cpp
 */

BOOL FEInstallEngine::ConfigDeExecutes()
{
    BOOL    brc = TRUE;

    // go thru the list of all packages that were installed
    list<FEJobBase*>::iterator JobStart = _pAllJobsList->begin(),
                               JobEnd = _pAllJobsList->end();
    for (; JobStart != JobEnd; ++JobStart)
    {
        FEJobBase *pTemp = *JobStart;
        FEInstallJob *pJobThis;
        if (pJobThis = pTemp->IsInstallJob())
        {
            if (pJobThis->Is2BeInstalled())
            {
                FEArcPackagePck *pPackageThis = &pJobThis->_ArcPackage;
                // now go thru the BSDeExecute objects in the list in
                // FEPackage

                // go thru the list of all config objects; this
                // list holds all objects as base objects, so
                // we need to type check
                list<BSConfigBase*>::iterator
                        CfgStart = pPackageThis->_pDecl->_listConfigObjects.begin(),
                        CfgEnd = pPackageThis->_pDecl->_listConfigObjects.end();
                for (; CfgStart != CfgEnd; ++CfgStart)
                {
                    // check if this is our class
                    DYNAMIC_CAST(BSDeExecute, pExecThis, *CfgStart);
                    // BSDeExecute *pExecThis = (**CfgStart).IsDeExecute();
                    if (!pExecThis)
                        // not the class we need: take next then
                        continue;

                    // resolve macros and store the resolved thingie
                    ResolveMacros(pExecThis->_ustrExecutable);

                    pPackageThis->_logDeExecutesResolved.Append(pExecThis->_ustrExecutable);

                    if (pExecThis->_ustrParams())
                    {
                        ResolveMacros(pExecThis->_ustrParams);

                        pPackageThis->_logDeExecutesResolved.Append("|", 2);
                        pPackageThis->_logDeExecutesResolved.Append(pExecThis->_ustrExecutable);
                    }
                }
            }
        }
    } // end for (; JobStart != JobEnd; ++JobStart)

    return brc;
}


