
/*
 *@@sourcefile fe_base.cpp:
 *      some shared code for the front end.
 *
 *      Most importantly, this has the FELocals
 *      implementation.
 *
 *@@added V0.9.0 (99-11-07) [umoeller]
 *@@header "engine\fe_base.h"
 */

/*
 *      This file Copyright (C) 1999-2002 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#define OS2EMX_PLAIN_CHAR
    // this is needed for "os2emx.h"; if this is defined,
    // emx will define PSZ as _signed_ char, otherwise
    // as unsigned char

#define INCL_DOSEXCEPTIONS
#define INCL_DOSPROCESS
#define INCL_DOSMODULEMGR
#define INCL_DOSNLS
#define INCL_DOSERRORS
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <limits.h>
#include <time.h>               // needed for WIFileHeader
#include <setjmp.h>             // needed for except.h
#include <assert.h>             // needed for except.h

#include "setup.h"              // was missing (99-12-23) [umoeller]
#include "bldlevel.h"

// include's from helpers
#include "helpers\configsys.h"
#include "helpers\dosh.h"
#include "helpers\except.h"
#include "helpers\nls.h"
#include "helpers\nlscache.h"
#include "helpers\procstat.h"
#include "helpers\standards.h"
#include "helpers\tree.h"
#include "helpers\xstring.h"

#include "helpers\tmsgfile.h"

#include "encodings\base.h"

// base includes (99-11-07) [umoeller]
#include "base\bs_base.h"
#include "base\bs_list.h"
#include "base\bs_string.h"
// #include "base\bs_map.h"
#include "base\bs_errors.h"
#include "base\bs_logger.h"
#include "base\bs_config.h"
#include "base\bs_config_impl.h"

// back-end includes
#include "wiarchive\wiarchive.h"
#include "libbz2\bzlib.h"               // for version info

// front-end includes
#include "engine\fe_base.h"

#pragma hdrstop

/* ******************************************************************
 *
 *  FELocals implementation
 *
 ********************************************************************/

DEFINE_CLASS(FEFileError, BSRoot);
DEFINE_CLASS(FELocals, BSRoot);

/*
 *@@ FELocals:
 *      the constructor.
 *
 *      Note that you must call FELocals::GetSettingsFromEnvironment
 *      manually after the locals have been constructed. See remarks
 *      there.
 *
 *@@changed V0.9.14 (2001-08-23) [umoeller]: added cThousands so that log file can respect NLS separators
 *@@changed V0.9.16 (2001-09-20) [umoeller]: adjusted FNSHOWMESSAGE prototype
 *@@changed V0.9.19 (2002-04-14) [umoeller]: fixed DosQueryCp call
 */

FELocals::FELocals(HAB habThread1,                      // in: anchor block of thread 1 (req. for INI processing)
                   CHAR cThousands,                     // in: thousands separator char (e.g. '.')
                   void *paEntities,                    // in: entities or NULL
                   ULONG cEntities)                     // in: entities count or 0
    : BSRoot(tFELocals),
      _habThread1(habThread1),
      _paEntities(paEntities),
      _cEntities(cEntities),
      _TempFilenamesList(STORE),
      _varConfigSys("CONFIGSYS",
                     "Modify CONFIG.SYS",
                     TRUE),
      _varWPSClasses("REGISTERWPSCLASSES",
                      "Register Workplace Shell classes",
                      TRUE),
      _varWPSObjects("CREATEWPSOBJECTS",
                      "Create Workplace Shell  objects",
                      TRUE),
      _varWriteProfiles("WRITEPROFILES",
                         "Write profile data",
                         FALSE),
      _varClearProfiles("CLEARPROFILES",
                         "Clear profile data",
                         FALSE),
      _varExecutes("EXECUTES",
                    "Execute install scripts",
                    FALSE),
      _varDeExecutes("DEEXECUTES",
                      "Execute deinstall scripts",
                      FALSE),
      _varKillProcesses("KILLPROCESSES",
                         "Kill processes",
                         FALSE)
{
    CHAR sz[100];

    // file exists
    _ulIfSameDate = 2;           // overwrite
    _ulIfExistingOlder = 2;      // overwrite
    _ulIfExistingNewer = 0;      // prompt

    _pLogFile = NULL;

    _pCodecProcess = NULL;
    _pCodecGui = NULL;

    // create codec for TMF files (850)
    _pCodec850 = new BSUniCodec(850);

    // create codec for process codepage
    ULONG acp[8];       // fixed V0.9.19 (2002-04-14) [umoeller], this needs an array
    ULONG cb = 0;
    APIRET arcCP;
    if (!(arcCP = DosQueryCp(sizeof(acp),
                             acp,
                             &cb)))
        _pCodecProcess = new BSUniCodec(acp[0]);

    // store the WARPIN.EXE executable in the globals
    // for later use
    // V0.9.14 (2001-08-03) [umoeller]: now using DosQueryModuleName
    PPIB    pib;
    CHAR    szWarpINExe[CCHMAXPATH];
    DosGetInfoBlocks(NULL, &pib);
    DosQueryModuleName(pib->pib_hmte,
                       sizeof(szWarpINExe),
                       szWarpINExe);

    // extract the path from that
    PSZ p = strrchr(szWarpINExe, '\\');
    *p = 0;
    _ustrWarpINPath.assignCP(_pCodecProcess, szWarpINExe);

    /* #ifdef __DEBUG__
        BSRoot::OpenDebugLog();             // V0.9.14 (2001-07-07) [umoeller]
    #endif */

    // compose the name of the message (tmf) file (99-10-24) [umoeller];
    // this is referenced by fe_script.cpp
    string strMessageFile;
    strMessageFile._printf("%s\\warpin.tmf", szWarpINExe);

    tmfOpenMessageFile(strMessageFile.c_str(),
                       (PTMFMSGFILE*)&_pvMessageFile);

    _cThousands = cThousands;

    _fConfigSysChanged = FALSE;
    _fWPSClassesChanged = FALSE;
    _fWPSObjectsCreated = FALSE;

    if (arcCP)
    {
        // error above:
        sprintf(sz, "DosQueryCp returned %d.", arcCP);
        throw BSExcptBase(sz);
    }

}

/*
 *@@ GetSettingsFromEnvironment:
 *      post-constructor method that initializes
 *      a bunch of members from environment variables,
 *      if present.
 *
 *      This is in a separate method because the GUI
 *      might preload the settings from other sources
 *      and can then later call this method to have
 *      those values overridden if environment variables
 *      are present.
 *
 *@@added V0.9.19 (2002-04-14) [umoeller]
 */

void FELocals::GetSettingsFromEnvironment()
{
    // get general environment
    // duh, this used to be in GetGeneralEnvironment in warpin.cpp
    // and was completely missing in V0.9.18
    // V0.9.19 (2002-04-14) [umoeller]
    const char  *pcszCommand = 0,
                *pcszValue = 0;

    // SET WARPIN_DEFAULTAPPSPATH=[path]
    pcszCommand = "WARPIN_DEFAULTAPPSPATH";
    if (    (!(pcszValue = getenv(pcszCommand)))
         && (!(_strDefaultAppsPath()))      // not set by GUI? V0.9.19 (2002-04-14) [umoeller]
       )
    {
        // not set:
        _strDefaultAppsPath._printf(
                "%s=%c:\\APPS",
                pcszCommand,
                doshQueryBootDrive());
        putenv(_strDefaultAppsPath.c_str());
    }

    // SET WARPIN_DEFAULTTOOLSPATH=[path]
    pcszCommand = "WARPIN_DEFAULTTOOLSPATH";
    if (    (!(pcszValue = getenv(pcszCommand)))
         && (!(_strDefaultToolsPath()))     // not set by GUI? V0.9.19 (2002-04-14) [umoeller]
       )
    {
        // not set:
        _strDefaultToolsPath._printf(
                "%s=%c:\\TOOLS",
                pcszCommand,
                doshQueryBootDrive());
        putenv(_strDefaultToolsPath.c_str());
    }

    // SET WARPIN_IFSAMEDATE={PROMPT|SKIP|OVERWRITE}
    pcszCommand = "WARPIN_IFSAMEDATE";
    if ((pcszValue = getenv(pcszCommand)))
        if (stricmp(pcszValue, "PROMPT") == 0)
            _ulIfSameDate = FE_PROMPT;
        else if (stricmp(pcszValue, "SKIP") == 0)
            _ulIfSameDate = FE_SKIP;
        else if (stricmp(pcszValue, "OVERWRITE") == 0)
            _ulIfSameDate = FE_OVERWRITE;
        else
            // invalid value:
            throw FEEnvironmentExcept(*this, pcszCommand, pcszValue);

    // SET WARPIN_IFEXISTINGOLDER={PROMPT|SKIP|OVERWRITE}
    pcszCommand = "WARPIN_IFEXISTINGOLDER";
    if ((pcszValue = getenv(pcszCommand)))
        if (stricmp(pcszValue, "PROMPT") == 0)
            _ulIfExistingOlder = FE_PROMPT;
        else if (stricmp(pcszValue, "SKIP") == 0)
            _ulIfExistingOlder = FE_SKIP;
        else if (stricmp(pcszValue, "OVERWRITE") == 0)
            _ulIfExistingOlder = FE_OVERWRITE;
        else
            // invalid value:
            throw FEEnvironmentExcept(*this, pcszCommand, pcszValue);

    // SET WARPIN_IFEXISTINGNEWER={PROMPT|SKIP|OVERWRITE}
    pcszCommand = "WARPIN_IFEXISTINGNEWER";
    if ((pcszValue = getenv(pcszCommand)))
        if (stricmp(pcszValue, "PROMPT") == 0)
            _ulIfExistingNewer = FE_PROMPT;
        else if (stricmp(pcszValue, "SKIP") == 0)
            _ulIfExistingNewer = FE_SKIP;
        else if (stricmp(pcszValue, "OVERWRITE") == 0)
            _ulIfExistingNewer = FE_OVERWRITE;
        else
            // invalid value:
            throw FEEnvironmentExcept(*this, pcszCommand, pcszValue);

    // SET WARPIN_LOGFILEPATH=<logfilepath> V0.9.9 (2001-03-30) [umoeller]
    pcszCommand = "WARPIN_LOGFILEPATH";
    if ((pcszValue = getenv(pcszCommand)))
        if (0 == doshIsValidFileName(pcszValue,
                                     TRUE))         // must be fully qualified
            _ustrLogFilename.assignCP(_pCodecProcess, pcszValue);
        else
            // invalid value:
            throw FEEnvironmentExcept(*this, pcszCommand, pcszValue);
}

/*
 *@@ ~FELocals:
 *      the destructor.
 *
 *      This cleans up all temp files that were stored
 *      in _TempFilenamesList. Note that even though these
 *      are presently only created by FEInstallEngine, this
 *      allows other places of WarpIn to create temp
 *      files too in the future.
 */

FELocals::~FELocals()
{
    CloseLogger();

    tmfCloseMessageFile((PTMFMSGFILE*)&_pvMessageFile);

    // delete all temp files
    list<string*>::iterator tempBegin = _TempFilenamesList.begin(),
                            tempEnd = _TempFilenamesList.end();
    for (;
         tempBegin != tempEnd;
         ++tempBegin)
    {
        string *pstr = *tempBegin;
        if (pstr->size())
        {
            DosForceDelete(pstr->c_str());
        }
    }

    /*
    const char *pStart = _strTempFiles.c_str(),
               *pEnd;

    while (    (*pStart)
            && (pEnd = strchr(pStart, '|'))
          )
    {
        string strThis(pStart, pEnd);
        if (strThis())
        {
            DosForceDelete(strThis.c_str());
            pStart = pEnd + 1;
        }
        else
            break;
    }
    */
}

/*
 *@@ wpiGetMessage:
 *      like DosGetMessage, but automatically finds the
 *      (NLS) WarpIN message file.
 *
 *      This has been moved here because otherwise we
 *      can't use NLS messages with the parsers.
 *
 *@@changed (99-10-22) [umoeller]: now using TMF functions.
 *@@changed V0.9.9 (2001-02-28) [umoeller]: moved here from fe_base.cpp
 *@@changed V0.9.18 (2002-03-08) [umoeller]: now using ustrings for the array
 *@@changed V0.9.18 (2002-03-08) [umoeller]: added entities
 */

APIRET FELocals::GetMessage(ustring &ustrBuf,  // out: composed message string
                            ULONG ulMsgNumber, // in: TMF message number to retrieve
                            ustring *paStrings, // in: array of UTF-8 strings, defaults to NULL
                            ULONG ulTable)     // in: array item count (defaults to 0)
{
    XSTRING xstr;
    APIRET  arc;
    CHAR    szMsgID[30];
    sprintf(szMsgID, "WPI%04d", ulMsgNumber);

    // build an array of PSZ's from the ustrings given to us
    // V0.9.18 (2002-03-08) [umoeller]
    PCSZ     *pTable = NULL;
    string   *paCPStrings = NULL;
    if (ulTable)
    {
        pTable = (PCSZ*)malloc(sizeof(PSZ) * ulTable);
        paCPStrings = new string[ulTable];
        PCSZ *pThis = pTable;
        ULONG ul;
        for (ul = 0;
             ul < ulTable;
             ++ul, ++pThis)
        {
            // convert this to the CP850 because that's
            // what we have in the TMF file
            paCPStrings[ul].assignUtf8(_pCodec850,
                                       paStrings[ul]);
            *pThis = paCPStrings[ul].c_str();
        }
    }

    xstrInit(&xstr, 0);
    if (arc = tmfGetMessage((PTMFMSGFILE)_pvMessageFile,
                            szMsgID,
                            &xstr,
                            pTable, ulTable))
        ustrBuf._printf("tmfGetMessageExt returned error %u for msg %u",
                        arc,
                        ulMsgNumber);
    else
    {
        // this is now in codepage 850
        if (_cEntities)
        {
            ULONG ul;
            for (ul = 0;
                 ul < _cEntities;
                 ++ul)
            {
                ULONG ulOfs = 0;
                PCSTRINGENTITY pa = (PCSTRINGENTITY)_paEntities;
                PCSTRINGENTITY pThis = &pa[ul];
                while (xstrFindReplaceC(&xstr,
                                        &ulOfs,
                                        pThis->pcszEntity,
                                        *(pThis->ppcszString)))
                    ;
            }
        }

        // convert to utf-8
        ustrBuf.assignCP(_pCodec850, xstr.psz);
    }

    if (pTable)
        free(pTable);
    if (paCPStrings)            // V0.9.19 (2002-05-07) [umoeller]
        delete[] paCPStrings;

    xstrClear(&xstr);

    return arc;
}

/*
 *@@ MessageBox:
 *      this displays a message box using
 *      message IDs for the title and the
 *      message.
 *
 *      Returns what FELocals::ShowMessage returns
 *      (the virtual method to be implemented by
 *      a subclass), which must be a standard
 *      WinMessageBox value like MBID_OK.
 *
 *@@added V0.9.0 (99-11-01) [umoeller]
 *@@changed V0.9.14 (2001-07-26) [umoeller]: renamed from wpiMessageBox, turned into a method
 *@@changed V0.9.18 (2002-03-08) [umoeller]: now using ustrings for the array
 */

ULONG FELocals::MessageBox(ULONG ulTitleMsg, // in: TMF msg ID for title
                           ULONG ulMsgMsg,   // in: TMF msg ID for message
                           ULONG fl, // in: as in WinMessageBox (MB_OK etc.)
                           ustring *paStrings, // in: array of UTF-8 strings, defaults to NULL
                           ULONG cTable)   // in: string replacement count, defaults to 0
{
    ustring     strTitle, strMsg;
    GetMessage(strTitle,
               ulTitleMsg);
    GetMessage(strMsg,
               ulMsgMsg,
               paStrings, cTable);

    // call the ShowMessage implementation of the subclass
    // (pure virtual method in FELocals)
    return ShowMessage(strTitle, strMsg, fl);
}

/*
 *@@ OpenLogger:
 *      opens the install logger, if _strLogFilename
 *      has been set to anything.
 *
 *@@added V0.9.14 (2001-07-26) [umoeller]
 */

VOID FELocals::OpenLogger()
{
    if (    (_ustrLogFilename())
         && (!_pLogFile)
       )
    {
        // logging enabled:
        string str(_pCodecProcess, _ustrLogFilename);
        _pLogFile = new BSFileLogger(0,
                                     str.c_str());
    }
}

/*
 *@@ Log:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

VOID FELocals::Log(const char *pcszFormat,
                   ...)
{
    if (_pLogFile)
    {
        va_list arg_ptr;
        va_start(arg_ptr, pcszFormat);
        _pLogFile->WriteV(pcszFormat, arg_ptr);
        va_end(arg_ptr);
    }
}

/*
 *@@ IncIndent:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

void FELocals::IncIndent(int i)
{
    if (_pLogFile)
        _pLogFile->IncIndent(i);
}

/*
 *@@ CloseLogger:
 *
 *@@added V0.9.14 (2001-07-26) [umoeller]
 */

VOID FELocals::CloseLogger()
{
    if (_pLogFile)
    {
        delete _pLogFile;
        _pLogFile = NULL;
    }
}

/*
 *@@ KillOneProcess:
 *      checks the process list for the specified process
 *      and prompts for whether it should be killed.
 *
 *      Returns TRUE if the process list should be checked
 *      again.
 *
 *      Gets called in two situations:
 *
 *      -- from FEInstallEngine::ConfigKillProcesses, before the
 *         Install thread is started;
 *
 *      -- from guiStartRemove, before the Database thread
 *         starts removing packages.
 *
 *      This really belongs to FEInstallEngine, but since we don't
 *      have an engine in database mode, this was added to
 *      FELocals instead.
 *
 *@@added V0.9.4 (2000-07-01) [umoeller]
 *@@changed V0.9.6 (2000-10-26) [umoeller]: added exception handling
 *@@changed V0.9.9 (2001-03-30) [umoeller]: added pLogFile
 *@@changed V0.9.10 (2001-04-08) [umoeller]: added DosQProcStat check
 *@@changed V0.9.11 (2001-04-25) [umoeller]: fixed infinite questioning if DosQProcStat failed
 *@@changed V0.9.14 (2001-07-26) [umoeller]: renamed from wpiKillOneProcess, turned into a method
 */

BOOL FELocals::KillOneProcess(BSKillProcess *pKill)
{
    BOOL            fReloop = FALSE;
    PQPROCSTAT16    pInfo = NULL;
    CHAR            szPID[30];

    ustring         aStrings[2];
    aStrings[0] = pKill->_ustrKillProcess;

    TRY_LOUD(excpt1)
    {
        APIRET arc;
        if (arc = prc16GetInfo(&pInfo))
        {
            // DosQProcStat failed: V0.9.10 (2001-04-08) [umoeller]
            ustring astr[2];
            astr[0]._itoa10(arc, _cThousands);
            astr[1] = pKill->_ustrKillProcess;
            throw FEFatalErrorExcpt(*this,
                                    205, // "DosQProcStat failed, retry"
                                    astr, 2);
                // now throwing exception
        }
        else
        {
            // check if that process is running
            PQPROCESS16 pProcess;
            string strProcess(_pCodecProcess, pKill->_ustrKillProcess);
            if (pProcess = prc16FindProcessFromName(pInfo,
                                                    strProcess.c_str()))
            {
                // running: prompt
                aStrings[1]._printf("0x%lX", pProcess->usPID);
                ULONG mbrc = MessageBox(108, // warning
                                        180, // "kill these programs first"
                                        MSG_CONFIRM_YESNOCANCEL_DEFYES,
                                        aStrings, 2);
                switch (mbrc)
                {
                    case MBID_YES:
                        // kill process
                        DosKillProcess(DKP_PROCESS, pProcess->usPID);

                        // sleep a second:
                        // call the Sleep implementation of the subclass
                        // (pure virtual method in FELocals)
                        Sleep(1000);
                        fReloop = TRUE;
                    break;

                    case MBID_NO:
                        // check again
                        fReloop = TRUE;
                    break;

                    case MBID_CANCEL:
                        throw BSCancelExcpt();
                }
            }
        }
    }
    CATCH(excpt1)
    {
        ULONG mbrc = MessageBox(108, // warning
                                196, // "exception killing %1"
                                MSG_WARNING_OK,
                                aStrings, 1);
        fReloop = FALSE;
    } END_CATCH();

    if (pInfo)
        prc16FreeInfo(pInfo);

    return fReloop;
}

/*
 *@@ GetCopyrightInfo:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

VOID FELocals::GetCopyrightInfo(ustring &ustr,          // out: Unicode copyright info
                                const string &strDatabaseFile)  // in: database (process codepage)
{
    ustring aStrings[7];
    aStrings[0].assignUtf8(BLDLEVEL_VERSION);
    aStrings[1].assignUtf8(__DATE__);
#ifdef __IBMCPP__
    aStrings[2].assignUtf8("IBM VisualAge C++ 3.08");
#else
    CHAR    szEMXRev[40];
    sprintf(szEMXRev, "EMX/GCC %s", _emx_vprt);
    aStrings[2].assignUtf8(szEMXRev);
#endif
    aStrings[3]._itoa10(QueryGuiCodec().QueryCodepage(), 0);
    aStrings[4] = QueryWarpINPath();
    aStrings[5].assignCP(_pCodecProcess, strDatabaseFile);
    aStrings[6].assignUtf8(BZ2_bzlibVersion());

    GetMessage(ustr,
               276,
               aStrings,
               7);
}

/* ******************************************************************
 *
 *  Helpers
 *
 ********************************************************************/

/*
 *@@ wpiCTime2FTime:
 *      useful helper function for converting
 *      the C runtime "time_t" format to the
 *      OS/2 FDATE and FTIME format.
 */

void wpiCTime2FTime(time_t *pCTime,     // in: C runtime time_t format
                    FDATE *pfDate,      // out: OS/2 FDATE format
                    FTIME *pfTime)      // out: OS/2 FTIME format
{
    struct tm* pTimeBuf = localtime(pCTime);
    pfDate->year      = (pTimeBuf->tm_year + 1900) - 1980;
    pfDate->month     = (pTimeBuf->tm_mon + 1);
    pfDate->day       = (pTimeBuf->tm_mday);
    pfTime->hours     = (pTimeBuf->tm_hour);
    pfTime->minutes   = (pTimeBuf->tm_min);
    pfTime->twosecs   = (pTimeBuf->tm_sec) / 2;
}

/*
 *@@ wpiCompareFileDates:
 *      this helper compares the last write date
 *      as found in PFILESTATUS3 (which should have
 *      been filled using DosQueryPathInfo on an
 *      existing file) to the last write date in
 *      the specified WIFileHeader.
 *
 *      The header file's last write date will be
 *      written in the two specified buffers in
 *      OS/2 date/time format. These buffers _must_
 *      be specified (or this will crash).
 *
 *      Returns:
 *          // < 0:  existing file is newer
 *          // == 0: existing file is same write date
 *          // > 0:  existing file is older
 *
 *@@changed V0.9.14 (2001-08-03) [umoeller]: added VAC 3.6.5 time_t fix
 */

int wpiCompareFileDates(WIFileHeader* pHeader,      // in: file header to compare
                        FILESTATUS3 *pfs3,          // in: actual file to compare
                        FDATE *pfdateLastWrite,     // out: last-write date of pHeader in OS/2 format
                        FTIME *pftimeLastWrite)     // out: last-write time of pHeader in OS/2 format
{
    // convert file header's time information
    time_t lastwrite = pHeader->lastwrite;      // V0.9.14 (2001-08-03) [umoeller]
    wpiCTime2FTime(&lastwrite,
                   pfdateLastWrite,
                   pftimeLastWrite);

    // and compare to the actual file's information
    static const PSZ pszFormatTimestamp = "%4u%02u%02u%02u%02u%02u";
                                        // >  M   D   H   M   2S

    CHAR szHeaderStamp[30],
         szExistingStamp[30];

    // create timestamps
    sprintf(szHeaderStamp,
            pszFormatTimestamp,
            pfdateLastWrite->year + 1980,
            pfdateLastWrite->month,
            pfdateLastWrite->day,
            pftimeLastWrite->hours,
            pftimeLastWrite->minutes,
            pftimeLastWrite->twosecs * 2);
    sprintf(szExistingStamp,
            pszFormatTimestamp,
            pfs3->fdateLastWrite.year + 1980,
            pfs3->fdateLastWrite.month,
            pfs3->fdateLastWrite.day,
            pfs3->ftimeLastWrite.hours,
            pfs3->ftimeLastWrite.minutes,
            pfs3->ftimeLastWrite.twosecs * 2);

    return strcmp(szHeaderStamp, szExistingStamp);
}

/* ******************************************************************
 *
 *  Exception implementations
 *
 ********************************************************************/

/*
 *@@ FEConstructorExcpt:
 *      constructor for exceptions thrown in
 *      constructors.
 */

FEConstructorExcpt::FEConstructorExcpt(FELocals &Locals,
                                       const char *strClass,  // in: class name
                                       const char *pszSourceFile,
                                       int iSourceLine,
                                       int iMessage)        // in: msg id
{
    ustring     ustrMsg2;
    Locals.GetMessage(ustrMsg2,
                      iMessage);

    ustring     ustrMsg,
                ustrSourceLine;
    ustrSourceLine._itoa10( iSourceLine, Locals._cThousands);

    ustring     aStrings[4];
    aStrings[0].assignUtf8(strClass);
    aStrings[1].assignUtf8(pszSourceFile);
    aStrings[2] = ustrSourceLine;
    aStrings[3] = ustrMsg2;

    Locals.GetMessage(ustrMsg,
                      166,      // error in constructor
                      aStrings, 4);
    _ustrDescription = ustrMsg;
}

/*
 *@@ FEFatalErrorExcpt:
 *
 *@@added V0.9.18 (2002-03-08) [umoeller]
 */

FEFatalErrorExcpt::FEFatalErrorExcpt(FELocals &Locals,
                                     const ustring &ustrMessage)
{
    _ustrDescription = ustrMessage;

    Locals.Log("Fatal error: %s", _ustrDescription);
}

/*
 *@@ FEFatalErrorExcpt:
 *      constructor 2 for generic fatal error messages.
 */

FEFatalErrorExcpt::FEFatalErrorExcpt(FELocals &Locals,
                                     int iMessage,     // in: TMF msg id
                                     ustring *paStrings,    // in: ustring array (def: 0)
                                     int iCount)        // in: array item count (def: 0)
{
    Locals.GetMessage(_ustrDescription,
                      iMessage,
                      paStrings, iCount);

    Locals.Log("Fatal error: %s", _ustrDescription);
}

/*
 *@@ FEEnvironmentExcept:
 *      constructor for environment errors.
 */

FEEnvironmentExcept::FEEnvironmentExcept(FELocals &Locals,
                                         const char *pcszCommand,       // process codepage
                                         const char *pcszValue)         // process codepage
{
    ustring     aStrings[2];
    aStrings[0].assignCP(Locals._pCodecProcess, pcszCommand);
    aStrings[1].assignCP(Locals._pCodecProcess, pcszValue);
    Locals.GetMessage(_ustrDescription,
                      167,
                      aStrings, 2);
}


