
/*
 * fe_script.h:
 *      header file for fe_script.cpp. See remarks there.
 *
 *@@include #include "base\bs_string.h"
 *@@include #include "base\bs_logger.h"
 *@@include #include "base\bs_config.h"
 *@@include #include "engine\fe_script.h"
 */

/*
 *      This file Copyright (C) 1999-2001 Ulrich Mller.
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, in version 2 as it comes in the COPYING
 *      file of this distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#ifndef WARPIN_SCRIPT_HEADER_INCLUDED
    #define WARPIN_SCRIPT_HEADER_INCLUDED

    /* ******************************************************************
     *
     *  Forward declarations
     *
     ********************************************************************/

    class BSConfigBase;
    class FEScriptBase;
    class FEGroupDecl;
    class FERexx;

    /* ******************************************************************
     *
     *  FEScriptExcpt declaration
     *
     ********************************************************************/

    /*
     *@@ FEScriptExcpt:
     *      exception thrown by the various install script parsers.
     *
     *V0.9.9 (2001-02-28) [umoeller]
     */

    class FEScriptExcpt : public BSExcptBase
    {
        public:
            unsigned long       _ulLine,
                                _ulColumn;

            FEScriptExcpt(ULONG ulLine,
                          ULONG ulColumn,
                          FELocals &Locals,
                          int iMessage,
                          ustring *paStrings = NULL,
                          int iCount = 0);
    };

    /* ******************************************************************
     *
     *  FEPageInfo declaration
     *
     ********************************************************************/

    /*
     *@@ FEPageInfo:
     *      one of these structures is created for each PAGE block
     *      found in the install script by the FEScriptBase::Parse
     *      override.
     *
     *      A list of all the pages in an archive is stored in
     *      FEScriptBase::_PageInfoList.
     *
     *@@changed V0.9.0 (99-10-31) [umoeller]: this used to be WPIPAGEINFO
     */

    class FEPageInfo : public BSRoot
    {
        public:
            DECLARE_CLASS(FEPageInfo);

        public:
            LONG        _lPageIndex;    // index of this page

            enum { MODE_TEXT,           // this page is plain text
                   MODE_README,         // this page is a readme (MLE)
                   MODE_CONTAINER,      // this is the "Container" page
                   MODE_CONFIGURE       // this is the "Configure" page
                 }      _PageMode;

            ustring     _ustrInfoText;    // has text found in <TEXT> ... </TEXT> block

            ustring     _ustrReadmeSrc;   // for the README page, has the <README> ... </README> block

            ULONG       _ulExtractFromPck; // if != 0, extract from package and use
                                           // _ustrReadmeSrc as the filename V0.9.9
                                           // GUI is responsible for codepage conversions!

            // <README> attribute, one of the following:
            enum {  README_PLAIN,        // plain text
                    README_FLOW,         // reformat line breaks
                    README_HTML          // limited HTML support
                 }
                        _ReadmeFormat;   // set to README_PLAIN by the constructor

            LONG        _lNextButton;
                    // the "Next" button:
                    // -- if > 0, the button is visible and enabled
                    //    and has the index of the next page;
                    // -- if == 0 --> "Install" button, start install
            ustring     _ustrNextButtonTitle;
                    // button text

            /*
             *@@ FEPageInfo:
             *      default constructor.
             */

            FEPageInfo() : BSRoot(tFEPageInfo)
            {
                _lPageIndex = 0;
                _PageMode = MODE_TEXT;

                _ulExtractFromPck = 0;
                _ReadmeFormat = README_PLAIN;

                _lNextButton = 0;
            }
    };

    /* ******************************************************************
     *
     *  FEVarPrompt declaration
     *
     ********************************************************************/

    #define VPT_NUM         1
    #define VPT_ALPHANUM    2
    #define VPT_PATH        3
    #define VPT_FAIL        4

    /*
     *@@ FEVarPrompt:
     *      class representing a VARPROMPT declaration.
     *      This is passed to guiVarPrompt if a variable
     *      is not defined.
     *
     *      FEScriptBase has a static FEVarPromptsMap of
     *      all defined FEVarPrompt's.
     *
     *@@added V0.9.2 (2000-03-11) [umoeller]
     */

    class FEVarPrompt : public BSRoot
    {
        public:
            DECLARE_CLASS(FEVarPrompt);

        public:
            string      _strVarName;
                    // variable name; e.g. "ETC";
                    // specified with NAME attribute
            string      _strDescription;
                    // description; block within <VARPROMPT> ... </VARPROMPT>
            ULONG       _ulType,
                        _ulMin,
                        _ulMax;

            /*
             *@@ FEVarPrompt:
             *      default constructor.
             */

            FEVarPrompt(const char *pcszVarName) : BSRoot(tFEVarPrompt)
            {
                _strVarName = pcszVarName;
                _strDescription = "";
                _ulType = VPT_ALPHANUM;
                _ulMin = 0;
                _ulMax = 0;
            }

            void CopyFrom(const FEVarPrompt &p)
            {
                _strVarName = p._strVarName;
                _strDescription = p._strDescription;
                _ulType = p._ulType;
                _ulMin = p._ulMin;
                _ulMax = p._ulMax;
            }

            FEVarPrompt(const FEVarPrompt &p) : BSRoot(tFEVarPrompt)
            {
                CopyFrom(p);
            }

            FEVarPrompt& operator=(const FEVarPrompt &p)
            {
                CopyFrom(p);
                return *this;
            }

            bool Validate(const char *pcszValue)
                 const;
    };

    int operator==(const FEVarPrompt &v1, const FEVarPrompt &v2);
    int operator<(const FEVarPrompt &v1, const FEVarPrompt &v2);

    /* ******************************************************************
     *
     *  FEPckDeclBase declaration
     *
     ********************************************************************/

    // flags for FEPackage.ulPathType
    #define PATH_DISABLED   0x00        // path cannot be changed
    #define PATH_VARIABLE   0x01        // path can be freely changed
    #define PATH_BASE       0x02        // path is base for all packages

    // flags for FEPackage.ulConfigureFlags
    /*
    #define PCK_CONFIGSYS       0x0001    // CONFIG.SYS can be updated
    #define PCK_WPSCLASSES      0x0002    // WPS classes can be installed
    #define PCK_WPSOBJECTS      0x0004    // WPS objects can be created
    #define PCK_CLEARPROFILES   0x0008    // has clear-profile data -- V0.9.1 (2000-02-07) [umoeller]
    #define PCK_WRITEPROFILES   0x0010    // has write-profile data -- V0.9.1 (2000-02-07) [umoeller]
    #define PCK_EXECUTES        0x0020    // needs user program to be executed -- V0.9.1 (2000-02-03) [umoeller]
    #define PCK_KILLPROCESS     0x0040    // pck has KILLPROCESS tags -- V0.9.1 (2000-02-12) [umoeller]

    #define PCK_FILES           0x0080    // has files; this is only used
                                          // in database mode

    #define PCK_DEEXECUTES      0x0100    // has DEEXECUTE data
    */

    /*
    #define CONFIGSYS           "ConfigSys"
    #define WPSCLASSES          "RegisterWPSClasses"
    #define WPSOBJECTS          "CreateWPSObjects"
    #define CLEARPROFILES       "ClearProfiles"
    #define WRITEPROFILES       "WriteProfiles"
    #define EXECUTES            "Executes"
    #define DEEXECUTES          "DeExecutes"
    #define KILLPROCESSES       "KillProcesses"
    */

    enum enumPckType { GROUP,           // FEGroupDecl
                       INTERNAL,        // FEPckDeclBase, internal (no EXTERNAL attrib)
                       EXTERNAL,        // FEPckDeclBase, external, but optional
                       REQEXTERNAL,     // FEPckDeclBase, external and required
                       DBPACKAGE };     // FEDBPckDecl

    /*
     *@@ FEPckDeclBase:
     *      represents a package declaration in a script.
     *
     *      There are two subclasses of this for old-style
     *      (FEOldPckDecl) and new-style scripts, but
     *      the data fields are mostly identical for the two.
     *
     *@@added V0.9.9 (2001-02-28) [umoeller]
     *@@changed V0.9.18 (2002-03-03) [umoeller]: added _fLongFilenames
     */

    class FEPckDeclBase : public BSRoot
    {
        public:
            DECLARE_CLASS(FEPckDeclBase);

        public:
            enumPckType             _Type;

            // BOOL                    _fIsGroup;
                    // TRUE if this represents a group

            FEGroupDecl             *_pGroup;
                    // the group this belongs to or NULL if root

            ustring                 _ustrTitle;
                    // TITLE attribute V0.9.9 (2001-02-28) [umoeller];
                    // required for both packages and groups

            // the following are only used for "real" packages...

            ustring                 _ustrID;
                    // V0.9.9 (2001-02-28) [umoeller]
                    // copy of PACKAGEID attribute, not normalized;
                    // empty for groups

            ustring                  _ustrExternalArchive;
                    // if external, the external archive name
                    // now using ustring V0.9.19 (2002-04-14) [umoeller]

            ULONG                   _ulIndexInArchive;
                    // index of this package in the archive (>= 1);
                    // 0 for groups

            ustring                 _ustrDescription;
                    // old style: stuff between PCK and /PCK
                    // new style: contents of DESCRIPTION element
                    // empty for groups

            ustring                  _ustrTargetPath;
                    // TARGET attribute V0.9.9 (2001-02-28) [umoeller]
                    // empty for groups
                    // V0.9.19 (2002-04-14) [umoeller]: now using ustring

            enum { UNKNOWN,
                   SELECTED,        // package is initially selected
                   DESELECTED,      // package is initially deselected
                   NODESELECT }     // package must always be selected
                                    _Selected;

            ULONG                   _ulPathType;
                    // any combination of the following:
                    // -- PATH_DISABLED:   path cannot be changed
                    // -- PATH_VARIABLE: path can be freely changed
                    // -- PATH_BASE: path is base for all packages
                    // 0 for groups

            BOOL                    _fLongFilenames;
                    // V0.9.18 (2002-03-03) [umoeller]
                    // if TRUE, package can only be installed on
                    // drives which support LFNs (not FAT)

            // this holds all configuration items, which
            // can be of a whole lotta subclasses actually
            // (BSCfgSysManip, BSRegisterClass, BSReplaceClass,
            // BSCreateWPSObject, BSClearProfile, BSWriteProfile,
            // BSExecute);
            // we use a list of the shared base class to avoid
            // template bloat which we had by creating one list
            // for each of these subclasses... saved about 100 KB code!
            list<BSConfigBase*>     _listConfigObjects;

            // ULONG                   _ulConfigData;
                    // any combination of the following:
                    // -- PCK_CONFIGSYS: CONFIG.SYS can be updated;
                    // -- PCK_WPSCLASSES: WPS classes can be installed;
                    // -- PCK_WPSOBJECTS: WPS objects can be created.

            list<FEInstallVar*>     _listVariables;
                    // list of install variables used by this
                    // package (STORE mode)

            BSRequiresStringsLogger _logRequiresStrings;
                    // "REQUIRES=" attributes, as in script.
                    // Each of these can either be a decimal string
                    // (which is resolved later) or a five- or
                    // six-part package ID; V0.9.1 (2000-01-07) [umoeller]

            // configuration attributes from install script:
            // either set by FEArchive or loaded from database
            BSCfgSysAttrsLogger     _logCfgSysAttrs;
            BSRegisterAttrsLogger   _logRegisterClassAttrs;
            BSReplaceAttrsLogger    _logReplaceClassAttrs;
            BSWPSObjectAttrsLogger  _logWPSObjectAttrs;
            BSExecuteAttrsLogger    _logExecuteAttrs; // added V0.9.1 (2000-02-03) [umoeller]
            BSDeExecuteAttrsLogger  _logDeExecuteAttrs; // added V0.9.9 (2001-03-27) [umoeller]
            BSClearPrfAttrsLogger   _logClearPrfAttrs; // added V0.9.1 (2000-02-07) [umoeller]
            BSWritePrfAttrsLogger   _logWritePrfAttrs; // added V0.9.1 (2000-02-07) [umoeller]
            BSKillProcessAttrsLogger _logKillProcessAttrs; // added V0.9.1 (2000-02-07) [umoeller]

        protected:
            FEPckDeclBase(FEGroupDecl *pGroup,
                          BSClassID &Class);
    };

    /*
     *@@ FEDBPckDecl:
     *      pseudo-declaration for packages from a database.
     *
     *      Yes, this isn't pretty, but since FEPackageBase has
     *      a pDecl member pointing to a FEPckDeclBase instance
     *      and many of the package data members have now been
     *      encapsulated in FEPckDeclBase, we must use this for
     *      database packages.
     *
     *@@added V0.9.9 (2001-02-28) [umoeller]
     */

    class FEDBPckDecl : public FEPckDeclBase
    {
        public:
            DECLARE_CLASS(FEDBPckDecl);

        public:
            FEDBPckDecl(const ustring &ustrID)
                : FEPckDeclBase(NULL,       // group: none here
                                tFEDBPckDecl)
            {
                _Type = DBPACKAGE;
                _ustrID = ustrID;
            };
    };

    /*
     *@@ FEGroupDecl:
     *      FEPckDeclBase subclass for group declarations
     *      in a script (see FEScriptBase).
     *
     *      Note that this class does not parse itself
     *      since we only have a couple of data members.
     *
     *@@added V0.9.9 (2001-02-28) [umoeller]
     */

    class FEGroupDecl : public FEPckDeclBase
    {
        public:
            DECLARE_CLASS(FEGroupDecl);

        public:
            list<FEPckDeclBase*>       _DeclarationsList;

            BOOL        _fInitiallyExpanded;

            /*
             *@@ FEGroupDecl:
             *      constructor. This expects the group title
             *      and the EXPANDED boolean attribute directly,
             *      which must have been parsed by the caller.
             *
             *@@added V0.9.9 (2001-02-28) [umoeller]
             */

            FEGroupDecl(FEGroupDecl *pGroup,
                        const ustring &ustrTitle,
                        BOOL fInitiallyExpanded)
                : FEPckDeclBase(pGroup,
                                tFEGroupDecl),
                  _DeclarationsList(STORE)
            {
                _Type = GROUP;
                _ustrTitle = ustrTitle;
                _fInitiallyExpanded = fInitiallyExpanded;
            }
    };

    /* ******************************************************************
     *
     *  FEScriptBase and subclasses
     *
     ********************************************************************/

    /*
     *@@ FEScriptBase:
     *      base class for WarpIN scripts. This is an abstract
     *      base class and must be implemented by one of the
     *      two subclasses:
     *
     *      -- FEOldScript for "old-style" scripts;
     *
     *      -- BSXmlScript for XML scripts (not implemented yet).
     *
     *      See @class_rels for an overview of how the front-end
     *      classes interact.
     *
     *      The concept is that FEScriptBase has a static method
     *      (FEScriptBase::CreateScript) which gets called by FEArchive
     *      when a script needs to be parsed. FEArchive no longer cares
     *      about how the script is interpreted, but only wants
     *      the member fields in FEScriptBase to point to the data
     *      in the script.
     *
     *      In other words, only FEScriptBase is "seen" by
     *      FEArchive and is assumed to contain the parsed
     *      script data. One of the subclasses will do the
     *      actual parsing in the virtual FEScriptBase::Parse
     *      method, which must be overridden by subclasses
     *      (script implementations).
     *
     *      FEPckDeclBase and FEGroupDecl in turn
     *      implement package declarations in the script and
     *      must be created by the FEScriptBase::Parse override.
     *
     *      To implement a different script language, a check has
     *      to be added to FEScriptBase::CreateScript, which must
     *      then create an instance of the new script subclass and
     *      invoke that class's Parse method.
     */

    class FEScriptBase : public BSRoot
    {
        public:
            DECLARE_CLASS(FEScriptBase);

        protected:
            FEScriptBase(FELocals &Locals,
                         FERexx &Rexx,
                         BSClassID &Class);

        public:
            FELocals                &_Locals;

            FERexx                  &_Rexx;

            ULONG                   _ulCodepage;
                    // CODEPAGE tag found with WARPIN tag;
                    // defaults to 850 with old-style scripts.
                    // If this is 0, the script was in Unicode.

            // void                    *_VarPromptsMap;
            // static FEVarPromptsMap  VarPromptsMap;
                    // string map with all VARPROMPTS ever found

            ustring                 _ustrTitle;
                    // contents of optional <TITLE> tag;
                    // this can be empty if not specified

            list<FEPckDeclBase*>    _DeclarationsList;
                    // list of all root package and group declarations...
                    // if this is an old-style script, these are
                    // really instances of FEOldPckDecl.
                    // NOTE: This does not contain declarations which
                    // are non-root, i.e. members of a group package.
                    // Instead, those declarations are in the group only.

            list<FEPageInfo*>       _PageInfoList;
                    // list of pages in the install script;
                    // this is passed to the GUI
                    // (valid only after explicit call to ParsePages).

            static void StoreVarPrompt(const char *pcszVarNameAttr,
                                       string &strVarBlock,
                                       ULONG ulType,
                                       ULONG ulMin,
                                       ULONG ulMax);

            static FEVarPrompt* FindVarPrompt(const char *pcszVarNameAttr);

            // virtual parse method... implemented by subclasses
            virtual VOID Parse(const char *pcszScript) = 0;

            VOID Validate();

            static FEScriptBase* CreateScript(const char *pcszScript,
                                              FELocals &Locals,
                                              FERexx &Rexx);
    };

#endif

