

#include <stdio.h>
#include <math.h>
 
#include "RTTypes.h"
#include "RTVecOps.h"

#include "RTObjOps.h"
#include "RTIMOps.h"
#include "RayTrace.h"





int RT_CompareI( Vector Iul, Vector Iur, Vector Ill, Vector Ilr);

/*
 *******************************************************************************
 *
 *  Name:  RT_SuperSample
 *
 *  Purpose:  this routine super samples a pixel, or (in a recursive call)
 *            a subarea of an already super sampled pixel, according to
 *            the algorithm in section 15.10.4 of:
 *
 *               Computer Graphics (Principles and Practice)
 *
 *            This routine is called recursively for each subarea of the
 *            area currently being super sampled.  If the subarea does
 *            not require super sampling or if the recursive limit of
 *            super sampling has been reached, the corner intensities for
 *            the subarea are averaged.
 *
 *
 *
 *  Input Parameters
 *
 *     Iul, Iur, Ill, Ilr - intensities at corners of super sample area
 *     Rul, Rur, Rll, Rlr - rays through corners of super sample area
 *     scan - scan line containing "Rlr"
 *     pix - pixel containing "Rlr"
 *     d - current recursive depth in super sampling
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
Vector
   RT_SuperSample(Vector Iul, Vector Iur, Vector Ill, Vector Ilr,
                  Ray    Rul, Ray    Rur, Ray    Rll, Ray    Rlr,
                  float scan, float pix, int d)
      {
       Vector Imt, Imb;
       Vector Iml, Imr;
       Vector Ic;

/*
 *     Ray Rul, Rur;
 */
       Ray Rmt, Rmb;
       Ray Rml, Rmr;
       Ray Rc;

       Vector pixval;
       Vector newpix;

       float dsp;


/*
 *     initialization
 */
       dsp =((float) 1.0) / (float) (d * 2);
       vset(&pixval, (float) 0.0,  (float) 0.0,  (float) 0.0);

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  if ss is not needed - average I values ar corners
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       if ( (RT_CompareI(Iul, Iur, Ill, Ilr) == 0)  || (d >= 4) )  {
           pixval = Vector_Sum(&Iul, &Iur);
           pixval = Vector_Sum(&pixval, &Ill);
           pixval = Vector_Sum(&pixval, &Ilr);

           pixval = VectorScaler_Division(&pixval, (float)4.0);
          }

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  if ss is needed - subdivide current area for ss
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       else  {

/*
 *         determine additional rays for super sampling
 */
           Rmt.origin = Rlr.origin;
           Rmt.dir = RT_DirectionVector(pix-dsp,       scan+( ((float)2.0)*dsp));

           Rmb.origin = Rlr.origin;
           Rmb.dir = RT_DirectionVector(pix-dsp,       scan);

           Rml.origin = Rlr.origin;
           Rml.dir = RT_DirectionVector(pix-( ((float)2.0)*dsp), scan+dsp);

           Rmr.origin = Rlr.origin;
           Rmr.dir = RT_DirectionVector(pix,           scan+dsp);

           Rc.origin =  Rlr.origin;
           Rc.dir =  RT_DirectionVector(pix-dsp,       scan+dsp);

/*
 *         determine intensities for super sample rays
 */
           Imt = RT_Trace(Rmt, (float) -1.0, 1);
           Imb = RT_Trace(Rmb, (float) -1.0, 1);
           Iml = RT_Trace(Rml, (float) -1.0, 1);
           Imr = RT_Trace(Rmr, (float) -1.0, 1);
           Ic  = RT_Trace(Rc , (float) -1.0, 1);


/*
 *         super sample each subarea in super sample area & average results
 */
           pixval = RT_SuperSample(Ic,  Imr, Imb, Ilr, Rc,  Rmr, Rmb, Rlr,
                                   scan,     pix,      d+1);


           newpix = RT_SuperSample(Iml, Ic,  Ill, Imb, Rml, Rc,  Rll, Rmb,
                                   scan,     pix-dsp,  d+1);
           pixval = Vector_Sum(&pixval, &newpix);


           newpix = RT_SuperSample(Imt, Iur, Ic,  Imr, Rmt, Rur, Rc,  Rmr,
                                   scan+dsp, pix,      d+1);
           pixval = Vector_Sum(&pixval, &newpix);


           newpix = RT_SuperSample(Iul, Imt, Iml, Ic,  Rul, Rmt, Rml, Rc,
                                   scan+dsp, pix-dsp,  d+1);
           pixval = Vector_Sum(&pixval, &newpix);


           pixval = VectorScaler_Division(&pixval, (float)4.0);
          }
/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */

       return(pixval);
      }



/*
 *******************************************************************************
 *
 *  Name:  RT_CompareI
 *
 *  Purpose:  this routine compares the intensities at the corners of a
 *            super sampled area to see if it should be subdivided.  The
 *            comparison done on a component by component basis and deter-
 *            mines if any of the components differ significantly from the
 *            the component average.
 *            
 *
 *
 *
 *  Input Parameters
 *
 *     Iul, Iur, Ill, Ilr - intensities at corners of super sampled area
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
int
   RT_CompareI( Vector Iul, Vector Iur, Vector Ill, Vector Ilr)
      {
       float avg;


/*
 *     compare Red components of intensity
 */
       avg = (Iul.x + Iur.x + Ill.x + Ilr.x) / ((float) 4.0);

       if ( fabs(avg - Iul.x) > 0.05 || fabs(avg - Iur.x) > 0.05 ||
            fabs(avg - Ill.x) > 0.05 || fabs(avg - Ilr.x) > 0.05    ) 
          return(1);


/*
 *     compare Green components of intensity
 */
       avg = (Iul.y + Iur.y + Ill.y + Ilr.y) / ((float) 4.0);

       if ( fabs(avg - Iul.y) > 0.05 || fabs(avg - Iur.y) > 0.05 ||
            fabs(avg - Ill.y) > 0.05 || fabs(avg - Ilr.y) > 0.05    ) 
          return(1);


/*
 *     compare Blue components of intensity
 */
       if ( fabs(avg - Iul.z) > 0.05 || fabs(avg - Iur.z) > 0.05 ||
            fabs(avg - Ill.z) > 0.05 || fabs(avg - Ilr.z) > 0.05    ) 
          return(1);

       return(0);
      }



