

#include <stdio.h>
#include <math.h>
 
#include "RTTypes.h"
#include "RTMatMac.h"
#include "RTVecOps.h"

#include "RTObjOps.h"
#include "RTIMOps.h"
#include "RayTrace.h"
#include "RTIntOps.h"



/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  File:  nearest_int.c
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */


/*
 *******************************************************************************
 *
 *  Name:  nearest_int
 *
 *  Purpose:  this routine determines which objects the specified ray in-
 *            tersects.  It then chooses the intersection closest to the
 *            origin of the ray and returns the parameters associated with
 *            this intersection.
 *
 *            NOTE:  this routine uses a tolerance factor in order to
 *                   screen out the intersection between a ray and its
 *                   origin point.
 *
 *
 *
 *  Input Parameters
 *
 *     i_ray - ray currently being traced
 *
 *
 *  Output Parameters
 *
 *     obj_no - index of intersected object in array of objects
 *     int_val - point of intersection in object
 *     Pt_norm -  normal at point
 *     patch_norm -normal of patch containing point
 *
 *******************************************************************************
 */
Boolean
   nearest_int(Ray i_ray, int *obj_no, Vector *int_val, Vector *Pt_norm, Vector *patch_norm)
      {
       Sphere_Obj *s_ptr;
       Poly_Mesh_Obj *poly_ptr;
       Rect_Obj *rect_ptr;

       Vector int_pt;
       Vector int_norm, norm;

       Boolean norm_flg = TRUE;
       Boolean int_flg = FALSE;
       Boolean return_int_flg = FALSE;

       float t, t_near = (float)99999.0;

       int i;


/*
 +++++++++++++++++++++++++++++++++++++++++++++++++
 * determine nearest intersection
 +++++++++++++++++++++++++++++++++++++++++++++++++
 */
       for (i=0; i<shared->obj_num; i++)  {

           switch (shared->obj_list[i].label)  {

/*
 *             handle case of sphere object
 */
               case SPHERE:
                   s_ptr = (Sphere_Obj *) shared->obj_list[i].ptr;

                   int_flg = sphere_int(i_ray, s_ptr, norm_flg, &int_pt,
                                        &int_norm, &norm, &t);
                   break;

/*
 *             handle case of polygon mesh object
 */
               case POLY_MESH:
                   poly_ptr = (Poly_Mesh_Obj *) shared->obj_list[i].ptr;
                   int_flg = poly_mesh_int(i_ray, poly_ptr, norm_flg, NEAREST,
                                           &t, &int_pt, &int_norm, &norm);
                   break;

/*
 *             handle case of rectangle object
 */
               case RECTANGLE:
                   rect_ptr = (Rect_Obj *) shared->obj_list[i].ptr;
                   int_flg = rect_int(i_ray, rect_ptr, norm_flg, &t, &int_pt,
                                      &int_norm);
                   norm = int_norm;
                   break;
              }


/*
 *         record this intersection if:
 *            - there is an intersection   (int_flg)
 *            - the intersection is not at the origin of ray   (t > 0.1)
 *            - the intersection is the nearest so far   (t < t_near)
 */

           if ( int_flg && (t > 0.001) && (t < t_near) )  {
               t_near = t;
               *obj_no = i;
               *int_val = int_pt;
               *Pt_norm = int_norm;
               *patch_norm = norm;
               return_int_flg = TRUE;
              }

          }

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++
 +++++++++++++++++++++++++++++++++++++++++++++++++
 */

       return(return_int_flg);
      }



/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  File:  sphere_int.c
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */



/*
 *******************************************************************************
 *
 *  Name:  sphere_int
 *
 *  Purpose:  this routine determines the intersection between the specified
 *            ray and sphere by solving the quadratic equation representing
 *            their intersection.  It at least one intersection exists, the
 *            closest one is returned along with the ray "t" value at the
 *            point of intersection.
 *
 *            NOTE:  the sphere normal is only returned if the "norm_flag" 
 *                   is set to TRUE.
 *
 *
 *
 *  Input Parameters
 *
 *     ray - specified ray
 *     s_ptr - ptr to sphere object
 *     norm_flg - flag indicating whether normal should be returned
 *
 *
 *  Output Parameters
 *
 *     int_pt - point of intersection between sphere & ray
 *     pt_norm - normal at point
 *     norm - normal of sphere
 *     t - distance along ray where intersection occures
 *
 *******************************************************************************
 */
Boolean
   sphere_int (Ray ray, Sphere_Obj *s_ptr, Boolean norm_flg,
               Vector *int_pt, Vector *pt_norm, Vector *norm, float *t)
      {
       Vector tmp_vec;
       Boolean int_flg;

       float A, B, C;
       float Bsq_4AC;
       float t1, t2;
       float r_sq;


/*
 *     initialization - calculate squared radius of sphere
 */
       r_sq = SQR(s_ptr->center_pt.x - s_ptr->radius_pt.x) + 
              SQR(s_ptr->center_pt.y - s_ptr->radius_pt.y) + 
              SQR(s_ptr->center_pt.z - s_ptr->radius_pt.z);

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  calculate elements of quadratic formula
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       A = SQR(ray.dir.x) + SQR(ray.dir.y) + SQR(ray.dir.z);

       B = 2 * (  (ray.dir.x * (ray.origin.x - s_ptr->center_pt.x))  +
                  (ray.dir.y * (ray.origin.y - s_ptr->center_pt.y))  +
                  (ray.dir.z * (ray.origin.z - s_ptr->center_pt.z))     );

       C = SQR(ray.origin.x - s_ptr->center_pt.x)  +
           SQR(ray.origin.y - s_ptr->center_pt.y)  +
           SQR(ray.origin.z - s_ptr->center_pt.z)  - 
           r_sq;

       Bsq_4AC = (B*B) - ( ((float)4.0) * A * C);

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  determine solutions of quadratic formula
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *
 *     two imaginary solutions - no intersection
 */
       if (Bsq_4AC < 0.0)  {
           int_flg = FALSE;
          }

/* 
 *     one real solution - one intersection
 */
       else if (Bsq_4AC == 0.0)  {
           int_flg = TRUE;
           *t = B / ( ((float)2.0) * A);

           *int_pt = ray.origin;
           tmp_vec = VectorScaler_Product(&ray.dir, *t);
           *int_pt = Vector_Sum(int_pt, &tmp_vec);
          }

/* 
 *     two real solutions - two intersections
 */
       else if (Bsq_4AC > 0.0)  {
           int_flg = TRUE;
           t1 = ( (-B) + sqrt(Bsq_4AC) ) / ( ((float)2.0) * A);
           t2 = ( (-B) - sqrt(Bsq_4AC) ) / ( ((float)2.0) * A);

           if (t1 < 0.001)
              *t = t2;
           else if (t2 < 0.001)
              *t = t1;
           else
              *t = (fabs(t1) < fabs(t2))? t1 : t2;

           *int_pt = ray.origin;
           tmp_vec = VectorScaler_Product(&ray.dir, *t);
           *int_pt = Vector_Sum(int_pt, &tmp_vec);
          }

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  determine sphere normal if needed
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       if (int_flg && norm_flg)  {
           pt_norm->x = (int_pt->x - s_ptr->center_pt.x);
           pt_norm->y = (int_pt->y - s_ptr->center_pt.y);
           pt_norm->z = (int_pt->z - s_ptr->center_pt.z);

           *pt_norm = VectorScaler_Division(pt_norm, vector_len(pt_norm));
           *norm = *pt_norm;
          }
/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */

       return (int_flg);
      }



/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  File:  triang_int.c
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */



int triang_int(Ray ray, Triangle *tptr, Vector tdata[], Vector *nptr,
              Boolean norm_flg, Vector *Pint, Vector *p_norm, float *t);

void project_2D (Triangle *tptr, Vector tdata[], Vector Pi,
                 Point t_pr[], Point *Ppr);

float tri_area_x2 (Point pt_a, Point pt_b, Point pt_c);



/*
 *******************************************************************************
 *
 *  Name:  poly_mesh_int
 *
 *  Purpose:  this routine cycles through the triangles in a polygon mesh
 *            looking for an intersection with the specified ray.  In order
 *            to minimize cost it handles two different cases based on the
 *            parameter "type":
 *
 *               - case LF: Light (Shadow) Feeler
 *                   the ray is a light (shadow) feeler and therefore any
 *                   intersection will be sufficient.  The routine returns
 *                   with the first intersection found.
 *
 *               - case NEAREST: Nearest Intersection
 *                   the ray is not a light (shadow) feeler and therefore
 *                   the nearest intersection is being sought.  All tri-
 *                   angles are examined in order to insure that the inter-
 *                   section closest to the ray origin is returned.
 *
 *
 *
 *  Input Parameters
 *
 *     ray - specified ray
 *     obj_ptr - pointer to polygon mesh object
 *     norm_flg - flag indicating whether the normals should be returned
 *     type - param indicating whether first or closest int should be returned
 *
 *
 *  Output Parameters
 *
 *     t_val - distance along ray of intersection
 *     int_val - intersection point
 *     Pt_norm - interpolated normal at intersection
 *     t_norm - normal of triangle containing intersection point
 *
 *******************************************************************************
 */
Boolean
   poly_mesh_int(Ray ray, Poly_Mesh_Obj *obj_ptr, Boolean norm_flg,
                 int type, float *t_val,
                 Vector *int_val, Vector *Pt_norm, Vector *t_norm)
      {
       Sphere_Obj *s_ptr;

       Vector tdata[3];
       Vector new_int;

       Vector *nptr;

       Vector p_norm, norm;

       Boolean done = FALSE;
       Boolean int_flg = FALSE;
       Boolean return_int_flg = FALSE;

       float t;

       Surface *sptr;
       Triangle *tptr;

       int i, j, k;


       *t_val = (float) 9999.0;



/*
 *     run quick rejection test on object's bounding sphere 
 */
       //s_ptr = (Sphere_Obj *) &obj_ptr->b_sph;
       //int_flg = sphere_int(ray, s_ptr, FALSE, &new_int, NULL, NULL, &t);

       //if ( (! int_flg) || (t < 0.001) )
       //    return(FALSE);


       int_flg = FALSE;


/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *  cycle through triangles in polygon mesh object
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       for (i=0; i<obj_ptr->num_surf && !done; i++)  { 
           sptr = &obj_ptr->surfs[i];
           nptr = obj_ptr->surf_norms[i];

           for (j=0; j<sptr->num_triang && ! done; j++)  {
               tptr = &sptr->triangs[j];

/*
 *             put triangle vertices in more convienent sructure
 */
               for (k=0; k<3; k++)
                   tdata[k] = obj_ptr->data[ tptr->v[k] ];

/*
 *             determine intersection
 */
               int_flg = triang_int(ray, tptr, tdata, nptr, norm_flg, &new_int,
                                                                  &p_norm, &t);
               switch (type)  {

/*
 *                 if Light Feeler - prepare to stop if intersection
 */
                   case LF:
                      if ( (int_flg) && (t < *t_val) && (t > 0.001) ) {
						  done = int_flg;
						  return_int_flg = int_flg;
						  *int_val = new_int;
						  *t_val = t;
					     }
                      break;

/*
 *                 if Nearest Intersection - record int. info and continue
 */
                   case NEAREST:
                      if ( (int_flg) && (t < *t_val) && (t > 0.001) ) {
                          return_int_flg = int_flg;
                          *int_val = new_int;
                          *t_val = t;
                          if (norm_flg)  {
                              *Pt_norm = p_norm;
                              *t_norm = tptr->normal;
                             }
                         }
                      break;
                  }  /* end -- switch */

              }  /* end -- for each triangle in surface*/
          }  /* end -- outer for each surface in poly mesh*/
/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       return(return_int_flg);
      }



/*
 *******************************************************************************
 *
 *  Name:  triang_int
 *
 *  Purpose:  this routine determines if a ray intersects a triangle using
 *            a three step algorithm.  Steps two and three are performed
 *            ONLY if step one determines that some intersection has
 *            occured:
 *
 *               1. Determine equation of plane containing containing
 *                  triangle using normal of triangle patch.  Then deter-
 *                  mine if ray intersects this plane.
 *
 *               2. Project intersection of point & triangle into 2D space
 *                  along coordinate axis parallel to largest normal com-
 *                  ponent.
 *
 *               3. Determine barycentric coordinates of triangle projec-
 *                  tion.  Ray intersects triangle if all three barycen-
 *                  centric coordinates are >= zero.
 *
 *
 *
 *  Input Parameters
 *
 *     ray - specified ray
 *     tptr - pointer to specified triangle
 *     tdata - array containing data points for triangle vertices
 *     nptr - pointer to normals list containing normals at triangle vertices
 *     norm_flg - flag indicating whether triangle normals should be returned
 *
 *
 *  Output Parameters
 *
 *     Pint - point of intersection between ray and triangle
 *     p_norm - interpolated normal at point of intersection
 *     t - distance along ray of intersection
 *
 *******************************************************************************
 */
int
   triang_int(Ray ray, Triangle *tptr, Vector tdata[], Vector *nptr,
              Boolean norm_flg, Vector *Pint, Vector *p_norm, float *t)
      {
       Vector nfrac;
       Point T_pr[3];

       Point P_pr;

       Boolean int_flg = FALSE;

       float A, B, C, D;
       float numer, denom;
       float T_area;
       float u, v, w;


/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *  determine if ray intersects plane containing triangle
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       A = tptr->normal.x;
       B = tptr->normal.y;
       C = tptr->normal.z;

       D = ((-A) * tdata[0].x) + ((-B) * tdata[0].y) + ((-C) * tdata[0].z);

       numer = (A * ray.origin.x) + (B * ray.origin.y) + (C * ray.origin.z) + D;

       denom = (A * ray.dir.x) + (B * ray.dir.y) + (C * ray.dir.z);

       *t = (denom != 0.0)? -(numer / denom): (float)0.0;

/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *  determine if intersection (if any) is within triangle
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       if ( (numer == 0.0) || (denom == 0.0) || (*t <= 0.0) )  {
           int_flg = FALSE;
	      }


       else  {
/*
 *         determine intersection point and project to 2D space
 */
           *Pint = VectorScaler_Product(&ray.dir, *t);
           *Pint = Vector_Sum(&ray.origin, Pint);

           project_2D (tptr, tdata, *Pint, T_pr, &P_pr);

/*
 *         use barycentric coordinates to determine if triangle contains int
 */
           T_area = tri_area_x2(T_pr[0], T_pr[1], T_pr[2]);
           u = tri_area_x2(P_pr, T_pr[1], T_pr[2]) / T_area;

           if (u >= 0.0)  {
               v = tri_area_x2(T_pr[0], P_pr, T_pr[2]) / T_area;

               if (v >= 0.0)  {
                   w = tri_area_x2(T_pr[0], T_pr[1], P_pr) / T_area;
                   int_flg = (w >= 0)? TRUE : FALSE;
                  }
              }
          }
/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *  calculate interpolated normal at point - if necessary
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       if (int_flg && norm_flg)  {

           vset(p_norm,  (float) 0.0,  (float) 0.0,  (float) 0.0);

           nfrac = VectorScaler_Product(&nptr[ tptr->v[0] ], u);	// u
           *p_norm = Vector_Sum(p_norm, &nfrac);

           nfrac = VectorScaler_Product(&nptr[ tptr->v[1] ], v);	// v
           *p_norm = Vector_Sum(p_norm, &nfrac);

           nfrac = VectorScaler_Product(&nptr[ tptr->v[2] ], w);	// w
           *p_norm = Vector_Sum(p_norm, &nfrac);


		   *p_norm = VectorScaler_Division(p_norm, vector_len(p_norm));
          }
/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */

       return(int_flg);
      }



/*
 *******************************************************************************
 *
 *  Name:  project_2D
 *
 *  Purpose:  this routine projects points on to the two dimensional co-
 *            ordinate plane that is perpendicular to the largest compo-
 *            nent of the normal for the specified triangle.
 *           
 *
 *
 *
 *  Input Parameters
 *
 *     tptr - pointer to triangle in 3D space
 *     tdata - triangle points in 3D space
 *     Pi - intersection point in 3D space
 *
 *
 *  Output Parameters
 *
 *     t_pr - triangle points in 2D space
 *     Ppr - intersection point in 2D space
 *
 *******************************************************************************
 */
void
   project_2D (Triangle *tptr, Vector tdata[], Vector Pi,
               Point t_pr[], Point *Ppr)
      {
	   Boolean ProjectionFlag = FALSE;
       float Nx, Ny, Nz;
       int i;


       Nx = tptr->normal.x;
       Ny = tptr->normal.y;
       Nz = tptr->normal.z;

/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *  project || to z-axis if Nz is largest normal component
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       if (  ( fabs(Nz) >= fabs(Nx) )  &&  ( fabs(Nz) >= fabs(Ny) )  )  {
		   ProjectionFlag = TRUE;
           Ppr->x = Pi.x;
           Ppr->y = Pi.y;

           for (i=0; i<3; i++)  {
              t_pr[i].x = tdata[i].x;
              t_pr[i].y = tdata[i].y;
             }
          }

/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *  project || to y-axis if Ny is largest normal component
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       else if (  ( fabs(Ny) >= fabs(Nx) )  &&  ( fabs(Ny) >= fabs(Nz) )  )  {
		   ProjectionFlag = TRUE;
           Ppr->x = Pi.x;
           Ppr->y = Pi.z;

           for (i=0; i<3; i++)  {
              t_pr[i].x = tdata[i].x;
              t_pr[i].y = tdata[i].z;
             }
          }

/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *  project || to x-axis if Nx is largest normal component
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       else if (  ( fabs(Nx) >= fabs(Ny) )  &&  ( fabs(Nx) >= fabs(Nz) )  )  {
		   ProjectionFlag = TRUE;
           Ppr->x = Pi.y;
           Ppr->y = Pi.z;

           for (i=0; i<3; i++)  {
              t_pr[i].x = tdata[i].y;
              t_pr[i].y = tdata[i].z;
             }
          }
/*
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
	   if (!ProjectionFlag)  {
		   printf ("No Projection Made...   Exiting\n");
		   getchar();
	      }
 


      }



/*
 *******************************************************************************
 *
 *  Name:  tri_area_x2
 *
 *  Purpose:  this routine calculates the area of a triangle using Cramer's
 *            Rule.
 *
 *
 *
 *  Input Parameters
 *
 *     pt_a, pt_b, pt_c - vertices of triangle in x-y space
 *
 *
 *  Output Parameters
 *
 *     none
 *
 *******************************************************************************
 */
float
   tri_area_x2 (Point pt_a, Point pt_b, Point pt_c)
      {
	   float result;
	   
       result = (pt_a.x * pt_b.y) -
				(pt_a.x * pt_c.y) -
				(pt_b.x * pt_a.y) + 
				(pt_b.x * pt_c.y) +
				(pt_c.x * pt_a.y) - 
				(pt_c.x * pt_b.y);

	   return (result);
      }			


/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  File:  rect_int.c
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */



/*
 *******************************************************************************
 *
 *  Name:  rect_int
 *
 *  Purpose:  this routine determines the intersection (if any) between a
 *            ray and rectangle that is parallel to one of the coordinate
 *            axis planes.  It does this by finding the equation of the
 *            plane that contains the rectangle and determining if the
 *            ray intersects this plane.  If it does, a simple bounds test
 *            is used to determine if the intersection is within the rect-
 *            angle.
 *
 *            This routine returns the value TRUE if there the ray inter-
 *            sects the rectangle.  If a value of FALSE is returned, the
 *            values of the output parameters are undefined.
 *
 *            NOTE:  the normal of the rectangle will only be returned if
 *                   "norm_flg" is set to TRUE.
 *
 *
 *
 *  Input Parameters
 *
 *     ray - specified ray
 *     r_ptr - specified rectangle
 *     norm_flg - flag indicating whether the normal to plane is desired
 *
 *
 *  Output Parameters
 *
 *     t - ray parameter value at point of intersection
 *     Pint - point of intersection on rectangle
 *     r_norm - normal to rectangle surface (if norm_flg is set)
 *
 *******************************************************************************
 */
Boolean
   rect_int(Ray ray, Rect_Obj *r_ptr, Boolean norm_flg,
            float *t, Vector *Pint, Vector *r_norm)
      {
       Sphere_Obj *s_ptr;
       Boolean int_flg = FALSE;

       float A = (float)0.0, B = (float)0.0, C = (float)0.0, D = (float)0.0;
       float numer, denom;
       Vector s_int;


/*
 *     run quick rejection test on object's bounding sphere 
 */
       s_ptr = (Sphere_Obj *) &r_ptr->b_sph;
       int_flg = sphere_int(ray, s_ptr, FALSE, &s_int, NULL, NULL, t);

       if ( (! int_flg) || (*t < 0.001) )
           return(FALSE);


       int_flg = FALSE;

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  handle rectangle parallel to yz coordinate plane
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       if (r_ptr->normal == X_AXIS)  {

/*
 *         determine value for intersection
 */
           D = -(r_ptr->corners[0].x);

           numer = ray.origin.x + D;
           denom = ray.dir.x;
           *t = (denom != 0.0)? -(numer / denom): (float)0.0;

/*
 *         if intersection - determine if its in rectangle
 */
           if ((numer != 0.0) || (denom != 0.0) || (*t > 0.0) )  {
               *Pint = VectorScaler_Product(&ray.dir, *t);
               *Pint = Vector_Sum(&ray.origin, Pint);

               if (norm_flg)
                   vset(r_norm,  (float) 1.0,  (float) 0.0,  (float) 0.0);

               if (Pint->y <= r_ptr->max_pt.y && Pint->y >= r_ptr->min_pt.y &&
                   Pint->z <= r_ptr->max_pt.z && Pint->z >= r_ptr->min_pt.z)
                   int_flg = TRUE;
              }
          }

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  handle rectangle parallel to xz coordinate plane
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       else if (r_ptr->normal == Y_AXIS)  {

/*
 *         determine value for intersection
 */
           D = -(r_ptr->corners[0].y);

           numer = ray.origin.y + D;
           denom = ray.dir.y;
           *t = (denom != 0.0)? -(numer / denom): (float)0.0;

/*
 *         if intersection - determine if its in rectangle
 */
           if ((numer != 0.0) || (denom != 0.0) || (*t > 0.0) )  {
               *Pint = VectorScaler_Product(&ray.dir, *t);
               *Pint = Vector_Sum(&ray.origin, Pint);
               if (norm_flg)
                   vset(r_norm,  (float) 0.0,  (float) 1.0,  (float) 0.0);

               if (Pint->x <= r_ptr->max_pt.x && Pint->x >= r_ptr->min_pt.x &&
                   Pint->z <= r_ptr->max_pt.z && Pint->z >= r_ptr->min_pt.z)
                   int_flg = TRUE;
              }

          }

/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +  handle rectangle parallel to xy coordinate plane
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */
       else if (r_ptr->normal == Z_AXIS)  {

/*
 *         determine value for intersection
 */
           D = -(r_ptr->corners[0].z);

           numer = ray.origin.z + D;
           denom = ray.dir.z;
           *t = (denom != 0.0)? -(numer / denom): (float)0.0;

/*
 *         if intersection - determine if its in rectangle
 */
           if ((numer != 0.0) || (denom != 0.0) || (*t > 0.0) )  {
               *Pint = VectorScaler_Product(&ray.dir, *t);
               *Pint = Vector_Sum(&ray.origin, Pint);
               if (norm_flg)
                   vset(r_norm,  (float) 0.0,  (float) 0.0,  (float) 1.0);

               if (Pint->x <= r_ptr->max_pt.x && Pint->x >= r_ptr->min_pt.x &&
                   Pint->y <= r_ptr->max_pt.y && Pint->y >= r_ptr->min_pt.y)
                   int_flg = TRUE;
              }
          }
/*
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 */

       return (int_flg);
      }




