/****************************************************************************
    $Id: stats1.cpp 501.0 1995/03/07 12:27:02 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of class TLStats1<T>, a class to gather statistics for
    a single data series.

    $Log: stats1.cpp $
    Revision 501.0  1995/03/07 12:27:02  RON
    Updated for TLX 5.01
    Revision 1.6  1995/01/31 16:30:52  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.5  1995/01/06  16:04:32  ron
    Added sum to statistics output

    Revision 1.4  1995/01/05  15:39:38  ron
    Widened counting members to 32 bits
    Changed output format slightly

    Revision 1.3  1994/11/16  15:33:56  ron
    Changed TLX_TRACE calls to incorporate group

    Revision 1.2  1994/10/07  17:10:23  ron
    Renamed Insert() to Add()

    Revision 1.1  1994/10/06  17:51:43  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_STATS1_CPP
#define _TLX_STATS1_CPP

#include <iostream.h>
#include <math.h>

#ifndef _TLX_DEBUG_H
#include <tlx\501\debug.h>
#endif
#ifndef _TLX_RTTI_H
#include <tlx\501\rtti.h>
#endif
#ifndef _TLX_STATS_H
#include <tlx\501\stats.h>
#endif
#ifndef _TLX_UTIL_H
#include <tlx\501\util.h>
#endif

/*-------------------------------------------------------------------------*/
    template<class T> TLStats1<T>::TLStats1()

/*  Default constructor; resets all statistics.
---------------------------------------------------------------------------*/
{
    Clear();
}

/*-------------------------------------------------------------------------*/
    template<class T> void TLStats1<T>::Add(T aValue)

/*  Adds a single value to the data series, updating the statistics as
    necessary.
---------------------------------------------------------------------------*/
{
    mSum 	+= aValue;
    mSumSquares += aValue * aValue;
    mMax 	 = mCount ? tlMax(mMax, aValue) : aValue;
    mMin 	 = mCount ? tlMin(mMin, aValue) : aValue;

    mCount++;
    InvalidateCache(cvAverage | cvVariance);
}

/*-------------------------------------------------------------------------*/
    template<class T>
    void TLStats1<T>::Add(const T *aVector, size_t aCount)

/*  Adds all values of a given C-style vector to the data series.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aVector);

    for (size_t i = 0; i < aCount; i++)
	Add(aVector[i]);
}

/*-------------------------------------------------------------------------*/
    template<class T> void TLStats1<T>::Add(TLValueIter<T> &aIter)

/*  Adds all values addressable by the iterator to the data series.
---------------------------------------------------------------------------*/
{
    for (aIter.Reset(); aIter.Next(); )
	Add(aIter.Peek());
}

/*-------------------------------------------------------------------------*/
    template<class T> T TLStats1<T>::Average() const

/*  Returns the average of the data series so far. If the series is empty,
    0 is returned.
---------------------------------------------------------------------------*/
{
    if (IsCacheInvalid(cvAverage))
    {
	if (Count() < 1)
	{
	    TLX_TRACE_WARN(TLX, ("Average of empty series requested"));
	    TLX_ASSERT(mAverage == 0);
	}
	else
#ifdef _NOMUTABLE
	    ((TLStats1<T> *)this)->mAverage = mSum / Count();
#else
	    mAverage = mSum / Count();
#endif
	ValidateCache(cvAverage);
    }
    TLX_ASSERT(IsCacheValid(cvAverage));

    return mAverage;
}

/*-------------------------------------------------------------------------*/
    template<class T> void TLStats1<T>::Clear()

/*  Resets all statistics.
---------------------------------------------------------------------------*/
{
    mCount 	= 0;
    mCacheFlags = cvNone;
    mSum 	=
    mSumSquares =
    mMax 	=
    mMin 	=
    mAverage 	=
    mVariance   = 0;
}

/*-------------------------------------------------------------------------*/
    template<class T> T TLStats1<T>::StdDev() const

/*  Returns the standard deviation of the data series so far. If the number
    of data elements is < 2, 0 is returned.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(Variance() >= 0.0);

#ifdef _MSC_VER
    return sqrt((double)Variance());
#else
    return sqrt(Variance());
#endif
}

/*-------------------------------------------------------------------------*/
    template<class T> T TLStats1<T>::Variance() const

/*  Returns the variance of the data series so far. If the number of data
    elements is < 2, 0 is returned.
---------------------------------------------------------------------------*/
{
    if (IsCacheInvalid(cvVariance))
    {
	if (Count() < 2)
	{
	    TLX_TRACE_WARN(TLX, ("Variance of series length %u requested",
	    		      Count()));
	    TLX_ASSERT(mVariance == 0);
	}
	else
	{
#ifdef _NOMUTABLE
	    ((TLStats1<T> *)this)->mVariance = (Count() * mSumSquares - 
		mSum * mSum) / (Count() * (Count() - 1));
#else
	    mVariance = (Count() * mSumSquares - mSum * mSum) /
			(Count() * (Count() - 1));
#endif
	}
	ValidateCache(cvVariance);
    }
    TLX_ASSERT(IsCacheValid(cvVariance));

    return mVariance;
}

/*-------------------------------------------------------------------------*/
    template<class T>
    ostream & _TLXFUNC operator <<(ostream &os, const TLStats1<T> &aStats)

/*  Prints the statistics on the given output stream.
---------------------------------------------------------------------------*/
{
#ifdef _NOMUTABLE
    TLStats1<T> &s = CONSTCAST(TLStats1<T> &, aStats);
#else
    #define s aStats
#endif

    os << "Cnt=" << s.Count() << ", Sum=" << s.Sum();
    os << ", Distri=[" << s.Min();
    if (s.Count() > 0) os << "," << s.Average();
    os << "," << s.Max() << "]";
    if (s.Count() > 1) os << ", Sdev=" << s.StdDev();

    return os;
}

#endif	// _TLX_STATS1_CPP
