/****************************************************************************
    $Id: ptriter.cpp 501.0 1995/03/07 12:26:58 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Implementation of pointer iterator templates:

    - TLPtrIter
    - TLPtrIterConst

    $Log: ptriter.cpp $
    Revision 501.0  1995/03/07 12:26:58  RON
    Updated for TLX 5.01
    Revision 1.6  1995/01/31 16:30:44  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.5  1994/11/16  15:33:12  ron
    Chnaged names of UNREACHABLE assertions

    Revision 1.4  1994/10/06  17:51:06  ron
    Changed #defined name

    Revision 1.3  1994/10/05  18:48:19  ron
    Implemented TLPtrIter, TLPtrIterConst, TLPtrItemIter,
    and TLPtrItemIterConst

    Revision 1.2  1994/09/27  20:27:19  ron
    Changed path separator from / to \

    Revision 1.1  1994/09/26  15:32:03  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_PTRITER_CPP
#define _TLX_PTRITER_CPP

#include <iostream.h>

#ifndef _TLX_DEBUG_H
#include <tlx\501\debug.h>
#endif
#ifndef _TLX_EXCEPT_H
#include <tlx\501\except.h>
#endif
#ifndef _TLX_ITER_H
#include <tlx\501\iter.h>
#endif

/****************************************************************************

    Implementation of class TLPtrIter<T>

****************************************************************************/

/*-------------------------------------------------------------------------*/
    template<class T> TLPtrIter<T>::TLPtrIter()

/*  Default constructor. Does nothing.
---------------------------------------------------------------------------*/
{
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIter<T>::Contains(T *&aPtr)

/*  Checks of the given item appears in the associated collection. This
    operation will change the state of the iterator.
---------------------------------------------------------------------------*/
{
    for (Reset(); Next(); )
	if (Peek() == aPtr)
	    return true;
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> size_t TLPtrIter<T>::Count() const

/*  Returns the number of items in the underlying collection, which is
    always 0.
---------------------------------------------------------------------------*/
{
    return 0;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIter<T>::FirstPos()

/*  Advances the iterator to the first position in the associated
    collection. In this case, the operation always fails.
---------------------------------------------------------------------------*/
{
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIter<T>::FirstThat
    (
    	tTestFunc 	aTest, 		// Test to apply
	void *		aArg, 		// Optional argument
	T **&		aSuccess	// Set to first succeeding element
    )

/*  Applies a given test to elements in the associated collection until the
    test returns true. It then sets 'aSuccess' to point to the first element
    that made the test succeed and returns true. If no element made the
    test succeed, false is returned and 'aSuccess' is not changed.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aTest);

    for (Reset(); Next(); )
	if ((*aTest)(Peek(), aArg))
	{
	    aSuccess = &Peek();
	    return true;
	}
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIter<T>::FirstThatNot
    (
    	tTestFunc 	aTest,		// Test to apply
	void *		aArg, 		// Optional argument
	T **&		aFailure	// Set to first failing element
    )

/*  Applies a given test to elements in the associated collection as long as
    the test returns true. It then sets 'aFailure' to point to the first
    element that made the test fail and returns true. If no element made the
    test fail, false is returned and 'aFailure' is not changed.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aTest);

    for (Reset(); Next(); )
	if (!(*aTest)(Peek(), aArg))
	{
	    aFailure = &Peek();
	    return true;
	}
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> void TLPtrIter<T>::ForAll
    (
    	tApplyFunc 	aFunc,
	void *		aArg
    )

/*  Applies a given function to all elements in the associated collection.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aFunc);

    for (Reset(); Next(); )
	(*aFunc)(Peek(), aArg);
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIter<T>::NextPos()

/*  Advances the iterator to the next position in the associated collection.
    This version should never be called, since the iterator already fails
    the first position.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_UNREACHABLE;
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> T *&TLPtrIter<T>::Peek() const

/*  Returns a reference to the currently addressed item in the associated
    collection. This version should never be called, since the iterator
    fails all iteration attempts.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_UNREACHABLE;
    THROW(TLXEmpty(LOCUS));
#if defined(_MSC_VER) || defined(__SC__)
    // Microsoft and Symantec C++ don't know this is never reached
    static T *dummy = 0;
    return dummy;
#endif
}

/*-------------------------------------------------------------------------*/
    template<class T> ostream &TLPtrIter<T>::PrintAll(ostream &os)

/*  Prints all elements in the associated collection without any formatting.
    This function assumes the existence of operator <<(ostream &, const T &).
---------------------------------------------------------------------------*/
{
    return PrintFormat(os, TLPrintFormat());
}

/*-------------------------------------------------------------------------*/
    template<class T> ostream &TLPtrIter<T>::PrintFormat
    (
    	ostream &		os,
	const TLPrintFormat &	aFmt
    )

/*  Prints all elements in the associated collection with the given
    formatting around the collection and its elements. This function
    assumes the existence of operator <<(ostream &, const T &).
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aFmt.mLeader);
    TLX_ASSERT_PTR(aFmt.mPresep);
    TLX_ASSERT_PTR(aFmt.mPostsep);
    TLX_ASSERT_PTR(aFmt.mTrailer);

    os << aFmt.mLeader;
    for (Reset(); Next(); )
    {
	os << aFmt.mPresep;
/*	if (Peek())
	    os << *Peek();
	else
	    os << "NULL";
*/	os << aFmt.mPostsep;
    }

    return os << aFmt.mTrailer;
}

/****************************************************************************

    Implementation of class TLPtrIterConst<T>

****************************************************************************/

/*-------------------------------------------------------------------------*/
    template<class T> TLPtrIterConst<T>::TLPtrIterConst()

/*  Default constructor. Does nothing.
---------------------------------------------------------------------------*/
{
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIterConst<T>::Contains(T *aPtr)

/*  Checks of the given item appears in the associated collection. This
    operation will change the state of the iterator.
---------------------------------------------------------------------------*/
{
    for (Reset(); Next(); )
	if (Peek() == aPtr)
	    return true;
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> size_t TLPtrIterConst<T>::Count() const

/*  Returns the number of items in the underlying collection, which is
    always 0.
---------------------------------------------------------------------------*/
{
    return 0;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIterConst<T>::FirstPos()

/*  Advances the iterator to the first position in the associated
    collection. In this case, the operation always fails.
---------------------------------------------------------------------------*/
{
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIterConst<T>::FirstThat
    (
    	tTestFunc 	aTest, 		// Test to apply
	void *		aArg, 		// Optional argument
	T *&		aSuccess	// Set to first succeeding element
    )

/*  Applies a given test to elements in the associated collection until the
    test returns true. It then sets 'aSuccess' to point to the first element
    that made the test succeed and returns true. If no element made the
    test succeed, false is returned and 'aSuccess' is not changed.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aTest);

    for (Reset(); Next(); )
	if ((*aTest)(Peek(), aArg))
	{
	    aSuccess = Peek();
	    return true;
	}
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIterConst<T>::FirstThatNot
    (
    	tTestFunc 	aTest,		// Test to apply
	void *		aArg, 		// Optional argument
	T *&		aFailure	// Set to first failing element
    )

/*  Applies a given test to elements in the associated collection as long as
    the test returns true. It then sets 'aFailure' to point to the first
    element that made the test fail and returns true. If no element made the
    test fail, false is returned and 'aFailure' is not changed.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aTest);

    for (Reset(); Next(); )
	if (!(*aTest)(Peek(), aArg))
	{
	    aFailure = Peek();
	    return true;
	}
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> void TLPtrIterConst<T>::ForAll
    (
    	tApplyFunc 	aFunc,
	void *		aArg
    )

/*  Applies a given function to all elements in the associated collection.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aFunc);

    for (Reset(); Next(); )
	(*aFunc)(Peek(), aArg);
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrIterConst<T>::NextPos()

/*  Advances the iterator to the next position in the associated collection.
    This version should never be called, since the iterator already fails
    the first position.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_UNREACHABLE;
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> T *TLPtrIterConst<T>::Peek() const

/*  Returns a pointer to the currently addressed item in the associated
    collection. This version should never be called, since the iterator
    fails all iteration attempts.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_UNREACHABLE;
    THROW(TLXEmpty(LOCUS));
#if defined(_MSC_VER) || defined(__SC__)
    // Microsoft and Symantec C++ don't know this is never reached
    return 0;
#endif
}

/*-------------------------------------------------------------------------*/
    template<class T> ostream &TLPtrIterConst<T>::PrintAll(ostream &os)

/*  Prints all elements in the associated collection without any formatting.
    This function assumes the existence of operator <<(ostream &, const T &).
---------------------------------------------------------------------------*/
{
    return PrintFormat(os, TLPrintFormat());
}

/*-------------------------------------------------------------------------*/
    template<class T> ostream &TLPtrIterConst<T>::PrintFormat
    (
    	ostream &		os,
	const TLPrintFormat &	aFmt
    )

/*  Prints all elements in the associated collection with the given
    formatting around the collection and its elements. This function
    assumes the existence of operator <<(ostream &, const T &).
---------------------------------------------------------------------------*/
{
    TLX_ASSERT_PTR(aFmt.mLeader);
    TLX_ASSERT_PTR(aFmt.mPresep);
    TLX_ASSERT_PTR(aFmt.mPostsep);
    TLX_ASSERT_PTR(aFmt.mTrailer);

    os << aFmt.mLeader;
    for (Reset(); Next(); )
    {
	os << aFmt.mPresep;
	if (Peek())
	    os << *Peek();
	else
	    os << "NULL";
	os << aFmt.mPostsep;
    }

    return os << aFmt.mTrailer;
}

/*-------------------------------------------------------------------------*/
    template<class T> TLPtrItemIter<T>::TLPtrItemIter(T *&aItem)

/*  Constructor. Links to the given item.
---------------------------------------------------------------------------*/
: mItem(aItem)
{
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrItemIter<T>::Contains(T *&aItem)

/*  Returns true iff the associated collection (a single element) contains
    the item. In this case, a simple test for equality is performed.
---------------------------------------------------------------------------*/
{
    return aItem == mItem;
}

/*-------------------------------------------------------------------------*/
    template<class T> size_t TLPtrItemIter<T>::Count() const

/*  Returns the number of elements in the associated collection, which is
    always 1 in this case.
---------------------------------------------------------------------------*/
{
    return 1;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrItemIter<T>::FirstPos()

/*  Sets the iterator to the first available position (if any), returning
    true if that move succeeds.
---------------------------------------------------------------------------*/
{
    return true;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrItemIter<T>::NextPos()

/*  Advances the iterator to the next available position (if any), returning
    true if that move succeeds.
---------------------------------------------------------------------------*/
{
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> T *&TLPtrItemIter<T>::Peek() const

/*  Returns a reference to the currently iterated element.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(IsValid());
    return mItem;
}

/*-------------------------------------------------------------------------*/
    template<class T> TLPtrItemIterConst<T>::TLPtrItemIterConst(T *aItem)

/*  Constructor. Links to the given item.
---------------------------------------------------------------------------*/
: mItem(aItem)
{
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrItemIterConst<T>::Contains(T *aItem)

/*  Returns true iff the associated collection (a single element) contains
    the item. In this case, a simple test for equality is performed.
---------------------------------------------------------------------------*/
{
    return aItem == mItem;
}

/*-------------------------------------------------------------------------*/
    template<class T> size_t TLPtrItemIterConst<T>::Count() const

/*  Returns the number of elements in the associated collection, which is
    always 1 in this case.
---------------------------------------------------------------------------*/
{
    return 1;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrItemIterConst<T>::FirstPos()

/*  Sets the iterator to the first available position (if any), returning
    true if that move succeeds.
---------------------------------------------------------------------------*/
{
    return true;
}

/*-------------------------------------------------------------------------*/
    template<class T> bool TLPtrItemIterConst<T>::NextPos()

/*  Advances the iterator to the next available position (if any), returning
    true if that move succeeds.
---------------------------------------------------------------------------*/
{
    return false;
}

/*-------------------------------------------------------------------------*/
    template<class T> T *TLPtrItemIterConst<T>::Peek() const

/*  Returns a reference to the currently iterated element.
---------------------------------------------------------------------------*/
{
    TLX_ASSERT(IsValid());
    return mItem;
}

#endif	// _TLX_PTRITER_CPP
