/****************************************************************************
    $Id: stats.h 501.0 1995/03/07 12:26:46 RON Exp $

    Copyright (c) 1991-95 Tarma Software Research. All rights reserved.

    Project:	Tarma Library for C++ V5.0
    Author:	Ron van der Wal

    Declarations for statistics gathering classes.

    $Log: stats.h $
    Revision 501.0  1995/03/07 12:26:46  RON
    Updated for TLX 5.01
    Revision 1.6  1995/01/31 16:29:24  RON
    Update for release 012
    Added partial support for SunPro C++ compiler
    Revision 1.5  1995/01/06  16:01:33  ron
    Added Sum() member function

    Revision 1.4  1995/01/05  15:34:07  ron
    Widened count members of TLStats1<T>

    Revision 1.3  1994/10/07  17:08:26  ron
    Renamed Insert() operations to Add()

    Revision 1.2  1994/10/06  17:49:17  ron
    Replaced TLRealStats with a template class

    Revision 1.1  1994/10/05  18:24:38  ron
    Initial revision

****************************************************************************/

#ifndef _TLX_STATS_H
#define _TLX_STATS_H

#ifndef _TLX_ITER_H
#include <tlx\501\iter.h>
#endif

class _RTLCLASS ostream;

/*---------------------------------------------------------------------------
    TLStats1<T> -

    Statistics collecting class for values of type T in a single data
    series. The class employs lazy evalution techniques to minimize
    the calculations involved.

    Note: type T must allow arithmetic operations and be convertable
    from/to type double.
---------------------------------------------------------------------------*/

template<class T> class _TLXCLASS TLStats1
{
    // For efficiency reasons, we permanently store only a few values
    // that can be recalculated with little effort whenever a datum is
    // added. The other statistics are calculated on demand, although
    // their previous value is cached and will be reused if possible.
    // The following flags indicate which cached values are valid at a
    // given moment.

    enum {
	cvNone		= 0x0000,	// No cached values are valid
	cvMax		= 0x0001,	// Cached max value still valid
	cvMin		= 0x0002,	// Cached min value still valid
	cvAverage	= 0x0004,	// Cached average value still valid
	cvVariance	= 0x0008,	// Cached variance value still valid
	cvAll		= 0xffff	// All values are valid
    };

    // The following values are always valid: they can and will be updated
    // on each insertion.

    uint32		mCount;
    T			mSum;
    T			mSumSquares;
    T			mMax;
    T			mMin;

    // The next values are subject to caching; their validity depends on the
    // cache flags.

    mutable int		mCacheFlags;
    mutable T		mAverage;
    mutable T		mVariance;

public:
    TLStats1();

    // Addition of individual items and data series, and wholesale removal.

    void		Add(T);
    void		Add(const T *, size_t);
    void		Add(TLValueIter<T> &);
    void		Clear();

    // Statistics of the data series.

    uint32		Count() const;
    T			Sum() const;
    T			Max() const;
    T			Min() const;
    T			Average() const;
    T			Variance() const;
    T			StdDev() const;

    // Output operator

#if !defined(_MSC_VER) && !defined(__SC__)
    // Microsoft/Symantec C++ quirk
    friend ostream &	operator <<(ostream &, const TLStats1<T> &);
#endif

private:
    // Implementation helper functions, primarily for cache maintenance.

    bool		IsCacheInvalid(int) const;
    bool		IsCacheValid(int) const;
    void		ValidateCache(int) const;
    void		InvalidateCache(int) const;
};

#if defined(_MSC_VER) || defined(__SC__)
template<class T>
ostream & _TLXFUNC operator <<(ostream &, const TLStats1<T> &);
#endif

/*---------------------------------------------------------------------------
    Inline functions
---------------------------------------------------------------------------*/

//-----	TLStats1<T>

template<class T> inline uint32 TLStats1<T>::Count() const
    { return mCount; }

template<class T> inline T TLStats1<T>::Sum() const
    { return mSum; }

template<class T> inline bool TLStats1<T>::IsCacheInvalid(int aFlags) const
    { return (mCacheFlags & aFlags) != aFlags; }

template<class T> inline bool TLStats1<T>::IsCacheValid(int aFlags) const
    { return (mCacheFlags & aFlags) == aFlags; }

template<class T> inline void TLStats1<T>::ValidateCache(int aFlags) const
#ifdef _NOMUTABLE
    { ((TLStats1<T> *)this)->mCacheFlags |= aFlags; }
#else
    { mCacheFlags |= aFlags; }
#endif

template<class T> inline void TLStats1<T>::InvalidateCache(int aFlags) const
#ifdef _NOMUTABLE
    { ((TLStats1<T> *)this)->mCacheFlags &= ~aFlags; }
#else
    { mCacheFlags &= ~aFlags; }
#endif

template<class T> inline T TLStats1<T>::Max() const
    { return mMax; }

template<class T> inline T TLStats1<T>::Min() const
    { return mMin; }

#endif	// _TLX_STATS_H
